/*****************************************************************************/
/*
                                CGIsym.c

Creates global symbols required by the WASD CGI interface.  Essentially this
performs exactly the same role as SET_DCL_ENV.C and CGI_SYMBOLS.C without the
CGILIB.C and SCRIPTLIB.C paraphernalia.  One execption is PATH_TRANSLATED where
OSU provides a Unix-style VMS equivalent path where-as WASD requires it in an
RMS format.  This program is not designed as a drop-in replacement for either
of the above utilities, it's scope is deliberately more limited, and is to
support the used of WASD scripts in the OSU environment.


PARAMETERS
----------
P1      HTTP method (e.g. GET, POST, etc.)
P2      path (URL-style)
P3      protocol (e.g. HTTP/1.0)


VERSION HISTORY (update 'SoftwareID' as well)
---------------
01-AUG-98  MGD  v1.0.0, initial development
*/
/*****************************************************************************/

#ifdef __ALPHA
   char SoftwareID [] = "CGISYM AXP-1.0.0";
#else
   char SoftwareID [] = "CGISYM VAX-1.0.0";
#endif

#ifdef __ALPHA
#   pragma nomember_alignment
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <math.h>

#include <descrip.h>
#include <libdef.h>
#include <libclidef.h>
#include <syidef.h>
#include <ssdef.h>
#include <stsdef.h>

#define VMSok(x) ((x) & STS$M_SUCCESS)
#define VMSnok(x) !(((x) & STS$M_SUCCESS))

#define boolean int
#define true 1
#define false 0

char  Utility [] = "CGISYM";

boolean  Debug;

/* required prototypes */
char* QueryNetLink (char*, int);

/*****************************************************************************/
/*
*/

main
(
int argc,
char **argv
)
{
   register char  *cptr, *sptr, *zptr;

   int  KeyCount,
        IpAddress;
   char  PathInfo [256],
         PathTranslated [256],
         Scratch [256],
         ScriptName [256],
         SymbolName [256];

   /*********/
   /* begin */
   /*********/

   if (getenv ("CGISYM$DBUG") != NULL) Debug = true;

   if (argc < 4)
   {
      fprintf (stdout, "%%%s-E-USE, <method> <path> <protocol>\n", Utility);
      exit (STS$K_ERROR | STS$M_INHIB_MSG);
   }

   if (Debug)
   {
      /* let's have a look at what's available! */
      fprintf (stdout, "----------\n");
      QueryNetLink ("<DNETARG>", 1);
      QueryNetLink ("<DNETARG2>", 1);
      QueryNetLink ("<DNETHOST>", 1);
      QueryNetLink ("<DNETID>", 1);
      QueryNetLink ("<DNETID2>", 1);
      QueryNetLink ("<DNETRQURL>", 1);
      QueryNetLink ("<DNETBINDIR>", 1);
      QueryNetLink ("<DNETPATH>", 1);
      QueryNetLink ("<DNETHDR>", 999);
      fprintf (stdout, "----------\n");
   }

   /*****************/
   /* the following */
   /*****************/

   AssignGlobalSymbol ("WWW_GATEWAY_INTERFACE", "CGI/1.1", 0);

   AssignGlobalSymbol ("WWW_CONTENT_LENGTH", "0", 0);

   AssignGlobalSymbol ("WWW_CONTENT_TYPE", "", 0);

   AssignGlobalSymbol ("WWW_REQUEST_METHOD", argv[1], 0);

   AssignGlobalSymbol ("WWW_SERVER_PROTOCOL", argv[3], 0);

   /*******************************************/
   /* SCRIPT_NAME, PATH_INFO, PATH_TRANSLATED */
   /*******************************************/

   strcpy (sptr = ScriptName, QueryNetLink ("<DNETPATH>", 1));
   cptr = QueryNetLink ("<DNETRQURL>", 1);
   while (*cptr && *cptr && *cptr == *sptr) { cptr++; sptr++; }
   while (*cptr && *cptr != '/' && *cptr != '?') *sptr++ = *cptr++;
   *sptr = '\0';
   UrlDecode (ScriptName);
   AssignGlobalSymbol ("WWW_SCRIPT_NAME", ScriptName, 0);

   /* continue to get the path information from the request URL */
   sptr = PathInfo;
   while (*cptr && *cptr != '?') *sptr++ = *cptr++;
   *sptr = '\0';
   UrlDecode (PathInfo);

   if (*PathInfo && *(unsigned short*)PathInfo != '/\0')
   {
      QueryNetLink ("<DNETXLATE>", 0);
      sptr = QueryNetLink (PathInfo, 1);

      /* transmute the OSU Unix-style specification into VMS-style */
      zptr = NULL;
      if (*sptr == '/') sptr++;
      cptr = PathTranslated;
      while (*sptr && *sptr != '/') *cptr++ = toupper(*sptr++);
      if (*sptr)
      {
         sptr++;
         *cptr++ = ':';
         *cptr++ = '[';
      }
      while (*sptr)
      {
         while (*sptr && *sptr != '/') *cptr++ = toupper(*sptr++);
         if (!*sptr) break;
         /* directory separating '/' */
         sptr++;
         zptr = cptr;
         *cptr++ = '.';
      }
      /* revert the last '/' from a '.' to a ']' */
      if (zptr != NULL) *zptr = ']';
      *cptr = '\0';

      AssignGlobalSymbol ("WWW_PATH_INFO", PathInfo, 0);
      AssignGlobalSymbol ("WWW_PATH_TRANSLATED", PathTranslated, 0);
   }
   else
   {
      AssignGlobalSymbol ("WWW_PATH_INFO", "/", 0);
      AssignGlobalSymbol ("WWW_PATH_TRANSLATED", "", 0);
   }

   /***********************************/
   /* QUERY_STRING, FORM_..., KEY_... */
   /***********************************/

   cptr = QueryNetLink ("<DNETARG>", 1);
   if (*cptr == '?') cptr++;
   AssignGlobalSymbol ("WWW_QUERY_STRING", cptr, 0);

   KeyCount = 0;
   sptr = cptr;
   while (*cptr && *cptr != '=') cptr++;
   if (*cptr)
   {
      /* FORM-formatted query string */
      while (*cptr)
      {
         *cptr++ = '\0';
         sprintf (SymbolName, "WWW_FORM_%s", sptr);
         sptr = cptr;
         while (*cptr && *cptr != '&') cptr++;
         if (*cptr) *cptr++ = '\0';
         UrlDecode (SymbolName);
         UrlDecode (sptr);
         AssignGlobalSymbol (SymbolName, sptr, 0);
         sptr = cptr;
         while (*cptr && *cptr != '=') cptr++;
      }
   }
   else
   {
      /* ISQUERY-formatted query string */
      cptr = sptr;
      while (*cptr && *cptr != '+') cptr++;
      while (*sptr)
      {
         *cptr++ = '\0';
         sprintf (SymbolName, "WWW_KEY_%d", ++KeyCount);
         UrlDecode (sptr);
         AssignGlobalSymbol (SymbolName, sptr, 0);
         sptr = cptr;
         while (*cptr && *cptr != '+') cptr++;
      }
   }
   AssignGlobalSymbol ("WWW_KEY_COUNT", NULL, KeyCount);

   /**********************************************/
   /* SERVER_SOFTWARE, SERVER_NAME, SERVER_PORT, */
   /* REMOTE_HOST, REMOTE_ADDR, REMOTE_USER      */
   /**********************************************/

   cptr = sptr = QueryNetLink ("<DNETID2>", 1);

   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';
   AssignGlobalSymbol ("WWW_SERVER_SOFTWARE", sptr, 0);

   sptr = cptr;
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';
   AssignGlobalSymbol ("WWW_SERVER_NAME", sptr, 0);

   sptr = cptr;
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';
   AssignGlobalSymbol ("WWW_SERVER_PORT", sptr, 0);

   /* skip over client port */
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';

   /* host address */
   sptr = cptr;
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';
   for (zptr = sptr; *zptr && *zptr != '.'; zptr++);
   if (*zptr != '.')
   {
      IpAddress = atoi(sptr);
      sprintf (Scratch, "%d.%d.%d.%d",
               *(unsigned char*)&IpAddress,
               *((unsigned char*)(&IpAddress)+1),
               *((unsigned char*)(&IpAddress)+2),
               *((unsigned char*)(&IpAddress)+3));
      while (*cptr && !isspace(*cptr)) cptr++;
      sptr = Scratch;
   }
   AssignGlobalSymbol ("WWW_REMOTE_ADDR", sptr, 0);
   AssignGlobalSymbol ("WWW_REMOTE_HOST", sptr, 0);

   sptr = cptr;
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr) *cptr++ = '\0';
   AssignGlobalSymbol ("WWW_REMOTE_USER", sptr, 0);
   sptr = cptr;

   sptr = cptr;
   while (*cptr && !isspace(*cptr)) cptr++;
   if (*cptr)
   {
      /* DNS-resolved host name */
      *cptr++ = '\0';
      AssignGlobalSymbol ("WWW_REMOTE_HOST", sptr, 0);
   }

   /******************************************/
   /* CONTENT_LENGTH, CONTENT_TYPE, HTTP_... */
   /******************************************/

   cptr = sptr = QueryNetLink ("<DNETHDR>", 999);
   while (*cptr)
   {
      while (*cptr && *cptr != ':')
      {
         if (!isalnum(*cptr)) *cptr = '_';
         cptr++;
      }
      if (*cptr)
      {
         *cptr++ = '\0';
         sprintf (SymbolName, "WWW_HTTP_%s", sptr);
         while (*cptr && isspace(*cptr) && *cptr != '\n') *cptr++;
         sptr = cptr;
         while (*cptr && *cptr != '\n') *cptr++;
         if (*cptr) *cptr++ = '\0';
         UrlDecode (SymbolName);
         UrlDecode (sptr);
         AssignGlobalSymbol (SymbolName, sptr, 0);
         if (strsame (SymbolName+9, "CONTENT_TYPE" -1))
            AssignGlobalSymbol ("WWW_CONTENT_TYPE", sptr, 0);
         else
         if (strsame (SymbolName+9, "CONTENT_LENGTH" -1))
            AssignGlobalSymbol ("WWW_CONTENT_LENGTH", sptr, 0);
         sptr = cptr;
      }
   }

   exit (SS$_NORMAL);
}  

/*****************************************************************************/
/*
Write a single record to the OSU network link, then if expecting a response
wait for one or more records to be returned.  A single line response has no
carriage control.  A multi-line response has each record concatenated into a
single string of newline-separated records.  Returns pointer to static storage!
Open files to the link if first call.
*/

char* QueryNetLink
(
char *Query,
int ResponseLinesExpected
)
{
   static char  Response [4096];
   static FILE  *NetLinkRead,
                *NetLinkWrite;

   register char  *cptr;

   int  status;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "QueryNetLink() |%s|\n", Query);

   if (NetLinkRead == NULL)
   {
      /*************************/
      /* first call, open link */
      /*************************/

      if ((NetLinkRead = fopen ("NET_LINK:", "r", "ctx=rec")) == NULL)
      {
         status = vaxc$errno;
         fprintf (stdout, "%%%s-E-NETLINK, read open error\n", Utility);
         exit (status);
      }
      if ((NetLinkWrite = fopen ("NET_LINK:", "w", "ctx=bin")) == NULL)
      {
         status = vaxc$errno;
         fprintf (stdout, "%%%s-E-NETLINK, write open error\n", Utility);
         exit (status);
      }
   }

   /*********/
   /* write */
   /*********/

   if (fputs (Query, NetLinkWrite) == EOF)
   {
      status = vaxc$errno;
      fprintf (stdout, "%%%s-E-NETLINK, write error\n", Utility);
      exit (status);
   }
   fflush (NetLinkWrite);

   if (ResponseLinesExpected > 1)
   {
      /**************************/
      /* multiple line response */
      /**************************/

      cptr = Response;
      for (;;)
      {
         if (fgets (cptr, sizeof(Response)-(cptr-Response), NetLinkRead) == NULL)
         {
            status = vaxc$errno;
            fprintf (stdout, "%%%s-E-NETLINK, read error\n", Utility);
            exit (status);
         }
         /** if (Debug) fprintf (stdout, "cptr |%s|\n", cptr); **/
         if (*cptr != '\n')
         {
            while (*cptr) cptr++;
            continue;
         }
         *cptr = '\0';
         break;
      }
   }
   else
   if (ResponseLinesExpected > 0)
   {
      /************************/
      /* single line response */
      /************************/

      if (fgets (Response, sizeof(Response), NetLinkRead) == NULL)
      {
         status = vaxc$errno;
         fprintf (stdout, "%%%s-E-NETLINK, read error\n", Utility);
         exit (status);
      }

      for (cptr = Response; *cptr; cptr++);
      if (cptr > Response)
      {
         if (*--cptr == '\n')
            *cptr = '\0';
         else
            cptr++;
      }
   }

   if (Debug) fprintf (stdout, "%d |%s|\n", cptr-Response, Response);
   return (Response);
}  

/*****************************************************************************/
/*
Decode URL-encoded string.  Resultant string is always the same size or smaller
so it can be done in-situ!
*/

int UrlDecode (char *String)

{
   register char  c;
   register char  *cptr, *sptr;

   /*********/
   /* begin */
   /*********/

   if (Debug) fprintf (stdout, "UrlDecode() |%s|\n", String);

   cptr = sptr = String;
   while (*cptr)
   {
      switch (*cptr)
      {
         case '+' :

            *sptr++ = ' ';
            cptr++;

            continue;

         case '%' :

            cptr++;
            c = 0;
            if (*cptr >= '0' && *cptr <= '9')
               c = (*cptr - (int)'0') << 4;
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               c = (toupper(*cptr) - (int)'A' + 10) << 4;
            else
            {
               *String = '\0';
               return (-1);
            }
            if (*cptr) cptr++;
            if (*cptr >= '0' && *cptr <= '9')
               c += (*cptr - (int)'0');
            else
            if (toupper(*cptr) >= 'A' && toupper(*cptr) <= 'F')
               c += (toupper(*cptr) - (int)'A' + 10);
            else
            {
               *String = '\0';
               return (-1);
            }
            if (*cptr) cptr++;
            *sptr++ = c;

            continue;

         default :
            
            *sptr++ = *cptr++;

      }
   }
   *sptr = '\0';

   if (Debug) fprintf (stdout, "|%s|\n", String);
   return (sptr-String);
}

/****************************************************************************/
/*
Assign a global symbol.  If the string pointer is null the numeric value is
used.  Symbol lengths are adjusted according to the maximum allowed for the
version of the operating system.
*/ 

AssignGlobalSymbol
(
char *Name,
char *String,
int Value
)
{
   static int  GlobalSymbol = LIB$K_CLI_GLOBAL_SYM;
   static char  SyiVersion [16],
                ValueString [32];
   static $DESCRIPTOR (SymbolValueDsc, "");
   static $DESCRIPTOR (ValueFaoDsc, "!UL");
   static $DESCRIPTOR (ValueStringDsc, ValueString);

   static struct {
      short BufferLength;
      short ItemCode;
      void  *BufferPtr;
      void  *LengthPtr;
   }
   SyiItems [] =
   {
      { 8, SYI$_VERSION, &SyiVersion, 0 },
      { 0,0,0,0 }
   };

   int  status;
   $DESCRIPTOR (SymbolNameDsc, Name);

   /*********/
   /* begin */
   /*********/

   if (Debug)
      fprintf (stdout, "AssignGlobalSymbol() |%s|%s| %d\n",
               Name, String, Value);

   if (!SyiVersion[0])
   {
      if (VMSnok (status =
          sys$getsyiw (0, 0, 0, &SyiItems, 0, 0, 0)))
         exit (status);
      SyiVersion[8] = '\0';
      if (Debug) fprintf (stdout, "SyiVersion |%s|\n", SyiVersion);
   }

   SymbolNameDsc.dsc$w_length = strlen(Name);
   if (String == NULL)
   {
      SymbolValueDsc.dsc$a_pointer = ValueString;
      sys$fao (&ValueFaoDsc, &SymbolValueDsc.dsc$w_length, &ValueStringDsc,
               Value);
      ValueString[SymbolValueDsc.dsc$w_length] = '\0';
   }
   else
   {
      SymbolValueDsc.dsc$a_pointer = String;
      SymbolValueDsc.dsc$w_length = strlen(String);
      if (SymbolValueDsc.dsc$w_length > 255 && SyiVersion[1] <= '6')
         SymbolValueDsc.dsc$w_length = 255;
      else
      if (SymbolValueDsc.dsc$w_length > 1023)
         SymbolValueDsc.dsc$w_length = 1023;
   }

   if (VMSnok (status =
       lib$set_symbol (&SymbolNameDsc, &SymbolValueDsc, &GlobalSymbol)))
   {
      fprintf (stdout, "%%%s-E-SETSYMBOL, %s ... %d chars\n",
               Utility, Name, SymbolValueDsc.dsc$w_length);
      exit (status);
   }
}

/****************************************************************************/
/*
Does a case-insensitive, character-by-character string compare and returns 
true if two strings are the same, or false if not.  If a maximum number of 
characters are specified only those will be compared, if the entire strings 
should be compared then specify the number of characters as 0.
*/ 

boolean strsame
(
register char *sptr1,
register char *sptr2,
register int  count
)
{
   while (*sptr1 && *sptr2)
   {
      if (toupper (*sptr1++) != toupper (*sptr2++)) return (false);
      if (count)
         if (!--count) return (true);
   }
   if (*sptr1 || *sptr2)
      return (false);
   else
      return (true);
}

/*****************************************************************************/

