/* xmire -- mire under X
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*+
 * Title            : xmire
 * Subject          : Mire sous X.
 * File             : xmire.c
 * Author           : Martin Vicente <vicente@cena.dgac.fr> DGAC/CENA/SID
 * Last Modified By : Martin Vicente
 * Last Modified On : Thu Sep 21 10:18:26 1995
-*/



#ifdef VMS
#ifdef __DECC
#define NULL 0
#endif
#endif

# include <stdlib.h>
# include <stdio.h>
# include <X11/keysym.h>
# include <X11/Intrinsic.h>
# include <X11/StringDefs.h>
# include <X11/Shell.h>


#include "def.h"

#define STREQU(s1, s2)		(strcmp((s1), (s2)) == 0)
#define STREQU2(s, s1, s2)	(STREQU((s), (s1)) || STREQU((s), (s2)))
#define STREQU3(s, s1, s2, s3)	(STREQU((s), (s1)) || STREQU2((s), (s2), (s3)))

#define XCIRCLE			(360 * 64)
#define APPLICATION_CLASS	"Mire"
#define BACKGROUND		"black"
#define	BORDER_WIDTH		"0"
#define	BORDER_COLOR		"white"
#define DELAY			0
#define EVENT_MASK		(ExposureMask|ButtonPressMask)

#define XDrawCircle(dpy, win, gc, x, y, w, h) \
	XDrawArc((dpy), (win), (gc), (x), (y), (w), (h), 0, XCIRCLE)

#define XtNn	"n"
#define XtCN	"N"
#define XtNanim	"anim"
#define XtCAnim	"Anim"

struct app {int n; Boolean anim;};

enum {cyan, magenta, yellow, white, colorindexmax = white};

#ifdef __SunOS_4__
int system(char *string);
int fprintf(FILE *stream, char *format, ...);
#endif

static void  event_handler(Widget w, XtPointer client_data, XEvent *xevent);
static void  anim(XtPointer client_data, XtIntervalId *id);
static void  draw_mire(void);
static void  prev_color(void);
static void  next_color(void);
static void  set_color(unsigned short int colorindex);
static Pixel pixel(char *color_name);
static void  parse_options(int *argc, char **argv);
static void  help(void);
static void  usage(void);
static void  printGPL(void);

static char               *command_name;
static XtAppContext        app_context;
static Display            *display;
static Screen             *screen;
static GC                  gc;
static Colormap            colormap;
static unsigned short int  width, height;
static Window              window;
static unsigned long int   colors[colorindexmax+1];
static int                 colorindex = 0;
static Pixel               background;
static Dimension           border_width;
static struct app          app;
static unsigned int        nsegments;
static XSegment           *s;

static XrmOptionDescRec options[] = {
    {"-n",    ".n",    XrmoptionSepArg, null},
    {"-anim", ".anim", XrmoptionNoArg,  "True"}
};

static XtResource resources[] = {
    {XtNn, XtCN,
     XtRInt, sizeof (int),
     XtOffsetOf(struct app, n), XtRString, "8"},
    {XtNanim, XtCAnim,
     XtRBoolean, sizeof (Boolean),
     XtOffsetOf(struct app, anim), XtRString, "False"}
};

int main(int argc, char *argv[])
{
    char *fallback_resources[] = {
	APPLICATION_CLASS ".Background:"  BACKGROUND,
	APPLICATION_CLASS ".BorderWidth:" BORDER_WIDTH,
	APPLICATION_CLASS ".BorderColor:" BORDER_COLOR,
	NULL
    };

    Widget     wTopLevel;
    short int  x, y;
    int        i;

    command_name = argv[0];

    wTopLevel = XtVaAppInitialize(
	&app_context, APPLICATION_CLASS,
	options, XtNumber(options),
	&argc, argv, /* obligatoire */
	(String *)fallback_resources,
	XtNoverrideRedirect, true,
	null);

    parse_options(&argc, argv);

    XtVaGetValues(wTopLevel,
		  XtNbackground,  &background,
		  XtNborderWidth, &border_width,
		  null);

    XtGetApplicationResources(wTopLevel, &app,
			      resources, XtNumber(resources), NULL, 0);

    nsegments = 2 * app.n + 4;

    s = calloc(nsegments, sizeof (XSegment));

    display  = XtDisplay(wTopLevel);
    screen   = XtScreen(wTopLevel);
    gc       = DefaultGCOfScreen(screen);
    colormap = DefaultColormapOfScreen(screen);
    width    = WidthOfScreen(screen)  - 2 * border_width;
    height   = HeightOfScreen(screen) - 2 * border_width;
    window   = XtWindow(wTopLevel);

    colors[cyan]    = pixel("#00FFFF");
    colors[magenta] = pixel("#FF00FF");
    colors[yellow]  = pixel("#FFFF00");
    colors[white]   = pixel("#FFFFFF");

    set_color(colorindex);

    XtVaSetValues(wTopLevel, XtNwidth, width, XtNheight, height, null);

    x = 0;
    for (i = 0; i < app.n; i++) {
	s[i].x1 = x;
	s[i].y1 = 0;
	s[i].x2 = x;
	s[i].y2 = height;
	x += width/app.n;
    }
    
    y = 0;
    for (i = app.n; i < 2 * app.n; i++) {
	s[i].x1 = 0;
	s[i].y1 = y;
	s[i].x2 = width;
	s[i].y2 = y;
	y += height/app.n;
    }

    /* La limite a droite : */
    x = width-1;
    s[i].x1 = x;
    s[i].y1 = 0;
    s[i].x2 = x;
    s[i].y2 = height;
    ++i;
    /* La limite en bas : */
    y = height-1;
    s[i].x1 = 0;
    s[i].y1 = y;
    s[i].x2 = width;
    s[i].y2 = y;
    ++i;

    /* Les diagonales : */
    s[i].x1 = 0;
    s[i].y1 = 0;
    s[i].x2 = width;
    s[i].y2 = height;
    ++i;
    s[i].x1 = 0;
    s[i].y1 = height;
    s[i].x2 = width;
    s[i].y2 = 0;
    ++i;

    XtAddEventHandler(
	wTopLevel, EVENT_MASK, false,
	(XtEventHandler)event_handler, null);

    XtRealizeWidget(wTopLevel);

    XtAppMainLoop(app_context);

    return success;
}

static void event_handler(Widget w, XtPointer client_data, XEvent *xevent)
{
    if (window == None) {
	window = XtWindow(w);
	if (app.anim) XtAppAddTimeOut(app_context, DELAY, anim, null);
    }

    switch (xevent->type) {
    case Expose:
	draw_mire();
	break;
    case ButtonPress:
	switch (xevent->xbutton.button) {
	case 1:
	    prev_color();
	    XClearArea(display, window, 0, 0, 0, 0, true);
	    break;
	case 2:
	    next_color();
	    XClearArea(display, window, 0, 0, 0, 0, true);
	    break;
	case 3:
	    exit(success);
	    break;
	}
	break;
    }
}

static void anim(XtPointer client_data, XtIntervalId *id)
{
    static short int x, prev_x, y, prev_y, r, prev_r;

    XDrawLine(display, window, gc, x, 0, x, height);
    XDrawLine(display, window, gc, 0, y, width, y);
    XDrawCircle(display, window, gc, width/2-r, height/2-r, 2*r, 2*r);

    XSetForeground(display, gc, background);

    XDrawLine(display, window, gc, prev_x, 0, prev_x, height);
    XDrawLine(display, window, gc, 0, prev_y, width, prev_y);
    XDrawCircle(display, window, gc,
		width/2-prev_r, height/2-prev_r, 2*prev_r, 2*prev_r);

    set_color(colorindex);

    draw_mire();

    XSync(display, false);

    prev_x = x; x = (x < width  - 1 ? x + 1 : 0);
    prev_y = y; y = (y < height - 1 ? y + 1 : 0);
    prev_r = r; r = (r < height/2   ? r + 1 : 0);

    XtAppAddTimeOut(app_context, DELAY, anim, null);
}

static void draw_mire(void)
{
    XDrawSegments(display, window, gc, s, nsegments);
    XDrawCircle(display, window, gc, 0, 0, width-1, height-1);
    XDrawCircle(display, window, gc, width/4, height/4, width/2, height/2);
}

static void prev_color(void)
{
    set_color(colorindex = (colorindex == 0 ? colorindexmax : colorindex-1));
}

static void next_color(void)
{
    set_color(colorindex = (colorindex < colorindexmax ? colorindex+1 : 0));
}

static void set_color(unsigned short int colorindex)
{
    XSetForeground(display, gc, colors[colorindex]);
}

static Pixel pixel(char *color_name)
{
    XColor screen_def, exact_def;
    XAllocNamedColor(display, colormap, color_name, &screen_def, &exact_def);
    return screen_def.pixel;
}

static void parse_options(int *argc, char **argv)
{
    int n;
    for (n = 1; n < *argc; ++n) {
	char *arg = argv[n];
	if (arg[0] == '-' || arg[0] == '+' ) {
	    if (STREQU3(arg, "-H", "-h", "--help")) {
		help();
		exit(failure);
	    }
	    else if (STREQU3(arg, "-C", "-c", "--copyright")) {
		printGPL();
		exit(failure);
	    }
	    else if (STREQU3(arg, "-M", "-m", "--man")) {
		system("man xmire");
		exit(failure);
	    }
	    else {
		fprintf(stderr, "%s: invalid option `%s' (%s --help)\n",
			command_name, arg, command_name);
		exit(failure);
	    }
	}
    }
}

static void help(void)
{
    usage();
    fprintf(stderr, "\toptions:\n"
	    "\t\tX toolkit options and\n"
	    "\t\t-n  <n>               number of step\n"
	    "\t\t-anim                 animate\n"
	    "\t\t-H, -h, --help        print this help\n"
	    "\t\t-C, -c, --copyright   print the copyright\n"
	    "\t\t-M, -m, --man         print the manual page\n");
}

static void usage(void)
{
    fprintf(stderr, "%s: usage: %s [<options>]\n",
	    command_name, command_name);
}

static void printGPL(void)
{
    fprintf(stderr,
    "xmire -- Mire under X\n\n"
    "This program is free software; you can redistribute it and/or modify\n"
    "it under the terms of the GNU General Public License as published by\n"
    "the Free Software Foundation; either version 2, or (at your option)\n"
    "any later version.\n\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "GNU General Public License for more details.\n\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software\n"
    "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n");
}
