/* XTHING -- a visualization tool for 2Dimensional
 * representations of 3Dimensional data.
 *
 * Copyright 1991 John L. Cwikla
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appears in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of John L. Cwikla or
 * University of Illinois not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  John L. Cwikla and University of Illinois make no
 * representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 * John L. Cwikla and University of Illinois disclaim all warranties with
 * regard to this software, including all implied warranties of
 * merchantability and fitness, in no event shall John L. Cwikla or
 * University of Illinois be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of
 * use, data or profits, whether in an action of contract, negligence or
 * other tortious action, arising out of or in connection with the use or
 * performance of this software.
 *
 * Author:
 * 	John L. Cwikla
 * 	Materials Research Laboratory Center for Computation
 * 	University Of Illinois at Urbana-Champaign
 *	104 S. Goodwin
 * 	Urbana, IL 61801
 * 
 * 	cwikla@uimrl7.mrl.uiuc.edu
*/

#include <X11/Intrinsic.h> 
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/BulletinB.h>

#include <stdio.h>
#include "local.h"
#include "extern.h"

double log10();

Widget XThing, TopLevel;
PointStruct *PS;
int Xdimension, Ydimension;
int NumColors = 0;
int *IValues;
Display *TheDisplay;
Colormap TheColormap;
int ColorMode = COLOR;
int PlotType = LINEAR;
int TheWindow = OWN;
Drawable TheRootWindow;
int MaxRequests;

static float Min, Max;
static float *Values, *ValuePtr;
/*
static XtResource XThingResources[] = 
{
  {XtNfilename, XtCFilename, XtRString, sizeof(XtRString),
    XtOffset(AppResourcePtr, filename), XtRString, FILENAME},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(XtRInt),
    XtOffset(AppResourcePtr, numColors), XtRInt, NUMCOLORS},
  {XtNcolorMode, XtCColorMode, XtRString
*/

void InitPointStruct()
{
  int i;
#ifdef DDEBUG
printf("InitPointStruct\n");
#endif
  PS = (PointStruct *)XtMalloc(sizeof(PointStruct)*(NumColors)); 
  for(i=0;i<NumColors;i++)
	PS[i].length = 0;

}

void CompressPoints(int x, int y)
{
	int i,j,element;
	int usedColors = 0;
#ifdef DDEBUG
printf("CompressPoints\n");
#endif
	for(i=0;i<NumColors;i++)
	  if (PS[i].length != 0)
	  {
	    PS[i].points = (XPoint *)XtMalloc(sizeof(XPoint)*PS[i].length);
	    PS[i].currentPtr = PS[i].points;
	    usedColors++;
	 }
	
	element = 0;
	for(i=0;i<y;i++)
	  for(j=0;j<x;j++)
	  {
#ifdef DDEBUG
	    printf("%d element, %d IValues[element], %d j %d i\n",
		element, IValues[element], j, i);
#endif
	    PS[IValues[element]].currentPtr->x = j;
	    PS[IValues[element]].currentPtr->y = i;
	    PS[IValues[element]].currentPtr++;
	    element++;
	  }

	UpdateStatus("%d distinct colors.", usedColors);
}  

void Release()
{
	int i;
#ifdef DDEBUG
printf("Release\n");
#endif
	for(i=0;i<NumColors;i++)
	  if (PS[i].length)
	    XtFree(PS[i].points);
	if (Values)
	  XtFree(Values);
	if (IValues)
	  XtFree(IValues);
	PicClose();
	ZoomClose();
	PeriodClose();
	XFlush(TheDisplay);
}

int *ConvertValues(float min, float max, int x, int y, float *Values, int *IValues)
{
  float increment;
  int i, j;
  float *colors;

#ifdef DDEBUG
printf("ConvertValues\n");
#endif
  colors = (float *)XtMalloc(sizeof(float)*NumColors);
  if (PlotType == LINEAR)
  {
    increment = (max - min)/(float)NumColors;
#ifdef DDEBUG
    printf("Max is %f min is %f increment is %f\n", max, min, increment);
#endif
    for(i=0;i<NumColors;i++)
#ifdef DDEBUG
    {
#endif
      colors[i] = min + ((float)i * increment);
#ifdef DDEBUG
      printf("This value at %d is %f\n", i, colors[i]);
    }
#endif
  }
  else 
  if (PlotType == LOG)
  {
    increment = (log10(max) - log10(min))/(float)NumColors;
    for(i=0;i<NumColors;i++)
      colors[i] = log10(min) + ((float)i * increment);
  }

  for(i=0;i<x*y;i++)
  {
    j = 0;
    while( (j<(NumColors-1)) && (Values[i] > colors[j]) )
      j++;
    IValues[i] = j;
#ifdef DDEBUG
    printf("Put %d at IValues[%d]\n", j, i);
#endif
    PS[j].length++;
  }

  XtFree(colors);
  return IValues;
}

short ReadFile(String filename)
{
  FILE *fid;
  int i, j;
  short notFound = FALSE;
#ifdef DDEBUG
printf("ReadFile: %s",filename);
#endif
  if ( (fid = fopen(filename, "r")) == NULL )
  {
    PopUpError("File %s not found!", filename);
    UpdateStatus("File %s not found!", filename);
    return FALSE;
  }

  fscanf(fid,"%d %d", &Xdimension, &Ydimension);

  Values = (float *)XtMalloc(sizeof(float)*Xdimension*Ydimension);
  ValuePtr = Values;
  IValues = (int *)XtMalloc(sizeof(int)*Xdimension*Ydimension);
 
  for(i=0;i<Ydimension;i++)
  {
    for(j=0;j<Xdimension;j++)
    {
      fscanf(fid,"%f", ValuePtr);
	  if ( (i==0) && (j==0))
	  {
	      Min = *ValuePtr;
	      Max = *ValuePtr;
	  }
	  else
	  {
	      Min = MIN(Min, *ValuePtr);
	      Max = MAX(Max, *ValuePtr);
	  }
	  ValuePtr++;
    }
  }

  close(fid);
  UpdateStatus("Image is %d x %d", Xdimension, Ydimension);
  UpdateStatus("Minimum value: %f", Min);
  UpdateStatus("Maximum value: %f", Max);
  return TRUE;
}	

void ArgMessage(char *argv)
{
  if (!(strncmp(argv, "-help", 5)))
    printf("XThing by John L. Cwikla, Materials Research Laboratory Center for Computation.\n");
  else
    printf("XThing -- unsupported option %s.\n", argv);

  printf("Syntax: XThing [options] datafile\n");
  printf("Supported options are:\n");
  printf("-color              - color shades\n");
  printf("-gray               - gray\n");
  printf("-linear             - linear plot\n");
  printf("-log                - log plot\n");
  printf("-numcolors value    - number of colors to use\n");
  printf("-help               - this message\n");
  printf("\n");
  exit(0);
}

void ParseArg(int argc, char *argv[])
{
  int n;
  char filename[255];
#ifdef DDEBUG
printf("ParseArg\nn");
#endif
  if (argc > 1)
  {
    for(n=1;n<argc;n++)
    {
      strcpy(filename, argv[n]);
      if (filename[0] == '-')
      {
        switch(filename[1])
        {
          case 'g':
                if (!strncmp(filename, "-gray", 5))
                  ColorMode = GRAY;
                else
                  ArgMessage(filename);
                break;
          case 'c':
                if (!strncmp(filename, "-color", 6))
                  ColorMode = COLOR;
                else
                  ArgMessage(filename);
                break;
          case 'l':
                if (!strncmp(filename, "-log", 4))
                  PlotType = LOG;
                else
                if (!strncmp(filename, "-linear", 7))
                  PlotType = LINEAR;
                else
                  ArgMessage(filename);
                break;
          case 'n':
                if (!strncmp(filename, "-numcolors", 10))
                {
                  n++;
                  if (n == argc)
                  {
                    printf("Missing numcolor value!\n");
                    ArgMessage(filename);
                    break;
                  }
                  else
                  {
                    NumColors = atoi(argv[n]);
                    continue;
                  }
                }
                else
                  ArgMessage(filename);
                break;
	  case 'r':
		if (!strncmp(filename, "-root", 5))
		  TheWindow = ROOT;
		else
		  ArgMessage(filename);
		break;
          default:
                ArgMessage(filename);
        }
      }
    }
  }
  SetInputFileName(filename);
  SetColorMode(ColorMode);
}

void DoIt()
{
  char *filename;
#ifdef DDEBUG
printf("DoIt\n");
#endif
  CursorWait();
  FillColormap(NumColors, ColorMode);
  BackgroundColorChange();
  InitPointStruct();
  filename = GetInputFileName();
  if (filename[0] != 0)
  {
   
    UpdateStatus("Reading file %s.", filename);
    if (ReadFile(filename))
    {
      if (filename)
        XtFree(filename);
      
      UpdateStatus("Converting values.");
      IValues = ConvertValues(Min, Max, Xdimension, Ydimension, Values, IValues);
      UpdateStatus("Compressing colors.");
      CompressPoints(Xdimension, Ydimension);
      CreatePicShell(TopLevel, Xdimension, Ydimension);
      DrawIt(Xdimension, Ydimension);
      Sensitize();
    }
    else
     Desensitize();
  }
  else
    Desensitize();
  CursorNormal();
}

main(int argc, char *argv[])
{
  Widget bb;
  Arg warg[6];
  char filename[255];
  XtAppContext app;

  XtToolkitInitialize();
  app = XtCreateApplicationContext();
 
  TheDisplay = XtOpenDisplay (app, NULL, "XThing", "XThing", 
	NULL, NULL, &argc, argv);
  if (!TheDisplay)
  {
	XtWarning ("XThing: can't open display, exiting...");
	exit (0);
  }

  TheRootWindow = RootWindow(TheDisplay, DefaultScreen(TheDisplay)); 
  CreateCursors();
  XThing = XtAppCreateShell ("XThing", "XThing",
    applicationShellWidgetClass, TheDisplay, NULL, 0);

  MaxRequests = (XMaxRequestSize(TheDisplay)-3)/2;

#ifdef DDEBUG
printf("Making Controls\n");
#endif
  TopLevel = MakeControls(XThing);
#ifdef DDEBUG
printf("CreateErrorBox\n");
#endif
  CreateErrorBox(TopLevel);
  XtRealizeWidget(XThing);
#ifdef DDEBUG
printf("SettingIcon\n");
#endif
  SetIcon(XThing);

#ifdef DDEBUG
printf("Parsing Args\n");
#endif
  ParseArg(argc, argv);
#ifdef DDEBUG
printf("AssigningColors\n");
#endif
  NumColors = AssignColors(XThing, NumColors);

#ifdef DDEBUG
printf("Updating Status\n");
#endif 
  UpdateStatus("Using %d colors.", NumColors);
#ifdef DDEBUG
printf("DoingIn\n");
#endif
  DoIt();

#ifdef DDEBUG
printf("MainLoop\n");
#endif

  CursorNormal();
  XtAppMainLoop(app);
}
