/*
** XStar by Wayne Schiltt, wayne@cse.unl.edu
** Version 1.1.0  3/22/95
**
** This program now shows the gravitational interaction between a number
** of "stars."
**
**
** Insofar as I have any claim to this program (and I haven't much since it's
** a hack of somebody else's) the GNU General Public License would apply.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**
**
** XStar was based on XGrav 1.0.1 (ca 1/20/95)
** written by David Flater (dave@case50.ncsl.nist.gov)
** The README file for XGrav is in README.xgrav.
**
** XGrav was based on XSwarm 2.3 (ca 1/4/91)
** written by Jeff Butterworth (butterwo@cs.unc.edu)
**
** XSwarm was based on psychoII (date and author unknown)
** and on meltdown (date and author also unknown)
**
** psychoII was possibly based on ico (date and author unknown)
**
** ico was probably a blatant rip-off of some other program :->
*/

/* These are needed for the nap function. */
#include <sys/time.h>
#include <signal.h>

#ifdef VMS
#include "unix_time.h"
#endif

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>

/* This makes it work with virtual root window managers. */
#include "vroot.h"

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <assert.h>
#include <string.h>
#include "patchlevel.h"
#include "xstar.h"

/* Who needs the this header file?? */
#if defined(NEEDS_GETOPT_H)
#include <getopt.h>
#endif

int
main(argc, argv)
int argc;
char **argv;
{
    char        *geometry = NULL;


    Initialize();

    Parse_Arguments(argc, argv, &geometry);

    /* The program acts as a screen saver if timeout is non-zero. */
    if( !timeout )
    {
        Create_Window(geometry);
        Animate();
    }
    else
    {
        Change_Screen_Saver(False);
        while(TRUE)
        {
            Wait_For_Idleness();
            Create_Big_Window();
            Animate();
        }
    }

    return 0;
}


void
Quit( int signal )
{
    if( timeout )
    {
        fprintf(stderr, "Restoring screen saver state.\n");
        Change_Screen_Saver(True);
    }

    fprintf(stderr, "Terminating because of signal %d. ", signal);
    switch( signal )
    {
        case SIGHUP:    fprintf(stderr, "(Hangup)\n"); break;
        case SIGINT:    fprintf(stderr, "(Interrupt)\n"); break;
        case SIGQUIT:   fprintf(stderr, "(Quit)\n"); break;
        case SIGILL:    fprintf(stderr, "(Illegal Instruction)\n"); break;
        case SIGFPE:    fprintf(stderr, "(Floating Point Exception)\n"); break;
        case SIGKILL:   fprintf(stderr, "(Kill)\n"); break;
        case SIGBUS:    fprintf(stderr, "(Bus Error)\n"); break;
        case SIGSEGV:   fprintf(stderr, "(Segmentation Violation)\n"); break;
        case SIGTERM:   fprintf(stderr, "(Termination from Kill)\n"); break;
        case SIGSTOP:   fprintf(stderr, "(Stop not from tty)\n"); break;
        case SIGTSTP:   fprintf(stderr, "(Stop from tty)\n"); break;
#ifdef SIGXCPU
        case SIGXCPU:   fprintf(stderr, "(Exceeded CPU Time Limit)\n"); break;
#endif
        default:        fprintf(stderr, "(Unexpected Signal)\n");
    }

    exit(-signal);
}


void
Initialize()
{
    /* Get the random number generator ready. */
    srand48(time(NULL));

    signal(SIGHUP, Quit);
    signal(SIGINT, Quit);
    signal(SIGQUIT, Quit);
    signal(SIGILL, Quit);
    signal(SIGFPE, Quit);
    signal(SIGKILL, Quit);
    signal(SIGBUS, Quit);
    signal(SIGSEGV, Quit);
    signal(SIGTERM, Quit);
    signal(SIGSTOP, Quit);
    if( timeout ) signal(SIGTSTP, Quit);
#ifdef SIGXCPU
    signal(SIGXCPU, Quit);
#endif
}


/* Parse_Arguments()                                                    */
/* Analyze command line arguments and set variables accordingly.        */

void
Parse_Arguments(argc, argv, geometry)
int     argc;
char    **argv;
char    **geometry;
{
    /* These variables are used to analyze the command line parameters. */
    int         option;
    extern char *optarg;
    double	argf;

    /* Check the command line. */
    while( (option = getopt(argc,argv,"hrt:d:g:D:b:c:C:vRMa:m:") )
            != EOF)
    {
        switch( option )
        {
            case 'r':
                root = TRUE;
                break;
            case 't':
                timeout = atoi(optarg);
                break;
            case 'g':
                *geometry = optarg;
                break;
            case 'd':
                display.dname = optarg;
                break;
            case 'b':
                stars = atoi(optarg);
		max_stars = stars + MAX_COLLAPSAR;
		max_disp_skip = (stars*stars)/2;
                if( stars < 2 )
                  HandleError("You need at least two stars.",
                      FATAL);
                break;
            case 'D':
                delay = atoi(optarg) * 1000; /* convert to microseconds */
                break;
            case 'c':
                star_color = optarg;
                break;
            case 'C':
                bg_color = optarg;
                break;
	    case 'v':
		verbose++;
		break;
	    case 'R':
		rotate_colors++;
		break;
	    case 'M':
		multi_colors++;
		break;
	    case 'a':
		argf = atof(optarg);
		if( argf <= 0 )
		{
		    Usage(*argv);
		    HandleError("-a value must be greater than zero.", FATAL );
		}
		fv *= argf;
		fm /= fv;
		break;
	    case 'm':
		argf = atof(optarg);
		if( argf <= 0 )
		{
		    Usage(*argv);
		    HandleError("-m value must be greater than zero.", FATAL );
		}
		fm *= argf;
		collapsar = fm*COLLAPSAR;
		break;
            case 'h':
                Usage(*argv);
		exit( 0 );
		break;
	    case '?':
                Usage(*argv);
                HandleError("The command line parameters were incorrect.",
                    FATAL);
                break;
        }
    }

    /* The screen saver is incompatible with the */
    /* root window option. */
    if( timeout > 0 )
    {
        if( root )
        {
            printf("The root window option and the screen saver option are\n");
            printf("incompatible.  The screen saver takes precedence, so\n");
            printf("that's what you're getting.\n");
        }
        root = FALSE;
    }

    /* Open the display. */
    if( !(display.dpy = XOpenDisplay(display.dname)) )
    {
        HandleError("Cannot open display.\n", FATAL);
        exit(-1);
    }

    /* Record the screen number and root window. */
    display.screen = DefaultScreen(display.dpy);
    display.root = RootWindow(display.dpy, display.screen);

    /* Set the colors. */
    display.cmap = XDefaultColormap(display.dpy, display.screen);
    if( !display.cmap ) HandleError("There was no default colormap!", FATAL);

    if( star_color == NULL )
        display.star = WhitePixel(display.dpy, display.screen);
    else
        display.star = GetColor(&display, star_color, NULL);

    if( bg_color == NULL )
        display.bg = BlackPixel(display.dpy, display.screen);
    else
        display.bg = GetColor(&display, bg_color, &(display.bg_xcolor));

    /* Set up window size. */
    if( !root && !timeout )
    {
        winW = WINWIDTH;
        winH = WINHEIGHT;
        winX = (DisplayWidth(display.dpy, display.screen)
                - winW) / 2;
        winY = (DisplayHeight(display.dpy, display.screen)
                - winH) / 2;

        if( *geometry != NULL )
	{
	    int result;

            result = XParseGeometry(*geometry, &winX, &winY, &winW, &winH);
	    if( (result & XValue) && (result & XNegative) )
		winX += DisplayWidth(display.dpy, display.screen) - winW;
	    if( (result & YValue) && (result & YNegative) )
		winY += DisplayHeight(display.dpy, display.screen) - winH;
	}
    }
    else
    {
        winX = 0;
        winY = 0;
        if( *geometry != NULL )
            XParseGeometry(*geometry, &winX, &winY, &winW, &winH);
	root_xoff = winX;
	root_yoff = winY;
	
        winW = DisplayWidth(display.dpy, display.screen);
        winH = DisplayHeight(display.dpy, display.screen);
        winX = 0;
        winY = 0;
    }
    far_dist = (winW + winH)*1.5 + 500;
    half_winW = winW/2;
    half_winH = winH/2;
    winW_20 = winW + 20;
    winH_20 = winH + 20;
    


    /* If screen saving is on, then watch for events everywhere. */
    /* Traverse the window tree. */
    if( timeout ) Traverse_Tree(display.dpy, display.root);
}


/* Change_Screen_Saver()                                                        */
/* Turn the server's screen saver on or off.                            */
/* This routine should be called with on=False before it is called      */
/* with on=True.                                                        */

void
Change_Screen_Saver(on)
int     on;     /* True or False */
{
    static int  timeout, interval, blanking, exposures;
    static int  set_yet = FALSE;

    if( on )
    {
        if( set_yet )
        {
            /* Restore the old settings. */
            XSetScreenSaver(display.dpy, timeout, interval, blanking,exposures);
            XActivateScreenSaver(display.dpy);
            XResetScreenSaver(display.dpy);
            XSync(display.dpy, False);
        }
    }
    else
    {
        /* Save the old settings and turn off the server's screen saver. */
        XGetScreenSaver(display.dpy, &timeout, &interval, &blanking,&exposures);
        XSetScreenSaver(display.dpy, 0, 0, DefaultBlanking, DefaultExposures);
        XResetScreenSaver(display.dpy);
        set_yet = TRUE;
    }
}


/* Traverse_Tree()                                                      */
/* Select some events from every single window that is a decendent      */
/* of "current".                                                        */

void
Traverse_Tree(display, current)
Display *display;
Window  current;
{
    Window              my_root, parent, *children;
    unsigned int        num_children;
    int         i;


    /* Watch for signs of life from the user in this window. */
    XSelectInput(display, current, ALIVE_MASK);

    /* Who are my kids? */
    XQueryTree(display, current, &my_root, &parent, &children,
        &num_children);

    /* Examine all of the children too. */
    for( i = 0 ; i < num_children ; i++ )
        Traverse_Tree( display, children[i] );

    /* Let's not waste any memory. */
    if( num_children ) XFree( (char *) children );
}


/* Wait_For_Idleness()                                                  */
/* Wait for "timeout" seconds of user inactivity.                       */

void
Wait_For_Idleness()
{
    int         watching = TRUE;
    int         found;
    int         timer = 0;
    XEvent      event;

    while( watching )
    {
        if( timer >= timeout ) return;

        sleep(1);
        timer++;

        found = XCheckIfEvent(display.dpy, &event, Dummy_Predicate, NULL);
        if( found ) timer = 0;

        /* Flush events. */
        while( found )
        {
            switch( event.type )
            {
                /* Watch for events in new windows too. */
                case CreateNotify:
                    {
                        XCreateWindowEvent *ev = (XCreateWindowEvent *) &event;

                        /* What an insidious bug! */
                        /* I had always assumed that when I received a */
                        /* creation notification that the window in */
                        /* question would still exist.  However, the */
                        /* big screen saver window can sometimes be */
                        /* destroyed by the time we get to this point. */
                        /* We certainly don't want to do anything here */
                        /* with a window that no longer exists. */
                        if( display.win != ev->window )
                        {
                            XSelectInput(display.dpy, ev->window, ALIVE_MASK);
                            Traverse_Tree(display.dpy, ev->window);
                        }
                    }
                    break;
                default:
                    break;
            }
            /* Check for the existence of more events. */
            found = XCheckIfEvent(display.dpy, &event, Dummy_Predicate, NULL);
        }
    }
}


Bool
Dummy_Predicate(display, event, arg)
Display *display;
XEvent  *event;
char    *arg;
{
    return(True);
}


void
Create_Big_Window()
{
    /* Window Attributes */
    unsigned long               valuemask;
    XSetWindowAttributes        xswa;

    /* First time. */
    static int          first = TRUE;

    /* Cursor Stuff */
    static Cursor       cursor;

    /* Turn the cursor the color of the background. (Invisible) */
    if( first )
    {
        /* I don't care which cursor I get. */
        cursor = XCreateFontCursor(display.dpy, 0);

        XRecolorCursor(display.dpy, cursor, &(display.bg_xcolor),
                        &(display.bg_xcolor));
        first = FALSE;
    }

    /* Create a screen sized window. */
    xswa.cursor = cursor;
    xswa.background_pixel = display.bg;
    xswa.override_redirect = True;
    xswa.do_not_propagate_mask = KeyPressMask | KeyReleaseMask |
        ButtonPressMask | ButtonReleaseMask;
    valuemask = CWCursor | CWBackPixel | CWOverrideRedirect | CWDontPropagate;
    display.win = XCreateWindow(display.dpy, display.root,
        0, 0,
        winW, winH, 0,
        DefaultDepth(display.dpy, display.screen),
        InputOutput, DefaultVisual(display.dpy, display.screen),
        valuemask, &xswa);

    XMapWindow(display.dpy, display.win);

    /* Event Mask */
    XSelectInput(display.dpy, display.win, KeyPressMask);

    /* Set up the stars' graphics context. */
    display.star_gc = XCreateGC(display.dpy, display.win, 0, NULL);
    XSetForeground(display.dpy, display.star_gc, display.star);
    XSetBackground(display.dpy, display.star_gc, display.bg);

    /* Set up an erasing graphics context. */
    display.erase_gc = XCreateGC(display.dpy, display.win, 0, NULL);
    XCopyGC(display.dpy, display.star_gc, 0xffffffff, display.erase_gc);
    XSetForeground(display.dpy, display.erase_gc, display.bg);

    /* Clear the background. */
    XFillRectangle(display.dpy, display.win, display.erase_gc,
            0,0, winW, winH);
}



/* Create_Window()                                                      */
/* Create the window, making sure to set it up correctly.               */

void
Create_Window(geometry)
char    *geometry;
{
    XSetWindowAttributes xswa;
    XSizeHints          sizehint;
    XWMHints            wmhints;
    char	        wname[256];     /* Window Name */

    if( !root )
    {
        xswa.event_mask = 0;
        xswa.background_pixel = display.bg;
        display.win = XCreateWindow(display.dpy, display.root,
            winX, winY,
            winW, winH, 0,
            DefaultDepth(display.dpy, display.screen),
            InputOutput, DefaultVisual(display.dpy, display.screen),
            CWEventMask | CWBackPixel , &xswa);

        sizehint.x = winX;
        sizehint.y = winY;
        sizehint.width = winW;
        sizehint.height = winH;
        sizehint.min_width = WINWIDTH;
        sizehint.min_height = WINHEIGHT;
        if( geometry != NULL )
            sizehint.flags = USPosition | USSize | PMinSize;
        else
            sizehint.flags = PPosition | PSize | PMinSize;
        XSetNormalHints(display.dpy, display.win, &sizehint);

        display.protocol_atom = XInternAtom(display.dpy, "WM_PROTOCOLS", False);
        display.kill_atom = XInternAtom(display.dpy, "WM_DELETE_WINDOW", False);
#ifdef X11R4
        XSetWMProtocols(display.dpy, display.win, &display.kill_atom, 1);
#endif

        /* Title */
        if( PATCHLEVEL )
          sprintf( wname, "XStar %s.%d", VERSION, PATCHLEVEL);
        else
          sprintf( wname, "XStar %s", VERSION);
        XChangeProperty(display.dpy, display.win,
            XA_WM_NAME, XA_STRING, 8, PropModeReplace, (unsigned char *) wname,
            strlen(wname));

        /* Window Manager Hints (This is supposed to make input work.) */
        wmhints.flags = InputHint;
        wmhints.input = True;
        XSetWMHints(display.dpy, display.win, &wmhints);

        XMapWindow(display.dpy, display.win);
    }
    else
    {
        display.win = display.root;
        xswa.backing_store = Always;
        XChangeWindowAttributes(display.dpy, display.win,
            CWBackingStore, &xswa);
    }

    /* Event Mask */
    if( root )
        XSelectInput(display.dpy, display.win,
                        KeyPressMask | StructureNotifyMask);
    else
        XSelectInput(display.dpy, display.win,
                        KeyPressMask | ButtonPressMask | StructureNotifyMask);

    /* Set up the stars' graphics context. */
    display.star_gc = XCreateGC(display.dpy, display.win, 0, NULL);
    XSetForeground(display.dpy, display.star_gc, display.star);
    XSetBackground(display.dpy, display.star_gc, display.bg);

    /* Set up an erasing graphics context. */
    display.erase_gc = XCreateGC(display.dpy, display.win, 0, NULL);
    XCopyGC(display.dpy, display.star_gc, 0xffffffff, display.erase_gc);
    XSetForeground(display.dpy, display.erase_gc, display.bg);

    /* Clear the background. */
    XSetWindowBackground(display.dpy, display.win, display.bg);
    XFillRectangle(display.dpy, display.win, display.erase_gc,
      0,0, winW, winH);
}

/* Initialize point positions, velocities, etc. */
void init_stars( point_2d *cur, point_2d *prev, point_2d *v,
		double *m, point_2d *a, point_2d *a_prev,
		XPoint *last_disp,
		int stars, int max_stars, int *min_stars,
		time_t *tstamp, int *live_stars,
		int *live_erase, int *erase_disps,
		XPoint *points, int *points_used )
{
    int size;
    int b;
    int num_collapsar;
    int num_reverse = ( stars >= 4 ) ? (lrand48() % (stars/4)) : 0;

    double tv, tpx, tpy, tx, ty, tdist;

    static time_t	last_init = 0;

    /* update various time stamps */
    *tstamp = time(NULL);
    *live_stars = stars;

    *erase_disps = 0;
    *live_erase = *live_stars;

    if( verbose > 0 && last_init != 0 )
	fprintf( stderr, "%s:%d  total life span of the star system: %d sec\n", __FILE__, __LINE__, *tstamp - last_init );
    last_init = *tstamp;


    /* clean up display */
    if( *points_used > 0 )
	XDrawPoints(display.dpy, display.win, display.star_gc, points,
		    *points_used, CoordModeOrigin );
    XSync(display.dpy, False);
    sleep(1);
    XFillRectangle(display.dpy, display.win, display.erase_gc,
		   0,0, winW, winH);
    XSync(display.dpy, False);
    *points_used = 0;


    /* make sure the array is clean at the end... */
    for( b = 0; b < max_stars; b++ )
    {
	prev[b].x = cur[b].x = 0;
	prev[b].y = cur[b].y = 0;
	v[b].x = v[b].y = 0;
	m[b] = 0;
	a_prev[b].x = a[b].x = 0;
	a_prev[b].y = a[b].y = 0;
	last_disp[b].x = far_dist;
	last_disp[b].y = far_dist;
    }

    /* figure out how many collapsars to create */
    switch( lrand48() % 10 )
    {
    case 0:				/* 40%				*/
    case 1:
    case 2:
    case 3:
	num_collapsar = 0;
	break;
	
    case 4:				/* 10%				*/
	num_collapsar = 1;
	break;
	
    case 5:				/* 20%				*/
    case 6:
	num_collapsar = 2;
	break;
	
    case 7:				/* 20%				*/
    case 8:
	num_collapsar = 3;
	break;
	
    case 9:				/* 10%				*/
	num_collapsar = 4;
	break;
	
    default:
	num_collapsar = 0;
	break;
    }

#if defined(bounce_stars) || defined(bounce_debug)
    /* for easier debugging */
    num_collapsar = 0;
#endif

    if( num_collapsar + stars > max_stars )
	num_collapsar = max_stars - stars;

    if( num_collapsar > 1 )
	*min_stars = 0;
    else
	*min_stars = 1;


    /*
     * Try to put slower, more massive stars toward the center, and give
     * the stars on the outside enough velocity so that they can orbit, but
     * not so much that they fly off.  Also try to make the total momentum
     * of the system start out at zero to keep things from drifting off the
     * screen.
     *
     * the total mass of the system needs to be fairly large to keep
     * things bound together, so we have a fudge factor to make stars
     * more massive if there are fewer of them.
     */

    tpx = tpy = 0;
    size = sqrt((sqrt((double)stars)*2.5 + 5 + num_collapsar/2.75) * STARAREA);

    if( verbose > 0 )
	fprintf( stderr, "%s:%d  num stars=%d  num collapsars=%d  size=%d\n", __FILE__, __LINE__, stars, num_collapsar, size );

    for( b = 0 ; b < stars ; b++ )
    {
	prev[b].x = cur[b].x = (double) (lrand48() % size) + ((double)winW-size)/2.0 + root_xoff;
	prev[b].y = cur[b].y = (double) (lrand48() % size) + ((double)winH-size)/2.0 + root_yoff;

	tx = cur[b].x - half_winW - root_xoff;
	ty = cur[b].y - half_winH - root_yoff;
	tdist = sqrt( tx*tx + ty*ty );
	if( tdist < 70 )
	    tdist = 70;
	else if( tdist > 165 )
	    tdist = 165;
	
	
	v[b].x = ((double) RAND(200) * (num_collapsar*1.25 + 5) * tdist)
	    / (fv*1700000.0 + fv*100000.0*(stars - num_reverse/2));
	v[b].y = ((double) RAND(200) * (num_collapsar*1.25 + 5) * tdist)
	    / (fv*1700000.0 + fv*100000.0*(stars - num_reverse/2));
	if( fabs( v[b].x ) < .01/fv )
	    v[b].x = .01/fv;
	if( fabs( v[b].y ) < .01/fv )
	    v[b].y = .01/fv;

#if defined(bounce_stars) || defined(bounce_debug)
	/* for easier debugging */
	v[b].x = v[b].y = 0;
#endif
	
	
	/* put a counter clockwise spin on things... */
	if( cur[b].x - half_winW - root_xoff > 0 )
	    if( cur[b].y - half_winH - root_yoff > 0 )
		if( cur[b].x - half_winW - root_xoff > cur[b].y - half_winH - root_yoff )
		{ /* NEE */
		    if( fabs( v[b].x ) > fabs( v[b].y ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = fabs( v[b].x ), v[b].y = -fabs( v[b].y );
		}
		else
		{ /* NNE */
		    if( fabs( v[b].y ) > fabs( v[b].x ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = fabs( v[b].x ), v[b].y = -fabs( v[b].y );
		}
	    else
		if( cur[b].x - half_winW - root_xoff > -cur[b].y + half_winH + root_yoff )
		{ /* SEE */
		    if( fabs( v[b].x ) > fabs( v[b].y ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = -fabs( v[b].x ), v[b].y = -fabs( v[b].y );
		}
		else
		{ /* SSE */
		    if( fabs( v[b].y ) > fabs( v[b].x ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = -fabs( v[b].x ), v[b].y = -fabs( v[b].y );
		}
	else
	    if( cur[b].y - half_winH - root_yoff > 0 )
		if( -cur[b].x + half_winW + root_xoff > cur[b].y - half_winH - root_yoff )
		{ /* NWW */
		    if( fabs( v[b].x ) > fabs( v[b].y ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = fabs( v[b].x ), v[b].y = fabs( v[b].y );
		}
		else
		{ /* NNW */
		    if( fabs( v[b].y ) > fabs( v[b].x ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = fabs( v[b].x ), v[b].y = fabs( v[b].y );
		}
	    else
		if( -cur[b].x + half_winW + root_xoff > -cur[b].y + half_winH + root_yoff )
		{ /* SWW */
		    if( fabs( v[b].x ) > fabs( v[b].y ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = -fabs( v[b].x ), v[b].y = fabs( v[b].y );
		}
		else
		{ /* SSW */
		    if( fabs( v[b].y ) > fabs( v[b].x ) )
		    { tv = v[b].x; v[b].x = v[b].y; v[b].y = tv; }
		    v[b].x = -fabs( v[b].x ), v[b].y = fabs( v[b].y );
		}
	
	
	/* give it some mass */
	m[b] = (fm*1400000.)/(stars * (num_collapsar*2+3) * tdist * tdist);
	
	if( m[b] > fm*5 )
	    m[b] = fm*5;
	if( m[b] < fm*.4 )
	    m[b] = fm*.4;
	

	if( verbose > 2 )
	    fprintf( stderr, "%s:%d  star: %d at (%g,%g)=(%g,%g)*%g  tdist=%g\n", __FILE__, __LINE__, b, cur[b].x - half_winW, cur[b].y - half_winH, v[b].x, v[b].y, m[b], tdist );
	
	tpx += v[b].x*m[b];
	tpy += v[b].y*m[b];
    }

    /*
     * scale down the total momentum by the number of bodies
     */
    tpx = tpx / stars;
    tpy = tpy / stars;

    /* even out the total momentum of the system */
    for( b = 0 ; b < stars ; b++ )
    {
	v[b].x -= tpx/m[b];
	v[b].y -= tpy/m[b];
    }


    /*
     * create the collapsars
     *
     * note:  it is assumed that these will be at the end and thus won't
     *        have their velocities reversed.
     */
    for( b = stars; b < stars + num_collapsar; b++ )
    {
	prev[b].x = cur[b].x = (double) (lrand48() % winW) / 3.0 + (double)winW / 3.0 + root_xoff;
	prev[b].y = cur[b].y = (double) (lrand48() % winH) / 3.0 + (double)winH / 3.0 + root_yoff;
	v[b].x = 0.0;
	v[b].y = 0.0;
	m[b] = collapsar * 1.0001;
    }


    /* reverse the direction on a few stars */
    for( b = 0; b < num_reverse; b++ )
    {
	v[b].x = -v[b].x;
	v[b].y = -v[b].y;
    }
}

void init_colors( XColor *star_colors, int num_elem )
{
    int			c;

    long int		c_red, c_green, c_blue;
    long int		inc_red, inc_green, inc_blue;

    long int		inc_cval = (MAX_CVAL / (NUM_COLORS/6));

    char        error_str[STD_STR];

    c_red = MAX_CVAL;
    c_green = 0;
    c_blue = MAX_CVAL;

    inc_red = 0;
    inc_green = 0;
    inc_blue = -inc_cval;

    for( c = 0; c < NUM_COLORS; c++ )
    {
	/* allocate another color */
	star_colors[c].red = c_red;
	star_colors[c].green = c_green;
	star_colors[c].blue = c_blue;
	if( !XAllocColor(display.dpy, display.cmap, &star_colors[ c ]) )
	{
	    sprintf(error_str, "Color %d=(%d,%d,%d) couldn't be allocated.",
		    c, star_colors[c].red, star_colors[c].green,
		    star_colors[c].blue );
	    HandleError(error_str, FATAL);
	}
#if 0
	fprintf( stderr, "%s:%d  color[%d]=(%d,%d,%d)=%08X\n", __FILE__, __LINE__, c, star_colors[c].red, star_colors[c].green, star_colors[c].blue, star_colors[c].pixel );
#endif
	
	/* calc next color value */
	c_red += inc_red;
	c_green += inc_green;
	c_blue += inc_blue;
	
	if( inc_blue < 0 && c_blue <= 0 )
	{
	    inc_green = inc_cval;
	    inc_blue = 0;
	    c_green += inc_green;
	    c_blue = 0;
	}
	else if( inc_blue > 0 && c_blue >= MAX_CVAL )
	{
	    inc_green = -inc_cval;
	    inc_blue = 0;
	    c_green += inc_green;
	    c_blue = MAX_CVAL;
	}
	else if( inc_green < 0 && c_green <= 0 )
	{
	    inc_green = 0;
	    inc_red = inc_cval;
	    c_green = 0;
	    c_red += inc_red;
	}
	else if( inc_green > 0 && c_green >= MAX_CVAL )
	{
	    inc_green = 0;
	    inc_red = -inc_cval;
	    c_green = MAX_CVAL;
	    c_red += inc_red;
	}
	else if( inc_red < 0 && c_red <= 0 )
	{
	    inc_red = 0;
	    inc_blue = inc_cval;
	    c_red = 0;
	    c_blue += inc_blue;
	}
	else if( inc_red > 0 && c_red >= MAX_CVAL )
	{
	    inc_red = 0;
	    inc_blue = -inc_cval;
	    c_red = MAX_CVAL;
	    c_blue += inc_blue;
	}
    }
    init_colors_done = 1;
}



void
Animate()
{
    register int        b, c;              /* star index */
    XPoint		*points;
    XPoint		*tmp_pts;
    int			*pixels;
    int			points_used;
    int			num_disp_skipped;
    int			num_poll_skipped;
    int			max_points;
    point_2d		*cur, *prev;	/* current and previous positions */
    XPoint		*last_disp;	/* last loc star was displayed	*/
    point_2d		*temp_points;
    point_2d		*v;		/* velocities			*/
    double              *m;             /* masses */
    point_2d		*a, *a_prev;	/* accelerations		*/
    point_2d		Dt_a;		/* derivative of the acceleration */
    int			tx, ty;
    XEvent              xev;
    time_t              tstamp;
    int			erase_disps;	/* num displays since last erase*/
    int			live_erase;	/* number of stars when last cleared */
    int			live_stars;
    int			num_visable;
    int			num_seen;
    int			min_stars = 0;
    int			xpt, ypt;
    int			found;
    int			i;
    unsigned int	window_dist;
    double		found_mass, tmass, dist;

    /*
     * most compilers aren't smart enought to know that there can't be
     * aliasing between malloced arrays and so they can't promote these
     * variables into registers
     */
    register double	m_b, m_c;
    double		curb_x, curb_y;
    double		prevb_x, prevb_y;
    double		ab_x, ab_y;

    int			color_number = 0;
    int			color_skip = 0;

    int			paused = FALSE;


    /* set up colors */
    if( rotate_colors || multi_colors)
    {
	init_colors( star_colors, NUM_COLORS );
	display.star = star_colors[color_number].pixel;
	XSetForeground(display.dpy, display.star_gc, display.star);
    }


    /* Allocate memory. */
    max_points = max_stars*10;		/* will rarely be more than 1/4 full */
    points = (XPoint *) Safe_Malloc(sizeof(XPoint) * max_points);
    tmp_pts = (XPoint *) Safe_Malloc(sizeof(XPoint) * max_points);
    pixels = (int *) Safe_Malloc(sizeof(int) * max_points);
    cur = (point_2d *) Safe_Malloc( sizeof( point_2d ) * max_stars );
    prev = (point_2d *) Safe_Malloc( sizeof( point_2d ) * max_stars );
    last_disp = (XPoint *) Safe_Malloc( sizeof( XPoint ) * max_stars );
    v = (point_2d *) Safe_Malloc( sizeof( point_2d ) * max_stars );
    m = (double *) Safe_Malloc(sizeof(double) * max_stars);
    a = (point_2d *) Safe_Malloc(sizeof(point_2d) * max_stars);
    a_prev = (point_2d *) Safe_Malloc(sizeof(point_2d) * max_stars);

    
    /* initialize the system */
    points_used = 0;
    num_disp_skipped = 0;
    num_poll_skipped = 0;
    num_seen = 0;
    init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );


    /*
     * Seemingly endless loop.
     */
    while( TRUE )
    {
	num_visable = 0;
	
	/* Age the the points. */
	temp_points = prev;
	prev = cur;
	cur = temp_points;
	
	/* Age the the points. */
	temp_points = a_prev;
	a_prev = a;
	a = temp_points;
	
#ifdef bounce_stars
	/*
	 * this code doesn't work right.
	 *
	 * - need to put an if around the loop to make it conditional
	 *
	 * - I am not sure that this is the right place for the loop anyway, or
	 *   if it should even be a separate loop.
	 *
	 * - I would like to  have a separate BOUNCE constant that can tweak
	 *   the amount of energy converted to heat.
	 *
	 * - This code doesn't really work for collapsars, and I am not sure
	 *   how they should be handled.
	 *
	 * - should I add code to bounce things harder if they are deeper into
	 *   bounce region?
	 *
	 * - energy does not appear to be conserved.  A bouncing star system
	 *   tends to gain energy over time if there are a lot of sling shot
	 *   effects going on.  A larger FV value might "fix" this...?
	 *
	 * 
	 * Basically, the results of this code doesn't match reality.  If
	 * you start off with a system of stars that have no initial velocity,
	 * then you should see the stars bouncing around forever without
	 * leaving the screen.  This doesn't happen.
	 */
	
	/* bounce stars off of each other */
        for( b = 0 ; b < max_stars ; b++ )
        {
	    m_b = m[b];
            if( m_b <= 0.0 )
		continue;
	
	
            for( c = b+1; c < max_stars; c++ )
	    {
		double dx, dy, n2;
		point_2d tvb, tvc;
		
		m_c = m[c];
		if( m_c <= 0.0 )
		    continue;
		
		dx = prev[c].x - prev[b].x;
		dy = prev[c].y - prev[b].y;
		n2 = dx*dx + dy*dy;
		

		/* this is probably too large of a value... */
		if( n2 < 5*5 )
		{
		    /* collision:  perfectly elastic -> bounce */

		    double n = sqrt(n2);

		    		    
		    /* projected velocities along alternate coordinate sys */
		    point_2d	v1xp, v1yp, v2xp, v2yp;
		    point_2d	xp, yp;

		    /*
		     * figure out new coordinate system
		     *
		     * The x-prime axis is along the line connecting the
		     * two bodies.  The y-prime axis is "flat wall" that the
		     * bodies appear to bounce off of.  The momentum and
		     * velocity along the y-prime axis is not changed by
		     * the collision.
		     */
		    xp.x = dx / n;
		    xp.y = dy / n;

		    yp.x = xp.y;
		    yp.y = -xp.x;

		    /* figure out projections of velocities along new axii */
		    v1yp.x = (yp.x * v[b].x + yp.y * v[b].y) * yp.x;
		    v1yp.y = (yp.x * v[b].x + yp.y * v[b].y) * yp.y;

		    v2yp.x = (yp.x * v[c].x + yp.y * v[c].y) * yp.x;
		    v2yp.y = (yp.x * v[c].x + yp.y * v[c].y) * yp.y;

		    v1xp.x = (xp.x * v[b].x + xp.y * v[b].y) * xp.x;
		    v1xp.y = (xp.x * v[b].x + xp.y * v[b].y) * xp.y;

		    v2xp.x = (xp.x * v[c].x + xp.y * v[c].y) * xp.x;
		    v2xp.y = (xp.x * v[c].x + xp.y * v[c].y) * xp.y;

		    /*
		     * basically, we just need to treat the collision as a
		     * one dimensional collision along the x-prime axis.
		     */
		    tvb.x = v1yp.x + (m_b * v1xp.x - m_c * v1xp.x + 2 * m_c * v2xp.x) / (m_b + m_c);
		    tvb.y = v1yp.y + (m_b * v1xp.y - m_c * v1xp.y + 2 * m_c * v2xp.y) / (m_b + m_c);
		    				                 
		    tvc.x = v2yp.x + (m_c * v2xp.x - m_b * v2xp.x + 2 * m_b * v1xp.x) / (m_b + m_c);
		    tvc.y = v2yp.y + (m_c * v2xp.y - m_b * v2xp.y + 2 * m_b * v1xp.y) / (m_b + m_c);
		    
		    /*
		     * There has got to be a better way to see if the
		     * stars are moving away from each other than
		     * checking the distances like this.  This is way
		     * too expensive to calculate.
		     *
		     * the problem that I am fighting is that once inside the
		     * 5 unit distance, this code always says that the stars
		     * are colliding, even though after the first bounce, they
		     * are moving away from each-other.
		     *
		     * we know that the tvb and tvc vectors are collinear,
		     * and that should make things easier to calculate...
		     */
		    dx = (prev[c].x + tvc.x) - (prev[b].x + tvb.x);
		    dy = (prev[c].y + tvc.y) - (prev[b].y + tvb.y);
		    
		    if( n2 < (dx*dx + dy*dy) )
		    {
#ifdef bounce_debug
			static double	last_mag = 0, cur_mag = 0;
			
			fprintf( stderr, "%s:%d  v1=(%g,%g) %g  v1yp=(%g,%g) v1xp=(%g,%g)  (%g,%g)\n",
				__FILE__, __LINE__,
				v[b].x, v[b].y, sqrt((v[b].x*v[b].x) + (v[b].y*v[b].y)),
				v1yp.x, v1yp.y, v1xp.x, v1xp.y,
				v1xp.x + v1yp.x, v1yp.y + v1xp.y );
#endif
			
			
			v[b] = tvb;
			v[c] = tvc;
#ifdef bounce_debug
			cur_mag = sqrt((v[b].x*v[b].x) + (v[b].y*v[b].y));
			
			fprintf( stderr, "%s:%d  v1=(%g,%g) %g  diff=%.3g%%\n",
				__FILE__, __LINE__,
				v[b].x, v[b].y, cur_mag,
				100.*(last_mag/cur_mag-1.) );
			last_mag = cur_mag;
#endif
			
			/* this code isn't right.  We shouldn't need to move
			 * the body twice...  This just compensates for the
			 * lost energy due to FV being too small.  It doesn't
			 * even do a complete job at that...
			 */

			/* Move */
			cur[b].x = prev[b].x + v[b].x;
			cur[b].y = prev[b].y + v[b].y;

			cur[c].x = prev[c].x + v[c].x;
			cur[c].y = prev[c].y + v[c].y;

			/*
			 * what about the acceleration arrays?  what should
			 * happen to them?
			 */
		    }
#ifdef bounce_debug
		    else
			fprintf( stderr, "%s:%d  ignoring the bounce...\n", __FILE__, __LINE__ );
#endif
		}
	    }

	}
#endif

	/* calculate and display the new locations of each star */
        for( b = 0 ; b < max_stars ; b++ )
        {
	    m_b = m[b];
            if( m_b <= 0.0 )
		continue;
	
	    prevb_x = prev[b].x;
	    prevb_y = prev[b].y;
	    ab_x = a[b].x;
	    ab_y = a[b].y;

            /* solve the n-body problem in time for lunch */
            for( c = b+1; c < max_stars; c++ )
	    {
		double dx, dy, n2, f;
		register double f_d;
		
		m_c = m[c];
		if( m_c <= 0.0 )
		    continue;
		
		dx = prev[c].x - prevb_x;
		dy = prev[c].y - prevb_y;
		n2 = dx*dx + dy*dy;
		

		/* the size of this n2 check is important.  If you
		 * don't try to collapse two nearby stars together
		 * soon enough, they reach very high spin velocities.
		 * Then, the slightest disturbance can cause them to
		 * step over and past each other in a single movement.
		 * In reality they would have collided.  After this
		 * last step, they are now far enough apart and have
		 * such high velocities, that their attraction can no
		 * longer keep them together and they fly off at very
		 * high speeds.
		 *
		 * The smaller the FV value, the larger this value needs
		 * to be, because movement is coarser and each step
		 * can be larger.
		 *
		 * Be aware that besides this overstep artifact caused
		 * by discrete movement, there is also the sling shot
		 * effect which is quite real.  The sling shot effect is
		 * actually used for interplanetary probes and such.
		 * It works by transferring some of the energy from a
		 * larger body (planet, star) to the smaller body
		 * (probe), allowing it to gain a fair amount of
		 * speed.  You can usually tell the difference between
		 * the results of the two phenomenon, because the overstep
		 * artifact shoots stars off at a very high speed, and
		 * the sling shot effect only shoots stars off at a high
		 * speed.
		 *
		 * Since collapsars don't move, you will never see the
		 * sling shot effect from a collapsar.  Because collapsars
		 * are heavy, you will often see the overshoot artifact
		 * from them.
  		 */

		if( n2 < 1.2*1.2 )
		{
		    /* collision:  totally inelastic -> combine */
		    double pxv, pyv, tm;
		
		    pxv = v[b].x*m_b + v[c].x*m_c;
		    pyv = v[b].y*m_b + v[c].y*m_c;
		    tm = BUMPERS / (m_b + m_c);
		
		    v[b].x = pxv * tm;
		    v[b].y = pyv * tm;
		
		    /* don't create a new collapsar */
		    if( m_b >= collapsar || m_c >= collapsar )
			m_b += m_c;
		    else
		    {
			tmass = 1/(m_b + m_c);

			/* the new location of the combined stars */
			prev[b].x = prevb_x = (m_b*prevb_x + m_c*prev[c].x)*tmass;
			prev[b].y = prevb_y = (m_b*prevb_y + m_c*prev[c].y)*tmass;


			a[b].x = ab_x = (m_b*ab_x + m_c*a[c].x)*tmass;
			a[b].y = ab_y = (m_b*ab_y + m_c*a[c].y)*tmass;
			a_prev[b].x = (m_b*a_prev[b].x+m_c*a_prev[c].x)*tmass;
			a_prev[b].y = (m_b*a_prev[b].y+m_c*a_prev[c].y)*tmass;

			m_b += m_c;
			if( m_b >= collapsar )
			    m_b = collapsar * .999;
		    }
		
		    live_stars--;
		    tstamp = time(NULL);
		    if( verbose > 1 )
			fprintf( stderr, "%s:%d  live_stars=%d  (%g)+(%g)=(%g)\n", __FILE__, __LINE__, live_stars, m[b], m_c, m_b );
		
		    m[b] = m_b;
		    m[c] = m_c = 0.0;
		    cur[c].x = cur[c].y = prev[c].x = prev[c].y = -1.0;
		    v[c].x = v[c].y = 0.0;
		    a[c].x = a[c].y = 0.0;
		    a_prev[c].x = a_prev[c].y = 0.0;
		}
		else
		{
		    f = G / (sqrt(n2) * n2);
		
		    f_d = dx * f;		
		    ab_x += f_d * m_c;
		    a[c].x -= f_d * m_b;
		
		    f_d = dy * f;
		    ab_y += f_d * m_c;
		    a[c].y -= f_d * m_b;
		}
            }

            /* I think it looks better if the collapsars don't move ;-) */
            /* immobile collapsar = gravity well */

            if( m_b >= collapsar )
	    {
		v[b].x = v[b].y = 0.0;
		a[b].x = a[b].y = 0.0;
		ab_x = ab_y = 0.0;
		a_prev[b].x = a_prev[b].y = 0.0;
	    }
	
            /*
	     * Move the star
	     *
	     * You want each step to be as small as possible and this is done
	     * by making FV as large as possible.  If the steps are small
	     * enough, then we are approximating infinitesimal movements.
	     *
	     * The movement is based off of the formula:  (with t=1)
	     * x(t) = x0 + v t + 1/2 a t^2 + 1/(2*3) (Dt a) t^3
	     * v(t) = v0 + a t + 1/2 (Dt a) t^2
	     *
	     * In theory, these formulas should include an infinite number
	     * of derivatives of x(t), but the additional terms are not
	     * easy to calculate.
	     */
	    Dt_a.x = ab_x - a_prev[b].x;
	    Dt_a.y = ab_y - a_prev[b].y;
	    
            cur[b].x = curb_x = prevb_x + (v[b].x + ((1./2)*ab_x + (1./(2*3))*Dt_a.x));
            cur[b].y = curb_y = prevb_y + (v[b].y + ((1./2)*ab_y + (1./(2*3))*Dt_a.y));

	    v[b].x += ab_x + (1./2)*Dt_a.x;
	    v[b].y += ab_y + (1./2)*Dt_a.y;

	    
	    a[b].x = ab_x;
	    a[b].y = ab_y;
	    a_prev[b].x = a_prev[b].y = 0;
	
	    /* see if the star has left the system */
	    xpt = (int)curb_x;
	    ypt = (int)curb_y;
	    tx = xpt - half_winW;
	    ty = ypt - half_winH;
	    window_dist = abs(tx) + abs(ty);
	    if( window_dist > far_dist )
	    {
		live_stars--;
		if( verbose > 1 )
		    fprintf( stderr, "%s:%d  live_stars=%d  |(%g,%g)|=%g\n", __FILE__, __LINE__, live_stars, curb_x, curb_y, tx*tx+ty*ty );
		
                m[b] = m_b = 0.0;
                cur[b].x = cur[b].y = prev[b].x = prev[b].y = -1.0;
                v[b].x = v[b].y = 0.0;
	    }
	
	    /* if the star is on the screen, then plot it */
	    if( window_dist <= winW )
	    {
		/* mark the point where the body is at */
		
		if( xpt > -20 && ypt > -20 && xpt < winW_20 && ypt < winH_20 )
		{
		    num_visable++;
		
		    if( m_b < collapsar )
			num_seen++;
		
		    if( xpt != last_disp[b].x || ypt != last_disp[b].y
		       || (m_b >= collapsar &&
			   (num_poll_skipped & (POLL_SKIP/2 - 1)) == 0)
		       )
		    {
			last_disp[b].x = xpt;
			last_disp[b].y = ypt;
			
			if (multi_colors)
			    pixels[points_used] = ((b * NUM_COLORS)/max_stars) % NUM_COLORS;

			if( m_b >= collapsar )
			{
			    points[points_used].x = xpt + lrand48()%3 - 1;
			    points[points_used++].y = ypt + lrand48()%3 - 1;
			}
			else
			{
			    points[points_used].x = xpt;
			    points[points_used++].y = ypt;
			}
		    }
		}
	    }
	
        }
	
	
	/* if there are not enough stars to be interesting, then start over */
	if( live_stars <= min_stars )
	{
	    init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );
	    continue;
	}
	
        /* Display stars
	 *
	 * some of the terms in these formulas should really be adjustable
	 * depending on the speed of the computer and the quality of the
	 * X server and compiler.  max_disp_skip in particular.
	 */
	num_disp_skipped++;
	if( ( points_used > num_visable		/* long time between disps */
	     && num_disp_skipped > max_disp_skip - (live_stars*live_stars))
	   || points_used >= 10*num_visable	/* jerky lines */
	   || points_used + stars + MAX_COLLAPSAR >= max_points	/* buf full */
	   )
	{
	    erase_disps++;
	
	    if( multi_colors )
	    {
		int num_pts, pp, next_color;

 		for( i = 0; i < NUM_COLORS; i = next_color )
		{
		    next_color = NUM_COLORS;
		    
		    num_pts = 0;
		    for (pp = 0; pp < points_used; pp++)
		    {
			if( i == pixels[pp] )
			    tmp_pts[num_pts++] = points[pp];
			else if( i < pixels[pp] && pixels[pp] < next_color )
			    next_color = pixels[pp];
		    }
		    
		    if( num_pts )
		    {
			XSetForeground(display.dpy, display.star_gc,
				       star_colors[i].pixel );

			XDrawPoints(display.dpy, display.win, display.star_gc,
				    tmp_pts, num_pts, CoordModeOrigin );
		    }
		}
	    }
	    else
		XDrawPoints(display.dpy, display.win, display.star_gc, points,
			points_used, CoordModeOrigin );
	
	    /* if we're not generating much data, then force it out quickly */
	    if( num_disp_skipped > 5 + max_disp_skip-(live_stars*live_stars) )
		XSync(display.dpy, False);
	
	    points_used = 0;
	    num_disp_skipped = 0;
	}	
	
        /* Check for events. */
	num_poll_skipped++;
        if( num_poll_skipped >= POLL_SKIP )
        {
	    num_poll_skipped = 0;
	
	    if( XPending(display.dpy) )
	    {
		XNextEvent(display.dpy, &xev);
		HandleEvent(&xev);
	    }
	    else
	    {
		/* Delay so we don't use all of the cpu time. */
#ifdef HP_UX
		if( delay != 0 ) usleep(delay);
#else
		if( delay != 0 ) nap(0,delay);
#endif
#ifdef HP_UX
		if( paused ) usleep(1000*1000);
#else
		if( paused ) nap(1,0);
#endif
	    }
	
	
	    /* change the color of the stars */
 	    if( rotate_colors && color_skip++ > 80000/(POLL_SKIP*NUM_COLORS) )
	    {
		color_skip = 0;
		color_number = (color_number + 1) % NUM_COLORS;
		display.star = star_colors[color_number].pixel;
		XSetForeground(display.dpy, display.star_gc, display.star);
	    }
	
	
	    /* Screen saver/background:  restart after 5 minutes
	     * with out a star dieing */
	    if( timeout || root )
	    {
		time_t	cur_time = time(NULL);
		
		if( cur_time - tstamp > 420 )
		    init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );
		
		if( live_erase < live_stars )
		    live_erase = live_stars;
		
		/* this formula does not respond well to changes in FV */
		if( erase_disps > (rotate_colors ? 4300/fv : 2000/fv)
		   - (1100/fv)*(live_erase-live_stars)*(live_erase-live_stars)/(live_stars+2) )
		    erase = TRUE;
	    }
	
	
	    /* if we can't see any stars, then we should start over */
	    if( num_seen == 0 )
	    {
		if( verbose > 1 )
		    fprintf( stderr, "%s:%d  no stars are visable...\n", __FILE__, __LINE__ );
		
		init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );
	    }
	    num_seen = 0;
	
	
	    /* Add a collapsar or a star */
	    if( add || add_star )
	    {
		for( i = 0, found = -1; i < max_stars; i++ )
		    if( m[i] <= 0 )
		    {
			found = i;
			break;
		    }
		
		if( found == -1 || stars + MAX_COLLAPSAR >= max_stars )
		{
		    max_stars++;
		    if( max_points < max_stars*10 )
			max_points = max_stars*10;
		    points = (XPoint *) Safe_Realloc(points, sizeof(XPoint) * max_points );
		    tmp_pts = (XPoint *) Safe_Realloc(tmp_pts, sizeof(XPoint) * max_points );
		    pixels = (int *) Safe_Realloc(pixels, sizeof(int) * max_points);
		    cur = (point_2d *) Safe_Realloc(cur, sizeof(point_2d) * max_stars);
		    prev = (point_2d *) Safe_Realloc(prev, sizeof(point_2d) * max_stars);
		    last_disp = (XPoint *) Safe_Realloc( last_disp, sizeof(XPoint) * max_stars );
		    v = (point_2d *) Safe_Realloc(v, sizeof(point_2d) * max_stars);
		    m = (double *) Safe_Realloc(m, sizeof(double) * max_stars);
		    a = (point_2d *) Safe_Realloc(a, sizeof(point_2d) * max_stars);
		    a_prev = (point_2d *) Safe_Realloc(a_prev, sizeof(point_2d) * max_stars);

		    found = max_stars - 1;
		}
		
		prev[found].x = cur[found].x = (double) (lrand48() % winW) / 3.0 + (double)winW / 3.0 + root_xoff;
		prev[found].y = cur[found].y = (double) (lrand48() % winH) / 3.0 + (double)winH / 3.0 + root_yoff;
		a_prev[found].x = a[found].x = 0;
		a_prev[found].y = a[found].y = 0;
		last_disp[found].x = far_dist;
		last_disp[found].y = far_dist;

		
		if( add )
		{
		    v[found].x = 0.0;
		    v[found].y = 0.0;
		    m[found] = collapsar * 1.0001;
		}
		else
		{
		    v[found].x = ((double) RAND(100)) / (fv*2000.0);
		    v[found].y = ((double) RAND(100)) / (fv*2000.0);
		    m[found] = fm*3.0;
		    stars++;
		    live_stars++;
		    tstamp = time(NULL);
		}
		
		add = FALSE;
		add_star = FALSE;
		
	    }
	
	    /* delete a star */
	    if( del_star )
	    {
		del_star = FALSE;
		
		/* kill the least massive one */
		found_mass = collapsar*10;
		for( i = 0, found = -1; i < max_stars; i++ )
		{
		    dist = sqrt(cur[i].x*cur[i].x + cur[i].y*cur[i].y);
		    if( dist < 50 )
			dist = 50;

		    tmass = m[i] / dist;
		
		    if( m[i] > 0 && tmass < found_mass )
		    {
			found = i;
			found_mass = tmass;
		    }
		}
		
		if( found != -1 && stars > 2 )
		{
		    stars--;
		    live_stars--;
		    tstamp = time(NULL);
		    if( verbose > 1 )
			fprintf( stderr, "%s:%d  live_stars=%d  m[%d]=%g\n", __FILE__, __LINE__, live_stars, found, m[found] );
		
		    m[found] = 0.0;
		    cur[found].x = cur[found].y = prev[found].x = prev[found].y = -1.0;
		    v[found].x = v[found].y = 0.0;
		
		    if( live_stars <= min_stars )
			init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );
		}
	    }
	
	
	    /* Erase trails */
	    if( erase )
	    {
		if( points_used > 0 )
		    XDrawPoints(display.dpy, display.win, display.star_gc, points,
				points_used, CoordModeOrigin );
		XSync(display.dpy, False);
		points_used = 0;

		erase = FALSE;
		XFillRectangle(display.dpy, display.win, display.erase_gc,
			       0,0, winW, winH);
		
		erase_disps = 0;
		live_erase = live_stars;
	    }
	
	
	    /* new start trails */
	    if( new_start )
	    {
		new_start = FALSE;
		init_stars( prev, cur, v, m, a, a_prev, last_disp, stars, max_stars, &min_stars, &tstamp, &live_stars, &live_erase, &erase_disps, points, &points_used );
	    }
	
	
	    /* stop updating */
	    if( pause_updt )
	    {
		pause_updt = FALSE;
		paused = !paused;
	    }
	
	
	    /* Clean up and shut down. */
	    if( stop )
	    {
		stop = FALSE; /* reset the "stop" variable */
		
		XFillRectangle(display.dpy, display.win, display.erase_gc,
			       0,0, winW, winH);
		XSync(display.dpy, 0);
		
		if( !root )
		    XDestroyWindow(display.dpy, display.win);
		
		/* Free Memory */
		free(points);
		free(tmp_pts);
		free(pixels);
		free(cur);
		free(prev);
		free(last_disp);
		free(v);
		free(m);
		free(a);
		free(a_prev);
		
		/* Free the graphics contexts. */
		XFreeGC(display.dpy, display.star_gc);
		XFreeGC(display.dpy, display.erase_gc);
		
		return;
	    }
        }
	
    }
}


/*
** HandleEvent()
**
** process X events
*/

void
HandleEvent(event)
XEvent  *event;
{
    /* If the screen saver is on, then watch for signs of activity. */
    if( ((event->type == KeyPress) || (event->type == MotionNotify))
       && (timeout)
       )
	stop = TRUE;

    switch( event->type )
    {
        case ClientMessage: /* sent by f.delete from twm */
            {
                XClientMessageEvent     *ev = (XClientMessageEvent *) event;

                printf("Client message received.\n");
                if( ev->message_type == display.protocol_atom
		   && ev->data.l[0] == display.kill_atom )
                    stop = TRUE;
            }
            break;
        case ConfigureNotify:
            {
                XConfigureEvent *ev = (XConfigureEvent *) event;

                /* The screen saver should ignore resizes. */
                if( !timeout )
                {
                    winW = ev->width;
                    winH = ev->height;
		    far_dist = (winW + winH)*1.5 + 500;
		    half_winW = winW/2;
		    half_winH = winH/2;
		    winW_20 = winW + 20;
		    winH_20 = winH + 20;
                    winX = ev->x;
                    winY = ev->y;
                }
            };
            break;
        case KeyPress:
            {
                XKeyEvent *key_event = (XKeyEvent *) event;
                char buf[128];
                KeySym ks;
                XComposeStatus status;

                XLookupString(key_event,buf,128,&ks,&status);
                if( buf[0]=='q' || buf[0]=='Q' )
                    stop = TRUE;
                if( buf[0]=='d' || buf[0]=='D' )
                    add = TRUE;
                if( buf[0]=='+' )
                    add_star = TRUE;
                if( buf[0]=='-' )
                    del_star = TRUE;
                if( buf[0]=='e' || buf[0]=='E' )
                    erase = TRUE;
                if( buf[0]=='n' || buf[0]=='N' )
                    new_start = TRUE;
                if( buf[0]=='p' || buf[0]=='P' )
		    pause_updt = TRUE;
		if( buf[0]=='m' || buf[0]=='M' ) {
		    multi_colors = !multi_colors;
		    if (rotate_colors && multi_colors) {
			rotate_colors = 0;
		    }
		    if (init_colors_done == 0)
			init_colors(star_colors, NUM_COLORS);
		}
		if( buf[0]=='r' || buf[0]=='R' ) {
		    rotate_colors = !rotate_colors;
		    if (rotate_colors && multi_colors) {
			multi_colors = 0;
		    }
		    if (init_colors_done == 0)
			init_colors(star_colors, NUM_COLORS);
		}
            }
            break;
        default:
            break;
    }
}


#ifndef HP_UX

/*  Put the process to sleep for a while.  */
void nap(sec,usec)
long sec, usec;
{
    static struct timeval tv;

    tv.tv_sec = sec;
    tv.tv_usec = usec;
#ifndef VMS
    select(0, 0, 0, 0, &tv);
#else
    seconds = ((float)tv.tv_sec) +((float)tv.tv_usec)/1000000.0;
    statvms = LIB$WAIT(&seconds);
#endif
}

#else

static void alarmhandler()
{
}

sleepms(msec)
int msec;
{
    struct itimerval value,ovalue;
    struct sigvec vec;
    long savemask, sigblock(), sigpause();

    vec.sv_handler = alarmhandler;
    vec.sv_mask = 0x0;
    vec.sv_flags = 0;
    sigvector(SIGALRM, &vec, &vec); /* Set up alarmhandler for SIGALRM */
    savemask = sigblock((long)(1L << (SIGALRM - 1)));

    value.it_interval.tv_sec = 0;
    value.it_interval.tv_usec = 0;
    value.it_value.tv_sec = msec/1000;
    value.it_value.tv_usec = (msec%1000)*1000;
    setitimer(ITIMER_REAL,&value,&ovalue);

    (void)sigpause(0L);
    (void)sigsetmask(savemask);

    sigvector(SIGALRM, &vec, NULL); /* Restore previous signal handler */
}

usleep(us)
long us;
{
    sleepms(us / 1000);
}

#endif


void
Usage(program)
char *program;
{
    printf("%s [options]  where options are listed below\n", program);
    printf("-h                display this message\n");
    printf("-r                use root window\n");
    printf("-g geometry       window geometry\n");
    printf("-d host:display   X server to connect to\n");
    printf("-t timeout        screen saved after 'timeout' seconds\n");
    printf("-D delay          delay between updates (milliseconds)\n");
    printf("-b stars          number of stars\n");
    printf("-c star_clr       star color\n");
    printf("-C bg_color       background color\n");
    printf("-R                Rotate the star colors  (Rainbow)\n");
    printf("-M                multiple colors, one per star\n");
    printf("-a float          accuracy of possition calculations.   Larger values\n" );
    printf("                  increase accuracy.  Must be greater than zero.\n" );
    printf("-m float          mass of star system.  Larger values cause the stars\n" );
    printf("                  to fall to the center.  Must be greater than zero.\n" );
    printf("-v                enable display of debug output\n");
    printf("\nPress d in the window to add a gravity well.\n");
    printf("Press e in the window to erase trails.\n");
    printf("Press + in the window to add a star to the system.\n");
    printf("Press - in the window to delete a star from the system.\n");
    printf("Press n in the window to get a new set of stars.\n");
    printf("Press m in the window to toggle multi-color mode.\n");
    printf("Press r in the window to toggle rainbow mode.\n");
    printf("Press p in the window to pause the updating.  Press p again to start.\n" );
    printf("Press q in the window to quit.\n\n");
}


void
HandleError(description, degree)
char    *description;
int     degree;
{
    fprintf(stderr, "An error has occurred.  The description is below...\n");
    fprintf(stderr, "%s\n", description);

    if( degree == FATAL )
    {
        fprintf(stderr, "Program aborting...\n");
        exit(-1);
    }
}

long
GetColor(display, color, final_color)
disp            *display;
char            *color;
XColor          *final_color;
{
    XColor      cdef;
    char        error_str[STD_STR];

    if( !XParseColor(display->dpy, display->cmap, color, &cdef) ||
        !XAllocColor(display->dpy, display->cmap, &cdef) )
    {
        sprintf(error_str, "Color \"%s\" wasn't found.", color);
        HandleError(error_str, FATAL);
    }

    /* Copy the final color. */
    if( final_color != NULL ) *final_color = cdef;

    return(cdef.pixel);
}


/* Check the return code of malloc(). */
void *
Safe_Malloc(bytes)
int bytes;
{
    void *pointer;

    pointer = (void *) malloc(bytes);
    if( NULL == pointer )
    {
        fprintf(stderr, "Error allocating %d bytes.\n", bytes);
        exit(-1);
    }

    return(pointer);
}

void *
Safe_Realloc(ptr, bytes)
void *ptr;
int bytes;
{
    void *pointer;

    pointer = (void *) realloc(ptr, bytes);
    if( NULL == pointer )
    {
        fprintf(stderr, "Error allocating %d bytes.\n", bytes);
        exit(-1);
    }

    return(pointer);
}
