#!/usr/xpg4/bin/sh
####################################################################################
## FILENAME:    OA-setup.sh
## VERSION:     08.60.030
## DESCRIPTION: Solaris OMU8 server side  installation script for OpenVMS
##              Operation Agent
##              Script can be used for 
##              1. Pre-requisite check for server version, Disk Space for 
##              OpenVMS Operation Agent (8.6)
##              2. (De-)Installation of OpenVMS Operation Agent on OMU8 server.
## USAGE:       OA-setup.sh [-h][-c|i][-p IA|ALPHA]
## LANGUAGE:    sh
## Author:      asim.mandal@hp.com
# (c) Copyright 2010, Hewlett-Packard Development Company, L.P. all rights reserved.
####################################################################################
#set -x
####################################################################################

OMSVR_INST_DIR=/opt/OV/bin/OpC/agtinstall
OMSVR_INSTL_DIR=/opt/OV/bin/OpC/install
OMSVR_CONF_DIR=/opt/OV/bin/OpC/
OMSVR_VEND_DIR=/var/opt/OV/share/databases/OpC/mgd_node/vendor
OMSVR_CUST_DIR=/var/opt/OV/share/databases/OpC/mgd_node/customer
OMSVR_AGT_INST_DIR=/opt/OV/bin/OpC/agtinstall
OMSRV_VMS_OA_BACK_ROOT=/var/opt/OV/packages/ovms

INST_LOG_FILE="/var/opt/OV/log/ovms/vms_inst.log"
INST_VLOG_FILE="/var/opt/OV/log/ovms/vms_inst.verbose"
INST_LOCK_FILE="/var/tmp/vms_inst.lock"

PLAT_REG_FILE=AgentPlatform
SRV_BASE="A.08.10.160"
SRV_COMM="RPC_BBC"
SRC_DIR=" "
TRG_DIR=" "

VERBOSE=0
REMOVE=0
VERIFY=0
INSTALL=0
BACKUP=0
ROLLBACK=0
CHK_PLAT=0
VMS_VER=8_60
IN_PLAT="X"
ALPHA_SELECT=0
IA_SELECT=0


SRV_PKG_SIZE=50
PKG_SIZE_1=125000
PKG_SIZE_2=250000
PKG_SIZE_4=500000
MIN_OM_VER=A.08.30
MIN_OA_VER=A.08.60

#------------------------------------------------------------------------------
#  DESCRIPTION
#    Tracing and logging Function.
#
#  PARAMETERS
#    Message.
#
#  EXIT CODE
#     Return from Message Logging ststus
#------------------------------------------------------------------------------
Log_msg()
{
   MSG="$*"
   if [ ${VERBOSE} -eq 1 ]
   then
   echo "${MSG}" >> ${INST_VLOG_FILE} 2>&1
   fi
   echo "${MSG}" >> ${INST_LOG_FILE} 2>&1
   echo "${MSG}"
   unset MSG
   return 0
}

VLog_msg()
{
   if [ ${VERBOSE} -eq 1 ]
   then
      echo "$*" >> ${INST_VLOG_FILE} 2>&1
      echo "$*"
   fi 
   return 0
}
#------------------------------------------------------------------------------
#
#  Interlock multiple executions of install.ksh.
#
#------------------------------------------------------------------------------
check_lockfile()
{
   if [ -f ${INST_LOCK_FILE} ]
   then
      Log_msg " An instance of the OMU OpenVMS 8.6 OA  Maintenance script is"
      Log_msg " already running or has been terminated abnormally."
      Log_msg " If you are sure no other instance is running,"
      Log_msg " the lock file ${INST_LOCK_FILE} may be removed now."
      return 1
   else
      touch ${INST_LOCK_FILE}
      return 0
   fi
}
#------------------------------------------------------------------------------
remove_lockfile()
{
   rm -f $INST_LOCK_FILE
}

#------------------------------------------------------------------------------
get_dce_version()
{
  VERSIONS=`find ${OMSVR_VEND_DIR}/${SERIES}/ -follow -name '*_version'`
  for version in $VERSIONS
  do
        file=$version
        version=${version##*/}
        case "$version" in
             "opc_version" ) if [ -z "$OPC_VER" ]
                             then
                                 OPC_VER="    $version:  `cat $file`"
                             fi;;
             "comm_version" ) if [ -z "$COMM_VER" ]
                             then
                                 COMM_VER="    $version: `cat $file`"
                             fi;;
             "perf_version" ) if [ -z "$PERF_VER" ]
                             then
                                 PERF_VER="    $version: `cat $file`"
                             fi;;

        esac
  done
 
  if [ -z "$OPC_VER" ]
  then
      OPC_VER="    DCE OVO Agent Version:  $SRV_BASE"
  fi
  Log_msg "$OPC_VER"
 
  if [ -n "$COMM_VER" ]
  then
       Log_msg "$COMM_VER"
  fi

  if [ -n "$PERF_VER" ]
  then
       Log_msg "$PERF_VER"
  fi
}
#------------------------------------------------------------------------------
get_HTTPS_version()
{
  if [ -n "$SRV_BASE" ]
  then
     Log_msg "    Agent base version :  $SRV_BASE"
  fi
  Log_msg "----------------------------------------------------"
  XML_FILES=`find ${OMSVR_VEND_DIR}/${SERIES}/${SRV_BASE}/${SRV_COMM} -name 'HPOv*.xml'`
  for xml_file in $XML_FILES
  do
     component_name=`grep "@(#)FileN" $xml_file | cut -d":" -f2 | cut -d"." -f1` 2>>/dev/null
     version=`grep "@(#)Ver" $xml_file | cut -d":" -f2` 2>>/dev/null
     if [ -n "$component_name" ] && [ -n "$version" ]
     then
        Log_msg "    $component_name:  $version"
     fi
  done
  Log_msg "----------------------------------------------------"
}

#------------------------------------------------------------------------------
get_free_ds()
{
   df -k $1 \
         | /usr/bin/awk 'BEGIN { s = 0 } NF > 1 && NF < 6 { s = $3 } \
                         NF == 6 { s = $4 } END { printf "%d\n", s }'
}
#------------------------------------------------------------------------------
display_details()
{
#set -x
   SERIES=$1
   REQ_F=${OMSVR_VEND_DIR}/${SERIES}/require.dat
   
   if [ -f ${REQ_F} ]
   then
      VLog_msg  "  For Managed Node type: ${SERIES}"
      VLog_msg  "  ----------------------"
      VLog_msg  "  Base Version      Supported OS Version(s)"
      VLog_msg   "  ${SERIES}:"
      awk '{ \
             if ($1 != "#" && $1 != "#@" && NF >2) \
             { \
               printf "  %s          ", $1; \
               for (i=3; i<NF; i++) {printf "%s, ", $i}; \
               printf "%s\n", $(NF) \
             } \
          }' ${REQ_F}
      SRV_BASE=`awk '{ \
                          if ($1 != "#" && $1 != "#@" && NF >2) \
                           {\
                                printf "%s", $1;\
                           }\
                       }' ${REQ_F}`

      VER_NUMBER=`echo $SRV_BASE | cut -d"." -f2` 2>>/dev/null
      case "$VER_NUMBER" in
             "07" )
                 get_dce_version
                 Log_msg "-----------------------------------------------------------------------"
                 Log_msg " (OpenVMS OA 8.6 will not change this installation.)"  
                 Log_msg " (Upgradation from DCE Agent to OA 8.6 is not supported for OpenVMS.)"  
                 Log_msg " (Nothing will be removed from ${OMSVR_VEND_DIR}/${SERIES}.)"  
                 Log_msg "-----------------------------------------------------------------------"
                 ;;
                * )
                 get_HTTPS_version
                 Log_msg "------------------------------------------------------"
                 Log_msg "Upgrade to OpenVMS OA 8.6 on OM server not possible."
                 Log_msg "Please specify -b option with OA-setup.sh for backup existing version."
                 ;;
      esac

   else
      Log_msg ""
      Log_msg "(Operation Agent is not properly installed for $2.)"
   fi

   return 0
}
#------------------------------------------------------------------------------
verify_details()
{
#set -x
   SERIES=$1
   REQ_F=${OMSVR_VEND_DIR}/${SERIES}/require.dat
   
   if [ -f ${REQ_F} ]
   then
      VLog_msg  "  For Managed Node type: ${SERIES}"
      VLog_msg  "  ----------------------"
      VLog_msg  "  Base Version      Supported OS Version(s)"
      VLog_msg   "  ${SERIES}:"
      awk '{ \
             if ($1 != "#" && $1 != "#@" && NF >2) \
             { \
               printf "  %s          ", $1; \
               for (i=3; i<NF; i++) {printf "%s, ", $i}; \
               printf "%s\n", $(NF) \
             } \
          }' ${REQ_F}
      SRV_BASE=`awk '{ \
                          if ($1 != "#" && $1 != "#@" && NF >2) \
                           {\
                                printf "%s", $1;\
                           }\
                       }' ${REQ_F}`

      VER_NUMBER=`echo $SRV_BASE | cut -d"." -f2` 2>>/dev/null
      get_HTTPS_version
      Log_msg "------------------------------------------------------"
   else
      Log_msg ""
      Log_msg "(Operation Agent is not properly installed for $2.)"
   fi

   return 0
}
#------------------------------------------------------------------------------
verify_server()
{
  if [ ! -f /opt/OV/bin/ovconfget ]
  then 
    Log_msg "OMU Management Server not installed on local system."
    VLog_msg "/opt/OV/bin/ovconfget not installed on the system."
    return 1
  fi
  OMU_VERS=`/opt/OV/bin/ovconfget -ovrg server opc OPC_INSTALLED_VERSION`
  if [ -z "$OMU_VERS" ]
  then
    Log_msg "OMU Management Server not installed on local system."
    return 1
  else
    Log_msg "Installed OMU Management Server version: ${OMU_VERS}"
    Log_msg "OpenVMS 8.6 OA is supported with OMU8.3x."
  fi
  return 0
}
#------------------------------------------------------------------------------
verify_agent()
{
  if [ ! -f /opt/OV/bin/ovconfget ]
  then 
    VLog_msg "/opt/OV/bin/ovconfget not installed on the system."
    return 1
  fi
  OMU_VERS=`/opt/OV/bin/ovconfget eaagt OPC_INSTALLED_VERSION`
  if [ -z "$OMU_VERS" ]
  then
    Log_msg "OMU Managed Node not installed on local system. OpenVMS OA needs 8.6 OA on OM."
    return 1
  else
    Log_msg "Installed OMU Managed Node version: ${OMU_VERS}"
    Log_msg "OpenVMS Operation Agent needs 8.6 OA on OM local system."
  fi
  return 0
}
#------------------------------------------------------------------------------
verify_after ()
{
  if [ -d ${OMSVR_VEND_DIR} ]
  then
    Log_msg "Following OpenVMS OA versions are installed on the Management Server:"
    Log_msg "====================================================================="
    PLATFORMSV=`(cd ${OMSVR_VEND_DIR}; echo */*/* )`

  HP_PLAT="hp/ipf64/ovms hp/alpha/ovms"

  for platform in ${HP_PLAT}
  do
    if [ `echo ${PLATFORMSV}|grep -c ${platform}` -eq 1 ]
    then
       if [ `echo ${platform}|grep -c alpha` -eq 1 ]
       then
          Log_msg ""
          Log_msg "OpenVMS Alpha OA 8.x is installed."
          verify_details ${platform} "OpenVMS ALPHA"
       fi
       if [ `echo ${platform}|grep -c ipf64` -eq 1 ]
       then
          Log_msg ""
          Log_msg "OpenVMS IPF64/IA64 OA 8.x is installed."
          verify_details ${platform} "OpenVMS IA64"
       fi
       Log_msg ""
    fi
  done
fi

}
#------------------------------------------------------------------------------
chk_space()
{
  chk_ret=0
  chk_space=$SRV_PKG_SIZE
  
  free_ds=`get_free_ds /opt/OV`
  if [ $chk_space -gt $free_ds ]
  then
      Log_msg "$SRV_PKG_SIZE KB free disk space is require at /opt/OV: available $free_ds"
      chk_ret=1
  fi
  if [ BACKUP -eq 1 ]
  then
     if [ CHK_PLAT -eq 1 ]
     then
         chk_space=$PKG_SIZE_2
     else
         chk_space=$PKG_SIZE_4
     fi
  else
     if [ CHK_PLAT -eq 1 ]
     then
         chk_space=$PKG_SIZE_1
     else
         chk_space=$PKG_SIZE_2
     fi
  fi
  free_ds=`get_free_ds /var/opt/OV`
  if [ $chk_space -gt $free_ds ]
  then
      Log_msg "$chk_space KB free disk space is require at /var/opt/OV: available $free_ds"
      chk_ret=1
  fi
  if [ $chk_ret -ne 0 ]
  then
     VERIFY=0
     return 1
  fi
}
#------------------------------------------------------------------------------
chk_om()
{
  if [ -f /opt/OV/bin/OpC/agtinstall/opcversion ]
        then
           v_major=`/opt/OV/bin/OpC/agtinstall/opcversion -s |cut -c3-4`
           v_minor=`/opt/OV/bin/OpC/agtinstall/opcversion -s |cut -c6-7`
           in_v_major=`echo $MIN_OM_VER |cut -c3-4`
           in_v_minor=`echo $MIN_OM_VER |cut -c6-7`
           if [ ${v_major} -gt ${in_v_major} ]
           then
              return 0
           elif [ ${v_major} -eq ${in_v_major} ]
           then
              if [ ${v_minor} -ge ${in_v_minor} ]
              then
                 return 0
              else
                 return 1
              fi
           else
              return 1
           fi
        else
           return 1
      fi
      
  
}
#------------------------------------------------------------------------------
chk_oma()
{
  if [ -f /opt/OV/bin/OpC/agtinstall/opcversion ]
        then
           v_major=`/opt/OV/bin/OpC/agtinstall/opcversion -l |cut -c1-2`
           v_minor=`/opt/OV/bin/OpC/agtinstall/opcversion -l |cut -c4-5`
           in_v_major=`echo $MIN_OA_VER |cut -c3-4`
           in_v_minor=`echo $MIN_OA_VER |cut -c6-7`
           if [ ${v_major} -gt ${in_v_major} ]
           then
              return 0
           elif [ ${v_major} -eq ${in_v_major} ]
           then
              if [ ${v_minor} -ge ${in_v_minor} ]
              then
                 return 0
              else
                 return 1
              fi
           else
              return 1
           fi
        else
           return 1
      fi
}
#------------------------------------------------------------------------------
backup_vms()
{
  B_PATH_ROOT=`echo $OMSRV_VMS_OA_BACK_ROOT/pre-$VMS_VER`
  mkdir -p $B_PATH_ROOT 2>>/dev/null
  
  VLog_msg "Created backup Dir $B_PATH_ROOT"
  if [ $ALPHA_SELECT -eq 1 ]
  then
     mv -f $OMSVR_VEND_DIR/hp/alpha/ovms $B_PATH_ROOT/alpha  2>>/dev/null
     VLog_msg "Backup Alpha completed."
  fi
  

  if [ $IA_SELECT -eq 1 ]
  then
       mv -f $OMSVR_VEND_DIR/hp/ipf64/ovms $B_PATH_ROOT/ia  2>>/dev/null
       VLog_msg "Backup IA64 completed."
  fi
  

  B_PATH_ROOT=`echo $OMSVR_INST_DIR/pre-$VMS_VER`
  
  mkdir -p $B_PATH_ROOT 2>>/dev/null
  
  mv -f $OMSVR_INST_DIR/*vms* $B_PATH_ROOT/
  
  VLog_msg "Backup server script completed."
  
}
#------------------------------------------------------------------------------
chk_server()
{
  chk_space
  if [ $? -ne 0 ]
  then 
     return 1
  fi
  chk_om
  if [ $? -ne 0 ]
  then
     Log_msg "OM server version is less than $MIN_OM_VER"
     return 1
  fi
  
  chk_oma
  if [ $? -ne 0 ]
  then
     Log_msg "OA version on server is lower than $MIN_OA_VER"
     return 1
  fi
}
#------------------------------------------------------------------------------
install_vms()
{
   chk_server
  if [ $? -ne 0 ]
  then
     return 1
  fi
  
  if [ BACKUP -eq 1 ]
  then
      backup_vms
  else
      del_vms
  fi
  
  copy_srv_script
  if [ $? -ne 0 ]
  then 
     return 1
  fi
     
  copy_vms_kits
  
  reg_comp
}




#------------------------------------------------------------------------------
restore_vms()
{

  B_PATH_ROOT=`echo $OMSRV_VMS_OA_BACK_ROOT/pre-$VMS_VER`
  
  if [ $ALPHA_SELECT -eq 1 ]
  then
     SRC1_DIR=`echo $B_PATH_ROOT/alpha`
     if [ -d ${SRC1_DIR} ]
     then
        TRG1_DIR=`echo $OMSVR_VEND_DIR/hp/alpha/ovms`
        mkdir -p $TRG1_DIR 2>>/dev/null
        mv -f $SRC1_DIR $TRG1_DIR 2>>/dev/null
     else
        Log_msg "No backup available for VMS Alpha OA $VMS_VER."
     fi
  fi

  if [ $IA_SELECT -eq 1 ]
  then
     SRC1_DIR=`echo $B_PATH_ROOT/ia`
     if [ -d ${SRC1_DIR} ]
     then
        TRG1_DIR=`echo $OMSVR_VEND_DIR/hp/ipf64/ovms`
        mkdir -p $TRG1_DIR 2>>/dev/null
        mv -f $SRC1_DIR $TRG1_DIR 2>>/dev/null
     else
        Log_msg "No backup available for VMS IA64 OA $VMS_VER."
     fi
  fi
}
#------------------------------------------------------------------------------
om_pkg_unreg()
{
    plat_dir=$1


    ${OMSVR_CONF_DIR}/opcagtdbcfg -p ${plat_dir} -r 2>&1
    if [ $? -eq 0 ]
    then
           Log_msg "Deinstallation of $plat_dir is successful"
           rm -f ${OMSVR_VEND_DIR}/${plat_dir}/require.dat
           rm -f ${OMSVR_VEND_DIR}/${plat_dir}/req.dat 
           return 0 

    else 
           Log_msg  "Deinstallation of ${plat_dir} has failed"
           return 1
    fi 
}
#------------------------------------------------------------------------------
del_vms()
{
  if [ $IA_SELECT -eq 1 ]
  then
     TRG1_DIR=`echo ${OMSVR_VEND_DIR}/hp/ipf64/ovms`
     rm -Rf $TRG1_DIR 2>>/dev/null
  fi
  
  if [ $ALPHA_SELECT -eq 1 ]
  then
     TRG1_DIR=`echo ${OMSVR_VEND_DIR}/hp/alpha/ovms`
     rm -Rf $TRG1_DIR 2>>/dev/null  
  fi
}
#------------------------------------------------------------------------------
copy_srv_script()
{
   SRC_DIR="OMU8-SOL"

   SRC_DIR1=`echo $DIRNAME/ServerScripts/$SRC_DIR`

   cp -f $SRC_DIR1/* $OMSVR_INST_DIR 2>>/dev/null
   VLog_msg "Coping Server scripts completed." 
}
#------------------------------------------------------------------------------
om_pkg_reg()
{
    plat_dir=$1

    if [ ! -f ${OMSVR_VEND_DIR}/${plat_dir}/${PLAT_REG_FILE} ]
    then
            Log_msg "ERROR: There is no ${PLAT_REG_FILE} for ${plat_dir}"
            return 1
    fi

    if [ -f ${OMSVR_VEND_DIR}/${plat_dir}/req.dat ]
    then
           Log_msg "INFO: Requerment for ${plat_dir} is already present."
    else
           ${OMSVR_INSTL_DIR}/opcagtutil -platform ${plat_dir} -require 2>&1
           if [ $? -ne 0 -o ! -f ${OMSVR_VEND_DIR}/${plat_dir}/require.dat ]
           then
                Log_msg "ERROR:   Unable to extract requirement for ${plat_dir}."
                return 2
           fi
    fi
         
    ${OMSVR_CONF_DIR}/opcagtdbcfg -p ${plat_dir} -d -f 2>&1
    if [ $? -eq 0 ]
    then
           Log_msg "NOTE:    Updated database with new platform information for ${plat_dir}."
           Log_msg "Registration of ${plat_dir} is successful"
           return 0
    else
           Log_msg "ERROR: Couldn't update database with new platform information for ${plat_dir}."
           return 3
    fi
}

#------------------------------------------------------------------------------
reg_comp()
{
  if [ $IA_SELECT -eq 1 ]
  then
     om_pkg_reg "hp/ipf64/ovms"
     if [ $? -ne 0 ]
     then
        Log_msg "Problem in package registration for IPF64."
     fi
  fi
  
  if [ $ALPHA_SELECT -eq 1 ]
  then
     om_pkg_reg "hp/alpha/ovms"
     if [ $? -ne 0 ]
     then
        Log_msg "Problem in package registration for ALPHA."
     fi
     
  fi

}
#------------------------------------------------------------------------------
copy_vms_kits()
{
   SRC_DIR="OMU8-SOL"
   
  if [ $ALPHA_SELECT -eq 1 ]
  then
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/alpha/ovms/$SRV_BASE/$SRV_COMM/install`
     
     mkdir -p $TRC_DIR1 2>>/dev/null
     if [ $? -eq 0 ]
     then
        VLog_msg " $TRC_DIR1 created successfully."
     else   
        VLog_msg " $TRC_DIR1 creation failed."
     fi

     SRC_DIR1=`echo $DIRNAME/Pre-ReqChk`
     
     cp -f $SRC_DIR1/* $TRC_DIR1/  2>>/dev/null
     
     SRC_DIR1=`echo $DIRNAME/Package/VMS-ALPHA`
     
     cp -f $SRC_DIR1/OPC_INST.COM $TRC_DIR1/ 2>>/dev/null
     
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/alpha/ovms/$SRV_BASE/$SRV_COMM`
     
     cp -f $SRC_DIR1/* $TRC_DIR1 2>>/dev/null
     
     SRC_DIR1=`echo $DIRNAME/AgentConfig/$SRC_DIR/VMS-ALPHA`
     
     cp -f $SRC_DIR1/OVO-Agent.xml $TRC_DIR1 2>>/dev/null
#     cp -f $SRC_DIR1/OVO-Client.xml $TRC_DIR1 2>>/dev/null
     
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/alpha/ovms`
     
     cp -f $SRC_DIR1/AgentPlatform $TRC_DIR1 2>>/dev/null
  fi
  
  if [ $IA_SELECT -eq 1 ]
  then
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/ipf64/ovms/$SRV_BASE/$SRV_COMM/install`
     
     mkdir -p $TRC_DIR1 2>>/dev/null
     if [ $? -eq 0 ]
     then
        VLog_msg " $TRC_DIR1 created successfully."
     else   
        VLog_msg " $TRC_DIR1 creation failed."
     fi

     SRC_DIR1=`echo $DIRNAME/Pre-ReqChk`
     
     cp -f $SRC_DIR1/* $TRC_DIR1/  2>>/dev/null
     
     SRC_DIR1=`echo $DIRNAME/Package/VMS-IA`
     
     cp -f $SRC_DIR1/OPC_INST.COM $TRC_DIR1/ 2>>/dev/null
     
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/ipf64/ovms/$SRV_BASE/$SRV_COMM`
     
     cp -f $SRC_DIR1/* $TRC_DIR1 2>>/dev/null
     
     SRC_DIR1=`echo $DIRNAME/AgentConfig/$SRC_DIR/VMS-IA`
     
     cp -f $SRC_DIR1/OVO-Agent.xml $TRC_DIR1 2>>/dev/null
#     cp -f $SRC_DIR1/OVO-Client.xml $TRC_DIR1 2>>/dev/null
     
     TRC_DIR1=`echo $OMSVR_VEND_DIR/hp/ipf64/ovms`
     
     cp -f $SRC_DIR1/AgentPlatform $TRC_DIR1 2>>/dev/null
  fi

}
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
unreg_comp()
{
  if [ $IA_SELECT -eq 1 ]
  then
     om_pkg_unreg "hp/ipf64/ovms"
     if [ $? -ne 0 ]
     then
        Log_msg "Problem in package unregistration for IPF64."
     fi
  fi
  
  if [ $ALPHA_SELECT -eq 1 ]
  then
     om_pkg_unreg "hp/alpha/ovms"
     if [ $? -ne 0 ]
     then
        Log_msg "Problem in package unregistration for ALPHA."
     fi
     
  fi

}
#------------------------------------------------------------------------------
remove_vms()
{
  if [ $ROLLBACK -eq 1 ]
  then
    del_vms
    restore_vms
  else
    unreg_comp
    del_vms
  fi
}

#------------------------------------------------------------------------------
validate_input()
{
#set -x
   
   if [ $BACKUP -eq $ROLLBACK ]
   then
      if [ $BACKUP -eq 1 ]
      then
         Log_msg "ERROR: Both Backup & Rollback can't be selected."
         return 1   
      fi
   else
      if [ $BACKUP -eq 1 -a $REMOVE -eq 1 ]
      then
         Log_msg "ERROR: Wrong combination in selection Backup & Remove."
         return 1
      fi
      if [ $ROLLBACK -eq 1 -a $INSTALL -eq 1 ]
      then
         Log_msg "ERROR: Wrong combination in selection Rollback & Install."
         return 1
      fi
   fi
   
   if [ $INSTALL -eq $REMOVE ]
   then
      if [ $REMOVE -eq 1 ]
      then
         Log_msg "ERROR: Both Install & Remove can't be selected."
         return 1
      else
         Log_msg "ERROR: Install or Remove none has selected."
         return 1     
      fi
   fi
   
    if [ CHK_PLAT -eq 1 ]
    then

         if [ $IN_PLAT = "ALPHA" -o $IN_PLAT = "Alpha" -o $IN_PLAT = "alpha" ]
         then
           ALPHA_SELECT=1
         elif [ $IN_PLAT = "IPF64" -o  $IN_PLAT = "ipf64"]
         then
           IA_SELECT=1
         else
            Log_msg "Unknown platform ($IN_PLAT) selected."
            print_usage
            return 1
         fi
   else
     ALPHA_SELECT=1
     IA_SELECT=1
   fi
   
   VLog_msg "Successfully Validate the input."
   return 0
}
#------------------------------------------------------------------------------
find_all()
{ 
  Log_msg ""
  Log_msg "==================== Local OM Server ========================"
  verify_server
  if [ $? -ne 0 ]
  then
     Log_msg "   No OMU Server has detected. This can be installed only on OMU8.3x or higher."
     return 1
  fi
  Log_msg ""
  Log_msg "======================= Local OA ============================"
  verify_agent
  if [ $? -ne 0 ]
  then
     Log_msg "  No OA has detected on Local System. OpenVMS OA 8.6 will work only when 8.6x OA is present on OM."
     return 1
  fi
  Log_msg ""
  Log_msg "======================= Local Disk Space =========================="
  Log_msg "OpenVMS OA 8.6 needs $SRV_PKG_SIZE KB in /opt/OV."
  Log_msg "Available space `get_free_ds "/opt/OV"` KB" 
  Log_msg "OpenVMS OA 8.6 needs $PKG_SIZE_2 KB in /var/opt/OV. Incase of Backup it requires $PKG_SIZE_4 KB."
  Log_msg "Available space `get_free_ds "/var/opt/OV"` KB"
  Log_msg "" 
  Log_msg "======================= Local OpenVMS OA =========================="
  Log_msg ""
  if [ -d ${OMSVR_VEND_DIR} ]
  then
    Log_msg "Following OpenVMS OA versions are installed on the Management Server:"
    Log_msg "====================================================================="
    PLATFORMS=`(cd ${OMSVR_VEND_DIR}; echo */*/* )`

  HP_PLATF="hp/ia64/vms hp/alpha/vms hp/ipf64/ovms hp/alpha/ovms"

  for platform in ${HP_PLATF}
  do
    if [ `echo ${PLATFORMS}|grep -c ${platform}` -eq 1 ]
    then
       if [ `echo ${platform}|grep -c alpha` -eq 1 ]
       then
          Log_msg ""
          Log_msg "OpenVMS Alpha OA 8.x is already installed."
          display_details ${platform} "OpenVMS ALPHA"
       fi
       if [ `echo ${platform}|grep -c 64` -eq 1 ]
       then
          Log_msg ""
          Log_msg "OpenVMS IPF64/IA64 OA 8.x is already installed."
          display_details ${platform} "OpenVMS IA64"
       fi
       Log_msg ""
    fi
  done
fi
}
#------------------------------------------------------------------------------
print_usage()
{
echo "Usage: $SCRIPTNAME -[h|d|v]"
echo "Usage: $SCRIPTNAME -i [-v] [-b] [-p <plat>] [-vv]"
echo "Usage: $SCRIPTNAME -r [-v] [-rb] [-p <plat>] [-vv]"
echo "  -help|h       Display this message."
echo "  -discovery|d  Discover the local System."
echo "  -remove|r     Remove installed agents packages."
echo "  -verify|v     Check for installed agents packages."
echo "  -install|i    Install selected Architecture packages."
echo "  -platform|p   Valid with -i or -r. It should be one from ALPHA|IPF64."
echo "  -backup|b     Valid with -i. It will backup the existing agents packages."
echo "  -rollback|rb  Valid with -r. It will rollback the previous agents packages,if any."
echo "  -verbose|vv   Verbose output."
echo ""
echo "NOTE: you must be root to run this script."

  Log_msg "          OM OpenVMS 8.6 OA Maintenance script ends `date`."
}
#------------------------------------------------------------------------------

#MAIN

INST_LOG_DIR=`dirname ${INST_LOG_FILE}`
if [ ! -d ${INST_LOG_DIR} ]
then
  mkdir -p ${INST_LOG_DIR}
fi
touch ${INST_LOG_FILE}

Log_msg "          OM OpenVMS 8.6 OA Maintenance script starts `date`."
check_lockfile
if [ $? -ne 0 ]
then
  Log_msg "          OM OpenVMS 8.6 OA Maintenance script is already running - exiting `date`."
  return 1
fi

`who am i | grep root > /dev/null`
if [ $? -ne 0 ]
then
    Log_msg "User must be root."
    print_usage
    remove_lockfile
    return 1
fi

if [ $# -eq 0 ]
then
       print_usage
       remove_lockfile
       exit 0
fi

if [ $# -eq 1 ]
then
   case $1 in
      -discovery|-d)
       find_all
       remove_lockfile
       return 0
       ;;
      -help|-h)
       print_usage
       remove_lockfile
       exit 0
       ;;
   esac
fi

while [ $# -ge 1 ]
do
  case $1 in
    -remove|-r)
       REMOVE=1
       ;;
    -verify|-v)
       VERIFY=1
       ;;
    -install|-i)
       INSTALL=1
       ;;
    -backup|-b)
       INSTALL=1
       BACKUP=1
       ;;
    -rollback|-rb)
       ROLLBACK=1
       REMOVE=1
       ;;
    -verbose|-vv)
       VERBOSE=1
       ;;
    -platform|-p)
       CHK_PLAT=1
       IN_PLAT=$2
       shift
       ;;
    *)
       print_usage
       remove_lockfile
       exit 1
       ;;
  esac
  shift
done

validate_input
if [ $? -ne 0 ]
then
  Log_msg "Invalid Input provided."
  print_usage
  remove_lockfile
  return 1
fi


if [ $INSTALL -eq 1 ]
then
   install_vms
   if [ $? -ne 0 ]
   then
     remove_lockfile
     return 1
   fi 
elif [ $REMOVE -eq 1 ]
then
   remove_vms
   if [ $? -ne 0 ]
   then
     remove_lockfile
     return 1
   fi  
fi

   
if [ $VERIFY -eq 1 ]
then
   verify_after
fi
remove_lockfile
Log_msg " OM OpenVMS 8.6 OA Maintenance script ends `date`."
