
/*******************************************************************************

This software program is available to you under a choice of one of two
licenses. You may choose to be licensed under either the GNU General Public
License 2.0, June 1991, available at http://www.fsf.org/copyleft/gpl.html,
or the Intel BSD + Patent License, the text of which follows:

Recipient has requested a license and Intel Corporation ("Intel") is willing
to grant a license for the software entitled Common Statistics Manager
being provided by Intel Corporation. The following definitions apply to this
license:

"Licensed Patents" means patent claims licensable by Intel Corporation which
are necessarily infringed by the use of sale of the Software alone or when
combined with the operating system referred to below.

"Recipient" means the party to whom Intel delivers this Software.

"Licensee" means Recipient and those third parties that receive a license to
any operating system available under the GNU General Public License 2.0 or
later.

Copyright (c) 2004 - 2005 Intel Corporation.
All rights reserved.

The license is provided to Recipient and Recipient's Licensees under the
following terms.

Redistribution and use in source and binary forms of the Software, with or
without modification, are permitted provided that the following conditions
are met:

Redistributions of source code of the Software may retain the above
copyright notice, this list of conditions and the following disclaimer
Redistributions in binary form of the Software may reproduce the above
copyright notice, this list of conditions and the following disclaimer in
the documentation and/or materials provided with the distribution.

Neither the name of Intel Corporation nor the names of its contributors
shall be used to endorse or promote products derived from this Software
without specific prior written permission.

Intel hereby grants Recipient and Licensees a non-exclusive, worldwide,
royalty-free patent license under Licensed Patents to make, use, sell, offer
to sell, import and otherwise transfer the Software, if any, in source code
and object code form. This license shall include changes to the Software
that are error corrections or other minor changes to the Software that do
not add functionality or features when the Software is incorporated in any
version of an operating system that has been distributed under the GNU
General Public License 2.0 or later. This patent license shall apply to the
combination of the Software and any operating system licensed under the GNU
General Public License 2.0 or later if, at the time Intel provides the
Software to Recipient, such addition of the Software to the then publicly
available versions of such operating systems available under the GNU General
Public License 2.0 or later (whether in gold, beta or alpha form) causes
such combination to be covered by the Licensed Patents. The patent license
shall not apply to any other combinations which include the Software. NO
hardware per se is licensed hereunder.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**********************************************************************
*                                                                     *
* INTEL CORPORATION                                                   *
*                                                                     *
* This software is supplied under the terms of the license included   *
* above.  All use of this module must be in accordance with the terms *
* of that license.                                                    *
*                                                                     *
* Module Name: dma_list.h                                             *
*                                                                     *
* Environment: The linux kernel contains list manipulation macros,    *
* data structures and functions, which may be substituted for these   *
* functions. These may be found in the include/linux/list.h file.     *
*                                                                     *
*                                                                     *
**********************************************************************/

#ifndef DMA_LIST_H
#define DMA_LIST_H

#include <linux/stddef.h>


/* A generic structure for the operation of linked lists in the driver.*/
struct list_start
{
    char * next_ptr;
    char * prev_ptr;
};

/*
 * Lists in the driver, such as resource lists, are constructed
 * with the functions below and takes the form described in
 * the example:
 *
 * EXAMPLE:
 *
 *  ---- the list head prev should be NULL ----
 *  HEAD.next_ptr = ELEMENT1
 *  HEAD.prev_ptr = ELEMENT3
 *
 *  ELEMENT1.next_ptr = ELEMENT2
 *  ELEMENT1.prev_ptr = HEAD
 *
 *  ELEMENT2.next_ptr = ELEMENT3
 *  ELEMENT2.prev_ptr = ELEMENT1
 *
 *  ELEMENT3.next_ptr = NULL
 *  ELEMENT3.prev_ptr = ELEMENT2
 *
 */

/*
 * adjust_ptrs - adjusts the pointers when a list element is removed by
 * the removal functions.
 *
 * NOTE: used by the removal functions only.
 *
 * @prev: pointer to the element before the element being removed.
 * @head: pointer to the head of a list.
 *
 */
static inline void adjust_ptrs(struct list_start * prev, struct list_start * head)
{


    struct list_start * next = (struct list_start *) prev->next_ptr;
    prev->next_ptr = next->next_ptr;


    if(prev == head && next->next_ptr == NULL)
        head->prev_ptr = NULL;

    if(prev != head &&  next->next_ptr == NULL)
        head->prev_ptr = (char *)prev;

    if(next->next_ptr != NULL)
    {
        next = (struct list_start *)next->next_ptr;
        next->prev_ptr = (char *)prev;
    }


}

/*
 * remove_element - removes an element from a list.
 *
 * @element: list element to remove.
 * @head: head of the list from which you wish to remove the element.
 *
 */
static inline void remove_element(struct list_start * element, struct list_start * head)
{
    /* Obtain the element before the element that is being removed. */
    struct list_start * prev = (struct list_start *)element->prev_ptr;
    adjust_ptrs(prev, head);
}



/*
 * linked_list_reinit - initializes list to proper empty values.
 *
 * @new_list: pointer to the list that is being initialized
 *
 */
static inline void linked_list_reinit(struct list_start * list)
{
    if(list != NULL)
    {
        list->next_ptr = NULL;
        list->prev_ptr = NULL;
    }
}

/*
 * remove_each_element_front - removes an element from the front
 * of the list.
 *
 * RETURN: pointer to a list element that is being removed.
 *
 * NOTE: This function should be called within a loop
 * to free an entire list.
 *
 * @head: pointer to the head of a list.
 *
 */
static inline char *  remove_each_element_front(struct list_start * head)
{

    struct list_start * ret = (struct list_start *)head->next_ptr;
    struct list_start * next;

    if(ret != NULL)
    {

        ret = (struct list_start *)head->next_ptr;
        next =(struct list_start *)ret->next_ptr;
        head->next_ptr = (char *)next;
        if(next != NULL)
        {
            next->prev_ptr =(char*) head;
            if(next->next_ptr == NULL)
                head->prev_ptr = (char *)next;

        }
        else
        {
            head->next_ptr = head->prev_ptr = NULL;

        }

    }


    return (char *)ret;

}

/*
 * insert_element_front - inserts a list element at the front of the list.
 *
 * @element: pointer to element to insert
 * @head: pointer to head of list
 *
 */
static inline void insert_element_front(struct list_start * element, struct list_start * head)
{
    struct list_start * temp = NULL;
    if(head->next_ptr != NULL)
    {
        temp = (struct list_start *)head->next_ptr;
        temp->prev_ptr = (char *)element;
    }
    else
    {
        head->prev_ptr = (char *) element;
    }
    head->next_ptr = (char *)element;
    element->next_ptr = (char *)temp;
    element->prev_ptr = (char *)head;
}

/*
 * insert_element_at_end - inserts a list element at the end of the list.
 *
 * @element: pointer to element to insert
 * @head: pointer to head of list
 *
 */
static inline void insert_element_at_end(struct list_start * element, struct list_start * head)
{
    struct list_start * last = (struct list_start *) head->prev_ptr;
    if(last != NULL)
    {
        last->next_ptr = (char *)element;
        element->prev_ptr = (char *)last;
        element->next_ptr = NULL;
        head->prev_ptr = (char *) element;
    }
    else
    {
        head->next_ptr = (char *)element;
        head->prev_ptr = (char *)element;
        element->prev_ptr =(char *)head;
        element->next_ptr = NULL;
    }
        
}


/*
 * join_lists - joins list_to_add to list by inserting list_to_add
 * at the beginning of the list.
 *
 *
 * NOTE: Both parameters should be non-NULL values; otherwise, a
 * segmentation fault may occur.
 *
 * @list_to_add: pointer to head of list one
 * @list: pointer to head of list two
 *
 */
static inline void join_lists(struct list_start * list_to_add, struct list_start * list)
{
	
    struct list_start * at = list;
    struct list_start * last = (struct list_start *)list_to_add->prev_ptr;
    struct list_start * first = (struct list_start *)list_to_add->next_ptr;

    if(first != NULL)
    {
        at = (struct list_start *)list->next_ptr;
        list->next_ptr = (char *)first;
	last->next_ptr = (char *)at;
	if(at != NULL)
            at->prev_ptr = (char *)last;


    }
    first->prev_ptr = (char *)list;
    if(list->prev_ptr == NULL)
        list->prev_ptr = (char *)last;
    list_to_add->next_ptr = NULL;
    list_to_add->prev_ptr = NULL;

}

/*
 * join_lists_tail - joins list_to_add to list by inserting list_to_add
 * at the end of the list.
 *
 *
 * NOTE: Both parameters should be non-NULL values; otherwise, a
 * segmentation fault may occur.
 *
 * @list_to_add: pointer to head of list one
 * @list: pointer to head of list two
 *
 */
static inline void join_lists_tail(struct list_start * list_to_add, struct list_start * list)
{

    
    struct list_start * last_lta = (struct list_start *)list_to_add->prev_ptr;
    struct list_start * last_lt =  (struct list_start *)list->prev_ptr;
    struct list_start * first_lta = (struct list_start *)list_to_add->next_ptr;


    if(last_lt != NULL)
    {

        last_lt->next_ptr = list_to_add->next_ptr;
	first_lta->prev_ptr = (char *)last_lt;
	list->prev_ptr = (char *)last_lta;
    }
    else
    {
        list->next_ptr = list_to_add->next_ptr;
	list->prev_ptr = list_to_add->prev_ptr;
	first_lta->prev_ptr = (char *)list;
    }
	
	
    list_to_add->next_ptr = NULL;
    list_to_add->prev_ptr = NULL;

	
}
/*
 *  printlist - prints an entire list. For testing purposes only.
 *
 * @list: head of list to print.
 *
 */
 /*
void printlist(struct list_start * list)
{
    struct list_start * cnt = list;
    int i = 0;
    printk("List Start: list_ptr %x next_ptr %x, prev_ptr %x\n",
    (unsigned int)list, (unsigned int)list->next_ptr,(unsigned int)list->prev_ptr);
    printf("List Start: list_ptr %x next_ptr %x, prev_ptr %x\n",
    (unsigned int)list, (unsigned int)list->next_ptr,(unsigned int)list->prev_ptr);

    while(cnt->next_ptr != NULL)
    {
	cnt = (struct list_start *)cnt->next_ptr;
	printk("Element%x: element %x next_ptr %x, prev_ptr %x\n",i, (unsigned int)cnt,
	(unsigned int)cnt->next_ptr, (unsigned int)cnt->prev_ptr);
	printf("Element%x: element %x next_ptr %x, prev_ptr %x\n",i, (unsigned int)cnt,
	(unsigned int)cnt->next_ptr, (unsigned int)cnt->prev_ptr);
	i++;
     }
     printk("ENDOFPRINTLIST\n");
     printf("ENDOFPRINTLIST\n");
}*/

#endif
