

/*******************************************************************************

This software program is available to you under a choice of one of two
licenses. You may choose to be licensed under either the GNU General Public
License 2.0, June 1991, available at http://www.fsf.org/copyleft/gpl.html,
or the Intel BSD + Patent License, the text of which follows:

Recipient has requested a license and Intel Corporation ("Intel") is willing
to grant a license for the software entitled Common Statistics Manager
being provided by Intel Corporation. The following definitions apply to this
license:

"Licensed Patents" means patent claims licensable by Intel Corporation which
are necessarily infringed by the use of sale of the Software alone or when
combined with the operating system referred to below.

"Recipient" means the party to whom Intel delivers this Software.

"Licensee" means Recipient and those third parties that receive a license to
any operating system available under the GNU General Public License 2.0 or
later.

Copyright (c) 2004 - 2005 Intel Corporation.
All rights reserved.

The license is provided to Recipient and Recipient's Licensees under the
following terms.

Redistribution and use in source and binary forms of the Software, with or
without modification, are permitted provided that the following conditions
are met:

Redistributions of source code of the Software may retain the above
copyright notice, this list of conditions and the following disclaimer
Redistributions in binary form of the Software may reproduce the above
copyright notice, this list of conditions and the following disclaimer in
the documentation and/or materials provided with the distribution.

Neither the name of Intel Corporation nor the names of its contributors
shall be used to endorse or promote products derived from this Software
without specific prior written permission.

Intel hereby grants Recipient and Licensees a non-exclusive, worldwide,
royalty-free patent license under Licensed Patents to make, use, sell, offer
to sell, import and otherwise transfer the Software, if any, in source code
and object code form. This license shall include changes to the Software
that are error corrections or other minor changes to the Software that do
not add functionality or features when the Software is incorporated in any
version of an operating system that has been distributed under the GNU
General Public License 2.0 or later. This patent license shall apply to the
combination of the Software and any operating system licensed under the GNU
General Public License 2.0 or later if, at the time Intel provides the
Software to Recipient, such addition of the Software to the then publicly
available versions of such operating systems available under the GNU General
Public License 2.0 or later (whether in gold, beta or alpha form) causes
such combination to be covered by the Licensed Patents. The patent license
shall not apply to any other combinations which include the Software. NO
hardware per se is licensed hereunder.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MECHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL INTEL OR IT CONTRIBUTORS BE LIABLE FOR ANY
DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
ANY LOSS OF USE; DATA, OR PROFITS; OR BUSINESS INTERUPTION) HOWEVER CAUSED
AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY OR
TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

/**********************************************************************
*                                                                     *
* INTEL CORPORATION                                                   *
*                                                                     *
* This software is supplied under the terms of the license included   *
* above.  All use of this module must be in accordance with the terms *
* of that license.                                                    *
*                                                                     *
* File Name: dma.c                                                    *
*                                                                     *
* Environment:  This file is intended to be specific to the Linux     *
*               2.4.21 kernel distributed with                        *
*               Red Hat Enterprise Linux AS and WS Version 3.         *
**********************************************************************/


#ifndef __KERNEL__
#define __KERNEL__
#endif


#include <linux/module.h> //SET_MODULE_OWNER
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/kernel.h> /* printk() */
#include <linux/string.h>
#include <linux/init.h>
#include <linux/fs.h>      /* everything... */
#include <linux/errno.h>   /* error codes */
#include <linux/types.h>   /* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>   /* O_ACCMODE */
#include <asm/system.h>    /* cli, *_flags */
#include <asm/uaccess.h>   /* copy_to_user */
#include <asm/semaphore.h>
#include <asm/io.h>        /* inb(), outb() */
#include <linux/kmod.h>
#include <linux/ioport.h>  /* request_region */
#include <linux/pci.h>
#include <linux/time.h>
#include <linux/interrupt.h> /* tasklets */
#include <linux/list.h>
#include <asm/mman.h>
#include <asm/page.h>
#include <linux/mm.h>

#ifdef DMA_2_4
#include <linux/wrapper.h>
#endif

#include "dma_api.h"

#define DEBUG_DMA_L 0



MODULE_LICENSE("GPL");
MODULE_AUTHOR("Intel Corporation - Infrastructure Processor Division");
MODULE_DESCRIPTION("EDMA Reference Driver - IPD");

static int edma_chan_priority = -1; /* Default */
static int edma_mmio          =  1; /* Enabled */
#ifdef DMA_2_4
MODULE_PARM(edma_chan_priority, "i");
MODULE_PARM(edma_mmio, "i");
#endif

#ifdef DMA_2_6
#include <linux/moduleparam.h>
module_param(edma_chan_priority, int, S_IRUGO);
module_param(edma_mmio, int, S_IRUGO);
#endif
/*
*  Global variables -
*  Every module is linked to the whole kernel.
*  All symbols that you are not willing to export
*  should be declared static.
*/

static struct list_start g_edma_client_list     = {NULL, NULL};
static char g_drvr_name[]                      = PRODUCT_STRING;  // Name of driver in proc.
static char g_char_drvr_name[]                 = CHAR_DRV_NAME;
static struct pci_dev * g_dev_cfg_space;
static struct edma_mm_regs * g_mm_regs;
static struct edma_chan * g_edma_channel_list[MAX_NO_CHANS];
static unsigned long g_tasklet_data[] = { 0, 0, 0, 0};
static struct edma_descriptor_list g_descriptor_lists[MAX_NO_CHANS];
static unsigned int g_channel_ccr_offsets[] = { EDMA_CHAN0_CCR, EDMA_CHAN1_CCR,
    EDMA_CHAN2_CCR, EDMA_CHAN3_CCR};

static spinlock_t g_tasklet_lock;

/* Prototypes - Application Level Only */
int edma_init(void);
void edma_cleanup(void);
int edma_ioctl(struct inode * inode, struct file * filep, unsigned int cmd, unsigned long arg);
int edma_open(struct inode * inode, struct file * filep);
int edma_release(struct inode * inode, struct file * filep);

/* Application Level Available through IOCTL Call */
struct edma_client * edma_client_alloc_app(struct edma_client *client);
struct edma_chan * edma_chan_alloc_app(struct edma_client * client, struct edma_client *kern_client);
int allocate_buffer_app(struct file * , struct vm_area_struct *);
edma_cookie_t edma_memcpy_app(u32 memory_ops, struct edma_chan *chan, edma_addr_t dest,
edma_addr_t src,size_t size);



/* Prototypes - Misc. */
int create_internal_structures(void);
void free_internal_structures(void);
void release_client_resources(struct edma_client_internal *client_internal);
void  status_tasklet_func(unsigned long data);
int get_descriptors(int num, int channel);
void edma_descriptors_init(struct edma_chan * chan);

DECLARE_TASKLET(status_tasklet, status_tasklet_func,
(unsigned long) &g_tasklet_data);

struct file_operations edma_fileops = {
    ioctl: edma_ioctl,
    open:  edma_open,
    mmap: allocate_buffer_app,
    release: edma_release  };
    



static struct edma_api_export edma_api = {
	.edma_client_alloc = edma_client_alloc,
	.edma_client_release = edma_client_release,
	.edma_client_register = edma_client_register,
	.edma_client_unregister = edma_client_unregister,
	.edma_chan_alloc = edma_chan_alloc,
	.edma_chan_release = edma_chan_release,
	.edma_memcpy = edma_memcpy,
	.edma_is_complete = edma_is_complete
};




/*
 * edma_memcpy_app - performs the memory transfer operations
 * (Mem-to-Mem and Mem-to-IO). This particular memcpy function is
 * part of the application interface.
 *
 * RETURN: integer cookie value greater than 0.
 *
 * @memory_ops: type of memory tranfers (e.g. Memory to Memory)
 * @chan: channel
 * @dest: physical address of the memory destination
 * @src:  physical address of the memory source
 * @size: entire transfer size in bytes
 */
edma_cookie_t edma_memcpy_app(u32 memory_ops, struct edma_chan *chan,
edma_addr_t dest, edma_addr_t src,size_t size)
{
    edma_cookie_t cookie = 0;
    cookie = edma_memcpy(memory_ops, chan, dest, src, size, 0);

    return cookie;
}

/*
 * get_descriptors - returns the starting index for n number of free
 * descriptors.
 *
 * RETURN: an error message (-1) indicating there are not enough
 * descriptors or an integer index > -1 indicating
 * where in the list the free descriptors begin.
 *
 * @num: number of descriptors requested.
 * @channel: on which channel descriptors are requested.
 *
 *
 */
int get_descriptors(int num, int channel)
{

    int temp_head_index, num_used, num_free;
    int ret_code = ERROR_OUT_OF_DESCRIPTORS;
    temp_head_index = num_used = num_free = 0;
    struct edma_descriptor_internal * desc_array = (struct edma_descriptor_internal *)g_descriptor_lists[channel].descriptor_pool;

    if((channel >= 0) && (channel < MAX_NO_CHANS))
    {
        /*
         * If < condition is true, head pointer has wrapped
         * around to the beginning of the list.
         */
        
        if(g_descriptor_lists[channel].head_index <= g_descriptor_lists[channel].tail_index)
        {
	     
            /* Driver Loaded condition */
            if(g_descriptor_lists[channel].head_index == -1)
            {
                num_used = 0;
                g_descriptor_lists[channel].head_index = 0;
            } 
            /* All descriptors in use condition */
            else if(g_descriptor_lists[channel].head_index ==  g_descriptor_lists[channel].tail_index)
            {
                
		if(desc_array[DESCRIPTOR_NO - 1].status == EDMA_IN_PROGRESS)
                {
		     num_used = DESCRIPTOR_NO;     
                } 
                else
                {
                     num_used = 0;
                }
                
            }
            else
	    {
		temp_head_index = g_descriptor_lists[channel].head_index + DESCRIPTOR_NO;
                num_used = temp_head_index - g_descriptor_lists[channel].tail_index;
            }
           
        }
        else
        {
           num_used = g_descriptor_lists[channel].head_index - g_descriptor_lists[channel].tail_index;
        }

        
        num_free = DESCRIPTOR_NO - num_used;
       
       /*
        *  If the number of descriptors needed is less than number of free,
        *  an error is returned.
        */
        //KD
        
        if(num > num_free)
            ret_code = ERROR_OUT_OF_DESCRIPTORS;
        else
        {
            /* Return the starting index for descriptors */
            ret_code = g_descriptor_lists[channel].head_index;
            /* Adjust the head to reflect additional descriptors being used */
            g_descriptor_lists[channel].head_index += num;
	    g_descriptor_lists[channel].head_index %= DESCRIPTOR_NO;
        }
    }
    
    return ret_code;
}
/*
 * edma_memcpy - performs the memory transfer operations (Mem-to-Mem and Mem-to-IO).
 *
 * RETURN: integer cookie value greater than 0.
 *
 * @memory_ops: type of memory transfers (e.g. Memory to Memory)
 * @chan: channel
 * @dest: physical address of the memory destination
 * @src:  physical address of the memory source
 * @size: entire transfer size in bytes
 *
 */
edma_cookie_t edma_memcpy(unsigned int memory_ops, struct edma_chan *chan, edma_addr_t dest,
edma_addr_t src, size_t size, unsigned long user_data)
{
    /* Variables for setting up transactions */
    struct edma_descriptor_internal * desc_array;
    unsigned long chan_csr_val = 0;
    unsigned int len = 0;
    edma_cookie_t cookie = 0;
    int num_desc, desc_index, first_index, chain_index;
    num_desc = desc_index = first_index = chain_index = 0;

    /* Clear the first two bits. These shouldn't be set by the client. */
    memory_ops = memory_ops & CLEAR_MEMORY_OPS;
    /* Configure the first two bits properly. */
    memory_ops = memory_ops | INT_END_OF_CHAIN | INT_ABORT;

    /* Calculate the number of descriptors needed. */
    num_desc = size / MAX_TRANSFER_PER_DESC;
    if((size % MAX_TRANSFER_PER_DESC) != 0)
        num_desc++;
    
       
    /* Ensure DWORD alignment (only EDMA_L platform is concerned with alignment) */
    #ifdef DMA_L
        if(size % DWORD_SIZE)
            return -ENODEV;

        if(num_desc == 1)
            size = size/DWORD_SIZE;
    
        /* If num_desc > 1 descriptor, size is what is leftover div by 4. */
        else
        {
    
            if((size % MAX_TRANSFER_PER_DESC) == 0)
                size = MAX_TRANSFER_PER_DESC/DWORD_SIZE;
            else
                size = (size % MAX_TRANSFER_PER_DESC)/DWORD_SIZE;
    
        }
    #endif
   
    if((edma_mmio < 1) && ( memory_ops & DST_IO))
    {
        cookie = INVALID_COOKIE;
	return cookie;
    }

    if((desc_index = get_descriptors(num_desc, chan->channel_no)) == ERROR_OUT_OF_DESCRIPTORS)
    	return ERROR_OUT_OF_DESCRIPTORS;
    else
    	desc_array = (struct edma_descriptor_internal * )g_descriptor_lists[chan->channel_no].descriptor_pool;



    /* Incrementing Cookie */
    chan->cookie++;
    if(chan->cookie < 0)
        chan->cookie  = 1;
    cookie = chan->cookie;


    first_index = desc_index;
    chain_index = first_index - 1;
    if(chain_index < 0)
        chain_index = DESCRIPTOR_NO - 1;
   /*
    *   Break up the transfer into blocks of MAX_TRANSER_PER_DESC or smaller.
    *   Each block is assigned one descriptor. Chain the descriptors together in
    *   a linked list format.
    *
    */
    while(num_desc)
    {
        if(num_desc == 1)
        {
            desc_array[desc_index].hw.ndar  = 0;
            desc_array[desc_index].hw.nduar = 0;
            len = size;
            desc_array[desc_index].hw.tcr = len;

        }
        else
        {
            #ifdef DMA_L
                len = MAX_TRANSFER_PER_DESC/4;
            #elif EDMA_W
                len = MAX_TRANSFER_PER_DESC;
            #endif
            
            
            desc_array[desc_index].hw.ndar  = desc_array[desc_index].next_ndar;
            desc_array[desc_index].hw.nduar = desc_array[desc_index].next_nduar;
            desc_array[desc_index].hw.tcr =  len;

        }
        /* 
         * This section will cause problems if the kernel is not
         * configured to deal with 64 bit values. See programmer's guide 
         * for details.
         */
        desc_array[desc_index].hw.sar = src & LOWER_64;
        desc_array[desc_index].hw.suar = (src >> 32) & LOWER_64;
        desc_array[desc_index].hw.dar = dest & LOWER_64;
        desc_array[desc_index].hw.duar = (src >> 32) & LOWER_64;

       /*
        *  This debug section was inserted to unit test a dma abort condition.
        *  It may be removed without affecting driver behavior.
        *  It causes the src and destination addresses to be 'un-aligned'
        *  so that memcpy will abort during a transaction with
        *  multiple descriptors on a DMA_L platform.
        */
        if(DEBUG_DMA_L)
        {
            if(desc_index == 2)
            {
                src = src + 1;
                dest = dest + 1;
                desc_array[desc_index].hw.sar = src & LOWER_64;
                desc_array[desc_index].hw.suar =  (src >> 32) & LOWER_64;
                desc_array[desc_index].hw.dar = dest & LOWER_64;
                desc_array[desc_index].hw.duar = (dest >> 32) & LOWER_64;
             }

        }

        desc_array[desc_index].hw.dcr = memory_ops;
        desc_array[desc_index].cookie = cookie;
        desc_array[desc_index].status = EDMA_IN_PROGRESS;
        desc_array[desc_index].ferr   = -1;

        desc_index++;
        if(desc_index == DESCRIPTOR_NO)
            desc_index = 0;
        num_desc--;

        src = src + MAX_TRANSFER_PER_DESC;
        dest = dest + MAX_TRANSFER_PER_DESC;

    }

    /* Check the status register. */
    chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no] + EDMA_CSR);

    if(chan_csr_val & CHAN_ACTIVE)
    {
        printk("Channel Active\n");
        /* Suspend the current DMA Operation and Wait for the appropriate status value. */
        writel(SUSPEND_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no]);

        do
        {
            chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no] + EDMA_CSR);
        }while(!(chan_csr_val |  DMA_SUSPENDED));

        /* Append to the last batch of descriptors in the chain. */
        desc_array[chain_index].hw.ndar = desc_array[chain_index].next_ndar;
        desc_array[chain_index].hw.nduar = desc_array[chain_index].next_nduar;

        /* Clear the Suspend and Resume the Operation. */
        writel(0, (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no]);
        writel(RESUME_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no]);


    }
    else
    {
        printk("Channel Idle\n");
        writel(desc_array[first_index].phys & LOWER_64,
        (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no] + EDMA_NDAR);
        writel((desc_array[first_index].phys >> 32) & LOWER_64,
        (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no] + EDMA_NDUAR);
        writel(START_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan->channel_no]);
    }

    return cookie;
}

/*
 * allocate_buffer_app - allocates memory buffers that can be used
 * by application space and kernel space. This function is used for testing
 * only as part of an application interface. A unit test application
 * may call mmap, which will in turn call this function to obtain
 * the memory buffers. See Linux documentation for more details about mmap.
 *
 * RETURN: 0 successful, non-zero unsuccessful.
 *
 * See mmap documentation for details about parameters.
 *
 */


int allocate_buffer_app(struct file * filp, struct vm_area_struct * vma)
{
    #ifdef DMA_2_4
        struct page * paget;
        unsigned long phys_addr;
        unsigned int * edma_mem = NULL;
    #endif
    unsigned long size = vma->vm_end - vma->vm_start;
    unsigned long offset = vma->vm_pgoff * 0x1000;



   /*
    * NON-MALLOCED MEMORY
    * Total System memory in the test system was 512 MB.
    * The upper 128 MB of this memory was 'hidden'
    * from Linux for use in large dma memory transactions.
    *
    * The remap_page_range call below returns a virtual address (mapped
    * to the physical address of the upper portion of memory) to the application
    * for shared use.
    */
    if(offset >= 0x1800000)
        remap_page_range(vma,vma->vm_start,offset,size,vma->vm_page_prot);

   /*
    * MALLOCED MEMORY
    * This section is for smaller transactions. Memory is malloced for sharing
    * with the application. The mem_map_reserve statement reserves the memory,
    * which must be done to enable this malloced memory to be shared.
    */
    else
    {
        /* These memory operations work in 2.4 only.*/
        #ifdef DMA_2_4
        edma_mem = kmalloc(size, __GFP_DMA);
        if(edma_mem != NULL)
        {
             /*
             * Reserve the memory. Necessary for sharing kmalloced memory between
             * application and kernel space.
             */
            paget = virt_to_page(edma_mem);
            mem_map_reserve(paget);
            phys_addr = virt_to_phys(edma_mem);
            vma->vm_flags |= VM_LOCKED;
            if(remap_page_range(vma,vma->vm_start,phys_addr,size,vma->vm_page_prot))
            {

                return -ENXIO;

            }
           /* Save the physical address at the first dword. */
           *edma_mem = phys_addr;

        }
        else
        {
            return -ENXIO;
        }
       #endif
    }

    return 0;
}



/*
 * edma_client_alloc_app - creates and returns a client structure. This
 * function is used for testing only as part of an application interface.
 * The edma_client_alloc function returns a client structure to the
 * client driver which the client driver must initialize. Because the
 * unit test application was not able to initialize a kmalloced structure,
 * this function was created as a way to enable passing client information
 * back and forth.
 *
 * RETURN: NULL = unsuccessful, Non-null value = successful.
 *
 * @app_client - a edma_client created and initialized by the application.
 *
 */
struct edma_client * edma_client_alloc_app(struct edma_client *app_client)
{
    struct edma_client * new_client = edma_client_alloc();

    /*
    * The application supplies an initialized edma_client struct
    * which is copied to the driver's official edma_client struct.
    */
    copy_from_user(new_client, app_client, sizeof(struct edma_client));

    return new_client;
}
/*
 * edma_chan_alloc_app - creates and returns a channel structure. This
 * function is used for testing only as part of an application interface.
 * The edma_chan_alloc function returns a edma_chan structure to the
 * client driver.
 *
 * RETURN: NULL = unsucessful, Non-null value = successful.
 *
 * @app_client - a edma_client created and initialized by the application.
 *
 */
struct edma_chan * edma_chan_alloc_app(struct edma_client * app_client, struct edma_client *kern_client)
{
    struct edma_chan * new_chan;
    copy_from_user(kern_client, app_client, sizeof(struct edma_client));
    new_chan =  edma_chan_alloc(kern_client);
    return new_chan;
}
/*
 * edma_client_alloc - creates and returns a edma_client structure.
 *
 * NOTE: Creates a new client structure, along with additional allocations
 * internal to the driver.  Do not pass client structures to the
 * register function that have not been allocated with edma_client_alloc.
 * Similarly, only use edma_client_release to free the structure.
 *
 * RETURN: NULL = unsucessful, Non-NULL value = successful.
 *
 *
 */
struct edma_client * edma_client_alloc()
{
    struct edma_client_internal *client = NULL;

    /* Create the internal structure. */
    client = kmalloc(sizeof(struct edma_client_internal), GFP_KERNEL);
    if(client == NULL) 
    {
        printk(KERN_WARNING ERROR_CLIENT_ALLOC, g_drvr_name);
        return NULL;
    }
    /* Fill the region of mem with 0s. */
    memset(client, 0, sizeof(*client));

    return &client->pub;
}


/*
 * edma_client_release - releases @client structure.
 *
 * NOTE: Use this function after unregistering the client.
 * Releases the client structure, along with the additional internal
 * allocations made in edma_client_alloc.
 *
 * @client: pointer to driver-created client
 *
 */
void edma_client_release(struct edma_client *client)
{


    struct edma_client_internal *client_internal;
    /*
        This operation is very dependent upon the
        location of the client in the client_internal
        structure.

        In 2.6.x kernels, there is a container_of
        macro that returns the pointer of the container struct
        given a member of that struct, which is the another way
        of handling the structure-in-structure issue.
    */
    if(client != NULL)
    {
        client_internal = (struct edma_client_internal *)((char *)client - (2 * sizeof(void *)));
        kfree(client_internal);
    }

}



/*
 * create_internal_structures - creates structures and lists
 * internal to driver operations.
 *
 * NOTE: called at driver initialization.
 *
 * RETURN: SUCCESS (0), FAILED (-1)
 *
 */
int create_internal_structures()
{
    int i;
    int err = SUCCESS;
    int size = (sizeof(struct edma_descriptor_internal)  * DESCRIPTOR_NO) + DWORD_ALIGNED;
    int last_byte = 0;
    int add_on =0;

    spin_lock_init(&g_tasklet_lock);




    /* Create and initialize the channel list */
    for (i = 0; i < MAX_NO_CHANS; i++)
    {

        g_descriptor_lists[i].descriptor_pool = NULL;
        g_descriptor_lists[i].pool_addr = NULL;
        g_edma_channel_list[i] = kmalloc(sizeof(struct edma_chan),GFP_KERNEL);
        if(g_edma_channel_list[i] != NULL)
        {
            memset(g_edma_channel_list[i], 0, sizeof(struct edma_chan));
            /* not in use */
            g_edma_channel_list[i]->sw_in_use = FREE;
            g_edma_channel_list[i]->channel_no = i;
            g_edma_channel_list[i]->reg_offset = g_channel_ccr_offsets[i];
            g_edma_channel_list[i]->resource.pub.type = EDMA_RESOURCE;
            g_edma_channel_list[i]->resource.pub.ptr = g_edma_channel_list[i];
            spin_lock_init(&g_edma_channel_list[i]->desc_lock);


        }
        else
        {
            err = FAILED;
            return err;
        }
    }

    /* Initialize DESCRIPTOR_NO Descriptors Per Channel */
    for(i = 0; i < MAX_NO_CHANS; i++)
    {
        g_descriptor_lists[i].pool_addr = (char *)kmalloc(size,  GFP_KERNEL);
        if(g_descriptor_lists[i].pool_addr != NULL)
        {
            last_byte = (int)g_descriptor_lists[i].pool_addr & 0x000000FF;

           if((last_byte % DWORD_ALIGNED) != 0 )
           {
               add_on = DWORD_ALIGNED - (last_byte % DWORD_ALIGNED);
               g_descriptor_lists[i].descriptor_pool = g_descriptor_lists[i].pool_addr + add_on;
           }

           else
           {
               g_descriptor_lists[i].descriptor_pool = g_descriptor_lists[i].pool_addr;
           }
           memset(g_descriptor_lists[i].pool_addr, 0, size);
       }
       else
       {
          err = FAILED;
          i = MAX_NO_CHANS;
       }

    }

    if(g_descriptor_lists[0].descriptor_pool == NULL || g_descriptor_lists[1].descriptor_pool == NULL ||    g_descriptor_lists[2].descriptor_pool == NULL || g_descriptor_lists[3].descriptor_pool == NULL)
    {
        err = FAILED;
        return err;
    }
    else
    {
        memset(g_descriptor_lists[0].descriptor_pool, 0, size );
        memset(g_descriptor_lists[1].descriptor_pool, 0, size );
        memset(g_descriptor_lists[2].descriptor_pool, 0, size );
        memset(g_descriptor_lists[3].descriptor_pool, 0, size );

    }

    /* Initialize the head and tail per list. */
    g_descriptor_lists[0].head_index = g_descriptor_lists[0].tail_index = 0;
    g_descriptor_lists[1].head_index = g_descriptor_lists[1].tail_index = 0;
    g_descriptor_lists[2].head_index = g_descriptor_lists[2].tail_index = 0;
    g_descriptor_lists[3].head_index = g_descriptor_lists[3].tail_index = 0;



    return err;
}


/*
 * frees_internal_structures - frees structures and lists
 *  internal to driver operations.
 *
 * 
 * NOTE: called at driver cleanup.
 */
void free_internal_structures()
{
    int i;

    /* Free the kmalloced the channel list. */
    for (i = 0; i < MAX_NO_CHANS; i++)
    {
        if(g_edma_channel_list[i] != NULL)
            kfree(g_edma_channel_list[i]);
        if(g_descriptor_lists[i].pool_addr != NULL)
            kfree(g_descriptor_lists[i].pool_addr);
    }
}
/*
 * edma_chan_alloc - "allocates" a channel (1 channel/client)
 * to registered client.
 *
 * NOTE: The client structure contains a member through which the
 * client may request a particular channel (@client->chan_no).
 * If the channel is available the channel and client are
 * associated by the channel's resource struct being added to the
 * client's resource.
 *
 * @client: a registered client.
 *
 * RETURN a pointer to the channel, NULL if it is unavailable.
 *
 */
struct edma_chan * edma_chan_alloc(struct edma_client *client)
{
    struct edma_chan *chan = NULL;
    struct edma_client_internal *client_internal;

    client_internal = (struct edma_client_internal *)((char *)client - (2 * sizeof(void *)));

    if(client->chan_no >= 0  && client->chan_no < MAX_NO_CHANS)
    {
        /* Search for the channel requested. If available, return. */
       /* Implementation Detail: One client/channel */
        if(g_edma_channel_list[client->chan_no]->sw_in_use == FREE)
        {
            chan = g_edma_channel_list[client->chan_no];
            g_edma_channel_list[client->chan_no]->sw_in_use = IN_USE;
            g_edma_channel_list[client->chan_no]->resource.client = client;

            /*
             * Insert the channel's resource structure in the client's resource structure.
             */
            insert_element_at_end((struct list_start * )&(chan->resource),
            (struct list_start *)&client_internal->resources);

            edma_descriptors_init(chan);

        }


    }

    return chan;
}


/*
 * edma_chan_release- "releases" a channel from being
 * associated with a particular client.
 *
 * @chan: channel to release.
 *
 *
 */
void edma_chan_release(struct edma_chan * chan)
{

    struct edma_client_internal *client_internal;
    struct edma_client *client = NULL;
    struct list_start  * resource = NULL;
    struct list_start  * head = NULL;
    if(chan != NULL)
    {
        client = chan->resource.client;
        /* Struct to remove from the client's resource list. */
        resource = (struct list_start *)&(chan->resource);
        chan->resource.client = NULL;
        client_internal = (struct edma_client_internal *)((char *)client - (2 * sizeof(void *)));
        head = (struct list_start *)&(client_internal->resources);

        remove_element(resource, head);

        chan->sw_in_use = FREE;

        edma_descriptors_init(chan);
    }
}


/*
 * edma_get_functions - returns function ptrs to the
 * API.
 *
 * @funcs: a pointer to a struct to hold the API ptrs.
 *
 *
 */
int edma_get_functions(struct edma_api_export * funcs)
{
    int err = 0;
    *funcs = edma_api;
    return err;
}


/*
 * edma_descriptors_init - initializes descriptors in a descriptor pool.
 *
 * @chan: channel indicating which descriptor pool to initialize.
 *
 *
 */
void edma_descriptors_init(struct edma_chan * chan)
{

   struct edma_descriptor_internal * prev;
   edma_addr_t addr = 0;
   int i = 0;
   int internal_desc_size = sizeof(struct edma_descriptor_internal);
   struct edma_descriptor_internal * first_desc = (struct edma_descriptor_internal*)g_descriptor_lists[chan->channel_no].descriptor_pool;
   struct edma_descriptor_internal * desc = (struct edma_descriptor_internal*)g_descriptor_lists[chan->channel_no].descriptor_pool;
   g_descriptor_lists[chan->channel_no].head_index = -1;
   g_descriptor_lists[chan->channel_no].tail_index = 0;

   spin_lock(&chan->desc_lock);


   /*
    * This loop amounts to simply setting up the free descriptor list,
    * adjusting all of the pointers in individual descriptors to
    * point to the appropriate next and previous descriptors.
    *
    */

   for(i = 0; i < DESCRIPTOR_NO; i++)
   {
        memset(desc, 0, internal_desc_size);
        desc->phys = virt_to_phys(&desc->hw);
        desc->cookie = INVALID_COOKIE;
        desc->status = EDMA_ERROR;
        desc->ferr   = EDMA_ERROR;
        prev = desc;

        if(i != (DESCRIPTOR_NO - 1))
        {
            desc++;
            addr = virt_to_phys(&desc->hw);
            prev->next_ndar = (addr & LOWER_64);
            prev->next_nduar = (addr >> 32) & LOWER_64;

        }
        else
        {
            addr = virt_to_phys(&first_desc->hw);
            prev->next_ndar = (addr & LOWER_64);
            prev->next_nduar = (addr >> 32) & LOWER_64;

        }

   }

   spin_unlock(&chan->desc_lock);
}
/*
 * edma_ioctl - the ioctl function in a driver offers a method for creating
 * an entry point for specific driver commands (functions).
 *
 * NOTE: This function is used for testing mainly as part of
 * an application interface.
 *
 * @inode:
 * @filep: file descriptor
 * @cmd: command to issue
 * @arg: arguments to the commands
 *
 * RETURN: 0 or the return value of the function called by the ioctl.
 *
 */
int edma_ioctl(struct inode * inode, struct file * filep, unsigned int cmd, unsigned long arg)
{
    int ret = 0;
    int * data_array = (int *) arg;
    struct edma_memcpy_parameters * params = (struct edma_memcpy_parameters *)arg;

    switch(cmd)
    {
        case DMA_IOC_CLIENT_ALLOC_APP:
            ret = (int)(void *)edma_client_alloc_app((struct edma_client *)arg);
            break;
        case DMA_IOC_CLIENT_RELEASE:
            edma_client_release((struct edma_client *)arg);
            break;
        case DMA_IOC_CHANNEL_ALLOC_APP:
            ret = (int)(void*)edma_chan_alloc_app((struct edma_client *) data_array[0],
            (struct edma_client *) data_array[1]);
            break;
        case DMA_IOC_CHANNEL_RELEASE:
            edma_chan_release((struct edma_chan *)arg);
            break;
        case DMA_IOC_REGISTER:
            ret = edma_client_register((struct edma_client *)arg);
            break;
        case DMA_IOC_UNREGISTER:
            edma_client_unregister((struct edma_client *)arg);
            break;
        case DMA_IOC_MEMCPY:
            ret = (int)edma_memcpy_app(params->memory_ops,
            params->chan, params->dest, params->src, params->size );
            break;
        case DMA_IOC_IS_COMPLETE:
            ret = edma_is_complete((struct edma_chan *) data_array[0],
            (edma_cookie_t) ((signed long *)data_array[1]));
            break;
        case DMA_IOC_CHANNEL_PRIORITY:
            ret = edma_change_channel_priority((int)arg);
            break;
        case DMA_IOC_GET_ERRORS:
            ret = edma_get_errors((struct edma_chan *) data_array[0],
            (edma_cookie_t) ((signed long *)data_array[1]));
            break;
        default:
            printk(KERN_WARNING UNKNOWN_IOCTL,g_drvr_name);
            break;

    }
    return ret;
}
/*
 * edma_open - increments the usage count and does any prep work necessary
 * to initialize the driver.
 *
 * NOTE: The open and release functions were implemented so that
 * the unit test application could have a mechanism for opening and
 * closing the file structure associated with character drivers.
 * The functions are not used by the client driver.
 *
 * @inode: see Linux documentation for details.
 * @filep:
 *
 * RETURN: 0
 *
 */
int edma_open(struct inode * inode, struct file * filep)
{



    pci_write_config_dword(g_dev_cfg_space, EDMA_FERR_REG, EDMA_FERR_RESET);
    /*
    *  The MOD_INC_USE_COUNT is a macro for incrementing the driver
    *  reference count that is used in 2.4.x kernels but is not
    *  used in the 2.6.x kernels.
    */

    #ifdef DMA_2_4
        MOD_INC_USE_COUNT;
    #endif
    return 0;
}
/*
 * edma_release - decrements the usage count and does any prep work necessary
 * to initialize the driver.
 *
 *
 * @inode: see Linux documentation for details.
 * @filep:
 *
 * RETURN: 0
 *
 */
int edma_release(struct inode * inode, struct file * filep)
{
    /*
    *  The MOD_DEC_USE_COUNT is a macro for incrementing the driver
    *  reference count that is used in 2.4.x kernels but is not
    *  used in the 2.6.x kernels.
    */

    #ifdef DMA_2_4
       MOD_DEC_USE_COUNT;
    #endif

    return 0;
}

/*
 * release_client_resources - frees the resource structures associated with a client
 *
 * NOTE: The edma_client_internal structure is not to be confused with edma_client.
 * This function is not part of the client driver API or the application interface.
 * It is called internally.
 *
 * @client_internal: internal client
 *
 */
void release_client_resources(struct edma_client_internal *client_internal)
{
    struct edma_resource_internal *resource = NULL;

    if(client_internal != NULL )
    {
        resource = (struct edma_resource_internal *)remove_each_element_front(&client_internal->resources);
        while(resource != NULL)
        {
            edma_chan_release((struct edma_chan *)resource_to_dma(&resource->pub));
            resource = (struct edma_resource_internal*)remove_each_element_front(&client_internal->resources);

        }

    }
}
/*
 * edma_client_register - adds a client struct to an internal master client
 * list.
 *
 * RETURN: SUCCESS (0), FAILED (-1)
 * @client: edma_client created with edma_client_alloc
 *
 */
int edma_client_register(struct edma_client *client)
{
    struct edma_client_internal *client_internal;
    int ret = SUCCESS;

    if(client != NULL)
    {
        client_internal = (struct edma_client_internal *)((char *)client - (2 * sizeof(void *)));
        linked_list_reinit(&client_internal->resources);
        insert_element_at_end((struct list_start*)client_internal, &g_edma_client_list);


    }
    else
        ret = FAILED;

    return ret;
}
/*
 * edma_client_unregister - removes a client from the internal
 * master client list.
 *
 * @client: edma_client created with edma_client_alloc
 *
 */
void edma_client_unregister(struct edma_client *client)
{
    struct edma_client_internal *client_internal;

    if(client != NULL)
    {
        client_internal = (struct edma_client_internal *)((char *)client - (2 * sizeof(void *)));
        remove_element((struct list_start *)client_internal,&g_edma_client_list);
        release_client_resources(client_internal);
    }

}
/*
 * edma_get_errors - returns the contents of the FERR for a specified cookie.
 *
 *
 * RETURN: 0 = no error information was found for the particular cookie or
 * a positive integer containing the FERR error code (1st two bytes) and the
 * number of the problematic descriptor.
 *
 * EXAMPLE: a return value of 0x20014 indicates the problematic descriptor
 * was the second descriptor in the chain and the FERR value was 0x0014.
 *
 * @chan: channel allocated by edma_chan_alloc.
 * @cookie: integer cookie value returned by edma_memcpy.
 *
 */
unsigned int edma_get_errors(struct edma_chan * chan, edma_cookie_t cookie)
{
    int i = 0;
    struct edma_descriptor_internal * desc = (struct edma_descriptor_internal *) g_descriptor_lists[chan->channel_no].descriptor_pool;

    /*
     *  0 = No errors found, or there is no longer a record of that value in the descriptors.
     *  In other words, the descriptor has been re-used and the ferr value was written
     *  over.
     *
     */

    unsigned int error_value = 0;
    int ferr_value;

    spin_lock(&chan->desc_lock);
    while(i < DESCRIPTOR_NO)
    {
        if(desc->cookie == cookie)
        {
            /* Found a descriptor with an error in it! */
            if(desc->ferr != -1)
            {
                /* Place the error value and the descriptor number in the return value */
                ferr_value = desc->ferr;
                error_value = i << 16;
                ferr_value = ferr_value >> (chan->channel_no * 8);
                error_value = error_value | ferr_value;
                /* Break out of the loop */
                break;
            }
        }
        i++;
        if(i < (DESCRIPTOR_NO - 1))
            desc++;

    }


    spin_unlock(&chan->desc_lock);

    return error_value;
}

/*
 * edma_is_complete - polling function to determine if a transaction
 * has successfully completed.
 *
 * NOTE: Use this function for polling the status of a particular
 * transaction if the client has no callback function. Note, there
 * are 16 descriptors, so it is possible for cookie information to
 * expire.
 *
 * RETURN: EDMA_SUCCESS (0) (successfully completed)
 *	   EDMA_IN_PROGRESS (1)
 *	   EDMA_NO_RECORD (2) (cookie value expired)
 *         EDMA_ERROR (-1) (error occurred)
 *
 * @chan: channel allocated by edma_chan_alloc.
 * @cookie: integer cookie value returned by edma_memcpy.
 *
 */
unsigned int edma_is_complete(struct edma_chan * chan, edma_cookie_t cookie)
{
    int i = 0;
    if(chan == 0)
    {
        return(1);
    }
    struct edma_descriptor_internal * desc = (struct edma_descriptor_internal * )g_descriptor_lists[chan->channel_no].descriptor_pool;
    unsigned int error_value = EDMA_NO_RECORD;

    spin_lock(&chan->desc_lock);
    while(i < DESCRIPTOR_NO)
    {

        if(desc->cookie == cookie)
        {

            if(desc->status == EDMA_SUCCESS)
                error_value = EDMA_SUCCESS;
            else if(desc->status == EDMA_IN_PROGRESS)
                error_value = EDMA_IN_PROGRESS;

            /* Found a descriptor with an error in it! */
            if(desc->ferr != -1)
            {
                error_value = EDMA_ERROR;
                break;

            }
        }

        if(i < (DESCRIPTOR_NO - 1))
            desc++;

        i++;

    }

    spin_unlock(&chan->desc_lock);

    return error_value;


}

/*
 * edma_change_channel_priority - changes the channel arbitration scheme from
 * round robin to 50% bandwidth for a given channel.
 *
 * NOTE: Integer values 0-3 are valid channel numbers. Any other value will
 * reset the arbitration scheme to the default round robin.
 *
 * RETURN:  EDMA_CP_FAIL = -1	     //Failed to reset channel priority
 *	    EDMA_CP_DEFAULT 0         //Set channel priority to default (round robin)
 *	   EDMA_CP_VALID_CHANNEL 1    //Set user specified channel to have top priority
 *
 *
 * @channel: number of the channel to get 50% bandwidth.

 *
 */
int edma_change_channel_priority(unsigned int channel)
{
    unsigned long dgcr_val, chan_csr_val, chan_ccr_val;
    int chan_cnt;
    int ret = EDMA_CP_FAIL;


     /* If any of the channels are in the midst of a transaction,
        pause them. Resume them after setting the channel priority.  */
    for(chan_cnt = 0; chan_cnt < MAX_NO_CHANS; chan_cnt++)
    {
        chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CSR);
        if(chan_csr_val & CHAN_ACTIVE)
        {
            chan_ccr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);
            writel(chan_ccr_val | SUSPEND_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);
        }

        do
        {
            chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CSR);
        }while(!(chan_csr_val |  DMA_SUSPENDED));

    }



    dgcr_val = readl((char *)g_mm_regs + EDMA_DCGC_REG);


    /* If channel value is 0-3 set the priority, else set to default. */
    if((channel >= 0) && (channel < MAX_NO_CHANS))
    {
        /* Set channel priority */
        writel(dgcr_val | channel | CHANNEL_PRIORITY_ENABLED, (char *)g_mm_regs + EDMA_DCGC_REG);
        ret = EDMA_CP_VALID_CHANNEL;

    }
    else
    {
        /* Set channel priority to the default arbitration scheme */
        if(dgcr_val & CHANNEL_PRIORITY_ENABLED)
        {
            dgcr_val= dgcr_val & CHANNEL_PRIORITY_DISABLED;
            writel(dgcr_val, (char *)g_mm_regs + EDMA_DCGC_REG);
        }
        ret = EDMA_CP_DEFAULT;
    }

    /* Resume */
     for(chan_cnt = 0; chan_cnt < MAX_NO_CHANS; chan_cnt++)
     {
        chan_ccr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);


        if(chan_ccr_val & SUSPEND_DMA)
        {
            /* Clear the Suspend DMA bit and write the value back */
            writel(0, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);
            writel(RESUME_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);
            chan_ccr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);
            if(chan_ccr_val & SUSPEND_DMA)
            ret = EDMA_CP_FAIL;
        }
    }

    dgcr_val = readl((char *)g_mm_regs + EDMA_DCGC_REG);


    return ret;
}
/*
 * edma_irq_handler - Interrupt request handler, schedules
 * tasklets to do descriptor cleanup in the bottom half.
 *
 * NOTE: This IRQ handler is only concerned with DMA_ABORT
 * and END_OF_CHAIN events.
 *
 *
 * @irq: see Linux documentation.
 * @dev_id:
 * @reqs:
 */

#ifdef DMA_2_4
void edma_irq_handler(int irq, void * dev_id, struct pt_regs * regs)
{

    unsigned long clear_bits;
    unsigned int ferr_value;
    unsigned long chan_csr_val;
    int chan_cnt = 0;
    g_tasklet_data[0] = g_tasklet_data[1] = g_tasklet_data[2]= g_tasklet_data[3] = 0;


    while(chan_cnt < MAX_NO_CHANS)
    {
        chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CSR);
        clear_bits = chan_csr_val & (END_OF_CHAIN | END_OF_TRANSFER | DMA_ABORTED);
        writel(clear_bits, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]  + EDMA_CSR);

        if(chan_csr_val & DMA_ABORTED)
        {
            /* Print the FERR */
            pci_read_config_dword(g_dev_cfg_space, EDMA_FERR_REG, &ferr_value);
            printk("EDMA_ABORTED FERR %x \n", ferr_value);

        }
        if(chan_csr_val & END_OF_CHAIN)
        {
            /* End of chain message */
            printk("END_OF_CHAIN %x \n", (unsigned int)chan_csr_val);
        }
        g_tasklet_data[chan_cnt] = clear_bits;
        chan_cnt++;

    }
    tasklet_schedule(&status_tasklet);

}
#endif

#ifdef DMA_2_6
static irqreturn_t edma_irq_handler(int irq, void * dev_id, struct pt_regs * regs)
{

    unsigned long clear_bits;
    unsigned int ferr_value;
    unsigned long chan_csr_val;
    int chan_cnt = 0;
    int handled = 0;
    g_tasklet_data[0] = g_tasklet_data[1] = g_tasklet_data[2]= g_tasklet_data[3] = 0;


    while(chan_cnt < MAX_NO_CHANS)
    {
        chan_csr_val = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CSR);
        clear_bits = chan_csr_val & (END_OF_CHAIN | END_OF_TRANSFER | DMA_ABORTED);
        writel(clear_bits, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]  + EDMA_CSR);

        if(chan_csr_val & DMA_ABORTED)
        {
            /* Print the FERR */
            pci_read_config_dword(g_dev_cfg_space, EDMA_FERR_REG, &ferr_value);
            printk("EDMA_ABORTED FERR %x \n", ferr_value);
            handled = 1;
        }
        if(chan_csr_val & END_OF_CHAIN)
        {
            /* End of chain message */
            printk("END_OF_CHAIN %x \n", (unsigned int)chan_csr_val);
            handled = 1;
        }
        g_tasklet_data[chan_cnt] = clear_bits;
        chan_cnt++;

    }
    if(handled)
        tasklet_schedule(&status_tasklet);
    return IRQ_RETVAL(handled);
}
#endif
/*
 * status_tasklet_func - Bottom half tasklet that does EDMA_ABORT
 * and END_OF_CHAIN cleanup.
 *
 * @data: see Linux documentation.
 *
 */
void status_tasklet_func(unsigned long data)
{

    struct edma_client * client;
    unsigned long user_data = 0;
    struct edma_chan * channel ;
    edma_cookie_t cookie;
    edma_callback_t client_callback;
    struct edma_descriptor_internal * desc_array;
    unsigned long cdar;
    unsigned long cduar;
    unsigned long flags;
    unsigned int chan_cnt = 0;
    unsigned long data_array[] = {0,0,0,0};
    int desc_index = 0;
    int tail_index = 0;
    int cookie_index = 0;
    edma_addr_t physical_addr = 0;
    unsigned int ferr_value = 0;
    int i = 0;


    //need an irq spinlock
    spin_lock_irqsave(&g_tasklet_lock, flags);
    memcpy(data_array,(unsigned long *)data, sizeof(unsigned long) * MAX_NO_CHANS);
    spin_unlock_irqrestore(&g_tasklet_lock, flags);


    for(chan_cnt = 0; chan_cnt < MAX_NO_CHANS; chan_cnt++)
    {
        if(data_array[chan_cnt])
        {
            /* Get the parameters for the client callback */
            desc_array = (struct edma_descriptor_internal *)g_descriptor_lists[chan_cnt].descriptor_pool;
            client = g_edma_channel_list[chan_cnt]->resource.client;
            channel = g_edma_channel_list[chan_cnt];
            cookie = g_edma_channel_list[chan_cnt]->cookie;
            client_callback = client->edma_callback;
            g_edma_channel_list[chan_cnt]->completed_cookie = g_edma_channel_list[chan_cnt]->cookie;

            /* Obtain the next descriptor address */
            cdar = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CDAR);
            cduar = readl((char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_CDUAR);
            physical_addr = cduar;
            physical_addr = physical_addr << 32;
            physical_addr = physical_addr | cdar;

            if(data_array[chan_cnt] & END_OF_CHAIN)
            {

                /* Update cookie info - Latest completed cookie */
                g_edma_channel_list[chan_cnt]->completed_cookie = g_edma_channel_list[chan_cnt]->cookie;

                /* Correct the head and tail indices, freeing used descriptors in between */
                tail_index = g_descriptor_lists[chan_cnt].tail_index;
                cookie_index = desc_array[tail_index].cookie;
                
                do
                {

                    desc_array[tail_index].ferr = -1;
                    if(desc_array[tail_index].status == EDMA_IN_PROGRESS)
                    desc_array[tail_index].status = EDMA_SUCCESS;
                  

                    /* End of chain for a particular cookie */
                    if(cookie_index != desc_array[tail_index + 1].cookie)
                    {
                         if(client_callback != NULL)
                             (*client_callback)(client, channel, cookie_index, EDMA_SUCCESS, user_data);

                    }

                    if(physical_addr == desc_array[tail_index].phys)
                    {
                        g_descriptor_lists[chan_cnt].tail_index = 		g_descriptor_lists[chan_cnt].head_index;
                        break;
                    }

                    if(tail_index == (DESCRIPTOR_NO - 1))
                    {
                        tail_index = 0;
                        cookie_index = desc_array[tail_index].cookie;
                    }
                    else
                    {
                        tail_index++;
                        cookie_index = desc_array[tail_index].cookie;
                    }

                }while(tail_index != g_descriptor_lists[chan_cnt].head_index);

            }

            if(data_array[chan_cnt] & DMA_ABORTED)
            {

                pci_read_config_dword(g_dev_cfg_space, EDMA_FERR_REG, &ferr_value);
                pci_write_config_dword(g_dev_cfg_space, EDMA_FERR_REG, ~0);


                desc_index = -1;
                i = 0;

                /* Find the descriptor index associated with the cdar desc */
                while( i < DESCRIPTOR_NO)
                {

                    cookie_index = desc_array[i].cookie;
                    /* End of chain for a particular cookie */
                    if(cookie_index != desc_array[i + 1].cookie)
                    {
                        if(client_callback != NULL)
                            (*client_callback)(client, channel, cookie_index, EDMA_SUCCESS, user_data);


                    }


                    /* Found the index of the problematic descriptor */
                    if(desc_array[i].phys == physical_addr)
                    {
                        desc_index = i;
                        desc_array[desc_index].ferr = ferr_value;

                        if(client_callback != NULL)
                            (*client_callback)(client, channel, cookie, EDMA_ERROR, user_data);

                    }
                    /* Start checking for the end of the chain */
                    if(desc_index != -1)
                    {
                        if(desc_array[i].hw.ndar == 0 && desc_array[i].hw.nduar == 0)
                        {
                            /* Set the descriptor index to the end of the chain */
                            desc_index = i;
                            /* Stop checking */
                            break;
                        }
                    }
                    /* Reset the next descriptors pointers and cookie val */
                    if(desc_array[i].cookie == cookie)
                    {
                        desc_array[i].status = EDMA_ERROR;
                    }
                    i++;

                }


                /*
                * If the next descriptor after EOT is associated with a transaction,
                * restart the dma engine.
                *
                */
                if(desc_array[desc_index + 1].cookie != -1)
                {

                    g_descriptor_lists[chan_cnt].tail_index = desc_index + 1;
                    /* Write the next descriptor to NDAR/NDUAR and restart the operations */
                    writel(desc_array[desc_index + 1].phys & LOWER_64,
                    (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_NDAR);
                    writel((desc_array[desc_index + 1].phys >> 32) & LOWER_64,
                    (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt] + EDMA_NDUAR);
                    writel(START_DMA, (char *)g_mm_regs + g_channel_ccr_offsets[chan_cnt]);

                }
                else
                    g_descriptor_lists[chan_cnt].tail_index = g_descriptor_lists[chan_cnt].head_index;


            }
        }
    }



}

edma_version_t edma_version()
{
    return VERSION_STRING;
}


/*
 *  Prepares module for later invocations.
 */
int edma_init(void)
{

    /*
     * Initialize dev, NULL indicates that pci_find_device should
     * start at the beginning to look for the device.
     */
    int ret         = 0;
    #ifdef DMA_2_6
        int msi_result  = 0;
    #endif
    g_dev_cfg_space = NULL;
    g_mm_regs       = NULL;





     printk(KERN_INFO MESSAGE_INIT, g_drvr_name);
     /*
     * Create internal structures, which involves several kmallocs.
     * If the kmalloc calls fail, go to Exit_Error.
     */
    if(create_internal_structures())
    {
        printk(KERN_INFO"%s:Couldn't allocate structures properly\n",
           g_drvr_name);
        goto Exit_Error;
    }
    /*
    *  Print a kernel message acknowledging the loading of the driver
    */


   /*
    * Devices are identified internally by a file structure.
    * The kernel uses file_operations to access a driver's functions.
    * SET_MODULE_OWNER sets the owner field in the file_operations structure.
    *
    */
    #ifdef DMA_2_4
        SET_MODULE_OWNER(&edma_fileops);
    #endif
    /*
    *  Find the DMA device - there should only be one.
    */
    g_dev_cfg_space = pci_find_device(EDMA_VENDOR_ID, EDMA_DEVICE_ID, g_dev_cfg_space);


    /*
     * If the device was found, proceed with enabling.
     */
    if(g_dev_cfg_space != NULL)
    {
        pci_enable_device(g_dev_cfg_space);

       /*
        * Set the enable bit in the control register.
        * Reset the FERR (First Error Register), write all 1's.
        */


        if(pci_write_config_byte(g_dev_cfg_space, EDMA_CTL_REG, EDMA_CTL_ENABLE) < 0)
        {
            printk(KERN_INFO"%s:Couldn't initialize EDMA_CTL_REG\n",
            g_drvr_name);
            goto Exit_Error;
        }

        if(pci_write_config_dword(g_dev_cfg_space, EDMA_FERR_REG, EDMA_FERR_RESET) < 0)
        {
            printk(KERN_INFO"%s:Couldn't initialize EDMA_FERR_REG\n",
            g_drvr_name);
            goto Exit_Error;
        }

        if(pci_write_config_word(g_dev_cfg_space, EDMA_PCICMD, EDMA_CMD_VAL) < 0)
        {
            printk(KERN_INFO"%s:Couldn't initialize EDMA_PCICMD\n",
            g_drvr_name);
            goto Exit_Error;
        }

       /*
        *   Register the device as a character driver.
        *   NOTE: Here, we opted not to obtain a dynamic allocation
        *   of the major number. Dynamic allocation affects the
        *   ability to use load-on-command later on.
        */
        if(register_chrdev(MAJOR_NO, g_char_drvr_name, &edma_fileops) < SUCCESS)
        {
            printk(KERN_CRIT ERROR_REGISTRATION, g_drvr_name);
            goto Exit_Error;
        }

        if(pci_request_region(g_dev_cfg_space, EDMA_BAR0, g_char_drvr_name) == SUCCESS)
        {
            /*
             *  Obtain a virtual address for the memory mapped registers.
             *  The ioremap function makes bus memory CPU accessible via
             *  the readb/readw/readl/writeb/writew/writel functions.
             */
            g_mm_regs = (struct edma_mm_regs *)pci_resource_start(g_dev_cfg_space, 0);
            g_mm_regs = ioremap((unsigned long)g_mm_regs, sizeof(struct edma_mm_regs));

            if(g_mm_regs == NULL)
            {
                printk(KERN_WARNING ERROR_MM2, g_drvr_name);
                goto Exit_Error;
            }

        }
        else
        {
            printk(KERN_WARNING ERROR_MM1, g_drvr_name);
            goto Exit_Error;
        }

       #ifdef DMA_2_6
           if((msi_result = pci_enable_msi(g_dev_cfg_space)))
           {
               printk(KERN_NOTICE "MSI support not available: %i.\n", msi_result);
           }
           else
           {
               printk(KERN_NOTICE "MSI Interrupt vector: %i.\n", g_dev_cfg_space->irq);
           }
       #endif

        /*
         * Obtain a (shared) Interrupt Request (IRQ) Line from the OS.
         */
        if (request_irq(g_dev_cfg_space->irq, &edma_irq_handler, SA_SHIRQ,
                         g_char_drvr_name, g_dev_cfg_space) )
        {
            printk(KERN_WARNING ERROR_MM1, g_drvr_name);
            goto Exit_Error;
        }

        edma_change_channel_priority(edma_chan_priority);

    }
    else
    {
        printk(KERN_WARNING ERROR_DEVICE_NOT_FOUND, g_drvr_name);
        goto Exit_Error;
    }

    /* Intermodule Communication: Exposing the Client Driver API */
    #ifdef DMA_2_4
        inter_module_register("edma_get_functions", THIS_MODULE, edma_get_functions);
    #endif

    goto Exit;


/* A serious error occurred. */
Exit_Error:
    edma_cleanup();
    ret = -ENODEV;

Exit:
    printk(KERN_INFO MESSAGE_INIT_COMPLETE, g_drvr_name);

    return ret;

}
/*
 *  Invoked before the module is unloaded.
 */
void edma_cleanup(void)
{
   /*
    *  Print a kernel message acknowledging the unloading of the driver
    */
    printk(KERN_INFO MESSAGE_CLEANUP, g_drvr_name);

    free_internal_structures();


    if(g_dev_cfg_space != NULL)
    {
        /* free the IRQ line */
        free_irq(g_dev_cfg_space->irq, g_dev_cfg_space);
    }

    #ifdef DMA_2_6
        pci_disable_msi(g_dev_cfg_space);
    #endif

    if(g_mm_regs)
    {
        /* free all pci-related resources */
        iounmap(g_mm_regs);
    }

    if(g_dev_cfg_space != NULL)
    {
        pci_release_region(g_dev_cfg_space, 0);
        pci_disable_device(g_dev_cfg_space);
    }

    /* If the unregister_chrdev fails, it is a serious problem,
     * as the next invocation of insmod will produce errors.
     */
    if(unregister_chrdev(MAJOR_NO, g_char_drvr_name) < SUCCESS)
    {
            printk(KERN_WARNING ERROR_DEREGISTRATION, g_drvr_name);
    }
    else
    {

            printk(KERN_INFO MESSAGE_DEREGISTRATION, g_drvr_name);
    }


    #ifdef DMA_2_4
        inter_module_unregister("edma_get_functions");
    #endif
}
#ifdef DMA_2_6
EXPORT_SYMBOL(edma_get_functions);
#endif

module_init(edma_init);
module_exit(edma_cleanup);







