/**
 * @file IxLinuxEDDIoctl.h
 *
 * @brief Header file for the implementation of the IOCTLs features
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright (c) 2004-2007  Intel Corporation. All Rights Reserved. 
 * 
 * @par 
 * This software program is licensed subject to the GNU
 * General Public License (GPL). Version 2, June 1991, available at
 * http://www.fsf.org/copyleft/gpl.html
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * @par
 * -- End Intel Copyright Notice --
 */
 
#ifndef IX_LINUX_EDD_IOCTL_H_INCLUDED
#define IX_LINUX_EDD_IOCTL_H_INCLUDED

/**
 *
 * @defgroup IxLinuxEDDIoctl Linux IXP400 Ethernet Device Driver Private IOCTLs
 *
 * @brief information for the Linux EDD private IOCTLs
 */

/** @brief Definition of a physical port ID */
typedef u32 PortId;

/** @brief Definition for QoS traffic classes */
typedef u32 IxTrafficClass;

/** @brief Fast Path Flow ID */
typedef u32 IxFPathFlowId;

/** @brief Fast Path Classification Template */
typedef u32 IxFPathClsTemplate;


/**
 * @def IX_IOCTL_MHDR_OFFSET
 *
 * @brief The Ethernet frame header offset value is two
 */
#define IX_IOCTL_MHDR_OFFSET 2

/**
 * @def IX_IOCTL_QOS
 *
 * @ref LINUX sockios.h
 *
 * @brief The Device private ioctl for QoS configuration
 */
#define IX_IOCTL_QOS (SIOCDEVPRIVATE+3)

/**
 * @def IX_IOCTL_FPATH
 *
 * @ref LINUX sockios.h
 *
 * @brief The Device private ioctl for FastPath configuration
 */
#define IX_IOCTL_FPATH (SIOCDEVPRIVATE+4)

/**
 * @def IX_IOCTL_MISCFEATURES
 *
 * @ref LINUX sockios.h
 *
 * @brief The Device private ioctl for specific features configuration
 */
#define IX_IOCTL_MISCFEATURES (SIOCDEVPRIVATE+5)

/**
 * @def IX_IOCTL_MIN_IEEE802_3_ETH_TYPE
 *
 * @brief Minimum value of ethernet type, 
 *        0000-05DC	IEEE802.3 Length Field (0.:1500.)
 */
#define IX_IOCTL_MIN_IEEE802_3_ETH_TYPE (0x5DC)

/**
 * @def IX_IOCTL_MAX_IEEE802_3_ETH_TYPE
 *
 * @brief Maximum value of ethernet type
 */
#define IX_IOCTL_MAX_IEEE802_3_ETH_TYPE (0xFFFF)

/**
 * @def IX_IOCTL_MAX_FPDATA_CHANNELS
 *
 * @brief Maximum Fast path channels supported
 */
#define IX_IOCTL_MAX_FPDATA_CHANNELS 	(10)

/**
 * @def IX_IEEE803_MAC_ADDRESS_SIZE
 * @brief The size of the MAC address
 */
#define IX_IEEE803_MAC_ADDRESS_SIZE 	(6)

/**
 *
 * @brief The IEEE 802.3 Ethernet MAC address structure.
 * 
 * The data should be packed with bytes xx:xx:xx:xx:xx:xx 
 *
 * @note The data must be packed in network byte order.
 */
typedef struct  
{
   u8 macAddress[IX_IEEE803_MAC_ADDRESS_SIZE];
} __attribute__((__packed__)) IxMacAddr;


/**
 *
 * @brief The IPv4 address structure.
 * 
 * The data should be packed with bytes xx.xx.xx.xx
 *
 * @note The data must be packed in network byte order.
 */
typedef struct  
{
   u8 v4Address[4];
} __attribute__((__packed__)) IxIPv4Addr;


/**
 *
 * @brief The IPv6 address structure.
 * 
 * The data should be packed with bytes xx:xx:xx:xx:xx:xx:xx:xx
 *
 * @note The data must be packed in network byte order.
 */
typedef struct  
{
   u8 v6Address[16];
} __attribute__((__packed__)) IxIPv6Addr;


/**
 * @enum IxQoSCFGType
 * 
 * @brief Definition for QoS types @ref IxEthDB component
 */
typedef enum  {	
	TYPE_QOS_DSCP		= 1,	/**< IPv4 or IPv6 DSCP QoS */
	TYPE_QOS_VLAN		= 2,	/**< IEEE802.1P VLAN Priority */
	TYPE_QOS_ETHTYPE	= 4		/**< Ethernet Header-EthType */
}IxQoSCFGType;


/**
 * @enum IxQosIOAct
 *
 * @brief Definition for IX_IOCTL_QOS ioctl action types
 */
typedef enum  {	
	CMD_QOSCFG_SET		= 301,	/**< Configure traffic class for DSCP/VLAN priority/EtherType */
	CMD_QOSCFG_DELETE	= 302,	/**< delete the configuration of traffic class */
	CMD_QOSCFG_CLEAR	= 303,	/**< clear the traffic class policy */
	CMD_QOSCFG_GETINFO	= 304,	/**< get the information of traffic class */
	CMD_QOSCFG_SHOW		= 305	/**< Display Rx QoS classification configuration */
}IxQosIOAct;


/**
 *
 * @enum IxFastPathEntryType
 * 
 * @brief Definition for the Fast Path rule types
 */
typedef enum  {	
	TYPE_CLASSIFIER	= 0,	/**< Classification rule entry */
	TYPE_MODIFIER	= 1		/**< Modification rule entry */
}IxFastPathEntryType;


/**
 *
 * @enum IxFPathDeletionType
 * 
 * @brief Definition for the Fast path deletion type
 */
typedef enum{
    TYPE_DELETION_FLOW 		= 0,
    TYPE_DELETION_TEMPLATE 	= 1
} IxFPathDeletionType;


/**
 * @enum IxFPathProcessingType
 *
 * @brief Fast Path Traffic Processing Type. This value is used to
 *        keep track of the operation assigned to a Fast Path Flow. 
 */
typedef enum
{
    TYPE_PROCESS_IPv4		= 0, /**< Fast Path IPv4 Routing */
    TYPE_PROCESS_IPv6		= 1, /**< Fast Path IPv6 Routing */  
    TYPE_PROCESS_NAT		= 2, /**< Fast Path Basic NAT */   
    TYPE_PROCESS_NAPT		= 3, /**< Fast Path NAPT */
    TYPE_PROCESS_PPPOE		= 4, /**< Fast Path PPPoE */
    TYPE_PROCESS_PPPOENAT	= 5, /**< Fast Path PPPoE+NAT */
    TYPE_PROCESS_PPPOENAPT	= 6, /**< Fast Path PPPoE+NAPT */
    TYPE_PROCESS_USERDEFINED= 7  /**< User-defined Fast Path operation */
} IxFPathProcessingType;


/**
 *
 * @enum IxFPathTrafficDirection
 *
 * @brief Definition for Fast Path traffic direction
 */
typedef enum  
{
   TYPE_TRAFFIC_INBOUND    = 0,  /**< WAN to LAN Fast Path traffic*/
   TYPE_TRAFFIC_OUTBOUND   = 1,  /**< LAN to WAN Fast Path traffic*/
   TYPE_TRAFFIC_DIRECTION_MAX = 2   /**< Limiter*/
} IxFPathTrafficDirection;


/** @enum IxFPathChannelIdx
 * @brief Fast Path channle index type
*/
typedef enum
{
  IX_FPATH_CHANNEL0	= 0, /**< Fast Path Channel-0 */
  IX_FPATH_CHANNEL1	= 1, /**< Fast Path Channel-1 */
  IX_FPATH_CHANNEL2	= 2, /**< Fast Path Channel-2 */
  IX_FPATH_CHANNEL3	= 3, /**< Fast Path Channel-3 */
  IX_FPATH_CHANNEL4	= 4, /**< Fast Path Channel-4 */
  IX_FPATH_CHANNEL5	= 5, /**< Fast Path Channel-5 */
  IX_FPATH_CHANNEL6	= 6, /**< Fast Path Channel-6 */
  IX_FPATH_CHANNEL7	= 7, /**< Fast Path Channel-7 */
  IX_FPATH_CHANNEL_MAX   /**< Maximum Fast Path Channel = 10 */
} IxFPathChannelIdx;


/**
 *
 * @brief The Fast Path Modification template
 *
 */
typedef struct {
    u16 opCode;              /**< Modification operation: <br>
                              * 0x01 IPv4 Processing (IP Routing) <br>
                              * 0x02 IPv6 Processing (IP Routing) <br>
                              * 0x05 IPv4 TTL update + Inbound Basic NAT <br>
                              * 0x09 IPv4 TTL update + Outbound Basic NAT <br>
                              * 0x11 IPv4 TTL update + Inbound NAPT <br>
                              * 0x21 IPv4 TTL update + Outbound NAPT <br>
                              * 0x41 IPv4 TTL update + Inbound PPPoE <br>
                              * 0x81 IPv4 TTL update + Outbound PPPoE <br>
                              * 0x45 IPv4 TTL update + Inbound PPPoE + Inbound NAT <br>
                              * 0x89 IPv4 TTL update + Outbound PPPoE + Outbound NAT <br>
                              * 0x51 IPv4 TTL update + Inbound PPPoE + Inbound NAPT <br>
                              * 0xA1 IPv4 TTL update + Outbound PPPoE + Outbound NAPT <br> */ 
    u16 offset;              /**< The offset between the end of Ethernet header 
                              *  and the start of IP header */
    u16 pppoeSession;        /**< PPPoE session ID for PPPoE Header insertion 
                              *  in outbound PPPoE */
    u16 rsvd0;               /**< Reserved */
    u32 destMac03[4];        /**< Byte0-3 of Dest MAC Address for Fast Path Flow-0~3 */
    u16 destMac45[4];        /**< Byte4-5 of Dest MAC Address for Fast Path Flow-0~3 */
    IxIPv4Addr ipAddr[4];    /**< IP address of for Fast Path Flow-0~3 (Basic NAT and NAPT) <br>
                              *  For inbound operation, IP address = local node private IP. <br>
                              *  Dest IP of the Fast Path = IP address. <br>
                              *  For outbound operation, IP address = local gateway public IP. <br>
                              *  Source IP of the Fast Path = IP address. <br> */
    u16 l4Port[4];           /**< TCP/UDP Port number of the modified TCP/UDP Flow-0~3 (NAPT) <br>
                              *  For inbound operation, port number = local node private port. <br>
                              *  Dest TXP/UDP port of the Fast Path = port number. <br>
                              *  For outbound operation, port number = local gateway public port. <br>
                              *  Source IP of the Fast Path = port number. <br> */
    u16 ipChecksumDiff[4];   /**< IP header checksum difference update of Fast Path Flow-0~3
                              * (Basic NAT and NAPT) */
    u16 l4ChecksumDiff[4];   /**< TCP/UDP checksum difference update of Fast Path Flow-0~3 
                              * (Basic NAT and NAPT) */
} IxFPathMdfTemplate;


/**
 *
 * @brief The Fast Path IPv4 Routing configuration
 */
typedef struct
{
    IxIPv4Addr   localNodePublicIPAddr;   /**< Public IPv4 address of local node*/    
    IxIPv4Addr   remoteNodePublicIPAddr;  /**< Public IP Address of remote node*/
    IxMacAddr    localNodeMacAddr;        /**< MAC address of local node*/
    u8           ttlLowerLimit;           /**< Lower limit of IPv4 TTL */
    u8           reserv0;                 /**< Reserve field for word-aligned */
    IxMacAddr    remoteGwMacAddr;         /**< MAC address of remote gateway */
    u16          reserv1;                 /**< Reserve field for word-aligned */
} IxFastPathIPv4Config;


/**
 *
 * @brief The Fast Path IPv6 Routing configuration
 */
typedef struct
{
    IxIPv6Addr   localNodePublicIPAddr;   /**< Public IPv6 address of local node*/
    IxIPv6Addr   remoteNodePublicIPAddr;  /**< Public IPv6 address of remote node */
    IxMacAddr    localNodeMacAddr;        /**< MAC address of local node*/
    u8           hlLowerLimit;            /**< Lower limit of IPv6 Hop Limit */
    u8           reserv0;                 /**< Reserve field for word-aligned */
    IxMacAddr    remoteGwMacAddr;         /**< MAC address of remote gateway */
    u16          reserv1;                 /**< Reserve field for word-aligned */
} IxFastPathIPv6Config;


/**
 *
 * @brief The Fast Path NAT configuration
 */
typedef struct
{
    IxIPv4Addr   localNodePrivateIPAddr;  /**< Private IPv4 address of local node*/
    IxIPv4Addr   localGwPublicIPAddr;     /**< Public IP address of local gateway 
                                            *  assigned by service provider */
    IxIPv4Addr   remoteNodePublicIPAddr;  /**< Public IPv4 Address of remote node */
    IxMacAddr    localNodeMacAddr;        /**< MAC address of local node*/
    u8           ttlLowerLimit;           /**< Lower limit of IPv4 TTL */
    u8           reserv0;                 /**< Reserve field for word-aligned */
    IxMacAddr    remoteGwMacAddr;         /**< MAC address of remote gateway */        
    u16          reserv1;                 /**< Reserve field for word-aligned */
} IxFastPathNATConfig;


/** @enum IxEthDBFPathChannelIdx
 * @brief Fast Path channle index type
*/
typedef enum
{
  IX_FPATH_L4_TCP=0x06, /**< L4 protocol is TCP */
  IX_FPATH_L4_UDP=0x11, /**< L4 protocol is UDP */
  IX_FPATH_L4_NO =0xFF  /**< L4 protocol is not supported */
} IxFastPathL4Type;


/**
 *
 * @brief The Fast Path NAPT configuration
 */
typedef struct
{
    IxIPv4Addr   localNodePrivateIPAddr;   /**< Private IPv4 address of local node*/    
    IxIPv4Addr   localGwPublicIPAddr;      /**< Public IPv4 address of local gateway 
                                                     assigned by service provider */
    IxIPv4Addr   remoteNodePublicIPAddr;   /**< Public IPv4 Address of remote node */
    IxMacAddr    localNodeMacAddr;         /**< MAC address of local node*/
    u8           ttlLowerLimit;            /**< Lower limit of IPv4 TTL */
    u8           L4ProtoType;              /**< L4 Protocol Type (IxEthDBFPathL4Type) */ 
    IxMacAddr    remoteGwMacAddr;          /**< MAC address of remote gateway */
    u16          localNodePrivatePort;     /**< Private TCP/UDP Port of local node */    
    u16          localGwPublicPort;        /**< Public TCP/UDP Port of local gateway mapped 
                                                to private port of local node for NAPT*/
    u16          remoteNodePublicPort;     /**< Public TCP/UDP Port of remote node*/
} IxFastPathNAPTConfig;


/**
 *
 * @brief The Fast Path PPP over Ethernet configuration
 */
typedef struct
{
    IxIPv4Addr   localNodePublicIPAddr;   /**< Public IPv4 address of local node*/    
    IxIPv4Addr   remoteNodePublicIPAddr;  /**< Public IPv4 Address of remote node */
    IxMacAddr    localNodeMacAddr;        /**< MAC address of local node*/
    u8           ttlLowerLimit;           /**< Lower limit of IPv4 TTL */
    u8           reserv0;                 /**< Reserve field for word-aligned */
    IxMacAddr    remoteGwMacAddr;         /**< MAC address of remote gateway */
    u16          pppSessionId;            /**< PPPoE Session ID */ 
} IxFastPathPPPoEConfig;


/**
 *
 * @brief The Fast Path PPP over Ethernet NAT
 */
typedef struct
{
    IxIPv4Addr   localNodePrivateIPAddr;  /**< Private IPv4 address of local node*/
    IxIPv4Addr   localGwPublicIPAddr;     /**< Public IP address of local gateway 
                                            *  assigned by service provider */
    IxIPv4Addr   remoteNodePublicIPAddr;  /**< Public IPv4 Address of remote node */
    IxMacAddr    localNodeMacAddr;        /**< MAC address of local node*/
    u8           ttlLowerLimit;           /**< Lower limit of IPv4 TTL */
    u8           reserv0;                 /**< Reserve field for word-aligned */
    IxMacAddr    remoteGwMacAddr;         /**< MAC address of remote gateway */        
    u16          pppSessionId;            /**< PPPoE Session ID */ 
} IxFastPathPPPoENATConfig;


/**
 *
 * @brief The Fast Path PPP over Ethernet NAPT
 */
typedef struct
{
    IxIPv4Addr   localNodePrivateIPAddr;   /**< Private IPv4 address of local node*/    
    IxIPv4Addr   localGwPublicIPAddr;      /**< Public IPv4 address of local gateway 
                                                assigned by service provider */
    IxIPv4Addr   remoteNodePublicIPAddr;   /**< Public IPv4 Address of remote node */
    IxMacAddr    localNodeMacAddr;         /**< MAC address of local node*/
    u8           ttlLowerLimit;            /**< Lower limit of IPv4 TTL */
    u8           L4ProtoType;              /**< L4 Protocol Type (IxEthDBFPathL4Type) */ 
    IxMacAddr    remoteGwMacAddr;          /**< MAC address of remote gateway */
    u16          pppSessionId;             /**< PPPoE Session ID */ 
    u16          localNodePrivatePort;     /**< Private TCP/UDP Port of local node */    
    u16          localGwPublicPort;        /**< Public TCP/UDP Port of local gateway mapped 
                                                to private port of local node for NAPT*/
    u16          remoteNodePublicPort;     /**< Public TCP/UDP Port of remote node*/
    u16          reserv;                   /**< Reserve field for word-aligned */
} IxFastPathPPPoENAPTConfig;


/**
 *
 * @brief The Fast Path Flow configuration
 */
typedef struct
{
    IxFPathProcessingType  fpProcType; /**< Fast Path traffic processing type */
    /**
     *
     * @brief The Fast Path Flow types
     */
    union
    {
        IxFastPathIPv4Config      ipv4;      /**< IPv4 Configuration */
        IxFastPathNATConfig       ipv4nat;   /**< NAT Configuration */
        IxFastPathNAPTConfig      ipv4napt;  /**< NAPT Configuration */
        IxFastPathPPPoEConfig     pppoe;     /**< PPPoE Configuration */
        IxFastPathPPPoENATConfig  pppoenat;  /**< PPPoENAT Configuration */
        IxFastPathPPPoENAPTConfig pppoenapt; /**< PPPoENAPT Configuration */
        IxFastPathIPv6Config      ipv6;      /**< IPv6 Configuration */
    } flow;
} IxFPathFlowConfig;


/**
 * @enum IxFastPathIOAct
 *
 * @brief Definition for IX_IOCTL_FPATH ioctl action types
 */
typedef enum  {	
	CMD_FASTPATH_ADD		= 401,	/**< add one specific Fast Path entry */
	CMD_FASTPATH_DELETE		= 402,	/**< delete one specific Fast Path entry */
	CMD_FASTPATH_GETINFO	= 403,	/**< get one specific Fast Path entry */
	CMD_FASTPATH_SHOW		= 404,	/**< Display specific Fast Path flow entry */
	CMD_FASTPATH_ENABLE		= 405,	/**< Enable Fast Path feature */
	CMD_FASTPATH_DISABLE	= 406	/**< Disable Fast Path feature */
}IxFastPathIOAct;


/**
 * @enum IxMiscFeatureIOAct
 *
 * @brief Definition for IX_IOCTL_MISCFEATURES ioctl action types
 */
typedef enum  {	
	CMD_MISCFEATURE_FEATURESET_ENABLE	= 501,  /**< enables one specific NPE Ethernet feature */
	CMD_MISCFEATURE_FEATURESET_DISABLE	= 502,  /**< disables one specific NPE Ethernet feature */
	CMD_MISCFEATURE_FEATURESET_GETINFO	= 503,  /**< get the information about NPE Ethernet features */
	
	CMD_MISCFEATURE_MODUPGRADE_TRUE		= 511,  /**< enables module uploading with NPE reset */
	CMD_MISCFEATURE_MODUPGRADE_FALSE	= 512,  /**< disables module uploading with NPE reset */
	CMD_MISCFEATURE_MODUPGRADE_GETINFO	= 513,  /**< get the information about the result of module uploading */
		
	CMD_MISCFEATURE_EDBLEARNING_ENABLE	= 521,  /**< enables NPE learning & filtering */
	CMD_MISCFEATURE_EDBLEARNING_DISABLE	= 522,  /**< disables NPE learning & filtering */
	CMD_MISCFEATURE_EDBLEARNING_GETINFO	= 523   /**< get the information about the NPE learning & filtering */
}IxMiscFeatureIOAct;


/**
 *
 * @enum IxEthFeatureSet
 * 
 * @brief Definition for NPE Ethernet feature sets @ref IxEthDB component
 */
typedef enum
{
  IX_ETH_FEATURE_LEARNING				= 0x01,  /**< Learning feature; enables EthDB to learn MAC address (filtering) records, including 802.1Q enabled records */
  IX_ETH_FEATURE_FILTERING				= 0x02,  /**< Filtering feature; enables EthDB to communicate with the NPEs for downloading filtering information in the NPEs; depends on the learning feature */
  IX_ETH_FEATURE_FPATH_TRAFFIC			= 0x80,  /**< Fast path traffic */
  IX_ETH_FEATURE_SPATH_TRAFFIC			= 0x100  /**< Slow path traffic */
} IxEthFeatureSet;



/**
 *
 * @brief The request structure used for IX_IOCTL_QOS ioctl
 * 
 * Used to configure/retrieve QoS configuration 
 *
 * @note This data should be packed as the member ifru_data for struct ifreq (if.h)
 */
typedef struct  {
    IxQosIOAct		actType;			/**< action commands: Set/Delete/Clear/Get */
    IxQoSCFGType	cfgType;			/**< configured types: DSCP/VLAN/EtherType */
    PortId			pid;				/**< physical port id */
    u32				cfgValue;			/**< configured value */
	IxTrafficClass	tcMap;				/**< traffic class map */
    char			portName[IFNAMSIZ];	/**< interface name: ixp0/ixp1/ixp2 */
}   IxQosIOArgument;



/**
 *
 * @brief The request structure used for IX_IOCTL_FPATH ioctl
 * 
 * Used to configure/retrieve FastPath configuration 
 *
 * @note The data should be packed as the member ifru_data for struct ifreq (if.h)
 */
typedef struct  {
	PortId	pid;					/**< physical port id */
	char	portName[IFNAMSIZ];		/**< interface name: ixp0/ixp1/ixp2 */
    IxFastPathIOAct		actType;	/**< action commands: Add/Delete/Show/Get */
    IxFastPathEntryType	entryType;	/**< configured types: modification/classication */
    IxFPathProcessingType procType; /**< Traffic Processing Type: NAT/NAPT/PPPOE... */
    IxFPathDeletionType delType;	/**< deletion type: TEMPLATE/FLOW */
    IxTrafficClass tcValue;			/**< QoS traffic classes value */
    IxFPathTrafficDirection tfDir;  /**< traffic direction: INBOUND/OUTBOUND */
    u16 upperLen;					/**< max frame length */
    IxFPathFlowId flowId;           /**< flow id */
    IxFPathChannelIdx channelIdx;	/**< Channel id */
    IxFPathClsTemplate *clsTemplate;/**< user define classification template */
    u32 clsTemplateSizeInWord;      /**< size of classification template */
    IxFPathMdfTemplate mdfTemplate; /**< user define modification template */
    IxFPathFlowConfig flowConfig;   /**< Traffic flow configuration */
    BOOL parentFlag;                /**< Parent or child template flag */
}IxFastPathIOArgument;


/**
 *
 * @brief The request structure used for IX_IOCTL_MISCFEATURES ioctl
 * 
 * Used to configure/retrieve the configuration of NPE features 
 *
 * @note The data should be packed as the member ifru_data for struct ifreq (if.h)
 */
typedef struct  {
    IxMiscFeatureIOAct	actType;	/**< action commands: Enable/Disable/Get */
    PortId	pid;					/**< physical port id */
	u32		hdr_Offset;				/**< value of header offset */
	u32		FeatureSet;				/**< a specific NPE Ethernet feature */
	BOOL	FeaturePresent;			/**< specific feature is present? TRUE/FALSE */
	BOOL	FeatureEnable;			/**< specific feature is enabled? TRUE/FALSE */
	char	portName[IFNAMSIZ];		/**< interface name: ixp0/ixp1/ixp2 */
}   IxMiscFeatureIOArgument;

/**
 * @} IxLinuxEDDIoctl
 */
 
#endif	/* IX_LINUX_EDD_IOCTL_H_INCLUDED */
