/*
 * File: ixp400_eth.c
 *
 * Author: Intel Corporation
 *
 * IXP400 Ethernet Driver for Linux
 *
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright (c) 2004-2007  Intel Corporation. All Rights Reserved. 
 * 
 * @par 
 * This software program is licensed subject to the GNU
 * General Public License (GPL). Version 2, June 1991, available at
 * http://www.fsf.org/copyleft/gpl.html
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

/* 
 * DESIGN NOTES:
 * This driver is written and optimized for Intel Xscale technology.
 *
 * SETUP NOTES:
 * By default, this driver uses predefined MAC addresses.
 * These are set in global var 'default_mac_addr' in this file.
 * If required, these can be changed at run-time using
 * the 'ifconfig' tool.
 *
 * Example - to set ixp0 MAC address to 00:02:B3:66:88:AA, 
 * run ifconfig with the following arguments:
 *
 *   ifconfig ixp0 hw ether 0002B36688AA
 *
 * (more information about ifconfig is available thru ifconfig -h)
 *
 * Example - to set up the ixp1 IP address to 192.168.10.1
 * run ifconfig with the following arguments:
 * 
 * ifconfig ixp1 192.168.10.1 up
 *
 */

/*
 * System-defined header files
 */
#include <linux/autoconf.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/mii.h>
#include <linux/socket.h>
#include <linux/cache.h>
#include <asm/io.h>
#include <asm/errno.h>
#include <asm/mach-types.h>
#include <net/pkt_sched.h>
#include <net/ip.h>
#include <linux/sysctl.h>
#include <linux/unistd.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>

#ifdef CONFIG_XFRM
#include <net/xfrm.h>
#endif

/*
 * Intel IXP400 Software specific header files
 */
#include <IxQMgr.h>
#include <IxEthAcc.h>
#include <IxEthMii.h>
#include <IxEthNpe.h>
#include <IxNpeDl.h>
#include <IxNpeMh.h>
#include <IxFeatureCtrl.h>
#include <IxVersionId.h>
#include <IxOsal.h>
#include <IxQueueAssignments.h>
#include <IxErrHdlAcc.h>
#include <IxParityENAcc.h>

#ifdef CONFIG_IXP400_ETH_DB
#include <IxEthDB.h>
#include "IxLinuxEDDIoctl.h"
#endif

#ifdef CONFIG_XSCALE_PMU_TIMER
/* We want to use interrupts from the XScale PMU timer to
 * drive our NPE Queue Dispatcher loop.  But if this #define
 * is set, then it means the system is already using this timer
 * so we cannot.
 */
#error "XScale PMU Timer not available (CONFIG_XSCALE_PMU_TIMER is defined)"
#endif

/*
 * Module version information
 */
MODULE_DESCRIPTION("IXP400 NPE Ethernet driver");
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Intel Corporation");
#define MODULE_NAME "ixp400_eth"
#define MOD_VERSION "1.8"

/* 
 * Non-user configurable private variable 
 */

#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
/* NPE-A enabled flag for parity error detection configuration */
static IxParityENAccConfigOption parity_npeA_enabled = IX_PARITYENACC_DISABLE;
/* NPE-B enabled flag for parity error detection configuration */
static IxParityENAccConfigOption parity_npeB_enabled = IX_PARITYENACC_DISABLE;
/* NPE-C enabled flag for parity error detection configuration */
static IxParityENAccConfigOption parity_npeC_enabled = IX_PARITYENACC_DISABLE;
/* Flag to determine if NPE softreset is being initialized */
static u32 npe_error_handler_initialized = 0;
#endif

/*
 * Global variables
 */
static BOOL module_upgrade_shutdown = FALSE;
static int ethdb_mac_learning = 0;  /* default : NPE learning & filtering disable */

/* 
 * Module parameters 
 */
static int log_level = 0;         /* default : no log */
static int no_ixp400_sw_init = 0; /* default : init core components of the IXP400 Software */
static int no_phy_scan = 0;       /* default : do phy discovery */
static int phy_reset = 0;         /* default : no phy reset */
static int npe_error_handler = 0; /* default : no npe error handler */
static int hss_coexist = 0;	  /* default : HSS coexist disabled */
static int module_upgrade_init = 0; 	  /* default : normal loading */

#ifdef CONFIG_IXP400_ETH_DB
static int customized_eth_hdr_a = 0;	  /* default : customized 16-byte Ethernet
				             Header for NPE-A */
static int customized_eth_hdr_b = 0;	  /* default : customized 16-byte Ethernet
				             Header for NPE-B */
static int customized_eth_hdr_c = 0;	  /* default : customized 16-byte Ethernet
				             Header for NPE-C */
#endif

/* 
 * maximum number of ports supported by this driver ixp0, ixp1 ....
 * The default is to configure all ports defined in EthAcc component
 */
#if defined (CONFIG_IXP400_ETH_NPEA_ONLY) || \
    defined (CONFIG_IXP400_ETH_NPEB_ONLY) || \
    defined (CONFIG_IXP400_ETH_NPEC_ONLY)
static int dev_max_count = 1; /* only one NPE used */
#elif defined (CONFIG_ARCH_IXDP425) || defined(CONFIG_MACH_IXDPG425)\
      || defined (CONFIG_ARCH_ADI_COYOTE) || defined (CONFIG_MACH_KIXRP435)
static int dev_max_count = 2; /* only NPEB and NPEC or NPEA and NPEC on IXP43X*/
#elif defined (CONFIG_ARCH_IXDP465) || defined(CONFIG_MACH_IXDP465)
static int dev_max_count = 3; /* all NPEs are used */
#endif

module_param(log_level, int, 0);
MODULE_PARM_DESC(log_level, "Set log level: 0 - None, 1 - Verbose, 2 - Debug");
module_param(no_ixp400_sw_init, int, 0);
MODULE_PARM_DESC(no_ixp400_sw_init, "If non-zero, do not initialise Intel IXP400 Software Release core components");
module_param(no_phy_scan, int, 0);
MODULE_PARM_DESC(no_phy_scan, "If non-zero, use hard-coded phy addresses");
module_param(phy_reset, int, 0);
MODULE_PARM_DESC(phy_reset, "If non-zero, reset the phys");
module_param(dev_max_count, int, 0);
MODULE_PARM_DESC(dev_max_count, "Number of devices to initialize");
module_param(npe_error_handler, int, 0);
MODULE_PARM_DESC(npe_error_handler, "If non-zero, NPE error handling feature will be enabled");
module_param(hss_coexist, int, 0);
MODULE_PARM_DESC(hss_coexist, "If non-zero, HSS-Ethernet coexist feature will be enabled");
#ifdef CONFIG_IXP400_ETH_DB
module_param(customized_eth_hdr_a, int, 0);
MODULE_PARM_DESC(customized_eth_hdr_a, "If non-zero, customized 16-byte Ethernet header "\
		        		"for NPE-A will be enabled");
module_param(customized_eth_hdr_b, int, 0);
MODULE_PARM_DESC(customized_eth_hdr_b, "If non-zero, customized 16-byte Ethernet header "\
		        		"for NPE-B will be enabled");
module_param(customized_eth_hdr_c, int, 0);
MODULE_PARM_DESC(customized_eth_hdr_c, "If non-zero, customized 16-byte Ethernet header "\
		        		"for NPE-C will be enabled");
module_param(module_upgrade_init, int, 0);
MODULE_PARM_DESC(module_upgrade_init, "If non-zero, driver is assumed loaded from "\
				      "module upgrade recovery");
#endif
/* devices will be called ixp0 and ixp1 */
#define DEVICE_NAME "ixp"

/* boolean values for PHY link speed, duplex, and autonegotiation */
#define PHY_SPEED_10    		(0)
#define PHY_SPEED_100   		(1)
#define PHY_DUPLEX_HALF 		(0)
#define PHY_DUPLEX_FULL 		(1)
#define PHY_AUTONEG_OFF 		(0)
#define PHY_AUTONEG_ON  		(1)

/* Fast path macros */
#define FPATH_PORTS (1 << IX_ETH_PORT_2 | 1 << IX_ETH_PORT_3)
#define IS_FPATH_PORT(port_id) ((1 << (port_id)) & (FPATH_PORTS))

/* maintenance time (jiffies) */
#define DB_MAINTENANCE_TIME 		(IX_ETH_DB_MAINTENANCE_TIME*HZ)

/* 
 * Time before kernel will decide that the driver had stuck in transmit 
 * (jiffies) 
 */
#define DEV_WATCHDOG_TIMEO		(10*HZ)

/* Interval between media-sense/duplex checks (jiffies) */
#define MEDIA_CHECK_INTERVAL		(3*HZ)

/* Dictates the rate (per second) at which the NPE queues are serviced */
/*   4000 times/sec = 37 mbufs/interrupt at line rate */
#define QUEUE_DISPATCH_TIMER_RATE	(4000)

/* NPE Message Handler Polling Frequency (in milliseconds) */
#define NPEMH_MS_POLL_FREQ		(5)

/* 
 * Macro to convert number PMU timer rate to interrupt cycles. The value
 * computed is based on NPEMH_MS_POLL_FREQ and QUEUE_DISPATCH_TIMER_RATE 
 */
#define QUEUE_DISPATCH_TIMER_TO_INT_CYCLE_CONVERT \
		(NPEMH_MS_POLL_FREQ * 1000) / \
		(USEC_PER_SEC / QUEUE_DISPATCH_TIMER_RATE)  

#ifdef CONFIG_IXP400_ETH_NAPI
/* The "weight" value for NAPI defines how many receive buffers
 * are processed each time the device is polled.
 * This value is configurable.
 * - smaller values: polled more often, perhaps less latency
 * - higher values: polled less often, perhaps less CPU utilization
 * - recommended value: 32
 * - this value must less than IX_ETH_ACC_MAX_RX_MBUF_CONSUME_PER_CALLBACK
 *   (which is 240 by default)
 */
#define IXP400_NAPI_WEIGHT 		(32)
#else
#define RX_FRAME_RETRIEVE		(32)
#endif


/* number of packets to prealloc for the Rx pool (per driver instance) */
#define RX_MBUF_POOL_SIZE		(512)

/* Maximum number of packets in Tx+TxDone queue */
#define TX_MBUF_POOL_SIZE		(512)

/* VLAN header size definition. Only if VLAN 802.1Q frames are enabled */
#if defined(CONFIG_VLAN_8021Q) || defined(CONFIG_VLAN_8021Q_MODULE)
#define VLAN_HDR			(4)
#else
#define VLAN_HDR			(0)
#endif

/* Number of mbufs in sw queue */
#define MB_QSIZE  (512) /* must be a power of 2 and > TX_MBUF_POOL_SIZE */
#define SKB_QSIZE (512) /* must be a power of 2 and > RX_MBUF_POOL_SIZE */

/* Maximum number of addresses the EthAcc multicast filter can hold */
#define IX_ETH_ACC_MAX_MULTICAST_ADDRESSES (256)

/* Parameter passed to Linux in ISR registration (cannot be 0) */
#define IRQ_ANY_PARAMETER 		(1)

/* MAC recovery default stack size and priority */
#define MAC_RECOVERY_STACK_SIZE		(32)
#define MAC_RECOVERY_PRIORITY		(90)

/* Maximum supported Ethernet port */
#define MAX_ETH_PORTS			(3)

/* 
 * The size of the SKBs to allocate is more than the MSDU size.
 *
 * skb->head starts here
 * (a) 16 bytes reserved in dev_sk_alloc()
 * (b) 48 RNDIS optional reserved bytes
 * (c) 2 bytes for alignment of the IP header on a word boundary
 * skb->data starts here, the NPE will store the payload at this address. The
 * alignment is not required if customized 16-byte Ethernet
 * header(customized_eth_hdr) is enabled
 * (d)      14 bytes  (dev->dev_hard_len)
 * (d1)     2 bytes (if dev->customized_eth_hdr is enabled)
 * (e)      1500 bytes (dev->mtu, can grow up for Jumbo frames)
 * (f)      4 bytes fcs (stripped out by MAC core)
 * (g)      xxx round-up needed for a NPE 64 bytes boundary
 * skb->tail the NPE will not write more than these value
 * (h)      yyy round-up needed for a 32 bytes cache line boundary
 * (i) sizeof(struct sk_buff). there is some shared info sometimes
 *     used by the system (allocated in net/core/dev.c)
 *
 * The driver private structure stores the following fields
 *
 *  msdu_size = d+d1+e : used to set the msdu size
 *  replenish_size = d+d1+e+g : used for replenish
 *  pkt_size = b+c+d+d1+e+g+h : used to allocate skbufs
 *  alloc_size = a+b+c+d+d1+e+g+h+i, compare to skb->truesize
*/

#ifdef CONFIG_USB_RNDIS
#define HDR_SIZE			(2 + 48)
#else
#define HDR_SIZE			(2)
#endif

/* dev_alloc_skb reserves 16 bytes in the beginning of the skbuf
 * and sh_shared_info at the end of the skbuf. But the value used
 * used to set truesize in skbuff.c is struct sk_buff (???). This
 * behaviour is reproduced here for consistency.
*/
#define SKB_RESERVED_HEADER_SIZE	(16)
#define SKB_RESERVED_TRAILER_SIZE	sizeof(struct sk_buff)

/* NPE-A Functionality: Ethernet only */
#define IX_ETH_NPE_A_IMAGE_ID		IX_NPEDL_NPEIMAGE_NPEA_ETH

/* NPE-A Functionality: HSS co-exist */
#define IX_HSS_ETH_NPE_A_IMAGE_ID	IX_NPEDL_NPEIMAGE_NPEA_ETH_HSSCHAN_COEXIST

/* NPE-B Functionality: Ethernet only */
#define IX_ETH_NPE_B_IMAGE_ID		IX_NPEDL_NPEIMAGE_NPEB_ETH

/* NPE-C Functionality: Ethernet only  */
#define IX_ETH_NPE_C_IMAGE_ID		IX_NPEDL_NPEIMAGE_NPEC_ETH

/*
 * Macros to turn on/off debug messages
 */
/* Print kernel error */
#define P_ERROR(args...) \
    printk(KERN_ERR MODULE_NAME ": " args)
    
/* 
 * Print kernel warning. net_ratelimit is use to ensure we do not clog up syslog
 * */
#define P_WARN(args...) 				\
{							\
	printk(KERN_WARNING MODULE_NAME ": " args);	\
}

/*
 * Print kernel notice. net_ratelimit is use to ensure we do not clog up syslog
 */
#define P_NOTICE(args...) 				\
{							\
	printk(KERN_NOTICE MODULE_NAME ": " args);	\
}
/*
 * Print kernel info. net_ratelimit is use to ensure we do not clog up syslog
 */
#define P_INFO(args...) 				\
{							\
    	printk(KERN_INFO MODULE_NAME ": " args);	\
}

/* 
 * Print verbose message. Enabled/disabled by 'log_level' param. net_ratelimit
 * is use to ensure we do not clog up syslog
 */
#define P_VERBOSE(args...) 				\
{							\
	if (log_level >= 1) printk(MODULE_NAME ": " args);\
}

#ifdef DEBUG
#define LOG_SIZE 10240000 /* reduce this size on low memory platform */
char log_msg[LOG_SIZE];
int log_count = 0;
int log_stop = 0;

static void logstop(void)
{
    log_stop = 1;
}
EXPORT_SYMBOL(logstop);

static void logstart(void)
{
    log_stop = 0;
}
EXPORT_SYMBOL(logstart);

static void plog(void)
{
    int count = 0;
    while (count++ < log_count && !log_stop)
	printf("%c", *(log_msg+count));

    memset(log_msg, 0, sizeof(char)*LOG_SIZE);
    log_count = 0;
}
EXPORT_SYMBOL(plog);

#define LOG(args...) 					\
    if (log_count < LOG_SIZE-200)				\
    	log_count += sprintf(log_msg+log_count, args)

/* 
 * Print debug message. Enabled/disabled by 'log_level' param. net_ratelimit is
 * use to ensure we do not clog up syslog
 */
#define P_DEBUG(args...) 				\
{							\
    {							\
	if (log_level >= 2) 				\
	{ 						\
	    printk("%s: %s()\n", MODULE_NAME, __FUNCTION__); \
	    printk(args); 				\
	}						\
    }							\
}

/* Print trace message */
#define TRACE \
    if (log_level >= 2) \
    	printk("%s: %s(): line %d\n", MODULE_NAME, __FUNCTION__, __LINE__)
#else
/* no trace */

#define P_DEBUG(args...)
#define TRACE 
#endif

#ifdef CONFIG_IXP400_ETH_DB
#define ETHDB_MAINTENANCE_MUTEX_LOCK() {\
    down(ethdb_maintenance_mutex); \
}
#define ETHDB_MAINTENANCE_MUTEX_UNLOCK() {\
    up(ethdb_maintenance_mutex); \
}
#define CUSTOMIZED_ETH_HDR_OFFSET (IX_ETH_DB_MAX_ETH_HEADER_OFFSET)
#else
#define ETHDB_MAINTENANCE_MUTEX_LOCK() do{}while(0)
#define ETHDB_MAINTENANCE_MUTEX_UNLOCK() do{}while(0)
#define CUSTOMIZED_ETH_HDR_OFFSET (0) /* set the offset to 0 to ensure if the
					 macro is used accidentally (ethDB is
					 not enabled), it does not takes in any
					 effects
				       */
#endif

/* 
 * QMgr Configuration
 */
/* QMgr threshold configuration utility */
#define Q_THRESHOLD_MAX		(64)/* Maximum allowed threshold boundary */
/* 
 * macro to set the queue threshold. The macro ensures that the threshold set 
 * is <= Q_THRESHOLD_MAX for any given value
 */
#define Q_THRESHOLD_SET(x)	(((x) == Q_THRESHOLD_MAX)\
					?(x):(Q_THRESHOLD_MAX - 1) & (x))

/* Queue threshold  */
#define RX_Q_NE_THRESHOLD_TC0	(Q_THRESHOLD_SET(0)) 
#define RX_Q_NF_THRESHOLD_TC0	(Q_THRESHOLD_SET(1)) 
#define RX_Q_NE_THRESHOLD_TC1	(Q_THRESHOLD_SET(0)) 
#define RX_Q_NF_THRESHOLD_TC1	(Q_THRESHOLD_SET(1)) 
#define RX_Q_NE_THRESHOLD_TC2	(Q_THRESHOLD_SET(0)) 
#define RX_Q_NF_THRESHOLD_TC2	(Q_THRESHOLD_SET(1)) 
#define RX_Q_NE_THRESHOLD_TC3	(Q_THRESHOLD_SET(0)) 
#define RX_Q_NF_THRESHOLD_TC3	(Q_THRESHOLD_SET(1)) 
#define RX_FREE_Q_NE_THRESHOLD	(Q_THRESHOLD_SET(32))
#define RX_FREE_Q_NF_THRESHOLD	(Q_THRESHOLD_SET(64))
#define TX_DONE_Q_NE_THRESHOLD	(Q_THRESHOLD_SET(32))
#define TX_DONE_Q_NF_THRESHOLD	(Q_THRESHOLD_SET(1))
#define TX_Q_NE_THRESHOLD	(Q_THRESHOLD_SET(8))
#define TX_Q_NF_THRESHOLD	(Q_THRESHOLD_SET(8))

/* Queue notification */
#define RX_Q_NOTIFY_TC0		(IX_QMGR_Q_SOURCE_ID_NOT_E)
#define RX_Q_NOTIFY_TC1		(IX_QMGR_Q_SOURCE_ID_NOT_E)
#define RX_Q_NOTIFY_TC2		(IX_QMGR_Q_SOURCE_ID_NOT_E)
#define RX_Q_NOTIFY_TC3		(IX_QMGR_Q_SOURCE_ID_NOT_E)
#define RX_FREE_Q_NOTIFY	(IX_QMGR_Q_SOURCE_ID_NOT_NF)
#define TX_DONE_Q_NOTIFY	(IX_QMGR_Q_SOURCE_ID_NOT_NE)
#define TX_Q_NOTIFY		(IX_QMGR_Q_SOURCE_ID_NOT_NF)

#define RX_Q_DEPTH 	(IX_ETH_ACC_MAX_RX_MBUF_CONSUME_PER_CALLBACK)
#define TXDONE_Q_DEPTH 	(64)
#define RXFREE_Q_DEPTH	(128)

/* Number of buffers to replenish when rx_free_job is scheduled */
#define RX_FREE_REPLENISH 	(RX_FREE_Q_NF_THRESHOLD)

/* 
 * Macros to check if the queue may throttle 
 */
#define FILLING_Q_MAY_THROTTLE(threshold, source) 			\
   !(((source) == (IX_QMGR_Q_SOURCE_ID_NOT_E)) ||			\
     ((source) == (IX_QMGR_Q_SOURCE_ID_NOT_NE) && (threshold) == 0)	\
    )
     
#define DRAINING_Q_MAY_THROTTLE(threshold, source) 			\
   !(((source) == IX_QMGR_Q_SOURCE_ID_NOT_F) ||				\
     ((source) == IX_QMGR_Q_SOURCE_ID_NOT_NF && (threshold) == 0)	\
    )

struct timeout_work {
    struct work_struct task;
    struct net_device *ndev;
};

struct rx_free_work {
    struct work_struct task;
    struct net_device *ndev;
};

struct tx_done_work {
    struct work_struct task;
    struct net_device *ndev;
};


/* Private device data */
typedef struct {
    spinlock_t lock;  /* multicast management lock */
    
    unsigned int msdu_size;
    unsigned int replenish_size;
    unsigned int pkt_size;
    unsigned int alloc_size;

    struct net_device_stats stats; /* device statistics */

    IxEthAccPortId port_id; /* EthAcc port_id */

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED
    /* private scheduling discipline */
    struct Qdisc *qdisc;
#endif

    /* Implements a software queue for mbufs 
     * This queue is written in the tx done process and 
     * read during tx. Because there is 
     * 1 reader and 1 writer, there is no need for
     * a locking algorithm.
     */

    /* mbuf Tx queue indexes */
    unsigned int mbTxQueueHead;
    unsigned int mbTxQueueTail;

    /* mbuf Rx queue indexes */
    unsigned int mbRxQueueHead;
    unsigned int mbRxQueueTail;

    /* software queue containers */
    IX_OSAL_MBUF *mbTxQueue[MB_QSIZE];
    IX_OSAL_MBUF *mbRxQueue[MB_QSIZE];

    /* RX MBUF pool */
    IX_OSAL_MBUF_POOL *rx_pool;

    /* TX MBUF pool */
    IX_OSAL_MBUF_POOL *tx_pool;

    /* id of thread for the link duplex monitoring */
    int maintenanceCheckThreadId;

    /* mutex locked by thread, until the thread exits */
    struct semaphore *maintenanceCheckThreadComplete;

    /* Used to stop the kernel thread for link monitoring. */
    volatile BOOL maintenanceCheckStopped;

    /* workqueue used for tx timeout */
    struct workqueue_struct *timeout_workq;

    /* work structs */
    struct timeout_work timeout_w;
    struct rx_free_work rx_free_w;
    struct tx_done_work tx_done_w;

    /* used to control the message output */
    UINT32 devFlags;

    /* Customized 16-byte Ethernet Header flag */
    BOOL customized_eth_hdr;

#ifdef DEBUG
    int skb_alloc_count;
#endif

    IX_OSAL_MBUF *rxfree_mb[RXFREE_Q_DEPTH+1]; /*array for rxfree mbuf
    						 replenishment*/
    /* 
     * Implements a software queue for skbufs 
     * This queue is written in the tx done process and 
     * read during rxfree. Because there is 
     * 1 reader and 1 writer, there is no need for
     * a locking algorithm.
     *
     * This software queue is shared by both ports.
     */
#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
    /* skbuf queue indexes */
    unsigned int skQueueHead;
    unsigned int skQueueTail;

    /* software queue containers */
    struct sk_buff *skQueue[SKB_QSIZE];
#endif
} priv_data_t;

/* extern Linux kernel data */
extern unsigned long loops_per_jiffy; /* used to calculate CPU clock speed */

/*
 * Dataplane prototypes
 */
static int dev_ixp400_hard_start_xmit(struct sk_buff*, struct net_device*);
static void rx_frame_drop(priv_data_t*, IX_OSAL_MBUF*);
static void rx_ehandler_cb (UINT32);
static void rx_free_ehandler_cb(UINT32);
static void tx_ehandler_cb(UINT32 ,IxEthAccTxTrafficClass);
static void tx_done_ehandler_cb(UINT32);
static void rx_free_disabled_ehandler_cb(UINT32);
static void tx_disabled_ehandler_cb (UINT32, IxEthAccTxTrafficClass);
static inline void __tx_done_job (struct net_device *);
static void tx_done_job(struct work_struct *);
static int rx_free_replenish(priv_data_t *, u32);
static void rx_free_job(struct work_struct*);
static void tx_done_disable_cb(UINT32, IX_OSAL_MBUF*);
static void rx_disable_cb(UINT32, IX_OSAL_MBUF*, IxEthAccPortId);
#ifndef CONFIG_IXP400_ETH_NAPI
static void rx_job(struct work_struct*);
DECLARE_WORK(rx_work, rx_job);
#endif

/*
 * Port enabling and disabling prototypes
 */
static int port_enable(struct net_device*);
static void port_disable(struct net_device*);


/* 
 * prototypes for locally defined pmu_timer functions
 */
static int dev_pmu_timer_setup(void);
static void dev_pmu_timer_unload(void);
static void dev_pmu_timer_disable(void);
static void dev_pmu_timer_restart(void);
static int dev_pmu_timer_init(void);

/*
 * Prototype for message handler polling function
 */
static inline int npemh_poll(void *data);

/*
 * Prototype for parity error detection and handler
 */

#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
static int __init parity_npe_error_handler_init(void);
static void parity_npe_error_handler_uninit(void);
static void parity_npe_recovery_done_cb(IxErrHdlAccErrorEventType event_type);
static void parity_error_cb(void);
#else
static int __init parity_npe_error_handler_init(void){ return 0; }
static void parity_npe_error_handler_uninit(void){}
#endif

#ifdef CONFIG_IXP400_ETH_DB
static int fpath_enable(priv_data_t *priv);
static int fpath_disable(priv_data_t *priv);
static BOOL ethdb_mac_learning_status_get(void);
static int ethdb_mac_learning_enable(BOOL enable);
static int module_upgrade (BOOL enable);
static BOOL module_upgrade_status_get (void);
static void customized_eth_hdr_set(priv_data_t *priv);
static void fpath_cb(UINT32 callbackTag, IX_OSAL_MBUF *mbuf);
#else

/*
 * The below functions are not callable by the user. It does nothing shall
 * it is being called
 */
static void customized_eth_hdr_set(priv_data_t *priv)
{
}

static void ethdb_maintenance_timer_clear(void)
{
}

#endif


/* Collection of boolean PHY configuration parameters */
typedef struct {
    BOOL speed100;
    BOOL duplexFull;
    BOOL autoNegEnabled;
    BOOL linkMonitor;
} phy_cfg_t;

/* Structure to associate NPE image ID with NPE port ID */
typedef struct {
    UINT32 npeImageId;
    IxNpeDlNpeId npeId;
} npe_info_t;


static int __devinit dev_eth_probe(struct device *dev);
static int __devexit dev_eth_remove(struct device *dev);
static void dev_eth_release(struct device *dev);

static struct device_driver ixp400_eth_driver = {
    .name	= MODULE_NAME,
    .bus	= &platform_bus_type,
    .probe	= dev_eth_probe,
    .remove 	= dev_eth_remove,
};

static struct platform_device ixp400_eth_devices[MAX_ETH_PORTS] = {
    {
	.name 	= MODULE_NAME,
	.id   	= IX_ETH_PORT_1,
	.dev	= 
	    {
		.release	= dev_eth_release,
	    },
    },
    {
	.name 	= MODULE_NAME,
	.id   	= IX_ETH_PORT_2,
	.dev	= 
	    {
		.release	= dev_eth_release,
	    },
    },
    {
	.name 	= MODULE_NAME,
	.id   	= IX_ETH_PORT_3,
	.dev	= 
	    {
		.release	= dev_eth_release,
	    },
    }
};

/* Platform device is statically allocated, nothing to be released */
static void dev_eth_release(struct device *dev)
{
}

/*
 * STATIC VARIABLES
 *
 * This section sets several default values for each port.
 * These may be edited if required.
 */

/* values used inside the irq */
static unsigned long timer_countup_ticks;
static IxQMgrDispatcherFuncPtr dispatcherFunc;
static struct timeval  irq_stamp;  /* time of interrupt */

/* 
 * The PHY addresses mapped to Intel IXP400 Software EthAcc ports.
 *
 * These are hardcoded and ordered by increasing ports.
 * Overwriting these values by a PHY discovery is disabled by default but
 * can optionally be enabled if required
 * by passing module param no_phy_scan with a zero value
 *
 * Up to 32 PHYs may be discovered by a phy scan. Addresses
 * of all PHYs found will be stored here, but only the first
 * 2 will be used with the Intel IXP400 Software EthAcc ports.
 *
 * See also the function phy_init() in this file.
 *
 * NOTE: The hardcoded PHY addresses have been verified on
 * the IXDP425 and Coyote (IXP4XX RG) Development platforms.
 * However, they may differ on other platforms.
 */
static int phyAddresses[IXP400_ETH_ACC_MII_MAX_ADDR] =
{
#if defined(CONFIG_ARCH_IXDP425)
    /* 1 PHY per NPE port */
    0, /* Port 1 (IX_ETH_PORT_1 / NPE B) */
    1  /* Port 2 (IX_ETH_PORT_2 / NPE C) */

#elif defined(CONFIG_ARCH_IXDP465) || defined(CONFIG_MACH_IXDP465)
    /* 1 PHY per NPE port */
    0, /* Port 1 (IX_ETH_PORT_1 / NPE B) */
    1, /* Port 2 (IX_ETH_PORT_2 / NPE C) */
    2  /* Port 3 (IX_ETH_PORT_3 / NPE A) */

#elif defined(CONFIG_ARCH_ADI_COYOTE)
    4, /* Port 1 (IX_ETH_PORT_1) - Connected to PHYs 1-4      */
    5, /* Port 2 (IX_ETH_PORT_2) - Only connected to PHY 5    */

    3,  /******************************************************/
    2,  /* PHY addresses on Coyote platform (physical layout) */
    1   /* (4 LAN ports, switch)  (1 WAN port)                */
        /*       ixp0              ixp1                       */
        /*  ________________       ____                       */
        /* /_______________/|     /___/|                      */
	/* | 1 | 2 | 3 | 4 |      | 5 |                       */
        /* ----------------------------------------           */
#elif defined(CONFIG_MACH_IXDPG425)
    5, /* Port 1 (ixp0) - Connected to switch via PHY 5 */
    4, /* Port 2 (ixp1) - Only connected to PHY 4       */
    0, /* 4 port switch - PHY 0..3                      */
    1,
    2,
    3
#elif defined(CONFIG_MACH_KIXRP435)
    1, /* Port 1 (IX_ETH_PORT_2) - Connected to PHYs 1-4      */
    5, /* Port 2 (IX_ETH_PORT_3) - Only connected to PHY 5    */

    2,  /*********************************************************/
    3,  /* PHY addresses on KIXRP435 platform (physical layout)  */
    4   /* (4 LAN ports, switch)  (1 WAN port)                   */
        /*       ixp1              ixp2                          */
        /*  ________________       ____                          */
        /* /_______________/|     /___/|                         */
	/* | 1 | 2 | 3 | 4 |      | 5 |                          */
        /* ----------------------------------------              */
#else
    /* other platforms : suppose 1 PHY per NPE port */
    0, /* PHY address for EthAcc Port 1 (IX_ETH_PORT_1 / NPE B) */
    1  /* PHY address for EthAcc Port 2 (IX_ETH_PORT_2 / NPE C) */

#endif
};

/* 
 * Port ID to default_phy_cfg and phyAddresses index mapping table
 */
static long portIdPhyIndexMap[] =
{
#if defined (CONFIG_ARCH_IXDP465) || defined (CONFIG_MACH_IXDP465) 
	0, /* NPE-B */
	1, /* NPE-C */
	2  /* NPE-A */
#elif defined(CONFIG_MACH_KIXRP435)
	-1, /* Invalid */
	0,  /* NPE-C */
	1   /* NPE-A */
#else
/* 
 * CONFIG_CPU_IXP42X is not define by the kernel. Hence we assume IXP42X if
 * the flags above mismatch
 */
	0, /* NPE-B */
	1  /* NPE-C */
#endif
};

/* The default configuration of the phy on each Intel IXP400 Software EthAcc port.
 *
 * This configuration is loaded when the phy's are discovered.
 * More PHYs can optionally be configured here by adding more
 * configuration entries in the array below.  The PHYs will be
 * configured in the order that they are found, starting with the
 * lowest PHY address.
 *
 * See also function phy_init() in this file
 */
static phy_cfg_t default_phy_cfg[] =
{
#if defined(CONFIG_ARCH_IXDP425)
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}, /* Port 0: monitor the phy */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}  /* Port 1: monitor the link */

#elif defined(CONFIG_ARCH_IXDP465) || defined(CONFIG_MACH_IXDP465)
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}, /* Port 0: monitor the phy */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}, /* Port 1: monitor the link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE} /* Port 2: ignore the link */

#elif defined(CONFIG_ARCH_ADI_COYOTE)
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},/* Port 0: NO link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}, /* Port 1: monitor the link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE}

#elif defined(CONFIG_MACH_IXDPG425)
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE}, /* Port 0: NO link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE},  /* Port 1: monitor the link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE}
#elif defined(CONFIG_MACH_KIXRP435)
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE}, /* Port 1: NO link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE},  /* Port 2: monitor the link */
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE},
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,FALSE}
#else
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}, /* Port 0: monitor the link*/
    {PHY_SPEED_100, PHY_DUPLEX_FULL, PHY_AUTONEG_ON,TRUE}  /* Port 1: monitor the link*/
#endif
};

/* Default MAC addresses for EthAcc Ports 1 and 2 (using Intel MAC prefix) 
 * Default is 
 *   IX_ETH_PORT_1 -> MAC 00:02:b3:01:01:01
 *   IX_ETH_PORT_2 -> MAC 00:02:b3:02:02:02
 *   IX_ETH_PORT_3 -> MAC 00:02:b3:03:03:03
*/
static IxEthAccMacAddr default_mac_addr[] =
{
    {{0x00, 0x02, 0xB3, 0x01, 0x01, 0x01}}  /* EthAcc Port 0 */
    ,{{0x00, 0x02, 0xB3, 0x02, 0x02, 0x02}} /* EthAcc Port 1 */
#if defined (CONFIG_ARCH_IXDP465) || defined(CONFIG_MACH_IXDP465) ||\
    defined (CONFIG_MACH_KIXRP435)
    ,{{0x00, 0x02, 0xB3, 0x03, 0x03, 0x03}} /* EthAcc Port 2 */
#endif
};

/*
 * Array structure used to store net_device per port ID
 */
static struct net_device *net_devices[MAX_ETH_PORTS+1];

/* Default mapping of  NpeImageIds for EthAcc Ports 
 * Default is 
 *   IX_ETH_PORT_1 -> IX_ETH_NPE_B
 *   IX_ETH_PORT_2 -> IX_ETH_NPE_C
 *   IX_ETH_PORT_3 -> IX_ETH_NPE_A
 *
 * This mapping cannot be changed.
 * 
 */
static npe_info_t default_npeImageId[]=
{
    {IX_ETH_NPE_B_IMAGE_ID, IX_NPEDL_NPEID_NPEB}, /* Npe firmware for EthAcc Port 0  */
    {IX_ETH_NPE_C_IMAGE_ID, IX_NPEDL_NPEID_NPEC}, /* Npe firmware for EthAcc Port 1  */
#if defined (CONFIG_ARCH_IXDP465) || defined(CONFIG_MACH_IXDP465) || \
    defined (CONFIG_MACH_KIXRP435)
    {IX_ETH_NPE_A_IMAGE_ID, IX_NPEDL_NPEID_NPEA}  /* Npe firmware for EthAcc Port 2  */
#endif
};

/* Default mapping of ethAcc portId for devices ixp0 and ixp1 
 * Default is 
 *   device ixp0 ---> IX_ETH_PORT_1
 *   device ixp1 ---> IX_ETH_PORT_2
 *
 * On a system with NPE C, and a single PHY, 
 * - swap the entries from this array to assign ixp0 to IX_ETH_PORT_2
 * - set phyAddress[] with a single entry and the correct PHY address
 * - set default_phy_cfg[] with a single entry and the initialisation PHY setup
 * - starts the driver with the option dev_max_count=1 no_phy_scan=1 
 *      dev_max_count=1 will create 1 driver instance ixp0
 *      no_phy_scan=1 will bypass the default mapping set by phy_init
 *
*/
static IxEthAccPortId default_portId[] =
{
#if   defined (CONFIG_IXP400_ETH_NPEA_ONLY)
    IX_ETH_PORT_3 /* EthAcc Port 3 for ixp2 */
#elif defined (CONFIG_IXP400_ETH_NPEC_ONLY)
    IX_ETH_PORT_2 /* EthAcc Port 2 for ixp1 */
#elif defined (CONFIG_IXP400_ETH_NPEB_ONLY)
    /* configure port for NPE B first */
    IX_ETH_PORT_1 /* EthAcc Port 1 for ixp0 */
#elif defined (CONFIG_ARCH_IXDP465) || defined (CONFIG_MACH_IXDP465)
    /* configure port for NPE B first */
    IX_ETH_PORT_1, /* EthAcc Port 1 for ixp0 */
    IX_ETH_PORT_2, /* EthAcc Port 2 for ixp1 */
    IX_ETH_PORT_3  /* EthAcc Port 3 for ixp2 */
#elif defined (CONFIG_MACH_KIXRP435)
    /* configure port for NPE C first */
    IX_ETH_PORT_2, /* EthAcc Port 2 for ixp1 */
    IX_ETH_PORT_3  /* EthAcc Port 3 for ixp2 */
#else
    /* configure and use both ports */
    IX_ETH_PORT_1, /* EthAcc Port 1 for ixp0 */
    IX_ETH_PORT_2  /* EthAcc Port 2 for ixp1 */
#endif
};

/* Mutex lock used to coordinate access to IxEthAcc functions
 * which manipulate the MII registers on the PHYs
 */
static struct semaphore *miiAccessMutex;

#ifdef CONFIG_IXP400_ETH_DB
/* mutex locked when maintenance is being performed */
static struct semaphore *ethdb_maintenance_mutex;
#endif

#ifdef CONFIG_IXP400_ETH_NAPI
/* Pointer to net device which can handle NAPI rx polling. 
 * This device must be in the "open" state for NAPI rx polling to work.
 * When this device is closed, set it to another open device if possible.
 * Should be initialized to a default device in the probe routine,
 * then set/reset whenever a device is opened/closed.
 */
static struct net_device *rx_poll_dev = NULL;
#endif

/*
 * ERROR COUNTERS
 */

static UINT32 skbAllocFailErrorCount = 0;


/*
 * ERROR NUMBER FUNCTIONS
 */

/* Convert IxEthAcc return codes to suitable Linux return codes */
static int convert_error_ethAcc (IxEthAccStatus error)
{
    switch (error)
    {
	case IX_ETH_ACC_SUCCESS:            return  0;
	case IX_ETH_ACC_FAIL:               return -1;
	case IX_ETH_ACC_INVALID_PORT:       return -ENODEV;
	case IX_ETH_ACC_PORT_UNINITIALIZED: return -EPERM;
	case IX_ETH_ACC_MAC_UNINITIALIZED:  return -EPERM;
	case IX_ETH_ACC_INVALID_ARG:        return -EINVAL;
	case IX_ETH_TX_Q_FULL:              return -EAGAIN;
	case IX_ETH_ACC_NO_SUCH_ADDR:       return -EFAULT;
	default:                            return -1;
    };
}

/*
 * DEBUG UTILITY FUNCTIONS
 */
#ifdef DEBUG_DUMP

static void hex_dump(void *buf, int len)
{
    int i;

    for (i = 0 ; i < len; i++)
    {
	printk("%02x", ((u8*)buf)[i]);
	if (i%2)
	    printk(" ");
	if (15 == i%16)
	    printk("\n");
    }
    printk("\n");
}

static void mbuf_dump(char *name, IX_OSAL_MBUF *mbuf)
{
    printk("+++++++++++++++++++++++++++\n"
	"%s MBUF dump mbuf=%p, m_data=%p, m_len=%d, len=%d\n",
	name, mbuf, IX_OSAL_MBUF_MDATA(mbuf), IX_OSAL_MBUF_MLEN(mbuf), IX_OSAL_MBUF_PKT_LEN(mbuf));
    printk(">> mbuf:\n");
    hex_dump(mbuf, sizeof(*mbuf));
    printk(">> m_data:\n");
    hex_dump(__va(IX_OSAL_MBUF_MDATA(mbuf)), IX_OSAL_MBUF_MLEN(mbuf));
    printk("\n-------------------------\n");
}

static void skb_dump(char *name, struct sk_buff *skb)
{
    printk("+++++++++++++++++++++++++++\n"
	"%s SKB dump skb=%p, data=%p, tail=%p, len=%d\n",
	name, skb, skb->data, skb->tail, skb->len);
    printk(">> data:\n");
    hex_dump(skb->data, skb->len);
    printk("\n-------------------------\n");
}
#endif

/*
 * MEMORY MANAGEMENT
 */

/* XScale specific : preload a cache line to memeory */
static inline void dev_preload(void *virtualPtr) 
{
   __asm__ (" pld [%0]\n" : : "r" (virtualPtr));
}

static inline void dev_skb_preload(struct sk_buff *skb)
{
    /* from include/linux/skbuff.h, skb->len field and skb->data filed 
     * don't share the same cache line (more than 32 bytes between the fields)
     */
    dev_preload(&skb->len);
    dev_preload(&skb->data);
}

/*
 * BUFFER MANAGEMENT
 */

/*
 * Utility functions to handle software queues
 * Each driver has a software queue of skbufs and 
 * a software queue of mbufs for tx and mbufs for rx
 */
#define MB_QINC(index) ((index)++ & (MB_QSIZE - 1))
#define SKB_QINC(index) ((index)++ & (SKB_QSIZE - 1))
#define SKB_QFETCH(index) ((index) & (SKB_QSIZE - 1))

/**
 * mbuf_swap_skb : links/unlinks mbuf to skb
 */
static inline struct sk_buff *mbuf_swap_skb(IX_OSAL_MBUF *mbuf, struct sk_buff *skb)
{
    struct sk_buff *res = IX_OSAL_MBUF_PRIV(mbuf);

    IX_OSAL_MBUF_PRIV(mbuf) = skb;

    if (!skb)
	return res;
    
    IX_OSAL_MBUF_MDATA(mbuf) = skb->data;
    IX_OSAL_MBUF_MLEN(mbuf) = IX_OSAL_MBUF_PKT_LEN(mbuf) = skb->len;

    return res;
}

/*
 * dev_skb_mbuf_free: relaese skb to the kernel and mbuf to the pool 
 * This function is called during port_disable and has no constraint
 * of performances.
*/
static inline void mbuf_free_skb(IX_OSAL_MBUF *mbuf)
{
    TRACE;
    /* chained buffers can be received during port
     * disable after a mtu size change.
     */
    do
    {
	/* unchain and free the buffers */
	IX_OSAL_MBUF *next = IX_OSAL_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf);
	IX_OSAL_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) = NULL;
	if (IX_OSAL_MBUF_PRIV(mbuf) != NULL)
	{
	    /* this buffer has an skb attached, free both of them */	    
	    dev_kfree_skb_any(mbuf_swap_skb(mbuf, NULL));
	    IX_OSAL_MBUF_POOL_PUT(mbuf);
	}
	else
	{
	    /* this buffer has no skb attached, just ignore it */
	}
	mbuf = next;
#ifdef DEBUG
	if (mbuf != NULL)
	{
	    P_WARN("mbuf_free_skb: chained mbuf\n");
	}
#endif
    }
    while (mbuf != NULL);
    
    TRACE;
}


/* dev_skb_dequeue: remove a skb from the skb queue */
static inline struct sk_buff * dev_skb_dequeue(priv_data_t *priv)
{
    struct sk_buff *skb;

#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
    unsigned int repeat = 0;

    do {
	if (priv->skQueueHead != priv->skQueueTail)
	{
	    /* get from queue (fast) packet is ready for use 
	     * because the fields are reset during the enqueue
	     * operations
	     */
	    skb = priv->skQueue[SKB_QINC(priv->skQueueTail)];

	    if (priv->skQueueHead != priv->skQueueTail)
	    {
		/* preload the next skbuf : this is an optimisation to 
		 * avoid stall cycles when acessing memory.
		 */
		dev_skb_preload(priv->skQueue[SKB_QFETCH(priv->skQueueTail)]);
	    }
	    /* check the skb size fits the driver requirements 
	     */
	    if (skb->truesize >= priv->alloc_size)
	    {
		return skb;
	    }
	    /* the skbuf is too small : put it to pool (slow)
	     * This may occur when the ports are configured
	     * with a different mtu size.
	     */
	    dev_kfree_skb_any(skb);
#ifdef DEBUG
	    priv->skb_alloc_count--;
#endif
	}


	/*
	 * Attempt to replenish the skb pool from txDone queue once before
	 * trying allocating a new one
	 */
	if (!repeat++)
	{
	    int key = ixOsalIrqLock();
	    struct net_device *ndev = net_devices[priv->port_id];
	    TRACE;

	    /*
	     * Servicing TxDone queue to replenish internal TX Mbuf
	     */
	    __tx_done_job(ndev);
	    ixOsalIrqUnlock(key);
	}
    } while (repeat <= 1);

#endif

    /* get a skbuf from pool (slow) */
    skb = __dev_alloc_skb(priv->pkt_size, GFP_ATOMIC|__GFP_NOWARN);

    if (skb != NULL)
    {
	skb_reserve(skb,
	    (!priv->customized_eth_hdr)?HDR_SIZE:
	    HDR_SIZE - CUSTOMIZED_ETH_HDR_OFFSET);
	skb->len = priv->replenish_size;
	IX_ACC_DATA_CACHE_INVALIDATE(skb->data, priv->replenish_size);
#ifdef DEBUG
	priv->skb_alloc_count++;
#endif
    }
    else
    {
	skbAllocFailErrorCount++;
    }

    return skb;
}

#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
/* dev_skb_enqueue: add a skb to the skb queue */
static inline void dev_skb_enqueue(priv_data_t *priv, struct sk_buff *skb)
{
    /* check for big-enough unshared skb, and check the queue 
     * is not full If the queue is full or the complete ownership of the
     * skb is not guaranteed, just free the skb.
     * (atomic counters are read on the fly, there is no need for lock)
     */

    if ((skb->truesize >= priv->alloc_size) && 
	(atomic_read(&skb->users) == 1) && 
	(!skb_cloned(skb)) &&
	(skb->fclone == SKB_FCLONE_UNAVAILABLE) &&
        (skb->destructor == NULL) && 
	(atomic_read(&skb_shinfo(skb)->dataref) == 1) &&
	(skb->nohdr == 0) &&
        (skb_shinfo(skb)->nr_frags == 0) &&
        (skb_shinfo(skb)->frag_list == NULL) &&
	(priv->skQueueHead - priv->skQueueTail < SKB_QSIZE))
    {
 	/* put big unshared mbuf to queue (fast)
	 *
	 * The purpose of this part of code is to reset the skb fields
	 * so they will be ready for reuse within the driver.
	 * 
	 * The following fields are reset during a skb_free and 
	 * skb_alloc sequence (see dev/core/skbuf.c). We
	 * should call the function skb_headerinit(). Unfortunately,
	 * this function is static.
	 *
	 * This code resets the current skb fields to zero,
	 * resets the data pointer to the beginning of the 
	 * skb data area, then invalidates the all payload.
	 */

	dst_release(skb->dst);

#ifdef CONFIG_XFRM
	secpath_put(skb->sp);
	skb->sp = NULL;
#endif

#ifdef CONFIG_NETFILTER
	/* Some packets may get incorrectly process by netfilter firewall 
	 * software if CONFIG_NETFILTER is enabled and filtering is in use. 
	 * The solution is to reset the following fields in the skbuff 
	 * before re-using it on the Rx-path
	 */
        skb->mark = 0;
        nf_conntrack_put(skb->nfct);
        skb->nfct = NULL;
#ifdef CONFIG_BRIDGE_NETFILTER
/* We need to free the memory attached to the nf_bridge pointer to avoid a memory leak */
	nf_bridge_put(skb->nf_bridge);
	skb->nf_bridge = NULL;
#endif /* CONFIG_BRIDGE_NETFILTER */
#if defined(CONFIG_NF_CONNTRACK) || defined(CONFIG_NF_CONNTRACK_MODULE)
	nf_conntrack_put_reasm(skb->nfct_reasm);
	skb->nfct_reasm = NULL;
#endif /* CONFIG_NF_CONNTRACK || CONFIG_NF_CONNTRACK_MODULE */
#endif /* CONFIG_NETFILTER */
	skb->sk = NULL;
        skb->dst = NULL;
	skb->pkt_type = PACKET_HOST;    /* Default type */
        skb->ip_summed = 0;
        skb->priority = 0;
	skb->ipvs_property = 0;
	skb->nfctinfo = 0;
	skb->local_df = 0;
#ifdef CONFIG_NET_SCHED
	skb->tc_index = 0;
#ifdef CONFIG_NET_CLS_ACT
	skb->tc_verd = 0;
#endif /* CONFIG_NET_CLS_ACT */
#endif /* CONFIG_NET_SCHED */

#ifdef CONFIG_DMA_ENGINE
	skb->dma_cookie = 0;
#endif

#ifdef CONFIG_NETWORK_SECMARK
	skb->secmark = 0;
#endif
	/* reset the data pointer (skb_reserve is not used for efficiency) */
	skb->data = skb->head + SKB_RESERVED_HEADER_SIZE; 
	skb->data += HDR_SIZE - (priv->customized_eth_hdr)?
		CUSTOMIZED_ETH_HDR_OFFSET:0;
	skb->len = priv->replenish_size;

	/* invalidate the payload
	 * This buffer is now ready to be used for replenish.
	 */
	IX_ACC_DATA_CACHE_INVALIDATE(skb->data, skb->len);

	priv->skQueue[SKB_QINC(priv->skQueueHead)] = skb;
    }
    else
    {
	/* put to pool (slow) */
	dev_kfree_skb_any(skb);
#ifdef DEBUG
	priv->skb_alloc_count--;
#endif
    }
}
#endif /* ifdef CONFIG_IXP400_ETH_SKB_RECYCLE */


/* dev_skb_queue_drain: remove all entries from the skb queue */
static void dev_skb_queue_drain(priv_data_t *priv)
{
#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
    int key = ixOsalIrqLock();

    /* check for skbuf, then get it and release it */
    while (priv->skQueueHead != priv->skQueueTail)
    {
	dev_kfree_skb_any(dev_skb_dequeue(priv));
#ifdef DEBUG
	priv->skb_alloc_count--;
#endif
    }

    ixOsalIrqUnlock(key);
#endif /* CONFIG_IXP400_ETH_SKB_RECYCLE */
}

/* dev_rx_mb_dequeue: return one mbuf from the rx mbuf queue */
static inline IX_OSAL_MBUF * dev_rx_mb_dequeue(priv_data_t *priv)
{
    IX_OSAL_MBUF *mbuf;

    TRACE;

    if (priv->mbRxQueueHead != priv->mbRxQueueTail)
    {
	/* get from queue (fast) */
	mbuf = priv->mbRxQueue[MB_QINC(priv->mbRxQueueTail)];
    }
    else
    {
	/* get from pool (slow) */
	mbuf = IX_OSAL_MBUF_POOL_GET(priv->rx_pool);
    }

    return mbuf;
}

/* dev_rx_mb_enqueue: add one mbuf to the rx mbuf queue */
static inline void dev_rx_mb_enqueue(priv_data_t *priv, IX_OSAL_MBUF *mbuf)
{
    TRACE;

    /* check for queue not full */
    if (priv->mbRxQueueHead - priv->mbRxQueueTail < MB_QSIZE)
    {
	/* put to queue (fast) */
	priv->mbRxQueue[MB_QINC(priv->mbRxQueueHead)] = mbuf;
    }
    else
    {
	IX_OSAL_MBUF_POOL_PUT(mbuf);
    }
}

/* dev_rx_mb_queue_drain: remove all mbufs from the rx mbuf queue */
static void dev_rx_mb_queue_drain(priv_data_t *priv)
{
    IX_OSAL_MBUF *mbuf;
    int key = ixOsalIrqLock();

    /* free all queue entries */
    while(priv->mbRxQueueHead != priv->mbRxQueueTail)
    {
	mbuf = dev_rx_mb_dequeue(priv);
	IX_OSAL_MBUF_POOL_PUT(mbuf);
    }

    ixOsalIrqUnlock(key); 
}

/* dev_tx_mb_dequeue: remove one mbuf from the tx mbuf queue */
static inline IX_OSAL_MBUF * dev_tx_mb_dequeue(priv_data_t *priv)
{
    IX_OSAL_MBUF *mbuf;
    if (priv->mbTxQueueHead != priv->mbTxQueueTail)
    {
	/* get from queue (fast) */
	mbuf = priv->mbTxQueue[MB_QINC(priv->mbTxQueueTail)];
    }
    else
    {
	/* get from pool (slow) */
	mbuf = IX_OSAL_MBUF_POOL_GET(priv->tx_pool);
    }
    return mbuf;
}

/* dev_tx_mb_enqueue: add one mbuf to the tx mbuf queue */
static inline void dev_tx_mb_enqueue(priv_data_t *priv, IX_OSAL_MBUF *mbuf)
{
    /* check for queue not full */
    if (priv->mbTxQueueHead - priv->mbTxQueueTail < MB_QSIZE)
    {
	/* put to queue (fast) */
	priv->mbTxQueue[MB_QINC(priv->mbTxQueueHead)] = mbuf;
    }
    else
    {
	IX_OSAL_MBUF_POOL_PUT(mbuf);
    }
}

/* dev_tx_mb_queue_drain: remove all mbufs from the tx mbuf queue */
static void dev_tx_mb_queue_drain(priv_data_t *priv)
{
    IX_OSAL_MBUF *mbuf;
    int key = ixOsalIrqLock();

    /* free all queue entries */
    while(priv->mbTxQueueHead != priv->mbTxQueueTail)
    {
	mbuf = dev_tx_mb_dequeue(priv);
	IX_OSAL_MBUF_POOL_PUT(mbuf);
    }

    ixOsalIrqUnlock(key);
}

/* Allocate an skb for every mbuf in the rx_pool
 * and pass the pair to the NPEs
 */
static int dev_rx_buff_prealloc(priv_data_t *priv)
{
    IX_OSAL_MBUF *mbuf;
    int key;
    u32 count = 0;

    /* because this function may be called during monitoring steps
     * interrupt are disabled so it won't conflict with the
     * QMgr context.
     */
    key = ixOsalIrqLock();
    while(NULL != (mbuf = IX_OSAL_MBUF_POOL_GET(priv->rx_pool)))
    {
    	dev_rx_mb_enqueue(priv, mbuf);
    }
    ixOsalIrqUnlock(key);

    /*
     * Only call replenishment if there are buffers to replenish
     */
    if ((count = priv->mbRxQueueHead - priv->mbRxQueueTail))
    {
	rx_free_replenish(priv, count);
    }

    return 0;
}


/* Replenish if necessary and slowly drain the sw queues
 * to limit the driver holding the memory space permanently.
 * This action should run at a low frequency, e.g during
 * the link maintenance.
 */
static int dev_buff_maintenance(struct net_device *dev)
{
    priv_data_t *priv = netdev_priv(dev);

    dev_rx_buff_prealloc(priv);

#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
    {
	int key;

	/* Drain slowly the skb queue and free the skbufs to the 
	 * system. This way, after a while, skbufs will be 
	 * reused by other components, instead of being
	 * parmanently held by this driver.
	 */
	key = ixOsalIrqLock();

	/* check for queue not empty, then get the skbuff and release it */
	if (priv->skQueueHead != priv->skQueueTail)
	{
	    dev_kfree_skb_any(dev_skb_dequeue(priv));
#ifdef DEBUG
	    priv->skb_alloc_count--;
#endif
	}
	ixOsalIrqUnlock(key);
    }
#endif /* CONFIG_IXP400_ETH_SKB_RECYCLE */

    return 0;
}


/* 
 * KERNEL THREADS
 */

/* flush the pending signals for a thread and 
 * check if a thread is killed  (e.g. system shutdown)
 */
static BOOL dev_thread_signal_killed(void)
{
    int killed = FALSE;
    if (signal_pending (current))
    {
 	spin_lock_irq(&current->sighand->siglock);

	if (sigismember(&(current->pending.signal), SIGKILL)
	    || sigismember(&(current->pending.signal), SIGTERM))
	{
	    /* someone kills this thread */
	    killed = TRUE;
	}
	flush_signals(current);

	spin_unlock_irq(&current->sighand->siglock);
    }
    return killed;
}

/* This timer will check the PHY for the link duplex and
 * update the MAC accordingly. It also executes some buffer
 * maintenance to release mbuf in excess or replenish after
 * a severe starvation
 *
 * This function loops and wake up every 3 seconds.
 */
static int dev_media_check_thread (void* arg)
{
    struct net_device *dev = (struct net_device *) arg;
    priv_data_t *priv = netdev_priv(dev);
    int linkUp;
    int speed100;
    int fullDuplex = -1; /* unknown duplex mode */
    int newDuplex;
    int autonegotiate;
    unsigned phyNum = phyAddresses[portIdPhyIndexMap[priv->port_id]];
    u32 res;


    TRACE;

    /* Lock the mutex for this thread.
       This mutex can be used to wait until the thread exits
    */
    down (priv->maintenanceCheckThreadComplete);

    daemonize("ixp400 MediaCheck"); 
    spin_lock_irq(&current->sighand->siglock);

    sigemptyset(&current->blocked);
   
    recalc_sigpending();
    spin_unlock_irq(&current->sighand->siglock);
    
    snprintf(current->comm, sizeof(current->comm), "ixp400 %s", dev->name);

    TRACE;
    
    while (1)
    {
	/* We may have been woken up by a signal. If so, we need to
	 * flush it out and check for thread termination 
	 */ 
	if (dev_thread_signal_killed())
	{
	    priv->maintenanceCheckStopped = TRUE;
	}

	/* If the interface is down, or the thread is killed,
	 * or gracefully aborted, we need to exit this loop
	 */
	if (priv->maintenanceCheckStopped)
	{
	    break;
	}

	/*
	 * Determine the link status
	 */

	TRACE;

	if (default_phy_cfg[portIdPhyIndexMap[priv->port_id]].linkMonitor)
	{
	    /* lock the MII register access mutex */
	    down(miiAccessMutex);

	    res = ixEthMiiLinkStatus(phyNum,
				     &linkUp,
				     &speed100,
				     &newDuplex, 
				     &autonegotiate);
	    /* release the MII register access mutex */
	    up(miiAccessMutex);

	    /* We may have been woken up by a signal. If so, we need to
	     * flush it out and check for thread termination 
	     */ 
	    if (dev_thread_signal_killed())
	    {
		priv->maintenanceCheckStopped = TRUE;
	    }
	    
	    /* If the interface is down, or the thread is killed,
	     * or gracefully aborted, we need to exit this loop
	     */
	    if (priv->maintenanceCheckStopped)
	    {
		break;
	    }
	
	    if (res != IX_SUCCESS)
	    {
		P_WARN("ixEthMiiLinkStatus failed on PHY%d.\n"
		       "\tCan't determine\nthe auto negotiated parameters. "
		       "Using default values.\n",
		       phyNum); 
		/* something is bad, gracefully stops the loop */
		priv->maintenanceCheckStopped = TRUE;
		break;
	    }
	    
	    if (linkUp)
	    {
		if (! netif_carrier_ok(dev))
		{
		    /* inform the kernel of a change in link state */
		    netif_carrier_on(dev);
		}

		/*
		 * Update the MAC mode to match the PHY mode if 
		 * there is a phy mode change.
		 */
		if (newDuplex != fullDuplex)
		{
		    fullDuplex = newDuplex;
		    if (fullDuplex)
		    {
			ixEthAccPortDuplexModeSet (priv->port_id, IX_ETH_ACC_FULL_DUPLEX);
		    }
		    else
		    {
			ixEthAccPortDuplexModeSet (priv->port_id, IX_ETH_ACC_HALF_DUPLEX);
		    }
		}
	    }
	    else
	    {
		fullDuplex = -1;
		if (netif_carrier_ok(dev))
		{
		    /* inform the kernel of a change in link state */
		    netif_carrier_off(dev);
		}
	    }
	}
	else
	{
	    /* if no link monitoring is set (because the PHY do not
	     * require (or support) a link-monitoring follow-up, then assume 
	     * the link is up 
	     */
	    if (!netif_carrier_ok(dev))
	    {
		netif_carrier_on(dev);
	    }
	}
    
	TRACE;
    
	/* this is to prevent the rx pool from emptying when
	 * there's not enough memory for a long time
	 * It prevents also from holding the memory for too
	 * long
	 */
	dev_buff_maintenance(dev);

	/*
	 * Tx done queue may have 1 entry with huge chain, hence we service it
	 * here in event that the notification does not fire
	 */
	__tx_done_job(dev);

	/* Now sleep for 3 seconds */
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout(MEDIA_CHECK_INTERVAL);
    } /* while (1) ... */

    /* free the mutex for this thread. */
    up (priv->maintenanceCheckThreadComplete);

    return 0;
}

/*
 * NPE message polling function
 */
static inline int npemh_poll(void *data)
{

    /* Polling for NPE-A */
#if !defined (CONFIG_IXP400_ETH_NPEB_ONLY) &&\
    !defined (CONFIG_IXP400_ETH_NPEC_ONLY)

#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
    if (unlikely(IX_SUCCESS != ixNpeMhMessagesReceive(IX_NPEMH_NPEID_NPEA)))
    {
	P_ERROR("NPE-A npeMh read failure!\n");
    }
#endif
#endif

    /* Polling for NPE-B */
#if !defined (CONFIG_CPU_IXP43X)
#if defined (CONFIG_IXP400_ETH_NPEB_ONLY) || defined (CONFIG_IXP400_ETH_ALL)
    if (unlikely(IX_SUCCESS != ixNpeMhMessagesReceive(IX_NPEMH_NPEID_NPEB)))
    {
	P_ERROR("NPE-B npeMh read failure!\n");
    }
#endif
#endif 

    /* Polling for NPE-C */
#if defined (CONFIG_IXP400_ETH_NPEC_ONLY) || defined (CONFIG_IXP400_ETH_ALL)
    if (unlikely(IX_SUCCESS != ixNpeMhMessagesReceive(IX_NPEMH_NPEID_NPEC)))
    {
	P_ERROR("NPE-C npeMh read failure!\n");
    }
#endif

    return 0;
}
	    


/*
 * TIMERS
 *
 * PMU Timer : This timer based on IRQ  will call the NPEMH 
 * function every 5ms when NPE error handler is enabled
 *
 * Maintenance Timer : This timer run the maintanance action every 
 * 60 seconds approximatively.
 *
 */

/* PMU Timer reload : this should be done at each interrupt 
 *
 * Because the timer may overflow exactly between the time we
 * write the counter and the time we clear the overflow bit, all
 * irqs are disabled. Missing the everflow event and the timer 
 * will trigger only after a wrap-around.
*/
static void dev_pmu_timer_restart(void)
{
    unsigned long flags;

    local_irq_save(flags);

     __asm__(" mcr p14,0,%0,c1,c1,0\n"  /* write current counter */
            : : "r" (timer_countup_ticks));

    __asm__(" mrc p14,0,r1,c4,c1,0; "  /* get int enable register */
            " orr r1,r1,#1; "
            " mcr p14,0,r1,c5,c1,0; "  /* clear overflow */
            " mcr p14,0,r1,c4,c1,0\n"  /* enable interrupts */
            : : : "r1");

    local_irq_restore(flags);
}

static irqreturn_t dev_pmu_timer_npemhpoll_os_isr(int irg, void *dev_id)
{
    dev_pmu_timer_restart(); /* set up the timer for the next interrupt */

    npemh_poll(NULL);

    return IRQ_HANDLED;
}

/* initialize the PMU timer */
static int dev_pmu_timer_init(void)
{
    UINT32 controlRegisterMask =
        BIT(0) | /* enable counters */
        BIT(2);  /* reset clock counter; */

    /* 
    *   Compute the number of xscale cycles needed between each 
    *   PMU IRQ. This is done from the result of an OS calibration loop.
    *
    *   For 533MHz CPU, 533000000 tick/s / 4000 times/sec = 138250
    *   4000 times/sec = 37 mbufs/interrupt at line rate 
    *   The pmu timer is reset to -138250 = 0xfffde3f6, to trigger an IRQ
    *   when this up counter overflows.
    *
    *   The multiplication gives a number of instructions per second.
    *   which is close to the processor frequency, and then close to the
    *   PMU clock rate.
    *
    *      HZ : jiffies/second (global OS constant)
    *      loops/jiffy : global OS value cumputed at boot time
    *      2 is the number of instructions per loop
    *
    */
    UINT32 timer_countdown_ticks = (loops_per_jiffy * HZ * 2) /
        QUEUE_DISPATCH_TIMER_RATE;

    /*
     * NPE error handling require slower polling (default 5ms), so we
     * adjust the ticks count according to our polling frequency relatively
     * to QUEUE_DISPATCH_TIMER_RATE
     */
    timer_countdown_ticks *= QUEUE_DISPATCH_TIMER_TO_INT_CYCLE_CONVERT;

    timer_countup_ticks = -timer_countdown_ticks;

    /* enable the CCNT (clock count) timer from the PMU */
    __asm__(" mcr p14,0,%0,c0,c1,0\n"  /* write control register */
            : : "r" (controlRegisterMask));

    return 0;
}

/* stops the timer when the module terminates 
 *
 * This is protected from re-entrancy while the timeer is being restarted.
*/
static void dev_pmu_timer_disable(void)
{
    unsigned long flags;

    local_irq_save(flags);

    __asm__(" mrc p14,0,r1,c4,c1,0; "  /* get int enable register */
            " and r1,r1,#0x1e; "
            " mcr p14,0,r1,c4,c1,0\n"  /* disable interrupts */
            : : : "r1");
    local_irq_restore(flags);
}

static int dev_pmu_timer_setup(void)
{
    TRACE;

    /*
     * Setting up NPE error handler NPE message handler polling
     */
    if (request_irq(IX_OSAL_IXP400_XSCALE_PMU_IRQ_LVL,
		    dev_pmu_timer_npemhpoll_os_isr,
		    SA_SHIRQ,
		    "ixp400_eth PMU timer",
		    (void *)IRQ_ANY_PARAMETER))
    {
	P_ERROR("Failed to reassign irq to PMU timer interrupt!\n");
	return -1;
    }

    TRACE;

    if (dev_pmu_timer_init())
    {
        P_ERROR("Error initializing IXP400 PMU timer!\n");
        return -1;
    }

    TRACE;

    dev_pmu_timer_restart(); /* set up the timer for the next interrupt */

    return 0;
}

static void dev_pmu_timer_unload(void)
{
        dev_pmu_timer_disable(); /* stop the timer */
        free_irq(IX_OSAL_IXP400_XSCALE_PMU_IRQ_LVL,(void *)IRQ_ANY_PARAMETER);
}

/* Internal ISR : run a few thousand times per second and calls 
 * the queue manager dispatcher entry point.
 */
static irqreturn_t dev_qmgr_os_isr( int irg, void *dev_id)
{
    /* call the queue manager entry point */
    dispatcherFunc(IX_QMGR_QUELOW_GROUP);

    return IRQ_HANDLED;
}

#ifdef CONFIG_IXP400_ETH_DB
/* This timer will call ixEthDBDatabaseMaintenance every
 * IX_ETH_DB_MAINTENANCE_TIME jiffies
 */
static void ethdb_maintenance_timer_task(struct work_struct *data);

/* task spawned by timer interrupt for EthDB maintenance */
static DECLARE_DELAYED_WORK(ethdb_maintenance_work, ethdb_maintenance_timer_task);

static void ethdb_maintenance_timer_set(void)
{
    schedule_delayed_work(&ethdb_maintenance_work, DB_MAINTENANCE_TIME);
}

static void ethdb_maintenance_timer_clear(void)
{
    cancel_delayed_work(&ethdb_maintenance_work);
    flush_scheduled_work();
}

static void ethdb_maintenance_timer_task(struct work_struct *data)
{
    ETHDB_MAINTENANCE_MUTEX_LOCK();
    ixEthDBDatabaseMaintenance();
    ETHDB_MAINTENANCE_MUTEX_UNLOCK();
    ethdb_maintenance_timer_set();
}
#endif

/* the following function performs the operations normally done
 * in eth_type_trans() (see net/ethernet/eth.c) , and takes care about 
 * the flags returned by the NPE, so a payload lookup is not needed
 * in most of the cases.
 */
static inline void dev_eth_type_trans(unsigned int mflags, 
				      struct sk_buff *skb, 
				      struct net_device *dev)
{
    unsigned header_len = dev->hard_header_len;
    skb->mac.raw=skb->data;
    /* skip the mac header : there is no need for length comparison since
     * the skb during a receive is always greater than the header size and 
     * runt frames are not enabled.
     */
    skb_pull(skb, header_len);
   
    /* fill the pkt arrival time (set at the irq callback entry) */
    skb_set_timestamp(skb, &irq_stamp);
 
    /* fill the input device field */
    skb->dev = dev;
    
    /* set the protocol from the bits filled by the NPE */
    if (mflags & IX_ETHACC_NE_IPMASK)
    { 
	/* the type_length field is 0x0800 */
	skb->protocol = htons(ETH_P_IP); 
    } 
    else if (mflags & IX_ETHACC_NE_IPV6MASK)
    {
	/* the type_length field is 0x86DD */
	skb->protocol = htons(ETH_P_IPV6); 
    }
    else
    {
	/* use linux algorithm to find the protocol 
	 * from the type-length field. This costs a
	 * a lookup inside the packet payload. The algorithm
	 * and its constants are taken from the eth_type_trans()
	 * function.
	 */
	struct ethhdr *eth = eth_hdr(skb);
	unsigned short hproto = ntohs(eth->h_proto);
	
	if (hproto >= 1536)
	{
	    skb->protocol = eth->h_proto;
	}
	else
	{
	    unsigned short rawp = *(unsigned short *)skb->data;
	    if (rawp == 0xFFFF)
		skb->protocol = htons(ETH_P_802_3);
	    else
		skb->protocol = htons(ETH_P_802_2);
	}
    }

    /* set the packet type 
     * check mcast and bcast bits as filled by the NPE 
     */
    if (mflags & (IX_ETHACC_NE_MCASTMASK | IX_ETHACC_NE_BCASTMASK))
    {
	if (mflags & IX_ETHACC_NE_BCASTMASK)
	{
	    /* the packet is a broadcast one */
	    skb->pkt_type=PACKET_BROADCAST;
	}
	else
	{
	    /* the packet is a multicast one */
	    skb->pkt_type=PACKET_MULTICAST;
	    ((priv_data_t*)netdev_priv(dev))->stats.multicast++;
	}
    }
    else
    {
	if (dev->flags & IFF_PROMISC)
	{
	    /* check dest mac address only if promiscuous
	     * mode is set This costs
	     * a lookup inside the packet payload.
	     */
	    struct ethhdr *eth = eth_hdr(skb);
	    unsigned char *hdest = eth->h_dest;
	    
	    if (memcmp(hdest, dev->dev_addr, ETH_ALEN)!=0)
	    {
		skb->pkt_type = PACKET_OTHERHOST;
	    }
	}
	else
	{
	    /* promiscuous mode is not set, All packets are filtered
	     * by the NPE and the destination MAC address matches
	     * the driver setup. There is no need for a lookup in the 
	     * payload and skb->pkt_type is already set to PACKET_LOCALHOST;
	     */
	}
    }

    return;
}



/* Set promiscuous/multicast mode for the MAC */
static void ixp400_dev_set_multicast_list(struct net_device *dev)
{
    int res;
    priv_data_t *priv = netdev_priv(dev);
    IxEthAccMacAddr addr1 = {};

/* 4 possible scenarios here
 *
 * scenarios:
 * #1 - promiscuous mode ON
 * #2 - promiscuous mode OFF, accept NO multicast addresses
 * #3 - promiscuous mode OFF, accept ALL multicast addresses
 * #4 - promiscuous mode OFF, accept LIST of multicast addresses 
 */

    TRACE;

    /* 
     * WR19880: We removed IRQ lock from this API in order to allow NPE message
     * send with response API to work. Hence, ixp400_dev_set_multicast_list is
     * now not callable from IRQ!
     *
     * if called from irq handler, lock already acquired
     * if (!in_irq())
     *     spin_lock_irq(&priv->lock);
     */


    /* clear multicast addresses that were set the last time (if exist) */
    ixEthAccPortMulticastAddressLeaveAll (priv->port_id);

    TRACE;

/**** SCENARIO #1 ****/
    /* Set promiscuous mode */
    if (dev->flags & IFF_PROMISC)
    {
	if ((res = ixEthAccPortPromiscuousModeSet(priv->port_id)))
	{
	    P_ERROR("%s: ixEthAccPortPromiscuousModeSet failed on port %d\n",
		    dev->name, priv->port_id);
	}
	else
	{
	    TRACE;
	    /* avoid redundant messages */
	    if (!(priv->devFlags & IFF_PROMISC))
	    {
		P_VERBOSE("%s: Entering promiscuous mode\n", dev->name);
	    }
	    priv->devFlags = dev->flags;
	}

	goto Exit;
    }

    TRACE;


/**** SCENARIO #2 ****/

    /* Clear promiscuous mode */
    if ((res = ixEthAccPortPromiscuousModeClear(priv->port_id)))
    {
	/* should not get here */
	P_ERROR("%s: ixEthAccPortPromiscuousModeClear failed for port %d\n",
		dev->name, priv->port_id);
    }
    else
    {
    	TRACE;
	/* avoid redundant messages */
	if (priv->devFlags & IFF_PROMISC)
	{
	    P_VERBOSE("%s: Leaving promiscuous mode\n", dev->name);
	}
	priv->devFlags = dev->flags;
    }

    TRACE;


/**** SCENARIO #3 ****/
    /* If there's more addresses than we can handle, get all multicast
     * packets and sort the out in software
     */
    /* Set multicast mode */
    if ((dev->flags & IFF_ALLMULTI) || 
	(dev->mc_count > IX_ETH_ACC_MAX_MULTICAST_ADDRESSES))
    {
	/* ALL MULTICAST addresses will be accepted */
        ixEthAccPortMulticastAddressJoinAll(priv->port_id);

	P_VERBOSE("%s: Accepting ALL multicast packets\n", dev->name);
	goto Exit;
    }

    TRACE;

/**** SCENARIO #4 ****/
    /* Store all of the multicast addresses in the hardware filter */
    if ((dev->mc_count))
    {
	/* now join the current address list */
	/* Get only multicasts from the list */
	struct dev_mc_list *mc_ptr;

	/* Rewrite each multicast address */
	for (mc_ptr = dev->mc_list; mc_ptr; mc_ptr = mc_ptr->next)
	{
	    memcpy (&addr1.macAddress[0], &mc_ptr->dmi_addr[0],
		    IX_IEEE803_MAC_ADDRESS_SIZE);
	    ixEthAccPortMulticastAddressJoin (priv->port_id, &addr1);
	}
    }

Exit:

    TRACE;

    /*
     * WR19880: IRQ is not lock
     *
     * if (!in_irq())
     *     spin_unlock_irq(&priv->lock);
     */
}




/* Open the device.
 * Request resources and start interrupts
 */
static int do_dev_open(struct net_device *dev)
{
    int res;

#ifdef CONFIG_IXP400_ETH_NAPI
    /* if the current rx_poll_dev isn't running, set it to this one */
    if(!netif_running(rx_poll_dev))
    {
        rx_poll_dev = dev;
    }
#endif

    /* prevent the maintenance task from running while bringing down port */
    ETHDB_MAINTENANCE_MUTEX_LOCK();

    /* bring up the port */
    res = port_enable(dev);

    ETHDB_MAINTENANCE_MUTEX_UNLOCK();

    if (res == 0)
    {
	try_module_get(THIS_MODULE);
    }

    return res;
}

/* Close the device.
 * Free resources acquired in dev_start
 */
static int do_dev_stop(struct net_device *dev)
{
    int dev_idx;
    struct net_device *tmp_dev;
    int key;

    TRACE;

    dev_idx = 0;
    tmp_dev = NULL;

#ifdef CONFIG_IXP400_ETH_NAPI
    /* Wait for already scheduled NAPI polling function to complete */
    while (test_bit(__LINK_STATE_RX_SCHED, &rx_poll_dev->state))
    	msleep(1);
#endif

    key = ixOsalIrqLock();
    ixEthAccQMgrRxNotificationDisable();
    ixOsalIrqUnlock(key);

#ifdef CONFIG_IXP400_ETH_NAPI
    if(dev == rx_poll_dev)
    {
        /* closing the current poll device, change to an open one. 
         * if none are open, do nothing.
         */
        for(dev_idx = 0; dev_idx < dev_max_count; dev_idx++)
        {
	    tmp_dev = dev_get_drvdata(&ixp400_eth_devices[dev_idx].dev);

	    if (tmp_dev == dev)
	    {
	    	continue;
	    }

            if(netif_running(tmp_dev))
            {
                rx_poll_dev = tmp_dev;
                break;
            }
        }
    }
#endif

    /* prevent the maintenance task from running while bringing up port */
    ETHDB_MAINTENANCE_MUTEX_LOCK();

    /* bring the port down */
    port_disable(dev);

    ETHDB_MAINTENANCE_MUTEX_UNLOCK();

#ifdef CONFIG_IXP400_ETH_NAPI
    if (netif_running(rx_poll_dev))
    {
	netif_poll_enable(rx_poll_dev);
    }
#endif

    ixEthAccQMgrRxNotificationEnable();

    module_put(THIS_MODULE);

    return 0;
}

static void
dev_tx_timeout_task(struct work_struct *work)
{
    struct timeout_work *timeout_w = container_of(work, struct timeout_work,
					task);
    struct net_device *dev = timeout_w->ndev;
    priv_data_t *priv = netdev_priv(dev);

    P_WARN("%s: Tx Timeout for port %d\n", dev->name, priv->port_id);

    ETHDB_MAINTENANCE_MUTEX_LOCK();
    port_disable(dev);

    /* Note to user: Consider performing other reset operations here
     * (such as PHY reset), if it is known to help the Tx Flow to 
     * become "unstuck". This scenario is application/board-specific.
     * 
     * e.g.
     *
     *  if (netif_carrier_ok(dev))
     *  {
     *	down(miiAccessMutex);
     *	ixEthMiiPhyReset(phyAddresses[priv->port_id]);
     *	up(miiAccessMutex);
     *  }
     */
    
    /* enable traffic again if the port is up */
    if (dev->flags & IFF_UP)
    {
	port_enable(dev);
    }

    ETHDB_MAINTENANCE_MUTEX_UNLOCK();
}

/* This function is called when kernel thinks that TX is stuck */
static void dev_tx_timeout(struct net_device *dev)
{
    priv_data_t *priv = netdev_priv(dev);

    TRACE;
    queue_work(priv->timeout_workq, &priv->timeout_w.task);
}

/* update the maximum msdu value for this device */
static void dev_change_msdu(struct net_device *dev, int new_msdu_size)
{
    priv_data_t *priv = netdev_priv(dev);
    unsigned int new_size = new_msdu_size;

    /*
     * The replenishment size includes the customized Ethernet header offset
     */
    if (priv->customized_eth_hdr)
    {
    	new_size += CUSTOMIZED_ETH_HDR_OFFSET;
    }

    priv->msdu_size = new_size;

    /* ensure buffers are large enough (do not use too small buffers
     * even if it is possible to configure so. 
     */
    if (new_size < IX_ETHNPE_ACC_FRAME_LENGTH_DEFAULT)
    {
	new_size = IX_ETHNPE_ACC_FRAME_LENGTH_DEFAULT;
    }

    /* the NPE needs 64 bytes boundaries : round-up to the next
    * frame boundary. This size is used to invalidate and replenish.
    */
    new_size = IX_ETHNPE_ACC_RXFREE_BUFFER_ROUND_UP(new_size);
    priv->replenish_size = new_size;

    /* Xscale MMU needs a cache line boundary : round-up to the next
     * cache line boundary. This will be the size used to allocate
     * skbufs from the kernel.
     *
     * The 2 bytes header alignment is not required if customized Ethernet
     * header is enabled. 
     */
    if (priv->customized_eth_hdr)
    {
	new_size += HDR_SIZE - CUSTOMIZED_ETH_HDR_OFFSET;
    }
    else
    {
	new_size += HDR_SIZE;
    }

    new_size = L1_CACHE_ALIGN(new_size);
    priv->pkt_size = new_size;

    /* Linux stack uses a reserved header.
     * skb contain shared info about fragment lists 
     * this will be the value stored in skb->truesize
     */
    priv->alloc_size = SKB_DATA_ALIGN(SKB_RESERVED_HEADER_SIZE + new_size) + 
    	SKB_RESERVED_TRAILER_SIZE;
}

static int dev_change_mtu(struct net_device *dev, int new_mtu_size)
{
    priv_data_t *priv = netdev_priv(dev);

    /* the msdu size includes the ethernet header plus the 
     * mtu (IP payload), but does not include the FCS which is 
     * stripped out by the access layer.
     */
    unsigned int new_msdu_size = new_mtu_size + dev->hard_header_len + VLAN_HDR;

    if (new_msdu_size > IX_ETHNPE_ACC_FRAME_LENGTH_MAX)
    {
	/* Unsupported msdu size */
	return -EINVAL;
    }
    ETHDB_MAINTENANCE_MUTEX_LOCK();

    if (ixEthAccFilteringPortMaximumFrameSizeSet(priv->port_id, 
						new_msdu_size))
    {
	P_ERROR("%s: ixEthDBFilteringPortMaximumFrameSizeSet failed for port %d\n",
		dev->name, priv->port_id);
	ETHDB_MAINTENANCE_MUTEX_UNLOCK();

	return -1;
    }

    /* update the packet sizes needed */
    dev_change_msdu(dev, new_msdu_size);
    /* update the driver mtu value */
    dev->mtu = new_mtu_size;

    ETHDB_MAINTENANCE_MUTEX_UNLOCK();

    return 0;
}

#ifdef CONFIG_IXP400_ETH_DB
/**
 * @fn ixQoSIoctl(IxQosIOArgument *arg)
 *
 * @brief ioctl handler for user to configure/retrieve QoS traffic class 
 * on DSCP, VLAN priority and EtherType
 *
 * @param arg - request data for IX_IOCTL_QOS ioctl
 * @return IX_SUCCESS if the operation completed successfully
 * or IX_FAIL with display an appropriate error message otherwise
 *
 */
static int ixQoSIoctl(priv_data_t *priv, IxQosIOArgument *arg)
{
	int status = IX_FAIL;
	
	arg->pid = priv->port_id;
		
	switch ( arg->actType )
    {
    	case CMD_QOSCFG_SET:
    		
			if (arg->tcMap >= IX_ETH_DB_TRAFFIC_CLASS_MAX)	{
				P_ERROR("Invalid TC value %d\n", arg->tcMap);
				break;
			}
			
    		/* Configure traffic class for single DSCP */
    		if (arg->cfgType == TYPE_QOS_DSCP)	{
    			if ( arg->cfgValue >= IX_ETH_DB_QOS_DSCP_MAX )	{
    				P_ERROR("Invalid DSCP value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBSingleDscp2RxTClassMapSet((IxEthDBPortId)arg->pid, 
														arg->cfgValue, 
														(IxEthDBTrafficClass)arg->tcMap);
			}
			/* Configure VLAN traffic class */
    		else if (arg->cfgType == TYPE_QOS_VLAN)	{
    			if ( arg->cfgValue >= IX_ETH_DB_QOS_VLAN_PRI_MAX )	{
    				P_ERROR("Invalid VLAN priority value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBSingleVlanPri2RxTClassMapSet((IxEthDBPortId)arg->pid, 
    													(IxEthDBPriority)arg->cfgValue, 
    													(IxEthDBTrafficClass)arg->tcMap);
    		}
    		/* Configure Ether type traffic class */
			else if (arg->cfgType == TYPE_QOS_ETHTYPE)	{
    			if ((arg->cfgValue > IX_IOCTL_MAX_IEEE802_3_ETH_TYPE) || 
    				(arg->cfgValue <= IX_IOCTL_MIN_IEEE802_3_ETH_TYPE)) {
    				P_ERROR("Invalid ETHTYPE value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBEthType2RxTClassMapSet((IxEthDBPortId)arg->pid, 
    													arg->cfgValue, 
    													(IxEthDBTrafficClass)arg->tcMap);
    		}
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported type %d for SET\n", arg->cfgType);
    		break;
    		
    	case CMD_QOSCFG_DELETE:

    		/* Delete Ether type traffic class */
    		if (arg->cfgType == TYPE_QOS_ETHTYPE)	{
    			if ((arg->cfgValue > IX_IOCTL_MAX_IEEE802_3_ETH_TYPE) || 
    				(arg->cfgValue <= IX_IOCTL_MIN_IEEE802_3_ETH_TYPE)) {
    				P_ERROR("Invalid ETHTYPE value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBEthType2RxTClassMapDelete((IxEthDBPortId)arg->pid, 
    														arg->cfgValue);
			}
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported type %d for DELETE\n", arg->cfgType);
    		break;
    		
    	case CMD_QOSCFG_CLEAR:
    		
    		/* Clear DSCP traffic class configuration */
    		if (arg->cfgType == TYPE_QOS_DSCP)	{
    			status = ixEthDBDscp2RxTClassMapTableClear((IxEthDBPortId)arg->pid); 
			}
			/* Clear VLAN traffic class configuration */
    		else if (arg->cfgType == TYPE_QOS_VLAN)	{
    			status = ixEthDBVlanPri2RxTClassMapTableClear((IxEthDBPortId)arg->pid); 
    		}
    		/* Clear Ether type traffic class configuration */
			else if (arg->cfgType == TYPE_QOS_ETHTYPE)	{
    			status = ixEthDBEthType2RxTClassMapTableClear((IxEthDBPortId)arg->pid); 
    		}
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported type %d for CLEAR\n", arg->cfgType);
    		break;
    		
    	case CMD_QOSCFG_GETINFO:
			/* Get traffic class for given DSCP */
    		if (arg->cfgType == TYPE_QOS_DSCP)	{
    			if ( arg->cfgValue >= IX_ETH_DB_QOS_DSCP_MAX )	{
    				P_ERROR("Invalid DSCP value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBSingleDscp2RxTClassMapGet(
    											(IxEthDBPortId)arg->pid, 
    											arg->cfgValue,
    											&arg->tcMap);
			}
			/* Get traffic class for given VLAN priority */
    		else if (arg->cfgType == TYPE_QOS_VLAN)	{
    			if ( arg->cfgValue >= IX_ETH_DB_QOS_VLAN_PRI_MAX )	{
    				P_ERROR("Invalid VLAN priority 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBVlanPri2RxTClassMapGet((IxEthDBPortId)arg->pid, 
    													(IxEthDBPriority)arg->cfgValue,
    													&arg->tcMap);
			}
			/* Get traffic class for given Ethernet type */
    		else if (arg->cfgType == TYPE_QOS_ETHTYPE)	{
    			if ((arg->cfgValue > IX_IOCTL_MAX_IEEE802_3_ETH_TYPE) || 
    				(arg->cfgValue <= IX_IOCTL_MIN_IEEE802_3_ETH_TYPE)) {
    				P_ERROR("Invalid ETHTYPE value 0x%08x\n", arg->cfgValue);
    				break;
    			}
    			else
    				status = ixEthDBEthType2RxTClassMapGet((IxEthDBPortId)arg->pid, 
    													(IxEthDBPriority)arg->cfgValue,
    													&arg->tcMap);
			}			
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported type %d for GETINFO\n", arg->cfgType);
			break;
		
   		case CMD_QOSCFG_SHOW:
    		/* Display Rx QoS classification configuration */
    		status = ixEthDBRxQoSShow((IxEthDBPortId)arg->pid);
    		break;

		/* Unsupported/unknowned IOCTL action */	
		default:
			P_ERROR("Unknown action %d for IX_IOCTL_QOS IOCTL\n", arg->actType);
	}
	
	if (status != IX_SUCCESS) {
		P_ERROR("ixQoSIoctl failed in error # %d\n", status);
	}
	return status;
}

/**
 * @fn ixFastPathIoctl(IxFastPathIOArgument *arg)
 *
 * @brief ioctl handler for user to configure/retrieve NPE  
 * Ethernet Fast Path feature
 *
 * @param arg - request data for IX_IOCTL_FPATH ioctl
 * @return IX_SUCCESS if the operation completed successfully
 * or IX_FAIL with display an appropriate error message otherwise
 *
 */
static int ixFastPathIoctl(priv_data_t *priv, IxFastPathIOArgument *arg)
{
	int status = IX_FAIL;

	arg->pid = priv->port_id;

	switch ( arg->actType )
    {  	
    	case CMD_FASTPATH_ADD:
		if ((arg->procType != IX_ETH_DB_FPATH_USERDEFINED)&&
			(arg->tcValue >= IX_ETH_DB_TRAFFIC_CLASS_MAX))	{
			P_ERROR("Invalid Traffic Classic value %d\n", arg->tcValue);
			break;
		}
			
    		/* Configure IPv4 Routing Fast Path rule */
    		if (arg->procType == IX_ETH_DB_FPATH_IPv4)	{
    			status = ixEthDBFastPathIPv4RoutingConfig(
    						(IxEthDBTrafficClass)arg->tcValue,
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathIPv4Config *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
			}
			/* Configure IPv6 Routing Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_IPv6)	{
    			status = ixEthDBFastPathIPv6RoutingConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathIPv6Config *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure IPv4 NAT Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_NAT)	{
    			status = ixEthDBFastPathIPv4NATConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathNATConfig *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure IPv4 NAPT Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_NAPT)	{
    			status = ixEthDBFastPathIPv4NAPTConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathNAPTConfig *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure PPPoE Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_PPPOE)	{
    			status = ixEthDBFastPathIPv4PPPoEConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathPPPoEConfig *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure PPPoE + NAT Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_PPPOENAT)	{
    			status = ixEthDBFastPathIPv4PPPoENATConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathPPPoENATConfig *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure PPPoE + NAPT Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_PPPOENAPT)	{
    			status = ixEthDBFastPathIPv4PPPoENAPTConfig(
    						(IxEthDBTrafficClass)arg->tcValue, 
    						arg->upperLen,
    						(IxEthDBFPathTrafficDirection)arg->tfDir,
    						(IxEthDBFastPathPPPoENAPTConfig *)&arg->flowConfig.flow,
    						(IxEthDBFPathFlowId *)&arg->flowId);
    		}
    		/* Configure user defined Fast Path rule */
    		else if (arg->procType == IX_ETH_DB_FPATH_USERDEFINED)	{ 
    			
				if (arg->channelIdx >= IX_ETHDB_FPATH_CHANNEL_MAX) {
					P_ERROR("Invalid Channel index %d\n", arg->channelIdx);
					break;
				}		    			 			
				else if (arg->entryType == TYPE_CLASSIFIER)	{
 					status = ixEthDBFastPathClassificationTemplateDownload(
 							(IxEthDBPortId)arg->pid,
                            (IxEthDBFPathClsTemplate *)arg->clsTemplate,
                            arg->clsTemplateSizeInWord,
                            (IxEthDBFPathChannelIdx)arg->channelIdx, (BOOL)arg->parentFlag);
                }
                else if (arg->entryType == TYPE_MODIFIER)	{
					status = ixEthDBFastPathModificationTemplateDownload(
							(IxEthDBPortId)arg->pid,
                            (IxEthDBFPathMdfTemplate *)&arg->mdfTemplate,
                            (IxEthDBFPathChannelIdx)arg->channelIdx);
                }
				else	// Unsupported/unknowned type
					P_ERROR("Unsupported rule type %d for Template ADD\n", arg->entryType);                           
    		}
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported rule type %d for ADD\n", arg->entryType);
    		break;
    		
    	case CMD_FASTPATH_DELETE:
    		
    		if (arg->delType == TYPE_DELETION_FLOW)	{
    			status = ixEthDBFastPathFlowConfigurationDelete((IxEthDBFPathFlowId)arg->flowId);
			}
    		else if (arg->delType == TYPE_DELETION_TEMPLATE)	{
    			if (arg->channelIdx >= IX_ETHDB_FPATH_CHANNEL_MAX) {
					P_ERROR("Invalid Channel index %d\n", arg->channelIdx);
					break;
				}
				else {
    				status = ixEthDBFastPathTemplatesDelete(
    							(IxEthDBFPathTrafficDirection)arg->tfDir,
    							(IxEthDBFPathChannelIdx)arg->channelIdx);
    			}				
			}
			else	/* Unsupported/unknowned type */
				P_ERROR("Unsupported config type %d for DELETE\n", arg->delType);
    		break;

    	case CMD_FASTPATH_GETINFO:
    		status = ixEthDBFastPathFlowConfigurationGet(
    				(IxEthDBFPathFlowId)arg->flowId,
    				(IxEthDBFPathFlowConfig *)&arg->flowConfig);
			break;

    	case CMD_FASTPATH_SHOW:
    		status = ixEthDBFastPathShow((IxEthDBFPathFlowId)arg->flowId);
			break;

    	case CMD_FASTPATH_ENABLE:
    		status = fpath_enable(priv);
			break;
			
    	case CMD_FASTPATH_DISABLE:
    		status = fpath_disable(priv);
			break;

		/* Unsupported/unknowned IOCTL action */
		default:
			P_ERROR("Unknown action %d for IX_IOCTL_FASTPATH IOCTL\n", arg->actType);
	}

	if (status != IX_SUCCESS) {
		P_ERROR("ixFastPathIoctl failed in error # %d\n", status);
	}

	return status;
}


/**
 * @fn ixMiscFeatureIoctl(IxMiscFeatureIOArgument *arg)
 *
 * @brief ioctl handler for user to configure/retrieve some  
 * specific NPE Ethernet feature
 *
 * @param arg - request data for IX_IOCTL_MISCFEATURES ioctl
 * @return IX_SUCCESS if the operation completed successfully
 * or IX_FAIL with display an appropriate error message otherwise
 *
 */
static int ixMiscFeatureIoctl(priv_data_t *priv, IxMiscFeatureIOArgument *arg)
{
	int status = IX_FAIL;

	arg->pid = priv->port_id;
	
	switch ( arg->actType )
    {
    	case CMD_MISCFEATURE_MODUPGRADE_TRUE:
    	case CMD_MISCFEATURE_MODUPGRADE_FALSE:
			status = module_upgrade(arg->FeatureEnable);
    		break;
    	case CMD_MISCFEATURE_MODUPGRADE_GETINFO:
    		arg->FeatureEnable = module_upgrade_status_get();
    		status = IX_SUCCESS;
    		break;

    	case CMD_MISCFEATURE_EDBLEARNING_ENABLE:
    	case CMD_MISCFEATURE_EDBLEARNING_DISABLE:
			status = ethdb_mac_learning_enable(arg->FeatureEnable);
    		break;
    	case CMD_MISCFEATURE_EDBLEARNING_GETINFO:
    		arg->FeatureEnable = ethdb_mac_learning_status_get();
    		status = IX_SUCCESS;
    		break;
			
		case CMD_MISCFEATURE_FEATURESET_ENABLE:
		case CMD_MISCFEATURE_FEATURESET_DISABLE:
			if (arg->FeatureSet == 0)	{
				P_ERROR("Invalid FeatureSet %d\n", arg->FeatureSet);
			}
			else 
    			status = ixEthDBFeatureEnable((IxEthDBPortId)arg->pid, 
    										(IxEthDBFeature)arg->FeatureSet, 
    										(BOOL)arg->FeatureEnable);
			break;
			
		case CMD_MISCFEATURE_FEATURESET_GETINFO:
			status = ixEthDBFeatureStatusGet((IxEthDBPortId)arg->pid,
												(IxEthDBFeature)arg->FeatureSet,
												&arg->FeaturePresent,
												&arg->FeatureEnable );
			break;
			
		default:
			P_ERROR("Unknown action %d for IX_IOCTL_MISCFEATURES IOCTL\n", arg->actType);
	}

	if (status != IX_SUCCESS) {
		P_ERROR("ixMiscFeatureIoctl failed in error # %d\n", status);
	}	
	return status;
}
#endif	/* #ifdef CONFIG_IXP400_ETH_DB */

static int do_dev_ioctl(struct net_device *dev, struct ifreq *req, int cmd)
{
    priv_data_t *priv = netdev_priv(dev);
    struct mii_ioctl_data *data = (struct mii_ioctl_data *) & req->ifr_data;
    int phy = phyAddresses[portIdPhyIndexMap[priv->port_id]];
    int res = 0;

    TRACE;

    switch (cmd)
    {
	/*
	 * IOCTL's for mii-tool support
	 */

	/* Get address of MII PHY in use */
	case SIOCGMIIPHY:
	case SIOCDEVPRIVATE:
	    data->phy_id = phy;
	    break;

        /* Read MII PHY register */
	case SIOCGMIIREG:		
	case SIOCDEVPRIVATE+1:
	    down (miiAccessMutex);     /* lock the MII register access mutex */
	    if ((res = ixEthAccMiiReadRtn (data->phy_id, data->reg_num, &data->val_out)))
	    {
		P_ERROR("Error reading MII reg %d on phy %d\n",
		       data->reg_num, data->phy_id);
		res = -1;
	    }
	    up (miiAccessMutex);	/* release the MII register access mutex */
	    break;

	/* Write MII PHY register */
	case SIOCSMIIREG:
	case SIOCDEVPRIVATE+2:
	    down (miiAccessMutex);     /* lock the MII register access mutex */
	    if ((res = ixEthAccMiiWriteRtn (data->phy_id, data->reg_num, data->val_in)))
	    {
		P_ERROR("Error writing MII reg %d on phy %d\n",
                        data->reg_num, data->phy_id);
		res = -1;
	    }
	    up (miiAccessMutex);	/* release the MII register access mutex */
	    break;

	/* set the MTU size */
	case SIOCSIFMTU:
	    res = dev_change_mtu(dev, req->ifr_mtu);
	    break;

#ifdef CONFIG_IXP400_ETH_DB
	/* IOCTL's for NPE QoS features support */
	/* Get and set configuration for QoS */
	case IX_IOCTL_QOS:
		res = ixQoSIoctl(priv, (IxQosIOArgument *)req->ifr_data);
		if (res != IX_SUCCESS)
			P_ERROR("Error in IX_IOCTL_QOS IOCTL\n");		
		break;

	/* IOCTL's for NPE Fast Path support */
	/* Get and set configuration for NPE Fast Path feature */		

	case IX_IOCTL_FPATH:
		res = ixFastPathIoctl(priv, (IxFastPathIOArgument *)req->ifr_data);
		if (res != IX_SUCCESS)
			P_ERROR("Error in IX_IOCTL_FPATH IOCTL\n");
		break;

	/* IOCTL's for NPE misc. feature support */
	/* Get and set configuration for specific NPE Ethernet feature */				
	case IX_IOCTL_MISCFEATURES:
		res = ixMiscFeatureIoctl(priv, (IxMiscFeatureIOArgument *)req->ifr_data);
		if (res != IX_SUCCESS)
			P_ERROR("Error in IX_IOCTL_MISCFEATURES IOCTL\n");
		break;
#endif	/* #ifdef CONFIG_IXP400_ETH_DB */

	default:
	    res = -EOPNOTSUPP;

    }

    return res;
}

static struct net_device_stats *dev_get_stats(struct net_device *dev)
{
    int res;
    /* "stats" should be cache-safe.
     * we alligne "stats" and "priv" by 32 bytes, so that the cache
     * operations will not affect "res" and "priv"
     */
    IxEthEthObjStats ethStats __attribute__ ((aligned(32))) = {};
    priv_data_t *priv = netdev_priv(dev);

    TRACE;

    /* Get HW stats and translate to the net_device_stats */
    if (!netif_running(dev))
    {
	TRACE;
	return &priv->stats;
    }

    TRACE;

    IX_OSAL_CACHE_INVALIDATE(&ethStats, sizeof(ethStats));

    if ((res = ixEthAccMibIIStatsGetClear(priv->port_id, &ethStats)))
    {
	P_ERROR("%s: ixEthAccMibIIStatsGet failed for port %d, res = %d\n",
		dev->name, priv->port_id, res);
	return &priv->stats;
    }

    TRACE;

    /* bad packets received */
    priv->stats.rx_errors += 
        ethStats.dot3StatsAlignmentErrors +
        ethStats.dot3StatsFCSErrors +
        ethStats.dot3StatsInternalMacReceiveErrors;

    /* packet transmit problems */
    priv->stats.tx_errors += 
        ethStats.dot3StatsLateCollisions +
        ethStats.dot3StatsExcessiveCollsions +
        ethStats.dot3StatsInternalMacTransmitErrors +
        ethStats.dot3StatsCarrierSenseErrors;

    priv->stats.collisions +=
        ethStats.dot3StatsSingleCollisionFrames +
        ethStats.dot3StatsMultipleCollisionFrames;

    /* recved pkt with crc error */
    priv->stats.rx_crc_errors +=
        ethStats.dot3StatsFCSErrors;

    /* recv'd frame alignment error */
    priv->stats.rx_frame_errors += 
        ethStats.dot3StatsAlignmentErrors;

    /* detailed tx_errors */
    priv->stats.tx_carrier_errors +=
        ethStats.dot3StatsCarrierSenseErrors;

    /* Rx traffic dropped at the NPE level */
    priv->stats.rx_dropped +=
        ethStats.RxLearnedEntryDiscards +
        ethStats.RxLargeFramesDiscards +
        ethStats.RxSTPBlockedDiscards +
        ethStats.RxVLANTypeFilterDiscards +
        ethStats.RxVLANIdFilterDiscards +
        ethStats.RxInvalidSourceDiscards +
        ethStats.RxBlackListDiscards +
        ethStats.RxWhiteListDiscards +
	ethStats.RxMACFIFOOverrunErrors +
	ethStats.RxQTC0OverrunDiscards +
	ethStats.RxQTC1OverrunDiscards +
	ethStats.RxQTC2OverrunDiscards +
	ethStats.RxQTC3OverrunDiscards +
	ethStats.RxInternalFifoTC0OverrunDiscards +
	ethStats.RxInternalFifoTC1OverrunDiscards +
	ethStats.RxInternalFifoTC2OverrunDiscards +
	ethStats.RxInternalFifoTC3OverrunDiscards;

    /* Tx traffic dropped at the NPE level */
    priv->stats.tx_dropped += 
        ethStats.TxLargeFrameDiscards +
        ethStats.TxVLANIdFilterDiscards;

    return &priv->stats;
}


/* Initialize QMgr and bind it's interrupts */
static int qmgr_init(void)
{
    int res;

    /*
     * Enable live lock and disable polling mode if HSS co-exist is enabled
     */
    if (hss_coexist)
    {
	ixFeatureCtrlSwConfigurationWrite (IX_FEATURECTRL_ORIGB0_DISPATCHER,
	    IX_FEATURE_CTRL_SWCONFIG_DISABLED);
    }

    /* Initialise Queue Manager */
    P_VERBOSE("Initialising Queue Manager...\n");
    if ((res = ixQMgrInit()))
    {
	P_ERROR("Error initialising queue manager!\n");
	return -1;
    }

    TRACE;

    /* Get the dispatcher entrypoint */
    ixQMgrDispatcherLoopGet (&dispatcherFunc);

    TRACE;

    if (request_irq(IX_OSAL_IXP400_QM1_IRQ_LVL,
                    dev_qmgr_os_isr,
                    SA_SHIRQ,
                    "ixp400_eth QM1",
                    (void *)IRQ_ANY_PARAMETER))
    {
        P_ERROR("Failed to request_irq to Queue Manager interrupt!\n");
        return -1;
    }

    ixQMgrDispatcherInterruptModeSet(TRUE);

    TRACE;
    return 0;
}

static int ethacc_uninit(void)
{ 
    int res;

#ifdef CONFIG_IXP400_ETH_DB

    TRACE;

    /* we should uninitialize the components here */
    if (unlikely(res = ixEthDBUnload()))
    {
        P_ERROR("ixEthDBUnload Failed: %d!\n", res);
    }
#endif

    if (unlikely(res = ixEthAccUnload()))
    {
        P_ERROR("ixEthAccUnload Failed: %d!\n", res);
    }

    /* best effort, always succeed and return 0 */
    return 0;
}

static int ethacc_init(void)
{
    int res = 0;
    IxEthAccPortId portId;
    int dev_count;

    /* start all NPEs before starting the access layer */
    TRACE;
    for (dev_count = 0; 
	 dev_count < dev_max_count;  /* module parameter */
	 dev_count++)
    {
    	u32 npe_active = 0;

	portId = default_portId[dev_count];

        TRACE;

	if (TRUE == ixNpeDlNpeActiveCheck(default_npeImageId[portId].npeId))
	{
	    npe_active = 1;
	}

	/*
	 * Check if the NPE is being initialized with other NPE image. Do not
	 * download NPE image when the NPE containing other image
	 */
	if (0 == npe_active)
	{
	    /* Initialise and Start NPE */
	    if (IX_SUCCESS !=
		ixNpeDlNpeInitAndStart(default_npeImageId[portId].npeImageId))
	    {
		P_ERROR("Error starting NPE for Ethernet port %d!\n", portId);
		return -1;
	    }

	    /* 
	     * We reach here when the NPE image downloaded and the NPE engine
	     * started. So we mark the NPE engine started
	     */
	    npe_active = 1;
	}

	if (npe_error_handler && npe_active)
	{
	    /*
	     * Determine which NPE is to be enabled for parity error detection
	     * based on the NPE image ID that is successfully downloaded
	     */
#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
	    switch (default_npeImageId[portId].npeId)
	    {
	    	case IX_NPEDL_NPEID_NPEA:
			parity_npeA_enabled = IX_PARITYENACC_ENABLE;
			break;
	    	case IX_NPEDL_NPEID_NPEB:
			parity_npeB_enabled = IX_PARITYENACC_ENABLE;
			break;
	    	case IX_NPEDL_NPEID_NPEC:
			parity_npeC_enabled = IX_PARITYENACC_ENABLE;
			break;

		default:
			P_ERROR("Unknown NPE image ID\n");
	    }
#endif
	}
    }

    /* initialize the Ethernet Access layer */
    TRACE;
    if ((res = ixEthAccInit()))
    {
	P_ERROR("ixEthAccInit failed with res=%d\n", res);
	return convert_error_ethAcc(res);
    }

#ifdef CONFIG_IXP400_ETH_DB
    if ((res = ixEthDBInit()))
    {
	P_ERROR("ixEthDBInit failed with res=%d\n", res);
	return -ENODEV;
    }
#endif

    if (IX_ETH_ACC_SUCCESS != ixEthAccCallbackModeSet(CALLBACK_EVENT_DRIVEN))
    {
    	P_WARN("ixEthAccCallbackModeSet() failed\n");
    }

    if (IX_ETH_ACC_SUCCESS != ixEthAccTxDoneSwQEnable(FALSE))
    {
    	P_WARN("ixEthAccTxDoneSwQEnable() failed\n");
    }

    TRACE;

    return 0;
}

static int phy_init(void)
{
    int res;
    BOOL physcan[IXP400_ETH_ACC_MII_MAX_ADDR];
    int i, phy_found, num_phys_to_set, dev_count;

    /* initialise the MII register access mutex */
    miiAccessMutex = (struct semaphore *) kmalloc(sizeof(struct semaphore), GFP_KERNEL);
    if (!miiAccessMutex)
	return -ENOMEM;

    init_MUTEX(miiAccessMutex);

    TRACE;
    /* detect the PHYs (ethMii requires the PHYs to be detected) 
     * and provides a maximum number of PHYs to search for.
     */
    res = ixEthMiiPhyScan(physcan, 
			  sizeof(default_phy_cfg) / sizeof(phy_cfg_t));
    if (res != IX_SUCCESS)
    {
	P_ERROR("PHY scan failed\n");
	return convert_error_ethAcc(res);
    }

    /* Module parameter */
    if (no_phy_scan) 
    { 
	/* Use hardcoded phy addresses */
	num_phys_to_set = (sizeof(default_phy_cfg) / sizeof(phy_cfg_t));
    }
    else
    {
	/* Update the hardcoded values with discovered parameters 
	 *
	 * This set the following mapping
	 *  ixp0 --> first PHY discovered  (lowest address)
	 *  ixp1 --> second PHY discovered (next address)
	 *  .... and so on
	 *
	 * If the Phy address and the wiring on the board do not
	 * match this mapping, then hardcode the values in the
	 * phyAddresses array and use no_phy_scan=1 parameter on 
	 * the command line.
	 */
	for (i=0, phy_found=0; i < IXP400_ETH_ACC_MII_MAX_ADDR; i++)
	{
	    if (physcan[i])
	    {
		P_INFO("Found PHY %d at address %d\n", phy_found, i);
                if (phy_found < dev_max_count)
                {
                    phyAddresses[portIdPhyIndexMap[
		    	default_portId[phy_found]]] = i;
                }
                else
                {
                    phyAddresses[phy_found] = i;
                }
		
		if (++phy_found == IXP400_ETH_ACC_MII_MAX_ADDR)
		    break;
	    }
	}

	num_phys_to_set = phy_found;
    }

    /* Reset and Set each phy properties if module is not under upgrading */
    if (!ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_MODULE_UPGRADE))
    {
	for (i=0; i < num_phys_to_set; i++)
	{
	    P_VERBOSE("Configuring PHY %d\n", i);
	    P_VERBOSE("\tSpeed %s\tDuplex %s\tAutonegotiation %s\n",
		      (default_phy_cfg[i].speed100) ? "100" : "10",   
		      (default_phy_cfg[i].duplexFull) ? "FULL" : "HALF",  
		      (default_phy_cfg[i].autoNegEnabled) ? "ON" : "OFF");

	    if (phy_reset) /* module parameter */
	    {
		ixEthMiiPhyReset(phyAddresses[i]);
	    }

	    ixEthMiiPhyConfig(phyAddresses[i],
		default_phy_cfg[i].speed100,   
		default_phy_cfg[i].duplexFull,  
		default_phy_cfg[i].autoNegEnabled);
	}
    }

    /* for each device, display the mapping between the ixp device,
     * the IxEthAcc port, the NPE and the PHY address on MII bus. 
     * Also set the duplex mode of the MAC core depending
     * on the default configuration.
     */
    for (dev_count = 0; 
	 dev_count < dev_max_count  /* module parameter */
	     && dev_count <  num_phys_to_set;
	 dev_count++)
    {
	IxEthAccPortId port_id = default_portId[dev_count];
	char *npe_id = "?";

	if (port_id == IX_ETH_PORT_1) npe_id = "B";
	if (port_id == IX_ETH_PORT_2) npe_id = "C";
	if (port_id == IX_ETH_PORT_3) npe_id = "A";

	P_INFO("%s%d is using NPE%s and the PHY at address %d\n",
	       DEVICE_NAME, port_id, npe_id,
	       phyAddresses[portIdPhyIndexMap[port_id]]);

	/* Set the MAC to the same duplex mode as the phy */
	if (!ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_MODULE_UPGRADE))
	{
	    ixEthAccPortDuplexModeSet(port_id,
		(default_phy_cfg[portIdPhyIndexMap[port_id]].duplexFull) ?
		     IX_ETH_ACC_FULL_DUPLEX : IX_ETH_ACC_HALF_DUPLEX);
	}
    }

    return 0;
}

/* set port MAC addr and update the dev struct if successfull */
int ixp400_dev_set_mac_address(struct net_device *dev, void *addr)
{
    int res;
    IxEthAccMacAddr npeMacAddr;
    struct sockaddr *saddr = (struct sockaddr *)addr;
    priv_data_t *priv = netdev_priv(dev);

    TRACE;

    /* Get MAC addr from parameter */
    memcpy(&npeMacAddr.macAddress,
	   &saddr->sa_data[0],
	   IX_IEEE803_MAC_ADDRESS_SIZE);

    /* Set MAC addr in h/w (ethAcc checks for MAC address to be valid) */
    res = ixEthAccPortUnicastMacAddressSet(priv->port_id, &npeMacAddr);

#ifdef CONFIG_IXP400_ETH_DB
    if (IX_ETH_ACC_SUCCESS == res)
    {
    	res = ixEthDBPortAddressSet(priv->port_id, (IxEthDBMacAddr*)&npeMacAddr);
    }
#endif 

    if (res)
    {
        P_VERBOSE("Failed to set MAC address %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x for port %d\n",
	       (unsigned)npeMacAddr.macAddress[0],
	       (unsigned)npeMacAddr.macAddress[1],
	       (unsigned)npeMacAddr.macAddress[2],
	       (unsigned)npeMacAddr.macAddress[3],
	       (unsigned)npeMacAddr.macAddress[4],
	       (unsigned)npeMacAddr.macAddress[5],
	       priv->port_id);
        return convert_error_ethAcc(res);
    }

    /* update dev struct */
    memcpy(dev->dev_addr, 
	   &saddr->sa_data[0],
	   IX_IEEE803_MAC_ADDRESS_SIZE);

    return 0;
}


/* 
 *  TX QDISC
 */

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED

/* new tx scheduling discipline : the algorithm is based on a 
 * efficient JBI technology : Just Blast It. There is no need for
 * software queueing where the hardware provides this feature
 * and makes the internal transmission non-blocking.
 *
 * because of this reason, there is no need for throttling using
 * netif_stop_queue() and netif_start_queue() (there is no sw queue
 * that linux may restart)
 *
 * This tx queueing scheduling mechanism can be enabled
 * by defining CONFIG_IXP400_ETH_QDISC_ENABLED at compile time
 */
static int dev_qdisc_no_enqueue(struct sk_buff *skb, struct Qdisc * qdisc)
{
        return dev_ixp400_hard_start_xmit(skb, qdisc->dev);     
}

static struct sk_buff *dev_qdisc_no_dequeue(struct Qdisc * qdisc)
{
	return NULL;
}

static struct Qdisc_ops dev_qdisc_ops =
{
	.id		= "ixp400_eth", 
	.priv_size	= 0,
	.enqueue	= dev_qdisc_no_enqueue, 
	.dequeue	= dev_qdisc_no_dequeue,
	.requeue	= dev_qdisc_no_enqueue, 
	.owner		= THIS_MODULE,
};

#endif

static int dev_port_init(IxEthAccPortId portId, UINT32 dev)
{
    int res;

    /* register "safe" callbacks. This ensure that no traffic will be 
     * sent to the stack until the port is brought up (ifconfig up)
     */
    if ((res = ixEthAccPortDisableTxPathCallbackRegister(portId, 
						  tx_done_disable_cb,
						  dev)))

    {
	TRACE;
	P_ERROR("ixEthAccPortDisableTxPathCallbackRegister failed\n");

	return convert_error_ethAcc(res);
    }
    if ((res = ixEthAccPortDisableRxPathCallbackRegister(portId, 
					      rx_disable_cb, 
					      dev)))
    {
	TRACE;
	P_ERROR("ixEthAccPortDisableRxPathCallbackRegister failed\n");

	return convert_error_ethAcc(res);
    }

    /* set tx scheduling discipline */
    if ((res = ixEthAccTxSchedulingDisciplineSet(portId,
                                                 FIFO_NO_PRIORITY)))
    {
        TRACE;
        return convert_error_ethAcc(res);
    }
    /* enable tx frame FCS append */
    if ((res = ixEthAccPortTxFrameAppendFCSEnable(portId)))
    {
        TRACE;
        return convert_error_ethAcc(res);
    }
    /* disable rx frame FCS append */
    if ((res = ixEthAccPortRxFrameAppendFCSDisable(portId)))
    {
        TRACE;
	P_ERROR("ixEthAccPortRxFrameAppendFCSDisable failed with %d\n", res);
        return convert_error_ethAcc(res);
    }

#ifdef CONFIG_IXP400_ETH_DB
    /*
     * Register a fast-path callback to return replenished fast-path mbuf to 
     * the system during fast-path port disabling
     */
    if (IS_FPATH_PORT(portId))
    {
	res = ixEthAccPortFPathDisableCallbackRegister(portId,fpath_cb,portId);

	if (IX_ETH_ACC_SUCCESS != res)
	{
	    P_ERROR("Fast path callback registration failed for port %d: %d\n",
		portId, res);
	}
    }
#endif

    return 0;
}

    /**************************************************************
 *		PORT DISABLING FUNCTIONS      	      	      *
 **************************************************************/
/*
 * This callback is called when the port is under disabling and the RxFree
 * buffer is no longer required. The buffers will be returned to the system
 * memory pool
 *
 */
static void rx_disable_cb(UINT32 callbackTag, IX_OSAL_MBUF *mbuf, IxEthAccPortId portId)
{
    TRACE;

    /* this is a buffer returned by NPEs during a call to PortDisable: 
     * free the skb & return the mbuf to the pool 
     */

#ifdef DEBUG
#if 0
    if (IX_OSAL_MBUF_PRIV(mbuf) != NULL)
    {
    	struct net_device *ndev = (struct net_device*)callbackTag;
	priv_data_t *priv = netdev_priv(ndev);

	priv->skb_alloc_count--;
    }
#endif
#endif
    mbuf_free_skb(mbuf);

    TRACE;
}


/* This callback is called when transmission of the packed is done, and
 * IxEthAcc does not need the buffer anymore. The port is down or
 * a portDisable is running. The action is to free the buffers
 * to the pools.
 */
static void tx_done_disable_cb(UINT32 callbackTag, IX_OSAL_MBUF *mbuf)
{
    struct net_device *dev = (struct net_device *)callbackTag;
    priv_data_t *priv = netdev_priv(dev);

    TRACE;

    priv->stats.tx_packets++; /* total packets transmitted */
    priv->stats.tx_bytes += IX_OSAL_MBUF_MLEN(mbuf); /* total bytes transmitted */

#ifdef DEBUG
    if (IX_OSAL_MBUF_PRIV(mbuf) != NULL)
    {
	priv->skb_alloc_count--;
    }
#endif
    /* extract skb from the mbuf, free skb and return the mbuf to the pool */
    mbuf_free_skb(mbuf);

    TRACE;
}


/**************************************************************
 *		DATA TRANSMIT FUNCTIONS 		      *
 **************************************************************/

/* 
 * this function is called by the kernel to transmit packet 
 * It is expected to run in the context of the ksoftirq thread.
 *
 * It is important to note that DO NOT call dev_skb_enqueue and dev_skb_dequeue
 * from this function. That will cause reentry issue of the above function
 * because this function will be called from software interrupt context, which
 * is likely to preempt tasks under the job queue.
 */

static int dev_ixp400_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    int res;
    IX_OSAL_MBUF *mbuf;
    priv_data_t *priv = netdev_priv(dev);

    TRACE;

    /* get mbuf struct from tx software queue */
    mbuf = dev_tx_mb_dequeue(priv);

    if (mbuf == NULL)
    {
	priv->stats.tx_dropped++;

	/*
	 * Stops the net_device queue. The device queue will be enabled by
	 * TxDone job 
	 */
	netif_stop_queue(dev);

	dev_kfree_skb_any(skb);
#ifdef DEBUG
	priv->skb_alloc_count--;
#endif
	return NETDEV_TX_OK;
    }

#ifdef DEBUG_DUMP
    skb_dump("tx", skb);
#endif

    /*
     * Push the skb data by the size of CUSTOMIZED_ETH_HDR_OFFSET if customized
     * Ethernet header is enabled
     */
    if (priv->customized_eth_hdr)
    {
    	skb_push(skb, CUSTOMIZED_ETH_HDR_OFFSET);
    }

    /*
     * This function call is expected to return NULL, as mbufs in the pool have
     * no skb attached
     */
    mbuf_swap_skb(mbuf, skb); 
    IX_OSAL_MBUF_PKT_LEN(mbuf) = skb->len;

    /* set ethernet flags to zero */
    IX_ETHACC_NE_FLAGS(mbuf) = 0;

    /* flush the mbuf data from the cache */
    IX_OSAL_CACHE_FLUSH(IX_OSAL_MBUF_MDATA(mbuf), IX_OSAL_MBUF_MLEN(mbuf));

    dev->trans_start = jiffies;

    if ((res = ixEthAccPortTxQoSFrameSubmit(priv->port_id, mbuf,
	IX_ETH_ACC_TX_TRAFFIC_CLASS_3)))
    {

    	TRACE;

	priv->stats.tx_dropped++;

	/* Submit failed, return the mbuf to the mbuf pool */
	mbuf_swap_skb(mbuf, NULL); 
	dev_tx_mb_enqueue(priv, mbuf);

	dev_kfree_skb_any(skb);
#ifdef DEBUG
	priv->skb_alloc_count--;
#endif

    	if (IX_ETH_TX_Q_FULL == res)
	{
	    /*
	     * Stops the net_device queue and enable the Tx queue notification
	     */
	    netif_stop_queue(dev);

	    res = ixQMgrNotificationEnable(
	    	IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id,
		IX_ETH_ACC_TX_TRAFFIC_CLASS_3),
		TX_Q_NOTIFY);

	    if (IX_SUCCESS != res && IX_QMGR_WARNING != res)
	    {
		P_ERROR ("ixQMgrNotificationEnable failed for TX queue %d with "
			 "%d\n",
			IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id,
			IX_ETH_ACC_TX_TRAFFIC_CLASS_3),
			res);
	    }

	    P_DEBUG("Transmit queue full\n");
	}
	else /* Other errors */
	{
	    P_ERROR("%s: ixEthAccPortTxFrameSubmit failed for port %d, res = %d\n",
		    dev->name, priv->port_id, res);
	}

	return NETDEV_TX_OK;
    }

    TRACE;

    return NETDEV_TX_OK;
}

/**************************************************************
 *		DATA RECEIVE FUNCTIONS	      	      	      *
 **************************************************************/

static void rx_frame_drop(priv_data_t *priv, IX_OSAL_MBUF *mbuf)
{
    struct sk_buff *skb;
    /* 
     * Invalid frame, stop hammering the system
     * or we received an unexpected unsupported chained mbuf
     */
    TRACE;

    /* update the stats */
    priv->stats.rx_dropped++; 

    /* Return unchained mbufs to the RX mbuf pool */
    do
    {
    	IX_OSAL_MBUF *next;

	/* extract skb from the mbuf, free skb */
	if (NULL != (skb = mbuf_swap_skb(mbuf, NULL)))
	{
#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
	    /* recycle skb for later use in rx (fast) */
	    dev_skb_enqueue(priv, skb);
#else
	    /* put to kernel pool (slow) */
	    dev_kfree_skb_any(skb);
#ifdef DEBUG
	    priv->skb_alloc_count--;
#endif
#endif
	}

	next = IX_OSAL_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf);
	IX_OSAL_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) = NULL;
	IX_OSAL_MBUF_MLEN(mbuf) = priv->replenish_size;

	/* return the RX mbuf to the mbuf pool */
	dev_rx_mb_enqueue(priv, mbuf);

	mbuf = next;
    } while (mbuf != NULL);
}

/*
 * This is a utility function performs pre-processing of the raw mbuf data
 * before the skb can pass up to the stack
 */
static inline struct sk_buff *rx_frame_process(struct net_device *ndev,
					       IX_OSAL_MBUF *mbuf)
{
    priv_data_t *priv = netdev_priv(ndev);
    unsigned int mcastFlags;
    struct sk_buff *skb;
    int len;

    TRACE;

    len = IX_OSAL_MBUF_MLEN(mbuf);
    mcastFlags = IX_ETHACC_NE_FLAGS(mbuf);

    /* extract skb from mbuf */
    skb = mbuf_swap_skb(mbuf, NULL);

    /* return the RX mbuf to the mbuf pool */
    dev_rx_mb_enqueue(priv, mbuf);

    /* set the length of the received skb from the mbuf length  */
    skb->tail = skb->data + len;
    skb->len = len;

    /*
     * Offset the skb>-data by the size of CUSTOMIZED_ETH_HDR_OFFSET of 
     * customized Ethernet header is enabled
     */
    if (priv->customized_eth_hdr)
    {
    	skb_pull(skb, CUSTOMIZED_ETH_HDR_OFFSET);
    }
    
#ifdef DEBUG_DUMP
    skb_dump("rx", skb);
#endif

    /* Set the skb protocol and set mcast/bcast flags */
    dev_eth_type_trans(mcastFlags, skb, ndev);

    /* update the stats */
    priv->stats.rx_packets++; /* total packets received */
    priv->stats.rx_bytes += skb->len; /* total bytes received */

    /* Set the skb protocol and set mcast/bcast flags */
    return skb;
}


/* 
 * This is called when new packet received from MAC
 * and ready to be transfered up-stack.
 *
 */
inline int rx_poll(u32 entries_to_process)
{

#ifdef CONFIG_IXP400_ETH_NAPI
    static IxEthAccMbuf mbufs[IXP400_NAPI_WEIGHT+1];
#else
    static IxEthAccMbuf mbufs[RX_FRAME_RETRIEVE+1];
#endif
    struct net_device *ndev;
    priv_data_t *priv;
    IX_OSAL_MBUF *mbuf;
    UINT32 port_id;
    int ret;
    int i;

    TRACE;

    /* get the time of this interrupt : all buffers received during this
     * interrupt will be assigned the same time */
    do_gettimeofday(&irq_stamp);

    ret = ixEthAccRxQPriorityRetrieve(entries_to_process, mbufs);

    for (i = 0; i < ret; i++)
    {
    	mbuf = mbufs[i].mbufPtr;
	port_id = mbufs[i].callbackTag;
	ndev = net_devices[port_id];
	priv = netdev_priv(ndev);

	/* check if the system accepts more traffic and
	 * against chained mbufs 
	 */
	if (IX_OSAL_MBUF_NEXT_PKT_IN_CHAIN_PTR(mbuf) == NULL)
	{      
	    /* push upstack */
#ifdef CONFIG_IXP400_ETH_NAPI
	    netif_receive_skb(rx_frame_process(ndev, mbuf)); 
#else
	    netif_rx(rx_frame_process(ndev, mbuf)); 
#endif

#ifdef DEBUG
	    priv->skb_alloc_count--;
#endif
	}
	else
	{
	    rx_frame_drop(priv, mbuf);
	}
    }
    
    return ret;
}


#ifdef CONFIG_IXP400_ETH_NAPI
static int dev_rx_poll(struct net_device *netdev, int *budget)
{
    UINT32 entries_to_process = min(*budget, netdev->quota);
    UINT32 entries_processed = 0;
    UINT32 queue_entries = 0;

restart_poll:

    entries_processed = rx_poll(entries_to_process);

    *budget -= entries_processed;
    netdev->quota -= entries_processed;

    /* if not enough entries processed */
    if(entries_processed < entries_to_process)
    {
        netif_rx_complete(netdev);
        ixEthAccQMgrRxQEntryGet(&queue_entries);

        if(queue_entries > 0 && netif_rx_reschedule(netdev,entries_processed))
        {
            goto restart_poll;
        }

        ixEthAccQMgrRxNotificationEnable();

	/* 
	 * Polling is completed, remove the device from NAPI polling list
	 */
        return 0;
    }

    /*
     * There are more frames to poll, keep the device in the NAPI polling list
     */
    return 1;
}
#endif

/**************************************************************
 *		DATAPATH CALLBACK FUNCTIONS	      	      *
 **************************************************************/

/*
 * Data Receive Event handler
 */
static void rx_ehandler_cb (UINT32 callbackTag)
{
    /* Note: there are 2 possible race conditions where the normal
     * EthAcc QMgr receive callback will be invoked by dispatcherFunc()
     * to drain the rx queues rather than polling them via dev_rx_poll:
     * 1. interrupt occurs while dev_rx_poll is being closed and
     *    netif_rx_schedule_prep() fails because the device is no 
     *    longer open.
     * 2. traffic is received after dev_rx_poll enables interrupts,
     *    but before it de-schedules itself.
     */ 
#ifdef CONFIG_IXP400_ETH_NAPI
    if(netif_rx_schedule_prep(rx_poll_dev))
    {
        ixEthAccQMgrRxNotificationDisable();
        __netif_rx_schedule(rx_poll_dev);
    }
#else
    ixEthAccQMgrRxNotificationDisable();
    schedule_work(&rx_work);
#endif
}

/*
 * Receieve free event handler
 */
static void rx_free_ehandler_cb(UINT32 callbackTag)
{
    struct net_device *ndev = (struct net_device *) callbackTag;
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    ixEthAccQMgrRxFreeNotificationDisable(priv->port_id);

    schedule_work(&priv->rx_free_w.task);
}

/*
 * Frame transmitted event handler
 */
static void tx_ehandler_cb(UINT32 callbackTag, IxEthAccTxTrafficClass trafficClass)
{
    struct net_device *ndev = (struct net_device *) callbackTag;
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    if (likely(netif_queue_stopped(ndev)))
    {
    	netif_wake_queue(ndev);
    }

    ixEthAccQMgrTxNotificationDisable(priv->port_id, trafficClass);
}

/*
 * Frame transmitted done event handler
 */
static void tx_done_ehandler_cb(UINT32 callbackTag)
{
    struct net_device *ndev = (struct net_device *) callbackTag;
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    ixEthAccQMgrTxDoneNotificationDisable(priv->port_id);

    schedule_work(&priv->tx_done_w.task);
}


/*
 * The callback functions below are to be registered during port disabling. They
 * simply avoid infinite job rescheduling by disabling the notification when
 * callback is called when the port is disabled during heavy traffic
 */

static void rx_free_disabled_ehandler_cb(UINT32 callbackTag)
{
    struct net_device *ndev = (struct net_device *) callbackTag;
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    ixEthAccQMgrRxFreeNotificationDisable(priv->port_id);
}

static void tx_disabled_ehandler_cb (UINT32 callbackTag, 
				     IxEthAccTxTrafficClass trafficClass)
{
    struct net_device *ndev = (struct net_device *) callbackTag;
    priv_data_t *priv = netdev_priv(ndev);
    IxEthAccTxTrafficClass tc;

    TRACE;

    /*
     * Disable the notification from all of the traffic classes
     */
    for (tc = IX_ETH_ACC_TX_TRAFFIC_CLASS_0;
    	 tc < IX_ETH_ACC_TX_TRAFFIC_CLASS_MAX;
	 tc++)
    {
	ixEthAccQMgrTxNotificationDisable(priv->port_id, tc);
    }
}


/**************************************************************
 *		DATAPATH DEFERRED TASKS 	      	      *
 **************************************************************/

/* This callback is called when transmission of the packed is done, and
 * IxEthAcc does not need the buffer anymore. The buffers will be returned to
 * the software queues.
 */
static inline void tx_done_frame_process(priv_data_t *priv, IX_OSAL_MBUF *mbuf)
{
    IX_OSAL_MBUF *__mbuf = mbuf;
    IX_OSAL_MBUF *__mbuf_next;

    TRACE;

    /*
     * It is possible that the tx done mbuf is chained with interframe chain.
     * Loop thru the chain and free up each of the mbuf
     */
    do 
    {
	priv->stats.tx_packets++; /* total packets transmitted */
	priv->stats.tx_bytes += IX_OSAL_MBUF_MLEN(__mbuf);/* total bytes transmitted*/

	/* next mbuf from the interframe pointer */
	__mbuf_next = ixEthAccTxDoneInterFrameUnchain(priv->port_id, __mbuf);

	/* extract skb from the mbuf, free skb */
#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
	/* recycle skb for later use in rx (fast) */
	dev_skb_enqueue(priv, mbuf_swap_skb(__mbuf, NULL));
#ifdef DEBUG
	priv->skb_alloc_count++;
#endif
#else
	/* put to kernel pool (slow) */
	dev_kfree_skb_any(mbuf_swap_skb(__mbuf, NULL));
#endif

	/* return the mbuf to the queue */
	dev_tx_mb_enqueue(priv, __mbuf);

    } while (NULL != (__mbuf = __mbuf_next));
}

/*
 * Internal function that services the TxDone queue
 */
static inline void __tx_done_job (struct net_device *ndev)
{
    static IX_OSAL_MBUF *mbufs[TXDONE_Q_DEPTH];
    priv_data_t *priv = netdev_priv(ndev);
    int ret, i;

    TRACE;

    /*
     * Retrieve the mbuf from the hardware queue. The number of mbuf to retrieve
     * is the queue depth of the tx done hardware queue.
     */
    ret = ixEthAccTxDoneQRetrieve(priv->port_id, TXDONE_Q_DEPTH, mbufs);

    TRACE;

    /*
     * Process each of the mbuf retrieved
     */
    for (i = 0; i < ret;  i++)
    {
	tx_done_frame_process(priv, mbufs[i]);
    }

    TRACE;

    /*
     * Wake up the queue if it's stopped
     */
    if (netif_queue_stopped(ndev))
    {
    	netif_wake_queue(ndev);
    }

    TRACE;
}

/*
 * Deferred task to handler TX done event
 */
static void tx_done_job(struct work_struct *work)
{
    struct tx_done_work *tx_done_w = container_of(work, struct tx_done_work,
					task);
    struct net_device *ndev = tx_done_w->ndev;
    priv_data_t *priv = netdev_priv(ndev);

    __tx_done_job (ndev);

    /*
     * Enable the tx done notification
     */
    ixEthAccQMgrTxDoneNotificationEnable(priv->port_id);
}

static int rx_free_replenish(priv_data_t *priv, u32 num_to_replenish)
{
    IX_OSAL_MBUF *mbuf;
    struct sk_buff *skb;
    int buf_replenished = 0;
    int total_buf_to_replenish = 0;
    int __num_to_replenish;
    IX_OSAL_MBUF **__mbuf_array_ptr = priv->rxfree_mb;
    int i;
    UINT32 rx_free_q_entries;

    TRACE;

    rx_free_q_entries = 0;

#ifndef CONFIG_IXP400_ETH_SKB_RECYCLE
    ixQMgrQNumEntriesGet(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id), 
    	&rx_free_q_entries);
#endif
    __num_to_replenish = min(RXFREE_Q_DEPTH - rx_free_q_entries, 
    	(UINT32)num_to_replenish);

    /*
     * Filling up the mbuf array for replenishment up to the number indicated 
     * by the notification threshold
     */
    for (i = 0; i < __num_to_replenish; i++)
    {
    	/* Get an mbuf from the RX mbuf queue */
    	if (NULL == (mbuf = dev_rx_mb_dequeue(priv)))
	{
	    P_DEBUG("dev_rx_mb_dequeue() returning NULL for port %d\n",
	    	priv->port_id);
	    break;
	}

	/* Get an skbuff from the skbuff queue */
	if (unlikely (NULL == (skb = dev_skb_dequeue(priv))))
	{
	    P_DEBUG("dev_skb_dequeue() returning NULL for port %d\n", 
	    	priv->port_id);
	    dev_rx_mb_enqueue(priv, mbuf);
	    break;
	}

	/*
	 * Attach the skb to the mbuf 
	 */
	mbuf_swap_skb(mbuf, skb);

	/*
	 * Add the mbuf to the mbuf array
	 */
	*(__mbuf_array_ptr++) = mbuf;
    }

    *__mbuf_array_ptr = NULL; /* Set the last entry to NULL */

    /*
     * No buffer for replenishment
     */
    if (unlikely(0 == i))
    {
    	return 0;
    }

    total_buf_to_replenish = i;

    /*
     * Replenish the mbufs
     */
    buf_replenished = ixEthAccPortRxFreeMultiBufferReplenish(priv->port_id,
	priv->rxfree_mb);

    /*
     * Return over replenished buffers to the pool
     */
    for (;buf_replenished < i; i--)
    {
    	TRACE;

    	mbuf = *(--__mbuf_array_ptr); /* extract the last mbuf */

#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
	dev_skb_enqueue(priv, mbuf_swap_skb(mbuf, NULL));
#else
	dev_kfree_skb_any(mbuf_swap_skb(mbuf, NULL));
#ifdef DEBUG
	priv->skb_alloc_count--;
#endif
#endif
	dev_rx_mb_enqueue(priv, mbuf);
    }

    return buf_replenished;
}


#ifndef CONFIG_IXP400_ETH_NAPI
/*
 * Deferred task handling RX event
 */
static void rx_job(struct work_struct *work)
{
    rx_poll(RX_FRAME_RETRIEVE);
    ixEthAccQMgrRxNotificationEnable();
}
#endif

/*
 * Deferred task handling RX free event
 */
static void rx_free_job(struct work_struct *work)
{
    struct rx_free_work *rx_free_w = container_of (work, struct rx_free_work, 
    					task);
    struct net_device *ndev = rx_free_w->ndev;
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    rx_free_replenish(priv, RXFREE_Q_DEPTH);

    /* Enable the RX free notification */
    ixEthAccQMgrRxFreeNotificationEnable(priv->port_id);
}

/**************************************************************
 *		PORT ENABLING AND DISABLING FUNCTIONS         * 
 **************************************************************/

/* Enable the MAC port.
 * Called on do_dev_open, dev_tx_timeout and mtu size changes
 */

static int port_enable(struct net_device *dev)
{
    int res;
    IxEthAccMacAddr npeMacAddr;
    priv_data_t *priv = netdev_priv(dev);

    P_DEBUG("port_enable(%s)\n", dev->name);

    /* Set MAC addr in h/w (ethAcc checks for MAC address to be valid) */
    memcpy(&npeMacAddr.macAddress,
	   dev->dev_addr,
	   IX_IEEE803_MAC_ADDRESS_SIZE);

    res = ixEthAccPortUnicastMacAddressSet(priv->port_id, &npeMacAddr);

#ifdef CONFIG_IXP400_ETH_DB
    if (IX_ETH_ACC_SUCCESS == res)
    {
    	res = ixEthDBPortAddressSet(priv->port_id, (IxEthDBMacAddr*)&npeMacAddr);
    }
#endif 

    if (res)
    {
        P_VERBOSE("Failed to set MAC address %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x for port %d\n",
	       (unsigned)npeMacAddr.macAddress[0],
	       (unsigned)npeMacAddr.macAddress[1],
	       (unsigned)npeMacAddr.macAddress[2],
	       (unsigned)npeMacAddr.macAddress[3],
	       (unsigned)npeMacAddr.macAddress[4],
	       (unsigned)npeMacAddr.macAddress[5],
	       priv->port_id);
	return -EPERM;
    }

    /* restart the link-monitoring thread if necessary */
    if (priv->maintenanceCheckStopped)
    {
	/* Starts the driver monitoring thread, if configured */
	priv->maintenanceCheckStopped = FALSE;
	
	priv->maintenanceCheckThreadId = 
	    kernel_thread(dev_media_check_thread,
			  (void *) dev,
			  CLONE_FS | CLONE_FILES);
	if (priv->maintenanceCheckThreadId < 0)
	{
	    P_ERROR("%s: Failed to start thread for media checks\n", dev->name);
	    priv->maintenanceCheckStopped = TRUE;
	}
    }

    /* set the callback supporting the traffic */
    ixEthAccPortTxEventCallbackRegister(priv->port_id, 
					tx_ehandler_cb,
					(UINT32)dev);
    ixEthAccPortTxDoneEventCallbackRegister(priv->port_id, 
					tx_done_ehandler_cb,
				       	(UINT32)dev);
    ixEthAccPortRxEventCallbackRegister(priv->port_id, 
				       	rx_ehandler_cb, 
				       	(UINT32)dev);
    ixEthAccPortRxFreeEventCallbackRegister(priv->port_id,
					    rx_free_ehandler_cb,
					    (UINT32)dev);

    TRACE;

    /* force replenish if necessary */
    dev_rx_buff_prealloc(priv);

    ixQMgrNotificationEnable(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id),
	RX_FREE_Q_NOTIFY);

    if ((res = ixEthAccPortEnable(priv->port_id)))
    {
	P_ERROR("%s: ixEthAccPortEnable failed for port %d, res = %d\n",
		dev->name, priv->port_id, res);
	return convert_error_ethAcc(res);
    }

#ifdef CONFIG_IXP400_ETH_DB
    if ((res = ixEthDBPortEnable(priv->port_id)))
    {
	P_ERROR("%s: ixEthDBPortEnable failed for port %d, res = %d\n",
		dev->name, priv->port_id, res);
	return -1;
    }

    /* Do not enable aging unless learning is enabled (nothing to age otherwise) */
    if (ethdb_mac_learning)
    {
	if ((res = ixEthDBPortAgingEnable(priv->port_id)))
	{
	    P_ERROR("%s: ixEthDBPortAgingEnable failed for port %d, res = %d\n",
		    dev->name, priv->port_id, res);
	    return -1;
	}
    }
#endif

    TRACE;

    /* reset the current time for the watchdog timer */
    dev->trans_start = jiffies;

    netif_start_queue(dev);

#ifdef CONFIG_IXP400_ETH_NAPI
    netif_poll_enable(rx_poll_dev);
#endif

    TRACE;

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED
    /* restore the driver's own TX queueing discipline */
    dev->qdisc_sleeping = priv->qdisc;
    dev->qdisc = priv->qdisc;
#endif

    TRACE;

    return 0;
}

/* Disable the MAC port.
 * Called on do_dev_stop and dev_tx_timeout
 */
static void port_disable(struct net_device *dev)
{
    priv_data_t *priv = netdev_priv(dev);
    int res, ret, i;
    IX_STATUS status;
    IxEthAccMbuf mbufs[RX_Q_DEPTH];
    IX_OSAL_MBUF *mbuf;

    P_DEBUG("port_disable(%s)\n", dev->name);

    if (!netif_queue_stopped(dev))
    {
        dev->trans_start = jiffies;
        netif_stop_queue(dev);
    }

    if (priv->maintenanceCheckStopped)
    {
	/* thread is not running */
    }
    else
    {
	/* thread is running */
	priv->maintenanceCheckStopped = TRUE;
	/* Wake up the media-check thread with a signal.
	   It will check the 'running' flag and exit */
	if ((res = kill_proc (priv->maintenanceCheckThreadId, SIGKILL, 1)))
	{
	    P_ERROR("%s: unable to signal thread\n", dev->name);
	}
	else
	{
	    /* wait for the thread to exit. */
	    down (priv->maintenanceCheckThreadComplete);
	    up (priv->maintenanceCheckThreadComplete);
	}
    }

    TRACE;

    /* 
     * set the port disabling callbacks 
     * Design note: These callbacks are in place to protect the already
     * scheduled work from re-enabling the notifications during work flush. Rx
     * and TxDone disable handler is not required as we will manually service
     * these 2 queues in this function
     */
    ixEthAccPortTxEventCallbackRegister(priv->port_id, 
					tx_disabled_ehandler_cb,
					(UINT32)dev);
    ixEthAccPortRxFreeEventCallbackRegister(priv->port_id,
					    rx_free_disabled_ehandler_cb,
					    (UINT32)dev);
    TRACE;

    /* Flush scheduled work from the work queue */
    flush_scheduled_work();

    TRACE;

    __tx_done_job(dev);

    /*
     * Set the TxDone queue to not empty condition before disabling the port to
     * force queue flush
     */
    ixQMgrNotificationEnable(
    	IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
	IX_QMGR_Q_SOURCE_ID_NOT_E);

    if ((status = ixEthAccPortDisable(priv->port_id)) != IX_SUCCESS)
    {
	/* should not get here */
	P_ERROR("%s: ixEthAccPortDisable(%d) failed with %d\n",
		dev->name, priv->port_id, status);
    }

    /*
     * Restore the TxDone queue from not empty condition to the default state
     */
    ixQMgrNotificationEnable(
    	IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
	TX_DONE_Q_NOTIFY);

    TRACE;

    /*
     * The section below performs manual RX buffer drainage. The RX queues will
     * be drained and only valid frames that are not from the disabling port
     * will be submitted to the stack using non-NAPI interface
     */
    ret = ixEthAccRxQPriorityRetrieve(RX_Q_DEPTH, mbufs);

    TRACE;

    for (i = 0; i < ret; i++)
    {
    	struct net_device *__dev;
	priv_data_t *__priv;
	u32 port_id;

    	mbuf = mbufs[i].mbufPtr;
	port_id = mbufs[i].callbackTag;
	__dev = net_devices[port_id];
	__priv = netdev_priv(__dev);

	/* 
	 * check if the received frame is not from the disabling port and 
	 * against chained mbufs 
	 */
	if (dev != __dev &&
	    (IX_OSAL_MBUF_NEXT_PKT_IN_CHAIN_PTR(mbuf) == NULL))
	{    
	    TRACE;

	    /* push upstack */
	    netif_rx(rx_frame_process(__dev, mbuf)); 
#ifdef DEBUG
	    priv->skb_alloc_count--;
#endif
	}
	else
	{
	    TRACE;

	    rx_frame_drop(__priv, mbuf);
	}
    }
    TRACE;

#ifdef CONFIG_IXP400_ETH_DB
    /* Do not need to disable aging unless learning is enabled */
    if (ethdb_mac_learning)
    {
	if ((res = ixEthDBPortAgingDisable(priv->port_id)))
	{
	    P_ERROR("%s: ixEthDBPortAgingDisable failed for port %d, res=%d\n",
		    dev->name, priv->port_id, res);
	}
    }

    TRACE;

    /* Disable ethDB port */
    if ((status = ixEthDBPortDisable(priv->port_id)) != IX_ETH_DB_SUCCESS)
    {
	/* should not get here */
	P_ERROR("%s: ixEthDBPortDisable(%d) failed\n",
		dev->name, priv->port_id);
    }
    TRACE;
#endif

    /* remove all entries from the sw queues */
    dev_skb_queue_drain(priv);
    dev_tx_mb_queue_drain(priv);
    dev_rx_mb_queue_drain(priv);

    TRACE;
}

#ifdef CONFIG_IXP400_ETH_DB
/**************************************************************
 *	CUSTOMIZED 16-BYTE ETHERNET HEADER SUPPORT	      *
 **************************************************************/

static void customized_eth_hdr_set(priv_data_t *priv)
{
    TRACE;

    priv->customized_eth_hdr = FALSE; /* reset the header offset */

    /*
     * Check if the port have customized Ethernet header enabled
     */
    if ((IX_ETH_PORT_1 == priv->port_id && customized_eth_hdr_b) || 
    	(IX_ETH_PORT_2 == priv->port_id && customized_eth_hdr_c) ||
    	(IX_ETH_PORT_3 == priv->port_id && customized_eth_hdr_a))
    {
    	TRACE;

    	priv->customized_eth_hdr = TRUE;

    	if (IX_ETH_DB_SUCCESS != ixEthDBEthHeaderOffsetSet(priv->port_id,
	    CUSTOMIZED_ETH_HDR_OFFSET))
	{
	    P_WARN("ixEthDBEthHeaderOffsetSet() for port %d failed\n",
		priv->port_id);

	    priv->customized_eth_hdr = FALSE; /* reset the header offset after
	    					 failure */
	}
    }
}


/**************************************************************
 *		ETHDB MAC LEARNING SUPPORT		      *
 **************************************************************/

/*
 * Get the MAC learning enabling status
 */
static BOOL ethdb_mac_learning_status_get(void)
{
    return ethdb_mac_learning;
}

/*
 * Enable/disable the MAC learning
 */
static int ethdb_mac_learning_enable(BOOL enable)
{
    int dev_idx = 0;
    struct net_device *tmp_dev;
    priv_data_t *priv;


    /* Do not start the EthDB maintenance thread if learning & filtering feature
     * is disabled */
    if (enable) /* module parameter */
    {
	ixFeatureCtrlSwConfigurationWrite (IX_FEATURECTRL_ETH_LEARNING, TRUE);

	/*
	 * Check for each device running status and active aging to the
	 * active device
	 */
        for(dev_idx = 0; dev_idx < dev_max_count; dev_idx++)
        {
	    tmp_dev = dev_get_drvdata(&ixp400_eth_devices[dev_idx].dev);
	    priv = netdev_priv(tmp_dev);

            if(netif_running(tmp_dev))
            {
		if (IX_ETH_DB_SUCCESS != ixEthDBPortAgingEnable(priv->port_id))
		{
		    P_WARN("%s: ixEthDBPortAgingEnable failed for port %d\n",
			   tmp_dev->name, priv->port_id);
		}
            }
        }

	/* Setup maintenance task workqueue */
	ethdb_maintenance_timer_set();
    }
    else
    {
	/* stop the maintenance timer */
	ethdb_maintenance_timer_clear();

	/* Wait for maintenance task to complete (if started) */
	ETHDB_MAINTENANCE_MUTEX_LOCK();
	ETHDB_MAINTENANCE_MUTEX_UNLOCK();

	/*
	 * Check for each device running status and active aging to the
	 * active device
	 */
        for(dev_idx = 0; dev_idx < dev_max_count; dev_idx++)
        {
	    tmp_dev = dev_get_drvdata(&ixp400_eth_devices[dev_idx].dev);
	    priv = netdev_priv(tmp_dev);

            if(netif_running(tmp_dev))
            {
		if (IX_ETH_DB_SUCCESS != ixEthDBPortAgingDisable(priv->port_id))
		{
		    P_WARN("%s: ixEthDBPortAgingDisable failed for port %d\n",
			   tmp_dev->name, priv->port_id);
		}
            }
        }

	ixFeatureCtrlSwConfigurationWrite (IX_FEATURECTRL_ETH_LEARNING, FALSE);
    }

    ethdb_mac_learning = enable;

    return 0;
}

/**************************************************************
 *			MODULE UPGRADE SUPPORT		      *
 **************************************************************/

/*
 * Set module upgrade flag
 */
static int module_upgrade (BOOL enable)
{
    module_upgrade_shutdown = enable;

    return 0;
}

/*
 * Get the module upgrade flag
 */
static BOOL module_upgrade_status_get (void)
{
    return module_upgrade_shutdown;
}


/**************************************************************
 *			FAST PATH SUPPORT		      *
 **************************************************************/

#define FPATH_DATA_SIZE	1520 /* 1518 + 2 bytes customized 16-byte Ethernet
				header support */
#define FPATH_MBUF_POOL_SIZE 64
#define FPATH_REPLENISH_RETRY 10

/*
 * Fast-path callback function -- called during fast-path port disabling to
 * return the fast-path mbuf
 *
 * Design note: The callbackTag is the port ID
 */
static void fpath_cb(UINT32 callbackTag, IX_OSAL_MBUF *mbuf)
{
    IX_OSAL_MBUF_POOL_PUT(mbuf);
}

/*
 * Enabling the fast-path
 */
static int fpath_enable (priv_data_t *priv)
{
    IxEthDBFpathRxFreeMemInfo fpath_meminfo;
    IX_OSAL_MBUF_POOL *mbuf_pool = NULL;
    IX_OSAL_MBUF_POOL *__mbuf_pool = NULL;
    IX_OSAL_MBUF *mbuf;
    int res = 0;

    if (!priv)
    {
    	return -EINVAL;
    }

    /*
     * Initialize the fpath_meminfo before use
     */
    memset (&fpath_meminfo, 0, sizeof(IxEthDBFpathRxFreeMemInfo));

    /* 
     * Get the already saved fast-path mbuf pool base address
     */
    res = ixEthDBFpathRxFreeMemInfoGet(priv->port_id, &fpath_meminfo);
    if (IX_ETH_DB_SUCCESS != res)
    {
    	P_ERROR("ixEthDBFpathRxFreeMemInfoGet() failed for port %d: %d\n",
	    priv->port_id, res);
	return -EPERM;
    }

    /*
     * determine if the fast-path already enabled by checking the saved mbuf
     * pool base address
     */
    if (0 != fpath_meminfo.fpathMemInfo2)
    {
    	return 0; /* fast path already enabled */
    }

    /*
     * Create pool of mbuf for fast-path replenishment
     */
    mbuf_pool = IX_OSAL_MBUF_POOL_INIT(FPATH_MBUF_POOL_SIZE, FPATH_DATA_SIZE,
	"IXP400 Ethernet driver fast path RX pool");

    if (!mbuf_pool)
    {
    	P_ERROR("Fast path RxFree memory pool allocation failed for port %d\n",
	    priv->port_id);
	return -ENOMEM;
    }


    /*
     * Workaround to resolve OSAL mbuf limitation for module upgrade support
     */
    __mbuf_pool = (IX_OSAL_MBUF_POOL*)kmalloc(sizeof(IX_OSAL_MBUF_POOL),
	GFP_KERNEL);

    BUG_ON(!__mbuf_pool);

    /*
     * Replenishes the allocated mbuf
     */
    while (NULL != (mbuf = IX_OSAL_MBUF_POOL_GET(mbuf_pool)))
    {
    	int i;

	mbuf->ix_ctrl.ix_pool = __mbuf_pool;

    	for (i = 0; i < FPATH_REPLENISH_RETRY; i++)
	{
	    res = ixEthAccFPathPortRxFreeReplenish(priv->port_id, mbuf);
	    if (IX_ETH_ACC_SUCCESS == res)
	    {
	    	break;
	    }
	}

	/*
	 * Print a warning message if the fast-path replenishement failed for
	 * numerious number of time
	 */
	if (FPATH_REPLENISH_RETRY == i)
	{
	    P_WARN("Fast path RxFree replenishment failed after %d attempts: %d\n",
	    	    i, res);

	    IX_OSAL_MBUF_POOL_PUT(mbuf);
	    break;
	}
    }
    memcpy(__mbuf_pool, mbuf_pool, sizeof(IX_OSAL_MBUF_POOL));

    mbuf_pool->totalBufsInPool = 0;
    mbuf_pool->freeBufsInPool = 0;

    IX_OSAL_MBUF_POOL_UNINIT(mbuf_pool);

    /* 
     * Saves the mbuf_pool base address to the NPE
     */
    fpath_meminfo.fpathMemInfo2 = (UINT32)__mbuf_pool;

    res = ixEthDBFpathRxFreeMemInfoStore(priv->port_id, &fpath_meminfo);

    if (IX_ETH_DB_SUCCESS != res)
    {
    	P_WARN("Failed to store fast path RxFree mbuf pool base address: %d\n",
	    res);
    }

    return 0;
}

/*
 * Disabling the fast path and return the fast-path mbuf to the system pool
 */
static int fpath_disable (priv_data_t *priv)
{
    IxEthDBFpathRxFreeMemInfo fpath_meminfo;
    IX_OSAL_MBUF_POOL *mbuf_pool = NULL;
    IX_OSAL_MBUF_POOL *__mbuf_pool = NULL;
    IX_OSAL_MBUF *__next_free_buff = NULL;
    u32 pool_idx;

    TRACE;

    if (!priv)
    {
    	return -EINVAL;
    }

    /*
     * Safety net to protect user from calling fast path port disabling while
     * mode upgrade flag is set
     */
    if (module_upgrade_shutdown)
    {
    	P_WARN("Module upgrade flag is set, bailing out from fast path port "
	    "disabling for port %d\n", priv->port_id);

	return -EINVAL;
    }

    /* 
     * Initialize the fpath_meminfo before use
     */
    memset(&fpath_meminfo, 0, sizeof(IxEthDBFpathRxFreeMemInfo));

    TRACE;

    /*
     * Get the fast-path mbuf pool memory base address from the NPE
     */
    if (IX_ETH_DB_SUCCESS != ixEthDBFpathRxFreeMemInfoGet(priv->port_id,
	&fpath_meminfo))
    {
    	P_ERROR("ixEthDBFpathRxFreeMemInfoGet() failed\n");

	return -EPERM;
    }

    TRACE;

    /*
     * Check if the mbuf pool base address is NULL
     */
    if (0 == fpath_meminfo.fpathMemInfo2)
    {
    	return 0; /* fast path is already disabled */
    }

    /*
     * save the mbuf memory base address
     */
    mbuf_pool = (IX_OSAL_MBUF_POOL*)fpath_meminfo.fpathMemInfo2;

    /* 
     * Reset meminfo 
     */
    fpath_meminfo.fpathMemInfo1 = 0; /* memory size -- not used */
    fpath_meminfo.fpathMemInfo2 = 0; /* mbuf pool memory base address */
    if (IX_ETH_DB_SUCCESS != ixEthDBFpathRxFreeMemInfoStore(priv->port_id,
	&fpath_meminfo))
    {
    	P_ERROR("ixEthDBFpathRxFreeMemInfoStore() failed\n");
    }

    /* 
     * Disabling the fast path. EthAcc will call fpath_cb() to drain out all the
     * RxFree mbuf.
     */
    if (IX_ETH_ACC_SUCCESS != ixEthAccFPathPortDisable(priv->port_id))
    {
    	P_ERROR("ixEthAccFPathPortDisable() failed, port %d\n", priv->port_id);

	return -EPERM;
    }

    TRACE;

    /*
     * Workaround of mbuf pool limited support against module upgrade
     */
    __mbuf_pool = IX_OSAL_MBUF_POOL_INIT(1, 0, "dummy");
    BUG_ON(!__mbuf_pool);
    __next_free_buff = __mbuf_pool->nextFreeBuf;
    IX_OSAL_MBUF_NEXT_BUFFER_IN_PKT_PTR(__next_free_buff) =
	mbuf_pool->nextFreeBuf;
    mbuf_pool->nextFreeBuf = __next_free_buff;
    pool_idx = __mbuf_pool->poolIdx;
    mbuf_pool->totalBufsInPool++;
    mbuf_pool->freeBufsInPool++;
    memcpy(__mbuf_pool, mbuf_pool, sizeof(IX_OSAL_MBUF_POOL));
    __mbuf_pool->poolIdx = pool_idx;

    /*
     * Uninitialize the mbuf pool
     */
    IX_OSAL_MBUF_POOL_UNINIT(__mbuf_pool);

    kfree(mbuf_pool);

    return 0;
}
#endif

#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
/**************************************************************
 *      PARITY ERROR DETECTION & NPE SOFT RESET FUNCTIONS     *
 **************************************************************/

/*
 * Parity error callback which will be called upon parity error detected
 */
static void parity_error_cb (void)
{
    IxParityENAccParityErrorContextMessage	error_context;
    UINT32					irqlock;

    irqlock = ixOsalIrqLock();

    /*
     * Initialize the error context structure
     */
    memset (&error_context, 0xFF, 
	    sizeof (IxParityENAccParityErrorContextMessage));

    /*
     * Collect the source of error. Silently ignore if the error context get did
     * not return success. This is because it might due to data abort is not
     * caused by parity error
     */
    if (likely(IX_PARITYENACC_SUCCESS == 
    	ixParityENAccParityErrorContextGet (&error_context)))
    {
    	IxErrHdlAccFuncHandler func = NULL;

	/*
	 * Now find out the source of the error and retrive respective handling
	 * function
	 */
	switch (error_context.pecParitySource)
	{
	    case IX_PARITYENACC_NPE_A_IMEM:
	    case IX_PARITYENACC_NPE_A_DMEM:
	    case IX_PARITYENACC_NPE_A_EXT:
		ixErrHdlAccErrorHandlerGet(IX_ERRHDLACC_NPEA_ERROR, &func);
		break;

	    case IX_PARITYENACC_NPE_B_IMEM:
	    case IX_PARITYENACC_NPE_B_DMEM:
	    case IX_PARITYENACC_NPE_B_EXT:
		ixErrHdlAccErrorHandlerGet(IX_ERRHDLACC_NPEB_ERROR, &func);
		break;

	    case IX_PARITYENACC_NPE_C_IMEM:
	    case IX_PARITYENACC_NPE_C_DMEM:
	    case IX_PARITYENACC_NPE_C_EXT:
		ixErrHdlAccErrorHandlerGet(IX_ERRHDLACC_NPEC_ERROR, &func);
		break;

	    default:
	    	P_ERROR("Unknown error source\n");
	}

	/* 
	 * Call the handling function if the function pointer is not null
	 */
	if (likely(func))
	{
	    (*func)();
	}
    }

    ixOsalIrqUnlock(irqlock);
}

/*
 * NPE recovery done callback. This callback is to notify of any recovery
 * failure.
 */
void parity_npe_recovery_done_cb(IxErrHdlAccErrorEventType event_type)
{
    UINT32 status;

    /* 
     * Requires a 10ms delay or so , This is to allow the other Interrupt error
     * e.g NPE B and NPE C to trigger. This is required if you have NPE A, NPE B
     * and NPE C parity error triggered at the same time. If you don't have
     * to de-schedule code here, the interrupt ISR of NPE B and NPE C will not
     * be allow to execute and the right status will not be read. Why? Because
     * the thread/task in ixErrHdlAcc priority is higher (which where this
     * call-back is called) than the ISR priority level here.
     */
     ixOsalSleep(10);

    /*
     * Collect the status from the handler to detect any possibility failure on
     * recovery
     */
    ixErrHdlAccStatusGet(&status);

    switch (event_type)
    {
    	case IX_ERRHDLACC_NPEA_ERROR:
	    if (unlikely(IX_ERRHDLACC_NPEA_ERROR_MASK_BIT & status))
	    {
		P_ERROR("NPE-A recovery failed\n");
	    }

	    break;

    	case IX_ERRHDLACC_NPEB_ERROR:
	    if (unlikely(IX_ERRHDLACC_NPEB_ERROR_MASK_BIT & status))
	    {
		P_ERROR("NPE-B recovery failed\n");
	    }

	    break;

    	case IX_ERRHDLACC_NPEC_ERROR:
	    if (unlikely(IX_ERRHDLACC_NPEC_ERROR_MASK_BIT & status))
	    {
		P_ERROR("NPE-C recovery failed\n");
	    }

	    break;

	default:
		P_ERROR("Unknown NPE recovery error\n");
    }
}

/*
 *  Parity detection and NPE recovery initialization
 */
static int __init parity_npe_error_handler_init()
{
    /*
     * Parity detection hardware configuration parameter. The configured
     * fields are well explained by the self-explanatory data member name
     */
    IxParityENAccHWParityConfig parity_config =
    {
    	.npeAConfig = 
	    {
	    	.ideEnabled 			= parity_npeA_enabled,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },
	.npeBConfig =
	    {
	    	.ideEnabled			= parity_npeB_enabled,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },
	.npeCConfig =
	    {
	    	.ideEnabled			= parity_npeC_enabled,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },

	.mcuConfig =
	    {
	    	.singlebitDetectEnabled 	= IX_PARITYENACC_DISABLE,
	    	.singlebitCorrectionEnabled 	= IX_PARITYENACC_DISABLE,
	    	.multibitDetectionEnabled 	= IX_PARITYENACC_DISABLE,
	    },

	.swcpEnabled				= IX_PARITYENACC_DISABLE,

	.aqmEnabled				= IX_PARITYENACC_DISABLE,

	.pbcConfig =
	    {
	    	.pbcInitiatorEnabled		= IX_PARITYENACC_DISABLE,
		.pbcTargetEnabled		= IX_PARITYENACC_DISABLE,
	    },

	.ebcConfig =
	    {
	    	.ebcCs0Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs1Enabled 			= IX_PARITYENACC_DISABLE,
	    	.ebcCs2Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs3Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs4Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs5Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs6Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs7Enabled			= IX_PARITYENACC_DISABLE,
		.ebcExtMstEnabled		= IX_PARITYENACC_DISABLE,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },
    };
    IxParityENAccHWParityConfig parity_config_check;
    UINT32 error_handler_config = 0;

    if (npe_error_handler_initialized)
    {
    	return 0;
    }

    /*
     * We must set the initialized flag here so that un-initialization will pick
     * it up and uninitialize regardless of which phase in the initialization is
     * failed
     */
    npe_error_handler_initialized = 1;

    /*
     * Error recovery handler initialization 
     */
    if (IX_SUCCESS != ixErrHdlAccInit())
    {
    	P_ERROR("NPE error recovery initialization failed\n");

	return -EBUSY;
    }

    /* 
     * Error handler recovery done callback registration
     */
    if (IX_SUCCESS != ixErrHdlAccCallbackRegister(parity_npe_recovery_done_cb))
    {
    	P_ERROR("NPE error recovery callback registration failed\n");

	return -EBUSY;
    }

    /*
     * Configuring the events and hardware errors that the error recovery 
     * handler needs to pay attention on
     */
    if (IX_PARITYENACC_ENABLE == parity_npeA_enabled)
    {
    	error_handler_config |= IX_ERRHDLACC_NPEA_ERROR_MASK_BIT;
    }

    if (IX_PARITYENACC_ENABLE == parity_npeB_enabled)
    {
    	error_handler_config |= IX_ERRHDLACC_NPEB_ERROR_MASK_BIT;
    }

    if (IX_PARITYENACC_ENABLE == parity_npeC_enabled)
    {
    	error_handler_config |= IX_ERRHDLACC_NPEC_ERROR_MASK_BIT;
    }

    if (IX_SUCCESS != ixErrHdlAccEnableConfigSet(error_handler_config))
    {
    	P_ERROR("NPE error recovery configuration failed\n");

	return -EBUSY;
    }

    /*
     * Parity access layer initialization
     */
    if (unlikely(IX_PARITYENACC_SUCCESS != ixParityENAccInit()))
    {
	P_ERROR("Parity initialization failed\n");

	return -EBUSY;
    }

    /*
     * Registering parity error handling callback
     */
    if (unlikely(IX_PARITYENACC_SUCCESS != 
    	ixParityENAccCallbackRegister(parity_error_cb)))
    {
	P_ERROR("Parity callback registration failed\n");

	return -EBUSY;
    }

    /*
     * Configure parity error events that we are interested in
     */
    if (unlikely(IX_PARITYENACC_SUCCESS !=
	ixParityENAccParityDetectionConfigure(&parity_config)))
    {
	P_ERROR("Parity detection configuration failed\n");

	return -EBUSY;
    }
   	

    /*
     * Read back the configured value to detect possible inconsistency setup
     * errors
     */
    if (unlikely(IX_PARITYENACC_SUCCESS !=
	ixParityENAccParityDetectionQuery(&parity_config_check)))
    {
	P_ERROR("Parity detection query failed\n");

	return -EBUSY;
    }

    /*
     * Now we compare the value we read with the value we set. Report the error
     * if the data is inconsistent
     */
    if (unlikely(0 != memcmp((void*)&parity_config, (void*)&parity_config_check,
	sizeof(IxParityENAccHWParityConfig))))
    {
	P_ERROR("Parity configuration failed\n");

	return -EINVAL;
    }

    /* Initialization successful */
    P_INFO("NPE Error Handler Initialization Successful\n");

    return 0;
}

/*
 * Parity recovery and NPE error handling un-initialization
 */
static void parity_npe_error_handler_uninit(void)
{
    /*
     * We need to have ugly event disabling parameter here to disable the parity
     * access component due to lack of unload function
     */
    IxParityENAccHWParityConfig parity_config =
    {
    	.npeAConfig = 
	    {
	    	.ideEnabled 			= IX_PARITYENACC_DISABLE,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },

	.npeBConfig =
	    {
	    	.ideEnabled			= IX_PARITYENACC_DISABLE,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },

	.npeCConfig =
	    {
	    	.ideEnabled			= IX_PARITYENACC_DISABLE,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },

	.mcuConfig =
	    {
	    	.singlebitDetectEnabled 	= IX_PARITYENACC_DISABLE,
	    	.singlebitCorrectionEnabled 	= IX_PARITYENACC_DISABLE,
	    	.multibitDetectionEnabled 	= IX_PARITYENACC_DISABLE,
	    },

	.swcpEnabled				= IX_PARITYENACC_DISABLE,

	.aqmEnabled				= IX_PARITYENACC_DISABLE,

	.pbcConfig =
	    {
	    	.pbcInitiatorEnabled		= IX_PARITYENACC_DISABLE,
		.pbcTargetEnabled		= IX_PARITYENACC_DISABLE,
	    },

	.ebcConfig =
	    {
	    	.ebcCs0Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs1Enabled 			= IX_PARITYENACC_DISABLE,
	    	.ebcCs2Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs3Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs4Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs5Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs6Enabled			= IX_PARITYENACC_DISABLE,
	    	.ebcCs7Enabled			= IX_PARITYENACC_DISABLE,
		.ebcExtMstEnabled		= IX_PARITYENACC_DISABLE,
		.parityOddEven			= IX_PARITYENACC_EVEN_PARITY,
	    },
    };
    UINT32 irqlock;

    if (!npe_error_handler_initialized)
    {
	return;
    }

    /*
     * Disable the interrupts before changing the parity configuration
     */
    irqlock = ixOsalIrqLock();
    	
    /*
     * Disable all listening events
     */
    ixParityENAccParityDetectionConfigure(&parity_config);

    /*
     * Unregister the parity callback function by passing in NULL pointer
     */
    ixParityENAccCallbackRegister(NULL);

    /*
     * Workaround to unbind the IRQs bound by the parity access layer
     */
    ixOsalIrqUnbind(IX_OSAL_IXP400_NPEA_IRQ_LVL);
#ifdef CONFIG_CPU_IXP46X
    ixOsalIrqUnbind(IX_OSAL_IXP400_NPEB_IRQ_LVL);
#endif
    ixOsalIrqUnbind(IX_OSAL_IXP400_NPEC_IRQ_LVL);
    ixOsalIrqUnbind(IX_OSAL_IXP400_PCI_INT_IRQ_LVL);
#ifdef CONFIG_CPU_IXP46X
    ixOsalIrqUnbind(IX_OSAL_IXP400_SWCP_IRQ_LVL);
#endif
    ixOsalIrqUnbind(IX_OSAL_IXP400_AQM_IRQ_LVL);
    ixOsalIrqUnbind(IX_OSAL_IXP400_MCU_IRQ_LVL);
#ifdef CONFIG_CPU_IXP46X
    ixOsalIrqUnbind(IX_OSAL_IXP400_EBC_IRQ_LVL);
#endif

    /*
     * Parity uninitialization completed. Enable the interrupt here, error
     * handler unload function is sleepable
     */
    ixOsalIrqUnlock(irqlock); 

    /*
     * Finally we unload the error handler
     */
    ixErrHdlAccUnload();

    npe_error_handler_initialized = 0;

    P_INFO("Parity NPE Error Handler Uninitialized\n");
}
#endif

    
/* Initialize device structs.
 * Resource allocation is deffered until do_dev_open
 */
static int __devinit dev_eth_probe(struct device *dev)
{
    priv_data_t *priv = NULL;
    struct net_device *ndev = NULL;
    IxEthAccPortId portId = to_platform_device(dev)->id;
    IX_STATUS status = IX_SUCCESS;
    IxEthAccTxTrafficClass tc;
    int res;

    TRACE;

    BUG_ON(!(ndev = alloc_etherdev(sizeof(priv_data_t))));

    SET_MODULE_OWNER(ndev);
    SET_NETDEV_DEV(ndev, dev);
    dev_set_drvdata(dev, ndev);
    priv = netdev_priv(ndev);
    net_devices[portId] = ndev;

#ifdef CONFIG_IXP400_ETH_SKB_RECYCLE
    priv->skQueueHead = 0;
    priv->skQueueTail = 0;
    memset(priv->skQueue, 0, sizeof(struct sk_buff*) * SKB_QSIZE);
#endif

    TRACE;

    /* Initialize the ethAcc port */
    if ((res = ixEthAccPortInit(portId)))
    {
    	P_ERROR("ixEthAccPortInit portId %i: No such device, res = %d\n",
	    portId, res);
        return -ENODEV;
    }

#ifdef CONFIG_IXP400_ETH_DB
    /* Initialize the ethDB port */
    ixEthDBPortInit(portId);
#endif

    /* set the private port ID */
    priv->port_id  = portId;

    customized_eth_hdr_set(priv);

    if (hss_coexist)
    {
	TRACE;

	for (tc = IX_ETH_ACC_TX_TRAFFIC_CLASS_0;
	     tc < IX_ETH_ACC_TX_TRAFFIC_CLASS_MAX;
	     tc++)
	{  
	    /*
	     * Queue policy and priority setup for RX queue for each traffic
	     * class
	     */
	    status |= ixQMgrCallbackTypeSet(
		IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(tc), 
		IX_QMGR_TYPE_REALTIME_SPORADIC);
	    status |= ixQMgrDispatcherPrioritySet (
	    	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(tc),
	    	IX_QMGR_Q_PRIORITY_2);

	    /*
	     * Queue policy and priority setup for TX queue for each traffic
	     * class
	     */
	    status |= ixQMgrCallbackTypeSet(
			    IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id, tc),
			    IX_QMGR_TYPE_REALTIME_SPORADIC);
	    status |= ixQMgrDispatcherPrioritySet(
			    IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id, tc),
			    IX_QMGR_Q_PRIORITY_1);
	}

	/*
	 * Queue policy and priority setup for RX free queue
	 */
	status |= ixQMgrCallbackTypeSet(
			IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id),
			IX_QMGR_TYPE_REALTIME_SPORADIC);
	status |= ixQMgrDispatcherPrioritySet (
			IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id),
			IX_QMGR_Q_PRIORITY_1);

	/*
	 * Queue policy and priority setup for TxDone queue
	 */
	status |= ixQMgrCallbackTypeSet(
	    		IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
			IX_QMGR_TYPE_REALTIME_SPORADIC);
	status |= ixQMgrDispatcherPrioritySet(
			IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
			IX_QMGR_Q_PRIORITY_1);


	if (IX_SUCCESS != status)
	{
	    P_ERROR("Failed to set queue policy for HSS-Ethernet co-exist\n");
	}

	TRACE;
    }

    /*
     * Configuring QMgr watermark
     */
    status = ixQMgrWatermarkSet(
    	IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id,IX_ETH_ACC_TX_TRAFFIC_CLASS_3),
	TX_Q_NE_THRESHOLD, 
	TX_Q_NF_THRESHOLD);

    status |= ixQMgrWatermarkSet(
    	IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
	TX_DONE_Q_NE_THRESHOLD, 
	TX_DONE_Q_NF_THRESHOLD);

    status |= ixQMgrWatermarkSet(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id),
	RX_FREE_Q_NE_THRESHOLD, 
	RX_FREE_Q_NF_THRESHOLD);

    status |= ixQMgrWatermarkSet(
    	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(IX_ETH_ACC_RX_TRAFFIC_CLASS_0),
	RX_Q_NE_THRESHOLD_TC0, 
	RX_Q_NF_THRESHOLD_TC0);

    status |= ixQMgrWatermarkSet(
    	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(IX_ETH_ACC_RX_TRAFFIC_CLASS_1),
	RX_Q_NE_THRESHOLD_TC1, 
	RX_Q_NF_THRESHOLD_TC1);

    status |= ixQMgrWatermarkSet(
    	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(IX_ETH_ACC_RX_TRAFFIC_CLASS_2),
	RX_Q_NE_THRESHOLD_TC2, 
	RX_Q_NF_THRESHOLD_TC2);

    status |= ixQMgrWatermarkSet(
    	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(IX_ETH_ACC_RX_TRAFFIC_CLASS_3),
	RX_Q_NE_THRESHOLD_TC3, 
	RX_Q_NF_THRESHOLD_TC3);

    if (IX_SUCCESS != status)
    {
	P_WARN("Failed to configure queue watermark\n");
    }

    /*
     * Configuring and Enable Queue Notification
     * Design note 1: TX queue notification will not be enabled here (it's
     * currentlty disabled). It'll be enabled when the TX queue is full
     * Design note 2: RX Free queue notification will not be enabled here.
     * It's enabled enabled after RX free replenishement
     */
    status |= ixQMgrNotificationEnable(
    	IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
	TX_DONE_Q_NOTIFY);

    status |= ixQMgrNotificationEnable(
	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(
	    IX_ETH_ACC_RX_TRAFFIC_CLASS_0),
	RX_Q_NOTIFY_TC0);

    status |= ixQMgrNotificationEnable(
	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(
	    IX_ETH_ACC_RX_TRAFFIC_CLASS_1),
	RX_Q_NOTIFY_TC1);

    status |= ixQMgrNotificationEnable(
	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(
	    IX_ETH_ACC_RX_TRAFFIC_CLASS_2),
	RX_Q_NOTIFY_TC2);

    status |= ixQMgrNotificationEnable(
	IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(
	    IX_ETH_ACC_RX_TRAFFIC_CLASS_3),
	RX_Q_NOTIFY_TC3);

    if (IX_SUCCESS != status)
    {
	P_WARN("Failed to configure queue notification\n");
    }

    /* set device name */
    sprintf(ndev->name, DEVICE_NAME"%d", priv->port_id);

    TRACE;

    /* initialize RX pool */
    priv->rx_pool = IX_OSAL_MBUF_POOL_INIT(RX_MBUF_POOL_SIZE, 0,
				           "IXP400 Ethernet driver Rx Pool");
    if(priv->rx_pool == NULL)
    {
	P_ERROR("%s: Buffer RX Pool init failed on port %d\n",
		ndev->name, priv->port_id);
	goto error;
    }

    TRACE;

    /* initialize TX pool */
    priv->tx_pool = IX_OSAL_MBUF_POOL_INIT(TX_MBUF_POOL_SIZE, 0, 
				           "IXP400 Ethernet driver Tx Pool");
    if(priv->tx_pool == NULL)
    {
	P_ERROR("%s: Buffer TX Pool init failed on port %d\n",
		ndev->name, priv->port_id);
	goto error;
    }

     TRACE;

   /* initialise the MII register access mutex */
    priv->maintenanceCheckThreadComplete = (struct semaphore *)
	kmalloc(sizeof(struct semaphore), GFP_KERNEL);
    if (!priv->maintenanceCheckThreadComplete)
    {
	goto error;
    }
    priv->lock = SPIN_LOCK_UNLOCKED;
    init_MUTEX(priv->maintenanceCheckThreadComplete);
    priv->maintenanceCheckStopped = TRUE;

    /* initialize ethernet device (default handlers) */
    ether_setup(ndev);

    TRACE;

     /* fill in dev struct callbacks with customized handlers */
    ndev->open = do_dev_open;
    ndev->stop = do_dev_stop;

    ndev->hard_start_xmit = dev_ixp400_hard_start_xmit;

    ndev->watchdog_timeo = DEV_WATCHDOG_TIMEO;
    ndev->tx_timeout = dev_tx_timeout;
    ndev->change_mtu = dev_change_mtu;
    ndev->do_ioctl = do_dev_ioctl;
    ndev->get_stats = dev_get_stats;
    ndev->set_multicast_list = ixp400_dev_set_multicast_list;
    ndev->flags |= IFF_MULTICAST;

    ndev->set_mac_address = ixp400_dev_set_mac_address;

#ifdef CONFIG_IXP400_ETH_NAPI
    ndev->poll = &dev_rx_poll;
    ndev->weight = IXP400_NAPI_WEIGHT;

    /* initialize the rx_poll_dev device */
    if(NULL == rx_poll_dev)
        rx_poll_dev = ndev;
#endif

    TRACE;

    /* Defines the unicast MAC address
     *
     * Here is a good place to read a board-specific MAC address
     * from a non-volatile memory, e.g. an external eeprom.
     * 
     * This memcpy uses a default MAC address from this
     * source code.
     *
     * This can be overriden later by the (optional) command
     *
     *     ifconfig ixp0 ether 0002b3010101
     *
     */

    memcpy(ndev->dev_addr, 
	   &default_mac_addr[priv->port_id].macAddress,
	   IX_IEEE803_MAC_ADDRESS_SIZE);

    /* possibly remove this test and the message when a valid MAC address 
     * is not hardcoded in the driver source code. 
     */
    if (is_valid_ether_addr(ndev->dev_addr))
    {
	P_WARN("Use default MAC address %2.2x:%2.2x:%2.2x:%2.2x:%2.2x:%2.2x for port %d\n",
	       (unsigned)ndev->dev_addr[0],
	       (unsigned)ndev->dev_addr[1],
	       (unsigned)ndev->dev_addr[2],
	       (unsigned)ndev->dev_addr[3],
	       (unsigned)ndev->dev_addr[4],
	       (unsigned)ndev->dev_addr[5],
	       priv->port_id);
    }
    
    /* Set/update the internal packet size 
     * This can be overriden later by the command
     *                      ifconfig ixp0 mtu 1504
     */
    TRACE;

    dev_change_msdu(ndev, ndev->mtu + ndev->hard_header_len + VLAN_HDR);

    /* create timeout queue to handle transmission timeout */
    priv->timeout_workq = create_singlethread_workqueue(MODULE_NAME" timeout");
    BUG_ON(!priv->timeout_workq);

    priv->timeout_w.ndev = ndev;
    priv->rx_free_w.ndev = ndev;
    priv->tx_done_w.ndev = ndev;

    INIT_WORK(&priv->timeout_w.task, dev_tx_timeout_task);
    INIT_WORK(&priv->rx_free_w.task, rx_free_job);
    INIT_WORK(&priv->tx_done_w.task, tx_done_job);

    /* set the internal maximum queueing capabilities */
    ndev->tx_queue_len = TX_MBUF_POOL_SIZE;

    if (!netif_queue_stopped(ndev))
    {
	TRACE;

        ndev->trans_start = jiffies;
        netif_stop_queue(ndev);
    }

    TRACE;

    if (register_netdev(ndev))
    	goto error;

    TRACE;

    /* configuring the port  */
    if (dev_port_init(portId, (UINT32)ndev))
    {
    	goto error;
    }

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED
    /* configure and enable a fast TX queuing discipline */
    TRACE;

    priv->qdisc = qdisc_create_dflt(ndev, &dev_qdisc_ops, 0);
    ndev->qdisc_sleeping = priv->qdisc;
    ndev->qdisc = priv->qdisc;
    
    if (!ndev->qdisc_sleeping)
    {
	P_ERROR("%s: qdisc_create_dflt failed on port %d\n",
		ndev->name, priv->port_id);
	goto error;
    }
#endif /* CONFIG_IXP400_ETH_QDISC_ENABLED */

    goto done;

/* Error handling: enter here whenever error detected */
error:
    TRACE;

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED
    if (ndev->qdisc)
    	qdisc_destroy(ndev->qdisc);
#endif

    /* Step 1: Destroying workqueue */
    if (priv && priv->timeout_workq)
    {
    	flush_workqueue(priv->timeout_workq);
	destroy_workqueue(priv->timeout_workq);
	priv->timeout_workq = NULL;
    }

    /* Step 2: detaching ndev from dev */
    dev_set_drvdata(dev, NULL);

    /* Step 3: Unregister and free ndev */
    if (ndev)
    {
    	if (ndev->reg_state == NETREG_REGISTERED)
	    unregister_netdev(ndev);

	free_netdev(ndev);
    }

    TRACE;

    return -ENOMEM;

/* Escape from the routine if no error */
done:
    return 0;
}

static int __devexit dev_eth_remove(struct device *dev)
{
    struct net_device *ndev = dev_get_drvdata(dev);
    priv_data_t *priv = netdev_priv(ndev);

    TRACE;

    if (priv != NULL)
    {
	IxEthAccPortId portId = to_platform_device(dev)->id;

	if (priv->rx_pool)
	    IX_OSAL_MBUF_POOL_UNINIT(priv->rx_pool);

	if (priv->tx_pool)
	    IX_OSAL_MBUF_POOL_UNINIT(priv->tx_pool);

	if (!module_upgrade_shutdown) /* module upgrading disabled */
	{
	    if (IX_SUCCESS != 
		ixNpeDlNpeStopAndReset(default_npeImageId[portId].npeId))
	    {
		P_NOTICE("Error Halting NPE for Ethernet port %d!\n", portId);
	    }
	}

	TRACE;

	if (priv->timeout_workq)
	{
	    flush_workqueue(priv->timeout_workq);
	    destroy_workqueue(priv->timeout_workq);
	    priv->timeout_workq = NULL;
	}

#ifdef CONFIG_IXP400_ETH_QDISC_ENABLED
	if (ndev->qdisc)
	{
	    qdisc_destroy(ndev->qdisc);
	}
#endif
	unregister_netdev(ndev);

	free_netdev(ndev);
	dev_set_drvdata(dev,NULL);
    }

    return 0;
}


static void set_drv_data(void)
{
#if   defined (CONFIG_IXP400_ETH_NPEA_ONLY)
    ixp400_eth_devices[0].id = IX_ETH_PORT_3;
#elif defined (CONFIG_IXP400_ETH_NPEB_ONLY)
    ixp400_eth_devices[0].id = IX_ETH_PORT_1;
#elif defined (CONFIG_IXP400_ETH_NPEC_ONLY)
    ixp400_eth_devices[0].id = IX_ETH_PORT_2;
#elif defined (CONFIG_MACH_KIXRP435)
    ixp400_eth_devices[0].id = IX_ETH_PORT_2;
#else
    ixp400_eth_devices[0].id = IX_ETH_PORT_1;
#endif

#if   defined (CONFIG_MACH_KIXRP435)
    ixp400_eth_devices[1].id = IX_ETH_PORT_3;
#else
    ixp400_eth_devices[1].id = IX_ETH_PORT_2;
#endif

    ixp400_eth_devices[2].id = IX_ETH_PORT_3;
}

static int __init ixp400_eth_init(void)
{
    int res, dev_count;


    TRACE;

    P_INFO("Initializing IXP400 NPE Ethernet driver software v. " MOD_VERSION " \n");

    TRACE;

    /* check module parameter range */
    if (dev_max_count == 0 || dev_max_count > IX_ETH_ACC_NUMBER_OF_PORTS)
    {
	P_ERROR("Number of ports supported is dev_max_count <= %d\n", IX_ETH_ACC_NUMBER_OF_PORTS);
	return -1;
    }

    TRACE;

    set_drv_data();

    TRACE;

#ifndef DEBUG
    /* check module parameter range */
    if (log_level >= 2)  /* module parameter */
    {
	printk("Warning : log_level == %d and TRACE is disabled\n", log_level);
    }
#endif

    TRACE;

    /* display an approximate CPU clock information */
    P_INFO("CPU clock speed (approx) = %lu MHz\n",
	   loops_per_jiffy * 2 * HZ / 1000000);

    TRACE;

    /*
     * IXP43X and IXP46X supports the feature below:
     * 1. Ethernet on NPE-A
     * 2. Ethernet-HSS coexist capability
     * 3. NPE software error handler
     */
#if defined (CONFIG_CPU_IXP46X) || defined (CONFIG_CPU_IXP43X)
    {
         UINT32 expbusCtrlReg;

	/* Set the expansion bus fuse register to enable MUX for NPEA MII */
        expbusCtrlReg = ixFeatureCtrlRead ();
        expbusCtrlReg |= ((unsigned long)1<<8);
	ixFeatureCtrlWrite (expbusCtrlReg);

	/*
	 * HSS coexist NPE enabler
	 */
	if (hss_coexist)
	{
	    int i;
	    IxEthAccPortId portId;

	    for (i = 0; i < dev_max_count; i++)
	    {
		portId = default_portId[i];
	   	
		if (IX_NPEDL_NPEID_NPEA == default_npeImageId[portId].npeId)
		{
		    /*
		     * Prepare to download the HSS-Ethernet coexist image into 
		     * NPE-A
		     */
		    default_npeImageId[portId].npeImageId = 
		    	IX_HSS_ETH_NPE_A_IMAGE_ID;
		}
	    }
	}
    }
#else
    {
	if (npe_error_handler)
	{
	    P_ERROR("NPE error handling is not supported on this platform\n");
	    npe_error_handler = 0;
	}

	if (hss_coexist)
	{
	    P_ERROR("HSS-Ethernet co-exist is not supported on this "
		"platform\n");
	    hss_coexist = 0;
	}
    }
#endif

    TRACE;

    /* Enable/disable the EthDB MAC Learning & Filtering feature.
     * This is a half-bridge feature, and should be disabled if this interface 
     * is used on a bridge with other non-NPE ethernet interfaces.
     * This is because the NPE's are not aware of the other interfaces and thus
     * may incorrectly filter (drop) incoming traffic correctly bound for another
     * interface on the bridge.
     */
    TRACE;


    /* Do not initialise core components if no_ixp400_sw_init is set */
    if (no_ixp400_sw_init) /* module parameter */
    {
	P_WARN("no_ixp400_sw_init != 0, no IXP400 SW core component initialisation performed\n");
    }
    else
    {
    	u32 dev_count;
	u32 fastpath_npe_active = 0;
	IxEthAccPortId portId;

	if (module_upgrade_init)
	{
	    for (dev_count = 0; 
		 dev_count < dev_max_count;  /* module parameter */
		 dev_count++)
	    {
		portId = default_portId[dev_count];

		if (TRUE == ixNpeDlNpeActiveCheck(default_npeImageId[portId].npeId))
		{
		    /* Check if both fast-path NPEs are active */
		    if (IX_ETH_PORT_2 == portId || IX_ETH_PORT_3 == portId)
		    {
			fastpath_npe_active += 1;
		    }
		}
	    }

	    /* both fast-path NPEs are active */
	    if (2 == fastpath_npe_active)
	    {
		/* 
		 * Fast path NPEs are active, assuming recovery from module 
		 * upgrade 
		 */
		ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
		    TRUE);
	    }
	    else
	    {
		/* NPE is non-active, assuming it's the fresh loading */
		ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
		    FALSE);
	    }
	}
	else
	{
	    ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
		FALSE);
	}

	/* initialize the required components for this driver */
	if ((res = qmgr_init()))
	    return res;
        TRACE;

	/*
	 * NPE error handling is enabled, initializae the message handler with
	 * polling mode and setup the PMU timer to poll for NPE messages
	 */
	if (npe_error_handler)
	{

	    TRACE;

	    if (IX_SUCCESS != 
		(res = ixNpeMhInitialize(IX_NPEMH_NPEINTERRUPTS_NO)))
		return -1;

	    /* 
	     * Setting up PMU timer to poll for NPE messages
	     */
	    if ((res = dev_pmu_timer_setup()))
		return res;
	}
	else
	{
	    TRACE;

	    if (IX_SUCCESS != 
	    	(res = ixNpeMhInitialize(IX_NPEMH_NPEINTERRUPTS_YES)))
		return -1;
	}
    }

    /* Initialise the NPEs and access layer */
    TRACE;

    if ((res = ethacc_init()))
	return res;

    TRACE;

    /* Initialise the PHYs */
    if ((res = phy_init()))
	return res;

    TRACE;

    if ((res = driver_register(&ixp400_eth_driver)))
    	return res;

    TRACE;

    /* Initialise the driver structure */
    for (dev_count = 0; 
	 dev_count < dev_max_count;  /* module parameter */
	 dev_count++)
    {
        if ((res = platform_device_register(&ixp400_eth_devices[dev_count])))
        {
            P_ERROR("failure in registering device %d. res = %d\n", 
                    dev_count, res);
            return res;
        }

        TRACE;
    }

    if (IX_ETH_ACC_SUCCESS !=
	ixEthAccMacRecoveryLoopStart(MAC_RECOVERY_STACK_SIZE, 
	MAC_RECOVERY_PRIORITY))
    {
    	P_WARN("ixEthAccMacRecoveryLoopStart() failed\n");
    }

    TRACE;

    if (npe_error_handler)
    {
	/*
	 * Initialize parity detection and NPE error handler
	 */
	if (parity_npe_error_handler_init())
	{
	    P_ERROR("NPE error handler initialization failed\n");
	    parity_npe_error_handler_uninit();
	}
    }

    TRACE;

    /* initialise the DB Maintenance task mutex */
#ifdef CONFIG_IXP400_ETH_DB
    ethdb_maintenance_mutex = (struct semaphore *) kmalloc(sizeof(struct semaphore), GFP_KERNEL);
    if (!ethdb_maintenance_mutex)
	return -ENOMEM;

    init_MUTEX(ethdb_maintenance_mutex);

    TRACE;
#endif

    /* Initialization completed, turn off module upgrade support flag */
    ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
	FALSE);

    TRACE;

    return 0;
}

void __exit ixp400_eth_exit(void)
{
    int dev_count;

    TRACE;

    /* We can only get here when the module use count is 0,
     * so there's no need to stop devices.
     */

    /* stop the maintenance timer */
    if (ethdb_mac_learning)
    {
	ethdb_maintenance_timer_clear();

	/* Wait for maintenance task to complete (if started) */
	ETHDB_MAINTENANCE_MUTEX_LOCK();
	ETHDB_MAINTENANCE_MUTEX_UNLOCK();
    }

    TRACE;

#ifdef CONFIG_IXP400_ETH_DB
    /*
     * Check if module upgrade is enable. Disable fast path if 
     * it's not enabled.
     */
    if (!module_upgrade_shutdown) 
    {
	struct net_devices *tmp_dev;
	priv_data_t *priv;

	/*
	 * Workaround:
	 * Setting this to true so that ethAcc fast path port disabling is able
	 * to distinguish if the system is performing fast path port shutdown 
	 * after recovered from module upgrade
	 */
	ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
	    TRUE);

	for (dev_count = 0; 
	     dev_count < dev_max_count;  /* module parameter */
	     dev_count++)
	{
	    tmp_dev = dev_get_drvdata(&ixp400_eth_devices[dev_count].dev);
	    priv = (priv_data_t*) netdev_priv(tmp_dev);

	    fpath_disable(priv);
	}

	/*
	 * Setting the module upgrade flag to false state
	 */
	ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
	    FALSE);
    }
    else
    {
	ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_MODULE_UPGRADE,
	    TRUE);
    }
#endif

    /* uninitialize the access layers */
    ethacc_uninit();

    TRACE;

    for (dev_count = 0; 
	 dev_count < dev_max_count;  /* module parameter */
	 dev_count++)
    {
	platform_device_unregister(&ixp400_eth_devices[dev_count]);
    }

    TRACE;

    driver_unregister(&ixp400_eth_driver);

    TRACE;

    /*
     * Uninitialize parity detection and NPE error handler
     */
    if (npe_error_handler)
    {
	parity_npe_error_handler_uninit();
    }

    if (hss_coexist)
    {
	/*
	 * Disable live lock (default setting)
	 */
	ixFeatureCtrlSwConfigurationWrite (IX_FEATURECTRL_ORIGB0_DISPATCHER,
	    IX_FEATURE_CTRL_SWCONFIG_ENABLED);
    }


    if (0 == no_ixp400_sw_init) /* module parameter */
    {
        free_irq(IX_OSAL_IXP400_QM1_IRQ_LVL,(void *)IRQ_ANY_PARAMETER);

	if (IX_SUCCESS != ixQMgrUnload())
	{
	    P_ERROR("QMgr unloading failed!\n");
	}

	if (IX_SUCCESS != ixNpeMhUnload())
	{
		P_ERROR("NPE MH unloading failed\n");
	}

        if(npe_error_handler)
            dev_pmu_timer_unload();

	TRACE;
    }

    TRACE;

#ifdef CONFIG_IXP400_ETH_DB
    kfree(ethdb_maintenance_mutex);
#endif

    kfree(miiAccessMutex);

    P_VERBOSE("IXP400 NPE Ethernet driver software uninstalled\n");
}

#ifdef DEBUG
void mbuf_count_get(void)
{
    int dev_idx;
    struct net_device *ndev;
    priv_data_t *priv;

    for (dev_idx = 0; dev_idx < dev_max_count; dev_idx++)
    {
    	unsigned num_q_entries;
	int total_rx = 0; 
	int total_tx = 0;
	int tc;
	char message[2048];
	int msg_ost = 0;

	ndev = dev_get_drvdata(&ixp400_eth_devices[dev_idx].dev);
	priv = netdev_priv(ndev);

	memset(message, '\0', sizeof(message));
	msg_ost += sprintf(message, "\nport %d\n", priv->port_id);
	msg_ost += sprintf(message+msg_ost, "=======\n");

	for (tc = IX_ETH_ACC_RX_TRAFFIC_CLASS_0; 
	     tc < IX_ETH_ACC_RX_TRAFFIC_CLASS_MAX;
	     tc++)
	{
	    ixQMgrQNumEntriesGet(
		IX_ETH_ACC_RX_TRAFFIC_CLASS_TO_RX_Q_ID(tc), &num_q_entries);
	    total_rx += num_q_entries;

	    msg_ost += sprintf(message+msg_ost, "RX tc %d queue\t\t: %d\n", tc, 
		num_q_entries);
	}
	ixQMgrQNumEntriesGet(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(priv->port_id),
	    &num_q_entries);
	total_rx += num_q_entries;

	msg_ost += sprintf(message+msg_ost, "RX free queue\t\t: %d\n", 
	    num_q_entries);

	total_rx += IX_OSAL_MBUF_POOL_FREE_COUNT(priv->rx_pool) +
	    (priv->mbRxQueueHead - priv->mbRxQueueTail);

	msg_ost += sprintf(message+msg_ost, "RX MBUF pool\t\t: %d\n",
	    IX_OSAL_MBUF_POOL_FREE_COUNT(priv->rx_pool));

	msg_ost += sprintf(message+msg_ost, "RX MBUF software queue\t: %d\n",
	    priv->mbRxQueueHead - priv->mbRxQueueTail);

	msg_ost += sprintf(message+msg_ost, "*** Total RX mbufs(may short of "
	    "one due to NPE pre-fetch buffer)\t: %d\n\n", total_rx);

	for (tc = IX_ETH_ACC_TX_TRAFFIC_CLASS_0; 
	     tc < IX_ETH_ACC_TX_TRAFFIC_CLASS_MAX;
	     tc++)
	{
	    ixQMgrQNumEntriesGet(
	    	IX_ETH_ACC_PORT_TO_TX_Q_ID(priv->port_id, tc),
		&num_q_entries);
	    total_tx += num_q_entries;

	    msg_ost += sprintf(message+msg_ost, "TX tc %d queue\t\t: %d\n", tc, 
		num_q_entries);
	}
	ixQMgrQNumEntriesGet(IX_ETH_ACC_PORT_TO_TX_DONE_Q_ID(priv->port_id),
	    &num_q_entries);
	total_tx += num_q_entries;

	msg_ost += sprintf(message+msg_ost, "TX Done queue\t\t: %d\n", 
	    num_q_entries);

	total_tx += IX_OSAL_MBUF_POOL_FREE_COUNT(priv->tx_pool ) +
	    (priv->mbTxQueueHead - priv->mbTxQueueTail);

	msg_ost += sprintf(message+msg_ost, "TX MBUF pool\t\t: %d\n",
	    IX_OSAL_MBUF_POOL_FREE_COUNT(priv->tx_pool));

	msg_ost += sprintf(message+msg_ost, "TX MBUF software queue\t: %d\n", 
	    priv->mbTxQueueHead - priv->mbTxQueueTail);

	msg_ost += sprintf(message+msg_ost, "*** Total TX mbufs\t\t: %d\n", 
	    total_tx);

	printk("%s\n", message);
    }
}

EXPORT_SYMBOL(mbuf_count_get);

void skbuff_count_get(void)
{
    int dev_idx;
    struct net_device *ndev;
    priv_data_t *priv;

    for (dev_idx = 0; dev_idx < dev_max_count; dev_idx++)
    {
	ndev = dev_get_drvdata(&ixp400_eth_devices[dev_idx].dev);
	priv = netdev_priv(ndev);

	printk("port %d:\n", priv->port_id);
	printk("=======\n");
	printk("number of skbuff: %d\n", priv->skQueueHead - priv->skQueueTail);
	printk("number of active skbuff : %d\n", priv->skb_alloc_count);
    }
}

EXPORT_SYMBOL(skbuff_count_get);
#endif

module_init(ixp400_eth_init);
module_exit(ixp400_eth_exit);
