/**
 * drivers/i2c/i2c-adap-ixp400.c
 *
 * Description: Driver for dedicated I2C hardware based i2c adapter on
 *  IXP4XX systems.
 *
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright (c) 2004-2007  Intel Corporation. All Rights Reserved. 
 * 
 * @par 
 * This software program is licensed subject to the GNU
 * General Public License (GPL). Version 2, June 1991, available at
 * http://www.fsf.org/copyleft/gpl.html
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/slab.h>     /* kmalloc */
#include <asm/uaccess.h>    /* copy_to_user and copy_from_user */
#include <linux/version.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/i2c.h>      /* struct i2c_msg and others */
#include <linux/i2c-id.h>

#include "IxI2cDrv.h"
#include "i2c-adap-ixp400.h"

#ifdef MODULE
MODULE_AUTHOR("Intel Corp");
MODULE_LICENSE("GPL");
#endif /* MODULE */

static BOOL ixp400_i2c_init_complete = FALSE;

/**
 * static function declarations
 */
static int __init ixp400i2cdev_init (void);
static void ixp400i2cdev_cleanup (void);
static int ixp400i2cdev_add_bus (struct i2c_adapter *i2c_adap);
static int ixp400i2cdev_del_bus (struct i2c_adapter *i2c_adap);
static int ixp400i2cdev_master_xfer (
    struct i2c_adapter *i2c_adap,
    struct i2c_msg msgs[],
    int num);
static int ixp400i2cdev_algo_control (
    struct i2c_adapter *adapter,
    unsigned int cmd,
    unsigned long arg);

#ifdef MODULE
module_init(ixp400i2cdev_init);
module_exit(ixp400i2cdev_cleanup);
#endif /* MODULE */

static u32 ixp400i2c_func(struct i2c_adapter *i2c_adap)
{
    return I2C_FUNC_I2C | I2C_FUNC_SMBUS_EMUL;
}

/**
 * static struct
 */
static struct i2c_algorithm i2c_ixp_algorithm =
{
    master_xfer:    ixp400i2cdev_master_xfer,
    smbus_xfer:     NULL,
    algo_control:   ixp400i2cdev_algo_control,
    functionality:  ixp400i2c_func
};

/* The functions that are mapped below refer to the APIs in the algorithm
    module */
static struct i2c_algo_ixp_data i2c_ixp_algo_data =
{
    init:                       ixI2cDrvInit,
    uninit:                     ixI2cDrvUninit,
    slave_addr_set:             ixI2cDrvSlaveAddrSet,
    bus_scan:                   ixI2cDrvBusScan,
    master_receive:             ixI2cDrvReadTransfer,
    master_transmit:            ixI2cDrvWriteTransfer,
    slave_and_gen_call_check:   ixI2cDrvSlaveAddrAndGenCallDetectedCheck,
    slave_receive:              ixI2cDrvSlaveOrGenDataReceive,
    slave_transmit:             ixI2cDrvSlaveDataTransmit,
    slave_buf_replenish:        ixI2cDrvSlaveOrGenCallBufReplenish,
    stats_get:                  ixI2cDrvStatsGet,
    stats_reset:                ixI2cDrvStatsReset,
    show:                       ixI2cDrvShow,
    delay_select:               ixI2cDrvDelayTypeSelect
};

static struct i2c_adapter i2c_ixp_adap =
{
    name:                   "IXP-I2C-Adapter",
    owner:		    THIS_MODULE,
    algo:		    &i2c_ixp_algorithm,
    algo_data:              &i2c_ixp_algo_data,
#if KERNEL_VERSION(2,6,0) > LINUX_VERSION_CODE
    inc_use:                NULL,
    dec_use:                NULL,
#endif
    client_register:        NULL,
    client_unregister:      NULL,
};

/**
 * static function definitions
 */

/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_init (void)
 *
 * This API will perform the initialization required to install
 * the IXP I2C module.
 */
static int __init ixp400i2cdev_init (void)
{
    return (ixp400i2cdev_add_bus(&i2c_ixp_adap));
    
} /* end of ixp400i2cdev_init */


/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_cleanup (void)
 *
 * This API will perform the clean up before the IXP I2C module
 * can be removed.
 */
static void ixp400i2cdev_cleanup (void)
{
    struct i2c_algo_ixp_data *adap = i2c_ixp_adap.algo_data;

    /* Check if init completed else do not call uninit */
    if(TRUE == ixp400_i2c_init_complete)
    {
        adap->uninit();
    } /* end of TRUE == ixp400_i2c_init_complete */

    ixp400i2cdev_del_bus(&i2c_ixp_adap);
    printk(KERN_INFO "I2C: Successfully removed bus\n");
    
    return;
} /* end of ixp400i2cdev_cleanup */


/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_add_bus (struct i2c_adapter *i2c_adap)
 *
 * This API will register the IXP I2C adapter to the I2C module
 */
static int ixp400i2cdev_add_bus (struct i2c_adapter *i2c_adap)
{
    int status;
    struct i2c_algo_ixp_data *_adap = NULL;

    /* Check if i2c_adap is NULL */
    if(NULL == i2c_adap)
    {
        printk(KERN_ERR "i2c_adap NULL pointer provided\n");
        return -2;
    } /* end of i2c_adap is NULL */

    _adap = i2c_adap->algo_data;

    printk(KERN_INFO"I2C: Adding %s.\n", i2c_adap->name);

    /* 
     * Linux may call bus scan on adaptor registration (i.e. before
     * initialization. Initialize the driver with a default value if
     * this is happening.
     */
    if (FALSE == ixp400_i2c_init_complete)
    {
	IxI2cInitVars initI2cVarsSelected;

	initI2cVarsSelected.I2cSpeedSelect = IX_I2C_NORMAL_MODE;
	initI2cVarsSelected.I2cFlowSelect = IX_I2C_POLL_MODE;
	initI2cVarsSelected.MasterReadCBP = NULL;
	initI2cVarsSelected.MasterWriteCBP = NULL;
	initI2cVarsSelected.SlaveReadCBP = NULL;
	initI2cVarsSelected.SlaveWriteCBP = NULL;
	initI2cVarsSelected.GenCallCBP = NULL;
	initI2cVarsSelected.I2cGenCallResponseEnable = FALSE;
	initI2cVarsSelected.I2cSlaveAddrResponseEnable = FALSE;
	initI2cVarsSelected.SCLEnable = TRUE;
	initI2cVarsSelected.I2cHWAddr = 1; /* minimum slave address value */

	/* 
	 * We do not check for return value here but let bus_scan to
	 * return the error value shall initialization failed.
	 */
	_adap->init (&initI2cVarsSelected);

	/* 
	 * We do not set the ixp400_i2c_init_complete to TRUE here because we
	 * expect the user to perform initialization again for his/her preferred
	 * i2c setup value.
	 */
    }

    /* register the I2C adapter to the I2C module */
    status = i2c_add_adapter(i2c_adap);

    /* 
     * Bus scan should be done now, uninitialize the driver 
     */
    _adap->uninit();

    /* Check if adding bus is successful */
    if(0 != status)
    {
        return status;
    } /* end of adding bus unsuccessful */

    printk(KERN_INFO "I2C: Successfully added bus\n");

    return 0;
} /* end of ixp400i2cdev_add_bus */


/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_del_bus (struct i2c_adapter *i2c_adap)
 *
 * This API will de-register the IXP I2C adapter from the I2C module
 */
static int ixp400i2cdev_del_bus (struct i2c_adapter *i2c_adap)
{
    int res;

    /* Check if i2c_adap is NULL */
    if(NULL == i2c_adap)
    {
        printk(KERN_ERR "i2c_adap NULL pointer provided\n");
        return -2;
    } /* end of i2c_adap is NULL */

    /* de-register the I2C adapter from the I2C module */
    if ((res = i2c_del_adapter(i2c_adap)) < 0)
        return res;

    printk(KERN_INFO "I2C: Removing %s.\n", i2c_adap->name);

    return 0;
} /* end of ixp400i2cdev_del_bus */


/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_master_xfer (
    struct i2c_adapter *i2c_adap,
    struct i2c_msg msgs[],
    int num)
 *
 * This API will call the master_receive or master_transmit depending
 * on the access requested.
 */
static int ixp400i2cdev_master_xfer (
    struct i2c_adapter *i2c_adap,
    struct i2c_msg msgs[],
    int num)
{
    struct i2c_algo_ixp_data *adap = i2c_adap->algo_data;
    struct i2c_msg *pmsg=msgs;

    /* Check if i2c_adap is NULL */
    if(NULL == i2c_adap)
    {
        printk(KERN_ERR "i2c_adap NULL pointer provided\n");
        return -2;
    } /* end of i2c_adap is NULL */

    /* Check if msgs is NULL */
    if(NULL == msgs)
    {
        printk(KERN_ERR "msgs NULL pointer provided\n");
        return -2;
    } /* end of msgs is NULL */

    /* Check if it is a read or a write */
    if(I2C_M_RD == (pmsg->flags & I2C_M_RD)) /* master_read */
    {
        return adap->master_receive(pmsg->addr, pmsg->buf,
                                    pmsg->len, IX_I2C_NORMAL);
    } /* end of master_read (I2C_M_RD)*/
    else /* master_write */
    {
        return adap->master_transmit(pmsg->addr, pmsg->buf,
                                    pmsg->len, IX_I2C_NORMAL);
    } /* end of master_write */
} /* end of ixp400i2cdev_master_xfer */


/**
 * @ingroup IxI2cDrv
 * 
 * @fn ixp400i2cdev_algo_control(
    struct i2c_adapter *adapter,
    unsigned int cmd,
    unsigned long arg)
 *
 * This API will be called to support other of the capabilities of
 * the I2C algorith module.
 */
static int ixp400i2cdev_algo_control (
    struct i2c_adapter *i2c_adap,
    unsigned int cmd,
    unsigned long arg)
{
    struct i2c_algo_ixp_data *adap = i2c_adap->algo_data;
    IxI2cInitVars temp_initVarsSelected;
    IxI2cSlaveXferInfo temp_slave_xfer_info;
    IxI2cSlaveBufInfo temp_slave_buf_info;
    char *temp_buffer;
    UINT32 status;
    IxI2cMasterXferInfo temp_master_xfer_info;
    IxI2cStatsCounters temp_I2cStats;
    unsigned long ret; /* just to please the compiler */

    /* Check if i2c_adap is NULL */
    if(NULL == i2c_adap)
    {
        printk(KERN_ERR "i2c_adap NULL pointer provided\n");
        return -2;
    } /* end of i2c_adap is NULL */

    /* Function not allowed to be called if i2c not init except if it is
        i2c init, to replenish slave buffer, get stats, or to reset stats*/
    if((FALSE == ixp400_i2c_init_complete) &&
        (I2C_DRV_INIT != cmd) &&
        (I2C_DRV_SLAVE_OR_GEN_CALL_BUF_REPLENISH != cmd) &&
        (I2C_DRV_STATS_GET != cmd) &&
        (I2C_DRV_STATS_RESET != cmd))
    {
        printk(KERN_ERR "cmd = 0x%x\n", cmd);
        printk(KERN_ERR "I2C not initialized!!!\n");
        return -1;
    } /* end of FALSE == ixp400_i2c_init_complete... */

    /* Check if arg is zero (NULL) if it is to be used. Only I2C_DRV_UNINIT,
    I2C_DRV_BUS_SCAN, I2C_DRV_STATS_RESET, and I2C_DRV_SHOW do not use it. */
    if((0 == arg) && (I2C_DRV_BUS_SCAN != cmd) &&
        (I2C_DRV_STATS_RESET != cmd) && (I2C_DRV_SHOW != cmd) &&
        (I2C_SLAVE_ADDR_AND_GEN_CALL_CHECK != cmd) &&
        (I2C_DRV_UNINIT != cmd))
    {
        printk(KERN_ERR "NULL pointer provided\n");
        return -3;
    } /* end of 0 == arg ... */

    /* determine which command is being called */
    switch (cmd)
    {
        case I2C_DRV_INIT:
            /* copy the init config from user space */
            ret = copy_from_user(&temp_initVarsSelected, (size_t*)arg,
                            sizeof(IxI2cInitVars));

            /* Disallow callbacks to be used as kernel code should not call user code */
            if((IX_I2C_INTERRUPT_MODE ==
                temp_initVarsSelected.I2cFlowSelect))
            {
                if((NULL != temp_initVarsSelected.MasterReadCBP) ||
                   (NULL != temp_initVarsSelected.MasterWriteCBP) ||
                   (TRUE == temp_initVarsSelected.I2cSlaveAddrResponseEnable) ||
                   (TRUE == temp_initVarsSelected.I2cGenCallResponseEnable))
                {
                    printk(KERN_ERR "Interrupt callbacks are not supported\n");
                    return -1;
                } /* end of callbacks are used */
            } /* end of IX_I2C_INTERRUPT_MODE */

            /* call the API to init the IXP I2C unit */
            status = adap->init(&temp_initVarsSelected);
            if(status != 0)
            {
                return status;
            } /* end of status != 0 */

            /* set the flag to indicate I2C init complete */
            ixp400_i2c_init_complete = TRUE;
            break;

        case I2C_DRV_UNINIT:
            /* call the API to init the IXP I2C unit */
            status = adap->uninit();
            if(status != 0)
            {
                return status;
            } /* end of status != 0 */

            /* set the flag to indicate I2C init not complete */
            ixp400_i2c_init_complete = FALSE;
            break;
    
        case I2C_DRV_SET_I2C_SLAVE_ADDR:            
            /* copy the slave address from user space */
            /* call the API to set the IXP slave address */
            return adap->slave_addr_set(arg);
            break;

        case I2C_DRV_BUS_SCAN:
	    
            return adap->bus_scan();
            break;

        case I2C_SLAVE_ADDR_AND_GEN_CALL_CHECK:
            return adap->slave_and_gen_call_check();
            break;

        case I2C_DRV_SLAVE_OR_GEN_CALL_RECEIVE:
            /* copy slave receive info - bufSize */
            ret = copy_from_user(&temp_slave_xfer_info, (size_t*)arg,
                            sizeof(IxI2cSlaveXferInfo));

            /* allocate buffer according to buffer size defined by
                bufSize */
            temp_buffer = kmalloc(temp_slave_xfer_info.bufSize, GFP_KERNEL);

            /* Check if memore was allocated */
            if (NULL == temp_buffer)
            {
                return -ENOMEM;
            } /* end of no memory */

            /* call the API to transmit slave data in poll mode */
            status = adap->slave_receive(temp_buffer,
                                    temp_slave_xfer_info.bufSize,
                                    temp_slave_xfer_info.dataSizeXfer);

            /* copy data to user space - received data */
            ret = copy_to_user(temp_slave_xfer_info.bufP,
                        temp_buffer,
                        temp_slave_xfer_info.bufSize);

            /* copy slave receive info to user space (dataSizeXfer)*/
            ret = copy_to_user((size_t*)arg,
                        &temp_slave_xfer_info,
                        sizeof(IxI2cSlaveXferInfo));

            /* free the buffer allocated */
            kfree(temp_buffer);
            return status;
            break;

        case I2C_DRV_SLAVE_TRANSMIT:
            /* copy slave tranmit info - bufP, bufSize */
            ret = copy_from_user(&temp_slave_xfer_info,
                            (size_t*)arg,
                            sizeof(IxI2cSlaveXferInfo));

            /* allocate buffer according to buffer size defined by
                bufSize */
            temp_buffer = kmalloc(temp_slave_xfer_info.bufSize, GFP_KERNEL);

            /* Check if memore was allocated */
            if (NULL == temp_buffer)
            {
                return -ENOMEM;
            } /* end of no memory */

            /* copy data from user space */
            ret = copy_from_user(temp_buffer,
                            temp_slave_xfer_info.bufP,
                            temp_slave_xfer_info.bufSize);

            /* call the API to transmit slave data in poll mode */
            status = adap->slave_transmit(temp_buffer,
                                    temp_slave_xfer_info.bufSize,
                                    temp_slave_xfer_info.dataSizeXfer);

            /* copy data to user space - dataSizeXfer */
            ret = copy_to_user((size_t*)arg,
                        &temp_slave_xfer_info,
                        sizeof(IxI2cSlaveXferInfo));

            /* free the buffer allocated */
            kfree(temp_buffer);
            return status;
            break;

        case I2C_DRV_SLAVE_OR_GEN_CALL_BUF_REPLENISH:
            /* copy buffer info from user space */
            ret = copy_from_user(&temp_slave_buf_info,
                            (size_t*)arg,
                            sizeof(IxI2cSlaveBufInfo));

            /* call the API to provide slave with new buffer */
            adap->slave_buf_replenish(temp_slave_buf_info.bufP,
                                    temp_slave_buf_info.bufSize);
            break;

        case I2C_DRV_REPEATED_START_READ_TRANSFER:
            /* copy master receive info from user space */
            ret = copy_from_user(&temp_master_xfer_info,
                            (size_t*)arg,
                            sizeof(IxI2cMasterXferInfo));

            /* allocate buffer according to buffer size defined by
                bufSize */
            temp_buffer = kmalloc(temp_master_xfer_info.bufSize, GFP_KERNEL);

            /* Check if memore was allocated */
            if (NULL == temp_buffer)
            {
                return -ENOMEM;
            } /* end of no memory */

            /* call the API to receive data as master in
            repeated start mode */
            status = adap->master_receive(temp_master_xfer_info.slave_addr,
                                        temp_buffer,
                                        temp_master_xfer_info.bufSize,
                                        IX_I2C_REPEATED_START);

            /* copy data to user space */
            ret = copy_to_user(temp_master_xfer_info.bufP,
                        temp_buffer,
                        temp_master_xfer_info.bufSize);

            /* free the buffer allocated */
            kfree(temp_buffer);
            return status;
            break;

        case I2C_DRV_REPEATED_START_WRITE_TRANSFER:
            /* copy master transmit info from user space */
            ret = copy_from_user(&temp_master_xfer_info,
                            (size_t*)arg,
                            sizeof(IxI2cMasterXferInfo));

            /* allocate buffer according to buffer size defined by
                bufSize */
            temp_buffer = kmalloc(temp_master_xfer_info.bufSize, GFP_KERNEL);

            /* Check if memore was allocated */
            if (NULL == temp_buffer)
            {
                return -ENOMEM;
            } /* end of no memory */

            /* copy data from user space */
            ret = copy_from_user(temp_buffer,
                            temp_master_xfer_info.bufP,
                            temp_master_xfer_info.bufSize);

            /* call the API to transmit data as master in
            repeated start mode */
            status = adap->master_transmit(temp_master_xfer_info.slave_addr,
                                        temp_buffer,
                                        temp_master_xfer_info.bufSize,
                                        IX_I2C_REPEATED_START);

            /* free the buffer allocated */
            kfree(temp_buffer);
            return status;
            break;

        case I2C_DRV_STATS_GET:
            /* call the API to obtain the I2C stats */
            status = adap->stats_get(&temp_I2cStats);
            if(status != 0)
            {
                return status;
            } /* end of status != 0 */
            /* copy the stats to user space */
            ret = copy_to_user((size_t*)arg, &temp_I2cStats,
                        sizeof(IxI2cStatsCounters));
            break;

        case I2C_DRV_STATS_RESET:
            /* call the API to reset the I2C stats */
            adap->stats_reset();
            break;

        case I2C_DRV_SHOW:
            /* call the API to show the I2C stats and status */
            return adap->show();
            break;

        case I2C_DRV_DELAY_SELECT:
            /* call the API to change the delay type */
            adap->delay_select(arg);
            break;

        default:
            printk(KERN_ERR "Unsupported command\n");
            break;
    } /* end of switch(cmd) */
    
    return 0;
} /* end of ixp400i2cdev_algo_control */
