/******************************************************************************
*       @file  IxDspCodelet3WCall.c
*
* Contents: This file contains functions for 3 way call.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the user defined include files required.
 */

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define MAX_NAME_ARRAY_SIZE  20
#define MAX_3W_CALL_STATE  3

#define MIXER_NUM 1

#define IP_CHAN_1  1
#define IP_CHAN_2  2
#define IP_CHAN_3  3
#define TDM_CHAN_1  1

#define CALL_PARTY_1  0
#define CALL_PARTY_2  1
#define CALL_PARTY_3  2

/* 3-Way Call state index */
typedef enum
{
    IX_DSP_CODELET_3WST_IDLE = 0,
    IX_DSP_CODELET_3WST_2WCALL,
    IX_DSP_CODELET_3WST_3WCALL
} IxDspCodelet3WayCallDemoState;

/* 3-Way Call state */
typedef struct
{
    char stateName[MAX_NAME_ARRAY_SIZE];
} IxDspCodelet3WCallState;

static IxDspCodelet3WCallState state3WCall[MAX_3W_CALL_STATE] = {
    {"IDLE STATE"},
    {"2W-CALL STATE"},
    {"3W-CALL STATE"}
};
/* 3-way demo command defines */
typedef enum
{
    CMD_NULL = 0,
    CMD_SETUP_CALL,
    CMD_SWITCH_CALL,
    CMD_MAKE_3WAY_CALL,
    CMD_BACKTO_2WAY_CALL,
    CMD_EXIT,
} IxDspCodelet3WayDemoCmdType;

/* macro converts user command to message type */
#define IX_DSP_CODELET_3WMSG_BASE   IX_DSP_CODELET_MSG_END_OF_LIST
#define IX_DSP_CODELET_3WMSG(cmd)   (IX_DSP_CODELET_3WMSG_BASE + cmd)

/* static variables */
static int msgBuf[XMSG_MAX_WSIZE];

#define RFC_2833_PAYLOAD_TYPE_101  101

/* default call parameters */
static IxDspCodeletCallParms defCallParms =
{
    0xffff,                     /* decAutoSwitch */
    XCODER_TYPE_G722_64,    /* Decoder type */
    XCODER_TYPE_G722_64,    /* Encoder type */
    1,                          /* number frames per packet */
    XPARM_OFF,                   /* vad */
    XPARM_ON,                   /* RFC 2833 */
    RFC_2833_PAYLOAD_TYPE_101,  /* RFC 2833 payload type */
    XPARM_ON,                   /* tone Clamping */
    XPARM_TD_RPT_TONE_ON        /* tone event report */
};


/********************************
 * Start 3-Way Call demo
 ********************************/
void
ixDspCodelet3WCallSelect (int callId)
{
    /* call Id is used to identify the instance
       of 3-way call that is associated with a
       mixer instance. Currently callId should be
       set to 1 because there is only one mixer
       instance
     */

    int usrCmd =0;
    UINT32 trans =0;
    XMsgHdr_t msg;

    /* get 3-Way Call demo selection */
    do
    {
        usrCmd = ixDspCodeletGetNum ("    -----------------------\n"
                                     "        3-Way Call Demo    \n"
                                     "    -----------------------\n"
                                     "    0) Display menu and state\n"
                                     "    1) Setup Call\n"
                                     "    2) Switch 2-Way Call\n"
                                     "    3) Make 3-Way Call\n"
                                     "    4) Back to 2-Way Call\n"
                                     "    5) Exit\n"
                                     "    Please enter command : \n");

        if (usrCmd > CMD_EXIT)
        {
            continue;
        }

        trans = IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC, 
                                                                callId);
        XMSG_MAKE_HEAD (&msg, trans, 0, 0, sizeof (XMsgHdr_t),
                        IX_DSP_CODELET_3WMSG (usrCmd), 0) xMsgWrite (&msg);

    }
    while (usrCmd != CMD_EXIT);
}

/********************************
 * 3-Way Call State Machine
 ********************************/
int
ixDspCodelet3WCallSm (XMsgRef_t pMsg)
{
    static int state = IX_DSP_CODELET_3WST_IDLE;
    static int oldIp = IP_CHAN_1;
    static int curIp = IP_CHAN_1;
    int callId = MIXER_NUM;
    UINT32 trans;
    IxDspCodeletMsgCreate3wCall *pMesg = NULL;
    char *stateName = state3WCall[state].stateName;
    IxDspCodeletMsgLinkBreak *linkBreakMsg = NULL;
    IxDspCodeletSetCallParms *setParamsMsg;
    IxDspCodeletMsgSetupCallwParms *setCallWParamsMsg;

    /* display current state if a null command received */
    if (pMsg->type == IX_DSP_CODELET_3WMSG (CMD_NULL))
    {
        printf ("*** %s ***\n", stateName);
        return 0;
    }

    callId = IX_DSP_CODELET_TRANS_GET_CHAN (pMsg);

    if (callId != MIXER_NUM )
    {
        /* call Id must be 1 because we have only one mixer
           and can support only one 3-way call */

        printf ("invalid call ID (%d)\n", callId);
        return 0;
    }

    switch (state)
    {
    case IX_DSP_CODELET_3WST_IDLE:

        if (pMsg->type == IX_DSP_CODELET_MSG_SETUP_ACK)
        {
            /* setup call confirmed */
            printf ("complete\n");
            state = IX_DSP_CODELET_3WST_2WCALL;
            printf ("*** 2W-CALL STATE ***\n");
        }
        else
        {    

            if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_SETUP_CALL)))
            {
                printf ("[%s]: Setup call - ", stateName);

                /* setup a call from party 1 to ip1 */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
                /*setup call with params*/
                setCallWParamsMsg = (IxDspCodeletMsgSetupCallwParms *) msgBuf;
                setCallWParamsMsg->parms = defCallParms;                 
                setCallWParamsMsg->channelIP = IP_CHAN_1;
                setCallWParamsMsg->channelTDM = TDM_CHAN_1;
                    
                trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_3WC, 
                                                                    callId);
        
                IX_DSP_CODELET_MAKE_MSGHDR_SETUP_CALLWPARMS(setCallWParamsMsg,
                                                                    trans);
                if(xMsgSend(setCallWParamsMsg) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                                      
                curIp = 1;
                

                /*Setup the parameters for IP3*/
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);                
                setParamsMsg = (IxDspCodeletSetCallParms *) msgBuf;
                setParamsMsg->parms = defCallParms;
                setParamsMsg->channelIP = IP_CHAN_3;  
                                           
                IX_DSP_CODELET_MAKE_MSGHDR_SET_CALL_PARMS(setParamsMsg, 
                                                                trans);
                if(xMsgSend(setParamsMsg) != XSUCC)
                {
                    printf("ERROR - Can't set call params for IP3.\n");
                    return;
                }
       
                /*setup ip3*/
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);

                IX_DSP_CODELET_MAKE_MSG_START_IP (msgBuf, trans,
                                                  IP_CHAN_3);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                     
      
            }
            else if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_EXIT)))
            {
                printf ("Exit 3-Way Call Demo\n");
            }
            else
            {
                printf ("[%s]: Please select 1) Setup channel first\n",
                        stateName);
            }
        }

        break;

        case IX_DSP_CODELET_3WST_2WCALL:

        switch (pMsg->type)
        {
        case IX_DSP_CODELET_MSG_LINK_ACK:
            /* switch party confirmed. state unchanged */
            printf ("complete\n");
            break;

        case IX_DSP_CODELET_MSG_3W_ACK:
            /* make 3-way call confirmed */
            printf ("complete\n");
            state = IX_DSP_CODELET_3WST_3WCALL;
            printf ("*** 3W-CALL STATE ***\n");
            break;

        case IX_DSP_CODELET_MSG_STOP_ACK:
            printf ("complete\n");
            state = IX_DSP_CODELET_3WST_IDLE;
            printf ("Exit 3-Way Call Demo\n");
            break;

        default:

            if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_SWITCH_CALL)))
            {
                /* switch 2-way call between party2 & party3 */
                oldIp = curIp;
                curIp = (curIp == IP_CHAN_1 ? IP_CHAN_3 : IP_CHAN_1);
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
      
                IX_DSP_CODELET_MAKE_MSG_SWITCH_CALL (msgBuf, trans, 
                                        TDM_CHAN_1, oldIp, curIp);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }
                printf ("[%s]: Switch to party %d - ", stateName,
                        curIp == IP_CHAN_1 ? IP_CHAN_2 : IP_CHAN_3);
            }
            else if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_MAKE_3WAY_CALL)))
            {
                printf ("[%s]: Create 3-Way call - ", stateName);
                /* create 3-way call by summing TDM1, IP1 and IP3 */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
                IX_DSP_CODELET_MAKE_MSGHDR_CREATE_3WCALL (msgBuf, trans) pMesg
                                      = (IxDspCodeletMsgCreate3wCall *) msgBuf;

                pMesg->parties[CALL_PARTY_1].type = IX_DSP_CODELET_TERM_TDM;
                pMesg->parties[CALL_PARTY_1].channel = TDM_CHAN_1;

                pMesg->parties[CALL_PARTY_2].type = IX_DSP_CODELET_TERM_IP;
                pMesg->parties[CALL_PARTY_3].type = IX_DSP_CODELET_TERM_IP;
                /*Change the party 2 and 3 depending on 
                 * which is the current ip*/
                if (curIp == IP_CHAN_1)
                {
                    pMesg->parties[CALL_PARTY_2].channel = IP_CHAN_1;
                    pMesg->parties[CALL_PARTY_3].channel = IP_CHAN_3;
                }
                else
                {
                    pMesg->parties[CALL_PARTY_2].channel = IP_CHAN_3;
                    pMesg->parties[CALL_PARTY_3].channel = IP_CHAN_1;
                }               
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                 
            }
            else if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_EXIT)))
            {
                printf ("[%s] Exit 2-Way call - ", stateName);

                /* break the link between TDM1 and the currently used IP */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
                                               
                linkBreakMsg = (IxDspCodeletMsgLinkBreak *) msgBuf;
                
                linkBreakMsg->term1.type = IX_DSP_CODELET_TERM_TDM;
                linkBreakMsg->term1.channel = TDM_CHAN_1;
                linkBreakMsg->term2.type = IX_DSP_CODELET_TERM_IP;
                linkBreakMsg->term2.channel = curIp;
                
                IX_DSP_CODELET_MAKE_MSGHDR_LINK_BREAK (msgBuf,
                                                       trans);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                     


                /* stop current ip that party1 is using */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);

                IX_DSP_CODELET_MAKE_MSG_STOP_IP (msgBuf, trans,
                                                 curIp);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }


                /*Stop the other ip party that is not currently being used */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                                                    callId);
                IX_DSP_CODELET_MAKE_MSG_STOP_IP (msgBuf, trans,
                                                    (curIp ==
                                                    IP_CHAN_1 ? IP_CHAN_3 :
                                                    IP_CHAN_1));
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                                             

            }
            else
            {
                printf ("[%s]: Available selections - 2), 3) and 5)\n",
                        stateName);
            }
            break;
        }
        break;

    case IX_DSP_CODELET_3WST_3WCALL:

        switch (pMsg->type)
        {
        case IX_DSP_CODELET_MSG_3W_ACK:
            printf ("complete\n");
            state = IX_DSP_CODELET_3WST_2WCALL;
            printf ("*** 2W-CALL STATE ***\n");
            break;

        case IX_DSP_CODELET_MSG_STOP_ACK:
            printf ("complete\n");
            state = IX_DSP_CODELET_3WST_IDLE;
            printf ("Exit 3-Way Call Demo\n");
            break;

        default:
         
            if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_BACKTO_2WAY_CALL)))
            {
                printf ("[%s]: Back to 2-Way call - ", stateName);
                /* go back to 2way call */
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
                IX_DSP_CODELET_MAKE_MSGHDR_BACKTO_2WCALL (msgBuf,
                                                          trans);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                     
            }
            else if (pMsg->type == (IX_DSP_CODELET_3WMSG (CMD_EXIT)))
            {
                printf ("[%s]: Teardown 3-Way call - ", stateName);
                trans =
                    IX_DSP_CODELET_MAKE_TRANS (IX_DSP_CODELET_CATEGORY_3WC,
                                               callId);
                IX_DSP_CODELET_MAKE_MSGHDR_TEARDOWN_3WCALL (msgBuf,
                                                            trans);
                if(xMsgSend((XMsgRef_t) msgBuf) != XSUCC)
                {
                    printf("ERROR - Message send fails.\n");
                    return 0;
                }                     
            }
            else
            {
                printf ("[%s]: Available selections - 4) and 5)\n", stateName);
            }
            break;
        }
        break;

    default:
        printf ("Unknown state (%d)\n", state);
        break;

    }

    /* STOP_ACK & 3WMSG's are already handled, skip them */
    if ((pMsg->type != IX_DSP_CODELET_MSG_ACK) &&
        (pMsg->type != IX_DSP_CODELET_MSG_LINK_ACK) &&
        (pMsg->type != IX_DSP_CODELET_MSG_SETUP_ACK) &&
        (pMsg->type != IX_DSP_CODELET_MSG_3W_ACK) &&
        (pMsg->type != IX_DSP_CODELET_MSG_STOP_ACK) &&
        (pMsg->type < IX_DSP_CODELET_3WMSG_BASE))
    {
        ixDspCodeletPrtMsg (pMsg);
    }

    return 0;
}
