/******************************************************************************
*       @file    IxDspCodeletALSA.c
*
* Contents: This file contains the ALSA functions.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

#include <alsa/asoundlib.h>
#include "IxDspCodelet.h" 

#define ONE_CHANNEL 1

/* We need to set the read/write threshold. This is the point at which data
   will be read and written to the buffer. Too large a value will introduce
   excessive latency, too small will decrease reliability. 400 frames 
   corresponds to 400/RATE = 25msec @ 16kHz in both directions. So this will 
   introduce 50msec one-way delay while it should provide adequate 
   reliability. */
#define PLAYBACK_FRAMES_THRESHOLD 400 /* Playback threshold (frames)*/
#define CAPTURE_FRAMES_THRESHOLD  400 /* Capture threshold (frames)*/
#define PCM_STREAM_RATE           16000 /* pcm stream rate in hz */
#define BLOCKING                  0 /* used for snd_pcm_open to specify open
                                       in blocking mode */
#define TRY_EXACT                 0 /* used by set_rate_near specifying that
                                       we are trying to exactly specify the 
                                       rate */
#define SLEEP_10MS                10
char *pdevice = "plughw:1,0"; /* Audio playback device  */
char *cdevice = "plughw:1,0"; /* Audio capture device */

snd_pcm_t *phandle=NULL, *chandle=NULL; /* handle for playback and capture 
                                           streams */


static unsigned int rate = PCM_STREAM_RATE; /* stream rate 16kHz */



static snd_output_t *output = NULL;

static unsigned int chn;
static int err=0;

/* ALSA Init function */
int ixALSAInit(void)
{

    unsigned int rrate=0; /* Real sampling rate */
    snd_pcm_uframes_t p_min_val=0,
                      c_min_val=0;


    
    /* PCM hardware configuration containers, these contain the configuration
           of the hardware */ 
    snd_pcm_hw_params_t *p_HwParams, *c_HwParams; 

    /* PCM software configuration containers, these contain the configuration
           of the software */
    snd_pcm_sw_params_t *p_SwParams, *c_SwParams;

    /* Open in playback mode. */
    err = snd_pcm_open(&phandle, pdevice, SND_PCM_STREAM_PLAYBACK, BLOCKING);
    if (err  < 0)
    { 
        printf("Playback open error: %s\n", snd_strerror(err)); 
        return IX_FAIL; 
    }

    /* Open in capture mode. */
    err = snd_pcm_open(&chandle, cdevice, SND_PCM_STREAM_CAPTURE, BLOCKING);
    if (err  < 0)
    {
        printf("Capture open error: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    /* Allocate a hardware parameters object for each stream */
    snd_pcm_hw_params_alloca(&p_HwParams);
    snd_pcm_hw_params_alloca(&c_HwParams);
    /* Fill them in with default values. */
    snd_pcm_hw_params_any(phandle, p_HwParams);
    snd_pcm_hw_params_any(chandle, c_HwParams);

    /* Now set the SW parameters */
    snd_pcm_sw_params_alloca(&p_SwParams);
    snd_pcm_sw_params_alloca(&c_SwParams);


    /* Set access mode */
    err = snd_pcm_hw_params_set_access(phandle, p_HwParams,
                                       SND_PCM_ACCESS_RW_INTERLEAVED);
    if (err < 0)
    {
        printf("Access type not available for playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_hw_params_set_access(chandle, c_HwParams,
                                       SND_PCM_ACCESS_RW_INTERLEAVED);
    if (err < 0)
    {
        printf("Access type not available for capture: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    /* Set sample format */

    
    err = snd_pcm_hw_params_set_format(phandle, p_HwParams,
                                       SND_PCM_FORMAT_S16_BE);
    if (err < 0)
    {

        printf("Sample format not available for playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_hw_params_set_format(chandle, c_HwParams, 
                                       SND_PCM_FORMAT_S16_BE);
    if (err < 0)
    {
        printf("Sample format not available for capture: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    /* Set Number of channels to 1 for each stream */ 
    err = snd_pcm_hw_params_set_channels(phandle, p_HwParams, ONE_CHANNEL); 
    if (err < 0)
    {
        printf("Channel count 1 not available for playack: %s\n",     
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_hw_params_set_channels(chandle, c_HwParams, ONE_CHANNEL);
    if (err < 0)
    {
        printf("Channel count 1 not available for capture: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    /* Set sampling rate to 16000 Hz */
    rrate = rate;

    err = snd_pcm_hw_params_set_rate_near(phandle,p_HwParams,&rrate,TRY_EXACT);
    if (err < 0)
    {
        printf("Rate %uHz not available for playack: %s\n", rate,
                snd_strerror(err));
        return IX_FAIL;
    }

    if (rrate != rate) 
    {
               printf("Playback rate doesn't match (requested %uHz,"
                      " got %uHz)\n", rate, rrate);
               return IX_FAIL;
    }

    err = snd_pcm_hw_params_set_rate_near(chandle,c_HwParams,&rrate,TRY_EXACT);
    if (err < 0)
    {
        printf("Rate %uHz not available for capture: %s\n", rate,
                snd_strerror(err));
        return IX_FAIL;
    }

    if (rrate != rate) 
    {
               printf("Capture rate doesn't match (requested %uHz,"
                      " get %uHz)\n", rate, rrate);
               return IX_FAIL;
    }


    /* Write the parameters to the driver */
    err = snd_pcm_hw_params(phandle, p_HwParams);
    if (err < 0) 
    {
        printf("Unable to set hw params for Playback: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_hw_params(chandle, c_HwParams);
    if (err < 0) 
    {
        printf("Unable to set hw params for Capture: %s\n", snd_strerror(err));
        return IX_FAIL;
    }



    /* Associate SW params with handles */
    err = snd_pcm_sw_params_current(phandle, p_SwParams);
    if (err < 0)
    {
        printf("Unable to determine current sw params for Playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_sw_params_current(chandle, c_SwParams);
    if (err < 0)
    {
        printf("Unable to determine current sw params for Capture: %s\n", 
                snd_strerror(err));
        return IX_FAIL;
    }

    /* Set start threshold for playback & capture*/
    err = snd_pcm_sw_params_set_start_threshold(phandle,
                                                p_SwParams,
                                                PLAYBACK_FRAMES_THRESHOLD);
    if (err < 0)
    {
        printf("Unable to set start threshold mode for Playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_sw_params_set_start_threshold(chandle,
                                                c_SwParams,
                                                CAPTURE_FRAMES_THRESHOLD);
    if (err < 0) 
    {
        printf("Unable to set start threshold mode for Capture: %s\n",
               snd_strerror(err));
        return IX_FAIL;
    }

    /* Get period size to use for available min */
    err = snd_pcm_hw_params_get_period_size(p_HwParams, &p_min_val, NULL);
    if (err < 0)
    {
        printf("Unable to get period size for Playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_hw_params_get_period_size(c_HwParams, &c_min_val, NULL);
    if (err < 0)
    {
        printf("Unable to get period size for Capture: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }
    

    
    /* Set available min - for when ALSA notifies us that it is ready */
    err = snd_pcm_sw_params_set_avail_min(phandle, p_SwParams, p_min_val);
    if (err < 0)
    {
        printf("Unable to set available min for Playback: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_sw_params_set_avail_min(chandle, c_SwParams, c_min_val);
    if (err < 0)
    {
        printf("Unable to set available min for Capture: %s\n",
                snd_strerror(err));
        return IX_FAIL;
    }









    /* Write the SW params to the driver */
    err = snd_pcm_sw_params(phandle, p_SwParams);
    if (err < 0) 
    {
        printf("Unable to set sw params for Playback: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_sw_params(chandle, c_SwParams);
    if (err < 0) 
    {
        printf("Unable to set sw params for Capture: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    /* Prepare PCM for use */
    err = snd_pcm_prepare(chandle);
    if (err < 0)
    {
        printf("Prepare error for capture: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_prepare(phandle);
    if (err < 0)
    {
        printf("Prepare error for playback: %s\n", snd_strerror(err));
        return IX_FAIL;
    }


   return IX_SUCCESS;

}

/* ALSA Close function */
int ixALSAClose(void)
{

    xDspSysClose();
    /* Remove PCM hardware configuration and free associated resources */
    snd_pcm_hw_free(phandle);
    snd_pcm_hw_free(chandle);
  
    err = snd_pcm_close(phandle);
    if (err <0)
    {
        printf("Playback close error: %s\n", snd_strerror(err));
        return IX_FAIL;
    }

    err = snd_pcm_close(chandle);
    if (err<0)
    {
        printf("Capture close error: %s\n", snd_strerror(err));
        return IX_FAIL;
    }


    return IX_SUCCESS;
}

