/******************************************************************************
*       @file   IxDspCodeletCidBc.c
*
* Contents: This file contains funtions for Bellcore Caller ID.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the user defined include files required.
 */
#include "IxDspCodelet.h"

/*
 * defines and macros used in this file.
 */

/* Shift Constants */
#define SHIFT_7  7

/* constant in caller data string defined by NTT */
#define IX_DSP_CID_BELL_SDMF    0x04
#define IX_DSP_CID_BELL_MDMF    0x80
#define IX_DSP_CID_BELL_TIME    0x01
#define IX_DSP_CID_BELL_NUMBER  0x02
#define IX_DSP_CID_BELL_NAME    0x07
#define IX_DSP_CID_BELL_MAX_NO  20
#define IX_DSP_CID_BELL_TIME0   100  /* 1st ring duration (1 sec) */
#define IX_DSP_CID_BELL_TIME1   100  /* wait time after 1st ring (1 sec) */
#define IX_DSP_CID_BELL_TIME2   20   /* wait time before 2nd ring (200ms) */
#define IX_DSP_CID_BELL_TIME3   2000 /* 2nd ring timeout (20 sec) */

/* CRC generator initial value */
#define IX_DSP_CID_CRC_INIT     0x0

/* resolution of software timer */
#define IX_DSP_CID_TIMER_TICK   100 /* 0.1 second */

/* return value from caller id state machine */
#define IX_DSP_CID_ERROR       -1
#define IX_DSP_CID_UNKNOWN_MSG  1

/* General Constants */
#define INCREMENT_BY_2  2
#define INCREMENT_BY_3  3
#define CODELET_CID_PARITY_1  1
#define CODELET_CID_PARITY_2  2
 
/* states of Bellcore caller ID state machine */
typedef enum
{
    CID_IDLE = 0,          /* idle */
    CID_FIRST_RING,        /* in first ring, waiting to completion */
    CID_LOOP_CONN_DELAY,   /* delay after first ring */
    CID_SEND_FSK,          /* sending FSK modem data, waiting for completion */
    CID_LOOP_DISCONN_DELAY,/* FSK complete, delay before normal ring */
    CID_NORMAL_RING        /* normal ring, waiting for answer */
} IxDspCodeletStateCID;

/* channel data object */
typedef struct
{
    int         state;
    char        cidData[IX_DSP_CID_BELL_MAX_NO];
} IxDspCodeletCidObj, *IxDspCodeletCidHandle;


static IxDspCodeletCidObj cidObjBc[IX_DSP_CODELET_MAX_CHL];
static char *IxDspCodeletCidTime = "08220459";
static char *IxDspCodeletCidName = "Intel";
int IxDspCodeletCidParity = 2;

/******************************************
        parity check function
*******************************************/
unsigned char ixDspCodeletParity(unsigned char data)
{
    unsigned char temp;
    unsigned char parity = 1;

    if (IxDspCodeletCidParity > 1)
    {
        /* no parity */
        return data;
    }

    /* odd or even parity */
    parity = IxDspCodeletCidParity;

    temp = data &= 0x7f;
    while (data)
    {
        parity ^= data & 1;
        data >>= 1;
    }

    return temp | (parity << SHIFT_7);
}


/************************************************
        create modem data from the caller ID string
        according to the data format defined by Bellcore
*************************************************/
int ixDspCodeletCreateCidDataBc(char *pPhoneNo, unsigned char *pData)
{
    unsigned char crc = IX_DSP_CID_CRC_INIT;
    int n;
    int len = 0;
    unsigned char *pMsgLen;
    unsigned char *pParmLen;
    unsigned char *pCursor;
    unsigned char *IxDspCodeletCidStr;

    pCursor = pData;

    RETURN_IF_NULL_POINTER((pCursor), 0)

    /* create caller ID data header */
    *pCursor++ = IX_DSP_CID_BELL_MDMF;
    pMsgLen = pCursor++;

    /* add in time string */
    *pCursor++ = IX_DSP_CID_BELL_TIME;

    pParmLen = pCursor++;
    RETURN_IF_NULL_POINTER((pParmLen), 0)
    *pParmLen = 0;

    len += INCREMENT_BY_2;
    IxDspCodeletCidStr = IxDspCodeletCidTime;

    RETURN_IF_NULL_POINTER((IxDspCodeletCidStr), 0)
    /* copy the time string */
    while(*IxDspCodeletCidStr)
    {
        *pCursor++ = ixDspCodeletParity(*IxDspCodeletCidStr++);
        (*pParmLen)++;
        len++;
    }

    /* add in number string */
    *pCursor++ = IX_DSP_CID_BELL_NUMBER;
    pParmLen = pCursor++;
    *pParmLen = 0;
    len += INCREMENT_BY_2;
    /* copy the phone number string */
    while(*pPhoneNo && *pParmLen < IX_DSP_CID_BELL_MAX_NO)
    {
        *pCursor++ = ixDspCodeletParity(*pPhoneNo++);
        (*pParmLen)++;
        len++;
    }

    /* add in name string */
    *pCursor++ = IX_DSP_CID_BELL_NAME;
    pParmLen = pCursor++;
    *pParmLen = 0;
    len += INCREMENT_BY_2;
    IxDspCodeletCidStr = IxDspCodeletCidName;
    /* copy the time string */
    while(*IxDspCodeletCidStr)
    {
        *pCursor++ = ixDspCodeletParity(*IxDspCodeletCidStr++);
        (*pParmLen)++;
        len++;
    }

    RETURN_IF_NULL_POINTER((pMsgLen), 0)

    /* fill the message length */
    *pMsgLen = (unsigned char)(len);

    /* set cursor to the begin */
    pCursor = pData;

    n = len + INCREMENT_BY_2;
    /* calculate from the beginning */
    while(n) {
        crc += *pCursor++;
        n--;
    }
    *pCursor = ~crc + 1;

    return (len + INCREMENT_BY_3);
}

void ixDspCodeletCidInitBc(int chl, char *numStr)
{
    UINT32 trans;
    IxDspCodeletCidHandle handle;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CID, chl);
    handle = cidObjBc + chl - 1;

    /* update the state */
    handle->state = CID_FIRST_RING;

    strncpy(handle->cidData, numStr, IX_DSP_CID_BELL_MAX_NO);


    /* set timer to 1st ring duration */
    IX_DSP_CODELET_SET_TIMER(trans, chl, IX_DSP_CID_BELL_TIME0)

}

void ixDspCodeletCidInitPrc(int chl, char *numStr)
{
    IxDspCodeletCidParity = CODELET_CID_PARITY_1;     /* odd parity */
    ixDspCodeletCidInitBc(chl, numStr);
}

void ixDspCodeletCidInitUs(int chl, char *numStr)
{
    IxDspCodeletCidParity = CODELET_CID_PARITY_2;     /* no parity */
    ixDspCodeletCidInitBc(chl, numStr);
}

/************************************************
        Caller id state machine.
        Implement Bellcore caller id protocol,
        State machine is started by DSP's XMSG_START
        message, then driven by timer and slic events
        and XMSG_TG_PLAY_CMPLT message.
*************************************************/
int ixDspCodeletCallerIdSmBc(XMsgRef_t pMsg)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    int len;
    int rc;
    UINT32 trans;
    IxDspCodeletCidHandle handle;
    unsigned char *pData;
    XMsgTGPlayFSK_t msg;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);
    
    if((chl<=0)||(chl>IX_DSP_CODELET_MAX_CHL))
    {
      pMsg->type = XMSG_ERROR;
      printf("Invalid channel number, Not supported");
      ixDspCodeletPrtMsg(pMsg);
      error = IX_DSP_CID_UNKNOWN_MSG;
      return error;
    }
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CID, chl);
    handle = cidObjBc + chl - 1;

    switch(handle->state)
    {
    case CID_FIRST_RING:        /* first ring, wait to transmit */
        /* check if first ring is complete */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {

        /* set timer to wait after 1st ring */
            IX_DSP_CODELET_SET_TIMER(trans, chl, IX_DSP_CID_BELL_TIME1)

            /* update the state */
            handle->state = CID_LOOP_CONN_DELAY;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_LOOP_CONN_DELAY:   /* delay, wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* buid DSP FSK play message */
            XMSG_FIELD_TG_PLAY_FSK(&msg, pData);
            len = ixDspCodeletCreateCidDataBc(handle->cidData, pData);
            assert(len > 0);

            XMSG_MAKE_TG_PLAY_FSK(&msg, trans, chl, len);

            /* send FSK play message to DSP */
            rc = xMsgSend(&msg);
            if(rc!=XSUCC)
            {
                printf("ERROR - xMsgSend() fails\n");
            }

            /* update the state */
            handle->state = CID_SEND_FSK;

        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_SEND_FSK:  /* start modem message sent, wait for DSP response */
        /* check if DSP replied */
        if(pMsg->type == XMSG_TG_PLAY_CMPLT)
        {
            /* set timer to wait before 2nd ring */
            IX_DSP_CODELET_SET_TIMER(trans, chl, IX_DSP_CID_BELL_TIME2)
            handle->state = CID_LOOP_DISCONN_DELAY;
        }
        else
        {
            /* no response from DSP, abortion */
            handle->state = CID_IDLE;
            error = IX_DSP_CID_ERROR;
            printf("ERROR - DSP no response\n");
        }
        break;

    case CID_LOOP_DISCONN_DELAY:        /* delay, wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* set timer to 2nd ring timeout */
            IX_DSP_CODELET_SET_TIMER(trans, chl, IX_DSP_CID_BELL_TIME3)

            /* update the state */
            handle->state = CID_NORMAL_RING;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_NORMAL_RING:               /* normal ring, wait for answer */
        /* check if time out or answered */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT &&
           (evtCode == XEVT_NET_TIMER || evtCode == XEVT_NET_HOOK_STATE))
        {

            /* clean timer */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

            /* update the state */
            handle->state = CID_IDLE;

        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;
    default:
    ;/* No default */
    }

    if(error == IX_DSP_CID_UNKNOWN_MSG)
    {
        ixDspCodeletPrtMsg(pMsg);
    }

    return error;
}

