/******************************************************************************
*       @file  IxDspCodeletCidJp.c 
*
* Contents: This file contains functions for Caller ID for Japan.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/


/*
 * Put the user defined include files required.
 */
#include "IxDspCodelet.h"

/*
 * defines and macros used in this file.
 */

/* constant in caller data string defined by NTT */
#define IX_DSP_CID_NTT_DLE      0x10
#define IX_DSP_CID_NTT_SOH      0x01
#define IX_DSP_CID_NTT_HEADER   0x07
#define IX_DSP_CID_NTT_STX      0x02
#define IX_DSP_CID_NTT_SVC_TP   0x40
#define IX_DSP_CID_NTT_PARM_TP  0x02
#define IX_DSP_CID_NTT_ETX      0x03
#define IX_DSP_CID_NTT_MAX_NO   20

/* CRC generator polynommial and initial value */
#define IX_DSP_CID_GEN_POLY     0x8408
#define IX_DSP_CID_CRC_INIT     0x0

/* resolution of software timer */
#define IX_DSP_CID_TIMER_TICK   100 /* 0.1 second */

/* return value from caller id state machine */
#define IX_DSP_CID_ERROR       -1
#define IX_DSP_CID_UNKNOWN_MSG  1

/* Cid Data Buffer Max Size */
#define CID_DATA_BUFFER_SIZE  20

/* Shift Constants */
#define SHIFT_7  7
#define SHIFT_8  8

/* General Constants */
#define MAX_CRC_ITERATIONS  8
#define TONE_IIT  79
#define TONE_CAT  250
#define OFFSET_2  2
#define CID_HEADER_LEN  9
#define TOTAL_FSK_FIELDS  13

void play_iitorcat(int value);
/* states of Japan caller ID state machine */
typedef enum
{
    CID_IDLE = 0,          /* idle */
    CID_POLARITY_INV,      /* polarity inversed */
    CID_SHORT_RING,        /* in short ring, waiting for connection */
    CID_LOOP_CONN_DELAY,   /* delay after connection */
    CID_SEND_FSK,       /* sending FSK modem data, waiting for disconnection */
    CID_LOOP_DISCONN,      /* disconnected, delay before normal ring */
    CID_LOOP_DISCONN_DELAY,/* delay after disconnection */
    CID_NORMAL_RING        /* normal ring, waiting for answer */
} IxDspCodeletStateCIDJP;

/* channel data object */
typedef struct
{
    int         state;
    char        cidData[CID_DATA_BUFFER_SIZE];
} IxDspCodeletCidObj, *IxDspCodeletCidHandle;


static IxDspCodeletCidObj cidObjJp[IX_DSP_CODELET_MAX_CHL];
static IxDspCodeletCidObj cidcwObjJp[IX_DSP_CODELET_MAX_CHL];


/* FOR CIDCW */
/* states of Japan caller CIDCW state machine */
typedef enum
{
    CIDCW_IDLE = 0,    /* idle */
    CIDCW_SEND_IIT,    /* Sending IIT */
    CIDCW_SEND_CAT,    /* Sending CAT */
    CIDCW_SEND_FSK,    /* sending FSK modem data, waiting for disconnection */
    CIDCW_NXT,
    CIDCW_STOP
} IxDspCodeletStateCIDCWJP;


/******************************************
        parity check function
*******************************************/
unsigned char ixDspCodeletParityEven(unsigned char data)
{
    unsigned char temp;
    unsigned char parity = 0;

    temp = data &= 0x7f;
    while (data)
    {
        parity ^= data & 1;
        data >>= 1;
    }

    return temp | (parity << SHIFT_7);
}


/******************************************
        CRC check function
*******************************************/
unsigned int ixDspCodeletAdd2CRC16(unsigned char  newData,
                                   unsigned int   oldCRC,
                                   unsigned int   genPoly)
{
    int  i = MAX_CRC_ITERATIONS;

    oldCRC ^= newData;

    for(i = MAX_CRC_ITERATIONS; i; i--)
    {
        if(oldCRC & 1)
        {
            oldCRC = (oldCRC >>= 1) ^ genPoly;
        }
        else
        {
            oldCRC >>= 1;
        }
    }

    return oldCRC;
}

/************************************************
        create modem data from the caller ID string
        according to the data format defined by NTT
*************************************************/
int ixDspCodeletCreateCidDataJP(char *pPhoneNo, unsigned char *pData)
{
    int crc = IX_DSP_CID_CRC_INIT;
    int genPoly = IX_DSP_CID_GEN_POLY;
    int n;
    int len = 0;
    char *pMsgLen;
    char *pParmLen;
    char *pCursor;

    pCursor = pData;

    RETURN_IF_NULL_POINTER((pCursor), 0)

    /* create caller ID data header */
    *pCursor++ = IX_DSP_CID_NTT_DLE;
    *pCursor++ = IX_DSP_CID_NTT_SOH;
    *pCursor++ = IX_DSP_CID_NTT_HEADER;
    *pCursor++ = IX_DSP_CID_NTT_DLE;
    *pCursor++ = IX_DSP_CID_NTT_STX;
    *pCursor++ = IX_DSP_CID_NTT_SVC_TP;
    pMsgLen = pCursor++;
    *pCursor++ = IX_DSP_CID_NTT_PARM_TP;
    pParmLen = pCursor++;

    /* copy the phone number string */
    while(*pPhoneNo && len < IX_DSP_CID_NTT_MAX_NO)
    {
        *pCursor++ = *pPhoneNo++;
        len++;
    }

    RETURN_IF_NULL_POINTER((pParmLen), 0)
    RETURN_IF_NULL_POINTER((pMsgLen), 0)

    /* fill the message and parameter length */
    *pParmLen = (unsigned char)len;
    *pMsgLen = (unsigned char)(len + OFFSET_2);

    /* end of caller ID data */
    *pCursor++ = IX_DSP_CID_NTT_DLE;
    *pCursor++ = IX_DSP_CID_NTT_ETX;

    /* set cursor to the begin */
    pCursor = pData;

    /* add parity check bit to the first 2 bytes */
    *pCursor++ = ixDspCodeletParityEven(*pCursor);
    *pCursor++ = ixDspCodeletParityEven(*pCursor);

    n = len + CID_HEADER_LEN;
    /* calculate parity bits and crc from the 3rd bytes */
    while(n)
    {
        *pCursor = ixDspCodeletParityEven(*pCursor);
        crc = ixDspCodeletAdd2CRC16(*pCursor++, crc, genPoly);
        n--;
    }

    /* append the crc bytes */
    *pCursor++ = (unsigned char) crc;
    *pCursor++ = (unsigned char)(crc >> SHIFT_8);

    return (len + TOTAL_FSK_FIELDS);
}

void ixDspCodeletCidInitJp(int chl, char *numStr)
{
    UINT32 trans;
    IxDspCodeletCidHandle handle;

    ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_CID);
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CID, chl);
    handle = cidObjJp + chl - 1;

    /* update the state */
    handle->state = CID_POLARITY_INV;

    strncpy(handle->cidData, numStr, CID_DATA_BUFFER_SIZE);

    /* reverse the polarity */
    ixDspCodeletPolarityInv(chl);

    /* set timer to 0.2 s */
    IX_DSP_CODELET_SET_TIMER(trans, chl, 20)

}

/************************************************
        Caller id state machine.
        Implement NTT caller id protocol,
        State machine is started by DSP's XMSG_START
        message, then driven by timer and slic events
        and XMSG_TG_PLAY_CMPLT message.
*************************************************/
int ixDspCodeletCallerIdSmJp(XMsgRef_t pMsg)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    int len;
    int rc;
    UINT32 trans;
    IxDspCodeletCidHandle handle;
    unsigned char *pData;
    XMsgTGPlayFSK_t msg;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);
    if((chl<=0)||(chl>IX_DSP_CODELET_MAX_CHL))
    {
        pMsg->type = XMSG_ERROR;
    printf("Invalid channel number, Not supported");
    ixDspCodeletPrtMsg(pMsg);
    error = IX_DSP_CID_UNKNOWN_MSG;
    return error;
    }
    
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CID, chl);
    handle = cidObjJp + chl - 1;

    switch(handle->state)
    {
    case CID_POLARITY_INV:  
        /* delay after polarity inversed, wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* send short ring */
            ixDspCodeletShortRing(chl);

            /* set timer to 6.0 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 600)

            /* update the state */
            handle->state = CID_SHORT_RING;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_SHORT_RING:        /* short ring, wait for connection */
        /* check if short ring time out */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* station not response to short ring, go to normal ring */
            ixDspCodeletStopRing(chl);

            /* send normal ring */
            ixDspCodeletNormalRing(chl);

            /* set timer to 20 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 2000)

            /* update the state */
            handle->state = CID_NORMAL_RING;
        }
        /* station connected */
        else if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_HOOK_STATE)
        {
            /* short ring answered, delay 0.2 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 20)

            /* update the state */
            handle->state = CID_LOOP_CONN_DELAY;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_LOOP_CONN_DELAY:   /* delay, wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* buid DSP FSK play message */
            XMSG_FIELD_TG_PLAY_FSK(&msg, pData);
            len = ixDspCodeletCreateCidDataJP(handle->cidData, pData);
            assert(len > 0);

            XMSG_MAKE_TG_PLAY_FSK(&msg, trans, chl, len);

            /* send FSK play message to DSP */
            rc = xMsgSend(&msg);
            if(rc!=XSUCC)
            {
                printf("ERROR - xMsgSend() fails\n");
            }

            /* update the state */
            handle->state = CID_SEND_FSK;

        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_SEND_FSK:  /* start modem message sent, wait for DSP response */
        /* check if DSP replied */
        if(pMsg->type == XMSG_TG_PLAY_CMPLT)
        {
            /* set timer to 7 s, wait for on hook */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 700)
            handle->state = CID_LOOP_DISCONN;
        }
        else
        {
            /* no response from DSP, abortion */
            ixDspCodeletPolarityInv(chl);
            handle->state = CID_IDLE;
            error = IX_DSP_CID_ERROR;
            printf("ERROR - DSP not response\n");
        }
        break;

    case CID_LOOP_DISCONN:
        /* modem data sent to station, wait for disconnection */
        /* check if time out */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* station not response, abortion */
            ixDspCodeletPolarityInv(chl);
            handle->state = CID_IDLE;
            error = IX_DSP_CID_ERROR;
            printf("ERROR - station not response to CID data\n");
        }
        /* check if station disconnected */
        else if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_HOOK_STATE)
        {
            /* set timer to 0.3 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 30)

            /* update the state */
            handle->state = CID_LOOP_DISCONN_DELAY;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_LOOP_DISCONN_DELAY:        /* delay, wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* send normal ring */
            ixDspCodeletNormalRing(chl);

            /* set timer to 20.0 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 2000)

            /* update the state */
            handle->state = CID_NORMAL_RING;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CID_NORMAL_RING:               /* normal ring, wait for answer */
        /* check if time out or answered */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT &&
           (evtCode == XEVT_NET_TIMER || evtCode == XEVT_NET_HOOK_STATE))
        {
            /* if ring not answered, stop it */
            if(evtCode == XEVT_NET_TIMER)
            {
                ixDspCodeletStopRing(chl);
            }

                /* restore the polarity */
            ixDspCodeletPolarityInv(chl);

            /* clean timer */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

            /* update the state */
            handle->state = CID_IDLE;

        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;
     default:
        break;
    }

    if(error == IX_DSP_CID_UNKNOWN_MSG)
    {
        ixDspCodeletPrtMsg(pMsg);
    }

    return error;
}

/* Japnese CICW demo code */
/* if value = 0, play IIT, if value = 1, play CAT */
void play_iitorcat(int value)
{

    XMsgRef_t pMsg;
    static int msgBuf[XMSG_MAX_WSIZE];
    int type;
    int category = IX_DSP_CODELET_CATEGORY_CIDCW;
    int instance = 1;
    UINT8 *pToneID;
    int numTones = 1;
    UINT32 trans;

    pMsg = (XMsgRef_t)msgBuf;
    type = XMSG_TG_PLAY;

    XMSG_FIELD_TG_PLAY(pMsg, pToneID);
    pToneID[0] = ((value == 0) ? TONE_IIT:TONE_CAT);

    XMSG_MAKE_TG_PLAY(pMsg, 0, 0, numTones);

    trans = IX_DSP_CODELET_MAKE_TRANS(category, instance);

    /* IX_DSP_CODELET_FILL_MSGHDR(pMsg, type, instance, trans); */
    ((XMsgRef_t)(pMsg))->transactionId = trans;
    ((XMsgRef_t)(pMsg))->instance = instance;
    ((XMsgRef_t)(pMsg))->type = type;
    
    if (XSUCC != xMsgSend(pMsg))
    {
        printf("\t Error: Message Send Failed \n");
    }

}

void ixDspCodeletCidCwInitJp(int chl, char *numStr)
{
    UINT32 trans;
    IxDspCodeletCidHandle handle;

    ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_CIDCW);
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CIDCW, chl);
    handle = cidcwObjJp + chl - 1;

    /* update the state */
    handle->state = CIDCW_SEND_IIT;

    strncpy(handle->cidData, numStr, CID_DATA_BUFFER_SIZE);

    /* set timer to 0.2 s */
    IX_DSP_CODELET_SET_TIMER(trans, chl, 20)
}

/************************************************
        CidCw state machine.
        Implement NTT caller id protocol,
        State machine is started by DSP's XMSG_START
        message, then driven by timer and slic events
        and XMSG_TG_PLAY_CMPLT message.
*************************************************/
int ixDspCodeletCidCwSmJp(XMsgRef_t pMsg)
{
    int chl = 1;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    int len;
    int rc;
    UINT32 trans;
    IxDspCodeletCidHandle handle;
    unsigned char *pData;
    XMsgTGPlayFSK_t msg;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);
    
    if( (chl<=0)||(chl>IX_DSP_CODELET_MAX_CHL))
    {
        pMsg->type = XMSG_ERROR;
    printf("Invalid channel number, Not supported");
    ixDspCodeletPrtMsg(pMsg);
    error = IX_DSP_CID_UNKNOWN_MSG;
    return error;
    }    
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_CIDCW, chl);
    handle = cidcwObjJp + chl - 1;

    switch(handle->state)
    {
    case CIDCW_SEND_IIT: /*  wait for timer expiration */
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {

             play_iitorcat(0);

            /* set timer to 1.0 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 100)

            /* update the state */
            handle->state = CIDCW_SEND_CAT;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CIDCW_SEND_CAT:       
        /* check time out */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            /* send CAT */
        
             play_iitorcat(1);

            /* set timer to 0.3 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 30)

            /* update the state */
            handle->state = CIDCW_SEND_FSK;
        }
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }
        break;

    case CIDCW_SEND_FSK:

        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {
            
            /* buid DSP FSK play message */
            XMSG_FIELD_TG_PLAY_FSK(&msg, pData);
            
            len = ixDspCodeletCreateCidDataJP(handle->cidData, pData);
            assert(len > 0);

            XMSG_MAKE_TG_PLAY_FSK(&msg, trans, chl, len);
              
            /* send FSK play message to DSP */
            rc = xMsgSend(&msg);
            if(rc!=XSUCC)
            {
                printf("ERROR - xMsgSend() fails\n");
            }
      
            handle->state = CIDCW_NXT;
        } 
        else
        {
            error = IX_DSP_CID_UNKNOWN_MSG;
        }

         break;
    case CIDCW_NXT:
        /* Check if DSP replied */
        if (pMsg->type == XMSG_TG_PLAY_CMPLT)
        {
            handle->state = CIDCW_STOP;

            /* set timer to 6 s */
            IX_DSP_CODELET_SET_TIMER(trans, chl, 600);

            /* Play the CW tone for another 6s before stopping TG */
            play_iitorcat(0);

        }
        break;

    case CIDCW_STOP:
        /* check if timer expired */
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
        if(pMsg->type == XMSG_EVENT && evtCode == XEVT_NET_TIMER)
        {

            /* STOP THE TG resource */
            XMSG_MAKE_HEAD(pMsg, 0, 4, 0,sizeof(XMsgHdr_t), 0,0);
            ((XMsgRef_t)(pMsg))->transactionId = trans;
            ((XMsgRef_t)(pMsg))->instance = chl;
            ((XMsgRef_t)(pMsg))->type = XMSG_STOP;
    
            if (XSUCC != xMsgSend(pMsg))
            {
               printf("\t Error: Message Send Failed \n");
            }
        }
        break;
    default:
    ; /* No default */
    } /* switch */

    if(error == IX_DSP_CID_UNKNOWN_MSG)
    {
        ixDspCodeletPrtMsg(pMsg);
    }

    return error;
}

