/******************************************************************************
*       @file  IxDspCodeletDiag.c
*
* Contents: This file contains functions for Diagnostic menu.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define PARTY_1  0
#define PARTY_2  1
#define PARTY_3  2

#define CODER_TYPE_G711MU_10MS  1
#define CODER_TYPE_G711A_10MS   2
#define CODER_TYPE_G729A        3

#define SEC_TO_10MSEC_FACTOR  100
#define BYTES_PER_MS_80  80
#define BYTES_PER_MS_10  10

/* check the number of instances of a resource */
#define IX_CODELET_CHECK_INST_NUM(res, i, num) {\
        if((i)==0) {\
            (i) = 1;\
            (num) = (res) == XMPR_NET \
                    || res == XMPR_T38 ? ixDspCodeletResCfg.numChTDM : \
                    (res) == XMPR_PLY ? ixDspCodeletResCfg.numPlayers : \
                    (res) < XMPR_MIX  ? ixDspCodeletResCfg.numChIP : 1;\
        }\
        else { (num) = (i); } \
     }

/* fill the message header after making the message */
#define IX_DSP_CODELET_FILL_MSGHDR(pMsg, typ, inst, trans) \
    {\
        ((XMsgRef_t)(pMsg))->transactionId = (trans);\
        ((XMsgRef_t)(pMsg))->instance      = (inst);\
        ((XMsgRef_t)(pMsg))->type          = (typ);\
    }

typedef void (*make_Message) (XMsgRef_t pMsg);    /* message maker */

/* diagnostic menu structure */
typedef struct
{
    char *name;                         /* menu item name */
    make_Message makeMsg;               /* message maker */
    XMsgType_t  msgType;                /* control message type */
} IxDspCodeletDiagMenu;

/* menu functions */
static void makeMsgHeader(XMsgRef_t pMsg);
static void makeMsgSetmParms(XMsgRef_t pMsg);
static void makeMsgCoderStart(XMsgRef_t pMsg);
static void makeMsgPlayStart(XMsgRef_t pMsg);
static void makeMsgTgPlay(XMsgRef_t pMsg);
static void makeMsgTgPlayFSK(XMsgRef_t pMsg);
static void makeMsgTdRcv(XMsgRef_t pMsg);
static void makeMsgTdRcvFSK(XMsgRef_t pMsg);
static void makeMsgGetJBStat(XMsgRef_t pMsg);
static void makeMsgStartIP(XMsgRef_t pMsg);
static void makeMsgStopIP(XMsgRef_t pMsg);
static void makeMsgSetupCallwParms(XMsgRef_t pMsg);
static void makeMsgSetClearChan(XMsgRef_t pMsg);
static void makeMsgLink(XMsgRef_t pMsg);
static void makeMsgCreate3wCall(XMsgRef_t pMsg);
static void makeMsgBackto2wCall(XMsgRef_t pMsg);
static void makeMsgT38Switch(XMsgRef_t pMsg);
static void loadVoicePrompt(XMsgRef_t pMsg);
static void playRawData(XMsgRef_t pMsg);

static void printPrompts(void);
XMediaHandle_t ixDspCodeletLoadVoicePrompt(char *fname);
int ixDspCodeletLoadRawData(char *fname, unsigned char *pMem, int size);

typedef struct
{
    XMediaHandle_t  handle;
    char fname[IX_DSP_CODELET_STRLEN];
}IxDspCodeletPromptList;

static IxDspCodeletPromptList promptList[IX_DSP_CODELET_MAX_PROMPTS] =
{
    {0, "Hardcoded voice data"},
    {1, "Hardcoded voice data"}
};

static int numLoadedPrompts = 2;
static int rawDataPromptIndex = 0;
static XCachePromptDesc_t rawDesc;

/* diagnostic menu table */
IxDspCodeletDiagMenu ixDspCodeletDiagMenu[] =
{
/*      item name            msg maker               msg type
    -------------------------------------------------------------------------*/
    { "Print Menu",          NULL,                   (XMsgType_t)0},
    { "Reset a resource",    makeMsgHeader,          XMSG_RESET},
    { "Start a resource",    makeMsgHeader,          XMSG_START},
    { "Stop a resource",     makeMsgHeader,          XMSG_STOP},
    { "Ping a resource",     makeMsgHeader,          XMSG_PING},
    { "Set multiple params", makeMsgSetmParms,       XMSG_SET_MPARMS},
    { "Start codec",         makeMsgCoderStart,      XMSG_CODER_START},
    { "Start Player",        makeMsgPlayStart,       XMSG_PLY_START},
    { "Play tones",          makeMsgTgPlay,          XMSG_TG_PLAY},
    { "Play FSK data",       makeMsgTgPlayFSK,       XMSG_TG_PLAY_FSK},
    { "Receive digits",      makeMsgTdRcv,           XMSG_TD_RCV},
    { "Receive FSK",         makeMsgTdRcvFSK,        XMSG_TD_RCV_FSK},
    { "Get JB statistics",   makeMsgGetJBStat,       XMSG_GET_JBSTAT},
    { "Start an IP term",    makeMsgStartIP,        
                                     (XMsgType_t)IX_DSP_CODELET_MSG_START_IP},
    { "Stop an IP term",     makeMsgStopIP,          
                                      (XMsgType_t)IX_DSP_CODELET_MSG_STOP_IP},
    { "Setup call w/parms",  makeMsgSetupCallwParms,
                             (XMsgType_t)IX_DSP_CODELET_MSG_SETUP_CALLWPARMS},
    { "Set clear channel",   makeMsgSetClearChan,
                               (XMsgType_t)IX_DSP_CODELET_MSG_SET_CLEAR_CHAN},
    { "Link 2 terms",        makeMsgLink,    
                                         (XMsgType_t)IX_DSP_CODELET_MSG_LINK},
    { "Create 3W call",      makeMsgCreate3wCall, 
                                (XMsgType_t)IX_DSP_CODELET_MSG_CREATE_3WCALL},
    { "Back to 2W call",     makeMsgBackto2wCall,  
                                (XMsgType_t)IX_DSP_CODELET_MSG_BACKTO_2WCALL},
    { "T38 Switch",          makeMsgT38Switch,   
                                   (XMsgType_t)IX_DSP_CODELET_MSG_T38_SWITCH},
    { "Load a voice prompt", loadVoicePrompt,        (XMsgType_t)0},
    { "Play raw data file",  playRawData,            (XMsgType_t)0},
    { "Exit",                NULL,                   (XMsgType_t)0},
    { NULL,                  NULL,                   (XMsgType_t)0}
};


extern XDSPResConfig_t ixDspCodeletResCfg;
static int instance;
static int numInstance;
static UINT32 trans;
static IxDspCodeletMsgCreate3wCall msg3wCall;

/*******************************
 * Diagnostic menu main function
 *******************************/
void ixDspCodeletDiag(void)
{
    int i;
    int selectedItem=0;
    int lastItem = -1;
    int category;
    int type;
    XMsgRef_t pMsg;
    static int  msgBuf[XMSG_MAX_WSIZE];

    pMsg = (XMsgRef_t)msgBuf;

    do
    {
       /* print the test menu */
        if(!selectedItem)
        {
            printf("\n\t----------------------------------------\n"
                     "\t-      IxDspCodelet Diagnostic Menu    -\n"
                     "\t----------------------------------------\n");

            for (i = 0; ixDspCodeletDiagMenu[i].name != NULL; i++)
            {
                printf("\t%3d - %s\n", i, ixDspCodeletDiagMenu[i].name);
            }

            printf("\tPlease select test item - ");
            lastItem = i - 1;
        }

        /* select a test item */
        selectedItem = ixDspCodeletGetNum(NULL);

        /* skip if not a test item */
        if (selectedItem >= lastItem || !selectedItem)
        {
          continue;
        }

        type = ixDspCodeletDiagMenu[selectedItem].msgType;

        if(!type)
        {
            ixDspCodeletDiagMenu[selectedItem].makeMsg(pMsg);
        }
        else if(type < IX_DSP_CODELET_MSG_TYPE_BEGIN)
        {
            /* get instance # */
            instance =
                ixDspCodeletGetNum("\tEnter the instance number "
                                   "(1,2,...; 0 for all instances) - ");
  
            if((instance<0) || (instance >IX_DSP_CODELET_MAX_CHL))
            {
              printf("Invalid Entry\n"); 
              return;
            }   
  
            /* make a control message */
            ixDspCodeletDiagMenu[selectedItem].makeMsg(pMsg);

            IX_CODELET_CHECK_INST_NUM(pMsg->resource, instance, numInstance)
            for(; instance<=numInstance; instance++)
            {
                trans = IX_DSP_CODELET_MAKE_TRANS(category, instance);
                IX_DSP_CODELET_FILL_MSGHDR(pMsg, type, instance, trans)

                if(xMsgSend(pMsg) != XSUCC)
                {
                    printf("\tERROR - Message sender fails.\n");
                }
            }
        }
        else
        {

            /* make a user-defined control message */
            ixDspCodeletDiagMenu[selectedItem].makeMsg(pMsg);
            if(xMsgSend(pMsg) != XSUCC)
            {
                printf("\tERROR - Message sender fails.\n");
            }
        }

    } while(selectedItem != lastItem);
}


/**************************
 * make message header
 **************************/
static void makeMsgHeader(XMsgRef_t pMsg)
{
    int res;

    res   = ixDspCodeletGetNum("\tEnter the resource number\n"
                        "\t\tResource : (1:Net, 2:DEC, 3:ENC, 4:TG, 5:TD,"
                                        " 6:PLY, 7:MIX, 8:T38, 9:MA) - ");

    XMSG_MAKE_HEAD(pMsg, 0, res, 0, sizeof(XMsgHdr_t), 0, 0)
}


/***********************************
 * get all current parameter values
 **********************************/
void ixDspCodeletGetCurrentParms(int inst, int res)
{
    int numInst;

    IX_CODELET_CHECK_INST_NUM(res, inst, numInst)

    for(; inst<=numInst; inst++)
    {
        ixDspCodeletPrtResParms(res, inst);
    }
 }

/**************************************
 * make set multiple parameter message
 **************************************/
static void makeMsgSetmParms(XMsgRef_t pMsg)
{
    int res;
    int i;
    int numParms;
    UINT16 *pIDs;
    UINT16 *pVals;

    do
    {
        res = ixDspCodeletGetNum("\tEnter the resource number\n"
                                 "\t\tResource : (1:Net, 2:DEC, 3:ENC, 4:TG,"
                                       " 5:TD, 6:PLY, 7:MIX, 8:T38, 9:MA) - ");
    } while(res < 1 || res >= XMPR_EOL);

    numParms = ixDspCodeletGetNum("\tEnter the number of parameters"
                                                    " to be set - ");
    ixDspCodeletGetCurrentParms(instance, res);
    printf("\n");

    XMSG_FIELD_SET_MPARMS(pMsg, pIDs, pVals);
    for (i = 0; i < numParms; i++)
    {
        pIDs[i]  = ixDspCodeletGetNum("\tEnter the parameter id - ");
        pVals[i] = ixDspCodeletGetNum("\tEnter the parameter value - ");
    }

    XMSG_MAKE_SET_MPARMS(pMsg, 0, res, 0, numParms);
}

/**************************
 * make codec start message
 **************************/
static void makeMsgCoderStart(XMsgRef_t pMsg)
{
    int res;
    int fpp = 1;
    int codec;

    res   = ixDspCodeletGetNum("\tEnter the resource number\n"
                   "\t\tResource : (2:DEC, 3:ENC) - ");
    codec = ixDspCodeletGetNum(
                   "\t\t0 :PassThru (for debug only)\n"
                   "\t\t1 :G711_10ms u-Law\n"
                   "\t\t2 :G711_10ms A-Law\n"
                   "\t\t3 :G729a/b\n"
                   "\t\t4 :G723.1\n"
                   "\t\t5 :G722\n"
                   "\t\t6 :G726 40Kbps\n"
                   "\t\t7 :G726 32Kbps\n"
                   "\t\t8 :G726 24Kbps\n"
                   "\t\t9 :G726 16Kbps\n"
           "\t\t10:G729.1\n"
                   "\tEnter the type of Codec - ");

    if(res == XMPR_ENC)
    {
        fpp = ixDspCodeletGetNum("\tEnter frames per packet - ");
    }

    XMSG_MAKE_CODER_START(pMsg, 0, res, 0, codec, fpp);
}

/****************************
 * make play_start message
 ****************************/
static void makeMsgPlayStart(XMsgRef_t pMsg)
{
    int i;
    int numSeg;
    int tempGetNum;
    XPlyMediaDesc_t *pMedSeg;

    printf("\tEnter the number of prompts (1 ~ %d) - ", (int)XMAX_PLY_SEG);
    numSeg = ixDspCodeletGetNum(NULL);

    printPrompts();

    XMSG_FIELD_PLY_START(pMsg, pMedSeg)

    for (i = 0; i < numSeg; i++)
    {
        printf("\tSegement %d :\n", i+1);
        pMedSeg[i].handle = 
            (XMediaHandle_t)ixDspCodeletGetNum("\tEnter the prompt handle - ");

        tempGetNum = ixDspCodeletGetNum("\tEnter the offset "
                                               "(multiple of 10 if G.729) - ");
        pMedSeg[i].offset = (INT32)tempGetNum;

        tempGetNum = ixDspCodeletGetNum("\tEnter the length "
                                                 "(0: end of the segment) - ");
        pMedSeg[i].length = (INT32)tempGetNum;

        tempGetNum = ixDspCodeletGetNum("\tEnter the next link"
                                             "(fwd:>0, back:<=0, end:127) - ");
        pMedSeg[i].next = (INT16)tempGetNum;
    }

    XMSG_MAKE_PLY_START(pMsg, 0, 0, numSeg)
}

/****************************
 * make Tene Gen. play message
 ****************************/
static void makeMsgTgPlay(XMsgRef_t pMsg)
{
    int i;
    int numTones;
    unsigned char *pToneID;

    numTones = ixDspCodeletGetNum("\tEnter the number of tones "
                                              "to be played - ");

    XMSG_FIELD_TG_PLAY(pMsg, pToneID);

    printf("\t\tDTMF tone IDs = 0~15\n"
           "\t\tCall progress tone IDs = Dial:66, Ring:70, Busy:72, CW:79\n"
           "\t\tExamples of user-added tone IDs = 251~255\n");

    for (i = 0; i < numTones; i++)
    {
        pToneID[i] = ixDspCodeletGetNum("\tEnter the tone id - ");
    }

    XMSG_MAKE_TG_PLAY(pMsg, 0, 0, numTones);
}

/****************************
 * make Tene Gen. play FSK message
 ****************************/
static void makeMsgTgPlayFSK(XMsgRef_t pMsg)
{
    int numBytes;
    char *pData;

    XMSG_FIELD_TG_PLAY_FSK(pMsg, pData);

    numBytes = ixDspCodeletGetStr("\tEnter an ASCII string - ", pData);

    XMSG_MAKE_TG_PLAY_FSK(pMsg, 0, 0, numBytes);
}

/********************************
 * make Tone Det. receive message
 ********************************/
static void makeMsgTdRcv(XMsgRef_t pMsg)
{
    int numDigitsToRcv;
    int tmDig;
    int fstTmout;
    int intTmout;
    int totalTmout;

    numDigitsToRcv  = ixDspCodeletGetNum("\tEnter the number of digits to "
                                                              "receive - ");
    tmDig    = ixDspCodeletGetNum("\tEnter the termination digit bits "
                                              "(*=1024 #=2048 *#=3072) - ");
    fstTmout = SEC_TO_10MSEC_FACTOR * 
        ixDspCodeletGetNum("\tEnter the first digit time out (in second) - ");
    intTmout = SEC_TO_10MSEC_FACTOR * 
        ixDspCodeletGetNum("\tEnter the inter-digit time out (in second) - ");
    totalTmout = SEC_TO_10MSEC_FACTOR * 
        ixDspCodeletGetNum("\tEnter the total time out (in second) - ");

    XMSG_MAKE_TD_RCV(pMsg, 0, 0, numDigitsToRcv, tmDig,
             totalTmout, fstTmout, intTmout)
}

/********************************
 * make FSK receive message
 ********************************/
static void makeMsgTdRcvFSK(XMsgRef_t pMsg)
{
    int timeout;

    timeout  = SEC_TO_10MSEC_FACTOR * ixDspCodeletGetNum("\tEnter the timeout "
                                                   "(in 1 second unit) - ");

    XMSG_MAKE_TD_RCV_FSK(pMsg, 0, 0, timeout)
}

/********************************
 * make get-JB-stat message
 ********************************/
static void makeMsgGetJBStat(XMsgRef_t pMsg)
{
    int reset;

    reset  = ixDspCodeletGetNum("\tReset the statistics (1:yes, 0:no) ? ");

    XMSG_MAKE_GET_JBSTAT(pMsg, 0, 0, reset)
}


/********************************
 * make start-IP message
 ********************************/
static void makeMsgStartIP(XMsgRef_t pMsg)
{
    int ch;
    UINT32 trans;

    ch = ixDspCodeletGetNum("\tIP channel = ");
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, ch);
    IX_DSP_CODELET_MAKE_MSG_START_IP(pMsg, trans, ch)
}


/********************************
 * make stop-IP message
 ********************************/
static void makeMsgStopIP(XMsgRef_t pMsg)
{
    int ch;
    UINT32 trans;

    ch = ixDspCodeletGetNum("\tIP channel = ");
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, ch);
    IX_DSP_CODELET_MAKE_MSG_STOP_IP(pMsg, trans, ch)
}

/********************************
 * make SetupCallwParms message
 ********************************/
static void makeMsgSetupCallwParms(XMsgRef_t pMsg)
{
    IxDspCodeletMsgSetupCallwParms *pMsgSet;
    UINT32 trans;

    pMsgSet = (IxDspCodeletMsgSetupCallwParms *)pMsg;

    pMsgSet->channelTDM = (UINT8)ixDspCodeletGetNum("\tTDM channel = ");
    pMsgSet->channelIP = (UINT8)ixDspCodeletGetNum("\tIP channel = ");
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, 
                                                pMsgSet->channelTDM);

   pMsgSet->parms.decAutoSwitch=(UINT16)ixDspCodeletGetNum("\tAuto switch = ");
    pMsgSet->parms.decType = (UINT8)ixDspCodeletGetNum("\tDecoder type = ");
    pMsgSet->parms.encType = (UINT8)ixDspCodeletGetNum("\tEncoder type = ");
    pMsgSet->parms.vad = (UINT8)ixDspCodeletGetNum("\tVAD = ");
    pMsgSet->parms.frmsPerPkt = 
                        (UINT8)ixDspCodeletGetNum("\tFrames Per Packet = ");
    pMsgSet->parms.rfc2833 = (UINT8)ixDspCodeletGetNum("\tRFC2833 = ");
    pMsgSet->parms.rfc2833pyldType = (UINT8)ixDspCodeletGetNum("\tRFC2833 "
                                                       "payload type = ");
    pMsgSet->parms.toneClamp = (UINT8)ixDspCodeletGetNum("\tTone clamp = ");
    pMsgSet->parms.toneReport = (UINT8)ixDspCodeletGetNum("\tTone report = ");

    IX_DSP_CODELET_MAKE_MSGHDR_SETUP_CALLWPARMS(pMsg, trans);
}

/********************************
 * make SetClearChan message
 ********************************/
static void makeMsgSetClearChan(XMsgRef_t pMsg)
{
    int chTDM;
    int chIP;
    int code;
    UINT32 trans;

    chTDM = ixDspCodeletGetNum("\tTDM channel = ");
    chIP = ixDspCodeletGetNum("\tIP channel = ");
    code = ixDspCodeletGetNum("\tCode type = ");
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, chTDM);

    IX_DSP_CODELET_MAKE_MSG_SET_CLEAR_CHAN(pMsg, 0, chIP, chTDM, code)
}

/****************************
 * make Link message
 ****************************/
static void makeMsgLink(XMsgRef_t pMsg)
{
    IxDspCodeletMsgLink *pMsgLink;
    UINT32 trans;

    pMsgLink = (IxDspCodeletMsgLink *)pMsg;

    pMsgLink->term1.type =
      (UINT8)ixDspCodeletGetNum("\tType of Term 1(0:Null, \
1:TDM, 2:IP, 3:Mixer) = ");

    if(pMsgLink->term1.type != IX_DSP_CODELET_TERM_NULL)
    {
      pMsgLink->term1.channel = 
                (UINT8)ixDspCodeletGetNum("\tChannel of Term 1  = ");
    }

    pMsgLink->term2.type =
      (UINT8)ixDspCodeletGetNum("\tType of term 2 (0:Null, \
1:TDM, 2:IP, 3:Mixer) = ");

    if(pMsgLink->term2.type != IX_DSP_CODELET_TERM_NULL)
    {
      pMsgLink->term2.channel = 
                (UINT8)ixDspCodeletGetNum("\tChannel of Term 2 = ");
    }

    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST,
                                                   pMsgLink->term1.channel);
    IX_DSP_CODELET_MAKE_MSGHDR_LINK(pMsg, trans)
}

/**********************************
 * make create-3way-call message
 *********************************/
static void makeMsgCreate3wCall(XMsgRef_t pMsg)
{
    IxDspCodeletMsgCreate3wCall *pMsgC3w;
    UINT32 trans;

    pMsgC3w = (IxDspCodeletMsgCreate3wCall *)pMsg;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, 1);

    IX_DSP_CODELET_MAKE_MSGHDR_CREATE_3WCALL(pMsg, trans)

    if(pMsgC3w!=NULL) {

   pMsgC3w->parties[PARTY_1].type=(UINT8)ixDspCodeletGetNum("\tType of Party 1"
                                                     "  (1:TDM, 2:IP) = ");
    pMsgC3w->parties[PARTY_1].channel=(UINT8)ixDspCodeletGetNum("\tChannel of "
                                                            "party 1  = ");

   pMsgC3w->parties[PARTY_2].type=(UINT8)ixDspCodeletGetNum("\tType of Party 2"
                                                      " (1:TDM, 2:IP) = ");
    pMsgC3w->parties[PARTY_2].channel=(UINT8)ixDspCodeletGetNum("\tChannel of "
                                                             "Party 2 = ");

   pMsgC3w->parties[PARTY_3].type=(UINT8)ixDspCodeletGetNum("\tType of Party 3"
                                                      " (1:TDM, 2:IP) = ");
    pMsgC3w->parties[PARTY_3].channel=(UINT8)ixDspCodeletGetNum("\tChannel of "
                                                             "Party 3 = ");
    msg3wCall = *pMsgC3w;
    }
}

/**********************************
 * make back-to-2way-call message
 *********************************/
static void makeMsgBackto2wCall(XMsgRef_t pMsg)
{
    IxDspCodeletMsgBackto2wCall *pMsgB2w;
    UINT32 trans;

    pMsgB2w = (IxDspCodeletMsgBackto2wCall *)pMsg;

    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, 1);
    IX_DSP_CODELET_MAKE_MSGHDR_BACKTO_2WCALL(pMsg, trans)

    pMsgB2w->party1 = msg3wCall.parties[PARTY_1];
    pMsgB2w->party2 = msg3wCall.parties[PARTY_2];
    pMsgB2w->partyToDrop = msg3wCall.parties[PARTY_3];

}

/********************************
 * make T38-switch message
 ********************************/
static void makeMsgT38Switch(XMsgRef_t pMsg)
{
    int chTDM;
    int chIP;
    int mode;
    int attr = 0;
    UINT32 trans;

    chTDM = ixDspCodeletGetNum("\tTDM channel = ");
    chIP = ixDspCodeletGetNum("\tIP channel = ");
    mode = ixDspCodeletGetNum("\tmode (0:voice, 1:fax) = ");
    if(mode)
    {
      attr = ixDspCodeletGetNum("\tfax mode (32:receive, 36:send) = ");
    }

    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, chTDM);

    IX_DSP_CODELET_MAKE_MSG_T38_SWITCH(pMsg, trans, chIP, chTDM, mode, attr)
}

static void loadVoicePrompt(XMsgRef_t pMsg)
{
    char *fname;

    if(numLoadedPrompts == IX_DSP_CODELET_MAX_PROMPTS)
    {
        printf("too many prompts loaded\n");
        return;
    }

    fname = promptList[numLoadedPrompts].fname;

    /* now append file name */
    if (!ixDspCodeletGetStr("\tEnter full path wave file name "
                                   "(e.g. /wave/pcm.wav): ",fname))
    {
        printf("no file\n");
        return;
    }

    promptList[numLoadedPrompts].handle = ixDspCodeletLoadVoicePrompt(fname);

    if(promptList[numLoadedPrompts].handle != XMEDIA_HANDLE_NULL)
    {
        numLoadedPrompts++;
    }
    else
    {
        printPrompts();
    }
}

static void playRawData(XMsgRef_t pMsg)
{
    int inst;
    int numInst;
    int trans;
    int size, type;
    int tempGetNum;
    char *fname;
    XPlyMediaDesc_t *pMedSeg;
    XMsgPlyStart_t playMsg;

    if(numLoadedPrompts == IX_DSP_CODELET_MAX_PROMPTS)
    {
        printf("too many prompts loaded\n");
        return;
    }

    if (!rawDataPromptIndex)
    {
        /* static raw data storage hasn't been allocated yet */
        tempGetNum = ixDspCodeletGetNum("\tEnter maximum sample size "
                                                        "(8000=>1 sec.): ");
        rawDesc.size = (INT32)tempGetNum;
        if (!rawDesc.size)
        {
          return;
        }

        type = ixDspCodeletGetNum("\tCoder Type (1:uLaw;2:aLaw;3:G.729): ");
        switch (type)
        {
            case CODER_TYPE_G711MU_10MS: 
                    rawDesc.type = XCODER_TYPE_G711MU_10MS;
                    break;
            case CODER_TYPE_G711A_10MS: 
                    rawDesc.type = XCODER_TYPE_G711A_10MS;
                    break;
            case CODER_TYPE_G729A: 
                    rawDesc.type = XCODER_TYPE_G729A;
                    break;
            default:printf("invalid coder type\n");
                    return;
        }

        rawDesc.pBuffer = (unsigned char *)(malloc(rawDesc.size));
        if (!rawDesc.pBuffer)
        {
            printf("unable to allocate space\n");
            return;
        }

        /* register with DSP */
        promptList[numLoadedPrompts].handle = xDspRegCachePrompt(&rawDesc);

        if(promptList[numLoadedPrompts].handle == XMEDIA_HANDLE_NULL)
        {
            printf("unable to register cache prompt\n");
            free(rawDesc.pBuffer);
            return;
        }
        rawDataPromptIndex = numLoadedPrompts;
        numLoadedPrompts++;
    }

    fname = promptList[rawDataPromptIndex].fname;

    /* now get file name */
    if (!ixDspCodeletGetStr("\tEnter full path raw data file name "
                                         "(e.g. /data/pcm.dat): ",fname))
    {
        printf("no file\n");
        return;
    }
    size = ixDspCodeletLoadRawData(fname, rawDesc.pBuffer, rawDesc.size);
    if (!size)
    {
        printf("unable to read file\n");
        return;
    }
#ifdef OS_USRLINUX
    if(!ixDspCodeletSyncData(promptList[rawDataPromptIndex].handle, 
                                           rawDesc.pBuffer, rawDesc.size))
     {
    printf("unable to sync \n");
    return;

     }
#endif
    printf("\tCurrent settings: max. size = %d, type = ", rawDesc.size);
    switch (rawDesc.type)
    {
        case XCODER_TYPE_G711MU_10MS:
            size = size/BYTES_PER_MS_80;     /* 80 bytes per 10ms */
            printf("uLaw\n");
            break;
        case XCODER_TYPE_G711A_10MS:
            size = size/BYTES_PER_MS_80;     /* 80 bytes per 10ms */
            printf("aLaw\n");
            break;
        case XCODER_TYPE_G729A:
            size = size/BYTES_PER_MS_10;     /* 10 bytes per 10ms */
            printf("G.729A\n");
            break;
        default:
            printf("unknown\n");
    }

    inst = ixDspCodeletGetNum("\tEnter the instance number (1,2,...; 0 "
                                                  "for all instance) - ");

    if((inst<0) || (inst >IX_DSP_CODELET_MAX_CHL))
    {
       printf("Invalid Entry\n"); 
     return;
    }

    IX_CODELET_CHECK_INST_NUM(XMPR_PLY, inst, numInst)

    XMSG_FIELD_PLY_START(&playMsg, pMedSeg)

    pMedSeg[0].handle = promptList[rawDataPromptIndex].handle;
    pMedSeg[0].offset = 0;
    pMedSeg[0].length = size;
    pMedSeg[0].next = XPLY_MEDIA_SEG_EOP;

    for(; inst<=numInst; inst++)
    {
        trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST, inst);
        XMSG_MAKE_PLY_START(&playMsg, trans, inst, 1)
        if(xMsgSend(&playMsg) != XSUCC)
        {
            printf("ERROR - Message sender fails.\n");
            break;
        }
    }
}

static void printPrompts(void)
{
    int i;

    printf(    "\tRegistered Prompt List:\n"
               "\tprompt handle   file name\n"
               "\t----------------------------------\n");
    for(i=0; i<numLoadedPrompts; i++)
    {
        printf("\t   %3d          %s\n",
                promptList[i].handle, promptList[i].fname);
    }

    printf("\n");
}

