/*
 * @file IxDspCodeletGw.c
 *
 * Contents: This file contains funtions for the Gateway Demo.
 *
 * -- Intel Copyright Notice --
 * 
 * Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors. The software is licensed under 
 * IXA SDK license.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * 
 * 
*/

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define IX_DSP_CODELET_GWST_UNCHANGE    (-1)
#define IX_DSP_CODELET_NUN_LEN          3
#define IX_DSP_CODELET_ERR_UNKNOWN_MSG  1
#define IX_DSP_CODELET_CALL_STACK       20

/* fax transmition mode */
#define IX_DSP_CODELET_FAX_MODE_OFF     0
#define IX_DSP_CODELET_FAX_MODE_BYPASS  1
#define IX_DSP_CODELET_FAX_MODE_T38     2

/* channel mode */
#define IX_DSP_CODELET_CHAN_MODE_VOICE  IX_DSP_CODELET_FAX_MODE_OFF
#define IX_DSP_CODELET_CHAN_MODE_CLEAR  IX_DSP_CODELET_FAX_MODE_BYPASS
#define IX_DSP_CODELET_CHAN_MODE_T38    IX_DSP_CODELET_FAX_MODE_T38

/* Channel Number */
#define CHANNEL_ID_1 1
#define CHANNEL_ID_2 2
#define CHANNEL_ID_3 3
#define CHANNEL_ID_4 4

/* General Defines */
#define GW_STATE_NAME  20
#define RFC_2833_PAYLOAD_TYPE  101
#define GW_MAX_PARAMS  5
#define GW_PARAM_ID_1  0
#define GW_PARAM_ID_2  1
#define GW_PARAM_ID_3  2
#define GW_PARAM_ID_4  3
#define GW_PARAM_ID_5  4
#define SECONDS_25_MSEC  2500
#define SECONDS_20_MSEC  2000
#define SECONDS_3_MSEC   300

/* print the state information */
#define IX_DSP_CODELET_GW_TRACE(x) \
    {if(traceEnable) {printf x;}}

#define IX_DSP_CODELET_GW_SND_MSG(pMsg) \
    {if(xMsgSend(pMsg) != XSUCC) {printf("%s\n", msgSendErrStr);}}

/* simulate the communication to call stack */
#define IX_DSP_CODELET_SEND_CALL_MSG(dialNum, typ) { UINT32 trans; \
        int chl = 0; \
        if (!strcmp(dialNum, "111")) { chl=CHANNEL_ID_1; } \
        else if (!strcmp(dialNum, "222")) { chl=CHANNEL_ID_2; } \
        else if (!strcmp(dialNum, "333")) { chl=CHANNEL_ID_3; } \
        else if (!strcmp(dialNum, "444")) { chl=CHANNEL_ID_4; } \
        trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl); \
        XMSG_MAKE_HEAD (msgBuf, trans, IX_DSP_CODELET_CALL_STACK, \
                        chl, sizeof(XMsgHdr_t), typ, 0) \
       xMsgWrite(msgBuf); }

/* GW state index */
typedef enum
{
    IX_DSP_CODELET_GWST_IDLE = 0,     /* idle */
    IX_DSP_CODELET_GWST_SETUP,        /* get calling number and setup call*/
    IX_DSP_CODELET_GWST_WT_ANS,       /* wait for answer */
    IX_DSP_CODELET_GWST_CONN,         /* connected */
    IX_DSP_CODELET_GWST_RING,         /* ring */
    IX_DSP_CODELET_GWST_BUSY,         /* busy */
} IxDspCodeletGwStateIndex;

/* Simulated messages to/from call stack */
typedef enum
{
    IX_DSP_CODELET_GWMSG_SETUP = IX_DSP_CODELET_MSG_END_OF_LIST,
    IX_DSP_CODELET_GWMSG_ACCEPT,
    IX_DSP_CODELET_GWMSG_REJECT,
    IX_DSP_CODELET_GWMSG_CONN,
    IX_DSP_CODELET_GWMSG_DISCONN,
    IX_DSP_CODELET_GWMSG_CLR_CHAN,
    IX_DSP_CODELET_GWMSG_T38
} IxDspCodeletGwMsgType;

/* simulated call setup message to/from call stack */
typedef struct{
    XMsgHdr_t    header;
    UINT8        callingNum[IX_DSP_CODELET_NUN_LEN + 1];
    UINT8        callerNum[IX_DSP_CODELET_NUN_LEN + 1];
} IxDspCodeletGwMsgSetup;

typedef struct ixDspCodeletGwChan_t IxDspCodeletGwChan;
typedef void (*IxDspCodeletGwStateInit)(IxDspCodeletGwChan *pChl);
typedef int (*IxDspCodeletGwMsgHndlr)(IxDspCodeletGwChan *pChl, 
                               XMsgRef_t pMsgBuf, int *pNextState);

/* GW state */
typedef struct{
    char stateName[GW_STATE_NAME];      /* name */
    IxDspCodeletGwStateInit init;       /* state init function */
    IxDspCodeletGwMsgHndlr  handler;    /* state msg handler function */
} IxDspCodeletGwState;

/* channel parameters, they are not changed call by call */
typedef struct{
    UINT8   ec;             /* ec flag */
    UINT8   alc;            /* ALC flag */
    UINT8   cng;            /* CNG flag */
    UINT8   agc;            /* AGC flag */
    UINT8   faxByPasCoder;  /* fax bypass coder type */
} IxDspCodeletChanParms;

/* GW channel */
struct ixDspCodeletGwChan_t
{
    int                     channelId;  /* channel Id */
    IxDspCodeletGwState     *pState;    /* ptr to current state */
    IxDspCodeletCallParms   callParms;  /* params involved in call
                                                 setup negotiation */
    IxDspCodeletChanParms   chanParms;  /* params involved in clear 
                                                   channel setting */

    char    localNum[IX_DSP_CODELET_NUN_LEN + 1]; /* local phone number */
    UINT8   mode;                       /* current channel mode */
    UINT8   faxMode;                    /* fax transmition mode */
    char    dialledNum[IX_DSP_CODELET_NUN_LEN + 1]; /* Dialed  phone number */
};


/* state functions */
void ixDspCodeletGwInitIdle(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrIdle(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodeletGwInitSetup(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrSetup(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodeletGwInitWtAns(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrWtAns(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodeletGwInitConn(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrConn(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodeletGwInitBusy(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrBusy(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodeletGwInitRing(IxDspCodeletGwChan *pChl);
int ixDspCodeletGwHndlrRing(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                     int *pNextState);
void ixDspCodelGwSmExcept(XMsgRef_t pMsg, IxDspCodeletGwChan *pChl);

/* GW state machine table */
static IxDspCodeletGwState gwStates[] =
{
    {"IDLE",      ixDspCodeletGwInitIdle,   ixDspCodeletGwHndlrIdle},
    {"SETUP",     ixDspCodeletGwInitSetup,  ixDspCodeletGwHndlrSetup},
    {"WAIT ANS",  ixDspCodeletGwInitWtAns,  ixDspCodeletGwHndlrWtAns},
    {"CONNECTED", ixDspCodeletGwInitConn,   ixDspCodeletGwHndlrConn},
    {"RING",      ixDspCodeletGwInitRing,   ixDspCodeletGwHndlrRing},
    {"BUSY",      ixDspCodeletGwInitBusy,   ixDspCodeletGwHndlrBusy}
};

/* declare GW 2 channels and default parameters */

static IxDspCodeletGwChan gwChannels[] =
{
    {
        CHANNEL_ID_1,               /* channel Id */
        gwStates,                   /* initial state */
        {                           /* call parameters set call by call */
            0xffff,                 /* auto switch mask */
            XCODER_TYPE_G729A,      /* decoder type */
            XCODER_TYPE_G729A,      /* encoder type */
            1,                      /* frames per packet */
            XPARM_OFF,              /* vad */
            XPARM_ON,               /* RFC 2833 */
            RFC_2833_PAYLOAD_TYPE,  /* RFC 2833 payload type */
            XPARM_ON,               /* tone clamping */
            XPARM_TD_RPT_TONE_ON    /* tone reporting */
        },
        {  /* channel parameter involved in clear channel setting */
            XPARM_ON,               /* EC */
            XPARM_ON,               /* ALC */
            XPARM_ON,               /* CNG */
            XPARM_OFF,              /* AGC */
            XCODER_TYPE_G711MU_10MS /* coder type for fax bypass */
        },
        "111",                      /* local phone number */
        IX_DSP_CODELET_CHAN_MODE_CLEAR,  /* current channel mode */
        0,                           /* fax transmission mode */
    "000"                           /* dialled phone number */
    },
    {
        CHANNEL_ID_2,               /* channel Id */
        gwStates,                   /* initial state */
        {                           /* call parameters set call by call */
            0xffff,                 /* auto switch mask */
            XCODER_TYPE_G729A,      /* decoder type */
            XCODER_TYPE_G729A,      /* encoder type */
            1,                      /* frames per packet */
            XPARM_OFF,              /* vad */
            XPARM_ON,               /* RFC 2833 */
            RFC_2833_PAYLOAD_TYPE,  /* RFC 2833 payload type */
            XPARM_ON,               /* tone clamping */
            XPARM_TD_RPT_TONE_ON    /* tone reporting */
        },
        {  /* channel parameter involved in clear channel setting */
            XPARM_ON,               /* EC */
            XPARM_ON,               /* ALC */
            XPARM_ON,               /* CNG */
            XPARM_OFF,              /* AGC */
            XCODER_TYPE_G711MU_10MS /* coder type for fax bypass */
        },
        "222",                      /* local phone number */
        IX_DSP_CODELET_CHAN_MODE_CLEAR,  /* current channel mode */
        0,                           /* fax transmission mode */
    "000"                           /* dialled phone number */
    },
    {
        CHANNEL_ID_3,               /* channel Id */
        gwStates,                   /* initial state */
        {                           /* call parameters set call by call */
            0xffff,                 /* auto switch mask */
            XCODER_TYPE_G729A,      /* decoder type */
            XCODER_TYPE_G729A,      /* encoder type */
            1,                      /* frames per packet */
            XPARM_OFF,              /* vad */
            XPARM_ON,               /* RFC 2833 */
            RFC_2833_PAYLOAD_TYPE,  /* RFC 2833 payload type */
            XPARM_ON,               /* tone clamping */
            XPARM_TD_RPT_TONE_ON    /* tone reporting */
        },
        {  /* channel parameter involved in clear channel setting */
            XPARM_ON,               /* EC */
            XPARM_ON,               /* ALC */
            XPARM_ON,               /* CNG */
            XPARM_OFF,              /* AGC */
            XCODER_TYPE_G711MU_10MS /* coder type for fax bypass */
        },
        "333",                      /* local phone number */
        IX_DSP_CODELET_CHAN_MODE_CLEAR,  /* current channel mode */
        0,                           /* fax transmission mode */
    "000"                           /* dialled phone number */
    },
    {
        CHANNEL_ID_4,               /* channel Id */
        gwStates,                   /* initial state */
        {                           /* call parameters set call by call */
            0xffff,                 /* auto switch mask */
            XCODER_TYPE_G729A,      /* decoder type */
            XCODER_TYPE_G729A,      /* encoder type */
            1,                      /* frames per packet */
            XPARM_OFF,              /* vad */
            XPARM_ON,               /* RFC 2833 */
            RFC_2833_PAYLOAD_TYPE,  /* RFC 2833 payload type */
            XPARM_ON,               /* tone clamping */
            XPARM_TD_RPT_TONE_ON    /* tone reporting */
        },
        {  /* channel parameter involved in clear channel setting */
            XPARM_ON,               /* EC */
            XPARM_ON,               /* ALC */
            XPARM_ON,               /* CNG */
            XPARM_OFF,              /* AGC */
            XCODER_TYPE_G711MU_10MS /* coder type for fax bypass */
        },
        "444",                      /* local phone number */
        IX_DSP_CODELET_CHAN_MODE_CLEAR,  /* current channel mode */
        0,                           /* fax transmission mode */
    "000"                           /* dialled phone number */
    },
};

/* message buf */
static int msgBuf[XMSG_MAX_WSIZE];
static int traceEnable;
static char *msgSendErrStr = "GW: ERROR - xMsgSend() fails";

/************************************
 * GW state machine
 ************************************/
int ixDspCodeletGwSm(XMsgRef_t pMsg)
{

    int rc;
    int chl;
    int nextState = IX_DSP_CODELET_GWST_UNCHANGE;
    IxDspCodeletGwChan *pChl;
    IxDspCodeletGwState *pState;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);

    /* valid channel number */
    if(chl < CHANNEL_ID_1 || chl > CHANNEL_ID_4) 
     /* Gateway demo only for 4 channels supported */
    {
        IX_DSP_CODELET_GW_TRACE(("GW: Invalid channel number %d\n", chl))
        return -1;
    }

    /* retrieve channel object and current state */
    pChl = gwChannels + chl - 1;
    pState = pChl->pState;

    /* call state message handler */
    rc = pState->handler(pChl, pMsg, &nextState);

    /* call execption message handler */
    if(rc == IX_DSP_CODELET_ERR_UNKNOWN_MSG)
    {
        ixDspCodelGwSmExcept(pMsg, pChl);
    }

    /* goto new state if current state completed */
    if(nextState != IX_DSP_CODELET_GWST_UNCHANGE)
    {
        pChl->pState = pState = gwStates + nextState;

        IX_DSP_CODELET_GW_TRACE(("GW[%d]: GOTO %s state\n", chl, 
                                              pState->stateName))

        /* initialize new state */
        pState->init(pChl);
    }

    return rc;
}

/************************************
 * check if GW demo is in idle state
 ************************************/
int ixDspCodeletGwActive(void)
{
    return

        gwChannels[CHANNEL_ID_1-1].pState != gwStates ||
        gwChannels[CHANNEL_ID_2-1].pState != gwStates ||
        gwChannels[CHANNEL_ID_3-1].pState != gwStates ||
        gwChannels[CHANNEL_ID_4-1].pState != gwStates;
}

/************************************
 * GW demo initialization
 ************************************/
void ixDspCodeletGwSmInit(void)
{
    int chl;
    int codeType, faxMode;

    codeType = ixDspCodeletGetNum(
                             "\t1 :G711_10ms u-Law\n"
                             "\t2 :G711_10ms A-Law\n"
                             "\t3 :G729a/b\n"
                             "\t4 :G723.1\n"
                             "\t5 :G722\n"
                             "\t6 :G726 40Kbps\n"
                             "\t7 :G726 32Kbps\n"
                             "\t8 :G726 24Kbps\n"
                             "\t9 :G726 16Kbps\n"
                             "\t10:G729.1\n"
                             "Enter the coder type - ");

    faxMode = ixDspCodeletGetNum("Fax mode (0:Disabled, 1:Fax Bypass,"
                                                        " 2:T38) ? ");
    traceEnable = ixDspCodeletGetNum("Enable trace (0:no, 1:yes) ? ");


    for(chl=CHANNEL_ID_1; chl<=CHANNEL_ID_4; chl++)

    {
        gwChannels[chl - 1].pState = gwStates;
        gwChannels[chl - 1].callParms.decType = codeType;
        gwChannels[chl - 1].callParms.encType = codeType;
        gwChannels[chl - 1].faxMode = faxMode;
        ixDspCodeletGwInitIdle(gwChannels + chl - 1);
    }
}

/******************************************
 * handling exception messages out of states
 ******************************************/
void ixDspCodelGwSmExcept(XMsgRef_t pMsg, IxDspCodeletGwChan *pChl)
{
    int evtCode;
    int evtData1;
    int evtData2;
    int chl;
    int error = 0;
    UINT32 trans;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    switch(pMsg->type)
    {
    case IX_DSP_CODELET_GWMSG_SETUP:
        /* receive setup at non-idle state, reject */
        IX_DSP_CODELET_GW_TRACE(("GW[%d]: receive SETUP, rejected\n", chl))


    IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, IX_DSP_CODELET_GWMSG_REJECT)

        break;

    case IX_DSP_CODELET_GWMSG_CLR_CHAN:

        IX_DSP_CODELET_GW_TRACE(
            ("GW[%d]: SET_CLR_CHAN msg received\n", chl))

        IX_DSP_CODELET_MAKE_MSG_SET_CLEAR_CHAN
            (msgBuf, trans, chl, chl, pChl->chanParms.faxByPasCoder)

        IX_DSP_CODELET_GW_SND_MSG(msgBuf)

        pChl->mode = IX_DSP_CODELET_CHAN_MODE_CLEAR;

        break;

    default:
      
    if (pMsg->type == IX_DSP_CODELET_MSG_ACK)
    {
        break;
    }
    else if(pMsg->type == XMSG_EVENT)
    {
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

            if(evtCode == XEVT_CODE_TD_TONEON)
            {
                /* switch to clear channel if fax tones detected */
                if(pChl->faxMode == IX_DSP_CODELET_FAX_MODE_BYPASS &&
                   (evtData1 == RFC_TID_FAX_CED ||
                    evtData1 == RFC_TID_FAX_CNG ||
                    evtData1 == RFC_TID_FAX_V21))
                {
                    IX_DSP_CODELET_GW_TRACE(
                        ("GW[%d]: fax tone (%d) detected, switch to clear"
                                                  " channel\n", chl, evtData1))

                    IX_DSP_CODELET_MAKE_MSG_SET_CLEAR_CHAN
                       (msgBuf, trans, chl, chl, pChl->chanParms.faxByPasCoder)

                    IX_DSP_CODELET_GW_SND_MSG(msgBuf)

                    pChl->mode = IX_DSP_CODELET_CHAN_MODE_CLEAR;

                    /* inform the another party to switch to clear channel */


            IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                        IX_DSP_CODELET_GWMSG_CLR_CHAN)

                }
                else
                { 
                    IX_DSP_CODELET_GW_TRACE(
                       ("GW[%d]: tone(id=%d) detected, ignored\n", 
                                                       chl, evtData1))
                }
            }
            else
            {
                error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
            }
    }
    else 
    {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
    }
    break;
    }

    if(traceEnable && error == IX_DSP_CODELET_ERR_UNKNOWN_MSG )
    {
        ixDspCodeletPrtMsg(pMsg);
    }
}

/************************************
 * set channel parameters
 ************************************/
void ixDspCodeletGwSetChanParms(IxDspCodeletGwChan *pChl)
{
    int chl;
    UINT32  trans;
    IxDspCodeletMsgSetParms *pSetParms;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS( IX_DSP_CODELET_CATEGORY_GW, chl);

    pSetParms = (IxDspCodeletMsgSetParms *)msgBuf;

    IX_DSP_CODELET_MAKE_MSGHDR_SET_PARMS(pSetParms, trans)

    pSetParms->numParms = GW_MAX_PARAMS;
    pSetParms->parms[GW_PARAM_ID_1].parmID = XPARMID_NET_ECENABLE;
    pSetParms->parms[GW_PARAM_ID_1].value = pChl->chanParms.ec;
    pSetParms->parms[GW_PARAM_ID_1].dspResource = XMPR_NET;
    pSetParms->parms[GW_PARAM_ID_1].dspResInstance = chl;
    pSetParms->parms[GW_PARAM_ID_2].parmID = XPARMID_DEC_ALC;
    pSetParms->parms[GW_PARAM_ID_2].value = pChl->chanParms.agc;
    pSetParms->parms[GW_PARAM_ID_2].dspResource = XMPR_DEC;
    pSetParms->parms[GW_PARAM_ID_2].dspResInstance = chl;
    pSetParms->parms[GW_PARAM_ID_3].parmID = XPARMID_DEC_CNG;
    pSetParms->parms[GW_PARAM_ID_3].value = pChl->chanParms.cng;
    pSetParms->parms[GW_PARAM_ID_3].dspResource = XMPR_DEC;
    pSetParms->parms[GW_PARAM_ID_3].dspResInstance = chl;
    pSetParms->parms[GW_PARAM_ID_4].parmID = XPARMID_ENC_AGC;
    pSetParms->parms[GW_PARAM_ID_4].value = pChl->chanParms.agc;
    pSetParms->parms[GW_PARAM_ID_4].dspResource = XMPR_ENC;
    pSetParms->parms[GW_PARAM_ID_4].dspResInstance = chl;
    pSetParms->parms[GW_PARAM_ID_5].parmID = XPARMID_TD_RPT_EVENTS;
    pSetParms->parms[GW_PARAM_ID_5].value = XPARM_TD_RPT_TONE_ON;
    pSetParms->parms[GW_PARAM_ID_5].dspResource = XMPR_TNDET;
    pSetParms->parms[GW_PARAM_ID_5].dspResInstance = chl;

    IX_DSP_CODELET_GW_SND_MSG(pSetParms)
}


/************************************
 * init function for idle state
 ************************************/
void ixDspCodeletGwInitIdle(IxDspCodeletGwChan *pChl)
{
    int chl;
    UINT32 trans;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    /* clear timer */
    IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

    if(pChl->mode != IX_DSP_CODELET_CHAN_MODE_VOICE)
    {
        /* restore the call paramters if switched to fax mode */
        IX_DSP_CODELET_GW_TRACE(
            ("GW[%d]: Reset channel parameters\n", chl))

        pChl->mode = IX_DSP_CODELET_CHAN_MODE_VOICE;
        ixDspCodeletGwSetChanParms(pChl);
        return;
    }

    /* stop IP resource */
    IX_DSP_CODELET_MAKE_MSG_STOP_IP(msgBuf, trans, chl);
    IX_DSP_CODELET_GW_SND_MSG(msgBuf)
}

/************************************
 * message handler for idle state
 ************************************/
int ixDspCodeletGwHndlrIdle(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                        int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    IxDspCodeletGwMsgSetup *pSutupMsg;
    IxDspCodeletMsgLink *pMsgLink;
    UINT32 trans;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    switch(pMsg->type){

  case IX_DSP_CODELET_MSG_ACK:
        /* this is the ACK to set channel parameters */

        pMsgLink = (IxDspCodeletMsgLink *)msgBuf;

        /* link TDM and IP terminations in case it's broken by other app */
        pMsgLink->term1.type = IX_DSP_CODELET_TERM_TDM;
        pMsgLink->term2.type = IX_DSP_CODELET_TERM_IP;
        pMsgLink->term1.channel = pMsgLink->term2.channel = chl;

        IX_DSP_CODELET_MAKE_MSGHDR_LINK(msgBuf, trans)
        IX_DSP_CODELET_GW_SND_MSG(msgBuf)

        break;

    case IX_DSP_CODELET_MSG_LINK_ACK:
        /* this is the ACK to link TDM and IP terminations */

        /* stop IP resource */
        IX_DSP_CODELET_MAKE_MSG_STOP_IP(msgBuf, trans, chl);
        IX_DSP_CODELET_GW_SND_MSG(msgBuf)

        break;

    case IX_DSP_CODELET_MSG_STOP_ACK:

        if(ixDspCodeletGetHookState(chl))
        {
            *pNextState = IX_DSP_CODELET_GWST_SETUP;
        }

        break;

    default:
        
    if (pMsg->type == IX_DSP_CODELET_GWMSG_SETUP)
    {
        pSutupMsg = (IxDspCodeletGwMsgSetup *)pMsg;

            IX_DSP_CODELET_GW_TRACE(
                ("GW[%d]: SETUP msg received from '%s'\n", chl, 
                                          pSutupMsg->callerNum))

            if(strcmp(pSutupMsg->callingNum, pChl->localNum))
            {

            IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                   IX_DSP_CODELET_GWMSG_REJECT)

            }
            else
            {

            IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                                IX_DSP_CODELET_GWMSG_ACCEPT)


                *pNextState = IX_DSP_CODELET_GWST_RING;
            }
    }
    else if(pMsg->type == XMSG_EVENT)
    {
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

            if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_OFF)
            {
                *pNextState = IX_DSP_CODELET_GWST_SETUP;
            }
            else
            {
                error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
            } 
    }
    else 
    {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
    }
    break;
    }

    return error;
}


/************************************
 * init function for setup state
 ************************************/
void ixDspCodeletGwInitSetup(IxDspCodeletGwChan *pChl)
{
    UINT32 trans;

    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW,
                                                      pChl->channelId);

    /* start tone detector */
    XMSG_MAKE_START(msgBuf, trans, XMPR_TNDET, pChl->channelId)

    IX_DSP_CODELET_GW_SND_MSG(&msgBuf)
}

/************************************
 * message handler for setup state
 ************************************/
int ixDspCodeletGwHndlrSetup(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                        int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;

    int reason;
    int num;
    UINT8  *pDigits;
    UINT8  *pTones;

    UINT32 trans;
    int i;
    int channel;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    switch(pMsg->type){
    case XMSG_ACK:

        /* play dial tone */
        XMSG_FIELD_TG_PLAY(msgBuf, pTones)
        pTones[0] = RFC_TID_DIAL;
        XMSG_MAKE_TG_PLAY(msgBuf, trans, chl, 1)
        IX_DSP_CODELET_GW_SND_MSG(msgBuf)

        /* connect calling number */
        XMSG_MAKE_TD_RCV
            (msgBuf,
             trans,
             chl,
             IX_DSP_CODELET_NUN_LEN,    /* num digits */
             0,                         /* term digits */
             SECONDS_25_MSEC,           /* total 25 sec. */
             SECONDS_20_MSEC,           /* 1st digit 20 sec */
             SECONDS_3_MSEC             /* inter digit 3 sec */
            )
        IX_DSP_CODELET_GW_SND_MSG(msgBuf)
    
        break;

    case XMSG_TD_RCV_CMPLT:

        XMSG_FIELD_TD_RCV_CMPLT(pMsg, reason, num, pDigits)

        if(num)
        {
            /*IX_DSP_CODELET_SEND_CALL_SETUP_MSG(pChl, pDigits, num)*/

            /* simulate the communication to call stack */
            for(i=0; i<num; i++)
            {    ((IxDspCodeletGwMsgSetup *)msgBuf)->callingNum[i] =
                          pChl->dialledNum[i] = pDigits[i] + '0';
            }
            ((IxDspCodeletGwMsgSetup *)msgBuf)->callingNum[i] =
                                              pChl->dialledNum[i] =0;
            strncpy(((IxDspCodeletGwMsgSetup *)msgBuf)->callerNum, 
                      pChl->localNum, (IX_DSP_CODELET_NUN_LEN + 1) );
            if (!strcmp(pChl->dialledNum, "111")) {
                 channel= IX_DSP_CODELET_CHANNEL_1;}
            else if (!strcmp(pChl->dialledNum, "222")) {
                      channel= IX_DSP_CODELET_CHANNEL_2;}
            else if (!strcmp(pChl->dialledNum, "333") ){
                      channel= IX_DSP_CODELET_CHANNEL_3;}
            else if (!strcmp(pChl->dialledNum, "444")) {
                      channel= IX_DSP_CODELET_CHANNEL_4;}
            else { printf("ERROR - Invalid Number \n");
                    break; }

            strncpy((gwChannels[channel-1]).dialledNum, pChl->localNum,
                        (IX_DSP_CODELET_NUN_LEN + 1));
            trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, 
                                              channel);
            XMSG_MAKE_HEAD
                   (msgBuf,
                   trans,
                   IX_DSP_CODELET_CALL_STACK,
                   channel,
                   sizeof(IxDspCodeletGwMsgSetup),
                   IX_DSP_CODELET_GWMSG_SETUP,
                   0)
            xMsgWrite(msgBuf);

            IX_DSP_CODELET_GW_TRACE(("GW[%d]: calling '%s'...\n",
                    chl, ((IxDspCodeletGwMsgSetup *)msgBuf)->callingNum))

            /* set timer 0.5 sec */
            IX_DSP_CODELET_SET_TIMER(trans, pChl->channelId, 50)
        }
        else
        {
            *pNextState = IX_DSP_CODELET_GWST_BUSY;
        }

        break;

   case XMSG_EVENT:

        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

        if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_ON)
        {
            *pNextState = IX_DSP_CODELET_GWST_IDLE;
        }
        else if(evtCode == XEVT_CODE_TD_TONEON)
        {
            /* stop Tone Gen. on the first digit entering */
            XMSG_MAKE_STOP(msgBuf, trans, XMPR_TNGEN, chl);
            IX_DSP_CODELET_GW_SND_MSG(msgBuf)
        }
        else
        {
            error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
        }
        break;

    case XMSG_TG_PLAY_CMPLT:
        break;

    default:
        
    if (pMsg->type == IX_DSP_CODELET_GWMSG_ACCEPT)
    {
        IX_DSP_CODELET_GW_TRACE(("GW[%d]: ACCEPT msg received\n", chl))
            *pNextState = IX_DSP_CODELET_GWST_WT_ANS;
    }
    else if (pMsg->type == IX_DSP_CODELET_GWMSG_REJECT)
    {
        IX_DSP_CODELET_GW_TRACE(("GW[%d]: REJECT msg received\n", chl))
            *pNextState = IX_DSP_CODELET_GWST_BUSY;
    }
    else 
    {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
    }
    break;
    }

    return error;
}

/************************************
 * init function for wait answer state
 ************************************/
void ixDspCodeletGwInitWtAns(IxDspCodeletGwChan *pChl)
{
    UINT8  *pTones;
    UINT32 trans;

    trans = IX_DSP_CODELET_MAKE_TRANS( IX_DSP_CODELET_CATEGORY_GW, 
                                                   pChl->channelId);

    /* play ring tone */
    XMSG_FIELD_TG_PLAY(msgBuf, pTones)
    pTones[0] = RFC_TID_RING;
    XMSG_MAKE_TG_PLAY(msgBuf, trans, pChl->channelId, 1)
    IX_DSP_CODELET_GW_SND_MSG(msgBuf)

    /* wait for 20 sec */
    IX_DSP_CODELET_SET_TIMER(trans, pChl->channelId, 2000)
    
 
}

/************************************
 * message handler for wait answer state
 ************************************/
int ixDspCodeletGwHndlrWtAns(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                       int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;

    chl = pChl->channelId;

    if(pMsg->type == XMSG_EVENT) {
      
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

        if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_ON)
        {
           
        IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum,
                                       IX_DSP_CODELET_GWMSG_DISCONN)

            *pNextState = IX_DSP_CODELET_GWST_IDLE;
        }
        else if(evtCode == XEVT_NET_TIMER)
        {
              
        IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                       IX_DSP_CODELET_GWMSG_DISCONN)

            *pNextState = IX_DSP_CODELET_GWST_BUSY;
        }
        else
        {
    
            error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
        }
    }
    else {    
      if(pMsg->type == IX_DSP_CODELET_GWMSG_CONN)
      {
        IX_DSP_CODELET_GW_TRACE(("GW[%d]: CONNECT received\n", chl))

            *pNextState = IX_DSP_CODELET_GWST_CONN;
      }
      else 
      {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
      }
    }

    return error;
}

/************************************
 * init function for connnection state
 ************************************/
void ixDspCodeletGwInitConn(IxDspCodeletGwChan *pChl)
{
    int chl;
    UINT32 trans;
    IxDspCodeletMsgSetupCallwParms *pSetupMsg;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    /* stop timer */
    IX_DSP_CODELET_SET_TIMER(trans, chl, 0)
    
   
    /* setup IP resources (start ENC, DEC and stop TG) */
    if(pChl->mode == IX_DSP_CODELET_CHAN_MODE_CLEAR)
    {
 
        /* already set clear channel parameters, only start */
        IX_DSP_CODELET_MAKE_MSG_START_IP(msgBuf, trans, chl)
    }
    else
    {
        /* start with call parameters */
        pSetupMsg = (IxDspCodeletMsgSetupCallwParms *)msgBuf;
        IX_DSP_CODELET_MAKE_MSGHDR_SETUP_CALLWPARMS(pSetupMsg, trans)
        pSetupMsg->channelIP = chl;
        pSetupMsg->channelTDM = chl;
        pSetupMsg->parms = pChl->callParms;

    }

    IX_DSP_CODELET_GW_SND_MSG(msgBuf)
    
    /* reset tone generator -in attempt to stop it */
  /*  XMSG_MAKE_RESET(msgBuf, trans, XMPR_TNGEN, pChl->channelId) */
  /*  IX_DSP_CODELET_GW_SND_MSG(&msgBuf) */
    

}

/************************************
 * message handler for connection state
 ************************************/
int ixDspCodeletGwHndlrConn(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg, 
                                                       int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    UINT32 trans;
    IxDspCodeletMsgAck *pAckMsg;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    switch(pMsg->type){

    case IX_DSP_CODELET_GWMSG_DISCONN:

        if(pChl->mode == IX_DSP_CODELET_CHAN_MODE_T38)
        {
           break;
        }

        IX_DSP_CODELET_GW_TRACE(("GW[%d]: DISCONNECT received\n", chl))

        *pNextState = IX_DSP_CODELET_GWST_BUSY;

        break;

    case IX_DSP_CODELET_GWMSG_T38:

        IX_DSP_CODELET_GW_TRACE(
            ("GW[%d]: SWITCH_T38 msg received\n", chl))

        IX_DSP_CODELET_MAKE_MSG_T38_SWITCH
            (msgBuf, trans, chl, chl, IX_DSP_CODELET_MSG_MODE_T38, 0)

        IX_DSP_CODELET_GW_SND_MSG(msgBuf)

        pChl->mode = IX_DSP_CODELET_CHAN_MODE_T38;

        break;

    default:
        /*error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;*/
    if(pMsg->type == IX_DSP_CODELET_MSG_SETUP_ACK)
    {
        pAckMsg = (IxDspCodeletMsgAck *)pMsg;

            if(pAckMsg->numErrors)
            {
                IX_DSP_CODELET_GW_TRACE(
                    ("GW[%d]: Call setup completed, (err=%d)\n", chl, 
                                                       pAckMsg->numErrors))
            }
    }
    else if(pMsg->type == IX_DSP_CODELET_MSG_T38_ACK)
    {
        pAckMsg = (IxDspCodeletMsgAck *)pMsg;

            IX_DSP_CODELET_GW_TRACE(
                ("GW[%d]: Switched to %s mode\n", chl, pChl->mode ? 
                                                          "T.38" : "voice"))

            if(pChl->mode != IX_DSP_CODELET_CHAN_MODE_T38)
            {
                /* if the line has gone on-hook state during T.38 session */
                if(!ixDspCodeletGetHookState(chl))
                {
                    *pNextState = IX_DSP_CODELET_GWST_IDLE;
                }

            }

            if(pAckMsg->numErrors)
            {
                IX_DSP_CODELET_GW_TRACE(
                    ("GW[%d]: T38 switch completed with %d error(s)\n", chl,
                                                         pAckMsg->numErrors))
            }
    }
    else if(pMsg->type == XMSG_T38_CMPLT)
    {
        /* switch back to voice mode if end of T38 session */

            IX_DSP_CODELET_GW_TRACE(
               ("GW[%d]: T38 session completed, switch back to voice mode\n",
                                                                         chl))

            IX_DSP_CODELET_MAKE_MSG_T38_SWITCH
               (msgBuf, trans, chl, chl, IX_DSP_CODELET_MSG_MODE_VOICE, 0)

            IX_DSP_CODELET_GW_SND_MSG(msgBuf)

            pChl->mode = IX_DSP_CODELET_CHAN_MODE_VOICE;
    }
    else if (pMsg->type == XMSG_EVENT)
    {
        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

            if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_ON &&
               pChl->mode != IX_DSP_CODELET_CHAN_MODE_T38)
            {

                IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                            IX_DSP_CODELET_GWMSG_DISCONN)
                *pNextState = IX_DSP_CODELET_GWST_IDLE;
            }

            else if (evtCode == XEVT_CODE_TD_TONEON &&
                    pChl->faxMode == IX_DSP_CODELET_FAX_MODE_T38 &&
                    (evtData1 == RFC_TID_FAX_CED || evtData1 
                                                      == RFC_TID_FAX_V21))
            {
                /* switch to T38 mode if fax tones detected */
                IX_DSP_CODELET_GW_TRACE(
                    ("GW[%d]: fax tone (%s) detected, switch to T.38 mode\n",
                    chl, evtData1 == RFC_TID_FAX_CED ? "CED" : "V21"))

                IX_DSP_CODELET_MAKE_MSG_T38_SWITCH
                    (msgBuf, trans, chl, chl, IX_DSP_CODELET_MSG_MODE_T38,
                                                                   evtData1)

                IX_DSP_CODELET_GW_SND_MSG(msgBuf)

                pChl->mode = IX_DSP_CODELET_CHAN_MODE_T38;

                /* inform the another party to switch to T.38 */

            IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum, 
                                          IX_DSP_CODELET_GWMSG_T38)


            }

            else
            {
                error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
            }
    }
    else 
    {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
    }
    break;
    }

    return error;
}

/************************************
 * init function for busy state
 ************************************/
void ixDspCodeletGwInitBusy(IxDspCodeletGwChan *pChl)
{
    UINT8  *pTones;
    UINT32 trans;

    trans = IX_DSP_CODELET_MAKE_TRANS( IX_DSP_CODELET_CATEGORY_GW, pChl->channelId);

    /* play busy tone for 30 sec */
    IX_DSP_CODELET_SET_TIMER(trans, pChl->channelId, 3000)

    /* play busy tone */
    XMSG_FIELD_TG_PLAY(msgBuf, pTones)
    pTones[0] = RFC_TID_BUSY;
    XMSG_MAKE_TG_PLAY(msgBuf, trans, pChl->channelId, 1)
    IX_DSP_CODELET_GW_SND_MSG(msgBuf)
}

/************************************
 * message handler for busy state
 ************************************/
int ixDspCodeletGwHndlrBusy(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                       int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    UINT32 trans;

    chl = pChl->channelId;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_GW, chl);

    switch(pMsg->type){

    case XMSG_ERROR:
        /* error happens if no off-hook warning tone for the country */
        break;

    case XMSG_EVENT:

        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

        if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_ON)
        {
            *pNextState = IX_DSP_CODELET_GWST_IDLE;
        }
        else if(evtCode == XEVT_NET_TIMER)
        {
            UINT8  *pTones;

            /* play off hook warning tone */
            XMSG_FIELD_TG_PLAY(msgBuf, pTones)
            pTones[0] = RFC_TID_OFFHK_WARN;
            XMSG_MAKE_TG_PLAY(msgBuf, trans, chl, 1)
            IX_DSP_CODELET_GW_SND_MSG(msgBuf)
        }
        else
        {
            error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
        }
        break;

    default:
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
    }

    return error;
}

/************************************
 * init function for ring state
 ************************************/
void ixDspCodeletGwInitRing(IxDspCodeletGwChan *pChl)
{
    ixDspCodeletNormalRing(pChl->channelId);
}

/************************************
 * message handler for ring state
 ************************************/
int ixDspCodeletGwHndlrRing(IxDspCodeletGwChan *pChl, XMsgRef_t pMsg,
                                                      int *pNextState)
{
    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;

    chl = pChl->channelId;

    if(pMsg->type == XMSG_EVENT) {

        XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

        if(evtCode == XEVT_NET_HOOK_STATE && evtData1 == XHOOK_STATE_OFF)
        {

        IX_DSP_CODELET_SEND_CALL_MSG(pChl->dialledNum,
                                               IX_DSP_CODELET_GWMSG_CONN)
       

            *pNextState = IX_DSP_CODELET_GWST_CONN;
        }
        else
        {
            error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
        }
    }
    else {
      /*error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;*/
      if (pMsg->type == IX_DSP_CODELET_GWMSG_DISCONN)
      {
        IX_DSP_CODELET_GW_TRACE(("GW[%d]: DISCONNECT msg received\n", chl))
            ixDspCodeletStopRing(chl);
            *pNextState = IX_DSP_CODELET_GWST_IDLE;
      }
      else 
      {
        error = IX_DSP_CODELET_ERR_UNKNOWN_MSG;
      }
    }

    return error;
}

