/******************************************************************************
*       @file   IxDspCodeletMsgPrint.c
*
* Contents: This file contains functions to print messages. 
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the user defined include files required.
 */

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define MAX_STR_LEN     200
#define MAX_COMPONENTS  5
#define DTMF_TONE_10    10
#define DTMF_TONE_11    11
#define DTMF_TONE_12    12

/* union for handling the sign bit properly */
typedef union {
    INT16  i16;
    UINT16 ui16;
} XParmUnion_t;

static char *pResName[] = {
    "NULL",
    "NET",
    "DEC",
    "ENC",
    "TG",
    "TD",
    "PLY",
    "MIX",
    "T38",
    "MA"
};

static int numRes = (sizeof pResName) / sizeof(char *);


static char *pParmNameNet[] = {
    "Current state (0:Idle, 1:Active)",
    NULL,
    "PCM format (0:A-law, 1:Mu-Law, 2:PassThru)",
    "EC enable/disable (0:OFF, 1:ON)",
    "EC tail length (2 ~ 64, multiple of 2, in 1-ms unit)",
    "EC NLP (0:OFF, 1:NLP ON & SUP OFF, 2:NLP & SUP ON)",
    "EC Freeze (0:adaptive, 1:freeze)",
    "EC Delay compensation (0 ~ 240 in 0.125 ms unit)",
    "Flash hook detection window (in 10-ms unit)",
    "Timer counter (in 10-ms unit)",
    "Rx Gain (+15 ~ -40 in 1 dB unit)",
    "Tx Gain (+15 ~ -40 in 1 dB unit)",
    "Short bypass enable/disable (0:OFF, 1:ON)",
    "AEC enable/disable (0:OFF, 1:ON)",
    "AEC Select algorithm (0: Sub Band, 1:Sub Band Fast)",
    "AEC Tail length (1 ~ 128 ms tail length, in 1-ms unit)",
    "AEC NLP enable/disbale (0:OFF, 1: ON )",
    "AEC Adaptation freeze (0:adaptive, 1:freeze)",
    "AEC Howling control enable/disable (0: OFF, 1: ON)",
    "AEC TD enable/disable (0:OFF, 1:ON)",
    "Lip Sync Tx(PCM->IP)delay(0~1000ms in 1 ms unit)",
    "Lip Sync Rx(PCM<-IP)delay(0~1000ms in 1 ms unit)"
};

static char *pParmNameDec[] = {
    "Current state (0:Idle, 1:Active)",
    "Volume adjustment in dB (+15 ~ -40). -99 for mute",
    "ALC enable/disable (0:OFF, 1:ON)",
    "CNG enable/disable (0:OFF, 1:ON)",
    "Code type (0:PassThru, 1:G711u, 2:G711a, 3:G729a,\n\t\t\t"
    "   4:G723, 5:G722_64, 6:G722_56, 7:G722_48,\n\t\t\t"
    "   8:G726_40, 9:G726_32, 10:G726_24, 11:G726_16, 12:G729.1)",
    "Packet-loss report (0:OFF, 1:ON)",
    "Report packet type change (0:OFF, 1:ON)",
    "OR'd auto-switch cntrl bits (1:G711u, 2:G711a, 4:G729a,\n\t\t\t"
    "   8:G723, 16:G722, 32:G726_40, 64:G726_32, 128:G726_24, \n\t\t\t"
    "   256:G726_16, 512:G729.1)",
    "Jitter buffer maximum delay in ms",
    "Jitter buffer packet loss rate (in 0.1%% unit)",
    "Jitter buffer mode (0:Static 1:Histogram 2:RFC3550)",
    "G726 40Kbps RTP payload type",
    "G726 32Kbps RTP payload type",
    "G726 24Kbps RTP payload type",
    "G726 16Kbps RTP payload type",
    "G726 packing format (0:LSB - VoIP, 1:MSB - AAL)",
   "G729.1 RTP payload type"
};

static char *pParmNameEnc[] = {
    "Current state (0:Idle, 1:Active)",
    NULL,
    "AGC enable/disable (0:OFF, 1:ON)",
    "VAD enable/disable (0:OFF, 1:ON)",
    "Code type (0:PassThru, 1:G711u, 2:G711a, 3:G729a,\n\t\t\t"
    "   4:G723, 5:G722_64, 6:G722_56, 7:G722_48,\n\t\t\t"
    "   8:G726_40, 9:G726_32, 10:G726_24, 11:G726_16, 12:G729.1)",
    "Number of frames per packet",
    "Packet-loss report (0:OFF, 1:ON)",
    "G723 encoding rate (0:6.3k, 1:5.3k)",
    "G726 40Kbps RTP payload type",
    "G726 32Kbps RTP payload type",
    "G726 24Kbps RTP payload type",
    "G726 16Kbps RTP payload type",
    "G726 packing format (0:LSB - VoIP, 1:MSB - AAL)",
    "Volume adjustment in dB (+15 ~ -40). -99 for mute",
   "G729.1 RTP payload type",
   "G729.1 Encoding rate(0:8k, 1:12k, 2:14k, 3:16k,\n\t\t\t"
   "   4:18k, 5:20k, 6:22k, 7:24K, 8:26k, 9:28k, 10:30k, 11:32k)"
};

static char *pParmNameTg[] = {
    "Current state (0:Idle, 1:Active)",
    "Volume adjustment in dB (+15 ~ -20)",
    "Mode (0: V.23, 1: Bellcore 202)",
    "Number of CS bits for FSK transmitter",
    "Number of mark bits for FSK transmitter",
    "FSK Baud rate (8:1200, 16:600, 32:300, 64:150, 128:75)",
    "RFC2833 enable/disable (0:OFF, 1:ON)",
    "Number of post mark bits for FSK transmitter"
};

static char *pParmNameTd[] = {
    "Current state (0:Idle, 1:Active)",
    NULL,
    "Tone clamping enable/disable (0:OFF, 1:ON)",
    "Tone clamping buffer size",
    "Tone event report(0:None,1:Tone On,2:Tone Off,3:Both)",
    "RFC2833 event enable/disable (0:OFF, 1:ON)",
    "Update rate of RFC2833 packets (in 10-ms unit)",
    "Redundancy of end-of-RFC2833-packet",
    "Redundancy of begin-of-RFC2833-packet",
    "Suppress Encoder in RFC2833 event (0:NO, 1:YES)",
    "RFC2833 payload type",
    "Minimum CS-bit length required by FSK receiver ",
    "Minimum mark-bit length required by FSK receiver",
    "Extra stop bits allowed between the data",
    "FSK receiver Baud rate (reserved for future)",
    "FSK adaptive threshold (0: Non-adaptive, 1:Adaptive)",
    "FSK Minimum No. of data bytes to receive (0: default, 0~15)"
};

static char *pParmNamePly[] = {
    "Current state (0:Idle, 1:Active)",
    "Volume adjustment in dB (+15 ~ -30)",
};

static char *pParmNameMix[] = {
    "Current state (0:Idle, 1:Active)",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};

static char *pParmNameT38[] = {
    "Current state (0:Idle, 1:Active)",
    "Ellipsis (0:OFF, 1:ON)",
    "FEC (0:OFF, 1:ON)",
    "redundancy level (0 ~ 7)",
    "modem rate negotiation (0:end-to-end, 1:local)",
    "TCF error threshold (in percentage)",
    "packet transport (0:UDP, 1:TCP)",
    "mode (0:ITU-T.38, 1:China-T.38)",
    "Use China T38 Disconnect type (1:YES, 0:NO)",
    "Number of FEC messages (1 ~ 5)",
    "Number of Pkts per FEC message (2 ~ 3)"
};

static char *pParmNameMa[] = {
    "Current state (0:Idle, 1:Active)",
    "Debug (0:OFF, 1:ON)"
};

typedef struct{
    int numParms;
    char **ppParmNames;
}IxDspCodeletParmDesc;

static IxDspCodeletParmDesc parmDesc[] =
{
    {0,                                         NULL},
    {(sizeof pParmNameNet)/sizeof(char **),     pParmNameNet},
    {(sizeof pParmNameDec)/sizeof(char **),     pParmNameDec},
    {(sizeof pParmNameEnc)/sizeof(char **),     pParmNameEnc},
    {(sizeof pParmNameTg)/sizeof(char **),      pParmNameTg},
    {(sizeof pParmNameTd)/sizeof(char **),      pParmNameTd},
    {(sizeof pParmNamePly)/sizeof(char **),     pParmNamePly},
    {(sizeof pParmNameMix)/sizeof(char **),     pParmNameMix},
    {(sizeof pParmNameT38)/sizeof(char **),     pParmNameT38},
    {(sizeof pParmNameMa)/sizeof(char **),      pParmNameMa}
};

/****************************************
 * fill the T-Port base informtion to
 * the parameter description text
 ****************************************/
void ixDspCodletFillBaseStrmInfo(void)
{
    static char p[MAX_STR_LEN], pp[MAX_COMPONENTS][MAX_STR_LEN];
    static int i=0;

    if(i)
    {
      return;
    }

    snprintf(p, MAX_STR_LEN, "L-Port Stream ID (T-Port Base -\n\t\t\t"
               "                  TDM:%d, IP:%d, Player:%d, Mixer:%d)",
           ixDspCodeletResCfg.streamBaseTDM, ixDspCodeletResCfg.streamBaseIP,
           ixDspCodeletResCfg.streamBasePly, ixDspCodeletResCfg.streamBaseMix);

    pParmNameNet[1] = p;
    pParmNameEnc[1] = p;
    pParmNameTd[1] = p;

    for(i=0; i<MAX_COMPONENTS; i++)
    {
      snprintf(pp[i],MAX_STR_LEN,"L-Port[%d] Stream ID (T-Port Base -\n\t\t\t"
                       "                    TDM:%d,IP:%d, Player:%d,Mixer:%d)",
           i+1, ixDspCodeletResCfg.streamBaseTDM,
           ixDspCodeletResCfg.streamBaseIP,
           ixDspCodeletResCfg.streamBasePly, ixDspCodeletResCfg.streamBaseMix);

      pParmNameMix[i+1] = pp[i];
    }
}

/****************************************
 * convert resource type to text
 ****************************************/
char *ixDspCodeletResId2Text(UINT8 res)
{
    static char *pText;

    pText = "UNKNOWN";

    if(res < numRes)
    {
        pText = pResName[res];
    }

    return pText;
}

/****************************************
 * convert stop reason code to text
 ****************************************/
char *ixDspCodeletRsnId2Text(int rsn)
{
    static char *pText;

    if(rsn == XMSG_STOP_REASON_USER)
    {
        pText = "USR";
    }
    else if(rsn == XMSG_STOP_REASON_EOD)
    {
        pText = "EOD";
    }
    else if(rsn == XMSG_STOP_REASON_TERM)
    {
        pText = "TERM";
    }
    else if(rsn == XMSG_STOP_REASON_TIMEOUT)
    {
        pText = "TIMEOUT";
    }
    else
    {
        pText = "UNKNOWN";
    }

    return pText;
}

/****************************************
 * convert event code to text
 ****************************************/
char *ixDspCodeletEvtCode2Text(int evtCode)
{
    static char *pText;


    if(evtCode == XEVT_CODE_TD_TONEON)
    {
        pText = "Tone_On";
    }
    else if(evtCode == XEVT_CODE_TD_TONEOFF)
    {
        pText = "Tone_Off";
    }
    else if(evtCode == XEVT_LOST_PACKET)
    {
        pText = "Packet Lost";
    }
    else if(evtCode == XEVT_DEC_PACKET_CHNG)
    {
        pText = "Packet Type Changed";
    }
    else if(evtCode == XEVT_NET_HOOK_STATE)
    {
        pText = "Hook State";
    }
    else if(evtCode == XEVT_NET_TIMER)
    {
        pText = "Timer Expired";
    }
    else{
        pText = "Unknown event";
    }

    return pText;
}

/****************************************
 * get parameter definition
 ****************************************/
char *ixDspCodeletParmId2Text(UINT8 resType, UINT16 id)
{
    static char *pText;
    IxDspCodeletParmDesc *pParmDesc;

    pText = "Unknown parameter";

    if(resType < numRes)
    {
        pParmDesc = parmDesc + resType;

        if(id < pParmDesc->numParms)
        {
            pText = pParmDesc->ppParmNames[id];
        }
    }

    return pText;
}

/**************************
 * print ack message
 **************************/
void ixDspCodeletPrtMsgAck(XMsgRef_t pMsg, UINT8 chl)
{
    printf("<<MsgAck received (ch=%d res=%s)>>\n",
           chl, ixDspCodeletResId2Text(pMsg->resource));
}


/*********************************
 * print get all parmeters ack message
 *********************************/
void ixDspCodeletPrtMsgGetAllParmsAck(XMsgRef_t pMsg, UINT8 chl)
{
    int i;
    int n;
    UINT16 *pIDs;
    INT16  *pVals;
    XParmUnion_t tval;

    XMSG_FIELD_GET_ALLPARMS_ACK(pMsg, n, pIDs, pVals);

    printf("\n<<MsgGetAllParmsAck received (ch=%d res=%s)>>\n"
           " id\tvalue\t\tdefinition\n"
           "--------------------------------------------------------------\n",
           chl, ixDspCodeletResId2Text(pMsg->resource));

    for(i = 0; i < n; i++)
    {
      if(pVals != NULL) {
        tval.ui16 = pVals[i];
      }
      else {
        assert(0);
      }

      if(pIDs != NULL) {
        printf("%3d\t%d\t\t%s\n", pIDs[i], tval.i16,
               ixDspCodeletParmId2Text(pMsg->resource, pIDs[i]));
      }
      else {
        assert(0);
      }
    }
}

/**************************
 * print error message
 **************************/
void ixDspCodeletPrtMsgError(XMsgRef_t pMsg, UINT8 chl)
{
    int errCode;
    int errData1;
    int errData2;

    XMSG_FIELD_ERROR(pMsg, errCode, errData1, errData2);

    printf("<<MsgError received (ch=%d res=%s c=0x%x d1=0x%x d2=0x%x)>>\n",
           chl, ixDspCodeletResId2Text(pMsg->resource),
           errCode, errData1, errData2);
}

/**************************
 * print event message
 **************************/
void ixDspCodeletPrtMsgEvent(XMsgRef_t pMsg, UINT8 chl)
{
    int code;
    int data1;
    int data2;

    XMSG_FIELD_EVENT(pMsg, code, data1, data2);

    printf("<<MsgEvent received (%s : ch=%d res=%s d1=0x%x d2=0x%x)>>\n",
           ixDspCodeletEvtCode2Text(code), chl,
           ixDspCodeletResId2Text(pMsg->resource),
           data1, data2);
}

/********************************
 * print codec start ack message
 ********************************/
void ixDspCodeletPrtMsgCoderStopAck(XMsgRef_t pMsg, UINT8 chl)
{
    int frames;
    int bad_frames;

    XMSG_FIELD_CODEC_STOP_ACK(pMsg, frames, bad_frames);

    printf("<<MsgCodecStopAck received (ch=%d res=%s total_frames=%d"
           " bad_frames=%d)>>\n", chl, ixDspCodeletResId2Text(pMsg->resource),
           frames, bad_frames);
}

/********************************
 * print TG play complete message
 ********************************/
void ixDspCodeletPrtMsgTgPlayCmplt(XMsgRef_t pMsg, UINT8 chl)
{
    int reason;
    int num;

    XMSG_FIELD_TG_PLAY_CMPLT(pMsg, reason, num);

    printf("<<MsgTgPlayCmplt received (ch=%d reason=%s tones played=%d)>>\n",
           chl, ixDspCodeletRsnId2Text(reason), num);
}

/***********************************
 * print TD receive complete message
 ***********************************/
void ixDspCodeletPrtMsgTdRcvCmplt(XMsgRef_t pMsg, UINT8 chl)
{
    int reason;
    int num;
    int i;
    UINT8  *digits;
    char digStr[XMAX_DIGITBUFSIZE + 1];

    XMSG_FIELD_TD_RCV_CMPLT(pMsg, reason, num, digits)

    printf("<<MsgTdRcvCmplt received (ch=%d reason=%s numDigits=%d)>> \n",
            chl, ixDspCodeletRsnId2Text(reason), num);

    for (i = 0; i < num; i++)
    {
       if(digits != NULL) {
          digStr[i] = digits[i] < DTMF_TONE_10 ? digits[i] + '0' :
                    digits[i] == DTMF_TONE_10 ? '*' :
                    digits[i] == DTMF_TONE_11 ? '#' : 
                    digits[i] + 'a' - DTMF_TONE_12;
       }
    }

    digStr[i] = 0;
    printf("\tdigits rcvd = '%s'\n", digStr);
}

/***********************************
 * print FSK receive complete message
 ***********************************/
void ixDspCodeletPrtMsgTdFSKRcvCmplt(XMsgRef_t pMsg, UINT8 chl)
{
    int reason;
    int num;
    int i;
    UINT8 *bytes;

    XMSG_FIELD_TD_RCV_FSK_CMPLT(pMsg, reason, num, bytes)

    printf("<<MsgTdFSKRcvCmplt received (ch=%d reason=%s numBytes=%d)>>\n",
            chl, ixDspCodeletRsnId2Text(reason), num);

    if(num)
    {
        printf("Received data (in Hex):\n");
        for(i=0;i<num;i++)
        {
          if(bytes != NULL) {
              printf("%02x ", (unsigned int)bytes[i]);
          }
        }
        printf("\n");
    }
}

/********************************
 * print PLY play complete message
 ********************************/
void ixDspCodeletPrtMsgPlyPlayCmplt(XMsgRef_t pMsg, UINT8 chl)
{
    int reason;

    XMSG_FIELD_PLY_CMPLT(pMsg, reason);

    printf("<<MsgPlayCmplt received (ch=%d reason=%s)>>\n",
           chl, ixDspCodeletRsnId2Text(reason));
}

/********************************
 * print DEC get JBstat complete message
 ********************************/
void ixDspCodeletPrtMsgGetJBStatCmplt(XMsgRef_t pMsg, UINT8 chl)
{
    XJBStatistics_t *pStat;

    XMSG_FIELD_GET_JBSTAT_CMPLT(pMsg, pStat);

    printf("<<MsgGetJBStatCmplt received (ch=%d)>>\n"
           "\tTotal received packets   : %u\n"
           "\tLost packets via nework  : %u\n"
           "\tDecoder's erasure frames : %u\n"
           "\tReceived Tone packets    : %u\n",
           chl, pStat->rcvdPackets,
           pStat->lostPackets,
           pStat->badFrames,
           pStat->rcvdTonePackets);
}

/********************************
 * print T38 complete message
 ********************************/
void ixDspCodeletPrtMsgT38Cmplt(XMsgRef_t pMsg, UINT8 chl)
{
    int reason;

    XMSG_FIELD_T38_CMPLT(pMsg, reason);

    printf("<<MsgT38Cmplt received (ch=%d reason=%s)>>\n",
           chl, ixDspCodeletRsnId2Text(reason));
}


/***********************************
 * print user Ack message
 ***********************************/
void ixDspCodeletPrtMsgUsrAck(XMsgRef_t pMsg, UINT8 chl)
{
    int i;
    IxDspCodeletMsgAck *pUsrAck;
    IxDspCodeletError *pErr;
    static char *msgText[] =
    {
        "IxDspCodeletMsgAck",
        "IxDspCodeletMsgLinkAck",
        "IxDspCodeletMsgSetupAck",
        "IxDspCodeletMsg3WAck",
        "IxDspCodeletMsgT38Ack"

    };

    pUsrAck = (IxDspCodeletMsgAck *)pMsg;
    pErr = pUsrAck->error;

    printf("<<%s received (ch=%d, numDspReply=%d, numErr=%d)>> \n",
            msgText[pMsg->type - IX_DSP_CODELET_MSG_ACK],
            chl, pUsrAck->numDspReplies, pUsrAck->numErrors);

    for (i = 0; i < pUsrAck->numErrors; i++, pErr++)
    {
        printf("\tError[%d] - %s[%d], code=0x%x, data=%d\n",
                i+1, ixDspCodeletResId2Text(pErr->dspResource),
                pErr->dspResInstance, pErr->errCode, pErr->errData);
    }
}

/***********************************
 * print user Stop Ack message
 ***********************************/
void ixDspCodeletPrtMsgUsrStopAck(XMsgRef_t pMsg, UINT8 chl)
{
    int i;
    IxDspCodeletMsgStopAck *pUsrAck;
    IxDspCodeletError *pErr;

    pUsrAck = (IxDspCodeletMsgStopAck *)pMsg;
    pErr = pUsrAck->error;

    printf("<<IxDspCodeletMsgStopAck received (ch=%d, numDspReply=%d,"
           " numErr=%d)>> \n", chl, pUsrAck->numDspReplies,
                                        pUsrAck->numErrors);

    for (i = 0; i < pUsrAck->numErrors; i++, pErr++)
    {
        printf("\tError[%d] - %s[%d], code=0x%x, data=%d\n",
                i+1, ixDspCodeletResId2Text(pErr->dspResource),
                pErr->dspResInstance, pErr->errCode, pErr->errData);
    }
}


/***********************************
 * print message
 ***********************************/
int ixDspCodeletPrtMsg(XMsgRef_t pMsg)
{
    int chl;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);

    switch (pMsg->type)
    {
    case XMSG_GET_ALLPARMS_ACK:
        ixDspCodeletPrtMsgGetAllParmsAck(pMsg, chl);
        break;

    case XMSG_ACK:
        ixDspCodeletPrtMsgAck(pMsg, chl);
        break;

    case XMSG_ERROR:
        ixDspCodeletPrtMsgError(pMsg, chl);
        break;

    case XMSG_EVENT:
        ixDspCodeletPrtMsgEvent(pMsg, chl);
        break;

    case XMSG_CODER_STOP_ACK:
        ixDspCodeletPrtMsgCoderStopAck(pMsg, chl);
        break;

    case XMSG_TG_PLAY_CMPLT:
        ixDspCodeletPrtMsgTgPlayCmplt(pMsg, chl);
        break;

    case XMSG_TD_RCV_CMPLT:
        ixDspCodeletPrtMsgTdRcvCmplt(pMsg, chl);
        break;

    case XMSG_TD_RCV_FSK_CMPLT:
        ixDspCodeletPrtMsgTdFSKRcvCmplt(pMsg, chl);
        break;

    case XMSG_PLY_CMPLT:
        ixDspCodeletPrtMsgPlyPlayCmplt(pMsg, chl);
        break;

    case XMSG_GET_JBSTAT_CMPLT:
        ixDspCodeletPrtMsgGetJBStatCmplt(pMsg, chl);
        break;

    case XMSG_T38_CMPLT:
        ixDspCodeletPrtMsgT38Cmplt(pMsg, chl);
        break;

    default:

       if((pMsg->type == IX_DSP_CODELET_MSG_ACK) || 
           (pMsg->type == IX_DSP_CODELET_MSG_LINK_ACK) || 
           (pMsg->type == IX_DSP_CODELET_MSG_SETUP_ACK) || 
              (pMsg->type == IX_DSP_CODELET_MSG_3W_ACK) ||
      (pMsg->type == IX_DSP_CODELET_MSG_T38_ACK))
       {
           ixDspCodeletPrtMsgUsrAck(pMsg, chl);
       }
       else if (pMsg->type == IX_DSP_CODELET_MSG_STOP_ACK)          
       {
           ixDspCodeletPrtMsgUsrStopAck(pMsg, chl);
       }
       else 
       {
           printf("<<Unknown message received (type = 0x%x, ch=%d)>>\n",
               pMsg->type, chl);
       }
       break;
    }

    return IX_SUCCESS;
}

/***********************************
 * print resource parameters
 ***********************************/
void ixDspCodeletPrtResParms(int res, int inst)
{
    int rc;
    UINT16 parmId;
    INT16 parmVal;
    XParmUnion_t tval;

    printf("\n DSP resource %s[%d] parameters\n"
           " id\tvalue\t\tdefinition\n"
           "--------------------------------------------------------------\n",
               ixDspCodeletResId2Text(res), inst);

    for(parmId = 0; ; parmId++)
    {
        rc = xDspParmRead(res, inst, parmId, &parmVal);
        if(rc != XSUCC)
        {
          break;
        }

        tval.ui16 = parmVal;

        printf("%3d\t%d\t\t%s\n",
               parmId, tval.i16, ixDspCodeletParmId2Text(res, parmId));
    }
}
