/******************************************************************************
*       @file   IxDspCodeletPlay.c
*
* Contents: This file contains functions for the Player demo.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the user defined include files required.
 */

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define WAVCODE_G711_ULAW   0x0007
#define WAVCODE_G711_ALAW   0x0006
#define WAVCODE_G729A       0x0134
#define MAX_PATH_LENGTH     256

#define PLAYER_OFFSET_DIAL_NUM    0
#define PLAYER_LENGTH_DIAL_NUM    160
#define MAX_NUM_DIGIT_REC         10
#define TOTAL_TIMEOUT             2000
#define FIRST_DIGIT_TIMEOUT       400
#define INTER_DIGIT_TIMEOUT       250
#define PLAYER_OFFSET_INVAL_NUM   3400
#define PLAYER_LENGTH_INVAL_NUM   197
#define PLAYER_OFFSET_DIALED_NUM  1600
#define PLAYER_LENGTH_DIALED_NUM  178

#define MAX_PLAYER_NUMBER_SIZE  20
#define MAX_INPUT_STR_INT_SIZE  5
#define MAX_AUDIO_FMT_INP_STR   3
#define SHIFT_8  8
#define SAMPLING_RATE_8K  8000
#define NEXT_VOICE_PROMPT_CHUNK_OFFSET  8
#define NUM_PLAYER_PARAMETERS  2

#define READ_INT(i, str, fp) \
{ \
    fgets(str, 5, fp); \
    i = str[0] + (str[1]<<8) + (str[2]<<16) + (str[3]<<24); \
}

#define IX_DSP_CODELET_SND_MSG(pMsg) \
    {if(xMsgSend(pMsg) != XSUCC) {printf("%s\n", msgSendErrStr);}}

/* states of Japan caller ID state machine */
typedef enum
{
    PLY_ST_IDLE = 0,
    PLY_ST_START,
    PLY_ST_GET_NUMBER,
    PLY_ST_PLAY_NUMBER,
    PLY_ST_EXIT
} IxDspCodeletStatePlayer;

/* channel data object */
static struct{
    int         state;
    char        number[MAX_PLAYER_NUMBER_SIZE];
} player[IX_DSP_CODELET_MAX_CHL];

static int msgBuf[XMSG_MAX_WSIZE];
static char *msgSendErrStr = "PLAY DEMO: ERROR - xMsgSend() fails";

#define IX_DSP_CODELET_VOICE_PROMPT     0
#define IX_DSP_CODELET_VOICE_DIGITS     1
#define IX_DSP_UNKNOWN_MSG              -1

static struct {
   int offset;
   int length;
} digitIndex[] = {
    {   0, 85},    /* zero */
    { 860, 56},    /* one */
    {1420, 51},    /* two */
    {1910, 57},    /* three */
    {2500, 60},    /* four */
    {3080, 66},    /* five */
    {3740, 91},    /* six */
    {4660, 76},    /* seven */
    {5410, 57},    /* eight */
    {6000, 81}     /* nine */
};

/********************************
 * load a voice prompt from file
 ********************************/
XMediaHandle_t ixDspCodeletLoadVoicePrompt(char *fname)
{
    FILE *fp;
    int n, size, headerRead;
    char sString[IX_DSP_CODELET_STRLEN+MAX_PATH_LENGTH];
    char iString[MAX_INPUT_STR_INT_SIZE];
    char pwd_sString[MAX_PATH_LENGTH];

    unsigned char *pMem;
    XMediaHandle_t promptHandle;
    XCachePromptDesc_t desc;

    /* get default device path */
    if (!getcwd(pwd_sString, MAX_PATH_LENGTH))
    {
        printf("Unable to get current path\n");
        return (XMEDIA_HANDLE_NULL);
    }

    /* now copy current working dir and append file name */
    strncpy(sString, pwd_sString, MAX_PATH_LENGTH);
    strncat(sString, fname, IX_DSP_CODELET_STRLEN);

    /* open the file as read binary */
    fp = fopen(sString,"rb");

    if (!fp)
    {
        printf("unable to open file\n");
        return (XMEDIA_HANDLE_NULL);
    }
    else
    {
        /* wave file reader, only in uLaw, aLaw, or G.729A formats */
        fgets(sString, MAX_INPUT_STR_INT_SIZE, fp);
        if (strncmp(sString, "RIFF", (MAX_INPUT_STR_INT_SIZE-1)) )
        {
            printf("Unsupported format: not RIFF\n");
            fclose(fp);
            return (XMEDIA_HANDLE_NULL);
        }
        READ_INT(size, iString, fp)
        fgets(sString, MAX_INPUT_STR_INT_SIZE, fp);
        if (strncmp(sString, "WAVE", (MAX_INPUT_STR_INT_SIZE-1)) )
        {
            printf("Unsupported format: not WAVE\n");
            fclose(fp);
            return (XMEDIA_HANDLE_NULL);
        }
        headerRead = 1;
        while (headerRead)
        {
            /* read the 4-byte chunk ID */
            fgets(sString, MAX_INPUT_STR_INT_SIZE, fp);
            /* read the 4-byte chunk size */
            READ_INT(size, iString, fp)
            if (!strncmp(sString, "fmt ", (MAX_INPUT_STR_INT_SIZE-1)) )
            {
                /* read the 2-byte Audio Format */
                fgets(sString, MAX_AUDIO_FMT_INP_STR, fp);
                n = (sString[1]<<SHIFT_8) + sString[0];
                switch (n)
                {
                    case WAVCODE_G711_ULAW:
                        desc.type = XCODER_TYPE_G711MU_10MS;
                        break;
                    case WAVCODE_G711_ALAW:
                        desc.type = XCODER_TYPE_G711A_10MS;
                        break;
                    case WAVCODE_G729A:
                        desc.type = XCODER_TYPE_G729A;
                        break;
                    default:
                        printf("Unsupported format: coder type\n");
                        fclose(fp);
                        return (XMEDIA_HANDLE_NULL);
                }
                /* read the 2-byte NumChannels */
                fgets(sString, MAX_AUDIO_FMT_INP_STR, fp);
                if (sString[0] != 0x01)
                {
                    printf("Unsupported format: not mono data\n");
                    fclose(fp);
                    return (XMEDIA_HANDLE_NULL);
                }
                /* read the 4-byte Sampling Rate */
                READ_INT(n, iString, fp)
                if (n != SAMPLING_RATE_8K)
                {
                    printf("Unsupported format: not 8KHz sampling\n");
                    fclose(fp);
                    return (XMEDIA_HANDLE_NULL);
                }
                /* advance file pointer to next chunk */
                fseek(fp, (size-NEXT_VOICE_PROMPT_CHUNK_OFFSET), SEEK_CUR);
            }
            else if (!strncmp(sString, "data", (MAX_INPUT_STR_INT_SIZE-1)) )
            {
                /* header reading is complete, data area starts */
                headerRead = 0;
            }
            else
            {
                /* skip over all other chunks */
                fseek(fp, size, SEEK_CUR);
            }
        }
        /* allocate the memory for the data */
        pMem = (unsigned char *)(malloc(size));
        if (!pMem)
        {
            printf("unable to allocate space\n");
            fclose(fp);
            return (XMEDIA_HANDLE_NULL);
        }
        /* read in the data */
        for (n = 0; n < size; n++)
        {
            pMem[n] = (unsigned char)(fgetc(fp));
        }

        /* register with DSP */
        desc.pBuffer = pMem;
        desc.size = size;
        promptHandle = xDspRegCachePrompt(&desc);

        if(promptHandle == XMEDIA_HANDLE_NULL)
        {
            printf("unable to register cache prompt\n");
            free(pMem);

        }
        fclose(fp);
    }
    return (promptHandle);
}

/********************************
 * load a raw data file into
 * provided buffer with specified
 * maximum size; returns size read
 ********************************/
int ixDspCodeletLoadRawData(char *fname, unsigned char *pMem, int size)
{
    int n;
    FILE *fp;
    char sString[IX_DSP_CODELET_STRLEN+MAX_PATH_LENGTH];
    char pwd_sString[MAX_PATH_LENGTH];

    /* get default device path */
    if (!getcwd(pwd_sString, MAX_PATH_LENGTH))
    {
        printf("Unable to get current path\n");
        return (0);
    }

    /* now copy current working dir and append file name */
    strncpy(sString, pwd_sString, MAX_PATH_LENGTH);
    strncat(sString, fname, IX_DSP_CODELET_STRLEN);

    /* open the file as read binary */
    fp = fopen(sString,"rb");

    n = 0;
    if (fp)
    {
        /* read in the data */
        while (!feof(fp) && (n < size))
        {
            pMem[n++] = (unsigned char)(fgetc(fp));
        }
        fclose(fp);
    }
    return (n);
}

void ixDspCodeletPlayInit(int chl)
{
    UINT32 trans;
    IxDspCodeletMsgSetParms *pSetParms;

    pSetParms = (IxDspCodeletMsgSetParms *)msgBuf;
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_PLAY, chl);

    IX_DSP_CODELET_MAKE_MSGHDR_SET_PARMS(pSetParms, trans)

    pSetParms->numParms = NUM_PLAYER_PARAMETERS;
    pSetParms->parms[0].parmID = XPARMID_NET_LP_STREAM;
    pSetParms->parms[0].value = ixDspCodeletResCfg.streamBasePly + chl - 1;
    pSetParms->parms[0].dspResource = XMPR_NET;
    pSetParms->parms[0].dspResInstance = chl;
    pSetParms->parms[1].parmID = XPARMID_TD_RPT_EVENTS;
    pSetParms->parms[1].value = XPARM_OFF;
    pSetParms->parms[1].dspResource = XMPR_TNDET;
    pSetParms->parms[1].dspResInstance = chl;

    IX_DSP_CODELET_SND_MSG(pSetParms)
    player[chl-1].state = PLY_ST_START;
}

/************************************************
        State machine for player demo
*************************************************/
int ixDspCodeletPlayerSm(XMsgRef_t pMsg)
{
    int chl;
    int error = 0;
    UINT32 trans;
    XPlyMediaDesc_t *pMedSeg;

    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);
    
    if( (chl>=0)||(chl<IX_DSP_CODELET_MAX_CHL))
    {
        pMsg->type = XMSG_ERROR;
   printf("Invalid channel number, Not supported");
   ixDspCodeletPrtMsg(pMsg);
   error = IX_DSP_UNKNOWN_MSG;
   return error; 
    }
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_PLAY, chl);

    switch(player[chl-1].state)
    {
    case PLY_ST_START:
        if(pMsg->type == IX_DSP_CODELET_MSG_ACK)
        {
            /* start tone detector */
            XMSG_MAKE_START(msgBuf, trans, XMPR_TNDET, chl)
            IX_DSP_CODELET_SND_MSG(msgBuf)
       }
       else {
            if((pMsg->type == XMSG_ACK) || (pMsg->type == XMSG_ERROR)
                              || (pMsg->type == XMSG_PLY_CMPLT))
            {
                /* play prompt of dial number */
                XMSG_FIELD_PLY_START(msgBuf, pMedSeg)
                pMedSeg[0].handle = IX_DSP_CODELET_VOICE_PROMPT;
                pMedSeg[0].offset = PLAYER_OFFSET_DIAL_NUM;
                pMedSeg[0].length = PLAYER_LENGTH_DIAL_NUM;
                pMedSeg[0].next = XPLY_MEDIA_SEG_EOP;

                XMSG_MAKE_PLY_START(msgBuf, trans, chl, 1)
                IX_DSP_CODELET_SND_MSG(msgBuf)

                player[chl-1].state = PLY_ST_GET_NUMBER;
             }
             else
             {
                error = IX_DSP_UNKNOWN_MSG;
             }
        }
        break;

    case PLY_ST_GET_NUMBER:
        if(pMsg->type == XMSG_PLY_CMPLT)
        {
            /* receive digits */
            XMSG_MAKE_TD_RCV(msgBuf,
                          trans,
                          chl,
                          MAX_NUM_DIGIT_REC, /* number of digits to receive */
                          XTD_TERM_DIGIT_STAR|XTD_TERM_DIGIT_POUND,
                          TOTAL_TIMEOUT,         /* total time out 20 sec. */
                          FIRST_DIGIT_TIMEOUT,   /* first digit timeout */
                          INTER_DIGIT_TIMEOUT)   /* inter digit timeout */

            IX_DSP_CODELET_SND_MSG(msgBuf)

            player[chl-1].state = PLY_ST_PLAY_NUMBER;

        }
        else
        {
            error = IX_DSP_UNKNOWN_MSG;
        }
        break;

    case PLY_ST_PLAY_NUMBER:
        if(pMsg->type == XMSG_TD_RCV_CMPLT)
        {
            int reason, num, i;
            UINT8  *digits;

            XMSG_FIELD_TD_RCV_CMPLT(pMsg, reason, num, digits)

            if(reason == XMSG_STOP_REASON_TERM)
            {
              num--;
            }

            XMSG_FIELD_PLY_START(msgBuf, pMedSeg)

            if(!num)
            {
                /* play prompt of invalid number */
                pMedSeg[0].handle = IX_DSP_CODELET_VOICE_PROMPT;
                pMedSeg[0].offset = PLAYER_OFFSET_INVAL_NUM;
                pMedSeg[0].length = PLAYER_LENGTH_INVAL_NUM;
                pMedSeg[0].next = XPLY_MEDIA_SEG_EOP;

                player[chl-1].state = PLY_ST_EXIT;

            }
            else
            {
                /* play dialed number */
                pMedSeg[0].handle = IX_DSP_CODELET_VOICE_PROMPT;
                pMedSeg[0].offset = PLAYER_OFFSET_DIALED_NUM;
                pMedSeg[0].length = PLAYER_LENGTH_DIALED_NUM;
                pMedSeg[0].next = 1;

                for(i=0; i<num; i++)
                {
                    pMedSeg[i+1].handle = IX_DSP_CODELET_VOICE_DIGITS;
                    pMedSeg[i+1].offset = digitIndex[digits[i]].offset;
                    pMedSeg[i+1].length = digitIndex[digits[i]].length;
                    pMedSeg[i+1].next = 1;
                }

                pMedSeg[num].next = XPLY_MEDIA_SEG_EOP;

                player[chl-1].state = PLY_ST_START;
            }

            XMSG_MAKE_PLY_START(msgBuf, trans, chl, num+1)
            IX_DSP_CODELET_SND_MSG(msgBuf)

        }
        else
        {
            error = IX_DSP_UNKNOWN_MSG;
        }
        break;

    case PLY_ST_EXIT:
        if(pMsg->type == XMSG_PLY_CMPLT)
        {
            /* restore link */
            XMSG_MAKE_SET_PARM(msgBuf, trans, XMPR_NET, chl,
                               XPARMID_NET_LP_STREAM,
                               ixDspCodeletResCfg.streamBaseIP + chl - 1)

            IX_DSP_CODELET_SND_MSG(msgBuf)
        }
        else if(pMsg->type == XMSG_ACK)
        {
            printf("End of player demo\n");
            player[chl-1].state = PLY_ST_IDLE;
        }
        break;
    default:
    ; /* No default */
    }

    if(error == IX_DSP_UNKNOWN_MSG)
    {
        ixDspCodeletPrtMsg(pMsg);
    }

    return error;
}

