/*****************************************************************************
*       @file   IxDspCodeletPulseDial.c
*
* Contents: This file contains functions for Pulse Dialling
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
*****************************************************************************/

#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"

#define CHAN_DATA_OBJ_SIZE  20

static char *msgSendErrStr = "PULSE DIAL DEMO: ERROR - xMsgSend() fails";

#define IX_DSP_CODELET_SND_MSG(pMsg) \
    {if(xMsgSend(pMsg) != XSUCC) { printf("%s\n", msgSendErrStr);}}
    
#define  XMSG_PULSEDIAL_ERROR  10
#define  MAX_PULSE_COUNT    99
#define  MAX_DECADIC_PULSES    10
#define  SEC_10MS_FACTOR       100
#define  FACTOR_10             10

/* Number of digits expected */
static int exp_num_digits;

/* Inter Digit Timeout value in secs */
static int inter_digit_timeout_value;

static int inter_digit_timeout_counter;

static int current_flash_hook_pulse_length;
static int expected_flash_hook_pulse_length;

/* message buf */
static int msgBuf[XMSG_MAX_WSIZE];

/* states of Pulse Dialling state machine */
typedef enum
{
    PULSEDIAL_ST_IDLE =0,              /* Idle state */
    PULSEDIAL_ST_DIALTONE,             /* Dial Tone */
    PULSEDIAL_ST_BREAK,                /* Pulse Break */
    PULSEDIAL_ST_MAKE,                 /* Pulse Make */
    PULSEDIAL_ST_NXT_DIGIT            /* Next Digit */
} IxDspCodeletStatePulseDial;

/* Dial tone state */
typedef enum
{
  DIALTONE_NOT_PROVIDED =0,    /* Dial tone not provided */
  DIALTONE_PROVIDED            /* Dial tone provided */
}dialtonestate;

/* channel data object */
static struct{
    int         state;
    char        number[CHAN_DATA_OBJ_SIZE];
} pulse_dial[IX_DSP_CODELET_MAX_CHL];

static int dialtone_provided = DIALTONE_NOT_PROVIDED;
static int previous_state;
static int digits[XMAX_DIGITBUFSIZE];
static int pulse_count = 0;
static int digit_count = 0;
static int flaskhook_cnt_reset = 1;
static UINT16 parmVal;

#define DIAL_TONE_TIMER 3000  /* Dial tone timer value - 10sec */
#define BREAK_TIMER 9        /* Break Timer value - 33 msec + 47msec latency */
#define MAKE_TIMER  13        /* Make Timer value - 66 msec + 45msec latency */

void pulsedialPrintDigits(void);
void pulsedialResetFlashhookWindow(int channel, UINT32 transid);
/************************************
 * Pulse Dial state machine
 ************************************/
int ixDspCodeletPulseDialSm(XMsgRef_t pMsg)
{

    int chl;
    int error = 0;
    int evtCode;
    int evtData1;
    int evtData2;
    UINT32 trans;
    UINT8  *pTones;
    XStatus_t rc;
    chl = IX_DSP_CODELET_TRANS_GET_CHAN(pMsg);
    
    if ((chl<=0)||(chl>IX_DSP_CODELET_MAX_CHL))
    {
        pMsg->type = XMSG_ERROR;
    printf("Invalid channel number, Not supported");
    ixDspCodeletPrtMsg(pMsg);
    error = XMSG_PULSEDIAL_ERROR;
    return error;
    }
    trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_PULSE_DIAL, chl);
    if(flaskhook_cnt_reset == 1)
     { 
       rc = xDspParmRead(XMPR_NET, chl, XPARMID_NET_FLASH_HK, &parmVal);
       if(rc != XSUCC) 
         {
           printf("\n Unable to retrieve Flash hook Window Size"); 
           /* Go to MAIN MENU */
           ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_TEST);
           ixDspCodeletPrtMsg(pMsg);
           return error;
         }
      
       rc = xDspParmWrite(XMPR_NET, chl, XPARMID_NET_FLASH_HK, 0, trans);
       if(rc != XSUCC)
         {
           printf("\n Unable to Set Flash hook Window Size");
           /* Go to MAIN MENU */
           ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_TEST);
           ixDspCodeletPrtMsg(pMsg);
           return error;
         }
       flaskhook_cnt_reset = 0;

     }
    switch(pulse_dial[chl-1].state)
    {
     case PULSEDIAL_ST_IDLE:
         if (pMsg->type == XMSG_EVENT)   /* Event received */
         {
              XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
              if( (evtCode == XEVT_NET_HOOK_STATE) && 
                  (evtData1 == XHOOK_STATE_OFF) && (!dialtone_provided))
                  /* Check for OFF HOOK */
                {
                  /* play dial tone */
                  XMSG_FIELD_TG_PLAY(msgBuf, pTones)
                  pTones[0] = RFC_TID_DIAL;
                  XMSG_MAKE_TG_PLAY(msgBuf, trans, chl, 1)
                  IX_DSP_CODELET_SND_MSG(msgBuf)
                  
                  /* set dialtone provided flag to TRUE */
                  dialtone_provided = DIALTONE_PROVIDED;

                  /* set dialtone timer to 10 secs */
                  IX_DSP_CODELET_SET_TIMER(trans, chl, DIAL_TONE_TIMER)

                  /*  set state to DIALTONE */
                  pulse_dial[chl-1].state = PULSEDIAL_ST_DIALTONE;
               }
               else 
               {
                  /* continue in IDLE state only */
                  pulse_dial[chl-1].state = PULSEDIAL_ST_IDLE;
                  /* save the current state */
                  previous_state = PULSEDIAL_ST_IDLE;
               }
         }
         else /* no event is received */
         {
             /* unexpected event, continue to be in IDLE state */
             pulse_dial[chl-1].state = PULSEDIAL_ST_IDLE;
             /* save the current state */
             previous_state = PULSEDIAL_ST_IDLE;
         }
         break;

     case PULSEDIAL_ST_DIALTONE:
        if (pMsg->type == XMSG_EVENT)
        {
            XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
            if (evtCode == XEVT_NET_TIMER) /* DIALTONE TIMER expired */
            {
                /* stop dial tone */
                if (dialtone_provided)
                {
                   XMSG_MAKE_STOP(msgBuf, trans, XMPR_TNGEN, chl);
                   IX_DSP_CODELET_SND_MSG(msgBuf)

                  /* reset dialtone timer */
                  IX_DSP_CODELET_SET_TIMER(trans, chl, 0)
                }
                /* No digits dialled after Dial tone, disconnect */
                printf("\n No digits are dialled after Dial Tone provided \n");
                printf("\n Please go ONHOOK and Press ENTER to "
                       "go to MAIN MENU\n");
               /* Reset Flash hook Window size */
               pulsedialResetFlashhookWindow(chl, trans);
                /* Go to MAIN MENU */
                ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_TEST);
                ixDspCodeletPrtMsg(pMsg);
            }
            else if ((evtCode == XEVT_NET_HOOK_STATE) && 
                    (evtData1 == XHOOK_STATE_ON))  /* break pulse received */
            {
                /* stop dial tone */
                if (dialtone_provided)
                {
                   XMSG_MAKE_STOP(msgBuf, trans, XMPR_TNGEN, chl);
                   IX_DSP_CODELET_SND_MSG(msgBuf)

                   /* reset dialtone timer */
                   IX_DSP_CODELET_SET_TIMER(trans, chl, 0)
                }

                /* set BREAK timer */
                IX_DSP_CODELET_SET_TIMER(trans, chl, BREAK_TIMER)
            
                /* go to BREAK state */
                pulse_dial[chl-1].state = PULSEDIAL_ST_BREAK;
               
                /* first pulse */
                /*  pulse_count = 1; */
             }
        /* save the current state */
        previous_state = PULSEDIAL_ST_DIALTONE;
        }
        break;

     case PULSEDIAL_ST_BREAK:
         if (pMsg->type == XMSG_EVENT)   /* Event received */
         {
            XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

            if (evtCode == XEVT_NET_TIMER) /* BREAK Timer expired */
            {
               /* Went ONHOOK from DIAL TONE, Disconnect */
               if (previous_state == PULSEDIAL_ST_DIALTONE)
                  /* timeout from dialtone state */
               {
                   /* reset Break timer */
                   IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

                   printf("\n Gone ON HOOK after the Dial tone\n");
                   printf("\n Please press ENTER to go to MAIN MENU\n");
                   /* Reset Flash hook Window size */
                   pulsedialResetFlashhookWindow(chl, trans);
                   ixDspCodeletSetHookEventCategory
                        (IX_DSP_CODELET_CATEGORY_TEST);
                   ixDspCodeletPrtMsg(pMsg);

               }
               else if (previous_state == PULSEDIAL_ST_MAKE)
               /* Not all the digits dialled */
               { 
                     /* reset Break timer */
                     IX_DSP_CODELET_SET_TIMER(trans, chl, 0)
                     /* Reset Flash hook Window size */
                     pulsedialResetFlashhookWindow(chl, trans);
                     /* Print the digits and go to MAIN MENU */
                      ixDspCodeletSetHookEventCategory
                            (IX_DSP_CODELET_CATEGORY_TEST);
                      pulsedialPrintDigits();
                      ixDspCodeletPrtMsg(pMsg);
               }
               else if ((previous_state == PULSEDIAL_ST_NXT_DIGIT) ||
                        (previous_state == PULSEDIAL_ST_BREAK)) 
                         /* Flash hook not detected */
               {
                    /* not a flash hook pulse, it is onhook, print digits
                       and go to main menu */
                    if (current_flash_hook_pulse_length >= 
                              expected_flash_hook_pulse_length)
                    {
                         /* set BREAK Timer */
                         IX_DSP_CODELET_SET_TIMER(trans, chl, 0)
                         /* Reset Flash hook Window size */
                         pulsedialResetFlashhookWindow(chl, trans);
                         ixDspCodeletSetHookEventCategory
                                    (IX_DSP_CODELET_CATEGORY_TEST);
                         pulsedialPrintDigits();
                         ixDspCodeletPrtMsg(pMsg);

                    }
                    else {
                       current_flash_hook_pulse_length += 
                                              (BREAK_TIMER * FACTOR_10);

                       /* set BREAK Timer */
                       IX_DSP_CODELET_SET_TIMER(trans, chl, BREAK_TIMER)
                       pulse_dial[chl-1].state = PULSEDIAL_ST_BREAK;

                       previous_state = PULSEDIAL_ST_BREAK; 

                    }
               }
            }
            else if ((evtCode == XEVT_NET_HOOK_STATE) &&
                     (evtData1 == XHOOK_STATE_OFF)) /* make pulse received */
            {
              switch(previous_state)
              {
               case PULSEDIAL_ST_MAKE:
                    /* Reset BREAK timer */
                    IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

                    /* increment the Pulse counter */
                    pulse_count+=1; 

                    /* set MAKE timer */
                    IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                    /* go to MAKE state */
                    pulse_dial[chl-1].state = PULSEDIAL_ST_MAKE; 

                    break;
               case PULSEDIAL_ST_NXT_DIGIT:
                    pulse_count = 1;   /* First pulse */

                    /* set MAKE timer */
                    IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                    /* go to MAKE state */
                    pulse_dial[chl-1].state = PULSEDIAL_ST_MAKE;

                    break;
               case PULSEDIAL_ST_DIALTONE:
                    /* first pulse */
                    pulse_count = 1; 

                    /* set MAKE timer */
                    IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                    /* go to MAKE state */
                    pulse_dial[chl-1].state = PULSEDIAL_ST_MAKE;
                    break;
               case PULSEDIAL_ST_BREAK:
                    if( (current_flash_hook_pulse_length > 
                                           (BREAK_TIMER * FACTOR_10))
                           && (current_flash_hook_pulse_length <= 
                                   expected_flash_hook_pulse_length))
                    { 
                        /* store Flash hook as 99 */
                        digits[digit_count] = MAX_PULSE_COUNT;

                        /* increment the digit count by 1 */
                        digit_count +=1;

                        /* go to NEXT DIGIT state */
                        pulse_dial[chl-1].state = PULSEDIAL_ST_NXT_DIGIT;
                        inter_digit_timeout_counter = 0;
                        current_flash_hook_pulse_length = 0;

                        /* reset BREAK  timer */
                        IX_DSP_CODELET_SET_TIMER(trans, chl, 0) 
                        /* set MAKE timer */
                        IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)
                    }
                    else 
                    if (current_flash_hook_pulse_length == 
                                              (BREAK_TIMER * FACTOR_10))
                    {
                      pulse_count = 1;   /* First pulse */

                      /* reset BREAK  timer */
                      IX_DSP_CODELET_SET_TIMER(trans, chl, 0) 

                      /* set MAKE timer */
                      IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                      /* go to MAKE state */
                      pulse_dial[chl-1].state = PULSEDIAL_ST_MAKE;
                    }
                    else
                    { 
                       /* Reset Flash hook Window size */
                       pulsedialResetFlashhookWindow(chl, trans); 
                        /* Print the digits and go to MAIN MENU */
                        ixDspCodeletSetHookEventCategory
                             (IX_DSP_CODELET_CATEGORY_TEST);
                        pulsedialPrintDigits();
                        ixDspCodeletPrtMsg(pMsg);
                    }
                    break;  
               default:
                    printf("\n Unknown state received in BREAK state\n");
              }
            }
        /* save the current state */
        previous_state = PULSEDIAL_ST_BREAK;
     }
     break;

     case PULSEDIAL_ST_MAKE:
         if (pMsg->type == XMSG_EVENT)     /* event received */
         {
             XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);

             if (evtCode == XEVT_NET_TIMER)   /* MAKE Timer expired */
             {
                if ((previous_state == PULSEDIAL_ST_BREAK) && (!pulse_count))
                {
                  pulse_count += 1;  /* digit 1 */
                }

                /* store the digit value */
                digits[digit_count] = pulse_count;
                
                /* increment the digit count by 1 */
                digit_count +=1;
 
                /* reset the pulse count */
                pulse_count = 0;

                /* increment the the inter-digit counter */
                inter_digit_timeout_counter += MAKE_TIMER;


                /*set MAKE Timer, this timer is used for INTER DIGIT timeout*/
                IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                pulse_dial[chl-1].state = PULSEDIAL_ST_NXT_DIGIT; 

             }
             else if ((evtCode == XEVT_NET_HOOK_STATE) && 
                    (evtData1 == XHOOK_STATE_ON)) /* break pulse received */
             { 
                /* reset MAKE timer */
                IX_DSP_CODELET_SET_TIMER(trans, chl, 0)

                /* set BREAK timer */
                IX_DSP_CODELET_SET_TIMER(trans, chl, BREAK_TIMER)

                /* reset the inter-digit counter */
                inter_digit_timeout_counter = 0;

                /* go to BREAK state */
                pulse_dial[chl-1].state = PULSEDIAL_ST_BREAK;
             }
        /* save the current state */
        previous_state = PULSEDIAL_ST_MAKE;
        }
        break;

     case PULSEDIAL_ST_NXT_DIGIT:
         /* not all the digits are received */
         if (digit_count < exp_num_digits) 
         {
            if (pMsg->type == XMSG_EVENT)  /* event received */
            {
              /* Extract Event code and Event data  */
              XMSG_FIELD_EVENT(pMsg, evtCode, evtData1, evtData2);
            
              if (evtCode == XEVT_NET_TIMER)   /* MAKE Timer expired */
              { 
                /* increment the the inter-digit counter */
                inter_digit_timeout_counter += MAKE_TIMER;

                /* if INTER-DIGIT timer expires, go to state REPORT-DIGITS */
                if (inter_digit_timeout_counter > inter_digit_timeout_value)
                {  
                   /* Reset Flash hook Window size */
                    pulsedialResetFlashhookWindow(chl, trans); 
                   /* Not all the digits dialled, print digits */
                    printf("\n Not all the digits dialled \n");
                    printf("\n Please go ONHOOK and Press ENTER"
                                         " to go to MAIN MENU\n");
                    ixDspCodeletSetHookEventCategory
                              (IX_DSP_CODELET_CATEGORY_TEST);
                    pulsedialPrintDigits();
                    ixDspCodeletPrtMsg(pMsg);

                }
                else /* be in the same state */
                {
                  /* set MAKE Timer, this timer is used for 
                                     INTER DIGIT timeout */
                  IX_DSP_CODELET_SET_TIMER(trans, chl, MAKE_TIMER)

                   pulse_dial[chl-1].state =  PULSEDIAL_ST_NXT_DIGIT;
                }
              }  
              else if ((evtCode == XEVT_NET_HOOK_STATE) && 
                 (evtData1 == XHOOK_STATE_ON)) /* break pulse received */
              {
                       /* set BREAK Timer */
                       IX_DSP_CODELET_SET_TIMER(trans, chl, BREAK_TIMER)
                       pulse_dial[chl-1].state =  PULSEDIAL_ST_BREAK;
              }
            } /* if (pMsg->type == .... */

            /* save the current state */
            previous_state = PULSEDIAL_ST_NXT_DIGIT;
         } /* if digit_count < .....*/
          
         else  /* if digit_count > exp_num_digits */
          {
               /* Reset Flash hook Window size */
           pulsedialResetFlashhookWindow(chl, trans);
           /* print digits  and go to MAIN MENU */
           ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_TEST);
           if (exp_num_digits != 0)
           {
             pulsedialPrintDigits();
           }
           ixDspCodeletPrtMsg(pMsg);

         }     
      break;

     default:
     {
         printf("\n UNKNOWN STATE"); 
         error = XMSG_PULSEDIAL_ERROR;
     }
  } /* end of switch */
     return error;
}  /* end of Pulse dial state machine */


/********************************/
/* Pulse Dial Init Function     */
/********************************/
void ixDspCodeletPulseDialInit(int chl, int numdigits, int intdigtimeout,
                                 int flash_hook_timeout) 
{
    IxDspCodeletMsgSetParms *pSetParms;

    pSetParms = (IxDspCodeletMsgSetParms *)msgBuf;


    dialtone_provided =  DIALTONE_NOT_PROVIDED;

    digit_count = 0;
    current_flash_hook_pulse_length = 0;
    inter_digit_timeout_counter = 0;

    exp_num_digits = numdigits;
 
    inter_digit_timeout_value = intdigtimeout * SEC_10MS_FACTOR;

    expected_flash_hook_pulse_length =  flash_hook_timeout;

    ixDspCodeletSetHookEventCategory(IX_DSP_CODELET_CATEGORY_PULSE_DIAL);

    pulse_dial[chl-1].state = PULSEDIAL_ST_IDLE;
    printf("\nNumber of expected digits -> %d", exp_num_digits);
    printf("\nInter Digit TImeout value-> %d", intdigtimeout);
    printf("\nChannel Number-> %d", chl);
    printf("\nFlash Hook Window Size (ms)-> %d", flash_hook_timeout);
}


/***********************************/
/*      Print the collected digits */
/***********************************/
void pulsedialPrintDigits(void)
{
      int i;

      /* Print the collected digits */
      printf("\n Dialled Digits = ");

      for (i = 0; i < digit_count ; i++)
        {
            if (digits[i] < MAX_DECADIC_PULSES)
             {
                printf("%d ", digits[i]);
             }
             else if (digits[i] == MAX_DECADIC_PULSES) 
             {
               printf("%d ", digits[i]-MAX_DECADIC_PULSES);
             }
             else if (digits[i] == MAX_PULSE_COUNT)
             {
               printf("F ");
             }
        }

        inter_digit_timeout_counter = 0;
        current_flash_hook_pulse_length = 0;
        previous_state = PULSEDIAL_ST_IDLE;

        printf("\n Please go ONHOOK and Press ENTER to go to MAIN MENU\n");
}

/***************************************************************
 *                            Reset the Flash Hook Window Size
 ****************************************************************/
void pulsedialResetFlashhookWindow(int channel, UINT32 transid)
{
     XStatus_t rc;
        /* Reset Flash hook Window size */
        rc = xDspParmWrite(XMPR_NET, channel, XPARMID_NET_FLASH_HK,
                                                  parmVal, transid);
        if(rc != XSUCC)
             {
               printf("\n Unable to ReSet Flash hook Window Size");
            }
        flaskhook_cnt_reset = 1;         
}
 
