/******************************************************************************
*       @file    IxDspCodeletSocket.c
*
* Contents: this file contains functions for UDP socket interface.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the system defined include files required.
 */

#include <stdio.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>

/*
 * Put the user defined include files required.
 */
#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"
#include "IxDspCodeletSocket.h"
#include "dspcis.h"

#include "FWTypes.h"
#include <pthread.h>

#include <errno.h>
#include <string.h>

#ifdef SRTP_ACCEL_FLAG
#define SRTP_HEADER_LEN  12
#define SRTP_BLOCK_LEN   16
#define PAD_BYTES_8      8
#endif

#define MAX_LEN                     120
#define RTP_VERSION                 2
#define FACTOR_2                    2
#define OFFSET_2                    2
#define HEADER_MASK_VALUE           0x7f
#define MAX_IP_ADDRESS_SIZE         20
#define PAYLOAD_SIZE_1_KB           1024
#define MAX_STRING_LENGTH_TASKNAME  20

#define DEFAULT_SOCKET_THREAD_PRIORITY_LINUX   20
#define DEFAULT_SOCKET_THREAD_PRIORITY_VXWORK  50

typedef pthread_t thread_T;
typedef void  *(*pthread_startroutine_t) (void *);
typedef void  *pthread_addr_t;
typedef void threadArg_T;
typedef void (*threadFunc_T)(void *);

#ifndef OS_VXWORK
thread_T Thread_Create_RT(int priority, threadFunc_T startFunc,
                          threadArg_T *arg);
#else
thread_T Thread_Create(int priority, threadFunc_T startFunc, threadArg_T *arg);
#endif

int Thread_Cancel(thread_T thread);

/*
 * The DSP library treats channel 0 as a wild card channel number used to
 * reference all of the channels. Valid DSP voice channels range from 1 to 32.
 * In this socket level code the valid voice channels range from 0 to 
 * IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS.
 */
#define DSP_LVL_CHAN_TO_SOCK_LVL_CHAN_MAP(x) (x-1)
#define SOCK_LVL_CHAN_TO_DSP_LVL_CHAN_MAP(x) (x+1)

#define INET_ADDR_LEN        4
#define MAX_IP_ADDR_LEN      15

#define MAX_CHANNEL_COMBINATION 50

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
    BOOL socketReceiveThreadCreated = FALSE;
    thread_T socketReceiveThreadId;
#endif

IxDspCodeletSocketStats 
ixDspCodeletSocketStats[IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS];

IxDspCodeletSocketChanConfigInfo 
ixDspCodeletSocketChanConfigInfo[IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS];

UINT8 ixDspCodeletSocketRxStorage0
[IX_DSP_CODELET_SOCKET_MAX_READ_FROM_SOCKET_SIZE + sizeof(XPacket_t) - 1];


BOOL ixDspCodeletSocketAtLeastOneChanelIsConfigured = FALSE;

#ifdef SRTP
rtp_receiver_t  rcvr[IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS];
rtp_sender_t snd[IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS];
#endif

/* 
* This structure and its initialization is used to configure loop-back socket connections
* between the channel combinations as initializd below.
* 
* User has to take care that no channel number should be repeated
* otherwise that combination will be skipped and next combination will be scanned.
* Also if the any of the channel numbers in a combination exceeds the
* limit(0 - MAX_SOCKET) it will be ignored and next comination will be processed.
* 
* Channel combination entry should end with a {0,0} entry.
* Also there should not be more than 50 combinations in the structure initialization 
* other wise rest all will be ignored.
* 
*/

/*
IX_DSP_CODELET_CHANNEL_
*/

typedef struct 
{
    UINT16 source;
    UINT16 destination;
}IxDspCodeletUserProfile;

IxDspCodeletUserProfile ixDspCodeletUserProfile[]=
{
    { IX_DSP_CODELET_CHANNEL_1, IX_DSP_CODELET_CHANNEL_2 },
    { IX_DSP_CODELET_CHANNEL_3, IX_DSP_CODELET_CHANNEL_8 },
    { IX_DSP_CODELET_CHANNEL_4, IX_DSP_CODELET_CHANNEL_6 },
    { IX_DSP_CODELET_CHANNEL_23, IX_DSP_CODELET_CHANNEL_24 },
    { 0, 0 }   /* "0,0" indicates that this channel entry is done */
};

/* Codelet Socket Menu Function Pointer Type */
typedef void (*IxDspCodeletSocketMenuFunc) (void);

typedef struct {
    char *name;                     /* Menu Display Name */
    IxDspCodeletSocketMenuFunc menuFunc;  /* Service routine */
} IxDspCodeletSocketMenu;

IxDspCodeletSession ixDspCodeletSession[IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS];


void ixDspCodeletSockMenuConfig(void);
void ixDspCodeletSockMenuDisable(void);
void ixDspCodeletSockMenuShowConfig(void);
void ixDspCodeletSockMenuShowStats(void);
void ixDspCodeletSockMenuInit(void);
void ixDspCodeletSockMenuDefaultConfig(void);
void ixDspCodeletSockMenuMaxDefaultConfig(void);
void ixDspCodeletSockMenuUserDefinedConfig(void);

IxDspCodeletSocketMenu ixDspCodeletSocketMenu[] =
{
/*          Display Name         Service Routine              */
    {"Print Menus",            NULL},
    {"Channel Configuration",    ixDspCodeletSockMenuConfig},
    {"Disable Channel",        ixDspCodeletSockMenuDisable},
    {"Show Channel Configuration",    ixDspCodeletSockMenuShowConfig},
    {"Show Channel Statistics",    ixDspCodeletSockMenuShowStats},
    {"Create Sockets",        ixDspCodeletSockMenuInit},
    {"Default Configuration",    ixDspCodeletSockMenuDefaultConfig},
    {"Max-Default Configuration",    ixDspCodeletSockMenuMaxDefaultConfig},
    {"User Defined Configuration",    ixDspCodeletSockMenuUserDefinedConfig},
    {"Exit",            NULL},
    {NULL,                NULL}
};

/****************************************************************************
 *
 * This routine can configure how the source and destination addresses map to
 * voice channel numbers. This will determine the addressing info that the
 * socket will be interested in listening to and creating.
 *
 ****************************************************************************/
IX_STATUS ixDspCodeletSocketChanConfig(unsigned voiceChannel,
                       UINT16 localUdpPort)
{
    /*
     * Validate input parameters
     */
    if (voiceChannel > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || 
                         voiceChannel<0)
    {
    printf("%s(): ERROR - voice channel must be <= %d and >0 \n",
           __FUNCTION__,
           IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS);
    return IX_FAIL;
    }

    if (ixDspCodeletSocketChanConfigInfo[voiceChannel].isConfigured == TRUE)
    {
    printf("%s(): ERROR - voice channel is already configured. "
           "You must disable the channel before reconfiguring\n",
           __FUNCTION__);
    return IX_FAIL;
    }    


    if ( (localUdpPort < IX_DSP_CODELET_SOCKET_UDP_PORT_BASE) || 
     (localUdpPort >= IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + 
      IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS) )
    {
    printf("%s(): ERROR - UDP port must be >= %d and < %d\n",
           __FUNCTION__,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + 
           IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS);
    }

    ixDspCodeletSocketChanConfigInfo[voiceChannel].localUdpPort = localUdpPort;
    ixDspCodeletSocketChanConfigInfo[voiceChannel].isConfigured = TRUE;

    /*
     *  remote IP and port are now set dynamically as calls are 
     *  established.  Set to zero initially to indicate no call in progress.
     */

    ixDspCodeletSocketChanConfigInfo[voiceChannel].remoteUdpPort = 0;
    ixDspCodeletSocketChanConfigInfo[voiceChannel].remoteIpAddr = 0;

    /*
     * Set a flag to help check that this function was called at least once before
     * initialisation
     */
    ixDspCodeletSocketAtLeastOneChanelIsConfigured = TRUE;
    return IX_SUCCESS;
}

/*****************************************************************************
 *
 * This routine can be used to disable a channel
 *
 *****************************************************************************/
void ixDspCodeletSocketChanDisable(int channel)
{
   /*
     * Validate input parameters
     */
    if (channel > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || channel<0)
    {
    printf("%s(): ERROR - voice channel must be <= %d and >0 \n",
           __FUNCTION__,
           IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS);
    return;
    }

 if (channel >= IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS)
    {
    printf("%s(): WARNING - trying to disable an invalid channel\n",
           __FUNCTION__);
    return;
    }
    
 if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured == FALSE)
    {
    printf("%s(): WARNING - channel #%d is not configured.\n",
           __FUNCTION__, channel);
    return;
    }

    ixDspCodeletSocketChanConfigInfo[channel].isConfigured = FALSE;

    if (ixDspCodeletSocketChanConfigInfo[channel].isActivated == TRUE)
    {
#ifdef SRTP
        srtp_dealloc(snd[channel].srtp_ctx);
        srtp_dealloc(rcvr[channel].srtp_ctx);
#endif

#ifndef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
        Thread_Cancel(ixDspCodeletSocketChanConfigInfo[channel].threadID);
#endif
        ixDspCodeletSocketChanConfigInfo[channel].isActivated = FALSE;
    }

    /* Dont try to free the constant Default IP Address. Only free the malloc'ed
    IP String 
    */
    if ( 0 != strncmp(DEFAULT_IXP0_IP,
           ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr,
           sizeof(DEFAULT_IXP0_IP))) {
 
       if( 0 != strncmp(DEFAULT_IXP1_IP,
           ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr,
           sizeof(DEFAULT_IXP1_IP))) {
 
           free(ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr);
       }
    }
    /* Close the Rx and Tx Socket Associated with the Channel and Cancel
       the Thread 
    */    

    if (ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter)
    {
      close(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter);
    }

    if (ixDspCodeletSocketChanConfigInfo[channel].txSocketDescripter)
    {
      close(ixDspCodeletSocketChanConfigInfo[channel].txSocketDescripter);
    }

    /* Mark the channel as not configured */
    ixDspCodeletSocketChanConfigInfo[channel].isConfigured = FALSE;

    return;
}

void ixDspCodeletSocketRemoteSet (int channel,
                                  char *ip,
                  UINT16 port)
{
    /*
     * Validate input parameters
     */
    if (channel > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || channel<0)
    {
    printf("%s(): ERROR - voice channel must be <= %d and >0 \n",
           __FUNCTION__,
           IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS);
    }

    if(!(ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr))
    {
        ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr =
            (char *)malloc(MAX_IP_ADDR_LEN);
    }

    ixDspCodeletSocketChanConfigInfo[channel].remoteUdpPort = port;
    strncpy(ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr, 
                                   (char*)ip, MAX_IP_ADDRESS_SIZE);
}

void ixDspCodeletSocketRemoteGet (int channel,
                                  unsigned *ip,
                  int *port)
{
    /*
     * Validate input parameters
     */
    if (channel > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || channel<0)
    {
    printf("%s(): ERROR - voice channel must be <= %d and >0 \n",
           __FUNCTION__,
           IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS);
    }

    *port = ixDspCodeletSocketChanConfigInfo[channel].remoteUdpPort;
    *ip = (unsigned)*(ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr);
}


/*****************************************************************************
 *
 * This routine can be used to display the configured channel info
 *
 *****************************************************************************/
void ixDspCodeletSocketChanConfigShow(void)
{
    unsigned channel ;
    struct in_addr localAddr;
    struct in_addr remoteAddr;

    bzero((char *)&localAddr,sizeof(localAddr));
    bzero((char *)&remoteAddr,sizeof(remoteAddr));
    printf("\n*** DSP Codelet Socket And Voice Channel Config Info ***\n\n");

    printf("Chan#:Cfg:Act:Loc Port:      Remote IP:Rem Port\n");
    printf("-----:---:---:--------:---------------:--------\n");

    for (channel = 0;
     channel < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
     channel++)
    {
    if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured != TRUE)
    {
        continue;
    }
    
    printf("%5d: %1s : %1s :%8d:%15s:%8d\n",
           (channel+1),
           (ixDspCodeletSocketChanConfigInfo[channel].isConfigured ? "Y":"N"),
           (ixDspCodeletSocketChanConfigInfo[channel].isActivated ? "Y":"N"),
           ixDspCodeletSocketChanConfigInfo[channel].localUdpPort,
           ixDspCodeletSocketChanConfigInfo[channel].remoteIpAddr,
           ixDspCodeletSocketChanConfigInfo[channel].remoteUdpPort);
    }
    printf("\n");
}

/*****************************************************************************
 *
 * This routine is used to show the statistical info that has been captured
 *
 *****************************************************************************/
void ixDspCodeletSocketStatsShow(void)
{
    unsigned channel ;

    printf("\n*** DSP Codelet Socket Statistics ***\n\n");

    for (channel = 0;
     channel < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
     channel++)
    {
    if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured != TRUE)
    {
        continue;
    }

    printf("Channel %d Statistics:\n", channel);
    printf("----------------------\n");
    printf("Number of Rx pkts: %d\n",
           ixDspCodeletSocketStats[channel].numRxPkts);
    printf("Total Rx bytes: %d\n",
           ixDspCodeletSocketStats[channel].totalRxBytes);
    printf("Number of invalid Rx data lengths: %d\n",
           ixDspCodeletSocketStats[channel].invalidRxDataLen);
    printf("Number of socket Rx failed: %d\n",
           ixDspCodeletSocketStats[channel].rxFailedCount);
    printf("Number of Successful Tx pkts: %d\n",
           ixDspCodeletSocketStats[channel].numSuccessTxPkts);
    printf("Total Successful Tx bytes: %d\n",
           ixDspCodeletSocketStats[channel].totalTxBytes);
    printf("Number of times all Tx data was not transmitted: %d\n",
           ixDspCodeletSocketStats[channel].allTxBytesNotTransmittedCount);
    printf("Number of times socket Tx failed: %d\n",
           ixDspCodeletSocketStats[channel].txFailedCount);
    printf("Number of remote signalling events sent: %d\n",
           ixDspCodeletSocketStats[channel].remoteEventsTx);
    printf("Number of remote signalling events received: %d\n",
           ixDspCodeletSocketStats[channel].remoteEventsRx);
    printf("Number of voice packets dropped: %d\n",
           ixDspCodeletSocketStats[channel].droppedPackets);
    printf("\n");
    }
    printf("\n");
}

/*****************************************************************************
 *
 * This routine is used to clear the statistical info that has been captured
 * for the specified channel
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletSocketStatsClear(int channel)
{
    if (channel >= IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || channel<0)
    {
    printf("%s(): ERROR: Attempted to clear stats for a channel that is out of"
           "range\n", __FUNCTION__);
    return IX_FAIL;
    }

    if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured == FALSE)
    {
    printf("%s(): ERROR: Attempted to clear stats for an unconfigured"
           " channel\n", __FUNCTION__);
    return IX_FAIL;
    }

    bzero((char *)&(ixDspCodeletSocketStats[channel]),
          sizeof(IxDspCodeletSocketStats));

    return IX_SUCCESS;
}

/*****************************************************************************
 *
 * This routine is used to clear all the statistical info that has been captured
 * for all of the configured channels
 *
 *****************************************************************************/
void ixDspCodeletSocketAllStatsClear(void)
{
    unsigned channel ;

    for (channel = 0;
     channel < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
     channel++)
    {
    if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured != TRUE)
    {
        continue;
    }

    bzero((char *)&(ixDspCodeletSocketStats[channel]),
          sizeof(IxDspCodeletSocketStats));
    }
}


/*****************************************************************************
 *
 * Open a socket and bind an address to it
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletSocketCreate(void)
{
    struct sockaddr_in sockAddrIn;
    unsigned channel ;
    short family = AF_INET;
    int switchFlag = 1;
    uint16_t tempSinPort;

    bzero((char *)&sockAddrIn, sizeof(sockAddrIn));

    family = NETWORK_FAMILY;

    /*
     * Create and bind to sockets for each voice channel that is configured
     */
    for (channel = 0;
     channel < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
     channel++)
    {
    /*
     * Only want to open a new socket and bind to it if the channel is 
     * configured, but has not yet been activated.
     */
    if ((!ixDspCodeletSocketChanConfigInfo[channel].isConfigured) ||
        (ixDspCodeletSocketChanConfigInfo[channel].isActivated))
    {
        continue;
    }
    
    /*
     * Open the Rx and Tx sockets and save the descripters
     */
    ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter = 
        socket(family, SOCK_DGRAM, 0);

    if (ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter == ERROR)
    {
        printf("%s(): ERROR - socket open failed for DSP data reception.\n",
           __FUNCTION__);
        return IX_FAIL;
    }

    ixDspCodeletSocketChanConfigInfo[channel].txSocketDescripter = 
        socket(family, SOCK_DGRAM, 0);

    if (ixDspCodeletSocketChanConfigInfo[channel].txSocketDescripter == ERROR)
    {
        printf("%s(): ERROR - socket open failed for DSP data transmission.\n",
           __FUNCTION__);
        return IX_FAIL;
    }

    sockAddrIn.sin_addr.s_addr = INADDR_ANY; 
    sockAddrIn.sin_family = AF_INET;

    tempSinPort = 
          htons(ixDspCodeletSocketChanConfigInfo[channel].localUdpPort);
    sockAddrIn.sin_port = (in_port_t)tempSinPort; 
        /*
        ixDspCodeletSocketChanConfigInfo[channel].localUdpPort;
            Note: In case of XScale htons call is not required as it is big endian,
        but in IA we have to convert into litle Endian so htons is called.    
       */
        /* Enable address reuse */
        if (setsockopt(ixDspCodeletSocketChanConfigInfo[channel].
                       rxSocketDescripter,
                       SOL_SOCKET,
                       SO_REUSEADDR,
                       &switchFlag,
                       sizeof(int)) == ERROR)
          {
        printf("%s(): ERROR - receive socket set reuse failed.\n",
           __FUNCTION__);
        return IX_FAIL;
    }

    /*
     * Only need to bind address to the receive socket
     */
    if (bind(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter,
         (struct sockaddr *)&sockAddrIn,
         sizeof(sockAddrIn) ) == ERROR)
    {
        printf("%s(): ERROR - receive socket bind failed.\n",
           __FUNCTION__);
        return IX_FAIL;
    }
    }

    return IX_SUCCESS;
}


/*****************************************************************************
 *
 * This function needs to read data in from the sockets and write that data
 * to a location accessible by the DSP. This function is running in its own
 * task. The priority of which is slightly higher than the tNetTask.
 *
 *****************************************************************************/

int ixDspCodeletSocketReceive(int channel)
{
    int receivedBytes = -1;
    unsigned dspLevelChannel = 0;
    struct sockaddr_in remoteSockAddr;
    int remoteSockAddrSize = 0; /* This variable has two purposes, to 
                 * indicate size of remoteSockAddr to the
                 * recvfrom function, and to store the size
                 * of the remoteSockAddr returned from the
                 * recvfrom function
                 */
    XPacket_t *dspBuffer;
    int maxRead=0;

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
    fd_set rdSockDs; /* set of read socket descriptors */
    int maxRdSockDs=0; /* to store the max value in rdSockDs */
#endif

#ifndef SRTP
    IxDspCodeletRtpMsg IxRecv;
#endif

    bzero((char *)&remoteSockAddr, sizeof(remoteSockAddr));

    maxRead = sizeof (XPacket_t);

    dspLevelChannel = SOCK_LVL_CHAN_TO_DSP_LVL_CHAN_MAP(channel);

    dspBuffer = (XPacket_t *)malloc (maxRead);

    while(1)
    {
    remoteSockAddrSize = sizeof(remoteSockAddr);

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
        FD_ZERO(&rdSockDs);
        maxRdSockDs= 0;
       /* Add the configured & activated channels into Read File
          Descriptor Set */
    for(channel=0;channel<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; channel++)
    {
        if(ixDspCodeletSocketChanConfigInfo[channel].isActivated == FALSE || 
           ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter ==
             IX_DSP_CODELET_SOCKET_INVALID_DESCRIPTER)
        {
                continue;
            }
        FD_SET(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter,
               &rdSockDs);
        if(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter >
                maxRdSockDs)
        {
          maxRdSockDs= ixDspCodeletSocketChanConfigInfo[channel].
                rxSocketDescripter;
        }

    }
    select(maxRdSockDs+1, &rdSockDs, NULL, NULL, NULL);
    for(channel = 0; channel<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; channel++)
    {
        if(!FD_ISSET(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter, &rdSockDs))
        {
                continue;
        }
        dspLevelChannel = SOCK_LVL_CHAN_TO_DSP_LVL_CHAN_MAP(channel);
#else
    if ( (ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter ==
          IX_DSP_CODELET_SOCKET_INVALID_DESCRIPTER) ||
         (ixDspCodeletSocketChanConfigInfo[channel].isActivated == FALSE) )
    {
        printf ("%s(): ERROR : receive thread started for channel %d "
                "but not configured\n", __FUNCTION__, channel);
            free(dspBuffer);
        return IX_FAIL;
    }
#endif

    ixDspCodeletSocketStats[channel].recvfromCalls++;

#ifdef SRTP
    bzero(dspBuffer,sizeof(dspBuffer));

    /* TO DO: change RTP_Recvfrom return the number of bytes received*/
    receivedBytes=ixDspCodelet_RTP_Recvfrom(&rcvr[channel],dspBuffer,&maxRead);

    ixDspCodeletSocketStats[channel].recvfromReturns++;

    if ( (receivedBytes == ERROR) || (receivedBytes == IX_FAIL) )
    {
        ixDspCodeletSocketStats[channel].rxFailedCount++;
        continue;
    }

    /* 
     * Record some stats on this receive routine
     */
    if (receivedBytes > 0)
    {
        ixDspCodeletSocketStats[channel].numRxPkts++;
        ixDspCodeletSocketStats[channel].totalRxBytes += receivedBytes;
    }

    ixDspCodeletSocketStats[channel].xPacketCalls++;


       dspBuffer->timeStamp = rcvr[channel].message.header.ts;
       dspBuffer->payloadType = rcvr[channel].message.header.pt;
       if ( (rcvr[channel].message.header.pt &  HEADER_MASK_VALUE) ==
             IX_DEF_RFC2833_RX_EVENT_PAYLOAD)
       {
       dspBuffer->mediaType = RTP_MD_TYPE_RFC2833_EVENT;
       }
       else if ( (rcvr[channel].message.header.pt &  HEADER_MASK_VALUE) ==
                  IX_DEF_RFC2833_RX_TONE_PAYLOAD)
       {
       dspBuffer->mediaType = RTP_MD_TYPE_RFC2833_TONE;
       }
       else
       {
       dspBuffer->mediaType = RTP_MD_TYPE_AUDIO;
       }


    /* Pass the received packet to DSR */

    xPacketReceive(dspLevelChannel, dspBuffer); 

#else
    bzero(dspBuffer,sizeof(dspBuffer));
    receivedBytes = 
        recvfrom(ixDspCodeletSocketChanConfigInfo[channel].rxSocketDescripter,
             (char *)&IxRecv,
             maxRead,
             0,     /* Flags */
             (struct sockaddr *)&remoteSockAddr,
             &remoteSockAddrSize);
         
        /* ToD: Validate Received Bytes here */

           ixDspCodeletSocketStats[channel].recvfromReturns++;

    if (receivedBytes == ERROR)
    {
        printf("receivedBytes = ERROR\n");
        ixDspCodeletSocketStats[channel].rxFailedCount++;
        continue;
    }

    /* 
     * Record some stats on this receive routine
     */
    if (receivedBytes > 0)
    {
        ixDspCodeletSocketStats[channel].numRxPkts++;
        ixDspCodeletSocketStats[channel].totalRxBytes += receivedBytes;
    }


        if (receivedBytes < IX_DSP_CODELET_RTPHDR_LENGTH)
        {
            /* The Receive pkt must contain the RTP header atleast */

            printf("Received Bad RTP Packet\n");
            continue;
            
        }


    /* Do some RTP header checks */
            
        if (ixDspCodeletSession[channel].state == 
                         IX_DSP_CODELET_ACTIVE_CHANNEL_RTP &&
               (IxRecv.header.rtpVersion & 0xc0) == 0x80 )

        {
             /* Copy the DSP buffer from the local structure to send to
                the DSR */
             dspBuffer->timeStamp = IxRecv.header.rtpTimestamp;
             dspBuffer->payloadType = IxRecv.header.rtpPayloadType;
             dspBuffer->payloadLen = receivedBytes - 
                IX_DSP_CODELET_RTPHDR_LENGTH;
             memcpy(&dspBuffer->payload[0],IxRecv.body,dspBuffer->payloadLen);
             if ((IxRecv.header.rtpPayloadType & HEADER_MASK_VALUE) ==
                        ixDspCodeletSession[channel].rfc2833RxEvent)
             {
                 dspBuffer->mediaType = RTP_MD_TYPE_RFC2833_EVENT;
                 /* RFC2833 event */
             }
             else if ((IxRecv.header.rtpPayloadType & HEADER_MASK_VALUE) ==
                       ixDspCodeletSession[channel].rfc2833RxTone)
             {
                  dspBuffer->mediaType = RTP_MD_TYPE_RFC2833_TONE;
                  /* RFC2833 tone */
             }
             else
             {
                  dspBuffer->mediaType = RTP_MD_TYPE_AUDIO; /* audio */
             }

                 
         ixDspCodeletSocketStats[channel].xPacketCalls++;

         xPacketReceive(dspLevelChannel, dspBuffer); 

        }

#endif

    ixDspCodeletSocketStats[channel].xPacketReturns++;
#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
        }
#endif
    }

    free(dspBuffer);
    return IX_SUCCESS;
}

/*****************************************************************************
 *
 * This routine is used to transmit data from the DSP to the network protocols
 *
 *****************************************************************************/
XStatus_t ixDspCodeletRtpSocketSend(UINT16 channel, unsigned char *buffer)
{
    /*
     * Internet socket address
     */
    struct sockaddr_in remoteSockAddr;
    unsigned bytesTransmitted = 0;
    unsigned totalDataLengthForTx = 0;
    XPacket_t *dspBuffer;
    uint16_t tempSinPort;

#ifndef SRTP
IxDspCodeletRtpMsg IxSnd;
#endif

    unsigned socketLvlChanNo = DSP_LVL_CHAN_TO_SOCK_LVL_CHAN_MAP(channel);

    bzero((char *)&remoteSockAddr, sizeof(remoteSockAddr));
#ifndef SRTP
    bzero((char *)&IxSnd, sizeof(IxSnd));
#endif

    /*
     * Validate the input paramters
     */
    if (socketLvlChanNo >= IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS || 
        socketLvlChanNo <0)
    {
    printf("%s: ERROR - DSP trying to transmit on an invalid channel\n",
           __FUNCTION__);
    return XERROR;
    }

    if (buffer == NULL)
    {
    printf("%s: ERROR - NULL input parameter detected\n",
           __FUNCTION__);
    return XERROR;
    }

    dspBuffer = (XPacket_t *)buffer;

    /* dspBuffer->channelID = channel; */

    if (ixDspCodeletSocketChanConfigInfo[socketLvlChanNo].isActivated == FALSE)
    {
    printf("%s: WARNING - Attempt to send on an unactivated channel (%d)\n",
           __FUNCTION__,
           socketLvlChanNo);
    return XERROR;
    }

    /*
     * The length of data to transmit includes the DSP Buffer header. As the 
     * header includes one kilo byte of payload, subtract this from the total length
     * for transmission
     */
    totalDataLengthForTx = 
        (dspBuffer->payloadLen + sizeof(XPacket_t)) - PAYLOAD_SIZE_1_KB;

    /*
     * Send the data to the socket
     */
#ifdef SRTP

           if (dspBuffer->mediaType == RTP_MD_TYPE_RFC2833_EVENT)
           {
             snd[socketLvlChanNo].message.header.pt=dspBuffer->payloadType |
                                         IX_DEF_RFC2833_TX_EVENT_PAYLOAD;
           }
           else if (dspBuffer->mediaType == RTP_MD_TYPE_RFC2833_TONE)
           {
             snd[socketLvlChanNo].message.header.pt=dspBuffer->payloadType |
                                         IX_DEF_RFC2833_TX_TONE_PAYLOAD;
           }
           else
           {
             snd[socketLvlChanNo].message.header.pt= dspBuffer->payloadType;
           }
           /* Use the time stamp from the DSP */
           snd[socketLvlChanNo].message.header.ts = dspBuffer->timeStamp;
           bytesTransmitted = ixDspCodelet_RTP_Sendto(&snd[socketLvlChanNo],
                                          dspBuffer, totalDataLengthForTx);
#else
 
    /*
     * Configure the Internet socket address structure
     */
    remoteSockAddr.sin_family = AF_INET;
#ifdef OS_VXWORK
     remoteSockAddr.sin_len = sizeof(remoteSockAddr);
#endif    
      
    inet_aton(ixDspCodeletSocketChanConfigInfo[socketLvlChanNo].remoteIpAddr,
              &remoteSockAddr.sin_addr);

    tempSinPort = 
       htons(ixDspCodeletSocketChanConfigInfo[socketLvlChanNo].remoteUdpPort);
    remoteSockAddr.sin_port = (in_port_t)tempSinPort; 

        /* increment sequence number */
        ixDspCodeletSession[channel - 1].rtpHeader.rtpSequence++;
        if (dspBuffer->mediaType == RTP_MD_TYPE_RFC2833_EVENT)
        {
            ixDspCodeletSession[channel - 1].rtpHeader.rtpPayloadType =
                dspBuffer->payloadType | 
                ixDspCodeletSession[channel-1].rfc2833TxEvent;
        }
        else if (dspBuffer->mediaType == RTP_MD_TYPE_RFC2833_TONE)
        {
            ixDspCodeletSession[channel - 1].rtpHeader.rtpPayloadType =
                dspBuffer->payloadType | 
                ixDspCodeletSession[channel-1].rfc2833TxTone;
        }
        else
        {
            ixDspCodeletSession[channel - 1].rtpHeader.rtpPayloadType =
              dspBuffer->payloadType;
        }
        /* use timestamp from DSP */
        ixDspCodeletSession[channel - 1].rtpHeader.rtpTimestamp = 
             dspBuffer->timeStamp;



    memcpy(&IxSnd.header,(&ixDspCodeletSession[channel - 1].rtpHeader),
             IX_DSP_CODELET_RTPHDR_LENGTH);

    memcpy((UINT8 *)IxSnd.body,  (UINT8 *)(&dspBuffer->payload[0]),
             dspBuffer->payloadLen);



   bytesTransmitted = 
   sendto(ixDspCodeletSocketChanConfigInfo[socketLvlChanNo].txSocketDescripter,
           &IxSnd,
           (IX_DSP_CODELET_RTPHDR_LENGTH + dspBuffer->payloadLen),
           0, 
           (struct sockaddr *)(&remoteSockAddr),
           sizeof(remoteSockAddr) );

#endif

    if (bytesTransmitted == ERROR)
    {

    ixDspCodeletSocketStats[socketLvlChanNo].txFailedCount++;
    return XERROR;
    }

    ixDspCodeletSocketStats[socketLvlChanNo].numSuccessTxPkts++;
    ixDspCodeletSocketStats[socketLvlChanNo].totalTxBytes += 
    totalDataLengthForTx;
    return XSUCC;
}

/*****************************************************************************
 *
 * Initialise the socket processing routines
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletSocketInit(void )
{
    char taskName[MAX_STRING_LENGTH_TASKNAME];
    int channel;

    /*
     * Ensure that at least one channel has been configured
     */
    if (!ixDspCodeletSocketAtLeastOneChanelIsConfigured)
    {
        printf("%s: ERROR - At least one channel must be configured before"
               " initialisation\n", __FUNCTION__);
        return IX_FAIL;
    }
    
    /*
     * Open the socket and bind an address to it
     */
    if (ixDspCodeletSocketCreate() == IX_FAIL)
    {
    printf("%s: ERROR - Aborting socket init\n", __FUNCTION__);
    return IX_FAIL;
    }

    /*
     * Kick off the socket receive tasks
     */
    for (channel = 0; channel < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
         channel++)
    {

        if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured == FALSE)
        {
            /* Channel is not configured, yet */
            continue;
        }

        if (ixDspCodeletSocketChanConfigInfo[channel].isActivated == TRUE)
        {
            /* Channel is already active */
            continue;
        }
        snprintf(taskName, MAX_STRING_LENGTH_TASKNAME, "dspSckRcv%d", channel);

        /* Mark the Channel as Active */
        ixDspCodeletSocketChanConfigInfo[channel].isActivated = TRUE;

         /* TO DO: Thread_Create does not return an error status. */
#ifndef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
#ifndef OS_VXWORK
         ixDspCodeletSocketChanConfigInfo[channel].threadID = 
                Thread_Create_RT(DEFAULT_SOCKET_THREAD_PRIORITY_LINUX,
                                (void*)ixDspCodeletSocketReceive, channel);
#else
         ixDspCodeletSocketChanConfigInfo[channel].threadID = 
                Thread_Create(DEFAULT_SOCKET_THREAD_PRIORITY_VXWORK, 
                              ixDspCodeletSocketReceive, channel);
#endif
#endif
    }

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
    if(!socketReceiveThreadCreated)
    {
#ifndef OS_VXWORK
        socketReceiveThreadId = 
                Thread_Create_RT(DEFAULT_SOCKET_THREAD_PRIORITY_LINUX, 
                                      (void*)ixDspCodeletSocketReceive, 0);
#else
        socketReceiveThreadId = 
                Thread_Create(DEFAULT_SOCKET_THREAD_PRIORITY_VXWORK, 
                                      ixDspCodeletSocketReceive, 0);

#endif
     socketReceiveThreadCreated = TRUE;
    }
#endif
#ifndef SRTP
    ixDspCodeletRtpDefaultSet();
#endif
#ifdef SRTP
    ixDspCodeletSRTP_SndRcv_Session_Init();
#endif

    return IX_SUCCESS;
}

/*****************************************************************************
 *
 * This routine has been coded as a simple and quick means of setting up a 
 * configuration from the windsh for testing purposes. 
 * 
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletSocketPreDefinedChanConfig(IxDspSocketConfigType 
                                                 configType)
{
    IX_STATUS retVal = IX_SUCCESS;
    int channelNo, count=0, maxCountUserProfile=0;

    switch(configType)
    {
    case DEFAULT_CONFIG_0:
      ixDspCodeletSocketChanConfig( (IX_DSP_CODELET_CHANNEL_1-1), 
            IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+(IX_DSP_CODELET_CHANNEL_1-1) );
      ixDspCodeletSocketChanConfig( (IX_DSP_CODELET_CHANNEL_2-1), 
            IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+(IX_DSP_CODELET_CHANNEL_2-1) );
      ixDspCodeletSocketChanConfig( (IX_DSP_CODELET_CHANNEL_3-1), 
            IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+(IX_DSP_CODELET_CHANNEL_3-1) );
      ixDspCodeletSocketChanConfig( (IX_DSP_CODELET_CHANNEL_4-1), 
            IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+(IX_DSP_CODELET_CHANNEL_4-1) );

     ixDspCodeletSocketRemoteSet((IX_DSP_CODELET_CHANNEL_1-1), DEFAULT_IXP1_IP,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + (IX_DSP_CODELET_CHANNEL_2-1));
     ixDspCodeletSocketRemoteSet((IX_DSP_CODELET_CHANNEL_2-1), DEFAULT_IXP0_IP,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + (IX_DSP_CODELET_CHANNEL_1-1));
     ixDspCodeletSocketRemoteSet((IX_DSP_CODELET_CHANNEL_3-1), DEFAULT_IXP1_IP,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + (IX_DSP_CODELET_CHANNEL_4-1));
     ixDspCodeletSocketRemoteSet((IX_DSP_CODELET_CHANNEL_4-1), DEFAULT_IXP0_IP,
           IX_DSP_CODELET_SOCKET_UDP_PORT_BASE + (IX_DSP_CODELET_CHANNEL_3-1));
      break;

    case MAX_DEFAULT_CONFIG:
        for(channelNo=0; channelNo<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
                    channelNo++)
        {
          ixDspCodeletSocketChanConfig(channelNo,
                IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+channelNo);
        }

        for(channelNo=0; channelNo<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS;
                   channelNo+=OFFSET_2)
        {
            ixDspCodeletSocketRemoteSet(channelNo, DEFAULT_IXP1_IP,
                 IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+channelNo+1);
            ixDspCodeletSocketRemoteSet(channelNo+1, DEFAULT_IXP0_IP,
                 IX_DSP_CODELET_SOCKET_UDP_PORT_BASE+channelNo);
        }
        break;
    case USER_DEFINED_CONFIG:
        maxCountUserProfile = (sizeof(ixDspCodeletUserProfile)) /
                              (sizeof(UINT16)*FACTOR_2);
        
        while(ixDspCodeletUserProfile[count].source!=0 ||
              ixDspCodeletUserProfile[count].destination!=0)
        {
            if( (ixDspCodeletUserProfile[count].source<1 && 
                    ixDspCodeletUserProfile[count].source >
                       IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS) ||
                (ixDspCodeletUserProfile[count].destination<1 && 
                    ixDspCodeletUserProfile[count].destination >
                       IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS) )
            {
                printf("Invalid channel combination{%d, %d} in user profile,"
                       " scanning next combination\n",
                       ixDspCodeletUserProfile[count].source,
                       ixDspCodeletUserProfile[count].destination);
                count++;
                continue;
            }
            if( (ixDspCodeletSocketChanConfigInfo[
                   (ixDspCodeletUserProfile[count].source)].isConfigured
                       == TRUE) || 
                   (ixDspCodeletSocketChanConfigInfo
                       [(ixDspCodeletUserProfile[count].destination)].
                          isConfigured == TRUE) )
            {
                printf("One of the channels already configured out of "
                       "{%d, %d} in user profile, scanning next conbination\n",
                       ixDspCodeletUserProfile[count].source,
                       ixDspCodeletUserProfile[count].destination);
                count++;
                continue;
            }

            ixDspCodeletSocketChanConfig(
                     ixDspCodeletUserProfile[count].source-1,
                     IX_DSP_CODELET_SOCKET_UDP_PORT_BASE +
                     ixDspCodeletUserProfile[count].source-1);
            ixDspCodeletSocketChanConfig(
                     ixDspCodeletUserProfile[count].destination-1, 
                     IX_DSP_CODELET_SOCKET_UDP_PORT_BASE +
                     ixDspCodeletUserProfile[count].destination-1);
           ixDspCodeletSocketRemoteSet(ixDspCodeletUserProfile[count].source-1,
                     DEFAULT_IXP1_IP, IX_DSP_CODELET_SOCKET_UDP_PORT_BASE +
                     ixDspCodeletUserProfile[count].destination-1);
            ixDspCodeletSocketRemoteSet(
                     ixDspCodeletUserProfile[count].destination-1,
                     DEFAULT_IXP0_IP,
                     IX_DSP_CODELET_SOCKET_UDP_PORT_BASE +
                     ixDspCodeletUserProfile[count].source-1);
            count++;
            if(count >= maxCountUserProfile)
            {
                printf("Not enough members in user profile structure, so"
                       " breaking the loop...\n", count+1);
                break;
            }
            if(count >= MAX_CHANNEL_COMBINATION )
            {
                printf("Too big user profile structure, %dth onwards ignoring"
                       " rest of the entries\n", count+1);
                break;
            }
        }
        break;

    default:
        break;
    }

    return retVal;
}


/**************************************************************************
*
*
*       SRTP code
*
*
*
***************************************************************************/

#ifdef SRTP

/*****************************************************************************
 *
 * This routine translates a error number to a string equivalent.
 *
 *****************************************************************************/
void error(err_status_t status)
{
    if(status==err_status_fail)
    {
        printf("Error: Unspecified failure\n");
    }
    else if (status==err_status_bad_param)
    {
        printf("Error: Unsupported paramneters were passed\n");
    }
    else if(status==err_status_alloc_fail)
    {
        printf("Error: Memory allocation wasn't successful\n");
    }
    else if(status==err_status_dealloc_fail)
    {
        printf("Error: Memory de-allocation failed\n");
    }
    else if(status==err_status_init_fail)
    {
        printf("Error: Initialization failed\n");
    }
    else if(status==err_status_terminus)
    {
        printf("Error: Could not process as much data as requested\n");
    }
    else if(status==err_status_auth_fail)
    {
        printf("Error: Authentication failed\n");
    }
    else if(status==err_status_cipher_fail)
    {
        printf("Error: Cipher failure\n");
    }
    else if(status==err_status_replay_fail)
    {
        printf("Error: Replay check failed - bad index\n");
    }
    else if(status==err_status_replay_old)
    {
        printf("Error: Replay check failed - old index");
    }
    else if(status==err_status_algo_fail)
    {
        printf("Error: Algorithm failed test routine\n");
    }
    else if(status==err_status_no_such_op)
    {
        printf("Error: Unsupported operation\n");
    }
    else if(status==err_status_no_ctx)
    {
        printf("Error: No appropriate context found\n");
    }
    else if(status==err_status_cant_check)
    {
        printf("Error: Unable to perform desired validation\n");
    }
    else if(status==err_status_key_expired)
    {
        printf("Error: Can't use key anymore\n");
    }
}


/*****************************************************************************
 *
 * This routine is used to initialize the SRTP library and has to be the first
 * routine to be called before using other API's from srtp library.
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletSRTPInit(srtp_policy_t *policy, UINT8 *key, UINT32 ssrc)
{
    err_status_t status;
    status = srtp_init();
    if(status)
    {
        error(status);
        printf("%s(): Error - srtp_init\n", __FUNCTION__);
        return IX_FAIL;
    }

    policy->rtp.sec_serv=sec_serv_conf_and_auth;
    policy->rtcp.sec_serv=sec_serv_none;

    crypto_policy_set_rtp_default(&policy->rtp);

    policy->ssrc.type=ssrc_specific;
    policy->ssrc.value=ssrc;
    policy->key=key;
    policy->next=NULL;

    return IX_SUCCESS;
}

/*****************************************************************************
 *
 * This routine is used to initialize the SRTP session for send and receive 
 * and has to be called before doind send/receive with srtp library.
 *
 *****************************************************************************/

IX_STATUS ixDspCodeletSRTP_SndRcv_Session_Init(void)
{
    struct sockaddr_in remoteSockAddr;
    uint16_t tempSinPort;
    int chan;
    UINT32 ssrc = 0xdeadbeef;

    for(chan=0; chan<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; chan++)
    {
       bzero((char *)&snd[chan], sizeof(rtp_sender_t));
       if(ixDspCodeletSocketChanConfigInfo[chan].isActivated == FALSE)
       {
           continue;
       }

      /*
       * Configure the Internet socket address structure
       */
       bzero((char *)&remoteSockAddr, sizeof(remoteSockAddr));
       remoteSockAddr.sin_family = AF_INET;
      
       inet_aton(ixDspCodeletSocketChanConfigInfo[chan].remoteIpAddr,
                        &remoteSockAddr.sin_addr);

       tempSinPort = 
         htons(ixDspCodeletSocketChanConfigInfo[chan].remoteUdpPort);
       remoteSockAddr.sin_port = (in_port_t)tempSinPort;

       if(ixDspCodeletRTPSendInit(&snd[chan],
           ixDspCodeletSocketChanConfigInfo[chan].txSocketDescripter, 
           chan+1, ssrc, &remoteSockAddr, &policy)!=IX_SUCCESS)
       {
          printf("Error: RTPSendInit failed for channel %d\n", chan+1);
          return IX_FAIL;
       }
    }

    for(chan=0; chan<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; chan++)
    {
       bzero((char *)&rcvr[chan], sizeof(rtp_receiver_t));
       if(ixDspCodeletSocketChanConfigInfo[chan].isActivated == FALSE)
       {
           continue;
       }
       bzero((char *)&remoteSockAddr, sizeof(remoteSockAddr));

       if(ixDspCodeletRTPRecvInit(&rcvr[chan], 
           ixDspCodeletSocketChanConfigInfo[chan].rxSocketDescripter,
           chan+1, ssrc, &remoteSockAddr, &policy)!=IX_SUCCESS)
       {
          printf("Error: RTPRecvInit failed for channel %d\n", chan+1);
          return IX_FAIL;
       }
    }
}

/*****************************************************************************
 *
 * This routine is used to delete the SRTP session for send and receive 
 * and has to be called after all the sockets are destroyed.
 *
 *****************************************************************************/

IX_STATUS ixDspCodeletSRTP_SndRcv_Session_Free(void)
{
        int chan;
        for(chan=0; chan<IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; chan++)
        {
           if(ixDspCodeletSocketChanConfigInfo[chan].isActivated == FALSE)
           {
                   continue;
           }
           srtp_dealloc(snd[chan].srtp_ctx);
           srtp_dealloc(rcvr[chan].srtp_ctx);
        }
}

/*****************************************************************************
 *
 * This routine initializes the sender context and fills in default values for
 * RTP header and allocates and initializes an SRTP session by calling 
 * the srtp_create() function .
 *
 *****************************************************************************/
IX_STATUS ixDspCodeletRTPSendInit(rtp_sender_t *sender, int TxSock,
                                  int channel, UINT32 ssrc,
                                  struct sockaddr_in *sendtoaddr,
                                  srtp_policy_t *rpolicy)
{
    err_status_t status;
    uint16_t seq_random=(uint16_t)random();
    
    sender->message.header.ssrc=htonl(ssrc);
    sender->message.header.ts=0;
    sender->message.header.seq=htons(seq_random);
    sender->message.header.m=0;
    sender->message.header.pt=0xf;
    sender->message.header.version = RTP_VERSION;
    sender->message.header.p=0;
    sender->message.header.x=0;
    sender->message.header.cc=0;
    
    sender->socket=TxSock;
    sender->addr.sin_family=sendtoaddr->sin_family;
    sender->addr.sin_port=sendtoaddr->sin_port;
    sender->addr.sin_addr.s_addr=sendtoaddr->sin_addr.s_addr;
    memset(&(sender->addr.sin_zero), '\0', sizeof(sender->addr.sin_zero) );
   
    status=srtp_create(&sender->srtp_ctx, rpolicy);
    if(status)
    {    
        error(status);
        printf("%s(): Error - srtp_create\n", __FUNCTION__);
        return IX_FAIL;
    }
    sender->message.header.seq=seq_random;
    return IX_SUCCESS;
}

/****************************************************************************
 *
 * This routine initializes the receive context and fills in default values for
 * RTP header and allocates and initializes an SRTP receive session by calling 
 * the srtp_create() function .
 *
 ****************************************************************************/
IX_STATUS ixDspCodeletRTPRecvInit(rtp_receiver_t *receiver, int RxSock,
                int channel, UINT32 ssrc, struct sockaddr_in *recvaddr,
                srtp_policy_t *policy)
{

    err_status_t status;

    receiver->message.header.ssrc=htonl(ssrc);
    receiver->message.header.ts=0;
    receiver->message.header.seq=0;
    receiver->message.header.m=0;
    receiver->message.header.pt=0xf;
    receiver->message.header.version = RTP_VERSION;
    receiver->message.header.p=0;
    receiver->message.header.x=0;
    receiver->message.header.cc=0;
    
    receiver->socket=RxSock;
    
    status=srtp_create(&receiver->srtp_ctx, policy);
    
    if(status)
    {
        error(status);
        printf("%s(): Error - srtp_create\n", __FUNCTION__);
        return IX_FAIL;
    }
    
    return IX_SUCCESS;
}


/****************************************************************************
 *
 * This function is the send side of the socket interface and uses the 
 * srtp_protect() function from the srtp library to apply the SRTP protection 
 * to the RTP packet that is being transmitted.
 *
 *****************************************************************************/
ssize_t ixDspCodelet_RTP_Sendto(rtp_sender_t *sender, const void *msg, int len)
{
    err_status_t status;
    int octets_sent = -1;
    int pkt_len = len+rtp_header_len;

#ifdef SRTP_ACCEL_FLAG
    int pad_counter = 0;
    int num_pad_bytes = SRTP_BLOCK_LEN - (len & 0xf);
    UINT8 *pad_start = NULL;
#endif

    memcpy(sender->message.body, msg, len);

#ifdef SRTP_ACCEL_FLAG
    /* Adding padding Bits */
    pad_start = (UINT8*)sender->message.body + len;
    for(pad_counter=0; pad_counter < num_pad_bytes; pad_counter++)
    {
        pad_start[pad_counter] = 0x0;
    }
    pkt_len += num_pad_bytes;
#endif

    /* update sequence number and timestamp */
    sender->message.header.seq=ntohs(sender->message.header.seq)+1;
    sender->message.header.seq=htons(sender->message.header.seq);

    sender->message.header.ts=ntohl(sender->message.header.ts)+1;
    sender->message.header.ts=htonl(sender->message.header.ts);

    status=srtp_protect(sender->srtp_ctx, &sender->message.header, &pkt_len); 

    if(status)
    {
        error(status);
        printf("%s(): Error - srtp_protect\n", __FUNCTION__);
        return IX_FAIL;
    }

    octets_sent=sendto(sender->socket, &sender->message, pkt_len, 0,
           (struct sockaddr*)&sender->addr, sizeof(struct sockaddr));

    if (octets_sent>0 && octets_sent!=pkt_len)
    {
        printf("%s(): Error - Couldn't send entire message\n", __FUNCTION__);
        return IX_FAIL;
    }
    else if(octets_sent<0)
    {
        printf("%s(): Error - Couldn't send message\n", __FUNCTION__);
        return IX_FAIL;
    }

    return octets_sent;
}


/****************************************************************************
 *
 * This function is the recive side of the socket interface and uses the 
 * srtp_unprotect() function from the srtp library to verify the SRTP
 * protection of the RTP packet that is received.
 *
 *****************************************************************************/
ssize_t ixDspCodelet_RTP_Recvfrom(rtp_receiver_t *receiver, 
                                  void *msg, 
                                  int *len)
{
    int octets_recd=0, loc_errno;
    int *octPtr;
    err_status_t status;
    socklen_t addrlen;

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
    int flags=MSG_WAITALL;
#else
    int flags=0;
#endif

    /* if the flags arg has MSG_DONTWAIT, then the recvfrom call will not
       block if there is no data to read from the socket */
    do
    {
#ifdef SRTP_ACCEL_FLAG
        octets_recd=recvfrom(receiver->socket, (void*)&receiver->message, 
                            (*len+PAD_BYTES_8), flags, 
                            (struct sockaddr*)&receiver->addr, &addrlen);
#else
        octets_recd=recvfrom(receiver->socket, (void*)&receiver->message, 
                            (*len), flags, 
                            (struct sockaddr*)&receiver->addr, &addrlen);
#endif
    } while(octets_recd < 0);

    if(receiver->message.header.version != RTP_VERSION)
    {
        *len=0;
        printf("%s(): Error - Wrong rtp header version\n", __FUNCTION__);
        return IX_FAIL;
    }

    octPtr = &octets_recd;

    status=srtp_unprotect(receiver->srtp_ctx, &receiver->message.header,
                          octPtr); 
    if(status)
    {
        error(status);
        printf("%s(): Error - srtp_unprotect\n", __FUNCTION__);
        return IX_FAIL;
    }

    memcpy(msg, receiver->message.body, octets_recd);

    return octets_recd;
}

#endif


/*************************************************************************
*
*        SOCKET MENU HANDLING ROUTINES
*
**************************************************************************/

/****************************************************************************
 *
 * This function is the main "socket configuration menu" handler.
 *
 ****************************************************************************/
void ixDspCodeletAppSockConfig(void)
{

  int count;
  int selectedItem = 0;
  int lastItem = -1;

  do {

    if(!selectedItem)
    {
        printf( "\n----------------------------------------\n"
                  "-      IxDspCodelet Socket Menu        -\n"
                  "----------------------------------------\n");

        for (count = 0; ixDspCodeletSocketMenu[count].name != NULL; count++)
        {
           printf("%3d - %s", count, ixDspCodeletSocketMenu[count].name);

           /* Print the range for Max Defalult Configiration */
           if( count == (MAX_DEFAULT_CONFIG-1) ) 
           {
             printf(";loop-back from '1<->2'..to..'%d<->%d'",
                 (int)IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS-1,
                 (int)IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS );
           }
           printf("\n");
        }

        printf("Please select test item - ");
        lastItem = count - 1;
    }

    /* select a menu item */
    selectedItem = ixDspCodeletGetNum(NULL);

    /* skip if not a menu item */
    if (selectedItem >= lastItem || !selectedItem) {
      continue;
    }

    ixDspCodeletSocketMenu[selectedItem].menuFunc();

  } while (selectedItem != lastItem);

}/* ixDspCodeletAppSockConfig */


/*****************************************************************************
 *
 * This function is the entry funtion of the channel configuration option under
 * the socket configuraion menu.
 *
 *****************************************************************************/
void ixDspCodeletSockMenuConfig(void)
{
    int num = 0;
    int srcCh,dstCh;
    char dstIp[MAX_IP_ADDR_LEN];
    UINT16 srcPort = 0, dstPort = 0;

       srcCh = ixDspCodeletGetNum("Enter the Src channel number "
                                  "(1,2,3....24) - ");
       if(srcCh < 0 || srcCh > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS)
       {
         printf("ERROR - invalid Src channel number %d\n", srcCh);
       }
       else
       {
         srcPort = (srcCh - 1) + IX_DSP_CODELET_SOCKET_UDP_PORT_BASE; 
       }

       dstCh = ixDspCodeletGetNum("Enter the Dst channel number "
                                  "(1,2,3....24) - ");

       if(dstCh < 0 || dstCh > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS)
       {
         printf("ERROR - invalid Dst channel number %d\n", dstCh);
       }
       else
       {
           /* Add 2000 to the channel number to get the port number */
           dstPort = (dstCh - 1) + IX_DSP_CODELET_SOCKET_UDP_PORT_BASE;
       }

       num = ixDspCodeletGetStr("Enter the Dst IP:",dstIp);

       if (!num)
       {
          printf("Error : No Destination IP address configured \n");
          return;
       }
           ixDspCodeletSocketChanConfig((srcCh - 1),srcPort);
           ixDspCodeletSocketRemoteSet((srcCh - 1),dstIp,dstPort);

       printf("\n");
}


/*****************************************************************************
 *
 * This function is the entry funtion of the Disable channel option under
 * the socket configuraion menu.
 *
 *****************************************************************************/
void ixDspCodeletSockMenuDisable(void)
{
   int chl, count;
   IxDspCodeletMsgStopIP msg;
   UINT32 trans;
   unsigned channel;

       chl = ixDspCodeletGetNum("Enter the channel number to Disable "
                                "(1,2,3....24 or 0 for all) - ");

       if (chl == 0)
       {
       for (count = 0; count < IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; count++)
           {
              /* Teardown the channel first */
              trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST,(count+1));

              IX_DSP_CODELET_MAKE_MSG_STOP_IP(&msg, trans, (count+1))

              if(xMsgSend(&msg) != XSUCC)
              {
                  printf("ERROR - Message sender fails.\n");
                  break;
              }

              ixDspCodeletSocketChanDisable(count);
           }
        /* All channels are disabled. */
    ixDspCodeletSocketAtLeastOneChanelIsConfigured = FALSE;
    printf("\n");
       }
       else
       {
           if(chl < 0 || chl > IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS)
           {
             printf("ERROR - invalid channel number %d\n", chl);
           }
           else
           {
               /* Teardown the channel first */
              trans = IX_DSP_CODELET_MAKE_TRANS(IX_DSP_CODELET_CATEGORY_TEST,chl);

              IX_DSP_CODELET_MAKE_MSG_STOP_IP(&msg, trans, chl)

              if(xMsgSend(&msg) != XSUCC)
              {
                  printf("ERROR - Message sender fails.\n");
              }

              
          printf("Calling ChanDisable with chl = %d \n",chl);
              ixDspCodeletSocketChanDisable(chl - 1);
          printf("\n");
              /* find whether all the channel is disabled */
              count = 0;
              for (channel = 0; channel < 
                         IX_DSP_CODELET_SOCKET_MAX_NUM_SOCKETS; channel++)
              {
                  if (ixDspCodeletSocketChanConfigInfo[channel].isConfigured
                       != FALSE)
                  {
                      count = 1;
                      break;
                  }
              }
              if(count == 0)
              {
                  ixDspCodeletSocketAtLeastOneChanelIsConfigured = FALSE;
              }
           }
        }

#ifdef IX_DSP_CODELET_SOCKET_SINGLE_THREAD
        if( !(ixDspCodeletSocketAtLeastOneChanelIsConfigured) && 
               socketReceiveThreadCreated )
        {
            Thread_Cancel(socketReceiveThreadId);
            socketReceiveThreadCreated = FALSE;
        }
#endif

        printf("\n");    
}


/*****************************************************************************
 *
 * This function is the entry funtion of the show channel config option under
 * the socket configuraion menu and display the current channel configuration.
 *
 *****************************************************************************/
void ixDspCodeletSockMenuShowConfig(void)
{
     ixDspCodeletSocketChanConfigShow();
     printf("\n");
}


/*****************************************************************************
 *
 * This function is the entry funtion of the channel statistis option under
 * the socket configuraion menu and displays the channel statistics.
 *
 *****************************************************************************/
void ixDspCodeletSockMenuShowStats(void)
{

    ixDspCodeletSocketStatsShow();
    printf("\n");
}


/****************************************************************************
 *
 * This function is the entry funtion of the create sokcets option under
 * the socket configuraion menu and creates and binds the sockets as per the
 * channel configuration done.
 *
 ****************************************************************************/
void ixDspCodeletSockMenuInit(void)
{

    if(ixDspCodeletSocketInit() !=IX_SUCCESS)
    {
       printf("Error: Socket Initilazation failed\n");
       return;
    }

    ixDspCodeletSocketChanConfigShow();
    printf("\n");

}

/*****************************************************************************
 *
 * This function is the entry funtion of the default channel configuration 
 * option under the socket configuraion menu.
 * 
 *****************************************************************************/
void ixDspCodeletSockMenuDefaultConfig(void)
{

     printf("Enabling Default Channel Configuration \n");
     printf("ixp0 IPAddrs of : %s ",DEFAULT_IXP0_IP);
     printf("ixp1 IPAddrs of : %s ",DEFAULT_IXP1_IP);
     printf("is assumed \n");
    
    ixDspCodeletSocketPreDefinedChanConfig(DEFAULT_CONFIG_0);

    if(ixDspCodeletSocketInit() !=IX_SUCCESS)
    {
       printf("Error: Socket Initilazation failed\n");
       return;
    }

    ixDspCodeletSocketChanConfigShow();
    printf("\n");
}

/*****************************************************************************
 *
 * This function is the entry funtion of the Maximum channel configuration 
 * option under the socket configuraion menu.
 * 
 *****************************************************************************/
void ixDspCodeletSockMenuMaxDefaultConfig(void)
{

     printf("Enabling Default Channel Configuration \n");
     printf("ixp0 IPAddrs of : %s ",DEFAULT_IXP0_IP);
     printf("ixp1 IPAddrs of : %s ",DEFAULT_IXP1_IP);
     printf("is assumed \n");
    
    ixDspCodeletSocketPreDefinedChanConfig(MAX_DEFAULT_CONFIG );

    if(ixDspCodeletSocketInit() !=IX_SUCCESS)
    {
       printf("Error: Socket Initilazation failed\n");
       return;
    }

    ixDspCodeletSocketChanConfigShow();
    printf("\n");
}

/*****************************************************************************
 *
 * This function is the entry funtion of the User defined channel configuration 
 * option under the socket configuraion menu.
 * 
 *****************************************************************************/
void ixDspCodeletSockMenuUserDefinedConfig(void)
{

     printf("Enabling Default Channel Configuration \n");
     printf("ixp0 IPAddrs of : %s ",DEFAULT_IXP0_IP);
     printf("ixp1 IPAddrs of : %s ",DEFAULT_IXP1_IP);
     printf("is assumed \n");
    
    ixDspCodeletSocketPreDefinedChanConfig(USER_DEFINED_CONFIG);

    if(ixDspCodeletSocketInit() !=IX_SUCCESS)
    {
       printf("Error: Socket Initialization failed\n");
       return;
    }

    ixDspCodeletSocketChanConfigShow();
    printf("\n");
}

/************************************************************************
       RTP ONLY code
*************************************************************************/

/*****************************************************************************
 *
 * This function is used to setup the default parameters of the pseudo RTP 
 * implementation and will be used when the SRTP library is not being used
 * for SRTP functionality.
 *
 *****************************************************************************/
void ixDspCodeletRtpDefaultSet(void)
{
    int count;


    for (count=0; count<IX_DSP_CODELET_IP_CHANNELS; count++)
    {

        ixDspCodeletSession[count].rtpHeader.rtpVersion = 0x80;
        ixDspCodeletSession[count].rtpHeader.rtpPayloadType = 0;
        ixDspCodeletSession[count].rtpHeader.rtpSequence = 0;
        ixDspCodeletSession[count].rtpHeader.rtpTimestamp = 0;
        if (count & 0x1)
        {
            /* odd */
            ixDspCodeletSession[count].rtpHeader.rtpSyncSource = 0x456f;
        }
        else
        {
            /* even */
            ixDspCodeletSession[count].rtpHeader.rtpSyncSource = 0x4570;
        }

        ixDspCodeletSession[count].rfc2833RxEvent = 
                       IX_DSP_DEF_RFC2833_RX_EVENT_PAYLOAD;
        ixDspCodeletSession[count].rfc2833RxTone = 
                       IX_DSP_DEF_RFC2833_RX_TONE_PAYLOAD;
        ixDspCodeletSession[count].rfc2833RxRed = 
                       IX_DSP_DEF_RFC2833_RX_RED_PAYLOAD;
        ixDspCodeletSession[count].rfc2833TxEvent = 
                       IX_DSP_DEF_RFC2833_TX_EVENT_PAYLOAD;
        ixDspCodeletSession[count].rfc2833TxTone = 
                       IX_DSP_DEF_RFC2833_TX_TONE_PAYLOAD;
        ixDspCodeletSession[count].rfc2833TxRed = 
                       IX_DSP_DEF_RFC2833_TX_RED_PAYLOAD;
        ixDspCodeletSession[count].state = IX_DSP_CODELET_ACTIVE_CHANNEL_RTP;
    }
}

#ifndef OS_VXWORK
thread_T Thread_Create_RT(int priority, threadFunc_T startFunc,
                          threadArg_T *arg)
{
    /* Creates Real Time thread with RR Scheduling */
        thread_T tid;
        pthread_attr_t attr;
        struct sched_param param;
    int retVal;
        pthread_attr_init(&attr);
        pthread_attr_setschedpolicy(&attr, SCHED_RR);
        pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED);
        param.sched_priority = priority;
    pthread_attr_setschedparam(&attr, &param);
        retVal=pthread_create(&tid, &attr, (pthread_startroutine_t)startFunc,
                              (pthread_addr_t)arg);

    if(retVal != 0)
    {    
        printf("\nFailed to create RT thread, Error No = %d \n", retVal);
        return 0;
    }
    return tid;
}
#endif

#ifdef OS_VXWORK 
thread_T Thread_Create(int priority, threadFunc_T startFunc, threadArg_T *arg)
{
    thread_T             thread;
    pthread_attr_t       thread_attr;
    struct sched_param   param;

    pthread_attr_init(&thread_attr);
    pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);

    param.sched_priority = priority;
    pthread_attr_setschedparam(&thread_attr, &param);

    pthread_create(&thread, &thread_attr, (pthread_startroutine_t)startFunc,
                   (pthread_addr_t)arg);
    return (thread);
}

int Thread_Cancel(thread_T thread)
{
    return (pthread_cancel(thread));
}
#endif

