/******************************************************************************
*       @file  IxDspCodeletUsrMsg.c
*
* Contents: This file contains functions for user messaging.
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

/*
 * Put the user defined include files required.
 */

#include <stdio.h>
#include <dspcfg.h>
#include "IxDspCodelet.h"
#include "IxDspCodeletUsrMsgDef.h"
#include <assert.h>
 
#define MAX_TALKING_TERMS          2
#define MAX_LISTENERS              4
#define IX_DSP_CODELET_MSG_LINK_2  2
#define NUM_IP_COMPONENTS_4        4
#define MSG_NUMBER_TO_DECODE_1     1
#define MSG_NUMBER_TO_DECODE_2     2
#define MSG_NUMBER_TO_DECODE_3     3
#define MSG_NUMBER_TO_DECODE_4     4
#define MSG_NUMBER_TO_DECODE_5     5
#define NUM_USER_MSGS_2            2
#define THREE_WAY_CALL_MAX_PARTIES 3
#define THREE_WAY_CALL_PARTY_1     0
#define THREE_WAY_CALL_PARTY_2     1
#define THREE_WAY_CALL_PARTY_3     2

/* prototypes of message decoder and encoder functions */
typedef int (*IxDspCodeletMsgDec)(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
              int sequenceNo);
typedef void (*IxDspCodeletMsgEnc)(XMsgRef_t pUsrRply, XMsgRef_t pDspRply,
              int sequenceNo, UINT8 replyMsgType);

/* structure of message decoding/encoding table */
typedef struct{
    IxDspCodeletMsgDec  decFunc;
    IxDspCodeletMsgEnc  encFunc;
    UINT8   replyMsgType;
} IxDspCodeletMsgCodeTable;

static int msgDecLink(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecLinkBreak(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecLinkSwitch(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecStartStopIP(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSetupCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSetCallParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSetupCallwParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSwitchCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDec3wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecTeardown3wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecBackto2wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSetClearChan(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecT38Switch(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static int msgDecSetParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                      int sequenceNo);
static void msgEncAck(XMsgRef_t pUsrReply, XMsgRef_t pDspReply,
                      int sequenceNo, UINT8 replyMsgType);
static void msgEncStopAck(XMsgRef_t pUsrReply, XMsgRef_t pDspReply,
                          int sequenceNo, UINT8 replyMsgType);

static int msgDecUnavailable(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                             int sequenceNo, UINT8 replyMsgType)
{
    printf("*** ERROR : Decoder not implemented for the mssage "
           "(type=%d, index=%d) ***\n",
            pUsrMsg->type, pUsrMsg->type - IX_DSP_CODELET_MSG_TYPE_BEGIN);

    return IX_DSP_CODELET_MSG_ERR_MSG_TYPE;
}

static IxDspCodeletMsgCodeTable msgCodeTable[]=
{/*     This message decode/encode table, the order must
        be exactly the same as message type definition.

     msg dec func            msg enc func    reply msg type
----------------------------------------------------------------------------*/
    /*IX_DSP_CODELET_MSG_LINK*/
    {msgDecLink, msgEncAck, IX_DSP_CODELET_MSG_LINK_ACK},
    /*IX_DSP_CODELET_MSG_LINK_BREAK*/
    {msgDecLinkBreak, msgEncAck, IX_DSP_CODELET_MSG_LINK_ACK},
    /*IX_DSP_CODELET_MSG_LINK_SWITCH*/
    {msgDecLinkSwitch, msgEncAck, IX_DSP_CODELET_MSG_LINK_ACK},
    /*IX_DSP_CODELET_MSG_START_IP*/
    {msgDecStartStopIP, msgEncAck, IX_DSP_CODELET_MSG_SETUP_ACK},
    /*IX_DSP_CODELET_MSG_STOP_IP*/
    {msgDecStartStopIP, msgEncStopAck, IX_DSP_CODELET_MSG_STOP_ACK},
    /*IX_DSP_CODELET_MSG_SETUP_CALL*/
    {msgDecSetupCall, msgEncAck, IX_DSP_CODELET_MSG_SETUP_ACK},
    /*IX_DSP_CODELET_MSG_SET_CALL_PARMS*/
    {msgDecSetCallParms, msgEncAck, IX_DSP_CODELET_MSG_ACK},
    /*IX_DSP_CODELET_MSG_SETUP_CALLWPARMS*/
    {msgDecSetupCallwParms, msgEncAck, IX_DSP_CODELET_MSG_SETUP_ACK}, 
    /*IX_DSP_CODELET_MSG_SWITCH_CALL*/
    {msgDecSwitchCall, msgEncAck, IX_DSP_CODELET_MSG_LINK_ACK},
    /*IX_DSP_CODELET_MSG_CREATE_3WCALL*/
    {msgDec3wCall, msgEncAck, IX_DSP_CODELET_MSG_3W_ACK},
    /*IX_DSP_CODELET_MSG_EXIT_3WCALL*/
    {msgDec3wCall, msgEncAck, IX_DSP_CODELET_MSG_3W_ACK},
    /*IX_DSP_CODELET_MSG_TEARDOWN_3WCALL*/
    {msgDecTeardown3wCall, msgEncStopAck, IX_DSP_CODELET_MSG_STOP_ACK}, 
    /*IX_DSP_CODELET_MSG_BACKTO_2WCALL*/
    {msgDecBackto2wCall, msgEncAck, IX_DSP_CODELET_MSG_3W_ACK},
    /*IX_DSP_CODELET_MSG_SET_CLEAR_CHAN*/
    {msgDecSetClearChan, msgEncAck, IX_DSP_CODELET_MSG_ACK},
    /*IX_DSP_CODELET_MSG_T38_SWITCH*/
    {msgDecT38Switch, msgEncAck, IX_DSP_CODELET_MSG_T38_ACK},
    /*IX_DSP_CODELET_MSG_SET_PARMS*/
    {msgDecSetParms, msgEncAck, IX_DSP_CODELET_MSG_ACK} 
};


static  XDSPResConfig_t dspResCfg;
static int availableDSPCfg = 0;
static INT16 streamBase[IX_DSP_CODELET_TERM_EOL];
static UINT8 dspRes[] =
{
    XMPR_DEC,
    XMPR_ENC,
    XMPR_TNDET,
    XMPR_TNGEN,
    XMPR_NET,
    XMPR_MIX,
    XMPR_MA
};

/******************************************
    get DSP resource configuration
    The information is used to calculate
    T-Port stream IDs
*******************************************/
static void getDSPCfgInfo(void)
{
    xDspGetResConfig(&dspResCfg);

    /* get the stream ID base for all types of talkers */
    streamBase[IX_DSP_CODELET_TERM_NULL] = -1;
    streamBase[IX_DSP_CODELET_TERM_TDM] = dspResCfg.streamBaseTDM;
    streamBase[IX_DSP_CODELET_TERM_IP] = dspResCfg.streamBaseIP;
    streamBase[IX_DSP_CODELET_TERM_MIXER_PORT] = dspResCfg.streamBaseMix;
    availableDSPCfg = 1;
}

/******************************************
    Message Decoder function to be registered
    with Message Agent resource
*******************************************/
int ixDspCodeletMsgDecoder(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                           int sequenceNo)
{
    int isUsrMsg;
    int index;

    index = pUsrMsg->type;
    isUsrMsg = (index >= IX_DSP_CODELET_MSG_TYPE_BEGIN &&
                index < IX_DSP_CODELET_MSG_END_OF_OUTMSG);

    index -= IX_DSP_CODELET_MSG_TYPE_BEGIN;

    return isUsrMsg ?
        msgCodeTable[index].decFunc(pUsrMsg, pDecodedMsg, sequenceNo) :
        IX_DSP_CODELET_MSG_ERR_MSG_TYPE;
}

/******************************************
    Message Encoder function to be registered
    with Message Agent resource
*******************************************/
void ixDspCodeletMsgEncoder(XMsgRef_t pUsrReply, XMsgRef_t pDspReply,
                            int sequenceNo, UINT8 usrMsgType)
{
    if (usrMsgType >= IX_DSP_CODELET_MSG_TYPE_BEGIN &&
        usrMsgType < IX_DSP_CODELET_MSG_END_OF_OUTMSG)
    {
        /* use msg-specifc encoder */
        usrMsgType -= IX_DSP_CODELET_MSG_TYPE_BEGIN;
        msgCodeTable[usrMsgType].encFunc(
            pUsrReply,
            pDspReply,
            sequenceNo,
            msgCodeTable[usrMsgType].replyMsgType);
    }
    else
    {
        /* use general ACK encoder if unknown message */
        msgEncAck(pUsrReply, pDspReply, sequenceNo, IX_DSP_CODELET_MSG_ACK);
    }
}

/******************************************
    Message Encoder function to composite
    user's acknowledge message
*******************************************/
static void msgEncAck(XMsgRef_t pUsrReply, XMsgRef_t pDspReply, int sequenceNo,
                      UINT8 replyMsgType)
{
    IxDspCodeletMsgAck *pMsgAck;

    pMsgAck = (IxDspCodeletMsgAck *)pUsrReply;

    if(sequenceNo == XMSG_MA_ENCODING_INIT){
        pMsgAck->numDspReplies = 0;
        pMsgAck->numErrors = 0;
        return;
    }

    if(sequenceNo == XMSG_MA_ENCODING_CMPLT){

        XMSG_MA_MAKE_HEADER(
            pUsrReply,
            0,                      /* trans ID, will be overwritten by MA */
            replyMsgType,
            sizeof(IxDspCodeletMsgAck))

        return;
    }

    pMsgAck->numDspReplies++;

    if(pDspReply->type == XMSG_ERROR)
    {
        int n, temp;

        n = pMsgAck->numErrors;

        if(n == IX_DSP_CODELET_MSG_ERR_OVERFLOW)
        {
            return;
        }

        pMsgAck->error[n].dspResource = pDspReply->resource;
        pMsgAck->error[n].dspResInstance = pDspReply->instance;

        XMSG_FIELD_ERROR(pDspReply,
            pMsgAck->error[n].errCode,
            pMsgAck->error[n].errData,
            temp)

        pMsgAck->numErrors++;

        if(pMsgAck->numErrors >= IX_DSP_CODELET_MAX_ERR_REPLY)
        {
            pMsgAck->numErrors = IX_DSP_CODELET_MSG_ERR_OVERFLOW;
        }
    }
}


/******************************************
    Message Encoder function to composite
    user's stop-acknowledge message
*******************************************/
static void msgEncStopAck(XMsgRef_t pUsrReply, XMsgRef_t pDspReply,
                          int sequenceNo, UINT8 replyMsgType)
{
    IxDspCodeletMsgStopAck *pMsgStopAck;

    pMsgStopAck = (IxDspCodeletMsgStopAck *)pUsrReply;

    msgEncAck(pUsrReply, pDspReply, sequenceNo, replyMsgType);

    if(sequenceNo == XMSG_MA_ENCODING_INIT)
    {
        pMsgStopAck->numStopAck = 0;
        return;
    }

    if(sequenceNo == XMSG_MA_ENCODING_CMPLT){

        XMSG_MA_MAKE_HEADER(
            pUsrReply,
            0,                      /* trans ID, will be overwritten by MA */
            IX_DSP_CODELET_MSG_STOP_ACK,
            sizeof(IxDspCodeletMsgStopAck))

            return;
    }

    if(pDspReply->type == XMSG_CODER_STOP_ACK)
    {
        int n, temp;

        n = pMsgStopAck->numStopAck;

        if(n == IX_DSP_CODELET_MSG_ERR_OVERFLOW)
        {
            return;
        }

        pMsgStopAck->stopAck[n].dspResource = pDspReply->resource;
        pMsgStopAck->stopAck[n].dspResInstance = pDspReply->instance;

        XMSG_FIELD_CODEC_STOP_ACK
        (
            pDspReply,
            pMsgStopAck->stopAck[n].totalFrames,
            temp
        )

        pMsgStopAck->numStopAck++;

        if(pMsgStopAck->numStopAck >= IX_DSP_CODELET_MAX_STOP_CMPLT)
        {
            pMsgStopAck->numStopAck = IX_DSP_CODELET_MSG_ERR_OVERFLOW;
        }
    }
}

/******************************************
    Message Decoder function to decomposite
    user's link message
*******************************************/
static int msgDecLink(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg, int sequenceNo)
{
    /* the message will be decoded to up to 4 native DSP massages */

    int termA, termB;
    IxDspCodeletMsgLink *pUsrMsgW;
    struct{                     /* temp working structure */
        IxDspCodeletTerm term;  /* term ID */
        UINT16 talker;          /* talker's stream ID of the termination */
    } wTerms[MAX_TALKING_TERMS];

    /* static information, used during whole decoding procedure */
    static int numListeners;
    static struct{              /* listeners involved in the connection */
        UINT8   dspRes;
        UINT8   dspResInstance;
        UINT16  parmIdLPort;
        UINT16  listenTo;
    } listeners[MAX_LISTENERS];

    if(!availableDSPCfg)
    {
      getDSPCfgInfo();
    }

    pUsrMsgW = (IxDspCodeletMsgLink *)pUsrMsg;

    /* resolve the listener's information once before decoding */
    if(sequenceNo == 1)
    {
        /* copy the termination IDs to working data structure */
        wTerms[0].term = pUsrMsgW->term1;
        wTerms[1].term = pUsrMsgW->term2;

        /* calculate the talker's stream IDs, set to -1 if null termination */
        wTerms[0].talker =
            (pUsrMsgW->term1.type != IX_DSP_CODELET_TERM_NULL) ?
            (streamBase[pUsrMsgW->term1.type] + pUsrMsgW->term1.channel - 1) :
            (-1);

        wTerms[1].talker =
            (pUsrMsgW->term2.type != IX_DSP_CODELET_TERM_NULL) ?
            (streamBase[pUsrMsgW->term2.type] + pUsrMsgW->term2.channel - 1) :
            (-1);

        /* find the listener's information in each termination */
        for(numListeners=0, termA=0; termA<MAX_TALKING_TERMS; termA ++){

            /* the index of the opposite termination */
            termB = 1 - termA;

            switch(wTerms[termA].term.type)
            {
            case IX_DSP_CODELET_TERM_TDM:
                listeners[numListeners].dspRes = XMPR_NET;
                listeners[numListeners].dspResInstance =
                   wTerms[termA].term.channel;
                listeners[numListeners].parmIdLPort = XPARMID_NET_LP_STREAM;
                listeners[numListeners++].listenTo = wTerms[termB].talker;
            break;

            case IX_DSP_CODELET_TERM_IP:
                listeners[numListeners].dspRes = XMPR_ENC;
                listeners[numListeners].dspResInstance = 
                   wTerms[termA].term.channel;
                listeners[numListeners].parmIdLPort = XPARMID_ENC_LP_STREAM;
                listeners[numListeners++].listenTo = wTerms[termB].talker;

                listeners[numListeners].dspRes = XMPR_TNDET;
                listeners[numListeners].dspResInstance =
                   wTerms[termA].term.channel;
                listeners[numListeners].parmIdLPort = XPARMID_TD_LP_STREAM;
                listeners[numListeners++].listenTo = wTerms[termB].talker;
            break;

            case IX_DSP_CODELET_TERM_MIXER_PORT:
                listeners[numListeners].dspRes = XMPR_MIX;
                listeners[numListeners].dspResInstance = 1;
                listeners[numListeners].parmIdLPort = XPARMID_MIX_LP_STREAM
                   + wTerms[termA].term.channel - 1;
                listeners[numListeners++].listenTo = wTerms[termB].talker;
            break;

            case IX_DSP_CODELET_TERM_NULL:
            break;

            default:
                return IX_DSP_CODELET_MSG_ERR_MSG_TYPE;
            }
        }
    }

    if(sequenceNo <= numListeners)
    {
        XMSG_MAKE_SET_PARM
        (
            pDecodedMsg,
            pUsrMsg->transactionId,
            listeners[sequenceNo - 1].dspRes,
            listeners[sequenceNo - 1].dspResInstance,
            listeners[sequenceNo - 1].parmIdLPort,
            listeners[sequenceNo - 1].listenTo
        )
    }

    return numListeners - sequenceNo;
}

/******************************************
    Message Decoder function to decomposite
    user's link-break message
*******************************************/
static int msgDecLinkBreak(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                           int sequenceNo)
{
    /* the message will be decoded to 2 IX_DSP_CODELET_MSG_LINK messages
     * that link the 2 'term1' and 'term2' to null */

    int numDecodedMsg = MAX_TALKING_TERMS;
    IxDspCodeletMsgLink *pDecodedMsgW;
    IxDspCodeletMsgLinkBreak *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgLinkBreak *)pUsrMsg;
 
    pDecodedMsgW = (IxDspCodeletMsgLink *)pDecodedMsg;

    /* copy the message header after a NULL pointer check */

    RETURN_IF_NULL_POINTER((pDecodedMsgW), IX_DSP_CODELET_MSG_ERR_INVALID_PARM)
    RETURN_IF_NULL_POINTER((pUsrMsgW), IX_DSP_CODELET_MSG_ERR_INVALID_PARM)

    *pDecodedMsgW = *pUsrMsgW;

    pDecodedMsg->type = IX_DSP_CODELET_MSG_LINK;
 
    if(sequenceNo == 1){
        /* copy the message since the two has the same structure */
        *pDecodedMsgW = *pUsrMsgW;

        /* change the type of the decoded msg */
        pDecodedMsg->type = IX_DSP_CODELET_MSG_LINK;

        /* link term1 to null */
        pDecodedMsgW->term2.type = IX_DSP_CODELET_TERM_NULL;
    }
    else
    {
        /* link term2 to null */
        pDecodedMsgW->term1.type = IX_DSP_CODELET_TERM_NULL;
    }

    return numDecodedMsg - sequenceNo;
}


/******************************************
    Message Decoder function to decomposite
    user's link-switch message
*******************************************/
static int msgDecLinkSwitch(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                            int sequenceNo)
{
    /* the message will be decoded to 2 IX_DSP_CODELET_MSG_LINK messages.
     * the 1st one link 'switchFrom' to null and the 2nd one links 'term'
     * to 'switchTo' */

    int numDecodedMsg = IX_DSP_CODELET_MSG_LINK_2;
    IxDspCodeletMsgLink *pDecodedMsgW;
    IxDspCodeletMsgLinkSwitch *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgLinkSwitch *)pUsrMsg;
    pDecodedMsgW = (IxDspCodeletMsgLink *)pDecodedMsg;

    if(sequenceNo == 1)
    {
        /* biuld the message header once by copying */
        *pDecodedMsg = *pUsrMsg;
        pDecodedMsg->type = IX_DSP_CODELET_MSG_LINK;

        /* link switchFrom to null */
        pDecodedMsgW->term1 = pUsrMsgW->switchFrom;
        pDecodedMsgW->term2.type = IX_DSP_CODELET_TERM_NULL;
    }
    else
    {
        /* link term to switchTo */
        pDecodedMsgW->term1 = pUsrMsgW->term;
        pDecodedMsgW->term2 = pUsrMsgW->switchTo;
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's start/stop IP termination message
**********************************************/
static int msgDecStartStopIP(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                             int sequenceNo)
{
    /* this message will be decoded to 0 ~ 4 DSP messages that
     * start or stop ENC, DEC and TD respectively, TG is always stopped */

    static int numDspMsg;
    static UINT8 type;
    static UINT8 dspInst;
    static UINT8 dspResToChange[NUM_IP_COMPONENTS_4];
    int i;
    UINT16 state;

    if(sequenceNo == 1)
    {
        dspInst = ((IxDspCodeletMsgStartIP *)pUsrMsg)->channel;

        numDspMsg = 0;

        if(pUsrMsg->type == IX_DSP_CODELET_MSG_START_IP)
        {
            type = XMSG_START;

            /* find the DSP resources to be started, not including TG */
            for(i=0; i<(NUM_IP_COMPONENTS_4-1); i++)
            {
                xDspParmRead(dspRes[i], dspInst, XPARMID_RES_STATE, &state);
                if(state == 0)
                {
                    dspResToChange[numDspMsg++] = dspRes[i];
                }
            }
        }
        else
        {
            type = XMSG_STOP;

            /* find the DSP resources to be stopped */
            for(i=0; i<(NUM_IP_COMPONENTS_4-1); i++)
            {
                xDspParmRead(dspRes[i], dspInst, XPARMID_RES_STATE, &state);
                if(state != 0)
                {
                    dspResToChange[numDspMsg++] = dspRes[i];
                }
            }
        }

        /* always stop TG */
        xDspParmRead(XMPR_TNGEN, dspInst, XPARMID_RES_STATE, &state);
        if(state != 0)
        {
            dspResToChange[numDspMsg++] = XMPR_TNGEN;
        }
    }

    XMSG_MAKE_HEAD
    (
        pDecodedMsg,
        pUsrMsg->transactionId,
        dspResToChange[sequenceNo - 1],
        dspInst,
        sizeof(XMsgHdr_t),
        dspResToChange[sequenceNo - 1] == XMPR_TNGEN ? XMSG_STOP : type,
        0
    )

    return numDspMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's call setup message
**********************************************/
static int msgDecSetupCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                           int sequenceNo)
{
    /* the message will be decoded to 2 user messages.
     * the 1st one links the TDM and IP terminations and
     * the 2nd one start the IP termination */

    int numDecodedMsg = IX_DSP_CODELET_MSG_LINK_2;
    IxDspCodeletMsgLink *pDecodedMsgW1;
    IxDspCodeletMsgSetupCall *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgSetupCall *)pUsrMsg;

    if(sequenceNo == 1)
    {
        IX_DSP_CODELET_MAKE_MSGHDR_LINK(pDecodedMsg, pUsrMsg->transactionId)

        /* link TDM and IP terminations */
        pDecodedMsgW1 = (IxDspCodeletMsgLink *)pDecodedMsg;
        pDecodedMsgW1->term1.type = IX_DSP_CODELET_TERM_TDM;
        pDecodedMsgW1->term1.channel = pUsrMsgW->channelTDM;
        pDecodedMsgW1->term2.type = IX_DSP_CODELET_TERM_IP;
        pDecodedMsgW1->term2.channel = pUsrMsgW->channelIP;
    }
    else
    {
        /* start the IP termination */
        IX_DSP_CODELET_MAKE_MSG_START_IP
        (
            pDecodedMsg,
            pUsrMsg->transactionId,
            pUsrMsgW->channelIP
        )
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's set-call-parms message
**********************************************/
static int msgDecSetCallParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                              int sequenceNo)
{
    /* this message will be decoded to 4 DSP messages that
     * set parameters to DEC, INC, TD and TG resources respectively */

    static int numDspMsg = NUM_IP_COMPONENTS_4;
    static UINT16 *pParmID;
    static UINT16 *pParmVal;
    static UINT8 dspResInstance;
    UINT16 numParms = 0;

    switch(sequenceNo)
    {
    case MSG_NUMBER_TO_DECODE_1:

        dspResInstance = ((IxDspCodeletSetCallParms *)pUsrMsg)->channelIP;

        XMSG_FIELD_SET_MPARMS(pDecodedMsg, pParmID, pParmVal)

        /* call parameters for DEC resource */
        pParmID[numParms] = XPARMID_DEC_CTYPE;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.decType;
        pParmID[numParms] = XPARMID_DEC_AUTOSW;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.decAutoSwitch;
        break;

    case MSG_NUMBER_TO_DECODE_2:

        /* call parameters for NEC resource */
        pParmID[numParms] = XPARMID_ENC_CTYPE;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.encType;
        pParmID[numParms] = XPARMID_ENC_VAD;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.vad;
        pParmID[numParms] = XPARMID_ENC_MFPP;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.frmsPerPkt;
        break;

    case MSG_NUMBER_TO_DECODE_3:

        /* call parameters for TD resource */
        pParmID[numParms] = XPARMID_TD_RFC2833E_ENABLE;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.rfc2833;
        pParmID[numParms] = XPARMID_TD_RFC2833E_PAYLOADTYPE;
        pParmVal[numParms++] =
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.rfc2833pyldType;
        pParmID[numParms] = XPARMID_TD_TC;
        pParmVal[numParms++] = 
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.toneClamp;
        pParmID[numParms] = XPARMID_TD_RPT_EVENTS;
        pParmVal[numParms++] = 
          ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.toneReport;
       break;

    case MSG_NUMBER_TO_DECODE_4:

        /* call parameters for TG resource */
        pParmID[numParms] = XPARMID_TNGEN_RFC2833;
        pParmVal[numParms++] = 
           ((IxDspCodeletSetCallParms *)pUsrMsg)->parms.rfc2833;
        break;
    default:
            ; /* No default */
    }

    XMSG_MAKE_SET_MPARMS
    (
        pDecodedMsg,
        pUsrMsg->transactionId,
        dspRes[sequenceNo - 1],
        dspResInstance,
        numParms
    )

    return numDspMsg - sequenceNo;
}


/*********************************************
    Message Decoder function to decomposite
    user's call-setup-with-parms message
**********************************************/
static int msgDecSetupCallwParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                                 int sequenceNo)
{
    /* the message will be decoded to 2 user messages.
     * the 1st one set call parameters to IP termination
     * the 2nd one setup the call */

    int numDecodedMsg = NUM_USER_MSGS_2;
    IxDspCodeletSetCallParms *pDecodedMsgW1;
    IxDspCodeletMsgSetupCallwParms *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgSetupCallwParms *)pUsrMsg;


    if(sequenceNo == 1)
    {
        pDecodedMsgW1 = (IxDspCodeletSetCallParms *)pDecodedMsg;
        IX_DSP_CODELET_MAKE_MSGHDR_SET_CALL_PARMS(pDecodedMsg, pUsrMsg->transactionId)
        pDecodedMsgW1->channelIP = pUsrMsgW->channelIP;
        pDecodedMsgW1->parms = pUsrMsgW->parms;
    }
    else
    {
        IX_DSP_CODELET_MAKE_MSG_SETUP_CALL
        (
            pDecodedMsg,
            pUsrMsg->transactionId,
            pUsrMsgW->channelIP,
            pUsrMsgW->channelTDM
        )
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's swith-call message
**********************************************/
static int msgDecSwitchCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                            int sequenceNo)
{
    /* the message will be decoded to 2 user messages.
     * the 1st one switch the TDM termination to another IP
     * termination and the 2nd one setup a new call */

    int numDecodedMsg = NUM_USER_MSGS_2;
    IxDspCodeletMsgLinkSwitch *pDecodedMsgW1;
    IxDspCodeletMsgSwitchCall *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgSwitchCall *)pUsrMsg;

    if(sequenceNo == 1)
    {
        /* TDM term to the new IP term and put original IP term on hold */
        IX_DSP_CODELET_MAKE_MSGHDR_LINK_SWITCH(pDecodedMsg, pUsrMsg->transactionId)

        pDecodedMsgW1 = (IxDspCodeletMsgLinkSwitch *)pDecodedMsg;
        pDecodedMsgW1->term.type = IX_DSP_CODELET_TERM_TDM;
        pDecodedMsgW1->term.channel = pUsrMsgW->channelTDM;
        pDecodedMsgW1->switchFrom.type = IX_DSP_CODELET_TERM_IP;
        pDecodedMsgW1->switchFrom.channel = pUsrMsgW->ipChanOnHold;
        pDecodedMsgW1->switchTo.type = IX_DSP_CODELET_TERM_IP;
        pDecodedMsgW1->switchTo.channel = pUsrMsgW->ipChanNewCall;
    }
    else
    {
        /* setup a new call */
        IX_DSP_CODELET_MAKE_MSG_SETUP_CALL
        (
            pDecodedMsg,
            pUsrMsg->transactionId,
            pUsrMsgW->ipChanNewCall,
            pUsrMsgW->channelTDM
        )
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's create/exit 3-way-call message
**********************************************/
static int msgDec3wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                        int sequenceNo)
{
    /* the message will be decoded to 3 user messages.
     * each of them links/breaks a party to a port of the mixer */

    IxDspCodeletMsgLink *pDecodedMsgW;
    IxDspCodeletMsgCreate3wCall *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgCreate3wCall *)pUsrMsg;
    pDecodedMsgW = (IxDspCodeletMsgLink *)pDecodedMsg;

    /* buid message header */
    if(sequenceNo == 1){
        if(pUsrMsg->type == IX_DSP_CODELET_MSG_CREATE_3WCALL)
        {
            IX_DSP_CODELET_MAKE_MSGHDR_LINK(pDecodedMsg, pUsrMsg->transactionId)
        }
        else
        {
            IX_DSP_CODELET_MAKE_MSGHDR_LINK_BREAK(pDecodedMsg, pUsrMsg->transactionId)
        }
    }

    if(sequenceNo < MSG_NUMBER_TO_DECODE_4)
    {
        /* link or break the parites to the ports of the mixer */
        pDecodedMsgW->term1 = pUsrMsgW->parties[sequenceNo - 1];
        pDecodedMsgW->term2.type = IX_DSP_CODELET_TERM_MIXER_PORT;
        pDecodedMsgW->term2.channel = sequenceNo;
    }

    if(sequenceNo == MSG_NUMBER_TO_DECODE_4)
    {
        /* start or stop Mixer resource */
        XMSG_MAKE_HEAD
        (
            pDecodedMsg,
            pUsrMsg->transactionId,
            XMPR_MIX,
            1,
            sizeof(XMsgHdr_t),
            pUsrMsg->type == IX_DSP_CODELET_MSG_CREATE_3WCALL ? XMSG_START :
                              XMSG_STOP, 0
        )
    }

    return (MSG_NUMBER_TO_DECODE_4 - sequenceNo);
}

/*********************************************
    Message Decoder function to decomposite
    user's teardown-3-way-call message
**********************************************/
static int msgDecTeardown3wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                                int sequenceNo)
{
    /* the message will be decoded to 1+ user messages.
     * the 1st one exit the 3-way call and following stop IP terminations */

    static int numDecodedMsg;
    static int numTermsIP;
    static int i;
    IxDspCodeletMsgTeardown3wCall *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgTeardown3wCall *)pUsrMsg;

    if(sequenceNo == 1)
    {
        numDecodedMsg = 1;

        /* buid the message by copy */
        *((IxDspCodeletMsgExit3wCall *)pDecodedMsg) = *pUsrMsgW;

        pDecodedMsg->type = IX_DSP_CODELET_MSG_EXIT_3WCALL;

        /* find number of IP terminations */
        for(numTermsIP=0, i=0; i<THREE_WAY_CALL_MAX_PARTIES; i++)
        {
            if(pUsrMsgW->parties[i].type == IX_DSP_CODELET_TERM_IP)
            {
               numTermsIP++;
            }
        }

        numDecodedMsg += numTermsIP;
        i = 0;
    }
    else
    {

         /* stop parties if they are IP terminations */
        while(pUsrMsgW->parties[i].type != IX_DSP_CODELET_TERM_IP)
        {
          i++;
        }

        IX_DSP_CODELET_MAKE_MSG_STOP_IP
            (
                pDecodedMsg,
                pUsrMsg->transactionId,
                pUsrMsgW->parties[i].channel
            )

        i++;
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's back-to-2-way-call message
**********************************************/
static int msgDecBackto2wCall(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                              int sequenceNo)
{
    /* the message will be decoded to 2+ user messages.
     * the 1st exits the 3-way call, the 2nd links two terms
     * and the 3rd stop the dropped term if it is an IP term */

    static int numDecodedMsg;
    IxDspCodeletMsgBackto2wCall *pUsrMsgW;
    IxDspCodeletMsgExit3wCall *pDecodedMsgW1;
    IxDspCodeletMsgLink *pDecodedMsgW2;

    pUsrMsgW = (IxDspCodeletMsgBackto2wCall *)pUsrMsg;

    if(sequenceNo == MSG_NUMBER_TO_DECODE_1)
    {
        numDecodedMsg = NUM_USER_MSGS_2;
        pDecodedMsgW1 = (IxDspCodeletMsgExit3wCall *)pDecodedMsg;

        /* exit the 3-way call */
      IX_DSP_CODELET_MAKE_MSGHDR_EXIT_3WCALL(pDecodedMsg,pUsrMsg->transactionId)
        pDecodedMsgW1->parties[THREE_WAY_CALL_PARTY_1] = pUsrMsgW->partyToDrop;
        pDecodedMsgW1->parties[THREE_WAY_CALL_PARTY_2] = pUsrMsgW->party1;
        pDecodedMsgW1->parties[THREE_WAY_CALL_PARTY_3] = pUsrMsgW->party2;
    }
    else if(sequenceNo == MSG_NUMBER_TO_DECODE_2)
    {
        pDecodedMsgW2 = (IxDspCodeletMsgLink *)pDecodedMsg;

        /* link party1 and party2 */
        IX_DSP_CODELET_MAKE_MSGHDR_LINK(pDecodedMsg, pUsrMsg->transactionId)

        pDecodedMsgW2->term1 = pUsrMsgW->party1;
        pDecodedMsgW2->term2 = pUsrMsgW->party2;

    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's set-clear-channel message
**********************************************/
static int msgDecSetClearChan(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                              int sequenceNo)
{
    /* this message will be decoded to 5 DSP messages that
     * set parameters to DEC, INC, TD, TG and NET resources respectively */

    static int numDspMsg = MSG_NUMBER_TO_DECODE_5;
    static UINT16 *pParmID;
    static UINT16 *pParmVal;
    static UINT16 cType;
    static UINT8 dspResInstance;
    UINT16 numParms = 0;

    switch(sequenceNo)
    {
    case MSG_NUMBER_TO_DECODE_1:

        cType = ((IxDspCodeletMsgSetClearChan *)pUsrMsg)->codeType;
        if(cType != XCODER_TYPE_G711MU_10MS && cType != XCODER_TYPE_G711A_10MS)
        {
            return IX_DSP_CODELET_MSG_ERR_INVALID_PARM;
        }

        dspResInstance = ((IxDspCodeletMsgSetClearChan *)pUsrMsg)->channelIP;

        XMSG_FIELD_SET_MPARMS(pDecodedMsg, pParmID, pParmVal)

        /* parameters for DEC resource */
        pParmID[numParms] = XPARMID_DEC_AUTOSW;
        pParmVal[numParms++] = XPARM_DEC_AUTOSW_G711MU |XPARM_DEC_AUTOSW_G711A;
        pParmID[numParms] = XPARMID_DEC_VOL;
        pParmVal[numParms++] = 0;
        pParmID[numParms] = XPARMID_DEC_ALC;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_DEC_CNG;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_DEC_CTYPE;
        pParmVal[numParms++] = cType;
        break;

    case MSG_NUMBER_TO_DECODE_2:

        /* parameters for NEC resource */
        pParmID[numParms] = XPARMID_ENC_AGC;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_ENC_VAD;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_ENC_CTYPE;
        pParmVal[numParms++] = cType;
        break;

    case MSG_NUMBER_TO_DECODE_3:

        /* parameters for TD resource */
        pParmID[numParms] = XPARMID_TD_TC;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_TD_RPT_EVENTS;
        pParmVal[numParms++] = XPARM_OFF;
        pParmID[numParms] = XPARMID_TD_RFC2833E_ENABLE;
        pParmVal[numParms++] = XPARM_OFF;
        break;

    case MSG_NUMBER_TO_DECODE_4:

        /* parameters for TG resource */
        pParmID[numParms] = XPARMID_TNGEN_RFC2833;
        pParmVal[numParms++] = XPARM_OFF;
        break;

    case MSG_NUMBER_TO_DECODE_5:

        dspResInstance = ((IxDspCodeletMsgSetClearChan *)pUsrMsg)->channelTDM;

        /* parameters for NET resource */
        pParmID[numParms] = XPARMID_NET_ECENABLE;
        pParmVal[numParms++] = XPARM_OFF;
        break;
    default:
        break;
    }
 
    XMSG_MAKE_SET_MPARMS
    (
        pDecodedMsg,
        pUsrMsg->transactionId,
        dspRes[sequenceNo - 1],
        dspResInstance,
        numParms
    )

    return numDspMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's T38-switch message
**********************************************/
static int msgDecT38Switch(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                           int sequenceNo)
{
    /* if switch to fax mode, the message will be decoded to 1 user message
     * and 2 DSP messages. The first 2 message stop all the voice resources
     * (IP and TDM terms) and the 3rd one starts T38 resource.
     * if switch to voice mode, the message will be decoded to 2 DSP message
     * and one user message. the 1st one stops T38 resource and the 2nd and 3rd
     * start the voice resources (IP and TDM terms) */

    UINT16 state;
    int numDecodedMsg = MSG_NUMBER_TO_DECODE_3;
    IxDspCodeletMsgT38Switch *pUsrMsgW;

    pUsrMsgW = (IxDspCodeletMsgT38Switch *)pUsrMsg;

    if(pUsrMsgW->mode == IX_DSP_CODELET_MSG_MODE_T38)
    {
        /* switch to fax. first stop all voice resources then start 
           T38 resource */
        if(sequenceNo == MSG_NUMBER_TO_DECODE_1)
        {
            IX_DSP_CODELET_MAKE_MSG_STOP_IP(
                pDecodedMsg, pUsrMsg->transactionId, pUsrMsgW->channelIP)
        }
        else if(sequenceNo == MSG_NUMBER_TO_DECODE_2)
        {
            XMSG_MAKE_SET_PARM(
                pDecodedMsg,
                pUsrMsg->transactionId,
                XMPR_NET,
                pUsrMsgW->channelTDM,
                XPARMID_NET_ECENABLE,
                XPARM_OFF)
        }
        else
        {
            /* start the T.38 session */
            /* Extended start message indicating fax tone type. */
            XMSG_MAKE_T38_START(
                pDecodedMsg,
                pUsrMsg->transactionId,
                pUsrMsgW->channelIP,
                pUsrMsgW->tone);
        }
    }
    else
    {
        /* switch to voice. setup voice resource and stop T.38  */
        if(sequenceNo == MSG_NUMBER_TO_DECODE_1)
        {
            /* enable EC */
            XMSG_MAKE_SET_PARM(
                pDecodedMsg,
                pUsrMsg->transactionId,
                XMPR_NET,
                pUsrMsgW->channelTDM,
                XPARMID_NET_ECENABLE,
                XPARM_ON)
        }
        else if(sequenceNo == MSG_NUMBER_TO_DECODE_2)
        {
            /* stop the T.38 session */

            /* start IP termination */
            IX_DSP_CODELET_MAKE_MSG_START_IP(
                pDecodedMsg, pUsrMsg->transactionId, pUsrMsgW->channelIP)


             /* check if T.38 has been stopped */
             xDspParmRead(XMPR_T38, pUsrMsgW->channelIP, XPARMID_RES_STATE,
                          &state);
             if(state == 0)
             {
               numDecodedMsg--;
             }
        }
        else
        {
            /* stop the T.38 session */

            /* stop T.38 component if not stopped yet */
            XMSG_MAKE_STOP(
                pDecodedMsg,
                pUsrMsg->transactionId,
                XMPR_T38,
                pUsrMsgW->channelIP)
        }
    }

    return numDecodedMsg - sequenceNo;
}

/*********************************************
    Message Decoder function to decomposite
    user's set-parmeters message
**********************************************/
static int msgDecSetParms(XMsgRef_t pUsrMsg, XMsgRef_t pDecodedMsg,
                          int sequenceNo)
{
    static int i;
    IxDspCodeletMsgSetParms *pUsrMsgW;
    IxDspCodeletParm *pDspParm;

    pUsrMsgW = (IxDspCodeletMsgSetParms *)pUsrMsg;

    if(sequenceNo == 1)
    {
        if(pUsrMsgW->numParms > IX_DSP_CODELET_MAX_PARMS)
        {
            return IX_DSP_CODELET_MSG_ERR_TOOMANY_PARMs;
        }

        i=0;
    }

    pDspParm = pUsrMsgW->parms + i++;

    XMSG_MAKE_SET_PARM
    (
        pDecodedMsg,
        pUsrMsg->transactionId,
        pDspParm->dspResource,
        pDspParm->dspResInstance,
        pDspParm->parmID,
        pDspParm->value
    )

    return (pUsrMsgW->numParms - i);
}

