/*****************************************************************************
 * @file  IxFramerDriverUsrAPIs.c
 *
 *      Contents: This file has T1/E1 Framer Driver Specific Implementations
 *
 * -- Intel Copyright Notice --
 * 
 * Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors. The software is licensed under 
 * IXA SDK license.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * 
 ****************************************************************************/

/*
 * This file has IA specific Framer Driver User APIs 
 */

#include <pthread.h>
#include "IxDspCodeletAppLinuxInc.h"
#include "icp_framerdrv.h"
#include "IxDspCodelet.h"
#include "IxFramerDriverUsrAPIs.h" 

/******************************************************************************

  Define any global variables
 
******************************************************************************/

static pthread_t cbFrmThread;
static char cbFrmThreadEnable = THREAD_ENABLE;


/***************************************************************************/
/*
 * Function definition - ixFramerOnHook
 * Set the hook status of a Framer timeslot to On Hook. 
 */
int ixFramerOnHook(int chl)
{
  unsigned int timeslot = 0;
  unsigned int dev_id = DEFAULT_DEVICE_ID; 
  unsigned int line_id = DEFAULT_LINE_ID;
  uint32_t state = FRAMER_ONHOOK;
  icp_status_t result = ICP_STATUS_SUCCESS;
  int rc = IX_DSP_DRV_SUCCESS;

  /* Convert ADS Channel to Framer Timeslot.
   * Note that Framer Timeslots 0 and 16 are reserved. */

  switch (slic_framer_choice)
  {
    case IX_DSP_FRAMER_ONLY:
    {
      if ((ADS_BASE_CHANNEL<=chl) && (chl<=(RESERVED_CAS_TS-1)))
      {
         timeslot = chl;
      }
      else if ((RESERVED_CAS_TS<=chl) && (chl<=ADS_MAX_CHANNEL))
      {
         timeslot = (chl+1);
      }
      else
      {
         printf("Error: Invalid ADS channel number.\n");
      }
      break;
    }

    case IX_DSP_SLIC_AND_FRAMER:
    {
      if(chl<=(NUM_SLIC_SLOTS*NUM_FXS_CHIPS))
      {
        /* Shouldn't call this function for SLIC channels.*/
        printf("Error: ixFramerOnhook called for SLIC channel.\n");
      }
      else
      {
        /* Convert ADS channel to Framer timeslot. */
        if((MIXED_MODE_FRAMER_BASE_TS<=chl) && (chl<=(RESERVED_CAS_TS-1)))
        {
          timeslot = (chl-MIXED_MODE_FRAMER_OFFSET);
        }
        else if((RESERVED_CAS_TS<=chl) && (chl<=ADS_MAX_CHANNEL))
        {
          timeslot = (chl-MIXED_MODE_FRAMER_OFFSET+1);
        }
        else
        {
          printf("Error: Invalid ADS channel number.\n");
        }
      }
      break;
    }

    default:
    {
      printf("Error: ixFramerOnHook SLIC/Framer Choice\n");
    }
  } /* end of switch statement. */

  result = icp_FramerDrvSignalStateSet(dev_id, line_id, timeslot, state);
  if(result!=ICP_STATUS_SUCCESS)
    {
      rc = IX_DSP_DRV_FAIL;
      printf("icp_FramerDrvSignalStateSet [On Hook] failed\n");
      return (rc);
    }
  else
  {
    rc = IX_DSP_DRV_SUCCESS;
    return (rc);
  }
}

/***************************************************************************/
/*
 * Function definition - ixFramerOffHook
 * Set the hook status of a Framer timeslot to Off Hook. 
 */
int ixFramerOffHook(int chl)
{

  unsigned int timeslot = 0;
  unsigned int dev_id = DEFAULT_DEVICE_ID;
  unsigned int line_id = DEFAULT_LINE_ID;
  uint32_t state = FRAMER_OFFHOOK; 
  icp_status_t result = ICP_STATUS_SUCCESS;
  int rc = IX_DSP_DRV_SUCCESS;

  /* Convert ADS Channel to Framer Timeslot.
   * Note that Framer Timeslots 0 and 16 are reserved. */

  switch (slic_framer_choice)
  {
    case IX_DSP_FRAMER_ONLY:
    {
      if ((ADS_BASE_CHANNEL<=chl) && (chl<=(RESERVED_CAS_TS-1)))
      {
         timeslot = chl;
      }
      else if ((RESERVED_CAS_TS<=chl) && (chl<=ADS_MAX_CHANNEL))
      {
         timeslot = (chl+1);
      }
      else
      {
         printf("Error: Invalid ADS channel number.\n");
      }
      break;
    }
    
    case IX_DSP_SLIC_AND_FRAMER:
    {
      if(chl<=(NUM_SLIC_SLOTS*NUM_FXS_CHIPS))
      {
        /* Shouldn't call this function for SLIC channels. */
        printf("Error: ixFramerOffhook called for SLIC channel.\n");
      }
      else
      {
        /* Convert ADS channel to Framer timeslot. */
        if((MIXED_MODE_FRAMER_BASE_TS<=chl) && (chl<=(RESERVED_CAS_TS-1)))
        {
          timeslot = (chl-MIXED_MODE_FRAMER_OFFSET);
        }
        else if((RESERVED_CAS_TS<=chl) && (chl<=ADS_MAX_CHANNEL))
        {
          timeslot = (chl-MIXED_MODE_FRAMER_OFFSET+1);
        }
        else
        {
          printf("Error: Invalid ADS channel number.\n");
        }
      }
      break;
    }
    default:
    {
      printf("Error: ixFramerOffHook SLIC/Framer choice\n");
    }
  } /* end of switch statement. */

  result = icp_FramerDrvSignalStateSet(dev_id, line_id, timeslot, state);
  if(result!=ICP_STATUS_SUCCESS)
    {
      rc = IX_DSP_DRV_FAIL; 
      printf("icp_FramerDrvSignalStateSet [Off Hook] failed\n");
      return (rc);
    }
  else
  {
    rc = IX_DSP_DRV_SUCCESS; 
    return (rc);
  }
}

/***************************************************************************/
/*
 * Function definition - ixFramerHookStatus
 * Get the hook status of a Framer timeslot. 
 */
int ixFramerHookStatus(int channel)
{
  unsigned int timeslot = 0; /* Framer timeslot */
  int result = IX_SUCCESS;
  uint8_t hookstate = FRAMER_ONHOOK; /* Framer hookstate */ 
  uint8_t ads_hookstate = IX_SC_ON_HOOK;


  icp_framerdrv_event_t event = {0,0,0};
  event.lineId = DEFAULT_LINE_ID; 

  /* convert ADS channel to Framer timeslot */

  /* Framer only mode */
  if(IX_DSP_FRAMER_ONLY == slic_framer_choice) 
    {
      /* ADS Channel 1 - 15 map to Framer TS 1 - 15. ADS channel 16 - 24
       * maps to Framer TS 17 - 25. */
      if((ADS_BASE_CHANNEL<=channel) && (channel<=(RESERVED_CAS_TS-1)))
      {
        timeslot = channel;
      }
      else if((RESERVED_CAS_TS<=channel) && (channel<=ADS_MAX_CHANNEL)) 
      {
        timeslot = channel+1;
      }
      else
      {
        printf("Error: Invalid ADS channel number.\n");
        return (IX_DSP_DRV_FAIL);

      }
    }
    
  /* SLIC and Framer mode */
  if(IX_DSP_SLIC_AND_FRAMER==slic_framer_choice)
    {
      if((MIXED_MODE_FRAMER_BASE_TS<=channel) 
              && (channel<=(RESERVED_CAS_TS-1)))
    {
      timeslot = channel-MIXED_MODE_FRAMER_OFFSET;
    }
      else if((RESERVED_CAS_TS<=channel) && (channel<=ADS_MAX_CHANNEL ))
      {
        timeslot = channel-MIXED_MODE_FRAMER_OFFSET+1;
      }
      else
      {
        printf("Error: Invalid ADS channel number.\n");
        return (IX_DSP_DRV_FAIL);
      }
    }
  result = icp_FramerDrvSignalStateGet(DEFAULT_DEVICE_ID,DEFAULT_LINE_ID,
            timeslot, &hookstate);
  if (result != ICP_STATUS_SUCCESS)
    {
      printf("icp_FramerDrvSignalStateGet failed \n");
      return (IX_DSP_DRV_FAIL);
    }

  if(FRAMER_ONHOOK==hookstate)
    {
      printf("ON Hook: Channel %d\n", channel);
      ads_hookstate = IX_SC_ON_HOOK;
    }
  else if(FRAMER_OFFHOOK==hookstate)
    {
      printf("OFF Hook: Channel %d\n", channel);
      ads_hookstate = IX_SC_OFF_HOOK;
    }
  else
    {
      printf("Error: Invalid Hookstate \n");
      return (IX_DSP_DRV_FAIL); 
    }

  return ((int)ads_hookstate);
}

/***************************************************************************/
/*
 * Function definition - ixFramerInit
 * Initialize the Framer device
 */
int ixFramerInit(void)
{
  int rc = IX_DSP_DRV_SUCCESS;
  icp_status_t result = ICP_STATUS_SUCCESS;
  unsigned int dev_id = DEFAULT_DEVICE_ID;

  printf("Initialising Slot 2...\n");

  result = icp_FramerDrvInit(dev_id);

  if(result!=ICP_STATUS_SUCCESS)
    {
      rc = IX_DSP_DRV_FAIL;
      printf("icp_FramerDrvInit failed\n");
      return (rc);
    }

  return IX_DSP_DRV_SUCCESS;
}

/***************************************************************************/
/*
 * Function definition - ixFramerConfig
 * Configure the Framer device
 */

int ixFramerConfig(void)
{
  int rc = IX_DSP_DRV_SUCCESS;
  icp_status_t result = ICP_STATUS_SUCCESS;
  unsigned int dev_id = DEFAULT_DEVICE_ID;
  unsigned int line_id = DEFAULT_LINE_ID;
  /* Will only be using 1 E1 line */
  uint32_t events = ICP_FRAMERDRV_EVENT_COSS;

  /* Using one of the ICP_FRAMERDRV_CFG defaults (E1 CAS)  */
  unsigned int config = ICP_FRAMERDRV_CFG_E1_CAS_HDB3_CRCMF;

  result = icp_FramerDrvConfigSet(dev_id, config);

  if(result!=ICP_STATUS_SUCCESS)
    {
      rc = IX_DSP_DRV_FAIL;
      printf("icp_FramerDrvConfigSet failed.\n");
      return (rc);
    }

  /* Need to set what events on the Framer to check for */
  result = icp_FramerDrvNotificationsSet(dev_id, line_id, events);

  if(result!=ICP_STATUS_SUCCESS)
    {
      rc = IX_DSP_DRV_FAIL;
      printf("icp_FramerDrvNotificationsSet failed.\n");
      return (rc);
    }

  return IX_DSP_DRV_SUCCESS;

}


/***************************************************************************/
/*
 * Function definition - ixFramerRegCallback
 * Register the call back function for Framer
 */
int ixFramerRegCallback(void)
{

  int rc = IX_SUCCESS;
  if (initFramerCallbackThread() != IX_DSP_DRV_SUCCESS)
    {
      printf("Unable to create call back thread ...\n");
      return (IX_FAIL);
    }

  return rc;
}

/***************************************************************************/
/*
 * Function definition - initCallbackThread
 * Intialises the callback thread
 */
int initFramerCallbackThread(void)
{
  int result = IX_DSP_DRV_SUCCESS;
  cbFrmThreadEnable = THREAD_ENABLE;
  pthread_attr_t     thread_attr;
  struct sched_param  param;

  if(pthread_attr_init(&thread_attr) != 0)
  {
    printf("pthread_attr_init failure\n");
    return (IX_DSP_DRV_FAIL);
  }
  
  if (pthread_attr_setscope(&thread_attr, PTHREAD_SCOPE_SYSTEM) !=0)
  {
    printf("pthread_attr_setscope failure\n");
    return (IX_DSP_DRV_FAIL);
  }

  if (pthread_attr_setinheritsched(&thread_attr, PTHREAD_EXPLICIT_SCHED) !=0)
  {
    printf("pthread_attr_setinheritsched failure\n");
    return (IX_DSP_DRV_FAIL);
  }

  if (pthread_attr_setschedpolicy(&thread_attr, SCHED_RR) != 0)
  {
    printf("pthread_attr_setschedpolicy failure\n");
    return (IX_DSP_DRV_FAIL);
  }

  param.sched_priority = FRAMER_CB_THREAD_PRIORITY; 

  if (pthread_attr_setschedparam(&thread_attr, &param) !=0)
  {
    printf("pthread_attr_setschedparam failure\n");
    return (IX_DSP_DRV_FAIL);
  }

  if (pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED) !=0)
  {
    printf("pthread_attr_setdetachstate failure\n");
    return (IX_DSP_DRV_FAIL);
  }

  result = pthread_create(&cbFrmThread, &thread_attr,
            (pthread_startroutine_t)framerCallbackHandler, 0);

  if (result != 0)
    {
      printf("Unable to initialise the thread .. %d \n", result);
    }
  return result;
}

/*************************************************************************/
/*
 * Function definition - callbackHandler
 * Thread which invokes the command for the callback function. If the 
 * callback triggers , the read returns (otherwise it'll block) and the thread
 * will call the callback handler defined in user space based on the data 
 * received from kernel
 */ 
void * framerCallbackHandler(void *arg)
{
  int ads_hookstate = IX_SC_ON_HOOK;  /* Convert from Framer format 
                                       * to ADS format */
  int ads_chan_num = 0;  /* Convert from framer TS to ADS channel number */
  int result = IX_SUCCESS;
  uint8_t hookstate = FRAMER_ONHOOK;
  icp_boolean_t invalid_hookstate = ICP_FALSE;
  unsigned int timeslot = 0; /* Convert from bitmap to integer */
  icp_framerdrv_event_t event = {0,0,0};
  event.lineId = DEFAULT_LINE_ID;

  /* Set up thread attributes to allow it to be cancelled */
  if(pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL) != 0)
    {
      printf("Error setting up callback thread - setcancelstate\n");
      cbFrmThreadEnable = FALSE;
    }

  if(pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL) != 0)
    {
      printf("Error setting up callback thread - setcanceltype\n");
      cbFrmThreadEnable = FALSE;
    }

  /* Keep processing until the thread is killed */
  while (cbFrmThreadEnable)
  {
    result = icp_FramerDrvRead(DEFAULT_DEVICE_ID, &event, TRUE);
    if (result != ICP_STATUS_SUCCESS)
    {
      printf("icp_FramerDrvRead failed \n");
    }
    /* if result equals 0, then there should be a callback to process */
    if ((result == ICP_STATUS_SUCCESS) && 
           (event.eventType == ICP_FRAMERDRV_EVENT_COSS))
    {
        /* Convert from uint32_t bitmap to an unsigned int for timeslot */

        timeslot = 0; /* Reset this from previous COSS event */

        if (event.ts == 1)
        {
          timeslot = 0;
        }
        else
        {
          while (event.ts != 1)
          {
            event.ts = event.ts >>1;
            timeslot++;
          }
        }

        result = icp_FramerDrvSignalStateGet(DEFAULT_DEVICE_ID,event.lineId,
                  timeslot, &hookstate);
        if (result != ICP_STATUS_SUCCESS)
        {
          printf("icp_FramerDrvSignalStateGet failed \n");
        }
        else
        {
          if  (FRAMER_OFFHOOK == hookstate)
          {
            ads_hookstate = IX_SC_OFF_HOOK; 
          }
          else if (FRAMER_ONHOOK == hookstate)
          {
            ads_hookstate = IX_SC_ON_HOOK; 
          }
          else
          {
            printf("Invalid Hookstate %u (E&M signalling mode)\n", hookstate);
            invalid_hookstate = ICP_TRUE;
          }

          /* Now convert Framer event TS to ADS channel. */

          if (IX_DSP_FRAMER_ONLY==slic_framer_choice)
          {
            /* Implement chan map for Framer only...*/
            /* Convert Framer TS into ADS channel as follows:
             * TS 0 reserved for signalling.
             * TS 1 maps to channel 1,
             * TS 2 maps to channel 2 etc.
             * TS 16 reserved for signalling (CAS mode).
             * TS 17 maps to channel 16,
             * TS 25 maps to channel 24. */

            if(event.ts>RESERVED_TS && event.ts<RESERVED_CAS_TS)
            {      
              ads_chan_num = event.ts;
            }
            else if(event.ts>RESERVED_CAS_TS 
                    && event.ts<=(ADS_MAX_CHANNEL+1))
            {
              ads_chan_num = event.ts-1;
            }

          } /* end of Framer only */
          else if (IX_DSP_SLIC_AND_FRAMER==slic_framer_choice)
          {
            /* Implement chan map for mixed choice...*/
            /* Convert Framer TS into ADS channel as follows:
             * TS 0 reserved for signalling.
             * TS 1 maps to channel 9,
             * TS 2 maps to channel 10 etc. 
             * TS 16 reserved for signalling (CAS mode).
             * TS 17 maps to channel 24.
             * Note: Similar to Framer Only case but allowing
             * SLIC to use first 8 channels. */

            if(event.ts>RESERVED_TS && event.ts<RESERVED_CAS_TS)
            {
              ads_chan_num = event.ts+MIXED_MODE_FRAMER_OFFSET;
            }
            else if(event.ts>RESERVED_CAS_TS && event.ts<=MIXED_MODE_MAX_TS) 
            {
              ads_chan_num = event.ts+(MIXED_MODE_FRAMER_OFFSET-1);
            }
          } /* end of SLIC and Framer */

          if((ADS_BASE_CHANNEL<=ads_chan_num && ads_chan_num<=ADS_MAX_CHANNEL)
                 && ICP_FALSE==invalid_hookstate)
          {
            ixDspCodeletFramerNotifyHookCB(ads_hookstate, ads_chan_num);
          }
          ads_chan_num=0; /* reset for next iteration. */
          invalid_hookstate=ICP_FALSE; /* reset for next iteration */
        } /* end result = ICP_STATUS_SUCCESS */
      } /* end callback to process */
      else
      {
        /* No callbacks to process, so just sleep for a little while */
        printf("Error in framerCallback handler" 
                " wait 1 second and try again.\n");
        sleep(THREAD_SLEEP_TIME); /*This sleeps for 1 second */
   
      }
    } /* end while cbFrmThreadEnable */

  pthread_exit("Exiting Framer Callback Thread\n");
  return;
}

void cancelFramerCallbackThread(void){

   int rc=0;
   
   rc=pthread_cancel(cbFrmThread);
   if(rc!=0){
      printf("Error in cancelling framer callback thread\n");
   }
   
   return;
}

int ixFramerClose(void)
{
  /*Allowing the driver to exit gracefully*/
  icp_status_t result=0;
  unsigned int dev_id = DEFAULT_DEVICE_ID;

  printf("ixFramerClose called\n");

  result = icp_FramerDrvUninit(dev_id);
  if(result!=ICP_STATUS_SUCCESS)
    {
      printf("icp_FramerDrvUninit failed.\n");
      return (IX_DSP_DRV_FAIL);
    }
 
  return (IX_DSP_DRV_SUCCESS);
}

