/******************************************************************************
*       @file  IxHssDriverUsrAPIs.c 

*       Contents: This file has HSS Driver Specific Implementations
       
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

#include <stdio.h>
#include <sys/ioctl.h>

#include "IxDspCodelet.h"

#include "icp_hssvoicedrv.h"

#define UNUSED_FRAMER_TS_16 16


int voiceDrvfd=0;  

/**********************************************************
 * IxHssDriverHssPortInit brings up port0 and 1 for use
 * with a slic device and port2 for use with a framer 
 * device
 *********************************************************/
int IxHssDriverHssPortInit(Hss_config *port_config) 
{
   int rc=0;
   icp_hssdrv_portup_t hssPort={0,0,0};
  
    /* open hss Driver file for input and output */
    voiceDrvfd = open("/dev/hss-voice",O_RDWR);
   
   /* voice driver returns a file descriptor>0 if opened without error*/
    if(voiceDrvfd<=0)
    {
        printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");
        return IX_DSP_DRV_FAIL;
    }

    /* Initialize HSS Ports */
    /* Bring HSS port0 up for SLIC */
    hssPort.portId = IX_HSS_PORT_0;
    hssPort.port_config = ICP_HSSDRV_PORT_ANALOG_VOICE_MEZZANINE_CONFIG;
    if(((*port_config).hssportTx_Rx_config).loopback==FALSE){
       hssPort.loopbackMode = ICP_HSSDRV_NO_LOOPBACK;
    }else{
       hssPort.loopbackMode = ICP_HSSDRV_INTERNAL_LOOPBACK;
    } 
    
    rc=ioctl(voiceDrvfd, ICP_HSSVOICEDRV_PORT_UP, &hssPort);
    if(rc!=0)
    {

        printf("ICP_HSSVOICEDRV_PORT_UP failed for port 0\n");
        return IX_DSP_DRV_FAIL;
    }
    
    /* Bring HSS port1 up for SLIC */
    hssPort.portId = IX_HSS_PORT_1;
    hssPort.port_config = ICP_HSSDRV_PORT_ANALOG_VOICE_MEZZANINE_CONFIG;
    if(((*port_config).hssportTx_Rx_config).loopback==FALSE){
       hssPort.loopbackMode = ICP_HSSDRV_NO_LOOPBACK;
    }else{
       hssPort.loopbackMode = ICP_HSSDRV_INTERNAL_LOOPBACK;
    }
    
    rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_PORT_UP,&hssPort);
    if(rc!=0)
    {
        printf("ICP_HSSVOICEDRV_PORT_UP failed for port 1\n");
        return IX_DSP_DRV_FAIL;
    }

#ifndef IX_DSP_CODELET_SLIC_ONLY    
    /* Bring HSS port2 up for FRAMER if not in slic only mode*/
    hssPort.portId = IX_HSS_PORT_2;
    hssPort.port_config = ICP_HSSDRV_PORT_HMVIP_FRAMER_MEZZANINE_CONFIG;
    if(((*port_config).hssportTx_Rx_config).loopback==FALSE){
       hssPort.loopbackMode = ICP_HSSDRV_NO_LOOPBACK;
    }else{
       hssPort.loopbackMode = ICP_HSSDRV_INTERNAL_LOOPBACK;
    }
    
    rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_PORT_UP,&hssPort);
    if(rc!=0)
    {
        printf("ICP_HSSVOICEDRV_PORT_UP failed for port 2\n");
        return IX_DSP_DRV_FAIL;
    }    
#endif
    
    return IX_DSP_DRV_SUCCESS;
}

/***********************************************************
 * IxHssDriverHssConfig configures channels on ports 0,1 & 2
 * depending on user defined mode,band slic_framer_choice
 **********************************************************/
int IxHssDriverHssConfig(int mode,int band)
{
   int rc=0;
   icp_hssvoicedrv_channeladd_t hssChannel={0,0,0,{0,0,0,0},0,0,0,0,0};
   icp_hssdrv_timeslot_map_t map={0,0,0,0};
   int slot_cnt=0;
   int chip_cnt=0; 
   int count=0;
   int num_framer_chan=0;
   int chan_label_base=0;
   int ts_base_shiftL=0;
   int ts_base_shiftH=0;
   
   if(voiceDrvfd<=0)
    {
        printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");             
        return IX_DSP_DRV_FAIL;
    }
   
/* Checking the mode - band combo to ensure it is valid */

   if((band==USER_CHOICE_NB) && (mode==LINEAR_MODE)){
      printf("Narrowband linear channels are not supported\n");      
      return IX_DSP_DRV_FAIL;
   }   
   if((band==USER_CHOICE_WB) && ((mode==ALAW_MODE)||(mode==MULAW_MODE))){
      printf("Only linear channels are supported for wideband\n");
      return IX_DSP_DRV_FAIL;
   }
   
   if(slic_framer_choice>IX_DSP_SLIC_AND_FRAMER){
   
      printf("Hss Chan Add Error: Invalid Configuration.\n");
      return IX_DSP_DRV_FAIL;
   }
   
/* Next config all channels associated with both SLICs */
/* if slic_framer_choice=0 or 2                        */
/* In this case, the associated voice channels are labelled 1-8 by ADS */
/* The voice driver labels these channels (i.e. channelId) 0-7 */
 
  if((slic_framer_choice==IX_DSP_SLIC_ONLY) || 
       (slic_framer_choice==IX_DSP_SLIC_AND_FRAMER)){
 
 /* Next add all FXS channels associated with both SLICs   */
 
 /* NOTES ON channelId HARDCODING..................   */
 /* ADS Channel1-8 are given voice driver ADS 0-7 */
 
 /* NOTES on ts HARDCODING................................................*/
 /* These have to align with the offsets given in the SLIC init function  */
 /* We need to provide the voice driver with a bitmap with                */
 /* a 1 in the location of the active timeslots for a given channel       */ 
 /* Channel1 has an offset of 1 therefore map.line0=2 in the nb case     */
 /* Channel1 has an offset of 1 therefore map.line0=2^{1}+2^{2}+2^{17}   */
 /* +2^{18}=393222 in the wb case                                        */
   

/* How the voice driver channels are configured depends on band */ 
      switch(band)
      {
  
      /* Configure 4 narrow band channels per SLIC */ 
         case USER_CHOICE_NB:
         {
            for(slot_cnt=0; slot_cnt<NUM_SLIC_SLOTS; slot_cnt++){

               for(chip_cnt=0; chip_cnt<NUM_FXS_CHIPS; chip_cnt++){
                  map.line0_timeslot_bit_map=1<<(SLICBASE_TS+(SLIC_TS_SPACING
                                             *chip_cnt));

                  hssChannel.channelId = chip_cnt+(NUM_FXS_CHIPS*slot_cnt);
                  hssChannel.portId = slot_cnt;
                  hssChannel.voicePacketSize=IX_HSS_NB_PKT_SIZE;
                  hssChannel.tsMap = map;
                  hssChannel.voiceIdleAction = 0;
                  hssChannel.voiceIdlePattern = 0x00;
                  hssChannel.channelDataInvert = 0;
                  hssChannel.channelBitEndianness = 1;
                  hssChannel.channelByteSwap = 0;

                  rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_ADD,&hssChannel);
                  if (rc != 0)
                  {
                     printf("ICP_HSSVOICEDRV_CHAN_ADD ioctl failed "
                            " for slot %d, chip %d \n", slot_cnt, chip_cnt);
                     return IX_DSP_DRV_FAIL;
                  }
               }
            }
            break;
         }
      
   /* Configure 4 Linear PCM wide band channels on both SLICs */
         case USER_CHOICE_WB: 
         {
            for(slot_cnt=0; slot_cnt<NUM_SLIC_SLOTS; slot_cnt++){

               for(chip_cnt=0; chip_cnt<NUM_FXS_CHIPS; chip_cnt++){
          
                   ts_base_shiftL=SLICBASE_TS+(SLIC_TS_SPACING*chip_cnt);
                   ts_base_shiftH=SLIC_WB_TS_OFFSET+ts_base_shiftL;
                   map.line0_timeslot_bit_map=(1<<ts_base_shiftL)
                   +(1<<(ts_base_shiftL+1))+(1<<(ts_base_shiftH))
                   +(1<<(ts_base_shiftH+1)); 
              
                  hssChannel.channelId = chip_cnt+NUM_FXS_CHIPS*slot_cnt;
                  hssChannel.portId = slot_cnt;
                  hssChannel.voicePacketSize=IX_HSS_WB_PKT_SIZE;
                  hssChannel.tsMap = map;
                  hssChannel.voiceIdleAction = 0;
                  hssChannel.voiceIdlePattern = 0x00;
                  hssChannel.channelDataInvert = 0;
                  hssChannel.channelBitEndianness = 1;
                  hssChannel.channelByteSwap = 0;
      
                  rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_ADD,&hssChannel);
                  if (rc != 0)
                  {
                     printf("ICP_HSSVOICEDRV_CHAN_ADD ioctl failed "
                            "for slot %d, chip %d \n", slot_cnt, chip_cnt);
                     return IX_DSP_DRV_FAIL;
                  }
                }
            }
            break;
         }
    /* If an invalid mode-band combination is chosen */
         default:
         {
            printf("Hss Chan Add Error: Invalid Configuration.\n");
            return IX_DSP_DRV_FAIL;
         }
      }
   }
      
   /* If framer is choosen channels should be added to slot2 */
   if((slic_framer_choice==IX_DSP_SLIC_AND_FRAMER) 
       || (slic_framer_choice==IX_DSP_FRAMER_ONLY)){
   /* Framer related channels need to be added also */
   
      if(slic_framer_choice==IX_DSP_SLIC_AND_FRAMER){
         num_framer_chan=NUM_FRAMER_MIXED_CHAN; 
         chan_label_base=NUM_SLIC_SLOTS*NUM_FXS_CHIPS;     
      }else{
         num_framer_chan=NUM_FRAMER_CHAN;
         chan_label_base=0;
      }
      
      /* If NUM_FRAMER_MIXED_CHAN>30                  */ 
      /* then there are not enough timeslots on line0 */
      
      if(num_framer_chan>((BYTE_TO_BIT_CONV*sizeof(map.line0_timeslot_bit_map))
             -FRAMER_UNUSED_TS_NUM)||num_framer_chan<(UNUSED_FRAMER_TS_16-1)){
            
         printf("ICP_HSSVOICEDRV_CHAN_ADD can only support "
                "15-30 nb framer channels \n");  
         return IX_DSP_DRV_FAIL;          
      }

      switch(band)
      {
      /* Configure narrow band channels on port2 */      
         case USER_CHOICE_NB:
         {
            
            /*Timeslot map starts at ts0 here and all timeslots are adjacent*/
            /* Wide band not possible */
            /* ts0 and in some modes ts16 are used to transmit signal info */
            /* Therefore, we will not send voice information on ts0 and ts16*/
            /* on ports connected to the framer card */

            for(count=1; count<UNUSED_FRAMER_TS_16; count++){
     
               map.line0_timeslot_bit_map=1<<count;

               hssChannel.channelId=chan_label_base+count-1;
               hssChannel.portId = IX_HSS_PORT_2;
               hssChannel.voicePacketSize=IX_HSS_NB_PKT_SIZE;
               hssChannel.tsMap = map;
               hssChannel.voiceIdleAction = 0;
               hssChannel.voiceIdlePattern = 0x00;
               hssChannel.channelDataInvert = 0;
               hssChannel.channelBitEndianness = 1;
               hssChannel.channelByteSwap = 0;

               rc = ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_ADD,&hssChannel);
               if (rc != 0)
               {
                  printf("ICP_HSSVOICEDRV_CHAN_ADD ioctl failed for"
                          " framer count %d \n", count);
                  return IX_DSP_DRV_FAIL;
               }
            }
   
            for(count=UNUSED_FRAMER_TS_16; count<=num_framer_chan;count++){
       
               map.line0_timeslot_bit_map=1<<(count+1);

               hssChannel.channelId=chan_label_base+count-1;
               hssChannel.portId = IX_HSS_PORT_2;
               hssChannel.voicePacketSize=IX_HSS_NB_PKT_SIZE;
               hssChannel.tsMap = map;
               hssChannel.voiceIdleAction = 0;
               hssChannel.voiceIdlePattern = 0x00;
               hssChannel.channelDataInvert = 0;
               hssChannel.channelBitEndianness = 1;
               hssChannel.channelByteSwap = 0;

               rc = ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_ADD,&hssChannel);
               if (rc != 0)
               {
                  printf("ICP_HSSVOICEDRV_CHAN_ADD ioctl failed for framer"
                         " count %d \n", count);
                  return IX_DSP_DRV_FAIL;
                }
            }
            
         break;
         }
      
   /* Wide band choice invalid in framer and mixed modes
       - framer does not support*/
         case USER_CHOICE_WB:
         {
   
            printf("Wideband channels are not supported with"
                   " the framer/mixed option\n");
            return IX_DSP_DRV_FAIL;
 
         }
    /* If an invalid mode-band combination is chosen */
         default:
         {
            printf("Hss Chan Add Error: Invalid Configuration.\n");
            return IX_DSP_DRV_FAIL;
         } 
      }
   }

   return IX_DSP_DRV_SUCCESS; 
}

/*********************************************************
* Enable channels
*******************************************************/
int IxHssDriverChanEnable(void)
{
   int count=0;
   int rc=0;
   /* Need a temporary buffer to read in the data built up between 
      channel enables. */ 
   UINT8 drain_buff[MAX_READ_SIZE];
   
   if(voiceDrvfd<=0)
    {
        printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");             
        return IX_DSP_DRV_FAIL;
    }
  
   /* If slic_framer_choice=0 only 8 channels enabled */
   /* For either other option 24 channels enabled */
   for(count=(IX_DSP_CODELET_MAX_CHL-1); count>=0; count--){
   
      rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_UP,count);
      if(rc!=0)
      {
         printf("Error ICP_HSSVOICEDRV_CHAN_UP count=%d\n",count);
         return IX_DSP_DRV_FAIL;
      }
      rc=read(voiceDrvfd,drain_buff,MAX_READ_SIZE);
      if(rc<0){
         printf("Error in drain read. Return code %d\n",rc);
      }
   }

        
    return IX_DSP_DRV_SUCCESS; 
}

/********************************************************
* Disable previously enabled channels
*******************************************************/
int IxHssDriverChanDisable(void)
{

/*Disables channels*/
   int count=0;
   int rc=0;
   
   if(voiceDrvfd<=0)
    {
        printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");             
        return IX_DSP_DRV_FAIL;
    }

   for(count=0; count<IX_DSP_CODELET_MAX_CHL; count++){
       
      /* Bring down channel */
      rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_DOWN,count);
      if(rc!=0)
      {
         printf("Error ICP_HSSVOICEDRV_CHAN_DOWN count=%d\n",count);
         return IX_DSP_DRV_FAIL;
      }
            
   }
   
   
   return IX_DSP_DRV_SUCCESS;
}

/**********************************************************************
 * This function removes the channels brought up by IxHssDriverConfig 
 * and brings down the ports
 **********************************************************************/
int IxHssDriverDown(void)
{
   int count=0;
   int rc=0;
   int max_num_chan=0;
   
   if(voiceDrvfd<=0)
    {
        printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");             
        return IX_DSP_DRV_FAIL;
    }

   if(slic_framer_choice==IX_DSP_SLIC_ONLY){
       /* If just analog phones then only 8 channels */
         max_num_chan=NUM_SLIC_SLOTS*NUM_FXS_CHIPS;       
    }else{
       if((slic_framer_choice==IX_DSP_FRAMER_ONLY) || 
             (slic_framer_choice==IX_DSP_SLIC_AND_FRAMER)){
    /*If digital/E1 phones involved then 24 channels */
            max_num_chan=NUM_FRAMER_CHAN;
          
       }else{
            printf("Invalid Configuration, ensure valid "
                   "slic_framer_choice.\n");
            return IX_DSP_DRV_FAIL;
       }
    }

   for(count=0; count<max_num_chan; count++){
 
      rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_CHAN_REMOVE,count);
      if(rc!=0)
      {
          printf("Error ICP_HSSVOICEDRV_CHAN_REMOVE count=%d\n",count);
          return IX_DSP_DRV_FAIL;
      }
   }
    
    /* Next bring down the ports */
    /* Note 2 ports brought down if IX_DSP_CODELET_SLIC_ONLY */
    /* is defined else all 3 ports are brought down  */
    
   rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_PORT_DOWN,IX_HSS_PORT_0);
   if(rc!=0)
   {
      printf("ICP_HSSVOICEDRV_PORT_DOWN failed for port 0\n");
      return IX_DSP_DRV_FAIL;
   } 
   rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_PORT_DOWN,IX_HSS_PORT_1);
   if(rc!=0)
   {
      printf("ICP_HSSVOICEDRV_PORT_DOWN failed for port 1\n");
      return IX_DSP_DRV_FAIL;
   }

#ifndef IX_DSP_CODELET_SLIC_ONLY
   rc=ioctl(voiceDrvfd,ICP_HSSVOICEDRV_PORT_DOWN,IX_HSS_PORT_2);
   if(rc!=0)
   {
      printf("ICP_HSSVOICEDRV_PORT_DOWN failed for port 2\n");
      return IX_DSP_DRV_FAIL;
   }
#endif
    
   return IX_DSP_DRV_SUCCESS;    
}

/********************************************************************
 * Closes voice driver
 ******************************************************************/
int IxHssDriverClose(void)
{
   int rc=0;

   if(voiceDrvfd<=0)
   {
      printf("Open failed. Ensure module hssdriver module is inserted "
               "and /dev/hss-voice exists\n");             
      return IX_DSP_DRV_FAIL;
   }

   /* Allowing the driver to exit gracefully */
   printf("\nIxHssDriverClose called\n");

   /* close buffers */ 
   rc=close(voiceDrvfd); 
   if(rc!=0){
      printf("Error in voice driver close\n");
      return IX_DSP_DRV_FAIL;
   }
   
   return IX_DSP_DRV_SUCCESS;
}
