/******************************************************************
* @file   IxSlicDriverUsrAPIs.c
*
* Contents:  This file has IA specific Slic Driver User APIs
*
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
*
****************************************************************/

#include <stdio.h>
#include <pthread.h>
#include <sys/ioctl.h>
#include <stdint.h>

#include "IxDspCodeletAppOssl.h"
#include "IxDspCodelet.h" 
#include "IxDspCodeletAppLinuxInc.h"

#include "icp_analogdrv.h"

static int slicfd=0;    /* SLIC file descriptor */ 
static pthread_t cbThread;
void * callbackHandler(void *arg);

static char cbThreadEnable = THREAD_ENABLE;

/****************************************************************************/
/*
 * Function definition - ixFXSStartRinging
 * Start FXS Ringing
 */
 
int ixFXSStartRinging(int chl) 
{
   int slicLineId=0;
   int rc=0;

/*Ensure Slic is open */ 
   if(slicfd<=0)
   {
      printf("StartRinging-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   }  
 
 /* Need to convert ADS channel number to SLIC line ID */
   slicLineId=(chl+((chl-1)/NUM_FXS_CHIPS));
 
   rc=ioctl(slicfd,ICP_ANALOGDRV_RING_ON,slicLineId);
 
   if (rc != 0) 
   {
      printf("ICP_ANALOGDRV_RING_ON ioctl failed .. %d \n", rc);      
      return IX_DSP_DRV_FAIL;
   }

   return IX_DSP_DRV_SUCCESS;

}
/****************************************************************************/
/*
 * Function definition - ixFXSStopRinging
 * Stop FXS Ringing
 */
int ixFXSStopRinging(int chl) 
{
      int slicLineId=0;
      int rc=0;

/*Ensure Slic is open */ 

   if(slicfd<=0)
   {
      printf("StopRinging-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   }  
 
   /* Need to convert ADS channel number to SLIC line ID */
   slicLineId=chl+((chl-1)/NUM_FXS_CHIPS);
 
   rc = ioctl(slicfd , ICP_ANALOGDRV_RING_OFF, slicLineId);
   
   if (rc != 0)
   {
      printf("ICP_ANALOGDRV_RING_OFF ioctl failed .. %d \n", rc);
      return IX_DSP_DRV_FAIL;
   }
   return IX_DSP_DRV_SUCCESS;

}
/***************************************************************************/
/*
 * Function definition - ixFXSHookStatus
 * Check the hook status of the phone
 */
int ixFXSHookStatus(int chl) 
{
    int hookstate=0;
    int slicLineId=0;
    int rc=IX_DSP_DRV_FAIL;
    
/*Ensure Slic is open */ 
   if(slicfd<=0)
   {
      printf("HookStatus-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   } 
   
   /* Need to convert ADS channel number to SLIC line ID */
   slicLineId=chl+((chl-1)/NUM_FXS_CHIPS);
   
   hookstate = ioctl(slicfd, ICP_ANALOGDRV_HOOKSTATE_GET, slicLineId);
      
   switch(hookstate)
   {
      case ANALOGDRV_ON_HOOK:
      {
      
         printf("Channel %d is ON hook\n",chl);
         rc= IX_SC_ON_HOOK;
    break;
      }
      case ANALOGDRV_OFF_HOOK:
      {
         printf("Channel %d is OFF hook\n",chl);
         /*returns to IxDspCodeletSlicUtils.c*/
         rc= IX_SC_OFF_HOOK;
    break;
      }
      default:
      {
         printf("ICP_ANALOGDRV_HOOKSTATE_GET ioctl failed %d \n", hookstate);
         rc=IX_DSP_DRV_FAIL;
    break;  

      }
   } 
   
   return rc;           
}

/***************************************************************************/
/*
 * Function definition - ixFXSGetPCMConfig
 * Get the PCM configuration of the FXS device 
 */


int ixFXSGetPCMConfig(int chl) 
{

/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
  
}
/***************************************************************************/
/*
 * Function definition - ixFXSOChipDetails
 * Get the FXS chip details
 */
int ixFXSOChipDetails(int lineId)
{
/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
   
}
/***************************************************************************/
/*
 * Function definition - ixFXSODirectRegRead
 * Read from direct registers of FXS/FXO
 */

/* Note this funtion prototype now takes data as a parameter which is the 
 * data read*/
/* This funciton now simply returns pass or fail */
int ixFXSODirectRegRead(int slicLineId,int reg,int data) 
{
   int rc=0;
   icp_analogdrv_gen_reg_t reg_read={0,0,0};

/*Ensure Slic is open */ 
   if(slicfd<=0)
   {
      printf("DirectRegRead-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   } 

   reg_read.lineId=slicLineId;
   reg_read.reg=reg;
      
   rc = ioctl(slicfd, ICP_ANALOGDRV_READ_REG, &reg_read);
   
   if(rc != 0)
   {
       printf("ICP_ANALOGDRV_READ_REG ioctl failed .. %d \n", rc);
       return IX_DSP_DRV_FAIL;
   }
   data= (int) reg_read.data; 
  
   return IX_DSP_DRV_SUCCESS;  

}

/***************************************************************************/
/*
 * Function definition - ixFXSODirectRegWrite
 * Write to direct register of FXS/FXO
 */
int ixFXSODirectRegWrite(int slicLineId,int reg,int data) 
{   
   int rc=0;
   icp_analogdrv_gen_reg_t reg_write={0,0,0};

/*Ensure Slic is open */ 
   if(slicfd<=0)
   {
      printf("DirectRegWrite-Ensure Slic Module is inserted & initialised\n");
      return IX_DSP_DRV_FAIL;
   } 

   reg_write.lineId=slicLineId;
   reg_write.reg=reg;
   
   reg_write.data= (uint8_t) data;
   
   rc = ioctl(slicfd, ICP_ANALOGDRV_WRITE_REG, &reg_write);
   if(rc != 0)
   {
       printf("ICP_ANALOGDRV_WRITE_REG ioctl failed .. %d \n", rc);
       return IX_DSP_DRV_FAIL;
   }            

   return IX_DSP_DRV_SUCCESS;
 }

/***************************************************************************/
/*
 * Function definition - ixFXSOSetDeviceLoopback
 * Set digital loop back of FXS/FXO device
 */
int ixFXSOSetDeviceLoopback(int slot,int board,int chip)
{
/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}
/***************************************************************************/
/*
 * Function definition - ixFXSOResetDeviceLoopback
 * Reset the digital loopback of FXS/FXO device
 */

int ixFXSOResetDeviceLoopback(int slot,int board,int chip)
{
/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}
/***************************************************************************/
/*
 * Function definition - ixFXSOGetDeviceLoopback
 * Get the device loop back status of FXS/FXO device
 */
int ixFXSOGetDeviceLoopback(int slot,int board,int chip)
{
/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}
/***************************************************************************/
/*
 * Function definition - ixFXSOVerifyComm
 * Verify the communication with  FXS/FXO device
 */
int ixFXSOVerifyComm(int slot,int board,int chip)
{
/* Can be added later if required */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}
/***************************************************************************/
/*
 * Function definition - ixFXSSetPCMConfig
 * Set the PCM configuration of FXS/FXO device
 */
int ixFXSSetPCMConfig(int slot,int board,int chip,int pcmed,int mode,
                      int pcmtfr)
{ 
   /* See ixFXSConfig(int , int) function */
   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}
/***************************************************************************/
/*
 * Function definition - ixFXSOSlicInit
 * Initialize the  FXS/FXO device
 */
 
int ixFXSOSlicInit(int geo) 
{                                             
   int rc=0;
   int slic_geo=0;
   icp_analogdrv_init_t config_init={0,0,0};


#ifndef UNIT_TEST   
   slicfd = open("/dev/analog-fxo-fxs",O_RDONLY);
   if(slicfd<=0)
   {
      printf("SlicInit-Open Failed. Ensure Slic Module is inserted.\n");
      return IX_DSP_DRV_FAIL;
   }
#endif   

/*Convert ADS geo info to analog driver geo info*/
   switch(geo){
      case IX_DSP_CODELET_COUNTRY_US:
      {
         slic_geo=ANALOGDRV_US_GEO;
         break;
      }
      case IX_DSP_CODELET_COUNTRY_JP:
      {
         slic_geo=ANALOGDRV_JP_GEO;
         break;
      }
      case IX_DSP_CODELET_COUNTRY_PRC:
      {
         /*Setting the geo to Japan in this case
     * as PRC not supported by current SLIC
     * Note this does not mater to ADS as
     * ADS does not use the SLIC to play tones*/
       slic_geo=ANALOGDRV_JP_GEO;
       break;
      }
      default:
      {
         printf("Unsupported country code\n");
         return IX_DSP_DRV_FAIL;
      }
   }

/* Init SLIC on slot 0  channels 1-4 */

   config_init.slot=IX_HSS_PORT_0; /* hardcoded to slot 0 */
   config_init.geo_config=slic_geo;
   config_init.clk_speed=ANALOGDRV_CLK_SPEED_2MHZ;
   
   printf("Initialising Slot 0...\n");
 
   rc = ioctl(slicfd,ICP_ANALOGDRV_INIT, &config_init);
   if (rc != 0)
   {
      printf("ICP_ANALOGDRV_INIT 0 ioctl failed  .. %d \n", rc);
      return IX_DSP_DRV_FAIL;
   }  
   
/* Init SLIC on slot 1  channels 5-8 */
   config_init.slot=IX_HSS_PORT_1; /* hardcoded to slot 1 */
   config_init.geo_config=slic_geo;
   config_init.clk_speed=ANALOGDRV_CLK_SPEED_2MHZ;

   printf("Initialising Slot 1...\n");

   rc = ioctl(slicfd,ICP_ANALOGDRV_INIT, &config_init);
   if (rc != 0)
   {
       printf("ICP_ANALOGDRV_INIT 1 ioctl failed  .. %d \n", rc);
       return IX_DSP_DRV_FAIL;
   }    
   
   return IX_DSP_DRV_SUCCESS;

} 
/***************************************************************************/
/*
 * Function definition - ixFXSConfig
 * Configure the FXS chips
 */
int ixFXSConfig(int mode, int band)
{
   int rc=0;
   icp_analogdrv_pcm_config_t pcm_config_set={0,0,0,0,0};
   int expr_switch=0;
   int slot_cnt=0;
   int chip_cnt=0;

   /*Ensure Slic is open */
   if(slicfd<=0)
   {
      printf("RegCallback-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   }


 /* How the SLICs are configured depends on mode and band */
 
    expr_switch=(band<<BAND_SHIFT_FACTOR)+mode;
 
 
 /* Next config all FXS channels on both SLICs   */
 /* Note the FXO chips are not configured as they are not yet used */
 
 /* NOTES ON lineId HARDCODING..................   */   
 /* FXSchannel1 - lineId1, FXSchannel2 -lineId2,   */
 /* FXSchannel3 - lineId3 and FXSchannel4 - lineId4*/
 /* If FXO required then lineId0                   */
 /* FXSchannel5 - lineId6, FXSchannel6 -lineId7,   */
 /* FXSchannel7 - lineId8 and FXSchannel8 - lineId9*/
 /* If FXO required then lineId5                   */
 /* If Slic driver is being used then ADS Channel x = FXSchannel x and the  */
 /* offset associated with this should correspond to the timeslot mapping of*/
 /* voice driver channel labelled x-1.                                      */


 /* NOTES on ts HARDCODING..................................................*/
 /* Timeslots 1-8 respectively on slot0 are reserved for FXSchannel1,2,3,4  */
 /* If these channels are used in wideband then timeslots 65-68 on slot 0   */
 /* are reserved for  FXSchannel1,2,3,4                                     */
 /* Similarly for FXSchannel5,6,7,8 on slot 1                               */
 /* Therefore the SLIC will config with offsets 1,3,5,7, no matter the slot */


   for(slot_cnt=0; slot_cnt<NUM_SLIC_SLOTS; slot_cnt++){
      
      for(chip_cnt=0; chip_cnt<NUM_FXS_CHIPS; chip_cnt++){
  
         pcm_config_set.lineId=(slot_cnt*NUM_SLIC_CHIPS)+chip_cnt+1;
         pcm_config_set.pcm_start_offset=BYTE_TO_BIT_CONV*(SLICBASE_TS
                                           +(SLIC_TS_SPACING*chip_cnt)); 

         switch(expr_switch)
         {
   /* Configure 4 ALaw narrow band channels on both SLICs */

            case ALAW_NB:  
            {
            /* Note for analogdrv Alaw format is 1 unlike for ADS it is 0 */
            /* i.e. ANALOGDRV_PCM_FORMAT_A_LAW is defined to be 1 */
               pcm_config_set.pcm_format=ANALOGDRV_PCM_FORMAT_A_LAW;
               pcm_config_set.pcm_width=ANALOGDRV_PCM_WIDTH_8_BIT;
               pcm_config_set.pcm_samp_freq=ANALOGDRV_PCM_SAMP_FREQ_8_KHZ;
          break;
       }
       
         /* Configure 4 MuLaw narrow band channels on both SLICs */
            case MULAW_NB: 
       {
            /* Note for analogdrv ulaw format is 0 unlike for ADS it is 1 */
            /* i.e. ANALOGDRV_PCM_FORMAT_U_LAW is defined to be 0 */
               pcm_config_set.pcm_format=ANALOGDRV_PCM_FORMAT_U_LAW;
               pcm_config_set.pcm_width=ANALOGDRV_PCM_WIDTH_8_BIT;
               pcm_config_set.pcm_samp_freq=ANALOGDRV_PCM_SAMP_FREQ_8_KHZ;
               break;
       }
    
   /* Configure 4 Linear PCM wide band channels on both SLICs */
       case LINEAR_WB:
       {
          pcm_config_set.pcm_format=ANALOGDRV_PCM_FORMAT_LINEAR;
               pcm_config_set.pcm_width=ANALOGDRV_PCM_WIDTH_16_BIT;
               pcm_config_set.pcm_samp_freq=ANALOGDRV_PCM_SAMP_FREQ_16_KHZ;
               break;
       }
       default:
       {
          /* If an invalid mode-band combination is chosen */
          printf("Slic Init Error: Invalid Configuration,"
                      "ensure valid mode/band choice.\n");
               return IX_DSP_DRV_FAIL;
       }
    }
         /* Configure each FXS chip */
         rc = ioctl(slicfd,ICP_ANALOGDRV_PCM_CONFIG_SET, &pcm_config_set);
         if (rc != 0)
         {
            printf("ICP_ANALOGDRV_PCM_CONFIG_SET ioctl failed with"
                   "error code %d for slot %d and chip %d.\n", 
                    rc,slot_cnt,chip_cnt+1);
            return IX_DSP_DRV_FAIL; 

         }
      }     
   }
                     
   return IX_DSP_DRV_SUCCESS;
}
 
/*
 * Function definition - ixFXSRegCallback
 * Register the call back function for FXS
 */
int ixFXSRegCallback(int chip)
{
  
/*EL this function takes chip= NO_OF_PORTS                              */
/* This is left as input to the function but is not used.               */

/* We want to reg call back on any of the SLIC lines with this function */

/*Ensure Slic is open */ 
   if(slicfd<=0)
   {   
      printf("RegCallback-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   }

   if (initCallbackThread() != 0)
   {
      printf("Unable to create SLIC call back thread .. \n");
      return IX_DSP_DRV_FAIL; 
      
   }
    
   return IX_DSP_DRV_SUCCESS;
}

/***************************************************************************/
/*
   * Function definition - initCallbackThread
   * Intialises the callback thread
   */
int initCallbackThread(void)
{
   int result = 0;
   cbThreadEnable = THREAD_ENABLE;
   pthread_attr_t       thread_attr;
   struct sched_param   param;
   
   if(pthread_attr_init(&thread_attr)!=0)
   {
      printf("slic pthread_attr_init failure\n");
      return (IX_DSP_DRV_FAIL);
   }

   if(pthread_attr_setscope(&thread_attr, PTHREAD_SCOPE_SYSTEM)!=0)
   {
      printf("slic pthread_attr_setscope failure\n");
      return (IX_DSP_DRV_FAIL);
   }
   
   if(pthread_attr_setinheritsched(&thread_attr, PTHREAD_EXPLICIT_SCHED)!=0)
   {
      printf("slic pthread_attr_setinheritsched failure\n");
      return (IX_DSP_DRV_FAIL);
   }

   if(pthread_attr_setschedpolicy(&thread_attr, SCHED_RR)!=0)
   {
      printf("slic pthread_attr_setschedpolicy failure\n");
      return (IX_DSP_DRV_FAIL);
   }

   param.sched_priority = SLIC_CB_THREAD_PRIORITY;
   pthread_attr_setschedparam(&thread_attr, &param);
   pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);

   result = pthread_create(&cbThread, &thread_attr,
                       (pthread_startroutine_t)callbackHandler, 0);

   if (result != 0)
   {
      printf("Unable to initialise the thread .. %d \n", result); 
      return (IX_DSP_DRV_FAIL);
   }
   
   return (IX_DSP_DRV_SUCCESS);
}

/*************************************************************************/
/**
 * Function definition - callbackHandler
 * Thread which invokes the read command for an event from the SLIC. If the 
 * callback triggers , the read returns (otherwise it'll block... 
 * ensure SLIC is opened in blocking mode) and the thread will call the 
 * callback handler defined in user space based on the data 
 * received from kernel
 */ 
 
void * callbackHandler(void *arg)
{
   icp_analogdrv_event_t retEvent={0,0};
   int slicLineId=0;
   int hookstate=0;
   int result = 0;
   int chl=0;

   /* Set up thread attributes to allow it to be cancelled */
   if(pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL) != 0)
   {
      printf("Error setting up callback thread\n");
      /*In this case make sure don't go into while loop*/
      cbThreadEnable=THREAD_DISABLE;
   }
        
   if(pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL) != 0)
   {
      printf("Error setting up callback thread\n");
      cbThreadEnable=THREAD_DISABLE;
      
   }

/* Keep processing until the thread is killed */

   while (cbThreadEnable)
   {
      result = read(slicfd,&retEvent,SLIC_EVT_READ_SIZE);
 
      if (result == sizeof(icp_analogdrv_event_t))
      {
         hookstate=retEvent.eventType;
         slicLineId=retEvent.lineId;
         /*Convert lineId to channel number*/
         chl=slicLineId-slicLineId/(NUM_FXS_CHIPS+1);

         if(chl>0 && chl<=(NUM_SLIC_SLOTS*NUM_FXS_CHIPS) 
                  && (hookstate==ICP_ANALOGDRV_EVENT_ON_HOOK  
                      || hookstate==ICP_ANALOGDRV_EVENT_OFF_HOOK)){
/*ixDspCodeletSlicHookCB adds 1 to "chl-1" value to get chl*/
            ixDspCodeletSlicHookCB(hookstate, chl-1);
         }else{

            printf("SlicCBThread error in read ...%d."
              " Will try again\n",result);
            sleep(THREAD_SLEEP_TIME);
       
         }
         
      }
   }
     
    pthread_exit("Exiting Slic Callback Thread\n");
   return;
}

void cancelCallbackThread(void){

   int rc=0;
   
   rc=pthread_cancel(cbThread);
   if(rc!=0){
      printf("Error in cancelling slic callback thread\n");
   }
   
   return;
}
/***************************************************************************/
/*
 * Function definition - ixFXOOffhook
 * Set the FXO device to OFF hook
 */
int ixFXOOffhook(int lineId)
{

   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL; 
}
/***************************************************************************/
/*
   * Function definition - ixFXOOnhook
   * Set the FXO device to ON hook
   */

int ixFXOOnhook(int lineId)
{

   printf("This function is not currently supported for ADS1.0\n");
   return IX_DSP_DRV_FAIL;
}

int ixFXSOSlicClose(void)
{
   int rc=0;
   /*Allowing the driver to exit gracefully*/
   /*Ensure Slic is open */ 
   if(slicfd<=0)
   {
      printf("SlicClose-Ensure Slic Module is inserted & initialised.\n");
      return IX_DSP_DRV_FAIL;
   } 

   printf("ixFXSOSlicClose called\n");
 
   /* close buffers */ 
   rc=close(slicfd);
   if(rc==0)
   { 
      return IX_DSP_DRV_SUCCESS;
   }
   else{
      printf("Error in slic close\n");
      return IX_DSP_DRV_FAIL;
   }
}

