/******************************************************************************
*       @file	IxHssAcc.h
* -- Intel Copyright Notice --
* 
* Copyright (c) 2002-2008 Intel Corporation All Rights Reserved.
* 
* The source code contained or described herein and all documents
* related to the source code ("Material") are owned by Intel Corporation
* or its suppliers or licensors.  Title to the Material remains with
* Intel Corporation or its suppliers and licensors. The software is licensed under 
* IXA SDK license.
* 
* The Material is protected by worldwide copyright and trade secret laws
* and treaty provisions. No part of the Material may be used, copied,
* reproduced, modified, published, uploaded, posted, transmitted,
* distributed, or disclosed in any way except in accordance with the
* applicable license agreement .
* 
* No license under any patent, copyright, trade secret or other
* intellectual property right is granted to or conferred upon you by
* disclosure or delivery of the Materials, either expressly, by
* implication, inducement, estoppel, except in accordance with the
* applicable license agreement.
* 
* Unless otherwise agreed by Intel in writing, you may not remove or
* alter this notice or any other notice embedded in Materials by Intel
* or Intel's suppliers or licensors in any way.
* 
* For further details, please see the file README.TXT distributed with
* this software.
* 
* -- End Intel Copyright Notice --
* 
******************************************************************************/

#ifndef IXHSSACC_H
#define IXHSSACC_H

#include "FWTypes.h"

#define PUBLIC
/*
 * #defines for function return types, etc.
 */

/**
 * @def IX_HSSACC_TSLOTS_PER_HSS_PORT
 *
 * @brief The max number of TDM timeslots supported per HSS port - 4E1's =
 *  32x4 = 128 
 */
#define IX_HSSACC_TSLOTS_PER_HSS_PORT 128

/* -----------------------------------------------------------
   The following are HssAccess return values returned through 
   service interfaces. The globally defined IX_SUCCESS (0) and
   IX_FAIL (1) in IxOsalTypes.h are also used.
   ----------------------------------------------------------- */
/**
 * @def IX_HSSACC_PARAM_ERR
 *
 * @brief HssAccess function return value for a parameter error
 */
#define IX_HSSACC_PARAM_ERR 2

/**
 * @def IX_HSSACC_RESOURCE_ERR
 *
 * @brief HssAccess function return value for a resource error
 */
#define IX_HSSACC_RESOURCE_ERR 3

/**
 * @def IX_HSSACC_PKT_DISCONNECTING
 *
 * @brief Indicates that a disconnect call is progressing and will 
 * disconnect soon
 */
#define IX_HSSACC_PKT_DISCONNECTING 4

/**
 * @def IX_HSSACC_Q_WRITE_OVERFLOW
 *
 * @brief Indicates that an attempt to Tx or to replenish an 
 * RxFree Q failed due to Q overflow.  
 */
#define IX_HSSACC_Q_WRITE_OVERFLOW 5

/* -------------------------------------------------------------------
   The following errors are HSS/NPE errors returned on error retrieval
   ------------------------------------------------------------------- */
/**
 * @def IX_HSSACC_NO_ERROR
 *
 * @brief HSS port no error present
 */
#define IX_HSSACC_NO_ERROR 0

/**
 * @def IX_HSSACC_TX_FRM_SYNC_ERR
 *
 * @brief HSS port TX Frame Sync error
 */
#define IX_HSSACC_TX_FRM_SYNC_ERR 1

/**
 * @def IX_HSSACC_TX_OVER_RUN_ERR
 *
 * @brief HSS port TX over-run error
 */
#define IX_HSSACC_TX_OVER_RUN_ERR 2

/**
 * @def IX_HSSACC_CHANNELISED_SW_TX_ERR
 *
 * @brief NPE software error in channelised TX
 */
#define IX_HSSACC_CHANNELISED_SW_TX_ERR 3

/**
 * @def IX_HSSACC_PACKETISED_SW_TX_ERR
 *
 * @brief NPE software error in packetised TX
 */
#define IX_HSSACC_PACKETISED_SW_TX_ERR 4

/**
 * @def IX_HSSACC_RX_FRM_SYNC_ERR
 *
 * @brief HSS port RX Frame Sync error
 */
#define IX_HSSACC_RX_FRM_SYNC_ERR 5

/**
 * @def IX_HSSACC_RX_OVER_RUN_ERR
 *
 * @brief HSS port RX over-run error
 */
#define IX_HSSACC_RX_OVER_RUN_ERR 6

/**
 * @def IX_HSSACC_CHANNELISED_SW_RX_ERR
 *
 * @brief NPE software error in channelised RX
 */
#define IX_HSSACC_CHANNELISED_SW_RX_ERR 7

/**
 * @def IX_HSSACC_PACKETISED_SW_RX_ERR
 *
 * @brief NPE software error in packetised TX
 */
#define IX_HSSACC_PACKETISED_SW_RX_ERR 8

/* -----------------------------------
   Packetised service specific defines
   ----------------------------------- */

/**
 * @def IX_HSSACC_PKT_MIN_RX_MBUF_SIZE
 *
 * @brief Minimum size of the Rx mbuf in bytes which the client must supply 
 * to the component. 
 */
#define IX_HSSACC_PKT_MIN_RX_MBUF_SIZE 64

/* --------------------------------------------------------------------
   Enumerated Types - these enumerated values may be used in setting up
   the contents of hardware registers
   -------------------------------------------------------------------- */
/**
 * @enum IxHssAccHssPort
 * @brief The HSS port ID - There are two identical ports (0-1). 
 * 
 */
typedef enum
{
    IX_HSSACC_HSS_PORT_0,   /**< HSS Port 0 */
    IX_HSSACC_HSS_PORT_1,   /**< HSS Port 1 */
    IX_HSSACC_HSS_PORT_MAX  /**< Delimiter for error checks */
} IxHssAccHssPort;

/**
 * @enum IxHssAccHdlcPort
 * @brief The HDLC port ID - There are four identical HDLC ports (0-3) per 
 * HSS port and they correspond to the 4 E1/T1 trunks.
 * 
 */
typedef enum
{
   IX_HSSACC_HDLC_PORT_0,   /**< HDLC Port 0 */
   IX_HSSACC_HDLC_PORT_1,   /**< HDLC Port 1 */
   IX_HSSACC_HDLC_PORT_2,   /**< HDLC Port 2 */
   IX_HSSACC_HDLC_PORT_3,   /**< HDLC Port 3 */
   IX_HSSACC_HDLC_PORT_MAX  /**< Delimiter for error checks */
} IxHssAccHdlcPort;

/**
 * @enum IxHssAccTdmSlotUsage
 * @brief The HSS TDM stream timeslot assignment types
 *
 */
typedef enum
{
    IX_HSSACC_TDMMAP_UNASSIGNED,    /**< Unassigned */
    IX_HSSACC_TDMMAP_HDLC,          /**< HDLC - packetised */
    IX_HSSACC_TDMMAP_VOICE56K,      /**< Voice56K - channelised */
    IX_HSSACC_TDMMAP_VOICE64K,      /**< Voice64K - channelised */
    IX_HSSACC_TDMMAP_MAX            /**< Delimiter for error checks */
} IxHssAccTdmSlotUsage;

/**
 * @enum IxHssAccFrmSyncType
 * @brief The HSS frame sync pulse type
 *
 */
typedef enum
{
    IX_HSSACC_FRM_SYNC_ACTIVE_LOW,   /**< Frame sync is sampled low */
    IX_HSSACC_FRM_SYNC_ACTIVE_HIGH,  /**< sampled high */
    IX_HSSACC_FRM_SYNC_FALLINGEDGE,  /**< sampled on a falling edge */
    IX_HSSACC_FRM_SYNC_RISINGEDGE,   /**< sampled on a rising edge */
    IX_HSSACC_FRM_SYNC_TYPE_MAX      /**< Delimiter for error checks */
} IxHssAccFrmSyncType;

/**
 * @enum IxHssAccFrmSyncEnable
 * @brief The IxHssAccFrmSyncEnable determines how the frame sync pulse is
 * used
 * */
typedef enum
{
    IX_HSSACC_FRM_SYNC_INPUT,          /**< Frame sync is sampled as an input */
    IX_HSSACC_FRM_SYNC_INVALID_VALUE,  /**< 1 is not used */
    IX_HSSACC_FRM_SYNC_OUTPUT_FALLING, /**< Frame sync is an output generated 
					  off a falling clock edge */
    IX_HSSACC_FRM_SYNC_OUTPUT_RISING,  /**< Frame sync is an output generated 
					  off a rising clock edge */
    IX_HSSACC_FRM_SYNC_ENABLE_MAX      /**< Delimiter for error checks */
} IxHssAccFrmSyncEnable;

/**
 * @enum IxHssAccClkEdge
 * @brief IxHssAccClkEdge is used to determine the clk edge to use for 
 * framing and data
 *
 */
typedef enum
{
    IX_HSSACC_CLK_EDGE_FALLING,  /**< Clock sampled off a falling edge */
    IX_HSSACC_CLK_EDGE_RISING,   /**< Clock sampled off a rising edge */
    IX_HSSACC_CLK_EDGE_MAX       /**< Delimiter for error checks */
} IxHssAccClkEdge;

/**
 * @enum IxHssAccClkDir
 * @brief The HSS clock direction
 *
 */
typedef enum
{
    IX_HSSACC_SYNC_CLK_DIR_INPUT,    /**< Clock is an input */
    IX_HSSACC_SYNC_CLK_DIR_OUTPUT,   /**< Clock is an output */
    IX_HSSACC_SYNC_CLK_DIR_MAX       /**< Delimiter for error checks */
} IxHssAccClkDir;

/**
 * @enum IxHssAccFrmPulseUsage
 * @brief The HSS frame pulse usage
 *
 */
typedef enum
{
    IX_HSSACC_FRM_PULSE_ENABLED,     /**< Generate/Receive frame pulses */
    IX_HSSACC_FRM_PULSE_DISABLED,    /**< Disregard frame pulses */
    IX_HSSACC_FRM_PULSE_MAX          /**< Delimiter for error checks */
} IxHssAccFrmPulseUsage;

/**
 * @enum IxHssAccDataRate
 * @brief The HSS Data rate in relation to the clock
 *
 */
typedef enum
{
    IX_HSSACC_CLK_RATE,      /**< Data rate is at the configured clk speed */
    IX_HSSACC_HALF_CLK_RATE, /**< Data rate is half the configured clk speed */
    IX_HSSACC_DATA_RATE_MAX  /**< Delimiter for error checks */
} IxHssAccDataRate;

/**
 * @enum IxHssAccDataPolarity
 * @brief The HSS data polarity type
 *
 */
typedef enum
{
    IX_HSSACC_DATA_POLARITY_SAME,   /**< Don't invert data between NPE and 
				       HSS FIFOs */
    IX_HSSACC_DATA_POLARITY_INVERT, /**< Invert data between NPE and HSS 
				       FIFOs */
    IX_HSSACC_DATA_POLARITY_MAX     /**< Delimiter for error checks */
} IxHssAccDataPolarity;

/**
 * @enum IxHssAccBitEndian
 * @brief HSS Data endianness
 *
 */
typedef enum
{
    IX_HSSACC_LSB_ENDIAN,    /**< TX/RX Least Significant Bit first */
    IX_HSSACC_MSB_ENDIAN,    /**< TX/RX Most Significant Bit first */
    IX_HSSACC_ENDIAN_MAX     /**< Delimiter for the purposes of error checks */
} IxHssAccBitEndian;


/**
 * @enum IxHssAccDrainMode
 * @brief Tx pin open drain mode
 *
 */
typedef enum
{
    IX_HSSACC_TX_PINS_NORMAL,       /**< Normal mode */
    IX_HSSACC_TX_PINS_OPEN_DRAIN,   /**< Open Drain mode */
    IX_HSSACC_TX_PINS_MAX           /**< Delimiter for error checks */
} IxHssAccDrainMode;

/**
 * @enum IxHssAccSOFType
 * @brief HSS start of frame types
 *
 */
typedef enum
{
    IX_HSSACC_SOF_FBIT,  /**< Framing bit transmitted and expected on rx */
    IX_HSSACC_SOF_DATA,  /**< Framing bit not transmitted nor expected on rx */
    IX_HSSACC_SOF_MAX    /**< Delimiter for error checks */
} IxHssAccSOFType;

/**
 * @enum IxHssAccDataEnable
 * @brief IxHssAccDataEnable is used to determine whether or not to drive 
 * the data pins
 *
 */
typedef enum
{
    IX_HSSACC_DE_TRI_STATE,   /**< TRI-State the data pins */
    IX_HSSACC_DE_DATA,        /**< Push data out the data pins */
    IX_HSSACC_DE_MAX          /**< Delimiter for error checks */
} IxHssAccDataEnable;

/**
 * @enum IxHssAccTxSigType
 * @brief IxHssAccTxSigType is used to determine how to drive the data pins
 *
 */
typedef enum
{
    IX_HSSACC_TXSIG_LOW,        /**< Drive the data pins low */
    IX_HSSACC_TXSIG_HIGH,       /**< Drive the data pins high */
    IX_HSSACC_TXSIG_HIGH_IMP,   /**< Drive the data pins with high impedance */
    IX_HSSACC_TXSIG_MAX         /**< Delimiter for error checks */
} IxHssAccTxSigType;

/**
 * @enum IxHssAccFbType
 * @brief IxHssAccFbType determines how to drive the Fbit
 *
 * @warning This will only be used for T1 @ 1.544MHz
 *
 */
typedef enum
{
    IX_HSSACC_FB_FIFO,        /**< Fbit is dictated in FIFO */
    IX_HSSACC_FB_HIGH_IMP,    /**< Fbit is high impedance */
    IX_HSSACC_FB_MAX          /**< Delimiter for error checks */
} IxHssAccFbType;

/**
 * @enum IxHssAcc56kEndianness
 * @brief 56k data endianness when using the 56k type
 *
 */
typedef enum
{
    IX_HSSACC_56KE_BIT_7_UNUSED,  /**< High bit is unused */
    IX_HSSACC_56KE_BIT_0_UNUSED,  /**< Low bit is unused */
    IX_HSSACC_56KE_MAX            /**< Delimiter for error checks */
} IxHssAcc56kEndianness;

/**
 * @enum IxHssAcc56kSel
 * @brief 56k data transmission type when using the 56k type
 *
 */
typedef enum
{
    IX_HSSACC_56KS_32_8_DATA,  /**< 32/8 bit data */
    IX_HSSACC_56KS_56K_DATA,   /**< 56K data */
    IX_HSSACC_56KS_MAX         /**< Delimiter for error checks */
} IxHssAcc56kSel;


/**
 * @enum IxHssAccClkSpeed
 * @brief IxHssAccClkSpeed represents the HSS clock speeds available
 *
 */
typedef enum
{
    IX_HSSACC_CLK_SPEED_512KHZ,     /**< 512KHz */
    IX_HSSACC_CLK_SPEED_1536KHZ,    /**< 1.536MHz */
    IX_HSSACC_CLK_SPEED_1544KHZ,    /**< 1.544MHz */
    IX_HSSACC_CLK_SPEED_2048KHZ,    /**< 2.048MHz */
    IX_HSSACC_CLK_SPEED_4096KHZ,    /**< 4.096MHz */
    IX_HSSACC_CLK_SPEED_8192KHZ,    /**< 8.192MHz */
    IX_HSSACC_CLK_SPEED_MAX      /**< Delimiter for error checking */
} IxHssAccClkSpeed;

/**
 * @enum IxHssAccPktStatus
 * @brief Indicates the status of packets passed to the client
 *
 */
typedef enum
{
    IX_HSSACC_PKT_OK,              /**< Error free.*/
    IX_HSSACC_STOP_SHUTDOWN_ERROR, /**< Errored due to stop or shutdown 
				      occurrance.*/
    IX_HSSACC_HDLC_ALN_ERROR,      /**< HDLC alignment error */
    IX_HSSACC_HDLC_FCS_ERROR,       /**< HDLC Frame Check Sum error.*/
    IX_HSSACC_RXFREE_Q_EMPTY_ERROR,       /**< RxFree Q became empty 
					     while receiving this packet.*/
    IX_HSSACC_HDLC_MAX_FRAME_SIZE_EXCEEDED,      /**< HDLC frame size 
						   received is greater than
						   max specified at connect.*/
    IX_HSSACC_HDLC_ABORT_ERROR,   /**< HDLC frame received is invalid due to an 
				   abort sequence received.*/
    IX_HSSACC_DISCONNECT_IN_PROGRESS     /**< Packet returned
					    because a disconnect is in progress */
} IxHssAccPktStatus;


/**
 * @enum IxHssAccPktCrcType
 * @brief HDLC CRC type
 *
 */
typedef enum
{
    IX_HSSACC_PKT_16_BIT_CRC = 16,  /**< 16 bit CRC is being used */
    IX_HSSACC_PKT_32_BIT_CRC = 32   /**< 32 bit CRC is being used */
} IxHssAccPktCrcType;

/**
 * @enum IxHssAccPktHdlcIdleType
 * @brief HDLC idle transmission type
 *
 */
typedef enum
{
    IX_HSSACC_HDLC_IDLE_ONES,    /**< idle tx/rx will be a succession of ones */
    IX_HSSACC_HDLC_IDLE_FLAGS    /**< idle tx/rx will be repeated flags */
} IxHssAccPktHdlcIdleType;

/**
 * @typedef IxHssAccChanRxCallback
 * @brief Prototype of the clients function to accept notification of 
 * channelised rx
 *
 * This callback, if defined by the client in the connect, will get called
 * in the context of an IRQ. The IRQ will be triggered when the hssSyncQMQ
 * is not empty. The queued entry will be dequeued and this function will
 * be executed.
 *
 * @param hssPortId @ref IxHssAccHssPort - The HSS port Id. There are two
 * identical ports (0-1).
 * @param txOffset unsigned [in] - an offset indicating from where within
 * the txPtrList the NPE is currently transmitting from.
 * @param rxOffset unsigned [in] - an offset indicating where within the
 * receive buffers the NPE has just written the received data to.
 * @param numHssErrs unsigned [in] - This is the number of hssErrors 
 * the Npe has received
 *
 * @return void
 */
typedef void (*IxHssAccChanRxCallback) (IxHssAccHssPort hssPortId,
                                        unsigned rxOffset,
                                        unsigned txOffset,
                                        unsigned numHssErrs);

/**
 * @brief Structure containing HSS port configuration parameters
 *
 * Note: All of these are used for TX. Only some are specific to RX.
 *
 */
typedef struct
{
    IxHssAccFrmSyncType frmSyncType;     /**< frame sync pulse type (tx/rx) */
    IxHssAccFrmSyncEnable frmSyncIO;     /**< how the frame sync pulse is 
					    used (tx/rx) */
    IxHssAccClkEdge frmSyncClkEdge;      /**< frame sync clock edge type 
					    (tx/rx) */
    IxHssAccClkEdge dataClkEdge;         /**< data clock edge type (tx/rx) */
    IxHssAccClkDir clkDirection;         /**< clock direction (tx/rx) */
    IxHssAccFrmPulseUsage frmPulseUsage; /**< whether to use the frame sync 
					    pulse or not (tx/rx) */
    IxHssAccDataRate dataRate;           /**< data rate in relation to the 
					    clock (tx/rx) */
    IxHssAccDataPolarity dataPolarity;   /**< data polarity type (tx/rx) */
    IxHssAccBitEndian dataEndianness;    /**< data endianness (tx/rx) */
    IxHssAccDrainMode drainMode;         /**< tx pin open drain mode (tx) */
    IxHssAccSOFType fBitUsage;           /**< start of frame types (tx/rx) */
    IxHssAccDataEnable dataEnable;       /**< whether or not to drive the data 
					    pins (tx) */
    IxHssAccTxSigType voice56kType;      /**< how to drive the data pins for 
					    voice56k type (tx) */
    IxHssAccTxSigType unassignedType;    /**< how to drive the data pins for 
					    unassigned type (tx) */
    IxHssAccFbType fBitType;             /**< how to drive the Fbit (tx) */
    IxHssAcc56kEndianness voice56kEndian;/**< 56k data endianness when using 
					    the 56k type (tx) */
    IxHssAcc56kSel voice56kSel;          /**< 56k data transmission type when 
					    using the 56k type (tx) */
    unsigned frmOffset;                  /**< frame pulse offset in bits wrt 
					    the first timeslot (0-1023) (tx/rx) */
    unsigned maxFrmSize;                 /**< frame size in bits (1-1024) 
					    (tx/rx) */
} IxHssAccPortConfig;

/**
 * @brief Structure containing HSS configuration parameters
 *
 */
typedef struct
{
    IxHssAccPortConfig txPortConfig; /**< HSS tx port configuration */
    IxHssAccPortConfig rxPortConfig; /**< HSS rx port configuration */
    unsigned numChannelised;         /**< The number of channelised 
					timeslots (0-32) */
    unsigned hssPktChannelCount;     /**< The number of packetised 
					clients (0 - 4) */
    UINT8 channelisedIdlePattern;    /**< The byte to be transmitted on 
					channelised service when there 
					is no client data to tx */
    BOOL loopback;                   /**< The HSS loopback state */
    unsigned packetizedIdlePattern;  /**< The data to be transmitted on 
					packetised service when there is 
					no client data to tx */
    IxHssAccClkSpeed clkSpeed;       /**< The HSS clock speed */
} IxHssAccConfigParams;

typedef void (*IxHssAccLastErrorCallback) (unsigned lastHssError, 
					   unsigned servicePort);

/**
 *
 * @ingroup IxHssAccAPI
 *
 * @fn IX_STATUS ixHssAccPortInit (IxHssAccHssPort hssPortId, 
               IxHssAccConfigParams *configParams, 
               IxHssAccTdmSlotUsage *tdmMap, 
               IxHssAccLastErrorCallback lastHssErrorCallback)
 *
 * @brief Initialise a HSS port. No channelised or packetised connections
 * should exist in the HssAccess layer while this interface is being called.
 *
 * @param hssPortId @ref IxHssAccHssPort [in] - The HSS port Id. There are two
 * identical ports (0-1). 
 * @param *configParams @ref IxHssAccConfigParams [in] - A pointer to the HSS 
 * configuration structure
 * @param *tdmMap @ref IxHssAccTdmSlotUsage [in] - A pointer to an array of size
 * IX_HSSACC_TSLOTS_PER_HSS_PORT, defining the slot usage over the HSS port
 * @param lastHssErrorCallback @ref IxHssAccLastErrorCallback [in] - Client 
 * callback to report last error
 *
 * @return 
 *         - IX_SUCCESS The function executed successfully
 *         - IX_FAIL The function did not execute successfully
 *         - IX_HSSACC_PARAM_ERR The function did not execute successfully due to a
 *                          parameter error
 */
PUBLIC IX_STATUS 
ixHssAccPortInit (IxHssAccHssPort hssPortId, 
		  IxHssAccConfigParams *configParams, 
		  IxHssAccTdmSlotUsage *tdmMap, 
		  IxHssAccLastErrorCallback lastHssErrorCallback);

/**
 *
 * @ingroup IxHssAccAPI
 *
 * @fn IX_STATUS ixHssAccInit ()
 *
 * @brief This function is responsible for initialising resources for use
 * by the packetised and channelised clients. It should be called after 
 * HSS NPE image has been downloaded into NPE-A and before any other
 * HssAccess interface is called. 
 * No other HssAccPacketised interface should be called while this interface
 * is being processed.
 *
 * @return 
 *         - IX_SUCCESS The function executed successfully
 *         - IX_FAIL The function did not execute successfully
 *         - IX_HSSACC_RESOURCE_ERR The function did not execute successfully due
 *                          to a resource error 
 */
PUBLIC IX_STATUS 
ixHssAccInit (void);



/**
 *
 * @ingroup IxHssAccAPI
 *
 * @fn IX_STATUS ixHssAccChanConnect (IxHssAccHssPort hssPortId, 
           unsigned bytesPerTSTrigger, 
	   UINT8 *rxCircular, 
	   unsigned numRxBytesPerTS, 
	   UINT32 *txPtrList, 
	   unsigned numTxPtrLists, 
	   unsigned numTxBytesPerBlk, 
	   IxHssAccChanRxCallback rxCallback)
 *
 * @brief This function allows the client to connect to the Tx/Rx NPE
 * Channelised Service. There can only be one client per HSS port. The
 * client is responsible for ensuring that the HSS port is configured
 * appropriately before its connect request. No other HssAccChannelised 
 * interface should be called while this interface is being processed.
 *
 * @param hssPortId @ref IxHssAccHssPort [in] - The HSS port Id. There are two
 * identical ports (0-1). 
 * @param bytesPerTSTrigger unsigned [in] - The NPE will trigger the access
 * component after bytesPerTSTrigger have been received for all trunk
 * timeslots. This figure is a multiple of 8 e.g. 8 for 1ms trigger, 16 for
 * 2ms trigger.
 * @param *rxCircular UINT8 [in] - A pointer to memory allocated by the
 * client to be filled by data received. The buffer at this address is part
 * of a pool of buffers to be accessed in a circular fashion. This address
 * will be written to by the NPE. Therefore, it needs to be a physical address.
 * @param numRxBytesPerTS unsigned [in] - The number of bytes allocated per
 * timeslot within the receive memory. This figure will depend on the
 * latency of the system. It needs to be deep enough for data to be read by
 * the client before the NPE re-writes over that memory e.g. if the client
 * samples at a rate of 40bytes per timeslot, numRxBytesPerTS may need to
 * be 40bytes * 3. This would give the client 3 * 5ms of time before
 * received data is over-written.
 * @param *txPtrList UINT32 [in] - The address of an area of contiguous
 * memory allocated by the client to be populated with pointers to data for
 * transmission. Each pointer list contains a pointer per active channel.
 * The txPtrs will point to data to be transmitted by the NPE. Therefore,
 * they must point to physical addresses.
 * @param numTxPtrLists unsigned [in] - The number of pointer lists in
 * txPtrList. This figure is dependent on jitter.
 * @param numTxBytesPerBlk unsigned [in] - The size of the Tx data, in
 * bytes, that each pointer within the PtrList points to.
 * @param rxCallback @ref IxHssAccChanRxCallback [in] - A client function
 * pointer to be called back to handle the actual tx/rx of channelised
 * data. If this is not NULL, an ISR will call this function. If this
 * pointer is NULL, it implies that the client will use a polling mechanism
 * to detect when the tx and rx of channelised data is to occur. The client
 * will use hssChanAccStatus for this.
 *
 * @return
 *         - IX_SUCCESS The function executed successfully
 *         - IX_FAIL The function did not execute successfully
 *         - IX_HSSACC_PARAM_ERR The function did not execute successfully due to a
 *                          parameter error
 */
 
PUBLIC IX_STATUS 
ixHssAccChanConnect (IxHssAccHssPort hssPortId, 
		     unsigned bytesPerTSTrigger, 
		     UINT8 *rxCircular, 
		     unsigned numRxBytesPerTS, 
		     UINT32 *txPtrList, 
		     unsigned numTxPtrLists, 
		     unsigned numTxBytesPerBlk, 
		     IxHssAccChanRxCallback rxCallback);


#endif /* IXHSSACC_H */

/**
 * @} defgroup IxHssAcc
 */
