/**
 * @file ixTimeSyncAcc_p.h
 *
 * @author Intel Corporation
 * @date 17 Mar 2006
 *
 * @brief  Private header file for Access Layer to IEEE 1588(TM)
 * Precision Clock Synchronisation Protocol Hardware Assist.
 *
 * @par
 * INTEL CONFIDENTIAL
 * Copyright 2007,2008,2009 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to the
 * source code ("Material") are owned by Intel Corporation or its suppliers or
 * licensors. Title to the Material remains with Intel Corporation or its
 * suppliers and licensors. The Material may contain trade secrets and proprietary
 * and confidential information of Intel Corporation and its suppliers and
 * licensors, and is protected by worldwide copyright and trade secret laws and
 * treaty provisions. No part of the Material may be used, copied, reproduced,
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Intels prior express written permission.
 * No license under any patent, copyright, trade secret or other intellectual
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise. Any license under such intellectual property rights must be
 * express and approved by Intel in writing.
 * 
 * Include any supplier copyright notices as supplier requires Intel to use.
 * Include supplier trademarks or logos as supplier requires Intel to use,
 * preceded by an asterisk.
 * An asterisked footnote can be added as follows: 
 *   *Third Party trademarks are the property of their respective owners.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Intel or Intels suppliers
 * or licensors in any way.
 * 
 *  version: Embedded.X.1.0.3-127
 */


/* 
 * System defined include files
 */

/*
 * #defines and macros used in this file.
 */

#ifndef _IXTIMESYNCACC_P_H_
#define _IXTIMESYNCACC_P_H_

#define TRUE 1
#define FALSE 0

/* Inline or Non-Inlined function declaration/definition macro */
#ifdef NO_INLINE_APIS
    #define ICP_TIMESYNCACC_INLINE   /* empty define */
#else /* else of ifdef NO_INLINE_APIS */
	#define ICP_TIMESYNCACC_INLINE  __inline
#endif /* end of ifdef NO_INLINE_APIS */

/* private function prototypes */
/*
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAmmsInterruptMaskSet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAsmsInterruptMaskSet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlTtmInterruptMaskSet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAtmInterruptMaskSet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPpsmInterruptMaskSet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlAmmsInterruptMaskGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlAsmsInterruptMaskGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlTtmInterruptMaskGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlAtmInterruptMaskGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPpsmInterruptMaskGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAmmsInterruptMaskClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAsmsInterruptMaskClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlTtmInterruptMaskClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlAtmInterruptMaskClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPpsmInterruptMaskClear(void);
*/
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlReset(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccEventAmmsFlagGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccEventAsmsFlagGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccEventTtmFlagGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccEventAtmFlagGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccEventPpsmFlagGet(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccEventAmmsFlagClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccEventAsmsFlagClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccEventTtmFlagClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccEventAtmFlagClear(void);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccEventPpsmFlagClear(void);
/*
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccSystemTimeSnapshotSet(UINT32 systemTimeLow, UINT32 systemTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccSystemTimeSnapshotGet(UINT32 *systemTimeLow, UINT32 *systemTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccTargetTimeSnapshotSet(UINT32 targetTimeLow, UINT32 targetTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccTargetTimeSnapshotGet(UINT32 *targetTimeLow, UINT32 *targetTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAuxTargetTimeSnapshotSet (UINT32 auxTargetTimeLow, UINT32 auxTargetTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAuxTargetTimeSnapshotGet(UINT32 *auxTargetTimeLow, UINT32 *auxTargetTimeHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPulsePerSecondCompareSet (UINT32 ppsTime);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPulsePerSecondCompareGet (UINT32 *ppsTime);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAddendFsvSet (UINT32 fsv);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAddendFsvGet (UINT32 *fsv);
*/
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAuxMasterModeSnapshotGet (UINT32 *ammsLow, UINT32 *ammsHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccAuxSlaveModeSnapshotGet (UINT32 *asmsLow, UINT32 *asmsHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccChannelReset(UINT32 ptpPort);
/*
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortMasterModeSet (UINT32 ptpPort, UINT32 masterMode);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortMasterModeGet (UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortPTPMsgTimestampSet (UINT32 ptpPort, UINT32 allMsg);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortPTPMsgTimestampGet(UINT32 ptpPort);
*/
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortValFlagGet (UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortOvrFlagGet (UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortValFlagClear(UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortOvrFlagClear(UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortRxsFlagGet (UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 ixTimeSyncAccControlPTPPortTxsFlagGet (UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortRxsFlagClear(UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccControlPTPPortTxsFlagClear(UINT32 ptpPort);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPTPPortCANSnapshotGet (UINT32 ptpPort, UINT32 *rxsLow, UINT32 *rxsHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPTPPortReceiveSnapshotGet (UINT32 ptpPort, UINT32 *rxsLow, UINT32 *rxsHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPTPPortTransmitSnapshotGet (UINT32 ptpPort, UINT32 *txsLow, UINT32 *txsHigh);
PRIVATE ICP_TIMESYNCACC_INLINE void ixTimeSyncAccPTPMsgUuidSeqIdGet (UINT32 ptpPort, UINT32 *uuidLow, UINT16 *uuidHigh, UINT16 *seqId);

void write32(UINT32 val, UINT32 addr);
UINT32 read32(UINT32 addr);

/* OS specific defines */

#ifdef __linux 
	#include <asm/io.h>

	void write32(UINT32 val, UINT32 addr)
	{
		iowrite32(val, (UINT32*)addr);
	}

	UINT32 read32(UINT32 addr)
	{
		return ioread32((UINT32*)addr);
	}

	#define printf printk 
#endif

#ifdef _WIN32 
	#include <stdio.h>
	#include <stdlib.h>
  
	void write32(UINT32 val, UINT32 addr)
	{
		WRITE_REGISTER_ULONG((UINT32*)addr, val);
	}

	UINT32 read32(UINT32 addr)
	{
		return READ_REGISTER_ULONG((UINT32*)addr);
	}

	#define printf DbgPrint 
#endif

#ifdef __FreeBSD__ 
	#include <sys/param.h>
	#include <sys/systm.h>

	void write32(UINT32 val, UINT32 addr)
	{
		writel((UINT32*)addr, val);
	}

	UINT32 read32(UINT32 addr)
	{
		return readl((UINT32*)addr);
	}
#endif

/*
 * SET, CLEAR & CHECK macros for bit manipulation on the 
 * memory mapped registers' contents
 */
#define ICP_TIMESYNCACC_BIT_SET(regAddr, bitMask) \
do { \
    /* *(regAddr) |= (bitMask); */ \
    write32(read32(regAddr) | bitMask, \
                  regAddr); \
} while (0) /* Don't include ';' here */

#define ICP_TIMESYNCACC_BIT_CLEAR(regAddr, bitMask) \
do { \
    /* *(regAddr) &= ~(bitMask); */ \
    write32(read32(regAddr) & ~(bitMask), \
                       regAddr); \
} while (0) /* Don't include ';' here */

#define ICP_TIMESYNCACC_BIT_SET_CHECK(regAddr, bitMask) \
    ((read32(regAddr)  & (bitMask)) == (bitMask))
/* Don't include ';' here */

/*
 * READ, WRITE macros for memory mapped registers
 */
#define ICP_TIMESYNCACC_REG_READ(regAddr, varRef) \
do { \
    *(varRef) = read32(regAddr); \
} while (0) /* Don't include ';' here */

#define ICP_TIMESYNCACC_REG_WRITE(regAddr, varValue) \
do { \
    write32(varValue, regAddr); \
} while (0) /* Don't include ';' here */

/* Masks to extract High and Low SHORTs from UINT32 values */
#define ICP_TIMESYNCACC_MSB_SHORT_MASK  (0xFFFF0000)
#define ICP_TIMESYNCACC_LSB_SHORT_MASK  (0x0000FFFF)

/* Location of SeqID in the register */
#define ICP_TIMESYNCACC_SID_LOC         (16)

/* Max number of time sync hw channels for each port type */
#define ICP_TIMESYNCACC_MAX_1588ETH_CHNL  (0x08)
#define ICP_TIMESYNCACC_MAX_1588CAN_CHNL  (0x02)

/* Max number of time sync ports */
#define ICP_TIMESYNCACC_MAX_1588PTP_PORT  (ICP_TIMESYNCACC_MAX_1588ETH_CHNL + \
									   ICP_TIMESYNCACC_MAX_1588CAN_CHNL)	

/* active hardware channel numbers */
#define ICP_TIMESYNCACC_GBE_0_1588PTP_CHNL (0x3)
#define ICP_TIMESYNCACC_GBE_1_1588PTP_CHNL (0x4)
#define ICP_TIMESYNCACC_CAN_0_1588PTP_CHNL (0x0)
#define ICP_TIMESYNCACC_CAN_1_1588PTP_CHNL (0x1)

/* Size of the each Block / Port Level Register */
#define ICP_TIMESYNCACC_BLPLREG_SIZE  (0x04)

/* Offsets to eth and can port level registers */
#define ICP_TIMESYNCACC_PLETH_OFFSET  (0x40)
#define ICP_TIMESYNCACC_PLCAN_OFFSET  (0x140)

/* Block wise offset of each Port Level Registers for a given ethernet PTP Port
 */
#define ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET  (0x20)

/* Block wise offset of each Port Level Registers for a given CAN PTP Port
 */
#define ICP_TIMESYNCACC_PLREGS_CAN_BOFFSET  (0x10)

/*
 * Block Level Registers Offset Values
 *
 * Please refer to the struct - IxTimeSyncAccBlockLevelRegisters defined
 * to hold the virtual addresses of the various block level registers of
 * time sync hardware 
 */
#define ICP_TIMESYNCACC_TSC_OFFSET    (0x00)    // TS_Control 
#define ICP_TIMESYNCACC_TSE_OFFSET    (0x04)    // TS_Event
#define ICP_TIMESYNCACC_ADD_OFFSET    (0x08)    // TS_Addend
#define ICP_TIMESYNCACC_ACC_OFFSET    (0x0C)    // TS_Accum
#define ICP_TIMESYNCACC_TST_OFFSET    (0x10)    // TS_Test 
#define ICP_TIMESYNCACC_PPS_OFFSET	  (0x14)    // TS_PPS_Compare
#define ICP_TIMESYNCACC_STL_OFFSET    (0x20)    // TS_SysTimeLo
#define ICP_TIMESYNCACC_STH_OFFSET    (0x24)    // TS_SysTimeHi
#define ICP_TIMESYNCACC_TTL_OFFSET    (0x28)    // TS_TrgtLo
#define ICP_TIMESYNCACC_TTH_OFFSET    (0x2c)    // TS_TrgtHi
#define ICP_TIMESYNCACC_ASSL_OFFSET   (0x30)    // TS_ASMSLo
#define ICP_TIMESYNCACC_ASSH_OFFSET   (0x34)    // TS_ASMSHi
#define ICP_TIMESYNCACC_AMSL_OFFSET   (0x38)    // TS_AMMSLo
#define ICP_TIMESYNCACC_AMSH_OFFSET   (0x3C)    // TS_AMMSHi
#define ICP_TIMESYNCACC_ATTL_OFFSET	  (0x1F0)   // TS_Aux_TrgtLo
#define	ICP_TIMESYNCACC_ATTH_OFFSET	  (0x1F4)   // TS_Aux_TrgtHi

/*
 * Port Level Registers Offset Values
 *
 * The following offset macros work as explained below.
 * 
 * Effective Address:= Starting Virtual Address + 
 *                     Vertical Offset + Block Offset for Port
 *
 * NOTE: a) Starting Virtual Address will be obtained using OSAL macro
 *       b) portNum (0 -> Max ICP_TIMESYNCACC_MAX_1588PTP_PORT)
 *       c) Block Offset for Port starts from Zero
 *
 * The example assumes that Starting Virtual Address has been 0x40.
 *
 * Eg., CC0:  0x40 + (0x04 * 0x00) + (0x20 * 0x00) = 0x40
 *      XSH1: 0x40 + (0x04 * 0x03) + (0x20 * 0x01) = 0x6C
 *      RSH2: 0x40 + (0x04 * 0x05) + (0x20 * 0x02) = 0x94
 */

/* Vertical/Relative Offset of a given ethernet Port Level Register 
 * within a single set/block for each ethernet PTP channel
 *
 * Please refer to the struct - IxTimeSyncAccPortLevelRegisters defined
 * to hold the virtual addresses of the various port level registers of
 * time sync hardware
 */
#define ICP_TIMESYNCACC_CC_VOFFSET     (0x00)
#define ICP_TIMESYNCACC_CE_VOFFSET     (0x04)
#define ICP_TIMESYNCACC_XSL_VOFFSET    (0x08)
#define ICP_TIMESYNCACC_XSH_VOFFSET    (0x0C)
#define ICP_TIMESYNCACC_RSL_VOFFSET    (0x10)
#define ICP_TIMESYNCACC_RSH_VOFFSET    (0x14)
#define ICP_TIMESYNCACC_UID_VOFFSET    (0x18)
#define ICP_TIMESYNCACC_SID_VOFFSET    (0x1C)


/* Vertical/Relative Offset of a given CAN Port Level Register 
 * within a single set/block for each CAN PTP channel
 *
 * Please refer to the struct - IxTimeSyncAccPortLevelRegisters defined
 * to hold the virtual addresses of the various port level registers of
 * time sync hardware
 */
#define ICP_TIMESYNCACC_CCE_VOFFSET		(0x00)
#define ICP_TIMESYNCACC_CXSL_VOFFSET	(0x04)
#define	ICP_TIMESYNCACC_CXSH_VOFFSET	(0x08)

/* Compounded Offsets for each of the Port Level Registers as explained
 * in the NOTE of the above comments
 */
#define ICP_TIMESYNCACC_CC_OFFSET(portNum)  \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_CC_VOFFSET +   \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_CE_OFFSET(portNum)  \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_CE_VOFFSET +   \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_XSL_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_XSL_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_XSH_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_XSH_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_RSL_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_RSL_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_RSH_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_RSH_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_UID_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_UID_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_SID_OFFSET(portNum) \
            (ICP_TIMESYNCACC_PLETH_OFFSET + ICP_TIMESYNCACC_SID_VOFFSET +  \
             ICP_TIMESYNCACC_PLREGS_ETH_BOFFSET * (portNum))
#define ICP_TIMESYNCACC_CCE_OFFSET(portNum) \
			(ICP_TIMESYNCACC_PLCAN_OFFSET + ICP_TIMESYNCACC_CCE_VOFFSET +   \
			 ICP_TIMESYNCACC_PLREGS_CAN_BOFFSET  * (portNum))				 
#define ICP_TIMESYNCACC_CXSL_OFFSET(portNum) \
			(ICP_TIMESYNCACC_PLCAN_OFFSET + ICP_TIMESYNCACC_CXSL_VOFFSET +   \
			 ICP_TIMESYNCACC_PLREGS_CAN_BOFFSET  * (portNum))
#define	ICP_TIMESYNCACC_CXSH_OFFSET(portNum) \
			(ICP_TIMESYNCACC_PLCAN_OFFSET + ICP_TIMESYNCACC_CXSH_VOFFSET +   \
			 ICP_TIMESYNCACC_PLREGS_CAN_BOFFSET  * (portNum))

/* 
 * Bit Masks of Block Level Control Register
 */
/* Auxiliary Target Time Interrupt Mask */
#define ICP_TIMESYNCACC_TSC_ATM_MASK    (1 << 5)
/* Pulse Per Second Interrupt Mask */
#define ICP_TIMESYNCACC_TSC_PPSM_MASK    (1 << 4)
/* Auxiliary Master Mode snapshot Interrupt Mask */
#define ICP_TIMESYNCACC_TSC_AMMS_MASK    (1 << 3)
/* Auxiliary Slave Mode snapshot Interrupt Mask */
#define ICP_TIMESYNCACC_TSC_ASMS_MASK    (1 << 2)
/* Target Time Interrupt Mask */
#define ICP_TIMESYNCACC_TSC_TTM_MASK     (1 << 1)
/* Hardware Assist Reset */
#define ICP_TIMESYNCACC_TSC_RESET        (1 << 0)

/*
 * Bit Masks of Block Level Event Register
 */
/* GBe1 Mode Status - 1 indicates the gbe1 port is running in MII mode
 * 0 indicates the gbe1 port is running in GMII mode
 */
#define ICP_TIMESYNCACC_TSE_GB1M     (1 << 9)
/* GBe0 Mode Status - 1 indicates the Gbe0 port is running in MII mode
 * 0 indicates the gbe1 port is running in GMII mode
 */
#define ICP_TIMESYNCACC_TSE_GB0M     (1 << 8)
/* Auxiliary Target Time Interrupt Pending Event */
#define ICP_TIMESYNCACC_TSE_ATP      (1 << 5)
/* Pulse Per Second Match */
#define ICP_TIMESYNCACC_TSE_PPM      (1 << 4)
/* Auxiliary Master Mode snapshot Event */
#define ICP_TIMESYNCACC_TSE_SNM          (1 << 3)
/* Auxiliary Slave Mode snapshot Event */
#define ICP_TIMESYNCACC_TSE_SNS          (1 << 2)
/* Target Time Interrupt Pending Event */
#define ICP_TIMESYNCACC_TSE_TTIPEND      (1 << 1)

/*
 * Bit Masks of Channel/Port Level Control Register
 */
/* Channel Reset Control Flag */
#define ICP_TIMESYNCACC_CC_CRST          (1 << 2)
/* Timestamp All Messages Control Flag */
#define ICP_TIMESYNCACC_CC_TA            (1 << 1)
/* Timestamp Master or Slave Mode Control Flag */
#define ICP_TIMESYNCACC_CC_MM            (1 << 0)

/*
 * Bit Masks of Channel/Port Level Event Register
 */
/* Receive Snapshot Locked Indicator Flag */
#define ICP_TIMESYNCACC_CE_RXS           (1 << 1)
/* Transmit Snapshot Locked Indicator Flag */
#define ICP_TIMESYNCACC_CE_TXS           (1 << 0)

/*
 * Bit Masks of CAN Channel/Port Level Event Register
 */
/* Valid Indicator Flag */
#define ICP_TIMESYNCACC_CE_VAL           (1 << 1)
/* Overrun Indicator Flag */
#define ICP_TIMESYNCACC_CE_OVR           (1 << 0)

/* Version bit */
#define ICP_TIMESYNCACC_CC_VERSION           (1 << 31)

#define ICP_TIMESYNCACC_CC_MODE_MASK  (0x001F0000)
#define ICP_TIMESYNCACC_CC_MODE_SHIFT  (0x10)


/*
 * Typedefs used in this file
 */

/* Block Level Registers */
typedef struct
{
    UINT32 tsControl;    /* Time Sync Control */
    UINT32 tsEvent;      /* Time Sync Event */
    UINT32 tsAddend;     /* Addend */
    UINT32 tsAccum;      /* Accumulator */
    UINT32 tsTest;       /* Test */
	UINT32 tsPpsCompare; /* Pulse Per Second Compare */
    UINT32 tsSysTimeLo;  /* SystemTime_Low */
    UINT32 tsSysTimeHi;  /* SystemTime_High */
    UINT32 tsTrgtLo;     /* TargetTime_Low */
    UINT32 tsTrgtHi;     /* TargetTime_High */
    UINT32 tsASMSLo;     /* AuxSlaveModeSnap_Low */
    UINT32 tsASMSHi;     /* AuxSlaveModeSnap_High */
    UINT32 tsAMMSLo;     /* AuxMasterModeSnap_Low */
    UINT32 tsAMMSHi;     /* AuxMasterModeSnap_High */
    UINT32 tsATTSHi;	   /*	AuxTargetTimeSnap_High */
    UINT32 tsATTSLo;	   /*	AuxTargetTimeSnap_Low */
} IxTimeSyncAccBlockLevelRegisters;

/* Port Level Registers */
typedef struct
{
    UINT32 tsChControl;  	/* TS Channel Control */
    UINT32 tsChEvent;    	/* TS Channel Event */
    UINT32 tsTxSnapLo;   	/* XMIT Snaphot Low */
    UINT32 tsTxSnapHi;   	/* XMIT Snapshot High */
    UINT32 tsRxSnapLo;   	/* RECV Snapshot Low */
    UINT32 tsRxSnapHi;   	/* RECV Snapshot High */
    UINT32 tsSrcUUIDLo;  	/* SourceUUID Low */
    UINT32 tsSrcUUIDHi;  	/* SequenceID0/SourceUUID High */
	UINT32 tsCANChEvent;	/* TS CAN Channel Event */
	UINT32 tsCANTxSnapLo;	/* CAN XMIT Snapshot Low */
	UINT32 tsCANTxSnapHi;	/* CAN XMIT Snapshot High */
} IxTimeSyncAccPortLevelRegisters;

/* Virtual addresses of time sync hardware registers */
typedef struct
{
    /* Block Level Registers */
    IxTimeSyncAccBlockLevelRegisters  blRegisters;
    /* Port Level Registers */
    IxTimeSyncAccPortLevelRegisters   plRegisters[
        ICP_TIMESYNCACC_MAX_1588PTP_PORT];
} IxTimeSyncAccRegisters;

/*
 * Variable declarations global to TimeSync access component
 */

/* TimeSync Hardware Registers */
static IxTimeSyncAccRegisters ixTsRegisters;

/*
 * Client registered callback routines for 
 * a) the target time reached or exceeded interrupt notification
 * b) the auxiliary time stamps availability interrupt notification
 * c) the auxiliary target time reached or exceeded interrupt notification
 * d) the pulser per second interrupt notification
 */
static IxTimeSyncAccTargetTimeCallback  ixTsTargetTimeCallback    = NULL; /*(a)*/
static IxTimeSyncAccAuxTimeCallback     ixTsAuxMasterTimeCallback = NULL; /*(b)*/
static IxTimeSyncAccAuxTimeCallback     ixTsAuxSlaveTimeCallback  = NULL; /*(b)*/
static IxTimeSyncAccAuxTargetTimeCallback  ixTsAuxTargetTimeCallback    = NULL; /*(c)*/
static IxTimeSyncAccPulsePerSecondCallback  ixTsPulsePerSecondCallback    = NULL; /*(d)*/

/*
 * The transmit and receive timestamp statistics 
 */
static IxTimeSyncAccStats ixTsStats = { 0,0 };

/*
 *   Indicates whether hardware access library has been initialized
 */
static UINT32 ixInitialized = 0;

/*
 *   Hw channel to PTP port mapping
 */
static UINT32 ixPortToChnlMap[ICP_TIMESYNCACC_MAX_1588PTP_PORT];

/*
 * Support functions declarations
 */

PRIVATE IxTimeSyncAcc1588PTPPortMode
ixTimeSyncAccPTPPortModeGet(IxTimeSyncAcc1588PTPPort ptpPort);

/*
 * Local functions definitions.
 */

/*
 * ------------------------------------------------------------------ *
 * Block level configuration support functions definitions
 * ------------------------------------------------------------------ *
 */

/* Enable Auxiliary Master Mode Snapshot Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAmmsInterruptMaskSet(void)
{
    /* SET the amms bit */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
                              ICP_TIMESYNCACC_TSC_AMMS_MASK);
} /* end of ixTimeSyncAccControlAmmsInterruptMaskSet() function */

/* Enable Auxiliary Slave Mode Snapshot Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAsmsInterruptMaskSet(void)
{
    /* SET the asms bit */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
                              ICP_TIMESYNCACC_TSC_ASMS_MASK);
} /* end of ixTimeSyncAccControlAsmsInterruptMaskSet() function */

/* Enable Target Time Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlTtmInterruptMaskSet(void)
{
    /* SET the ttm bit */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
                               ICP_TIMESYNCACC_TSC_TTM_MASK);
} /* end of ixTimeSyncAccControlTtmInterruptMaskSet() function */

/* Enable Auxiliary Target Time Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAtmInterruptMaskSet(void)
{
	/* SET the atm bit */
	ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
    	                       ICP_TIMESYNCACC_TSC_ATM_MASK);
} /* end of ixTimeSyncAccControlAtmInterruptMaskSet() function */

/* Enable Pulse Per Second Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPpsmInterruptMaskSet(void)
{
	/* SET the ppsm bit */
	ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
                           ICP_TIMESYNCACC_TSC_PPSM_MASK);
} /* end of ixTimeSyncAccControlPpsmInterruptMaskSet() function */

/* Get Auxiliary Master Mode Snapshot Interrupt Mask value */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 
ixTimeSyncAccControlAmmsInterruptMaskGet(void)
{
    /* Is the amms bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsControl,
                                            ICP_TIMESYNCACC_TSC_AMMS_MASK);
} /* end of ixTimeSyncAccControlAmmsInterruptMaskGet() function */

/* Get Auxiliary Slave Mode Snapshot Interrupt  Mask value */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlAsmsInterruptMaskGet(void)
{
    /* Is the asms bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsControl,
                                            ICP_TIMESYNCACC_TSC_ASMS_MASK);
} /* end of ixTimeSyncAccControlAsmsInterruptMaskGet() function */

/* Get Target Time Interrupt Mask value */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlTtmInterruptMaskGet(void)
{
    /* Is the ttm bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsControl,
                                            ICP_TIMESYNCACC_TSC_TTM_MASK);
} /* end of ixTimeSyncAccControlTtmInterruptMaskGet() function */

/* Clarkspoint specific code */
/* Get Auxiliary Target Time Interrupt Mask value */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlAtmInterruptMaskGet(void)
{
	/* Is the atm bit SET? */
	return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsControl,
   	                                     ICP_TIMESYNCACC_TSC_ATM_MASK);
} /* end of ixTimeSyncAccControlAtmInterruptMaskGet() function */

/* Get Pulse Per Second Interrupt Mask value */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPpsmInterruptMaskGet(void)
{
	/* Is the ppsm bit SET? */
	return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsControl,
   	                                     ICP_TIMESYNCACC_TSC_PPSM_MASK);
} /* end of ixTimeSyncAccControlPpsmInterruptMaskGet() function */

/* Disable Auxiliary Master Mode Snapshot Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAmmsInterruptMaskClear(void)
{
    /* CLEAR the amms bit */
    ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
                                 ICP_TIMESYNCACC_TSC_AMMS_MASK);
} /* end of ixTimeSyncAccControlAmmsInterruptMaskClear() function */

/* Disable Auxiliary Slave Mode Snapshot Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAsmsInterruptMaskClear(void)
{
    /* CLEAR the asms bit */
    ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
                                 ICP_TIMESYNCACC_TSC_ASMS_MASK);
} /* end of ixTimeSyncAccControlAsmsInterruptMaskClear() function */

/* Disable Target Time Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlTtmInterruptMaskClear(void)
{
    /* CLEAR the ttm bit */
    ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
                                 ICP_TIMESYNCACC_TSC_TTM_MASK);
} /* end of ixTimeSyncAccControlTtmInterruptMaskClear() function */

	/* Disable Target Time Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlAtmInterruptMaskClear(void)
{
	/* CLEAR the atm bit */
	ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
	                             ICP_TIMESYNCACC_TSC_ATM_MASK);
} /* end of ixTimeSyncAccControlAtmInterruptMaskClear() function */

/* Disable Pulse Per Second Interrupt */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPpsmInterruptMaskClear(void)
{
	/* CLEAR the ppsm bit */
	ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
	                             ICP_TIMESYNCACC_TSC_PPSM_MASK);
} /* end of ixTimeSyncAccControlPpsmInterruptMaskClear() function */

/* Reset Hardware Assist block */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlReset(void)
{
    /* SET the rst bit */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsControl,
                               ICP_TIMESYNCACC_TSC_RESET);
    /* CLEAR the rst bit */
    ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.blRegisters.tsControl,
                                 ICP_TIMESYNCACC_TSC_RESET);
} /* end of ixTimeSyncAccControlReset() function */

/* Poll for Auxiliary Master Mode Snapshot Captured event */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32 
ixTimeSyncAccEventAmmsFlagGet(void)
{
    /* Is the snm bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsEvent,
                                            ICP_TIMESYNCACC_TSE_SNM);
} /* end of ixTimeSyncAccEventAmmsFlagGet() function */

/* Poll for Auxiliary Slave Mode Snapshot Captured event */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccEventAsmsFlagGet(void)
{
    /* Is the sns bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsEvent,
                                            ICP_TIMESYNCACC_TSE_SNS);
} /* end of ixTimeSyncAccEventAsmsFlagGet() function */

/* Poll for Target Time Reached event function */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccEventTtmFlagGet(void)
{
    /* Is the ttipend bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsEvent,
                                            ICP_TIMESYNCACC_TSE_TTIPEND);
} /* end of ixTimeSyncAccEventTtmFlagGet() function */

/* Poll for Auxiliary Target Time Reached event function */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccEventAtmFlagGet(void)
{
	/* Is the atp bit SET? */
	return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsEvent,
    	                                    ICP_TIMESYNCACC_TSE_ATP);
} /* end of ixTimeSyncAccEventAtmFlagGet() function */

/* Poll for Pulse Per Second Reached event function */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccEventPpsmFlagGet(void)
{
	/* Is the ppsm bit SET? */
 	return ICP_TIMESYNCACC_BIT_SET_CHECK(ixTsRegisters.blRegisters.tsEvent,
  	                                       ICP_TIMESYNCACC_TSE_PPM);
} /* end of ixTimeSyncAccEventPpsmFlagGet() function */

/* Clear Auxiliary Master Mode Snapshot Captured event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccEventAmmsFlagClear(void)
{
    /* CLEAR the snm bit by writing '1' onto it */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsEvent,
                               ICP_TIMESYNCACC_TSE_SNM);
} /* end of ixTimeSyncAccEventAmmsFlagClear() function */

/* Clear Auxiliary Slave Mode Snapshot Captured event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccEventAsmsFlagClear(void)
{
    /* CLEAR the sns bit by writing '1' onto it */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsEvent,
                               ICP_TIMESYNCACC_TSE_SNS);
} /* end of ixTimeSyncAccEventAsmsFlagClear() function */

/* Clear Target Time Reached event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccEventTtmFlagClear(void)
{
    /* CLEAR the ttipend bit by writing '1' onto it */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsEvent,
                               ICP_TIMESYNCACC_TSE_TTIPEND);
} /* end of ixTimeSyncAccEventTtmFlagClear() function */

/* Clear Auxiliary Target Time Reached event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccEventAtmFlagClear(void)
{
	/* CLEAR the atp bit by writing '1' onto it */
	ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsEvent,
    	                       ICP_TIMESYNCACC_TSE_ATP);
} /* end of ixTimeSyncAccEventAtmFlagClear() function */

/* Clear Pulse Per Second Time Reached event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccEventPpsmFlagClear(void)
{
	/* CLEAR the pps bit by writing '1' onto it */
	ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.blRegisters.tsEvent,
    	                       ICP_TIMESYNCACC_TSE_PPM);
} /* end of ixTimeSyncAccEventAtmFlagClear() function */

/*
 * ------------------------------------------------------------------ *
 * Block level timestamp support functions definitions
 * ------------------------------------------------------------------ *
 */

/* Set System Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccSystemTimeSnapshotSet (
    UINT32 systemTimeLow,
    UINT32 systemTimeHigh)
{
    /* Update the System Time Low Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsSysTimeLo,
                                 systemTimeLow);
    /* Update the System Time High Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsSysTimeHi,
                                 systemTimeHigh);
} /* end of ixTimeSyncAccSystemTimeSnapshotSet() function */

/* Get System Time Low value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccSystemTimeSnapshotGet(
    UINT32 *systemTimeLow,
    UINT32 *systemTimeHigh)
{
    /* Fetch the System Time Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsSysTimeLo,
                                systemTimeLow);
    /* Fetch the System Time High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsSysTimeHi,
                                systemTimeHigh);
} /* end of ixTimeSyncAccSystemTimeSnapshotGet() function */

/* Set Target Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccTargetTimeSnapshotSet (
    UINT32 targetTimeLow,
    UINT32 targetTimeHigh)
{
    /* Update the Target Time Low Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsTrgtLo,
                                 targetTimeLow);
    /* Update the Target Time High Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsTrgtHi,
                                 targetTimeHigh);
} /* end of ixTimeSyncAccTargetTimeSnapshotSet() function */

/* Get Target Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccTargetTimeSnapshotGet(
    UINT32 *targetTimeLow,
    UINT32 *targetTimeHigh)
{
    /* Fetch the Target Time Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsTrgtLo,
                                targetTimeLow);
    /* Fetch the Target Time High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsTrgtHi,
                                targetTimeHigh);
} /* end of ixTimeSyncAccTargetTimeSnapshotGet() function */

/* Set Auxiliary Target Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAuxTargetTimeSnapshotSet (
	UINT32 auxTargetTimeLow,
	UINT32 auxTargetTimeHigh)
{
	/* Update the Auxiliary Target Time Low Register contents */
	ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsATTSLo,   		
    	                         auxTargetTimeLow);
	/* Update the Auxliliary Target Time High Register contents */
	ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsATTSHi,
    	                         auxTargetTimeHigh);
} /* end of ixTimeSyncAccAuxTargetTimeSnapshotSet() function */

/* Get Auxiliary Target Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAuxTargetTimeSnapshotGet(
	UINT32 *auxTargetTimeLow,
	UINT32 *auxTargetTimeHigh)
{
	/* Fetch the Auxiliary Target Time Low Register contents */
	ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsATTSLo,
    	                        auxTargetTimeLow);
	/* Fetch the Auxiliary Target Time High Register contents */
	ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsATTSHi,
    	                        auxTargetTimeHigh);
} /* end of ixTimeSyncAccAuxTargetTimeSnapshotGet() function */

/* Set Pulse Per Second Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPulsePerSecondCompareSet (UINT32 ppsTime)
{
    /* Update the Pulse Per Second Compare Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsPpsCompare, ppsTime);
} /* end of ixTimeSyncAccPulsePerSecondCompareSet() function */

/* Get Pulse Per Second Time value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPulsePerSecondCompareGet (UINT32 *ppsTime)
{
    /* Fetch the Pulse Per Second Compare Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsPpsCompare, ppsTime);
} /* end of ixTimeSyncAccPulsePerSecondCompareGet() function */

/* Set Frequency Scaling Value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAddendFsvSet (UINT32 fsv)
{
    /* Update the Addend Register contents */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.blRegisters.tsAddend, fsv);
} /* end of ixTimeSyncAccAddendFsvSet() function */

/* Get Frequency Scaling Value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAddendFsvGet (UINT32 *fsv)
{
    /* Fetch the Addend Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsAddend, fsv);
} /* end of ixTimeSyncAccAddendFsvGet() function */

/* Get AMMS value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAuxMasterModeSnapshotGet (
    UINT32 *ammsLow,
    UINT32 *ammsHigh)
{
    /* Fetch the Auxiliary Master Mode Snapshot Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsAMMSLo, ammsLow);
    /* Fetch the Auxiliary Master Mode Snapshot High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsAMMSHi, ammsHigh);
} /* end of ixTimeSyncAccAuxMasterModeSnapshotGet() function */

/* Get ASMS value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccAuxSlaveModeSnapshotGet (
    UINT32 *asmsLow,
    UINT32 *asmsHigh)
{
    /* Fetch the Auxiliary Slave Mode Snapshot Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsASMSLo, asmsLow);
    /* Fetch the Auxiliary Slave Mode Snapshot High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.blRegisters.tsASMSHi, asmsHigh);
} /* end of ixTimeSyncAccAuxSlaveModeSnapshotGet() function */


/*
 * ------------------------------------------------------------------ *
 * Port level configuration support functions definitions
 * ------------------------------------------------------------------ *
 */

/* Reset PTP channel hardware */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccChannelReset(UINT32 ptpPort)
{
    /* SET the rst bit */
    ICP_TIMESYNCACC_BIT_SET(ixTsRegisters.plRegisters[ptpPort].tsChControl,
                               ICP_TIMESYNCACC_CC_CRST);
    /* CLEAR the rst bit */
    ICP_TIMESYNCACC_BIT_CLEAR(ixTsRegisters.plRegisters[ptpPort].tsChControl,
                                 ICP_TIMESYNCACC_CC_CRST);
} /* end of ixTimeSyncAccControlReset() function */

/* Set the channel mode to 1588 Master */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortMasterModeSet (
    UINT32 ptpPort,
    UINT32 masterMode)
{
    /* SET or CLEAR the Master Mode */
    if (TRUE == masterMode)
    {
        /* SET the mm bit */
        ICP_TIMESYNCACC_BIT_SET(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_MM);
    }
    else /* else of if (TRUE == masterMode) */
    {
        /* CLEAR the mm bit */
        ICP_TIMESYNCACC_BIT_CLEAR(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_MM);
    } /* if (TRUE == masterMode) */
} /* end of ixTimeSyncAccControlPTPPortMasterModeSet() function */

/* Check for 1588 master mode of channel */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortMasterModeGet (UINT32 ptpPort)
{
    /* Is the mm bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsChControl,
               ICP_TIMESYNCACC_CC_MM);
} /* end of ixTimeSyncAccControlPTPPortMasterModeGet() function */

/* Set Timestamp all or only PTP messages flag */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortPTPMsgTimestampSet (
    UINT32 ptpPort,
    UINT32 allMsg)
{
    /* SET or CLEAR the All Message Timestamping */
    if (TRUE == allMsg)
    {
        /* SET the ta bit */
        ICP_TIMESYNCACC_BIT_SET(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_TA);
    }
    else /* else of if (TRUE == allMsg) */
    {
    /* CLEAR the ta bit */
        ICP_TIMESYNCACC_BIT_CLEAR(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_TA);
    } /* if (TRUE == allMsg) */
} /* end of ixTimeSyncAccControlPTPPortPTPMsgTimestampSet() function */

/* Check for Timestamp all or only PTP messages flag */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortPTPMsgTimestampGet(UINT32 ptpPort)
{
    /* Is the ta bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsChControl,
               ICP_TIMESYNCACC_CC_TA);
} /* end of ixTimeSyncAccControlPTPPortPTPMsgTimestampGet() function */



/* Set the 1588 version number */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPVersionSet (
    UINT32 ptpPort,
    UINT32 versionVal)
{
    /* SET or CLEAR the All Message Timestamping */
    if (TRUE == versionVal)
    {
        /* SET the ta bit */
        ICP_TIMESYNCACC_BIT_SET(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_VERSION);
    }
    else /* else of if (TRUE == allMsg) */
    {
    /* CLEAR the ta bit */
        ICP_TIMESYNCACC_BIT_CLEAR(
            ixTsRegisters.plRegisters[ptpPort].tsChControl,
            ICP_TIMESYNCACC_CC_VERSION);
    } /* if (TRUE == allMsg) */
} /* end of ixTimeSyncAccControlPTPVersionSet() function */

/* Get the 1588 version number */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPVersionGet (UINT32 ptpPort)
{
    /* Is the ta bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsChControl,
               ICP_TIMESYNCACC_CC_VERSION);
} /* end of ixTimeSyncAccControlPTPVersionGet() function */

/* Set the 1588 operation mode number */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPOperationModeSet (
    UINT32 ptpPort,
    UINT32 modeVal)
{
    UINT32 regVal;
    
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsChControl,
                                &regVal);

    regVal = (regVal & (~ICP_TIMESYNCACC_CC_MODE_MASK)) | ( modeVal << ICP_TIMESYNCACC_CC_MODE_SHIFT );  

    ICP_TIMESYNCACC_REG_WRITE( ixTsRegisters.plRegisters[ptpPort].tsChControl,
                                regVal );

} /* end of ixTimeSyncAccControlPTPOperationModeSet() function */

/* Get the 1588 operation mode number */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPOperationModeGet (UINT32 ptpPort)
{
    UINT32 regVal;
    UINT32 modeVal;
    
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsChControl,
                                &regVal);

    modeVal = ( regVal & ICP_TIMESYNCACC_CC_MODE_MASK) >> ICP_TIMESYNCACC_CC_MODE_SHIFT ;  

    return modeVal; 

} /* end of ixTimeSyncAccControlPTPOperationModeGet() function */




/*
 * ------------------------------------------------------------------ *
 * Port level timestamp support functions definitions
 * ------------------------------------------------------------------ *
 */

/* CAN Timestamp available */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortValFlagGet (UINT32 ptpPort)
{
    /* Is the valid bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsCANChEvent,
               ICP_TIMESYNCACC_CE_VAL);
} /* end of ixTimeSyncAccControlPTPPortValFlagGet() function */

/* CAN Timestamp overrun */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortOvrFlagGet (UINT32 ptpPort)
{
    /* Is the ovr bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsCANChEvent,
               ICP_TIMESYNCACC_CE_OVR);
} /* end of ixTimeSyncAccControlPTPPortOvrFlagGet() function */

/* Clear CAN Timestamp valid flag */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortValFlagClear(UINT32 ptpPort)
{
    /* CLEAR the valid bit by writing '1' onto it */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.plRegisters[ptpPort].tsCANChEvent,
                               ICP_TIMESYNCACC_CE_VAL);
} /* end of ixTimeSyncAccControlPTPPortValFlagClear() function */

/* Clear CAN Timestamp overrun flag */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortOvrFlagClear(UINT32 ptpPort)
{
    /* CLEAR the overrun bit by writing '1' onto it */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.plRegisters[ptpPort].tsCANChEvent,
                               ICP_TIMESYNCACC_CE_OVR);
} /* end of ixTimeSyncAccControlPTPPortOvrFlagClear() function */

/* Receive Timestamp available */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortRxsFlagGet (UINT32 ptpPort)
{
    /* Is the rxs bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsChEvent,
               ICP_TIMESYNCACC_CE_RXS);
} /* end of ixTimeSyncAccControlPTPPortRxsFlagGet() function */

/* Transmit Timestamp available */
PRIVATE ICP_TIMESYNCACC_INLINE UINT32
ixTimeSyncAccControlPTPPortTxsFlagGet (UINT32 ptpPort)
{
    /* Is the txs bit SET? */
    return ICP_TIMESYNCACC_BIT_SET_CHECK(
               ixTsRegisters.plRegisters[ptpPort].tsChEvent,
               ICP_TIMESYNCACC_CE_TXS);
} /* end of ixTimeSyncAccControlPTPPortTxsFlagGet() function */

/* Clear Receive Timestamp available event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortRxsFlagClear(UINT32 ptpPort)
{
    /* CLEAR the rxs bit by writing '1' onto it */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.plRegisters[ptpPort].tsChEvent,
                               ICP_TIMESYNCACC_CE_RXS);
} /* end of ixTimeSyncAccControlPTPPortRxsFlagClear() function */

/* Clear Transmit Timestamp available event */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccControlPTPPortTxsFlagClear(UINT32 ptpPort)
{
    /* CLEAR the txs bit by writing '1' onto it */
    ICP_TIMESYNCACC_REG_WRITE(ixTsRegisters.plRegisters[ptpPort].tsChEvent,
                               ICP_TIMESYNCACC_CE_TXS);
} /* end of ixTimeSyncAccControlPTPPortTxsFlagClear() function */

/* Get PTP CAN Port Timestamp value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPTPPortCANSnapshotGet (
    UINT32 ptpPort,
    UINT32 *rxsLow,
    UINT32 *rxsHigh)
{
    /* Fetch the Receive Timestamp/Snapshot Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsCANTxSnapLo,
                                rxsLow);
    /* Fetch the Receive Timestamp/Snapshot High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsCANTxSnapHi,
                                rxsHigh);
} /* end of ixTimeSyncAccPTPPortCANSnapshotGet() function */

/* Get PTP Port Rx Timestamp value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPTPPortReceiveSnapshotGet (
    UINT32 ptpPort,
    UINT32 *rxsLow,
    UINT32 *rxsHigh)
{
    /* Fetch the Receive Timestamp/Snapshot Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsRxSnapLo,
                                rxsLow);
    /* Fetch the Receive Timestamp/Snapshot High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsRxSnapHi,
                                rxsHigh);
} /* end of ixTimeSyncAccPTPPortReceiveSnapshotGet() function */

/* Get PTP Port Tx Timestamp value */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPTPPortTransmitSnapshotGet (
    UINT32 ptpPort,
    UINT32 *txsLow,
    UINT32 *txsHigh)
{
    /* Fetch the Transmit Timestamp/Snapshot Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsTxSnapLo,
                                txsLow);
    /* Fetch the Transmit Timestamp/Snapshot High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsTxSnapHi,
                                txsHigh);
} /* end of ixTimeSyncAccPTPPortTransmitSnapshotGet() function */

/* Get UUID High (16-bit value) & Sequence ID (16-bit value) of PTP message */
PRIVATE ICP_TIMESYNCACC_INLINE void
ixTimeSyncAccPTPMsgUuidSeqIdGet (
    UINT32 ptpPort,
    UINT32 *uuidLow,
    UINT16 *uuidHigh,
    UINT16 *seqId)
{
    /* Local variables */
    UINT32 seqIdUuidHigh = 0;

    /* Fetch the UUID Low Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsSrcUUIDLo,
                                uuidLow);
    /* Fetch the Sequence ID and Source UUID High Register contents */
    ICP_TIMESYNCACC_REG_READ(ixTsRegisters.plRegisters[ptpPort].tsSrcUUIDHi,
                                &seqIdUuidHigh);

    *seqId    = (seqIdUuidHigh >> ICP_TIMESYNCACC_SID_LOC);
    *uuidHigh = (ICP_TIMESYNCACC_LSB_SHORT_MASK & seqIdUuidHigh);
} /* end of ixTimeSyncAccPTPMsgUuidHighSeqIdGet() function */

#endif


