/*****************************************************************************
 * @ingroup 1588
 *
 * @file ixtimesyncacc.c
 *
 * @description
 *        The 1588 driver module enables a client application to have
 *         access to the 1588 time synchronization hardware assist block. 
 *         A client application can use this driver to retreive tx and rx timestamps 
 *         of IEEE 1588(TM) PTP messages captured on ethernet or CAN ports. 
 *         Access is provided by a set of APIs that can be called directly or through 
 *         ioctl commands.  
 *
 * INTEL CONFIDENTIAL
 * Copyright 2007,2008,2009 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to the
 * source code ("Material") are owned by Intel Corporation or its suppliers or
 * licensors. Title to the Material remains with Intel Corporation or its
 * suppliers and licensors. The Material may contain trade secrets and proprietary
 * and confidential information of Intel Corporation and its suppliers and
 * licensors, and is protected by worldwide copyright and trade secret laws and
 * treaty provisions. No part of the Material may be used, copied, reproduced,
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Intels prior express written permission.
 * No license under any patent, copyright, trade secret or other intellectual
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise. Any license under such intellectual property rights must be
 * express and approved by Intel in writing.
 * 
 * Include any supplier copyright notices as supplier requires Intel to use.
 * Include supplier trademarks or logos as supplier requires Intel to use,
 * preceded by an asterisk.
 * An asterisked footnote can be added as follows: 
 *   *Third Party trademarks are the property of their respective owners.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Intel or Intels suppliers
 * or licensors in any way.
 * 
 *  version: Embedded.X.1.0.3-127
 ****************************************************************************/

/*
 * User defined include files
 */
#include "ixtimesyncacc.h"
#include "IxTimeSyncAcc_p.h"

/*
 * Support functions definitions
 */


/* Function to determine the port mode */
PRIVATE IxTimeSyncAcc1588PTPPortMode
ixTimeSyncAccPTPPortModeGet(IxTimeSyncAcc1588PTPPort ptpPort)
{
    /* Local variables */
    UINT32 masterMode = FALSE;
    UINT32 allMsgMode = FALSE;
    IxTimeSyncAcc1588PTPPortMode ptpPortMode = 
                                 ICP_TIMESYNCACC_1588PTP_PORT_SLAVE;

    /* Get the Mode of the PTP Port */
    masterMode = ixTimeSyncAccControlPTPPortMasterModeGet
        (ixPortToChnlMap[ptpPort]);
    allMsgMode = ixTimeSyncAccControlPTPPortPTPMsgTimestampGet
        (ixPortToChnlMap[ptpPort]);

    /* Is ANY mode (all message timestamp mode) on? */
    if (FALSE == allMsgMode)
    {
        /* Is Master mode on? */
        if (TRUE == masterMode)
        {
            ptpPortMode = ICP_TIMESYNCACC_1588PTP_PORT_MASTER;
        } /* if (TRUE == masterMode)  */
        else
        {
            ptpPortMode = ICP_TIMESYNCACC_1588PTP_PORT_SLAVE;
        } /* end of if (TRUE == masterMode)  */
    } /* if (FALSE == allMsgMode) */
    else
    {
        /* 
         * When Any mode is on (the ta bit is set) we do not care
         * for Master/Slave mode (the mm bit status) since all the
         * packets get time stamped anyways.
         */
        ptpPortMode = ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE;
    } /* end of if (FALSE == allMsgMode) */

    return ptpPortMode;
} /* end of ixTimeSyncAccPTPPortModeGet() function */


/*
 * ------------------------------------------------------------------ *
 *                        Public API definitions
 * ------------------------------------------------------------------ *
 */

/* Function for setting the base address registers */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccBlPlBaseAddressesSet(UINT32 RegsVirtualBaseAddr)
{
    /* Local variables */
    UINT32 ptpChannelNum = 0;

    if ( !RegsVirtualBaseAddr )
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    }

    /* Virtual Addresses assignment for Block Level Registers */
    ixTsRegisters.blRegisters.tsControl   = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_TSC_OFFSET;
    ixTsRegisters.blRegisters.tsEvent     = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_TSE_OFFSET;
    ixTsRegisters.blRegisters.tsAddend    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ADD_OFFSET;
    ixTsRegisters.blRegisters.tsAccum     = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ACC_OFFSET;
    ixTsRegisters.blRegisters.tsTest     = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_TST_OFFSET;
    ixTsRegisters.blRegisters.tsSysTimeLo = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_STL_OFFSET;
    ixTsRegisters.blRegisters.tsSysTimeHi = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_STH_OFFSET;
    ixTsRegisters.blRegisters.tsTrgtLo    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_TTL_OFFSET;
    ixTsRegisters.blRegisters.tsTrgtHi    =
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_TTH_OFFSET;
    ixTsRegisters.blRegisters.tsASMSLo    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ASSL_OFFSET;
    ixTsRegisters.blRegisters.tsASMSHi    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ASSH_OFFSET;
    ixTsRegisters.blRegisters.tsAMMSLo    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_AMSL_OFFSET;
    ixTsRegisters.blRegisters.tsAMMSHi    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_AMSH_OFFSET;
    ixTsRegisters.blRegisters.tsPpsCompare     = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_PPS_OFFSET;                 
    ixTsRegisters.blRegisters.tsATTSLo    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ATTL_OFFSET;
    ixTsRegisters.blRegisters.tsATTSHi    = 
        RegsVirtualBaseAddr + ICP_TIMESYNCACC_ATTH_OFFSET;

    /* Virtual Addresses assignment for Ethernet Port Level Registers */
    for (ptpChannelNum = 0; 
         ptpChannelNum < ICP_TIMESYNCACC_MAX_1588ETH_CHNL;
         ptpChannelNum++)
    {
        ixTsRegisters.plRegisters[ptpChannelNum].tsChControl  = 
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_CC_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsChEvent    = 
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_CE_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsTxSnapLo   = 
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_XSL_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsTxSnapHi   = 
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_XSH_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsRxSnapLo   =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_RSL_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsRxSnapHi   =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_RSH_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsSrcUUIDLo  =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_UID_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsSrcUUIDHi  =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_SID_OFFSET(ptpChannelNum);
    } /* end of for (ptpPortNum = 0; 
                     ptpPortNum < ICP_TIMESYNCACC_MAX_1588ETH_CHNL;
                     ptpPortNum++ ) */

    /* Virtual Addresses assignment for CAN Port Level Registers */
    for (ptpChannelNum = 0; 
         ptpChannelNum < ICP_TIMESYNCACC_MAX_1588CAN_CHNL;
         ptpChannelNum++)
    {
        ixTsRegisters.plRegisters[ptpChannelNum].tsCANChEvent   =
               RegsVirtualBaseAddr + ICP_TIMESYNCACC_CCE_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsCANTxSnapLo  =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_CXSL_OFFSET(ptpChannelNum);
        ixTsRegisters.plRegisters[ptpChannelNum].tsCANTxSnapHi  =
            RegsVirtualBaseAddr + ICP_TIMESYNCACC_CXSH_OFFSET(ptpChannelNum);
    } /* end of for (ptpPortNum = 0; 
                     ptpPortNum < ICP_TIMESYNCACC_MAX_1588CAN_CHNL;
                     ptpPortNum++ ) */

    /* PTP port to hardware channel mapping */
    ixPortToChnlMap[ICP_TIMESYNCACC_GBE_0_1588PTP_PORT] = 
        ICP_TIMESYNCACC_GBE_0_1588PTP_CHNL;
    ixPortToChnlMap[ICP_TIMESYNCACC_GBE_1_1588PTP_PORT] = 
        ICP_TIMESYNCACC_GBE_1_1588PTP_CHNL;
    ixPortToChnlMap[ICP_TIMESYNCACC_CAN_0_1588PTP_PORT] = 
        ICP_TIMESYNCACC_CAN_0_1588PTP_CHNL;
    ixPortToChnlMap[ICP_TIMESYNCACC_CAN_1_1588PTP_PORT] = 
        ICP_TIMESYNCACC_CAN_1_1588PTP_CHNL;

    /* Clear the snapshot availability condition for both master aux and slave
        aux */
    ixTimeSyncAccEventAmmsFlagClear();
    ixTimeSyncAccEventAsmsFlagClear();

    /* Init Callback Routines */
    ixTsTargetTimeCallback    = (IxTimeSyncAccTargetTimeCallback) NULL;
    ixTsAuxMasterTimeCallback = (IxTimeSyncAccAuxTimeCallback) NULL;
    ixTsAuxSlaveTimeCallback  = (IxTimeSyncAccAuxTimeCallback) NULL;
    ixTsAuxTargetTimeCallback  = (IxTimeSyncAccAuxTargetTimeCallback) NULL;
    ixTsPulsePerSecondCallback = (IxTimeSyncAccPulsePerSecondCallback) NULL;

    /* Clear Stats */
    ixTsStats.rxMsgs = ixTsStats.txMsgs = 0;

    /* module is initialized */
    ixInitialized = 1;

    return ICP_TIMESYNCACC_SUCCESS;
}  /* end of ixTimeSyncAccBlPlBaseAddressesSet() function */

/*
 * Process TimeSync interrupt events
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccIsr(void)
{
    /* Local variables */
    IxTimeSyncAccTimeValue targetTime = {0, 0};
    IxTimeSyncAccTimeValue auxTime = {0, 0};
    IxTimeSyncAccTimeValue auxTargetTime = {0, 0};
    UINT32 pulsePerSecond = 0;
    
    /* 
     * Handle the Interrupts in the following order
     * 
     * 1 - Target Time Reached/Hit Condition
     * 2 - Auxiliary Master Timestamp
     * 3 - Auxiliary Slave Timestamp 
     * 4 - Auxiliary Target Time Reached/Hit Condition
     * 5 - Pulse Per Second
     */

     /*
     * Also verify that valid callbacks are available for each of the following
     * since the client application may choose to use one or more of the following
     * in interrupt mode while others in non-interrupt mode i.e., makes use of poll
     * or get methods in which case there is no valid callback registered.
     */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Handle Target Time Reached or Exceeded Interrupt */
    if ((NULL != ixTsTargetTimeCallback) && 
        (TRUE == ixTimeSyncAccEventTtmFlagGet()))
    {
        /* Disable interrupt */
        ixTimeSyncAccControlTtmInterruptMaskClear();

        /* Target Time registers contents  */
        ixTimeSyncAccTargetTimeSnapshotGet(&targetTime.timeValueLowWord,
                                           &targetTime.timeValueHighWord);

        /* Invoke client callback */
        (*ixTsTargetTimeCallback)(targetTime);

        /* Clear the target time reached condition (ttipend bit) */
        ixTimeSyncAccEventTtmFlagClear();

    } /* end of if ((NULL != ixTsTargetTimeCallback) && 
                (TRUE == ixTimeSyncAccEventTtmFlagGet())) */

    /* Handle Auxiliary Master Mode Snapshot Interrupt */
    if ((NULL != ixTsAuxMasterTimeCallback) &&
        (TRUE == ixTimeSyncAccEventAmmsFlagGet()))
    {
        /* Disable interrupt */
        ixTimeSyncAccControlAmmsInterruptMaskClear();

        /* Fetch Auxiliary Master Mode Snapshot */
        ixTimeSyncAccAuxMasterModeSnapshotGet(&auxTime.timeValueLowWord,
                                              &auxTime.timeValueHighWord);

        /* Return Auxiliary Master Mode Snapshot */
        (*ixTsAuxMasterTimeCallback)(ICP_TIMESYNCACC_AUXMODE_MASTER,auxTime);

        /* Clear the snapshot availability condition */
        ixTimeSyncAccEventAmmsFlagClear();

    } /* end of if ((NULL != ixTsAuxMasterTimeCallback) &&
                    (TRUE == ixTimeSyncAccEventAmmsFlagGet())) */

    /* Handle Auxiliary Slave Mode Snapshot Interrupt */
    if ((NULL != ixTsAuxSlaveTimeCallback) &&
        (TRUE == ixTimeSyncAccEventAsmsFlagGet()))
    {
        /* Disable interrupt */
        ixTimeSyncAccControlAsmsInterruptMaskClear();

        /* Fetch Auxiliary Slave Mode Snapshot */
        ixTimeSyncAccAuxSlaveModeSnapshotGet(&auxTime.timeValueLowWord,
                                             &auxTime.timeValueHighWord);

        /* Return Auxiliary Slave Mode Snapshot */
        (*ixTsAuxSlaveTimeCallback)(ICP_TIMESYNCACC_AUXMODE_SLAVE,auxTime);

        /* Clear the snapshot availability condition */
        ixTimeSyncAccEventAsmsFlagClear();

    } /* end of if ((NULL != ixTsAuxSlaveTimeCallback) &&
                    (TRUE == ixTimeSyncAccEventAsmsFlagGet())) */

    /* Handle Auxiliary Target Time Reached or Exceeded Interrupt */
    if ((NULL != ixTsAuxTargetTimeCallback) && 
        (TRUE == ixTimeSyncAccEventAtmFlagGet()))
    {
        /* Disable interrupt */
        ixTimeSyncAccControlAtmInterruptMaskClear();

        /* Auxiliary Target Time registers contents  */
        ixTimeSyncAccAuxTargetTimeSnapshotGet(&auxTargetTime.timeValueLowWord,
                                           &auxTargetTime.timeValueHighWord);

        /* Invoke client callback */
        (*ixTsAuxTargetTimeCallback)(auxTargetTime);

        /* Clear the auxiliary target time reached condition (atp bit) */
        ixTimeSyncAccEventAtmFlagClear();

    } /* end of if ((NULL != ixTsAuxTargetTimeCallback) && 
                (TRUE == ixTimeSyncAccEventAtmFlagGet())) */                

   /* Handle Pulse Per Second Interrupt */
    if ((NULL != ixTsPulsePerSecondCallback) && 
          (TRUE == ixTimeSyncAccEventPpsmFlagGet()))
      {
      /* Disable interrupt */  
      ixTimeSyncAccControlPpsmInterruptMaskClear();

      /* Pulse per Second registers contents */ 
      ixTimeSyncAccPulsePerSecondCompareGet(&pulsePerSecond);

      /* Invoke client callback */
      (*ixTsPulsePerSecondCallback)(pulsePerSecond);

     /* Clear the pulse per second (ppsm bit) */
      ixTimeSyncAccEventPpsmFlagClear();

   } /* end of if ((NULL != ixTsPulsePerSecondCallback) && 
               (TRUE == ixTimeSyncAccEventPpsmFlagGet())) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccIsr() function */

/* Configure IEEE 1588 Hardware Assist message detection 
 * on a particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPPortConfigSet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPPortMode ptpPortMode)
{
    /* Verify the parameters for proper values */
    if (((ptpPort != ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) &&
         (ptpPort != ICP_TIMESYNCACC_GBE_1_1588PTP_PORT)) ||
         (ptpPortMode >= ICP_TIMESYNCACC_1588PTP_PORT_MODE_INVALID) ||
         (ptpPortMode < 0)) 
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) || 
          (ICP_TIMESYNCACC_1588PTP_PORT_MODE_INVALID <= ptpPortMode)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Set the Mode of the PTP Port */
    switch (ptpPortMode)
    {
        case ICP_TIMESYNCACC_1588PTP_PORT_MASTER:
        {
             ixTimeSyncAccControlPTPPortMasterModeSet(
                 ixPortToChnlMap[ptpPort], TRUE);
             ixTimeSyncAccControlPTPPortPTPMsgTimestampSet(
                 ixPortToChnlMap[ptpPort], FALSE);
             break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_MASTER */
        case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE:
        {
             ixTimeSyncAccControlPTPPortMasterModeSet(
                 ixPortToChnlMap[ptpPort], FALSE);
             ixTimeSyncAccControlPTPPortPTPMsgTimestampSet(
                 ixPortToChnlMap[ptpPort], FALSE);
             break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE */
        case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE:
        {
             ixTimeSyncAccControlPTPPortMasterModeSet(
                 ixPortToChnlMap[ptpPort], FALSE);
             ixTimeSyncAccControlPTPPortPTPMsgTimestampSet(
                 ixPortToChnlMap[ptpPort], TRUE);
             break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE */
        default:
        {
            /* This part of the code should not be reached */
            printf("ixTimeSyncAccPTPPortConfigSet(): "
                "Invalid Port Mode\n");

            return ICP_TIMESYNCACC_FAILED;
        } /* end of case default */
    } /* end of switch (ptpPortMode) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigSet() function */

/* Retrieve IEEE 1588 Hardware Assist PTP operation mode on 
    particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPPortConfigGet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPPortMode *ptpPortMode)
{
    /* Verify the parameters for proper values */

    if (((ptpPort != ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) &&
         (ptpPort != ICP_TIMESYNCACC_GBE_1_1588PTP_PORT)) ||
        ((IxTimeSyncAcc1588PTPPortMode *)NULL == ptpPortMode))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) ||
                    ((IxTimeSyncAcc1588PTPPortMode *)NULL == ptpPortMode)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Get the Mode of the PTP Port */
    *ptpPortMode = ixTimeSyncAccPTPPortModeGet(ptpPort);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigGet() function */

/* 
 * Poll the IEEE 1588 Hardware Assist message/time stamp detect 
 * status on a particular 
 * PTP Port on the Receive side
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPRxPoll(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccPtpMsgData  *ptpMsgData)
{
    /* Local variables */
    UINT32 rxsFlag = FALSE;
    IxTimeSyncAcc1588PTPPortMode ptpPortMode = 
                                 ICP_TIMESYNCACC_1588PTP_PORT_SLAVE;

    /* Verify the parameters for proper values */
    if (((ptpPort != ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) &&
         (ptpPort != ICP_TIMESYNCACC_GBE_1_1588PTP_PORT)) ||
        ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) ||
                    ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Get the Mode of the PTP Port */
    ptpPortMode = ixTimeSyncAccPTPPortModeGet(ptpPort);
    
    /* If the Port Mode is ANY mode OR the receive timestamp available */
    rxsFlag = ixTimeSyncAccControlPTPPortRxsFlagGet(ixPortToChnlMap[ptpPort]);

    /* Neither the port is configured for 'Any Mode' nor there is a timestamp */
    if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE != ptpPortMode) &&
        (TRUE != rxsFlag))
    {
        return ICP_TIMESYNCACC_NOTIMESTAMP;
    } /* end of if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE != ptpPortMode) &&
                    (TRUE != rxsFlag)) */

    /* Fetch the receive timestamp */
    ixTimeSyncAccPTPPortReceiveSnapshotGet(ixPortToChnlMap[ptpPort], 
        &ptpMsgData->ptpTimeStamp.timeValueLowWord,
        &ptpMsgData->ptpTimeStamp.timeValueHighWord);

    /* Fetch the UUID & Seq# of PTP messages in 'Master/Slave Mode' only */
    if (TRUE == rxsFlag)
    {
        ixTimeSyncAccPTPMsgUuidSeqIdGet(ixPortToChnlMap[ptpPort],
            &ptpMsgData->ptpUuid.uuidValueLowWord,
            &ptpMsgData->ptpUuid.uuidValueHighHalfword,
            &ptpMsgData->ptpSequenceNumber);
    }
    /* Clear-off the UUID & Seq# of all the messages in 'Any Mode' */
    else
    {
        ptpMsgData->ptpUuid.uuidValueLowWord = 0;
        ptpMsgData->ptpUuid.uuidValueHighHalfword = 0;
        ptpMsgData->ptpSequenceNumber = 0;
    } /* end of if (TRUE == rxsFlag) */

    /* Fill-in the PTP message type */
    switch (ptpPortMode)
    {
        case ICP_TIMESYNCACC_1588PTP_PORT_MASTER:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_DELAYREQ;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_MASTER */
        case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_SYNC;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE */
        case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_UNKNOWN;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE */
        default:
        {
            printf("ixTimeSyncAccPTPRxPoll(): Invalid Port Mode\n");

            return ICP_TIMESYNCACC_FAILED;
        } /* end of case default */
    } /* end of switch (ptpPortMode) */

    /* Increment receive timestamp counter */
    ixTsStats.rxMsgs++;

    /* Allow next timestamp to be captured */
    ixTimeSyncAccControlPTPPortRxsFlagClear(ixPortToChnlMap[ptpPort]);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPRxPoll() function */

/*
 * Poll for the IEEE 1588 Hardware Assist message/time stamp detect 
 * status on a particular 
 * PTP Port on the Transmit side.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPTxPoll(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccPtpMsgData  *ptpMsgData)
{
    /* Local variables */
    UINT32 txsFlag = FALSE;
    IxTimeSyncAcc1588PTPPortMode ptpPortMode = 
                                 ICP_TIMESYNCACC_1588PTP_PORT_SLAVE;

    /* Verify the parameters for proper values */
    if (((ptpPort != ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) &&
         (ptpPort != ICP_TIMESYNCACC_GBE_1_1588PTP_PORT)) ||
        ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) ||
                    ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Get the Mode of the PTP Port */
    ptpPortMode = ixTimeSyncAccPTPPortModeGet(ptpPort);

    /* If the Port Mode is ANY mode OR the transmit timestamp available? */
    txsFlag = ixTimeSyncAccControlPTPPortTxsFlagGet(ixPortToChnlMap[ptpPort]);

    if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE == ptpPortMode) ||
       (TRUE == txsFlag))
    {
        /* Fetch the transmit timestamp */
        ixTimeSyncAccPTPPortTransmitSnapshotGet(ixPortToChnlMap[ptpPort], 
            &ptpMsgData->ptpTimeStamp.timeValueLowWord,
            &ptpMsgData->ptpTimeStamp.timeValueHighWord);

        /*
         * Fill the UUID and Seq# with invalid values (zeros) 
         * since they are not relevant for transmit timestamp 
         */
        ptpMsgData->ptpUuid.uuidValueLowWord = 0;
        ptpMsgData->ptpUuid.uuidValueHighHalfword = 0;
        ptpMsgData->ptpSequenceNumber = 0;
    }
    /* else of if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE == ptpPortMode) ||
     *             (TRUE == txsFlag)) 
     */
    else
    {
        return ICP_TIMESYNCACC_NOTIMESTAMP;
    } /* end of if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE == ptpPortMode) ||
                    (TRUE == txsFlag)) */

    /* Fill-in the PTP message type */
    switch (ptpPortMode)
    {
        case ICP_TIMESYNCACC_1588PTP_PORT_MASTER:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_SYNC;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_MASTER */
        case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_DELAYREQ;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_SLAVE */
        case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE:
        {
            ptpMsgData->ptpMsgType = ICP_TIMESYNCACC_1588PTP_MSGTYPE_UNKNOWN;
            break;
        } /* end of case ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE */
        default:
        {
            printf("ixTimeSyncAccPTPTxPoll(): Invalid Port Mode\n");

            return ICP_TIMESYNCACC_FAILED;
        } /* end of case default */
    } /* end of switch (ptpPortMode) */

    /* Increment transmit timestamp counter */
    ixTsStats.txMsgs++;

    /* Allow next timestamp to be captured */
    ixTimeSyncAccControlPTPPortTxsFlagClear(ixPortToChnlMap[ptpPort]);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPTxPoll() function */

/* Set the System Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccSystemTimeSet(IxTimeSyncAccTimeValue systemTime)
{
    /* Local variables */
    UINT32 oldFsv = 0;

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Retrieve old Frequency Scaling Value */
    ixTimeSyncAccAddendFsvGet(&oldFsv);

    /*
     * Set the Frequency Scaling Value to zero (0) so that
     * System Time doesn't get incremented while it is being
     * written into low and high registers 
     */
    ixTimeSyncAccAddendFsvSet(0);

    /* Update System Time with user specified values */
    ixTimeSyncAccSystemTimeSnapshotSet(systemTime.timeValueLowWord,
                                       systemTime.timeValueHighWord);

    /*
     * Let the hardware assist re-evaluate the target time reached 
     * condition based on the new system time
     */
    ixTimeSyncAccEventTtmFlagClear();

    /*
     * Let the hardware assist re-evaluate the aux target time reached 
     * condition based on the new system time
     */
    ixTimeSyncAccEventAtmFlagClear();

    /*
     * Restore old Frequency Scaling Value so that System Time
     * can be incremented 
     */
    ixTimeSyncAccAddendFsvSet(oldFsv);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccSystemTimeSet() function */

/* Get the System Time from the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccSystemTimeGet(IxTimeSyncAccTimeValue *systemTime)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccTimeValue *)NULL == systemTime)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccTimeValue *)NULL == systemTime) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Fetch System Time */
    ixTimeSyncAccSystemTimeSnapshotGet(&systemTime->timeValueLowWord,
                                       &systemTime->timeValueHighWord);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccSystemTimeGet() function */

/*
 * Set the Tick Rate (Frequency Scaling Value) in the IEEE 1588
 * hardware assist block
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTickRateSet(UINT32 tickRate)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Update the Frequency Scaling Value */
    ixTimeSyncAccAddendFsvSet(tickRate);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTickRateSet() function */

/*
 * Get the Tick Rate (Frequency Scaling Value) from the IEEE 1588
 * hardware assist block
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTickRateGet(UINT32 *tickRate)
{
    /* Verify the parameter */
    if ((UINT32 *)NULL == tickRate)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((UINT32 *)NULL == tickRate) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Retrieve Current Frequency Scaling Value */
    ixTimeSyncAccAddendFsvGet(tickRate);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTickRateGet() function */

/*
 * Enable the interrupt to verify the condition where the System Time is
 * greater or equal to the Target Time in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeInterruptEnable(
    IxTimeSyncAccTargetTimeCallback targetTimeCallback)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccTargetTimeCallback)NULL == targetTimeCallback)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccTargetTimeCallback)NULL == 
                     targetTimeCallback) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Register the Callback */
    ixTsTargetTimeCallback = targetTimeCallback;

    /* Set target time interrupt mask */
    ixTimeSyncAccControlTtmInterruptMaskSet();

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTargetTimeInterruptEnable() function */

/*
 * Disable the interrupt to verify the condition that the System Time is 
 * greater or equal to the Target Time in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeInterruptDisable(void)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Clear target time interrupt mask */
    ixTimeSyncAccControlTtmInterruptMaskClear();

    /* Unregister the Callback */
    ixTsTargetTimeCallback = (IxTimeSyncAccTargetTimeCallback) NULL;

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTargetTimeInterruptDisable() function */

/*
 * Poll to verify the condition where the System Time is greater or equal 
 * to the Target Time in the IEEE 1588 hardware assist block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimePoll(
    UINT32 *ttmPollFlag,
    IxTimeSyncAccTimeValue *targetTime)
{
    /* Verify the parameters */
    if (((UINT32 *)NULL == ttmPollFlag) ||
        ((IxTimeSyncAccTimeValue *)NULL == targetTime))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if (((UINT32 *)NULL == ttmPollFlag) ||
                    ((IxTimeSyncAccTimeValue)NULL == targetTime)) */

    /* Is interrupt mode of processing is enabled? */
    if ((IxTimeSyncAccTargetTimeCallback) NULL != ixTsTargetTimeCallback)
    {
        return ICP_TIMESYNCACC_INTERRUPTMODEINUSE;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Is the System Time reached or exceeded Target Time? */
    *ttmPollFlag = ixTimeSyncAccEventTtmFlagGet();
    if (FALSE == ttmPollFlag)
    {
        /* Target Time not to be returned yet */
        targetTime->timeValueLowWord = 0;
        targetTime->timeValueHighWord = 0;

        return ICP_TIMESYNCACC_SUCCESS;
    } /* if (FALSE == ttmPollFlag) */

    /* Fetch Target Time */
    ixTimeSyncAccTargetTimeSnapshotGet(&targetTime->timeValueLowWord,
                                  &targetTime->timeValueHighWord);

    /* Clear the target time reached condition (ttipend bit) */
    ixTimeSyncAccEventTtmFlagClear();

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTargetTimePoll() function */

/* Set the Target Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeSet(IxTimeSyncAccTimeValue targetTime)
{
    /* Local variables */
    UINT32 oldTtmMask = FALSE;

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Retrieve existing target time interrupt mask value */
    oldTtmMask = ixTimeSyncAccControlTtmInterruptMaskGet();

    /* Clear the target time interrupt mask value to prevent false 
     * interrupts from being asserted due to the increments of the 
     * values in the System Time low and high registers (i.e., the
     * target time reached or exceeded interrupt does not get
     * generated
     */
    ixTimeSyncAccControlTtmInterruptMaskClear();

    /* Update Target Time with user specified values */
    ixTimeSyncAccTargetTimeSnapshotSet(targetTime.timeValueLowWord,
                                       targetTime.timeValueHighWord);

    /*
     * Let the hardware assist re-evaluate the target time reached 
     * condition based on the new target time
     */
    ixTimeSyncAccEventTtmFlagClear();

    /* Restore the preserved target time interrupt mask value */
    if (TRUE == oldTtmMask)
    {
        ixTimeSyncAccControlTtmInterruptMaskSet();
    } /* end of if (TRUE == oldTtmMask) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTargetTimeSet() function */

/* Get the Target Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeGet(IxTimeSyncAccTimeValue *targetTime)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccTimeValue *)NULL == targetTime)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccTimeValue *)NULL == systemTime) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Fetch Target Time */
    ixTimeSyncAccTargetTimeSnapshotGet(&targetTime->timeValueLowWord,
                               &targetTime->timeValueHighWord);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccTargetTimeGet() function */

/*
 * Enable the interrupt to verify the condition where the System Time is 
 * greater or equal to the Auxiliary Target Time in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeInterruptEnable(
    IxTimeSyncAccAuxTargetTimeCallback auxTargetTimeCallback)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccAuxTargetTimeCallback)NULL == auxTargetTimeCallback)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccAuxTargetTimeCallback)NULL == 
                     auxTargetTimeCallback) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Register the Callback */
    ixTsAuxTargetTimeCallback = auxTargetTimeCallback;

    /* Set auxiliary target time interrupt mask */
    ixTimeSyncAccControlAtmInterruptMaskSet();

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTargetTimeInterruptEnable() function */

/*
 * Disable the interrupt to verify the condition that the System Time is 
 * greater or equal to the Auxiliary Target Time in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeInterruptDisable(void)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Clear auxiliary target time interrupt mask */
    ixTimeSyncAccControlAtmInterruptMaskClear();

    /* Unregister the Callback */
    ixTsAuxTargetTimeCallback = (IxTimeSyncAccAuxTargetTimeCallback) NULL;

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTargetTimeInterruptDisable() function */

/*
 * Poll to verify the condition where the System Time greater or equal 
 * to the Auxiliary Target Time in the IEEE 1588 hardware assist block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimePoll(
    UINT32 *atmPollFlag,
    IxTimeSyncAccTimeValue *auxTargetTime)
{
    /* Verify the parameters */
    if (((UINT32 *)NULL == atmPollFlag) ||
        ((IxTimeSyncAccTimeValue *)NULL == auxTargetTime))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if (((UINT32 *)NULL == atmPollFlag) ||
                    ((IxTimeSyncAccTimeValue)NULL == auxTargetTime)) */

    /* Is interrupt mode of processing is enabled? */
    if ((IxTimeSyncAccAuxTargetTimeCallback) NULL != ixTsAuxTargetTimeCallback)
    {
        return ICP_TIMESYNCACC_INTERRUPTMODEINUSE;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Is the System Time reached or exceeded Auxiliary Target Time? */
    *atmPollFlag = ixTimeSyncAccEventAtmFlagGet();
    if (FALSE == atmPollFlag)
    {
        /* Auxiliary Target Time not to be returned yet */
        auxTargetTime->timeValueLowWord = 0;
        auxTargetTime->timeValueHighWord = 0;

        return ICP_TIMESYNCACC_SUCCESS;
    } /* if (FALSE == atmPollFlag) */

    /* Fetch Auxiliary Target Time */
    ixTimeSyncAccAuxTargetTimeSnapshotGet(&auxTargetTime->timeValueLowWord,
                                  &auxTargetTime->timeValueHighWord);

    /* Clear the auxiliary target time reached condition (atp bit) */
    ixTimeSyncAccEventAtmFlagClear();

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTargetTimePoll() function */

/* Set the Auxiliary Target Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeSet(IxTimeSyncAccTimeValue auxTargetTime)
{
    /* Local variables */
    UINT32 oldAtmMask = FALSE;

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Retrieve existing auxiliary target time interrupt mask value */
    oldAtmMask = ixTimeSyncAccControlAtmInterruptMaskGet();

    /* Clear the auxiliary target time interrupt mask value to prevent false 
     * interrupts from being asserted due to the increments of the 
     * values in the System Time low and high registers (i.e., the
     * auxiliary target time reached or exceeded interrupt does not get
     * generated
     */
    ixTimeSyncAccControlAtmInterruptMaskClear();

    /* Update Auxiliary Target Time with user specified values */
    ixTimeSyncAccAuxTargetTimeSnapshotSet(auxTargetTime.timeValueLowWord,
                                       auxTargetTime.timeValueHighWord);

    /*
     * Let the hardware assist to re-evaluate the auxiliary target time reached 
     * condition based on the new auxiliary target time
     */
    ixTimeSyncAccEventAtmFlagClear();

    /* Restore the preserved auxiliary target time interrupt mask value */
    if (TRUE == oldAtmMask)
    {
        ixTimeSyncAccControlAtmInterruptMaskSet();
    } /* end of if (TRUE == oldAtmMask) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTargetTimeSet() function */

/* Get the Auxiliary Target Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeGet(IxTimeSyncAccTimeValue *auxTargetTime)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccTimeValue *)NULL == auxTargetTime)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccTimeValue *)NULL == systemTime) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Fetch Auxiliary Target Time */
    ixTimeSyncAccAuxTargetTimeSnapshotGet(&auxTargetTime->timeValueLowWord,
                               &auxTargetTime->timeValueHighWord);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTargetTimeGet() function */

/*
 * Enable the interrupt notification for the given mode of Auxiliary Time 
 * Stamp in the IEEE 1588 hardware assist block
 */ 
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimeInterruptEnable(
    IxTimeSyncAccAuxMode auxMode,
    IxTimeSyncAccAuxTimeCallback auxTimeCallback)
{
    /* Verify the parameters */
    if ((ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode) ||
        ((IxTimeSyncAccAuxTimeCallback)NULL == auxTimeCallback))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* if ((ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode) ||
             ((IxTimeSyncAccAuxTimeCallback)NULL == auxTimeCallback)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Register the Callback and SET the amm/asm bits on */
    if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode)
    {
        ixTsAuxMasterTimeCallback = auxTimeCallback;
        ixTimeSyncAccControlAmmsInterruptMaskSet();

    } /* else of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */
    else
    {
        ixTsAuxSlaveTimeCallback = auxTimeCallback;
        ixTimeSyncAccControlAsmsInterruptMaskSet();
    } /* end of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTimeInterruptEnable() function */

/*
 * Disable the interrupt for the indicated mode of Auxiliary Time Stamp
 * in the IEEE 1588 hardware assist block
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimeInterruptDisable(IxTimeSyncAccAuxMode auxMode)
{
    /* Verify the parameters */
    if (ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* if (ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Unregister the Callback and CLEAR the amm/asm bits on */
    if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode)
    {
        ixTimeSyncAccControlAmmsInterruptMaskClear();
        ixTsAuxMasterTimeCallback = (IxTimeSyncAccAuxTimeCallback) NULL;
    }
    /* else of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */
    else
    {
        ixTimeSyncAccControlAsmsInterruptMaskClear();
        ixTsAuxSlaveTimeCallback = (IxTimeSyncAccAuxTimeCallback) NULL;
    } /* end of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTimeInterruptDisable() function */

/*
 * Poll for the Auxiliary Time Stamp captured for the mode indicated 
 * (Master or Slave)
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimePoll(IxTimeSyncAccAuxMode auxMode,
    UINT32 *auxPollFlag,
    IxTimeSyncAccTimeValue *auxTime)
{
    /* Local variables */
    UINT32 ammsFlag = FALSE;
    UINT32 asmsFlag = FALSE;

    /* Verify the parameters */
    if (((UINT32 *)NULL == auxPollFlag) ||
        (ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode) ||
        ((IxTimeSyncAccTimeValue *)NULL == auxTime))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if (((UINT32 *)NULL == auxPollFlag) ||
                    (ICP_TIMESYNCACC_AUXMODE_INVALID <= auxMode) ||
                    ((IxTimeSyncAccTimeValue *)NULL == auxTime)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Fetch Auxiliary Master/Slave Mode Snapshot */
    if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode)
    {
        /* Is interrupt mode of processing is enabled? */
        if ((IxTimeSyncAccAuxTimeCallback) NULL != ixTsAuxMasterTimeCallback)
        {
            return ICP_TIMESYNCACC_INTERRUPTMODEINUSE;
        } /* end of if ((IxTimeSyncAccAuxTimeCallback) NULL != 
                        ixTsAuxMasterTimeCallback) */

        /* Is the Auxiliary Master Mode Snapshot available? */
        ammsFlag = ixTimeSyncAccEventAmmsFlagGet();
        if (FALSE == ammsFlag)
        {
            *auxPollFlag = FALSE;
            auxTime->timeValueLowWord =0;
            auxTime->timeValueHighWord = 0;
            return ICP_TIMESYNCACC_SUCCESS;
        } /* end of if (FALSE == ammsFlag) */

        /* Fetch Auxiliary Master Snapshot */
        ixTimeSyncAccAuxMasterModeSnapshotGet(&auxTime->timeValueLowWord,
                                              &auxTime->timeValueHighWord);
        *auxPollFlag = TRUE;

        /* Clear the snapshot availability condition */
        ixTimeSyncAccEventAmmsFlagClear();
    }
    /* else of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */
    else
    {
        /* Is interrupt mode of processing is enabled? */
        if ((IxTimeSyncAccAuxTimeCallback) NULL != ixTsAuxSlaveTimeCallback)
        {
            return ICP_TIMESYNCACC_INTERRUPTMODEINUSE;
        } /* end of if ((IxTimeSyncAccAuxTimeCallback) NULL != 
                        ixTsAuxSlaveTimeCallback) */

        /* Is the Auxiliary Slave Mode Snapshot available? */
        asmsFlag = ixTimeSyncAccEventAsmsFlagGet();
        if (FALSE == asmsFlag)
        {
            *auxPollFlag = FALSE;
            auxTime->timeValueLowWord =0;
            auxTime->timeValueHighWord = 0;
            return ICP_TIMESYNCACC_SUCCESS;
        } /* end of if (FALSE == asmsFlag) */

        /* Fetch Auxiliary Slave Snapshot */
        ixTimeSyncAccAuxSlaveModeSnapshotGet(&auxTime->timeValueLowWord,
                                             &auxTime->timeValueHighWord);
        *auxPollFlag = TRUE;

        /* Clear the snapshot availability condition */
        ixTimeSyncAccEventAsmsFlagClear();
    } /* end of if (ICP_TIMESYNCACC_AUXMODE_MASTER == auxMode) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccAuxTimePoll() function */

/* 
 * Poll the IEEE 1588 Hardware Assist message/time stamp detect 
 * status on a particular 
 * PTP CAN Port
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPCANPoll(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccTimeValue *ptpTimeStamp)
{
    /* Local variables */
    UINT32 valFlag = FALSE;
    UINT32 ovrFlag = FALSE;

    /* Verify the parameters for proper values */
    if (((ptpPort != ICP_TIMESYNCACC_CAN_0_1588PTP_PORT) &&
         (ptpPort != ICP_TIMESYNCACC_CAN_1_1588PTP_PORT)) ||
        ((IxTimeSyncAccTimeValue *)NULL == ptpTimeStamp))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) ||
                    ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData)) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Is a new timestamp available? */
    valFlag = ixTimeSyncAccControlPTPPortValFlagGet(ixPortToChnlMap[ptpPort]);

    /* there is not a valid timestamp */
    if ( TRUE != valFlag )
    {
        return ICP_TIMESYNCACC_NOTIMESTAMP;
    } /* end of if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE != ptpPortMode) &&
                    (TRUE != rxsFlag)) */

    /* check overrun bit before retreiving timestamp */
    ovrFlag = ixTimeSyncAccControlPTPPortOvrFlagGet(ixPortToChnlMap[ptpPort]);

    /* if the timestamp has been overwritten */
    if ( TRUE == ovrFlag )
    {
        /* reset valid and overrun bits */
        ixTimeSyncAccControlPTPPortValFlagClear(ixPortToChnlMap[ptpPort]);
        ixTimeSyncAccControlPTPPortOvrFlagClear(ixPortToChnlMap[ptpPort]);

        /* return no valid timestamp available */
        ptpTimeStamp->timeValueLowWord = 0;
        ptpTimeStamp->timeValueHighWord = 0;

        return ICP_TIMESYNCACC_NOTIMESTAMP;
    } /* end of if ((ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE != ptpPortMode) &&
                    (TRUE != rxsFlag)) */

    /* Fetch the receive timestamp */
    ixTimeSyncAccPTPPortCANSnapshotGet(ixPortToChnlMap[ptpPort], 
        &ptpTimeStamp->timeValueLowWord,
        &ptpTimeStamp->timeValueHighWord);

    /* check overrun bit again to ensure timestamp is valid */
    ovrFlag = ixTimeSyncAccControlPTPPortOvrFlagGet(ixPortToChnlMap[ptpPort]);

    /* if the timestamp has been overwritten */
    if ( TRUE == ovrFlag )
    {
        /* reset valid and overrun bits */
        ixTimeSyncAccControlPTPPortValFlagClear(ixPortToChnlMap[ptpPort]);
        ixTimeSyncAccControlPTPPortOvrFlagClear(ixPortToChnlMap[ptpPort]);

        /* return no valid timestamp available */
        ptpTimeStamp->timeValueLowWord = 0;
        ptpTimeStamp->timeValueHighWord = 0;

        return ICP_TIMESYNCACC_NOTIMESTAMP;
    } /* end of if ( TRUE == ovrFlag ) */

    /* reset valid bit */
    ixTimeSyncAccControlPTPPortValFlagClear(ixPortToChnlMap[ptpPort]);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPCANPoll() function */

/*
 * Enable the interrupt to verify the condition where the System Time is
 * greater or equal to the Pulse Per Second in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondInterruptEnable(
    IxTimeSyncAccPulsePerSecondCallback pulsePerSecondCallback)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccPulsePerSecondCallback)NULL == pulsePerSecondCallback)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccPulsePerSecondCallback)NULL == 
                     pulsePerSecondCallback) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Register the Callback */
    ixTsPulsePerSecondCallback = pulsePerSecondCallback;

    /* Set pulse per second interrupt mask */
    ixTimeSyncAccControlPpsmInterruptMaskSet();

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPulsePerSecondInterruptEnable() function */

/*
 * Disable the interrupt to verify the condition that the System Time is
 * greater or equal to the Pulse Per Second  in the IEEE 1588 hardware assist 
 * block.
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondInterruptDisable(void)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Clear pulse per second interrupt mask */
    ixTimeSyncAccControlPpsmInterruptMaskClear();

    /* Unregister the Callback */
    ixTsPulsePerSecondCallback = (IxTimeSyncAccPulsePerSecondCallback) NULL;

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPulsePerSecondInterruptDisable() function */

/* Set the pulse per second Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondTimeSet(UINT32 ppsTime)
{
    /* Local variables */
    UINT32 oldPpsMask = FALSE;

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Retrieve existing pulse per second time interrupt mask value */
    oldPpsMask = ixTimeSyncAccControlPpsmInterruptMaskGet();

    /* Clear the pulse per second time interrupt mask value to prevent false 
     * interrupts from being asserted due to the increments of the 
     * values in the System Time low and high registers (i.e., the
     * pulse per second time reached or exceeded interrupt does not get
     * generated
     */
    ixTimeSyncAccControlPpsmInterruptMaskClear();

    /* Update pulse per second Time with user specified values */
    ixTimeSyncAccPulsePerSecondCompareSet(ppsTime);

    /*
     * Let the hardware assist re-evaluate the pulse per second time reached 
     * condition based on the new pulse per second time
     */
    ixTimeSyncAccEventPpsmFlagClear();

    /* Restore the preserved pulse per second time interrupt mask value */
    if (TRUE == oldPpsMask)
    {
        ixTimeSyncAccControlPpsmInterruptMaskSet();
    } /* end of if (TRUE == oldPpsMask) */

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPulsePerSecondTimeSet() function */

/* Get the pulse per second Time in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondTimeGet(UINT32 *ppsTime)
{
    /* Verify the parameter */
    if ((UINT32 *)NULL == ppsTime)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((IxTimeSyncAccTimeValue *)NULL == ppsTime) */

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Fetch pulse per second Time */
    ixTimeSyncAccPulsePerSecondCompareGet(ppsTime);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPulsePerSecondTimeGet() function */


/* Configure IEEE 1588 Hardware Assist version
 * on a particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPVersionSet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPVersion ptpVersion)
{
    /* Verify the parameters for proper values */
    if ( (ptpPort >= ICP_TIMESYNCACC_1588PORT_INVALID ) ||
         (ptpPort < 0) ||
         (ptpVersion > ICP_TIMESYNCACC_1588PTP_VERSION_2_OR_1) ||
         (ptpVersion < 0)) 
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    ixTimeSyncAccControlPTPVersionSet( ptpPort, ptpVersion );
    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigSet() function */

/* Retrieve IEEE 1588 Hardware Assist version on 
    particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPVersionGet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPVersion *ptpVersion)
{
    /* Verify the parameters for proper values */
    if ( (ptpPort > ICP_TIMESYNCACC_1588PORT_INVALID ) ||
         (ptpPort < 0) 
         || (ptpVersion == NULL ) 
         ) 
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    *ptpVersion = ixTimeSyncAccControlPTPVersionGet( ptpPort );

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigGet() function */

/* Configure IEEE 1588 Hardware Assist operation mode 
 * on a particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPOperationModeSet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPOperationMode ptpOpMode)
{
    /* Verify the parameters for proper values */
    if ( (ptpPort >= ICP_TIMESYNCACC_1588PORT_INVALID ) ||
         (ptpPort < 0) ||
         ((ptpOpMode != ICP_TIMESYNCACC_1588PTP_OP_MODE_SYNC_DELAYREQ_MSGS ) && 
          (ptpOpMode != ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_ALL_MSGS ) && 
          (ptpOpMode != ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_V2_EVENT_MSGS ) && 
          (ptpOpMode != ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_V2_ALL_MSGS ) && 
          (ptpOpMode != ICP_TIMESYNCACC_1588PTP_OP_MODE_USER_MSGS )) )
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    ixTimeSyncAccControlPTPOperationModeSet( ptpPort, ptpOpMode );

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigSet() function */

/* Retrieve IEEE 1588 Hardware Assist PTP operation mode on 
    particular PTP port */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPOperationModeGet(
    IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPOperationMode *ptpOpMode)
{
    /* Verify the parameters for proper values */
    if ( (ptpPort >= ICP_TIMESYNCACC_1588PORT_INVALID ) ||
         (ptpPort < 0) 
         || (ptpOpMode == NULL) 
         ) 
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } 

    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    *ptpOpMode = ixTimeSyncAccControlPTPOperationModeGet( ptpPort );

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccPTPPortConfigGet() function */



/* Reset the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccReset(void)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Reset Hardware Assist */
    ixTimeSyncAccControlReset();

    /* Clear Stats */
    ixTsStats.rxMsgs = ixTsStats.txMsgs = 0;

    /* Unregister any Callback Routines */
    ixTsTargetTimeCallback    = (IxTimeSyncAccTargetTimeCallback) NULL;
    ixTsAuxMasterTimeCallback = (IxTimeSyncAccAuxTimeCallback) NULL;
    ixTsAuxSlaveTimeCallback  = (IxTimeSyncAccAuxTimeCallback) NULL;
    ixTsAuxTargetTimeCallback  = (IxTimeSyncAccAuxTargetTimeCallback) NULL;
    ixTsPulsePerSecondCallback = (IxTimeSyncAccPulsePerSecondCallback) NULL;

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccReset() function */

/* Reset a channel in the IEEE 1588 hardware assist block */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccChnlReset(IxTimeSyncAcc1588PTPPort ptpPort)
{
    /* Initialised before */
    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /* Verify the parameters for proper values */
    if ((ptpPort != ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) &&
        (ptpPort != ICP_TIMESYNCACC_GBE_1_1588PTP_PORT))
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of if ((ICP_TIMESYNCACC_1588PORT_INVALID <= ptpPort) ||
                    ((IxTimeSyncAccPtpMsgData *)NULL == ptpMsgData)) */

    /* Reset Hardware Assist */
    ixTimeSyncAccChannelReset(ixPortToChnlMap[ptpPort]);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccChnlReset() function */

/* Return the IxTimeSyncAcc Statistics */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccStatsGet(IxTimeSyncAccStats *timeSyncStats)
{
    /* Verify the parameter */
    if ((IxTimeSyncAccStats *) NULL == timeSyncStats)
    {
        return ICP_TIMESYNCACC_INVALIDPARAM;
    } /* end of ((IxTimeSyncAccStats *) NULL == timeSyncStats) */

    /* Return Stats */
    timeSyncStats->rxMsgs = ixTsStats.rxMsgs; 
    timeSyncStats->txMsgs = ixTsStats.txMsgs; 

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of IxTimeSyncAccStatsGet() function */

/* Reset Time Sync statistics */
void
ixTimeSyncAccStatsReset(void)
{
    /* Clear Stats */
    ixTsStats.rxMsgs = ixTsStats.txMsgs = 0;

    return;
} /* end of ixTimeSyncAccStatsReset() function */


/* Display the Time Sync current status */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccShow(void)
{
    /* Local Varaiables */
    UINT32 regValue = 0;
    UINT32 regLowValue = 0;
    UINT32 regHighValue = 0;
    UINT16 seqId = 0;
    UINT32 uuIdLow = 0;
    UINT16 uuIdHigh = 0;
    UINT32   bitSet   = FALSE;
    UINT32 ptpPortNum = 0;

    if ( !ixInitialized )
    {
        return ICP_TIMESYNCACC_UNINITIALIZED;
    }

    /*
    Dump Block Level Status 

    System Time registers contents 
    */
    regLowValue = 0;
    regHighValue = 0;
    ixTimeSyncAccSystemTimeSnapshotGet(&regLowValue, &regHighValue);
    printf("System Test (Low:High): %x : %x\n", regLowValue, regHighValue);

    /* Frequency Scaling Value */
    regValue = 0;
    ixTimeSyncAccAddendFsvGet(&regValue);
    printf("Frequency Scaling Value: %x\n", regValue);

    /* Target time reached/exceeded interrupt mask value */
    bitSet = ixTimeSyncAccControlTtmInterruptMaskGet();
    printf("Target Time Reached/Exceeded Interrupt Mask: %s\n",
        ((TRUE == bitSet) ? "ENABLED":"DISABLED"));

    /* Target time reached/exceeded event flag value */
    bitSet = ixTimeSyncAccEventTtmFlagGet();
    printf("Target Time Reached/Exceeded Event Flag: %s\n",
        ((TRUE == bitSet) ? "SET":"CLEAR"));

    /* Target Time registers contents  */
    regLowValue = 0;
    regHighValue = 0;
    ixTimeSyncAccTargetTimeSnapshotGet(&regLowValue, &regHighValue);
    printf("Target Time (Low:High): %x : %x\n",
        regLowValue, regHighValue);

    /* Auxiliary Master Mode Snapshot interrupt mask value */
    bitSet = ixTimeSyncAccControlAmmsInterruptMaskGet();
    printf("Auxiliary Master Mode Snapshot Interrupt Mask: %s\n",
        ((TRUE == bitSet) ? "ENABLED":"DISABLED"));

    /* Auxiliary Master Mode Snapshot event flag value */
    bitSet = ixTimeSyncAccEventAmmsFlagGet();
    printf("Auxiliary Master Mode Snapshot Event Flag: %s\n",
        ((TRUE == bitSet) ? "SET":"CLEAR"));

    /* Auxiliary Master Snapshot registers  */
    regLowValue = 0;
    regHighValue = 0;
    ixTimeSyncAccAuxMasterModeSnapshotGet(&regLowValue, &regHighValue);
    printf("Auxiliary Master Mode Snapshot (Low:High): %x : %x\n",
        regLowValue, regHighValue);

    /* Auxiliary Slave Mode Snapshot interrupt mask value  */
    bitSet = ixTimeSyncAccControlAsmsInterruptMaskGet();
    printf("Auxiliary Slave Mode Snapshot Interrupt Mask: %s\n",
        ((TRUE == bitSet) ? "ENABLED":"DISABLED"));

    /* Auxiliary Slave Mode Snapshot event flag value  */
    bitSet = ixTimeSyncAccEventAsmsFlagGet();
    printf("Auxiliary Slave Mode Snapshot Event Flag: %s\n",
        ((TRUE == bitSet) ? "SET":"CLEAR"));
    
    /* Auxiliary Slave Snapshot registers  */
    regLowValue = 0;
    regHighValue = 0;
    ixTimeSyncAccAuxSlaveModeSnapshotGet(&regLowValue, &regHighValue);
    printf("Auxiliary Slave Mode Snapshot (Low:High) : %x : %x\n",
        regLowValue, regHighValue);
        
    /* Auxiliary Target time reached/exceeded interrupt mask value  */
    bitSet = ixTimeSyncAccControlAtmInterruptMaskGet();
    printf("Auxiliary Target Time Reached/Exceeded Interrupt Mask: %s\n",
        ((TRUE == bitSet) ? "ENABLED":"DISABLED"));

    /* Auxiliary Target time reached/exceeded event flag value  */
    bitSet = ixTimeSyncAccEventAtmFlagGet();
    printf("Auxiliary Target Time Reached/Exceeded Event Flag: %s\n",
        ((TRUE == bitSet) ? "SET":"CLEAR"));

    /* Auxiliary Target Time registers contents   */
    regLowValue = 0;
    regHighValue = 0;
    ixTimeSyncAccAuxTargetTimeSnapshotGet(&regLowValue, &regHighValue);
    printf("Auxiliary Target Time (Low:High): %x : %x\n",
        regLowValue, regHighValue);

    /* Pulse Per Second reached/exceeded interrupt mask value  */
    bitSet = ixTimeSyncAccControlPpsmInterruptMaskGet();
    printf("Pulse Per Second Reached/Exceeded Interrupt Mask: %s\n",
        ((TRUE == bitSet) ? "ENABLED":"DISABLED"));

    /* Pulse Per Second reached/exceeded event flag value  */
    bitSet = ixTimeSyncAccEventPpsmFlagGet();
    printf("Pulse Per Second Reached/Exceeded Event Flag: %s\n",
        ((TRUE == bitSet) ? "SET":"CLEAR"));

    /* Pulse Per Second registers contents   */
    regLowValue = 0;
    ixTimeSyncAccPulsePerSecondCompareGet(&regLowValue);
    printf("Pulse Per Second: %x\n",regLowValue);
        
    /* Dump Port Level Status for ethernet ports  */
    for (ptpPortNum = 0;
         ptpPortNum < ICP_TIMESYNCACC_MAX_1588ETH_CHNL;
         ptpPortNum++)
    {
        /* output status for ethernet ports  */
        if ( (ptpPortNum == ICP_TIMESYNCACC_GBE_0_1588PTP_PORT) ||
             (ptpPortNum == ICP_TIMESYNCACC_GBE_1_1588PTP_PORT) )
        {
            /* Display the port number  */
            printf("PTP Eth Port #: %u\n",
                ptpPortNum == ICP_TIMESYNCACC_GBE_0_1588PTP_PORT ? 0 : 1);

            /* Get the Master Mode and Timestamp All PTP messages status  */
            bitSet = ixTimeSyncAccControlPTPPortMasterModeGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tMaster Mode: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            bitSet = ixTimeSyncAccControlPTPPortPTPMsgTimestampGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tTimestamp All Mode: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            /* Receive Timestamp Event Flag  */
            bitSet = ixTimeSyncAccControlPTPPortRxsFlagGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tReceive Timestamp Event Flag: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            /* Receive timestamp registers  */
            regLowValue = 0;
            regHighValue = 0;
            ixTimeSyncAccPTPPortReceiveSnapshotGet(ixPortToChnlMap[ptpPortNum],
                &regLowValue, &regHighValue);
            printf("\tReceive Timestamp (Low:High): %x : %x\n",
                regLowValue,regHighValue);

            /* UUID and Seq#  */
            ixTimeSyncAccPTPMsgUuidSeqIdGet(ixPortToChnlMap[ptpPortNum], 
                &uuIdLow, &uuIdHigh, &seqId);
            printf("\tUUID (Low: High (16-Bits)): %x : %x \n"
                "Seq# (16Bits Only): %x\n",
                uuIdLow, uuIdHigh, seqId);

            /* Transmit Timestamp Event Flag  */
            bitSet = ixTimeSyncAccControlPTPPortTxsFlagGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tTransmit Timestamp Event Flag: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            /* Transmit timestamp registers  */
            regLowValue = 0;
            regHighValue = 0;
            ixTimeSyncAccPTPPortTransmitSnapshotGet(
                ixPortToChnlMap[ptpPortNum], &regLowValue,&regHighValue);
            printf("\tTransmit Timestamp (Low:High): %x : %x\n",
                regLowValue,regHighValue);
        }
    } /* end of for (ptpPortNum = 0;
                     ptpPortNum < ICP_TIMESYNCACC_MAX_1588PTP_PORT;
                     ptpPortNum++) */

    /* Dump Port Level Status for CAN ports */
    for (ptpPortNum = 0;
         ptpPortNum < ICP_TIMESYNCACC_MAX_1588PTP_PORT;
         ptpPortNum++)
    {
        /* output status for ethernet ports  */
        if ( (ptpPortNum == ICP_TIMESYNCACC_CAN_0_1588PTP_PORT) ||
             (ptpPortNum == ICP_TIMESYNCACC_CAN_1_1588PTP_PORT) )
        {
            /* Display the port number  */
            printf("PTP CAN Port #: %u\n",
                ptpPortNum == ICP_TIMESYNCACC_CAN_0_1588PTP_PORT ? 0 : 1);

            /* Valid Timestamp Event Flag  */
            bitSet = ixTimeSyncAccControlPTPPortValFlagGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tValid Timestamp Event Flag: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            /* Overrun Event Flag  */
            bitSet = ixTimeSyncAccControlPTPPortOvrFlagGet(
                ixPortToChnlMap[ptpPortNum]);
            printf("\tOverrun Event Flag: %s\n",
                ((TRUE == bitSet) ? "SET":"CLEAR"));

            /* CAN timestamp registers  */
            regLowValue = 0;
            regHighValue = 0;
            ixTimeSyncAccPTPPortCANSnapshotGet(ixPortToChnlMap[ptpPortNum],
                &regLowValue, &regHighValue);
            printf("\tCAN Timestamp (Low:High): %x : %x\n",
                regLowValue,regHighValue);
        }
    } /* end of for (ptpPortNum = 0;
                     ptpPortNum < ICP_TIMESYNCACC_MAX_1588PTP_PORT;
                     ptpPortNum++) */

    /* Stats  */
    printf("Receive Timestamps Count: %u\n"
        "Transmit Timestamp Count: %u\n",
        ixTsStats.rxMsgs, ixTsStats.txMsgs);

    /* Callback Routine Addresses */
    printf("Target Time Callback: %p\n"
        "Auxiliary Master Mode Snapshot Callback: %p\n"
        "Auxiliary Slave Mode Snapshot Callback: %p\n"
          "Auxiliary Target Time Callback: %p\n"
           "Pulse Per Second Callback: %p\n",
        ixTsTargetTimeCallback,
        ixTsAuxMasterTimeCallback,
        ixTsAuxSlaveTimeCallback,
        ixTsAuxTargetTimeCallback,
        ixTsPulsePerSecondCallback);

    return ICP_TIMESYNCACC_SUCCESS;
} /* end of ixTimeSyncAccShow() function */

