/*****************************************************************************
 * @ingroup 1588
 *
 * @file ixtimesyncacc.h
 *
 * @description
 *        The 1588 driver module enables a client application to have
 *         access to the 1588 time synchronization hardware assist block. 
 *         A client application can use this driver to retreive tx and rx timestamps 
 *         of IEEE 1588(TM) PTP messages captured on ethernet or CAN ports. 
 *         Access is provided by a set of APIs that can be called directly or through 
 *         ioctl commands.  
 *
 * INTEL CONFIDENTIAL
 * Copyright 2007,2008,2009 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to the
 * source code ("Material") are owned by Intel Corporation or its suppliers or
 * licensors. Title to the Material remains with Intel Corporation or its
 * suppliers and licensors. The Material may contain trade secrets and proprietary
 * and confidential information of Intel Corporation and its suppliers and
 * licensors, and is protected by worldwide copyright and trade secret laws and
 * treaty provisions. No part of the Material may be used, copied, reproduced,
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Intels prior express written permission.
 * No license under any patent, copyright, trade secret or other intellectual
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise. Any license under such intellectual property rights must be
 * express and approved by Intel in writing.
 * 
 * Include any supplier copyright notices as supplier requires Intel to use.
 * Include supplier trademarks or logos as supplier requires Intel to use,
 * preceded by an asterisk.
 * An asterisked footnote can be added as follows: 
 *   *Third Party trademarks are the property of their respective owners.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Intel or Intels suppliers
 * or licensors in any way.
 * 
 *  version: Embedded.X.1.0.3-127
 ****************************************************************************/


#ifndef IXTIMESYNCACC_H
#define IXTIMESYNCACC_H

#ifdef __linux 
	#include <linux/types.h>

    typedef unsigned int UINT32;
    typedef unsigned int UINT16;
#endif

#ifdef _WIN32 

    #include <ntddk.h>
    #include <wdf.h>

#endif

#ifdef __FreeBSD__ 
	#include <sys/types.h>

    typedef __uint32_t UINT32;
    typedef __uint16_t UINT16;
#endif

/* empty defines to indicate a private or public API */
#define PUBLIC
#define PRIVATE

/**
 * Section for enum
 */

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum IxTimeSyncAccStatus
 *
 * @brief The status as returned from the API
 */
typedef enum /**< IxTimeSyncAccStatus */
{
    ICP_TIMESYNCACC_SUCCESS,		         /**< Requested operation successful */
    ICP_TIMESYNCACC_INVALIDPARAM,            /**< An invalid parameter was passed */
    ICP_TIMESYNCACC_NOTIMESTAMP,             /**< While polling no time stamp available */
    ICP_TIMESYNCACC_INTERRUPTMODEINUSE,      /**< Polling not allowed while operating in interrupt mode */
    ICP_TIMESYNCACC_FAILED,                  /**< Internal error occurred */
	ICP_TIMESYNCACC_UNINITIALIZED,			 /**< Hardware access library not yet initialized */
}IxTimeSyncAccStatus;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum IxTimeSyncAccAuxMode
 *
 * @brief Master or Slave Auxiliary Time Stamp (Snap Shot)
 */
typedef enum /**< IxTimeSyncAccAuxMode */
{
    ICP_TIMESYNCACC_AUXMODE_MASTER,          /**< Auxiliary Master Mode */
    ICP_TIMESYNCACC_AUXMODE_SLAVE,           /**< Auxiliary Slave Mode */
    ICP_TIMESYNCACC_AUXMODE_INVALID          /**< Invalid Auxiliary Mode */
}IxTimeSyncAccAuxMode;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum IxTimeSyncAcc1588PTPPort
 *
 * @brief IEEE 1588 PTP Communication Port(Channel)
 */
typedef enum /**< IxTimeSyncAcc1588PTPPort */
{
    ICP_TIMESYNCACC_GBE_0_1588PTP_PORT,      /**< PTP Communication Port on GBE-0 */
    ICP_TIMESYNCACC_GBE_1_1588PTP_PORT,      /**< PTP Communication Port on GBE-1 */
    ICP_TIMESYNCACC_CAN_0_1588PTP_PORT,      /**< PTP Communication Port on CAN-0 */
    ICP_TIMESYNCACC_CAN_1_1588PTP_PORT,      /**< PTP Communication Port on CAN-1 */
    ICP_TIMESYNCACC_1588PORT_INVALID 		/**< Invalid PTP Communication Port */ 
} IxTimeSyncAcc1588PTPPort;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum IxTimeSyncAcc1588PTPPortMode
 *
 * @brief Master or Slave mode for IEEE 1588 PTP Communication Port
 */
typedef enum  /**< IxTimeSyncAcc1588PTPPortMode */
{
    ICP_TIMESYNCACC_1588PTP_PORT_MASTER,       /**< PTP Communication Port in Master Mode */
    ICP_TIMESYNCACC_1588PTP_PORT_SLAVE,        /**< PTP Communication Port in Slave Mode */
    ICP_TIMESYNCACC_1588PTP_PORT_ANYMODE,      /**< PTP Communication Port in ANY Mode
                                                  allows time stamping of all messages
                                                  including non-1588 PTP */
    ICP_TIMESYNCACC_1588PTP_PORT_MODE_INVALID  /**< Invalid PTP Port Mode */
}IxTimeSyncAcc1588PTPPortMode;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum ixTimeSyncAcc1588PTPMsgType
 *
 * @brief 1588 PTP Messages types that can be detected on communication port
 *
 * Note that client code can determine this based on master/slave mode in which
 * it is already operating in and this information is made available for the sake
 * of convenience only.
 */
typedef enum  /**< ixTimeSyncAcc1588PTPMsgType */
{
    ICP_TIMESYNCACC_1588PTP_MSGTYPE_SYNC,     /**< PTP Sync message sent by Master or received by Slave */
    ICP_TIMESYNCACC_1588PTP_MSGTYPE_DELAYREQ, /**< PTP Delay_Req message sent by Slave or received by Master */
    ICP_TIMESYNCACC_1588PTP_MSGTYPE_UNKNOWN   /**< Other PTP and non-PTP message sent or received by both
                                                Master and/or Slave */
} ixTimeSyncAcc1588PTPMsgType;


/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum ixTimeSyncAcc1588PTPVersion
 *
 * @brief 1588 PTP version value that can be detected on communication port
 *
 */
typedef enum  /**< ixTimeSyncAcc1588PTPVersion */
{
    ICP_TIMESYNCACC_1588PTP_VERSION_1,     /**Value 0, support version 1 only */
    ICP_TIMESYNCACC_1588PTP_VERSION_2_OR_1     /**Value 1, support both version 1 and version 2 */
} IxTimeSyncAcc1588PTPVersion;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @enum ixTimeSyncAcc1588PTPOperationMode
 *
 * @brief 1588 PTP operation mode value that can be detected on communication port
 *
 */
typedef enum  /**< ixTimeSyncAcc1588PTPOperationMode */
{
    ICP_TIMESYNCACC_1588PTP_OP_MODE_SYNC_DELAYREQ_MSGS=0,     
            /* mode value 0, timestamp SYNC and DELAYED_REQ only */
    ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_ALL_MSGS=1,     
            /* mode value 1, timestamp version 1 all messages */
    ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_V2_EVENT_MSGS=2,     
            /* mode value 2, timestamp version 1 and version event messages only */
    ICP_TIMESYNCACC_1588PTP_OP_MODE_V1_V2_ALL_MSGS=3,     
            /* mode value 3, timestamp version 1 and 2 all messages */
    ICP_TIMESYNCACC_1588PTP_OP_MODE_USER_MSGS=8     
            /* mode value 8, timestamp user messages */
} IxTimeSyncAcc1588PTPOperationMode;

/**
 * Section for struct
 */

/**
 * @ingroup IxTimeSyncAcc
 *
 * @struct IxTimeSyncAccTimeValue
 *
 * @brief Struct to hold 64 bit SystemTime and TimeStamp values
 */
typedef struct  /**< IxTimeSyncAccTimeValue */
{
    UINT32 timeValueLowWord;               /**< Lower 32 bits of the time value */
    UINT32 timeValueHighWord;              /**< Upper 32 bits of the time value */  
} IxTimeSyncAccTimeValue;

/**
 * @ingroup IxTimeSyncAcc
 *
 * @struct ixTimeSyncAccUuid
 *
 * @brief Struct to hold 48 bit UUID values captured in Sync or Delay_Req messages
 */
typedef struct  /**< ixTimeSyncAccUuid */
{
    UINT32 uuidValueLowWord;               /**<The lower 32 bits of the UUID */
    UINT16 uuidValueHighHalfword;          /**<The upper 16 bits of the UUID */  
} ixTimeSyncAccUuid;

/**
 * @ingroup IxTimeSyncAcc
 *
 * @struct IxTimeSyncAccPtpMsgData
 *
 * @brief Struct for data from the PTP message returned when TimeStamp available
 */
typedef struct  /**< IxTimeSyncAccPtpMsgData */
{
    ixTimeSyncAcc1588PTPMsgType ptpMsgType; /**< PTP Messages type */
    IxTimeSyncAccTimeValue ptpTimeStamp;    /**< 64 bit TimeStamp value from PTP Message */
    ixTimeSyncAccUuid ptpUuid;              /**< 48 bit UUID value from the PTP Message */
    UINT16 ptpSequenceNumber;               /**< 16 bit Sequence Number from PTP Message */
} IxTimeSyncAccPtpMsgData;

/**
 * @ingroup IxTimeSyncAcc
 *
 * @struct IxTimeSyncAccStats
 *
 * @brief Statistics for the PTP messages
 */
typedef struct  /**< IxTimeSyncAccStats */
{
    UINT32 rxMsgs; /**< Count of timestamps for received PTP Messages */
    UINT32 txMsgs; /**< Count of timestamps for transmitted PTP Messages */
} IxTimeSyncAccStats;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccAuxTimeIoctl
 *
 * @brief Struct to pass aux time data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccAuxTimeIoctl */
{
    IxTimeSyncAccAuxMode auxMode;   /**< aux mode: master or slave */
    IxTimeSyncAccTimeValue auxTime; /**< aux time snapshot */
} IxTimeSyncAccAuxTimeIoctl;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccPortCfgIoctl
 *
 * @brief Struct to pass port config data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccPortCfgIoctl */
{
    IxTimeSyncAcc1588PTPPort ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IxTimeSyncAcc1588PTPPortMode ptpPortMode; /**< Master, Slave, or Any mode */
} IxTimeSyncAccPortCfgIoctl;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccRxTxPollIoctl
 *
 * @brief Struct to pass PTP message data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccRxTxPollIoctl */
{
    IxTimeSyncAcc1588PTPPort ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IxTimeSyncAccPtpMsgData ptpMsgData; /**< data from the PTP message */
} IxTimeSyncAccRxTxPollIoctl;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccCANPollIoctl
 *
 * @brief Struct to pass CAN timestamp data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccCANPollIoctl */
{
    IxTimeSyncAcc1588PTPPort ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IxTimeSyncAccTimeValue ptpTimeStamp; /**< CAN PTP timestamp */
} IxTimeSyncAccCANPollIoctl;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccTimePollIoctl
 *
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccTimePollIoctl */
{
    UINT32 PollFlag;   /**< time event */
    IxTimeSyncAccTimeValue timeVal; /**< timestamp value */
    IxTimeSyncAccAuxMode auxMode;   /**< Master or Slave mode */
} IxTimeSyncAccTimePollIoctl;

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccVersionIoctl
 *
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccVersionIoctl */
{
    IxTimeSyncAcc1588PTPPort ptpPort;   /**< IEEE 1588 PTP Communication Port */
    IxTimeSyncAcc1588PTPVersion ptpVersion; /**< version value */
} IxTimeSyncAccVersionIoctl;


/**
 * @ingroup IxTimeSyncAcc
 * 
 * @struct IxTimeSyncAccOperationModeIoctl
 *
 * @brief Struct to pass timestamp data for ioctl call
 */
typedef struct  /**< IxTimeSyncAccOperationModeIoctl */
{
    IxTimeSyncAcc1588PTPPort ptpPort;   /* IEEE 1588 PTP Communication Port */
    IxTimeSyncAcc1588PTPOperationMode ptpOpMode; /* 1588 operation mode */
} IxTimeSyncAccOperationModeIoctl;


/**
 * @ingroup IxTimeSyncAcc
 *
 * @typedef IxTimeSyncAccTargetTimeCallback
 *
 * @brief Callback for use by target time stamp interrupt
 */
typedef void (*IxTimeSyncAccTargetTimeCallback)(IxTimeSyncAccTimeValue targetTime);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @typedef IxTimeSyncAccAuxTimeCallback
 *
 * @brief Callback for use by auxiliary time interrupts
 */
typedef void (*IxTimeSyncAccAuxTimeCallback)(IxTimeSyncAccAuxMode auxMode,
             IxTimeSyncAccTimeValue auxTime);
             
/**
 * @ingroup IxTimeSyncAcc
 *
 * @typedef IxTimeSyncAccAuxTargetTimeCallback
 *
 * @brief Callback for use by auxiliary target time stamp interrupt
 */
typedef void (*IxTimeSyncAccAuxTargetTimeCallback)(IxTimeSyncAccTimeValue auxTargetTime);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @typedef IxTimeSyncAccPulsePerSecondCallback
 *
 * @brief Callback for use by pulse per second interrupt
 */
typedef void (*IxTimeSyncAccPulsePerSecondCallback)(UINT32 pulsePerSecond);

/*
 * Section for prototypes interface functions
 */

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccBlPlBaseAddressesSet(
									UINT32 RegsVirtualBaseAddr)
 *
 * @brief Sets virtual addresses for 1588 hardware registers.
 *
 * @param RegsVirtualBaseAddr [in] - register virtual address
 *
 * This API will set the virtual starting adresses for the 1588 
 * hardware registers.
 *          
 * @li Re-entrant   : Yes
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccBlPlBaseAddressesSet(UINT32 RegsVirtualBaseAddr);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPPortConfigSet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAcc1588PTPPortMode ptpPortMode)
 *
 * @brief Configures the IEEE 1588 message detect on particular PTP port.
 *
 * @param ptpPort [in] - PTP port to config
 * @param ptpPortMode [in]- Port to operate in Master or Slave mode
 *
 * This API will enable the time stamping on a particular PTP port.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPPortConfigSet(IxTimeSyncAcc1588PTPPort ptpPort,
       IxTimeSyncAcc1588PTPPortMode ptpPortMode);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPPortConfigGet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAcc1588PTPPortMode *ptpPortMode)
 *
 * @brief Retrieves IEEE 1588 PTP operation mode on particular PTP port.
 *
 * @param ptpPort [in] - PTP port
 * @param ptpPortMode [in]- Mode of operation of PTP port (Master or Slave)
 *
 * This API will identify the time stamping capability of a PTP port by means
 * of obtaining its mode of operation.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPPortConfigGet(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPPortMode *ptpPortMode);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPRxPoll(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAccPtpMsgData  *ptpMsgData)
 *
 * @brief Polls the IEEE 1588 message    ime stamp detect status on a particular 
 * PTP Port on the Receive side.
 *
 * @param ptpPort [in] - PTP port to poll
 * @param ptpMsgData [out] - Current TimeStamp and other Data
 *
 * This API will poll for the availability of a time stamp on the received Sync 
 * (Slave) or Delay_Req (Master) messages.
 * The client application will provide the buffer.
 *              
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_NOTIMESTAMP - No time stamp available
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPRxPoll(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccPtpMsgData  *ptpMsgData);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPTxPoll(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAccPtpMsgData  *ptpMsgData)
 *
 *
 * @brief Polls the IEEE 1588 message    ime stamp detect status on a particular 
 * PTP Port on the Transmit side.
 *
 * @param ptpPort [in] - PTP port to poll
 * @param ptpMsgData [out] - Current TimeStamp and other Data
 *                        
 * This API will poll for the availability of a time stamp on the transmitted
 * Sync (Master) or Delay_Req (Slave) messages.
 * The client application will provide the buffer.
 *              
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_NOTIMESTAMP - No time stamp available
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPTxPoll(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccPtpMsgData  *ptpMsgData);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccSystemTimeSet(
                               IxTimeSyncAccTimeValue systemTime)
 *
 * @brief Sets the System Time in the IEEE 1588 hardware assist block
 *
 * @param systemTime [in] - Value to set System Time
 *                        
 * This API will set the SystemTime to given value.
 *              
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccSystemTimeSet(IxTimeSyncAccTimeValue systemTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccSystemTimeGet(
                               IxTimeSyncAccTimeValue *systemTime)
 *
 * @brief Gets the System Time from the IEEE 1588 hardware assist block
 *
 * @param systemTime [out] - Copy the current System Time into the client 
 * application provided buffer
 *
 * This API will get the SystemTime from IEEE1588 block and return to client
 *      
 * @li Re-entrant   : no
 * @li ISR Callable : no
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccSystemTimeGet(IxTimeSyncAccTimeValue *systemTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTickRateSet(
                               UINT32 tickRate)
 *
 * @brief Sets the Tick Rate (Frequency Scaling Value) in the IEEE 1588
 * hardware assist block
 *
 * @param tickRate [in] - Value to set Tick Rate
 *                        
 * This API will set the Tick Rate (Frequency Scaling Value) in the IEEE
 * 1588 block to the given value. The Accumulator register (not client 
 * visible) is incremented by this TickRate value every clock cycle. When 
 * the Accumulator overflows, the SystemTime is incremented by one. This
 * TickValue can therefore be used to adjust the system timer.
 *           
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTickRateSet(UINT32 tickRate);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTickRateGet(
                               UINT32 *tickRate)
 *
 * @brief Gets the Tick Rate (Frequency Scaling Value) from the IEEE 1588
 * hardware assist block
 *
 * @param tickRate [out] - Current Tick Rate value in the IEEE 1588 block
 *
 * This API will get the TickRate on IEE15588 block. Refer to @ref 
 * ixTimeSyncAccTickRateSet for notes on usage of this value.
 *         
 * @li   Reentrant    : yes
 * @li   ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTickRateGet(UINT32 *tickRate);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTargetTimeInterruptEnable(
                               IxTimeSyncAccTargetTimeCallback targetTimeCallback)
 *
 * @brief Enables the interrupt to verify the condition where the System Time 
 * greater or equal to the Target Time in the IEEE 1588 hardware assist block. 
 * If the condition is true an interrupt will be sent to Intel XScale(R) Core .
 *
 * @param targetTimeCallback [in] - Callback to be invoked when interrupt fires
 *
 * This API will enable the Target Time reached/hit condition interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccTargetTimeInterruptEnable, @ref ixTimeSyncAccTargetTimeSet and
 * @ref ixTimeSyncAccTargetTimeInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed for callback
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeInterruptEnable(IxTimeSyncAccTargetTimeCallback targetTimeCallback);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTargetTimeInterruptDisable(
                               void)
 *
 * @brief Disables the interrupt for the condition explained in the function
 * description of @ref ixTimeSyncAccTargetTimeInterruptEnable.
 *
 * This API will disable the Target Time interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccTargetTimeInterruptEnable, @ref ixTimeSyncAccTargetTimeSet and
 * @ref ixTimeSyncAccTargetTimeInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *              
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeInterruptDisable(void);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTargetTimePoll(
                               UINT32 *ttmPollFlag,
                               IxTimeSyncAccTimeValue *targetTime)
 *
 * @brief Poll to verify the condition where the System Time greater or equal to
 * the Target Time in the IEEE 1588 hardware assist block. If the condition is
 * true an event flag is set in the hardware.
 *
 * @param ttmPollFlag [out] - TRUE  if the target time reached/hit condition
 *				    event is set
 *                            FALSE if the target time reached/hit condition 
 * 				    event is not set
 * @param targetTime [out] - Capture current targetTime into client provided buffer
 *
 * Poll the target time reached/hit condition status. Return true and the
 * current target time value, if the condition is true else return false.
 *
 * @note The client application will need to clear the event flag that will be
 * set as long as the condition that the System Time greater or equal to the
 * Target Time is valid, in one of the following ways:
 *     1) Invoke the API to change the target time
 *     2) Change the system timer value 
 *              
 * @li Re-entrant   : Yes
 * @li ISR Callable : No
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 *         @li ICP_TIMESYNCACC_INTERRUPTMODEINUSE - Interrupt mode in use
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimePoll(UINT32 *ttmPollFlag,
    IxTimeSyncAccTimeValue *targetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTargetTimeSet(
                               IxTimeSyncAccTimeValue targetTime)
 *
 * @brief Sets the Target Time in the IEEE 1588 hardware assist block
 *
 * @param targetTime [in] - Value to set Target Time
 *                        
 * This API will set the Target Time to a given value.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccTargetTimeInterruptEnable, @ref ixTimeSyncAccTargetTimeSet and
 * @ref ixTimeSyncAccTargetTimeInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *              
 * @li   Reentrant    : no
 * @li   ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeSet(IxTimeSyncAccTimeValue targetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccTargetTimeGet(
                               IxTimeSyncAccTimeValue *targetTime)
 *
 * @brief Gets the Target Time in the IEEE 1588 hardware assist block
 *
 * @param targetTime [out] - Copy current time to client provided buffer
 *                                                 
 * This API will get the Target Time from IEEE 1588 block and return to the 
 * client application
 *            
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccTargetTimeGet(IxTimeSyncAccTimeValue *targetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTimeInterruptEnable(
                               IxTimeSyncAccAuxMode auxMode, 
                               IxTimeSyncAccAuxTimeCallback auxTimeCallback)
 *
 * @brief Enables the interrupt notification for the given mode of Auxiliary Time 
 * Stamp in the IEEE 1588 hardware assist block
 * 
 * @param auxMode [in] - Auxiliary time stamp register (slave or master) to use
 * @param auxTimeCallback [in] - Callback to be invoked when interrupt fires 
 *                                                              
 * This API will enable the Auxiliary Master/Slave Time stamp Interrupt.
 *
 * <pre>
 * NOTE: An individual callback is to be registered for each Slave and Master 
 * Auxiliary Time Stamp registers. Thus to register for both Master and Slave time
 * stamp interrupts either the same callback or two separate callbacks the API has
 * to be invoked twice.
 * </pre>
 *              
 * @li Re-entrant   : no
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed for callback or
                                            invalid auxiliary snapshot mode
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimeInterruptEnable(IxTimeSyncAccAuxMode auxMode,
    IxTimeSyncAccAuxTimeCallback auxTimeCallback);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTimeInterruptDisable(
                               IxTimeSyncAccAuxMode auxMode)
 *
 * @brief Disables the interrupt for the indicated mode of Auxiliary Time Stamp
 * in the IEEE 1588 hardware assist block
 *
 * @param auxMode [in] - Auxiliary time stamp mode (slave or master) using which
 * the interrupt will be disabled.
 *                        
 * This API will disable the Auxiliary Time Stamp Interrupt (Master or Slave)
 *              
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimeInterruptDisable(IxTimeSyncAccAuxMode auxMode);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTimePoll(
                               IxTimeSyncAccAuxMode auxMode,
                               UINT32 *auxPollFlag,
                               IxTimeSyncAccTimeValue *auxTime)
 *
 * @brief Poll for the Auxiliary Time Stamp captured for the mode indicated 
 * (Master or Slave)
 *
 * @param auxMode [in] - Auxiliary Snapshot Register (Slave or Master) to be checked
 * @param auxPollFlag [out] - TRUE if the time stamp captured in auxiliary 
                                   snapshot register
 *                            FALSE if the time stamp not captured in 
                                   auxiliary snapshot register
 * @param auxTime [out] - Copy the current Auxiliary Snapshot Register value into the
 * client provided buffer
 *
 * Polls for the Time stamp in the appropriate Auxiliary Snapshot Registers based 
 * on the mode specified. Return true and the contents of the Auxiliary snapshot,
 * if it is available else return false.
 *
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed for auxPollFlag,
                   callback or invalid auxiliary snapshot mode
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 *         @li ICP_TIMESYNCACC_INTERRUPTMODEINUSE - Interrupt mode in use
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTimePoll(IxTimeSyncAccAuxMode auxMode,
    UINT32 *auxPollFlag,
    IxTimeSyncAccTimeValue *auxTime);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @fn IxTimeSyncAccStatus ixTimeSyncAccReset(void)
 *
 * @brief Resets the IEEE 1588 hardware assist block
 *
 * Sets the reset bit in the IEEE1588 silicon which fully resets the silicon block
 *            
 * @li Reentrant    : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED  - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccReset(void);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @fn IxTimeSyncAccStatus ixTimeSyncAccChnlReset(IxTimeSyncAcc1588PTPPort ptpPort)
 *
 * @brief Resets the IEEE 1588 hardware for a single channel
 *
 * @param ptpPort [in] - Port which is to be reset
 *
 * Sets the reset bit in the IEEE1588 silicon which fully resets the channel
 *            
 * @li Reentrant    : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED  - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccChnlReset(IxTimeSyncAcc1588PTPPort ptpPort);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @fn IxTimeSyncAccStatus ixTimeSyncAccStatsGet(IxTimeSyncAccStats
 *                              *timeSyncStats)
 *
 * @brief Returns the IxTimeSyncAcc Statistics in the client supplied buffer
 *
 * @param timeSyncStats [out] - TimeSync statistics counter values
 *
 * This API will return the statistics of the received or transmitted messages.
 * 
 * NOTE: 1) These counters are updated only when the client polls for the time 
 * stamps or interrupt are enabled. This is because the IxTimeSyncAcc module 
 * does not either transmit or receive messages and does only run the code 
 * when explicit requests received by client application.
 *
 *       2) These statistics reflect the number of valid PTP messages exchanged
 * in Master and Slave modes but includes all the messages (including valid 
 * non-PTP messages) while operating in the Any mode.
 *              
 * @li Reentrant    : no
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - NULL parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccStatsGet(IxTimeSyncAccStats *timeSyncStats);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @fn void ixTimeSyncAccStatsReset(void)
 *
 * @brief Reset Time Sync statistics
 *
 * This API will reset the statistics counters of the TimeSync access layer.
 *             
 * @li Reentrant    : yes
 * @li ISR Callable:  no
 *
 * @return @li None
 */
PUBLIC void
ixTimeSyncAccStatsReset(void);

/**
 * @ingroup IxTimeSyncAcc
 *
 * @fn IxTimeSyncAccStatus ixTimeSyncAccShow(void)
 *
 * @brief Displays the Time Sync current status
 *
 * This API will display status on the current configuration of the IEEE
 * 1588 hardware assist block, contents of the various time stamp registers,
 * outstanding interrupts and/or events.
 *
 * Note that this is intended for debug only, and in contrast to the other
 * functions, it does not clear the any of the status bits associated with
 * active timestamps and so is passive in its nature.
 *            
 * @li Reentrant    : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccShow(void);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTargetTimeInterruptEnable(
                               IxTimeSyncAccAuxTargetTimeCallback auxTargetTimeCallback)
 *
 * @brief Enables the interrupt to verify the condition where the System Time 
 * greater or equal to the Auxiliary Target Time in the IEEE 1588 hardware assist block. 
 * If the condition is true an interrupt will be sent to Intel XScale(R) Core .
 *
 * @param auxTargetTimeCallback [in] - Callback to be invoked when interrupt fires
 *
 * This API will enable the Auxiliary Target Time reached/hit condition interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccAuxTargetTimeInterruptEnable and
 * @ref ixTimeSyncAccAuxTargetTimeInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed for callback
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeInterruptEnable(IxTimeSyncAccAuxTargetTimeCallback auxTargetTimeCallback);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTargetTimeInterruptDisable(
                               void)
 *
 * @brief Disables the interrupt for the condition explained in the function
 * description of @ref ixTimeSyncAccAuxTargetTimeInterruptEnable.
 *
 * This API will disable the Auxiliary Target Time interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccAuxTargetTimeInterruptEnable and
 * @ref ixTimeSyncAccAuxTargetTimeInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *              
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeInterruptDisable(void);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTargetTimePoll(
                               UINT32 *atmPollFlag,
                               IxTimeSyncAccTimeValue *auxTargetTime)
 *
 * @brief Poll to verify the condition where the System Time greater or equal to
 * the Auxiliary Target Time in the IEEE 1588 hardware assist block. If the condition is
 * true an event flag is set in the hardware.
 *
 * @param atmPollFlag [out] - TRUE  if the auxiliary target time reached/hit condition event set
 *                            FALSE if the auxiliary target time reached/hit condition event is 
                                    not set
 * @param auxTargetTime [out] - Capture current auxTargetTime into client provided buffer
 *
 * Poll the auxiliary target time reached/hit condition status. Return true and the current
 * auxiliary target time value, if the condition is true else return false.
 *
 * NOTE: The client application will need to clear the event flag that will be set
 * as long as the condition that the System Time greater or equal to the Auxiliary Target Time is
 * valid, in one of the following ways:
 *     1) Invoke the API to change the target time
 *     2) Change the system timer value 
 *              
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 *         @li ICP_TIMESYNCACC_INTERRUPTMODEINUSE - Interrupt mode in use
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimePoll(UINT32 *atmPollFlag,
    IxTimeSyncAccTimeValue *auxTargetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTargetTimeSet(
                               IxTimeSyncAccTimeValue auxTargetTime)
 *
 * @brief Sets the Auxiliary Target Time in the IEEE 1588 hardware assist block
 *
 * @param auxTargetTime [in] - Value to set Auxiliary Target Time
 *                        
 * This API will set the Auxiliary Target Time to a given value.
 *
 * @note The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccAuxTargetTimeInterruptEnable(), @ref ixTimeSyncAccAuxTargetTimeSet() and
 * @ref ixTimeSyncAccAuxTargetTimeInterruptDisable() are accessed in mutual exclusive 
 * manner with respect to each other.
 *              
 * @li   Reentrant    : No
 * @li   ISR Callable : Yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeSet(IxTimeSyncAccTimeValue auxTargetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccAuxTargetTimeGet(
                               IxTimeSyncAccTimeValue *auxTargetTime)
 *
 * @brief Gets the Auxiliary Target Time in the IEEE 1588 hardware assist block
 *
 * @param auxTargetTime [out] - Copy current time to client provided buffer
 *                                                 
 * This API will get the Auxiliary Target Time from IEEE 1588 block and return to the 
 * client application
 *            
 * @li Re-entrant   : Yes
 * @li ISR Callable : No
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccAuxTargetTimeGet(IxTimeSyncAccTimeValue *auxTargetTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPulsePerSecondInterruptEnable(
                               IxTimeSyncAccPulsePerSecondCallback pulsePerSecondCallback)
 *
 * @brief Enables the interrupt to verify the condition where the Pulse Per Second Compare 
 * register indicates a match in the IEEE 1588 hardware assist block. 
 * If the condition is true an interrupt will be sent to Intel XScale(R) Core .
 *
 * @param pulsePerSecondCallback [in] - Callback to be invoked when interrupt fires
 *
 * This API will enable the Pulse Per Second reached/hit condition interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccPulsePerSecondInterruptEnable and
 * @ref ixTimeSyncAccPulsePerSecondInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed for callback
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondInterruptEnable(IxTimeSyncAccPulsePerSecondCallback pulsePerSecondCallback);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPulsePerSecondInterruptDisable(
                               void)
 *
 * @brief Disables the interrupt for the condition explained in the function
 * description of @ref ixTimeSyncAccPulsePerSecondInterruptEnable.
 *
 * This API will disable the Pulse Per Second interrupt.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccPulsePerSecondInterruptEnable and
 * @ref ixTimeSyncAccPulsePerSecondInterruptDisable are accessed in mutual exclusive 
 * manner with respect to each other.
 *              
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondInterruptDisable(void);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPulsePerSecondTimeGet(
                               UINT32 *ppsTime)
 *
 * @brief Gets the Pulse Per Second Time in the IEEE 1588 hardware assist block
 *
 * @param ppsTime [out] - Copy current time to client provided buffer
 *                                                 
 * This API will get the PPS Time from IEEE 1588 block and return to the 
 * client application
 *            
 * @li Re-entrant   : yes
 * @li ISR Callable : no
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Null parameter passed
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondTimeGet(UINT32 *ppsTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPulsePerSecondTimeSet(
                               UINT32 ppsTime)
 *
 * @brief Sets the Pulse Per Second Time in the IEEE 1588 hardware assist block
 *
 * @param ppsTime [in] - Value to set PPS Time
 *                        
 * This API will set the Pulse Per Second Time to a given value.
 *
 * NOTE: The client application needs to ensure that the APIs 
 * @ref ixTimeSyncAccPulsePerSecondInterruptEnable, 
 * @ref ixTimeSyncAccPulsePerSecondTimeSet and
 * @ref ixTimeSyncAccPulsePerSecondInterruptDisable are accessed 
 * in mutual exclusive manner with respect to each other.
 *              
 * @li   Reentrant    : no
 * @li   ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPulsePerSecondTimeSet(UINT32 ppsTime);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPCANPoll(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAccTimeValue *ptpTimeStamp)
 *
 * @brief Polls the IEEE 1588 message    ime stamp detect status on a particular 
 * CAN PTP Port.
 *
 * @param ptpPort [in] - PTP port to poll
 * @param ptpTimeStamp [out] - Current TimeStamp
 *
 * This API will poll for the availability of a time stamp on a CAN port.
 * The client application will provide the buffer.
 *              
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_NOTIMESTAMP - No time stamp available
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPCANPoll(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAccTimeValue *ptpTimeStamp);


/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPVersionSet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAcc1588PTPVersion ptpVersion)
 *
 * @brief Configures the IEEE 1588 version number on particular PTP port.
 *
 * @param ptpPort [in] - PTP port to config
 * @param ptpVersion [out]- version value '0' or '1'
 *
 * This API will the supporting version number on a particular PTP port.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_UNINITIALIZED - Uninitialized
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPVersionSet(IxTimeSyncAcc1588PTPPort ptpPort,
       IxTimeSyncAcc1588PTPVersion ptpVersion);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPVersionGet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAcc1588PTPVersion *ptpVersion)
 *
 * @brief Retrieves IEEE 1588 PTP version on particular PTP port.
 *
 * @param ptpPort [in] - PTP port
 * @param ptpPortMode [in]- Version supported of PTP port (Master or Slave)
 *
 * This API will retrieves IEEE 1588 PTP version on particular PTP port.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li IICP_TIMESYNCACC_UNINITIALIZED - Uninitialized
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPVersionGet(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPVersion *ptpVersion);


/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPOperationModeSet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               ixTimeSyncAcc1588PTPOperationMode ptpOpMode)
 *
 * @brief Configures the IEEE 1588 operation mode on particular PTP port.
 *
 * @param ptpPort [in] - PTP port to config
 * @param ptpOpMode [in]- Operation mode to configure the timestamp. 
 *
 * This API will set the operation mode on a particular PTP port.
 * The operation mode definition is depending on the version that the port 
 * supports.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_UNINITIALIZED - Uninitialized
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPOperationModeSet(IxTimeSyncAcc1588PTPPort ptpPort,
       IxTimeSyncAcc1588PTPOperationMode ptpOpMode);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccPTPVersionGet(
                               IxTimeSyncAcc1588PTPPort ptpPort,
                               IxTimeSyncAcc1588PTPOperationMode *ptpOpMode)
 *
 * @brief Retrieves IEEE 1588 PTP operation mode on particular PTP port.
 *
 * @param ptpPort [in] - PTP port
 * @param ptpOpMode [out]- operation mode of PTP port 
 *
 * This API will retrieves IEEE 1588 PTP operation mode on particular PTP port.
 *          
 * @li Re-entrant   : No
 * @li ISR Callable : No
 * 
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_INVALIDPARAM - Invalid parameters passed
 *         @li ICP_TIMESYNCACC_UNINITIALIZED - Uninitialized
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccPTPOperationModeGet(IxTimeSyncAcc1588PTPPort ptpPort,
    IxTimeSyncAcc1588PTPOperationMode *ptpOpMode);

/**
 * @ingroup IxTimeSyncAcc
 * 
 * @fn IxTimeSyncAccStatus ixTimeSyncAccIsr(void)
 *
 * @brief Processes 1588 interrupt events.
 *
 * This API will process interrupt events.
 *
 * @li Re-entrant   : no
 * @li ISR Callable : yes
 *
 * @return @li ICP_TIMESYNCACC_SUCCESS - Operation is successful
 *         @li ICP_TIMESYNCACC_FAILED - Internal error occurred
 */
PUBLIC IxTimeSyncAccStatus
ixTimeSyncAccIsr(void);


extern UINT32 ixTimeSyncAccControlAmmsInterruptMaskGet(void);
extern UINT32 ixTimeSyncAccControlAsmsInterruptMaskGet(void);
extern UINT32 ixTimeSyncAccControlTtmInterruptMaskGet(void);
extern UINT32 ixTimeSyncAccControlAtmInterruptMaskGet(void);
extern UINT32 ixTimeSyncAccControlPpsmInterruptMaskGet(void);

extern void ixTimeSyncAccControlAmmsInterruptMaskSet(void);
extern void ixTimeSyncAccControlAsmsInterruptMaskSet(void);
extern void ixTimeSyncAccControlTtmInterruptMaskSet(void);
extern void ixTimeSyncAccControlAtmInterruptMaskSet(void);
extern void ixTimeSyncAccControlPpsmInterruptMaskSet(void);

extern void ixTimeSyncAccControlAmmsInterruptMaskClear(void);
extern void ixTimeSyncAccControlAsmsInterruptMaskClear(void);
extern void ixTimeSyncAccControlTtmInterruptMaskClear(void);
extern void ixTimeSyncAccControlAtmInterruptMaskClear(void);
extern void ixTimeSyncAccControlPpsmInterruptMaskClear(void);

extern void ixTimeSyncAccSystemTimeSnapshotGet(UINT32 *systemTimeLow, UINT32 *systemTimeHigh);
extern void ixTimeSyncAccTargetTimeSnapshotGet(UINT32 *targetTimeLow, UINT32 *targetTimeHigh);
extern void ixTimeSyncAccAuxTargetTimeSnapshotGet(UINT32 *auxTargetTimeLow, UINT32 *auxTargetTimeHigh);
extern void ixTimeSyncAccPulsePerSecondCompareGet (UINT32 *ppsTime);
extern void ixTimeSyncAccAddendFsvGet (UINT32 *fsv);

extern void ixTimeSyncAccSystemTimeSnapshotSet(UINT32 systemTimeLow, UINT32 systemTimeHigh);
extern void ixTimeSyncAccTargetTimeSnapshotSet(UINT32 targetTimeLow, UINT32 targetTimeHigh);
extern void ixTimeSyncAccAuxTargetTimeSnapshotSet (UINT32 auxTargetTimeLow, UINT32 auxTargetTimeHigh);
extern void ixTimeSyncAccPulsePerSecondCompareSet (UINT32 ppsTime);
extern void ixTimeSyncAccAddendFsvSet (UINT32 fsv);

extern UINT32 ixTimeSyncAccControlPTPPortMasterModeGet (UINT32 ptpPort);
extern UINT32 ixTimeSyncAccControlPTPPortPTPMsgTimestampGet(UINT32 ptpPort);

extern void ixTimeSyncAccControlPTPPortMasterModeSet (UINT32 ptpPort, UINT32 masterMode);
extern void ixTimeSyncAccControlPTPPortPTPMsgTimestampSet (UINT32 ptpPort, UINT32 allMsg);

extern void ixTimeSyncAccControlPTPVersionSet (UINT32 ptpPort, UINT32 versionVal);
extern UINT32 ixTimeSyncAccControlPTPVersionGet (UINT32 ptpPort);

extern void ixTimeSyncAccControlPTPOperationModeSet (UINT32 ptpPort, UINT32 modeValue);
extern UINT32 ixTimeSyncAccControlPTPOperationModeGet (UINT32 ptpPort);

#endif /* ICP_TIMESYNCACC_H */

/**
 * @} defgroup IxTimeSyncAcc
 */

