/*****************************************************************************
 * @ingroup WDT
 *
 * @file wdt.h
 *
 * @description
 *  This file contains the common data defines required by both the WDT driver
 *  and applications that use the driver
 ****************************************************************************/
/*
 * INTEL CONFIDENTIAL
 * Copyright 2007,2008,2009 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents related to the
 * source code ("Material") are owned by Intel Corporation or its suppliers or
 * licensors. Title to the Material remains with Intel Corporation or its
 * suppliers and licensors. The Material may contain trade secrets and proprietary
 * and confidential information of Intel Corporation and its suppliers and
 * licensors, and is protected by worldwide copyright and trade secret laws and
 * treaty provisions. No part of the Material may be used, copied, reproduced,
 * modified, published, uploaded, posted, transmitted, distributed, or disclosed
 * in any way without Intels prior express written permission.
 * No license under any patent, copyright, trade secret or other intellectual
 * property right is granted to or conferred upon you by disclosure or delivery
 * of the Materials, either expressly, by implication, inducement, estoppel or
 * otherwise. Any license under such intellectual property rights must be
 * express and approved by Intel in writing.
 * 
 * Include any supplier copyright notices as supplier requires Intel to use.
 * Include supplier trademarks or logos as supplier requires Intel to use,
 * preceded by an asterisk.
 * An asterisked footnote can be added as follows: 
 *   *Third Party trademarks are the property of their respective owners.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or alter this
 * notice or any other notice embedded in Materials by Intel or Intels suppliers
 * or licensors in any way.
 * 
 *  version: Embedded.X.1.0.3-127
 */


#ifndef _WDT_H_
#define _WDT_H_

#include <DEVIOCTL.H>
#include <initguid.h>


#define DRIVERNAME "WDT"

DEFINE_GUID(GUID_DEVINTERFACE_WDT, 0x4D36E97D, 0xE325, 0x11CE, 0xBF, 0xC1, 0x08, 0x00, 0x2B, 0xE1, 0x03, 0x18);


/*----------------------------------------------------------------------
 * IOCTL Defines
 *----------------------------------------------------------------------
 */
/* Get device capabilities, defined by Microsoft */
#define IOCTL_GET_CAPABILITIES               \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x802,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* 0x803-0xCFF	Reserved for future use by Microsoft */

/* 0xD00-0xFFF	OEM-definable*/

/* Set WDT Timer Enable bit to Enabled */
#define IOCTL_ENABLE_WDT                      \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D00,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set WDT Timer Enable bit to Disabled */
#define IOCTL_DISABLE_WDT                     \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D01,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get WDT Timer Enable bit, returns WDT_ENABLED or WDT_DISABLED */
#define IOCTL_GET_WDT_ENABLE                  \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D02,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set WDT Timer Lock bit, locks the values of WDT Timeout Configuration
and WDT Timer Enable until a hard-reset occurs or power is cycled. */
#define IOCTL_LOCK_DEVICE                     \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D03,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set Preload Register 1 */
#define IOCTL_SET_TIMEOUT_1                   \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D04,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set Preload Register 2 */
#define IOCTL_SET_TIMEOUT_2                   \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D05,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get Preload Register 1 */
#define IOCTL_GET_TIMEOUT_1                   \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D06,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get Preload Register 2 */
#define IOCTL_GET_TIMEOUT_2                   \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D07,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Sets the WDT_RELOAD bit to 1 to prevent a tiemout */
#define IOCTL_PING_THE_WDT                    \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D08,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get Down Counter Registers */
#define IOCTL_GET_DOWN_COUNTER                \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D09,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set WDT Timeout Configuration bit, set WDT_MODE for Watchdog Timer Mode:
When enabled (i.e. WDT_ENABLE goes from 0 to 1) the timer reloads
Preload Value 1 and start decrementing. Upon reaching the second stage
timeout the WDT_TOUT# is driven low once and does not change again until
Power is cycled or a hard reset occurs. set FREE_MODE for Free Running
Mode: WDT_TOUT# changes from previous state when the next timeout occurs. The
timer ignores the first stage. The timer only uses Preload Value 2. In this
mode the timer is restarted whenever WDT_ENABLE goes from a 0 to a 1. This
means that the timer reloads Preload Value 2 and start decrementing every
time it is enabled.*/
#define IOCTL_SET_MODE                        \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0A,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get the WDT Timeout Configuration bit, returns WDT_MODE or FREE_MODE */
#define IOCTL_GET_MODE                        \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0B,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get WDT Timeout bit (TOUT), in WDT Mode the ioctlt returns UNSTABLE if the
   down counter reached zero for the second time in a row, in Free running
   mode the ioctl return UNSTABLE if the down counter has reached zero */
#define IOCTL_GET_TIMEOUT                     \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0C,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Clears the WDT Timeout bit (TOUT) in Reload Register 1 and returns
STABLE or UNSTABLE */
#define IOCTL_CLR_TIMEOUT                     \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0D,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set the WDT Prescaler Select bit, set HIGH_SCALE for the 20-bit Preload
Value is loaded into bits 34:15 of the main down counter, which the
approximate clock generated is 1 KHz, (1 ms to 10 min) or set LOW_SCALE for the
20-bit Preload Value is loaded into bits 24:05 of the main down counter,
which the approximate clock generated is 1 MHz, (1 s to 1sec) */
#define  IOCTL_SET_PRESCALER                  \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0E,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get the WDT Prescaler Select bit, returns HIGH_SCALE or LOW_SCALE. */
#define  IOCTL_GET_PRESCALER                  \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D0F,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set the WDT Timeout Output Enable bit, set EXTOUT_ENABLED for WDT to toggle
the external WDT_TOUT# pin if the WDT times out or set EXTOUT_DIABLED for
WDT to not toggle the external WDT_TOUT# pin. */
#define IOCTL_SET_EXTERNAL_OUT                \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D10,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get the WDT Timeout Output Enable bit, returns EXTOUT_ENABLED
or EXT_OUTDIABLED. */
#define IOCTL_GET_EXTERNAL_OUT                \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D11,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Set the WDT interrupt type, set to WDT_INT_TYPE_IRQ, WDT_INT_TYPE_NMI
WDT_INT_TYPE_SMI or WDT_INT_TYPE_DIS. */
#define IOCTL_SET_INTERRUPT_TYPE              \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D12,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* Get the WDT interrupt type, returns WDT_INT_TYPE_IRQ, WDT_INT_TYPE_NMI
WDT_INT_TYPE_SMI or WDT_INT_TYPE_DIS. */
#define IOCTL_GET_INTERRUPT_TYPE              \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D13,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

/* User App provides the user-mode-created event handle. */
#define IOCTL_USER_HANDLE                     \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D14,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )

#define IOCTL_TEST_CALLBACK                   \
    CTL_CODE( FILE_DEVICE_UNKNOWN, 0x0D15,    \
        METHOD_BUFFERED, FILE_ANY_ACCESS )






/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL code to return capabiltity information to
 * the caller
 *-----------------------------------------------------------------------------
 */

typedef struct _SAWD_CAPABILITY_OUT_BUFF {

    unsigned long Version;    /* version of interface used */
    unsigned long Capability; /* bit field indicating capabilities */
    unsigned long min;        /* minimum value in ms */
    unsigned long max;        /* maximum value in ms */

} SAWD_CAPABILITY_OUT_BUFF, *PSAWD_CAPABILITY_OUT_BUFF;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL code IOCTL_GET_WDT_ENABLE to get WDT Timer
 * Enable bit.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_WDTENABLE {

    unsigned long    State;     /* Holds WDT_ENABLE bit of WDT Lock register */

} SAWD_WDTENABLE, *PSAWD_WDTENABLE;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_SET_TIMEOUT_1, IOCTL_SET_TIMEOUT_2,
 * IOCTL_GET_TIMEOUT_1, & IOCTL_GET_TIMEOUT_2 to set or get Preload Register #1
 * or Preload Register #2.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_TIMEOUT {

    unsigned long    Timeout;     /* Preload registers */

} SAWD_TIMEOUT, *PSAWD_TIMEOUT;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL code IOCTL_GET_DOWN_COUNTER to get Down Counter
 * registers.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_DC {

    unsigned long    Counter;     /* Holds current down count */

} SAWD_DC, *PSAWD_DC;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_SET_MODE & IOCTL_GET_MODE to
 * set or get Mode.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_TOUTCONFIG {

    unsigned long    Mode; /* Holds Watchdog Timer Mode or Free Running mode */

} SAWD_TOUTCONFIG, *PSAWD_TOUTCONFIG;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_GET_TIMEOUT & IOCTL_CLR_TIMEOUT to
 * get or clr TIMEOUT.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_TOUT {

    unsigned long    State;     /* Holds Normal or Unstable */

} SAWD_TOUT, *PSAWD_TOUT;


/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_SET_PRESCALER & IOCTL_GET_PRESCALER
 * to set or get PreScaler.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_PRESCALER {

    unsigned long    Scale;     /* WDT_PRE_SEL Bit of WDT Config register */

} SAWD_PRESCALER, *PSAWD_PRESCALER;

/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_SET_EXTERNAL_OUT &
 * IOCTL_GET_EXTERNAL_OUT to set or get Timeout Ouput Enable.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_EXTERNALOUT {

    unsigned long    State;     /* WDT_TOUT_EN Bit of WDT Config register */

} SAWD_EXTERNALOUT , *PSAWD_EXTERNALOUT ;

/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL codes IOCTL_SET_INTERRUPT_TYPE &
 * IOCTL_GET_INTERRUPT_STATUS to set or get the interrupt type.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_IRQTYPE {

    unsigned long    Type;     /* Type of Interrupt */

} SAWD_IRQTYPE , *PSAWD_IRQTYPE ;

/*-----------------------------------------------------------------------------
 * Structure used by the IOCTL code IOCTL_USER_HANDLE to set handle of the user
 * callback.
 *-----------------------------------------------------------------------------
 */
typedef struct _SAWD_UHANDLE {

    HANDLE    UserEvent;     /* Handle to a Ring 3 user event */

} SAWD_UHANDLE , *PSAWD_UHANDLE ;


/*-----------------------------------------------------------------------------
 * Enumerated values
 *-----------------------------------------------------------------------------
 */

/* WDT Timer Enable */
enum {
    WDT_DISABLED,  /* 0 */
    WDT_ENABLED,   /* 1 */
};

/* WDT device modes of operation */
enum {
    WDT_MODE,  /* 0 */
    FREE_MODE, /* 1 */
};

/* System Status from WDT_TIMEOUT, Unstable when TIMEOUT occurs */
enum {
    NORMAL,   /* 0 */
    UNSTABLE, /* 1 */
};

/* Timeout value granulaties */
enum {
    HIGH_SCALE, /* 0 */
    LOW_SCALE,  /* 1 */
};

/* External Out PIN enable or disable */
enum {
    EXTOUT_ENABLED,   /* 0 */
    EXTOUT_DISABLED,  /* 1 */
};

/* Available interrupt types after 1st stage timeout */
enum {
    WDT_INT_TYPE_IRQ, /* 0 - SERIRQ default */
    WDT_INT_TYPE_NMI, /* 1 - NMI        */
    WDT_INT_TYPE_SMI, /* 2 - SMI        */
    WDT_INT_TYPE_DIS, /* 3 - Disabled     */
};


#endif

