/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: NPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Netscape Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/NPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is 
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or 
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the NPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the NPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/////////////////////////////////////////////////////////////////////////////
// Documentation by Shanyu
// The implementation of the actual NPAPI Plugin
//
// To add new a property to the plugin:
// 1. add a property to class ScriptablePluginObject
// 2. define its sequence number below like PROPERTY_SEQ_SPSTATUS
// 3. update NUM_PROPERTY
// 4. define the property name string in g_propertyNames
// 5. convert type of the property properly in ScriptablePluginObject::GetProperty
//
// To add a new method to the plugin:
// 1. define its sequence number below like METHOD_SEQ_SPGETSTATUS
// 2. update NUM_METHOD
// 3. define the method name string in g_methodNames
// 4. implement the method in ScriptablePluginObject:Invoke
/////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"

//////////////////////////////////////////////////
//
// CPlugin class implementation
//
#include <windows.h>
#include <windowsx.h>
#include <atlbase.h>
#include <atlcom.h>
#include <comutil.h>
#include <vector>
#include <string>

#include "plugin.h"
#include "npupp.h"
#include "SpBpError.h"

#define PLUGIN_VERSION		L"1.0.0.0"

//Define properties and methods of our scriptable plugin
#define NUM_PROPERTY				4
#define PROPERTY_SEQ_SPSTATUS		0
#define PROPERTY_SEQ_SPVERSION		1
#define PROPERTY_SEQ_SPRESPONSE		2
#define PROPERTY_SEQ_ERRORCODE		3

static const char* g_propertyNames[NUM_PROPERTY] = {
	"SpStatus",
	"SpVersion",
	"SpResponse",
	"errorCode"
};

#define NUM_METHOD						2
#define METHOD_SEQ_SPGETSTATUS			0
#define METHOD_SEQ_SPPROCESSREQUEST		1
static const char* g_methodNames[NUM_METHOD] = {
	"SpGetStatus",
	"SpProcessRequest"
};

using namespace std;

/////////////////////////////////////////////////////////////////////////////
// class ScriptablePluginObjectBase
// Helper class that can be used to map calls to the NPObject hooks
// into virtual methods on instances of classes that derive from this
// class.
class ScriptablePluginObjectBase : public NPObject
{
public:
  ScriptablePluginObjectBase(NPP npp)
    : mNpp(npp)
  {
  }

  virtual ~ScriptablePluginObjectBase()
  {
  }

  // Virtual NPObject hooks called through this base class. Override
  // as you see fit.
  virtual void Invalidate();
  virtual bool HasMethod(NPIdentifier name);
  virtual bool Invoke(NPIdentifier name, const NPVariant *args,
                      uint32_t argCount, NPVariant *result);
  virtual bool InvokeDefault(const NPVariant *args, uint32_t argCount,
                             NPVariant *result);
  virtual bool HasProperty(NPIdentifier name);
  virtual bool GetProperty(NPIdentifier name, NPVariant *result);
  virtual bool SetProperty(NPIdentifier name, const NPVariant *value);
  virtual bool RemoveProperty(NPIdentifier name);
  virtual bool Enumerate(NPIdentifier **identifier, uint32_t *count);
  virtual bool Construct(const NPVariant *args, uint32_t argCount,
                         NPVariant *result);

public:
  static void _Deallocate(NPObject *npobj);
  static void _Invalidate(NPObject *npobj);
  static bool _HasMethod(NPObject *npobj, NPIdentifier name);
  static bool _Invoke(NPObject *npobj, NPIdentifier name,
                      const NPVariant *args, uint32_t argCount,
                      NPVariant *result);
  static bool _InvokeDefault(NPObject *npobj, const NPVariant *args,
                             uint32_t argCount, NPVariant *result);
  static bool _HasProperty(NPObject * npobj, NPIdentifier name);
  static bool _GetProperty(NPObject *npobj, NPIdentifier name,
                           NPVariant *result);
  static bool _SetProperty(NPObject *npobj, NPIdentifier name,
                           const NPVariant *value);
  static bool _RemoveProperty(NPObject *npobj, NPIdentifier name);
  static bool _Enumerate(NPObject *npobj, NPIdentifier **identifier,
                         uint32_t *count);
  static bool _Construct(NPObject *npobj, const NPVariant *args,
                         uint32_t argCount, NPVariant *result);

protected:
  NPP mNpp;
};

#define DECLARE_NPOBJECT_CLASS_WITH_BASE(_class, ctor)                        \
static NPClass s##_class##_NPClass = {                                        \
  NP_CLASS_STRUCT_VERSION,                                               \
  ctor,                                                                       \
  ScriptablePluginObjectBase::_Deallocate,                                    \
  ScriptablePluginObjectBase::_Invalidate,                                    \
  ScriptablePluginObjectBase::_HasMethod,                                     \
  ScriptablePluginObjectBase::_Invoke,                                        \
  ScriptablePluginObjectBase::_InvokeDefault,                                 \
  ScriptablePluginObjectBase::_HasProperty,                                   \
  ScriptablePluginObjectBase::_GetProperty,                                   \
  ScriptablePluginObjectBase::_SetProperty,                                   \
  ScriptablePluginObjectBase::_RemoveProperty                                \
}

#define GET_NPOBJECT_CLASS(_class) &s##_class##_NPClass

void
ScriptablePluginObjectBase::Invalidate()
{
}

bool
ScriptablePluginObjectBase::HasMethod(NPIdentifier name)
{
  return false;
}

bool
ScriptablePluginObjectBase::Invoke(NPIdentifier name, const NPVariant *args,
                                   uint32_t argCount, NPVariant *result)
{
  return false;
}

bool
ScriptablePluginObjectBase::InvokeDefault(const NPVariant *args,
                                          uint32_t argCount, NPVariant *result)
{
  return false;
}

bool
ScriptablePluginObjectBase::HasProperty(NPIdentifier name)
{
  return false;
}

bool
ScriptablePluginObjectBase::GetProperty(NPIdentifier name, NPVariant *result)
{
  return false;
}

bool
ScriptablePluginObjectBase::SetProperty(NPIdentifier name,
                                        const NPVariant *value)
{
  return false;
}

bool
ScriptablePluginObjectBase::RemoveProperty(NPIdentifier name)
{
  return false;
}

bool
ScriptablePluginObjectBase::Enumerate(NPIdentifier **identifier,
                                      uint32_t *count)
{
  return false;
}

bool
ScriptablePluginObjectBase::Construct(const NPVariant *args, uint32_t argCount,
                                      NPVariant *result)
{
  return false;
}

// static
void
ScriptablePluginObjectBase::_Deallocate(NPObject *npobj)
{
  // Call the virtual destructor.
  delete (ScriptablePluginObjectBase *)npobj;
}

// static
void
ScriptablePluginObjectBase::_Invalidate(NPObject *npobj)
{
  ((ScriptablePluginObjectBase *)npobj)->Invalidate();
}

// static
bool
ScriptablePluginObjectBase::_HasMethod(NPObject *npobj, NPIdentifier name)
{
  return ((ScriptablePluginObjectBase *)npobj)->HasMethod(name);
}

// static
bool
ScriptablePluginObjectBase::_Invoke(NPObject *npobj, NPIdentifier name,
                                    const NPVariant *args, uint32_t argCount,
                                    NPVariant *result)
{
  return ((ScriptablePluginObjectBase *)npobj)->Invoke(name, args, argCount,
                                                       result);
}

// static
bool
ScriptablePluginObjectBase::_InvokeDefault(NPObject *npobj,
                                           const NPVariant *args,
                                           uint32_t argCount,
                                           NPVariant *result)
{
  return ((ScriptablePluginObjectBase *)npobj)->InvokeDefault(args, argCount,
                                                              result);
}

// static
bool
ScriptablePluginObjectBase::_HasProperty(NPObject * npobj, NPIdentifier name)
{
  return ((ScriptablePluginObjectBase *)npobj)->HasProperty(name);
}

// static
bool
ScriptablePluginObjectBase::_GetProperty(NPObject *npobj, NPIdentifier name,
                                         NPVariant *result)
{
  return ((ScriptablePluginObjectBase *)npobj)->GetProperty(name, result);
}

// static
bool
ScriptablePluginObjectBase::_SetProperty(NPObject *npobj, NPIdentifier name,
                                         const NPVariant *value)
{
  return ((ScriptablePluginObjectBase *)npobj)->SetProperty(name, value);
}

// static
bool
ScriptablePluginObjectBase::_RemoveProperty(NPObject *npobj, NPIdentifier name)
{
  return ((ScriptablePluginObjectBase *)npobj)->RemoveProperty(name);
}

// static
bool
ScriptablePluginObjectBase::_Enumerate(NPObject *npobj,
                                       NPIdentifier **identifier,
                                       uint32_t *count)
{
  return ((ScriptablePluginObjectBase *)npobj)->Enumerate(identifier, count);
}

// static
bool
ScriptablePluginObjectBase::_Construct(NPObject *npobj, const NPVariant *args,
                                       uint32_t argCount, NPVariant *result)
{
  return ((ScriptablePluginObjectBase *)npobj)->Construct(args, argCount,
                                                          result);
}


/////////////////////////////////////////////////////////////////////////////
// class ScriptablePluginObject
// Our custom implementation of NPClass functions such as HasMethod() and Invoke()

class ScriptablePluginObject : public ScriptablePluginObjectBase
{
public:
	ScriptablePluginObject(NPP npp)
	: ScriptablePluginObjectBase(npp),
		m_pProxy( NULL )
	{
		for( int i=0; i<NUM_PROPERTY; ++i )
		{
			m_propertyNameIds.push_back( NPN_GetStringIdentifier(g_propertyNames[i]) );
		}
		for( int i=0; i<NUM_METHOD; ++i )
		{
			m_methodNameIds.push_back( NPN_GetStringIdentifier(g_methodNames[i]) );
		}
	}

	virtual bool HasMethod(NPIdentifier name);
	virtual bool HasProperty(NPIdentifier name);
	virtual bool GetProperty(NPIdentifier name, NPVariant *result);
	virtual bool Invoke(NPIdentifier name, const NPVariant *args,
					  uint32_t argCount, NPVariant *result);
	virtual bool InvokeDefault(const NPVariant *args, uint32_t argCount,
							 NPVariant *result);

	void SetTokenSvcProxy( CTokenSvcProxy* pProxy )
	{
		NPLOG( "ScriptablePluginObject::SetTokenSvcProxy() called\n" );
		m_pProxy = pProxy;
	};

private:
	vector<NPIdentifier>	m_methodNameIds;
	vector<NPIdentifier>	m_propertyNameIds;
	
	//custom properties, strings are always UTF8!
	string					m_strSpStatus;
	string					m_strSpVersion;
	string					m_strSpResponse;
	string					m_strSpErrorCode;

	CTokenSvcProxy*			m_pProxy;
};

static NPObject *
AllocateScriptablePluginObject(NPP npp, NPClass *aClass)
{
  return new ScriptablePluginObject(npp);
}

DECLARE_NPOBJECT_CLASS_WITH_BASE(ScriptablePluginObject,
                                 AllocateScriptablePluginObject);

bool
ScriptablePluginObject::HasMethod(NPIdentifier name)
{
	NPUTF8* a = NPN_UTF8FromIdentifier( name );
	if( a ) 
	{
		NPLOG( "ScriptablePluginObject::HasMethod() called: \n" );
		NPLOG( a );
		NPN_MemFree(a);
	}

	for( int i=0; i<NUM_METHOD; ++i )
	{
		if( m_methodNameIds[i] == name ) return true;
	}
	return false;
}

bool
ScriptablePluginObject::HasProperty(NPIdentifier name)
{
	NPUTF8* a = NPN_UTF8FromIdentifier( name );
	if( a ) 
	{
		NPLOG( "ScriptablePluginObject::HasProperty() called: \n" );
		NPLOG( a );
		NPN_MemFree(a);
	}

	for( int i=0; i<NUM_PROPERTY; ++i )
	{
		if( m_propertyNameIds[i] == name ) return true;
	}
	return false;
}

bool
ScriptablePluginObject::GetProperty(NPIdentifier name, NPVariant *result)
{
	NPLOG( "Entering ScriptablePluginObject::GetProperty()\n" );
	if( ! result ) return false;

	if( name == m_propertyNameIds[PROPERTY_SEQ_SPSTATUS] )
	{
		uint32 len = (uint32)m_strSpStatus.size();
		char* res = (char*)NPN_MemAlloc( len+1 );
		if( res )
		{
			strncpy_s(res, len+1, m_strSpStatus.c_str(), m_strSpStatus.size() );
			STRINGN_TO_NPVARIANT( res, len, *result );
		}
	}
	else if( name == m_propertyNameIds[PROPERTY_SEQ_SPVERSION] )
	{
		uint32 len = (uint32)m_strSpVersion.size();
		char* res = (char*)NPN_MemAlloc( len+1 );
		if( res )
		{
			strncpy_s(res, len+1, m_strSpVersion.c_str(), m_strSpVersion.size() );
			STRINGN_TO_NPVARIANT( res, len, *result );
		}
	}
	else if( name == m_propertyNameIds[PROPERTY_SEQ_SPRESPONSE] )
	{
		uint32 len = (uint32)m_strSpResponse.size();
		char* res = (char*)NPN_MemAlloc( len+1 );
		if( res )
		{
			strncpy_s(res, len+1, m_strSpResponse.c_str(), m_strSpResponse.size() );
			STRINGN_TO_NPVARIANT( res, len, *result );
		}
	}
	else if( name == m_propertyNameIds[PROPERTY_SEQ_ERRORCODE] )
	{
		uint32 len = (uint32)m_strSpErrorCode.size();
		char* res = (char*)NPN_MemAlloc( len+1 );
		if( res )
		{
			strncpy_s(res, len+1, m_strSpErrorCode.c_str(), m_strSpErrorCode.size() );
			STRINGN_TO_NPVARIANT( res, len, *result );
		}
	}
	else
	{
		return false;
	}

	return true;
}

bool
ScriptablePluginObject::Invoke(NPIdentifier name, const NPVariant *args,
                               uint32_t argCount, NPVariant *result)
{
	NPLOG( "Entering ScriptablePluginObject::Invoke()\n" );
	if( !result ) return false;
	if( !m_pProxy )
	{
		m_strSpErrorCode = STR_ESPBP_NO_SP_SERVICE;
		BOOLEAN_TO_NPVARIANT( false, *result );
		return true;
	}

	if( name == m_methodNameIds[METHOD_SEQ_SPGETSTATUS] )
	{
		if( argCount == 0 || !args ) 
		{
			m_strSpErrorCode = STR_ESPBP_FAIL;
			BOOLEAN_TO_NPVARIANT( false, *result );
			return true;
		}

		_bstr_t bVer, bStatus;
		HRESULT hr = m_pProxy->GetSpStatusVersion( &bStatus.GetBSTR(), &bVer.GetBSTR() );
		if( SUCCEEDED(hr) )
		{
			m_strSpStatus = (char*)bStatus;
			m_strSpVersion = (char*)bVer;
			m_strSpErrorCode = STR_ESPBP_SUCCEEDED;
			BOOLEAN_TO_NPVARIANT( true, *result );
		}
		else
		{
			GetSpInternalError( m_strSpErrorCode, hr );
			BOOLEAN_TO_NPVARIANT( false, *result );
		}
	}
	else if( name == m_methodNameIds[METHOD_SEQ_SPPROCESSREQUEST] )
	{
		if( argCount < 2 || !args ) 
		{
			m_strSpErrorCode = STR_ESPBP_FAIL;
			BOOLEAN_TO_NPVARIANT( false, *result );
			return true;
		}

		_bstr_t bRequest = NPVARIANT_TO_STRING(args[0]).utf8characters;
		UINT timeout = NPVARIANT_TO_INT32(args[1]);
		_bstr_t bRes;
		HRESULT hr = m_pProxy->ProcessSpRequest( bRequest.GetBSTR(), timeout, &bRes.GetBSTR() );
		if( SUCCEEDED(hr) )
		{
			m_strSpResponse = (char*)bRes;
			m_strSpErrorCode = STR_ESPBP_SUCCEEDED;
			BOOLEAN_TO_NPVARIANT( true, *result );
		}
		else
		{
			GetSpInternalError( m_strSpErrorCode, hr );
			BOOLEAN_TO_NPVARIANT( false, *result );
		}
	}
	else
	{
		return false;
	}

	//Return false will cause javascript exception thrown!
	return true;
}

bool
ScriptablePluginObject::InvokeDefault(const NPVariant *args, uint32_t argCount,
                                      NPVariant *result)
{
	return true;
}


/////////////////////////////////////////////////////////////////////////////
// class CPlugin implementation

CPlugin::CPlugin(NPP pNPInstance) :
  m_pNPInstance(pNPInstance),
  m_bInitialized(FALSE),
  m_bInitSucceeded(false),
  m_pScriptableObject(NULL)
{
}

CPlugin::~CPlugin()
{
	if (m_pScriptableObject)
	{
		NPN_ReleaseObject(m_pScriptableObject);
	}
}

NPBool CPlugin::init(NPWindow* pNPWindow)
{
	NPLOG( "Entering CPlugin::init()\n" );

	//windowless case
	if( m_bInitialized ) return TRUE;
	
	HRESULT hr = m_svcProxy.Initialize(PLUGIN_VERSION);
	if( FAILED(hr) ) 
	{
		//should we fail early here or later? we decide to let it fail later
		//that's why we keep code structure like this.
		m_bInitialized = TRUE;
		return TRUE;
	}
	else
	{
		m_bInitSucceeded = true;
		m_bInitialized = TRUE;
		return TRUE;
	}
}

void CPlugin::shut()
{
	NPLOG( "Entering CPlugin::shut()\n" );

	//ignore the return value - we can't do anything about it
	m_svcProxy.UnInitialize();
	m_bInitialized = FALSE;
}

NPBool CPlugin::isInitialized()
{
	//LOG_FUNCSIG_SCOPED( NPTOKEN, LOG_LEVEL_TRACE );
	return m_bInitialized;
}

int16 CPlugin::handleEvent(void* event)
{
	return 0;
}

NPObject *
CPlugin::GetScriptableObject()
{
	NPLOG( "Entering CPlugin::GetScriptableObject()\n" );

	if (!m_pScriptableObject) 
	{
		m_pScriptableObject = NPN_CreateObject(m_pNPInstance,
					   GET_NPOBJECT_CLASS(ScriptablePluginObject));
		if( m_pScriptableObject )
		{
			ScriptablePluginObject* p = 
				static_cast<ScriptablePluginObject*>(m_pScriptableObject);
			
			if( m_bInitSucceeded )
			{
				p->SetTokenSvcProxy(&m_svcProxy);
			}
		}
	}

	if (m_pScriptableObject) 
	{
		NPN_RetainObject(m_pScriptableObject);
	}

	return m_pScriptableObject;
}
