/**
 * \file bmc_dev_app.c
 *
 * Description: BMC App Device
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <sys/time.h>
#include "liberic_misc.h"
#include "pp/base.h"
#include "pp/dlist.h"
#include "pp/cfg.h"
#include "pp/i2c.h"
#include "pp/um.h"
#if defined (PP_FEAT_PDU)
#include "pp/hal_rpc.h"
#endif /* PP_FEAT_PDU */

#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/session_manager.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/bmc_health.h"
#include "pp/bmc/bmc_config.h"
#include "pp/bmc/user_manager.h"
#include "pp/bmc/host_power.h"
#include "pp/bmc/lan_cipher_suites.h"
#include "pp/bmc/session_manager.h"
#if defined(PP_FEAT_IPMI_SERVER_SMI_CHAN)
#include "pp/bmc/smi_adapter.h"
#endif

#include "bmc_dev_app.h"
#include "bmc_dev_app_channel.h"
#include "bmc_dev_event_msg_buff.h"
#include "bmc_dev_sel.h"

#include "receive_message_queue.h"
#include "bmc_msg_tracker.h"
#include "event_receiver.h"
#include "nc_interrupt.h"

#include "lpc.h" // PP_BMC_SMI_MAX_MSG_LEN

#define IPMI_PRIV_NOACCESS  0x0F    /* ipmi constant only, use IPMI_PRIV_UNSPEC internally */

/*
 * Get Device ID
 */

/* Get Device Id command: field Additional Device Support */
#define IPMI_ADD_DEV_SUPP_CHASSIS         0x80
#define IPMI_ADD_DEV_SUPP_BRIDGE          0x40
#define IPMI_ADD_DEV_SUPP_IPMB_EVENT_GEN  0x20
#define IPMI_ADD_DEV_SUPP_IPMB_EVENT_RECV 0x10
#define IPMI_ADD_DEV_SUPP_FRU             0x08
#define IPMI_ADD_DEV_SUPP_SEL             0x04
#define IPMI_ADD_DEV_SUPP_SDR_REPO        0x02
#define IPMI_ADD_DEV_SUPP_SENSOR          0x01

struct get_dev_id_rs_s {
    unsigned char dev_id;
    BITFIELD3(unsigned char, dev_rev : 4, _res1 : 3, provide_dev_sdrs : 1);
    BITFIELD2(unsigned char, major_firmware_rev : 7, dev_busy : 1);
    unsigned char minor_firmware_rev;
    unsigned char ipmi_ver; /* 0x51 or 0x02 */
    unsigned char supported; /* see IPMI_ADD_DEV_SUPP_xxxx defines */
    unsigned int manufactor_id_le24 : 24; /* IANA 'Private Enterprise' ID (20bit) */
    unsigned short product_id_le16;
    unsigned int aux_firmware_rev_be32;
} __attribute__ ((packed));
typedef struct get_dev_id_rs_s get_dev_id_rs_t;

static int app_cmd_get_dev_id(imsg_t* imsg)
{
    get_dev_id_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_dev_id_rs_t));
    
    rs->dev_id = DEVICE_ID;
    rs->dev_rev = DEVICE_REV;
    rs->provide_dev_sdrs = 0;
    rs->dev_busy = 0;
    rs->major_firmware_rev = FIRMWARE_REV_MAJOR;
    rs->minor_firmware_rev = FIRMWARE_REV_MINOR;
    rs->ipmi_ver = 0x02; /* IPMI v2.0 */
    rs->supported =
	  IPMI_ADD_DEV_SUPP_SEL
	| IPMI_ADD_DEV_SUPP_FRU
	| IPMI_ADD_DEV_SUPP_SDR_REPO
	| IPMI_ADD_DEV_SUPP_SENSOR
#if defined(PP_FEAT_IPMI_SERVER_CHASSIS_CMDS)
	| IPMI_ADD_DEV_SUPP_CHASSIS
#endif
	;
    rs->manufactor_id_le24 = cpu_to_le24(MANUFACTURER_ID);
#if defined (PP_FEAT_PDU)
    int n = pp_hal_rpc_get_no_outlets();
    rs->product_id_le16 = cpu_to_le16(((n & 0xff) << 8) | (PRODUCT_ID & 0xff));
#else
    rs->product_id_le16 = cpu_to_le16(PRODUCT_ID);
#endif
    rs->aux_firmware_rev_be32 = cpu_to_be32(AUX_FIRMWARE_REV);

    return pp_bmc_router_send_msg(imsg);
}

#ifndef PP_FEAT_BMC_OEMCMDS_ONLY

/*
 * private prototypes
 */
static int send_msg_mtrack_cb(imsg_t* imsg, void* ctx);
static int send_msg_mtrack_to_cb(void* ctx);

/********************************************************************
 * App static vars
 */

/* ACPI dev pwr state (just stored here, accessed via Get/Set ACPI Power State ) */
static unsigned char acpi_dev_pwr_state = 0x2a; // unknown
/* The system power state is independend from the real system power state */
static unsigned char acpi_sys_pwr_state = 0x2a; // unknown

#endif /* PP_FEAT_BMC_OEMCMDS_ONLY */

/********************************************************************
 * App command handlers
 */

/*
 * Cold Reset
 */

static int app_cmd_cold_reset(imsg_t* imsg)
{
    pp_bmc_log_info("[APP] Cold Reset triggered");
#ifdef LARA_KIMMSI
  /* MSI: trigger_board_reset will cause the host to reset as well
   * (due to fpga reset -> power loss)
   * Thus the KIM reset is replaced by a restart of eric in the
   * moment.
   * FIXME: is this fixed in thge meanwhile?
   */
    pp_bmc_router_resp_err(imsg,IPMI_ERR_SUCCESS);
#if NDEBUG
    exit(0);
#else
    /* In devel fw (NFS), eric process is not automatically restarted.
    * So exit(0) will not work. We make a warm reset here instead...
    */
    pp_bmc_router_channel_reset();
#endif
#endif // LARA_KIMMSI

    unsigned char cc;
    switch (eric_misc_trigger_board_reset(0)) {
        case  0: cc = IPMI_ERR_SUCCESS; break;
        case -1: cc = IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL; break;
        case -2: cc = IPMI_ERR_DEV_IN_FIRMWARE_UPDATE_MODE; break;
        default: cc = IPMI_ERR_UNSPECIFIED;
    }

    /*  may never be reached (accepted by IPMI spec) */
    return pp_bmc_router_resp_err(imsg, cc);
}

/*
 * Warm Reset
 */

static int app_cmd_warm_reset(imsg_t* imsg)
{
    pp_bmc_router_channel_reset();

    pp_bmc_session_manager_close_all_sessions();
   
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

#ifndef PP_FEAT_BMC_OEMCMDS_ONLY

/*
 * Get Self Test Results
 */

struct get_self_test_res_rs_s {
    unsigned char res;
    unsigned char reason;
} __attribute__ ((packed));
typedef struct get_self_test_res_rs_s get_self_test_res_rs_t;

static int app_cmd_get_self_test_res(imsg_t* imsg)
{
    int i;

    get_self_test_res_rs_t *rs =
        pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_self_test_res_rs_t));
    
    if (pp_bmc_health_check() == PP_SUC) {
        rs->res = 0x55;
        rs->reason = 0;
    } else {
        // okay, we have errors, let's see if we can output them in detail

        // find first error
        for (i=0; i<BMC_HEALTH_MAX_INDEX; i++) {
            if (pp_bmc_health_get_fault(i) == 1)
                break;
        } 
        
        // standard parts error, format specified in spec
        if (i<8) {
            rs->res = 0x57;
            rs->reason = pp_bmc_health_get_fault(0);
            for (i=1; i<8; i++) {
                rs->reason = rs->reason << 1;
                rs->reason = pp_bmc_health_get_fault(i);
            }
        } else {
            // nonstandard error
            rs->res = 0x58;
            rs->reason = i;  // set bitfield to index of first error
        }
    }
    
    return pp_bmc_router_send_msg(imsg);
}

void pp_bmc_set_sys_acpi_pwr_state(unsigned char power_state) {
    acpi_sys_pwr_state = power_state;
}

static int app_cmd_set_acpi_power_state(imsg_t* imsg)
{
    int new_acpi_sys_pwr_state;
    unsigned char* rq = imsg->data;
    pp_bmc_log_info("[APP] Set ACPI Power State (sys=0x%02x dev=0x%02x)", rq[0], rq[1]);
    
    if (rq[0] & 0x80) {
        new_acpi_sys_pwr_state = rq[0] & 0x7f;
        /* command acpi encoding is different than host_power acpi encoding,
	 * convert */
        if (new_acpi_sys_pwr_state == 0x20) {
	    new_acpi_sys_pwr_state = HOST_POWER_STATE_LEGON;
	} else if (new_acpi_sys_pwr_state == 0x21) {
	    new_acpi_sys_pwr_state=HOST_POWER_STATE_LEGOFF;
	} else if (new_acpi_sys_pwr_state == 0x2a) {
	    new_acpi_sys_pwr_state=HOST_POWER_STATE_UNKNOWN;
	}
    
	pp_bmc_set_sys_acpi_pwr_state(new_acpi_sys_pwr_state);
    }
    if (rq[1] & 0x80) acpi_dev_pwr_state = rq[1] & 0x7f;
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int app_cmd_get_acpi_power_state(imsg_t* imsg)
{
    unsigned char* rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 2);
    
    int sys_pwr_state;
    sys_pwr_state = acpi_sys_pwr_state;
    /* convert acpi_sys_pwr_state encoding */
    if (sys_pwr_state == HOST_POWER_STATE_LEGON) sys_pwr_state = 0x20;
    if (sys_pwr_state == HOST_POWER_STATE_LEGOFF) sys_pwr_state = 0x21;
    if (sys_pwr_state == HOST_POWER_STATE_UNKNOWN) sys_pwr_state = 0x2a;
    rs[0] = sys_pwr_state;
    rs[1] = acpi_dev_pwr_state;
    
    pp_bmc_log_info("[APP] Get ACPI Power State => sys=0x%02x dev=0x%02x", rs[0], rs[1]);
    return pp_bmc_router_send_msg(imsg);
}

static int app_cmd_get_device_guid(imsg_t* imsg) {
    unsigned char* guid;
    if (imsg->data_size != 0) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    }
    
    guid = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 16);
    
    pp_bmc_get_guid(guid);
    
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set BMC Global Enables
 */

#define IPMI_ENABLE_OEM_2                   0x80
#define IPMI_ENABLE_OEM_1                   0x40
#define IPMI_ENABLE_OEM_0                   0x20
#define IPMI_ENABLE_SEL                     0x08
#define IPMI_ENABLE_EVENT_BUF               0x04
#define IPMI_ENABLE_EVENT_MSG_BUF_FULL_INTR 0x02
#define IPMI_ENABLE_RECV_MSG_QUEUE_INTR     0x01

static int app_cmd_set_bmc_global_enables(imsg_t* imsg)
{
    unsigned char b;

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    /* only allowed from SI; do allow it, if we have no SMI */
    if (imsg->chan != IPMI_CHAN_SI) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
    }
#endif
    
    b = (imsg->data[0] & IPMI_ENABLE_SEL) ? 1:0;
    bmc_dev_sel_set_enable(b);

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    b = (imsg->data[0] & IPMI_ENABLE_EVENT_BUF) ? 1:0;
    bmc_dev_event_msg_buf_enable_set(b);
    
    b = (imsg->data[0] & IPMI_ENABLE_EVENT_MSG_BUF_FULL_INTR) ? 1:0;
    bmc_interrupt_enable_set(BMC_INTERRUPT_EVENT_MSG_BUFFER, b);
    
    b = (imsg->data[0] & IPMI_ENABLE_RECV_MSG_QUEUE_INTR) ? 1:0;
    bmc_interrupt_enable_set(BMC_INTERRUPT_RECEIVE_MSG_QUEUE, b);
#endif
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get BMC Global Enables
 */
static int app_cmd_get_bmc_global_enables(imsg_t* imsg)
{
    unsigned char* buf;
    
    buf = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    *buf = 0;

    if (bmc_dev_sel_get_enable()) {
        *buf |= IPMI_ENABLE_SEL;
    }

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    if (bmc_dev_event_msg_buf_enable_get()) {
        *buf |= IPMI_ENABLE_EVENT_BUF;
    }

    if (bmc_interrupt_enable_get(BMC_INTERRUPT_EVENT_MSG_BUFFER)) {
        *buf |= IPMI_ENABLE_EVENT_MSG_BUF_FULL_INTR;
    }

    if (bmc_interrupt_enable_get(BMC_INTERRUPT_RECEIVE_MSG_QUEUE)) {
        *buf |= IPMI_ENABLE_RECV_MSG_QUEUE_INTR;
    }
#endif

    return pp_bmc_router_send_msg(imsg);
}

#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
/*
 * Clear Message Flags
 */
#define IPMI_MSG_FLAGS_OEM2             0x80
#define IPMI_MSG_FLAGS_OEM1             0x40
#define IPMI_MSG_FLAGS_OEM0             0x20
#define IPMI_MSG_FLAGS_WATCHDOG_INTR    0x08
#define IPMI_MSG_FLAGS_EVENT_MSG_BUF    0x02
#define IPMI_MSG_FLAGS_RECV_MSG_QUEUE   0x01

static int app_cmd_clear_msg_flags(imsg_t* imsg)
{
    if (imsg->data[0] & IPMI_MSG_FLAGS_WATCHDOG_INTR) {
        bmc_interrupt_flag_set(BMC_INTERRUPT_WATCHDOG_PRETIMEOUT, 0);  /* clear flag */
    }
    if (imsg->data[0] & IPMI_MSG_FLAGS_EVENT_MSG_BUF) bmc_dev_event_msg_buf_clear(); /* clear buffer */
    if (imsg->data[0] & IPMI_MSG_FLAGS_RECV_MSG_QUEUE) bmc_receive_queue_clear();    /* clear queue */

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get Message Flags
 */

static int app_cmd_get_msg_flags(imsg_t* imsg)
{
    unsigned char flags = 0;
    
    if (bmc_interrupt_flag_get(BMC_INTERRUPT_RECEIVE_MSG_QUEUE)) {
        flags |= IPMI_MSG_FLAGS_RECV_MSG_QUEUE;
    }
    if (bmc_interrupt_flag_get(BMC_INTERRUPT_EVENT_MSG_BUFFER)) {
        flags |= IPMI_MSG_FLAGS_EVENT_MSG_BUF;
    }
    if (bmc_interrupt_flag_get(BMC_INTERRUPT_WATCHDOG_PRETIMEOUT)) {
        flags |= IPMI_MSG_FLAGS_WATCHDOG_INTR;
    }
    
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &flags, sizeof(flags));
}

/*
 * Get Message
 */
struct get_msg_rs_s {
    unsigned char chan;
    ipmi_msg_hdr_t hdr;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct get_msg_rs_s get_msg_rs_t;

/* all this message bridging stuff has is absolutely unchecked */
static int app_cmd_get_msg(imsg_t* imsg)
{
    imsg_t *msg;
    get_msg_rs_t *rs;

    msg = bmc_receive_queue_get();
    
    if (msg == NULL) {
        /* receive message queue is empty */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_DATA_NOT_AVAIL);
    }

    /* create response */
    /** Exact structure of response is defined in Table 22-8, p266 **/
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                   sizeof(get_msg_rs_t) + msg->data_size + 1 /* csum */);

    rs->chan = msg->chan;

    rs->hdr.rs_addr = msg->rs_addr;
    rs->hdr.rs_lun  = msg->rs_lun;
    rs->hdr.netfn   = msg->netfn;
    rs->hdr.chk1    = pp_bmc_calc_ipmi_checksum(&rs->hdr.rs_addr, 0, 1); /* csum for first 2 bytes */
    rs->hdr.rq_addr = msg->rq_addr;
    rs->hdr.rq_lun  = msg->rq_lun;
    rs->hdr.rq_seq  = msg->rq_seq;
    rs->hdr.cmd     = msg->cmd;

    memcpy(rs->data, msg->data, msg->data_size);

    rs->data[msg->data_size] = pp_bmc_calc_ipmi_checksum(&rs->hdr.rq_addr, 0, 2 + msg->data_size); /* csum for rest */

    return pp_bmc_router_send_msg(imsg);
}
#endif /* PP_FEAT_IPMI_SERVER_SMI_CHAN */

/*
 * Send Message
 */
/* Send Message command: bridging mechanism */ 
#define IPMI_BRIDGING_LUN10B 0x0
#define IPMI_BRIDGING_TRACK  0x1
#define IPMI_BRIDGING_RAW    0x2

struct send_msg_rq_s {
    BITFIELD3(unsigned char, chan : 4, _res1 : 2, bridging_type : 2);
    ipmi_msg_hdr_t hdr;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct send_msg_rq_s send_msg_rq_t;

static int app_cmd_send_msg(imsg_t* imsg)
{
    /* ATTENTION: works for into IPMB channels only!!
     * In future implementations please do not forget to:                 *
     * - check if the target channel exists before bridging to            *
     * - prevent bridging to the loopi channel (invisible but routeable)  *
     * - do not forget to adapt the message callback buffer to handle     *
     *   session ids correctly (see below: send_msg_mtrack_cb() )         */

    send_msg_rq_t *rq = (void*)imsg->data;
    int chan = rq->chan;
    int btype = rq->bridging_type;
    unsigned int msg_size;
    imsg_t *msg;

    /* check for channel existence and implementation */
    if (pp_bmc_router_get_chan_adapter_info(chan) == NULL ||
	(chan != IPMI_CHAN_PRIMARY_IPMB && chan != IPMI_CHAN_SECDARY_IPMB)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_DEST_UNAVAILABLE);
    }
	
    /* special privilege reqs for all channels other than SI: OPERATOR */
    if (chan != IPMI_CHAN_SI && imsg->priv_level < IPMI_PRIV_OPERATOR) {
        return pp_bmc_router_resp_err(imsg,
                                      IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL);
    }

    /* raw bridging not supported */
    if (btype != IPMI_BRIDGING_LUN10B && btype != IPMI_BRIDGING_TRACK) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
        
    /* create new message */
    msg_size = imsg->data_size - sizeof(send_msg_rq_t)  - 1 /* final csum */;
    msg = pp_bmc_imsg_new(msg_size);

    /* fill message from Send Message command */
    msg->chan = rq->chan;

    msg->rs_addr    = rq->hdr.rs_addr;
    msg->rs_lun     = rq->hdr.rs_lun;
    msg->netfn      = rq->hdr.netfn;
    msg->rq_addr    = rq->hdr.rq_addr;
    msg->rq_lun     = rq->hdr.rq_lun;
    msg->rq_seq     = rq->hdr.rq_seq;
    msg->cmd        = rq->hdr.cmd;

    memcpy(msg->data, rq->data, msg_size);
    
    /* handle bridging type */
    if (btype == IPMI_BRIDGING_LUN10B) {
        if (IPMI_IS_REQUEST(msg->netfn)) msg->rq_lun = 2; /* LUN 10b */
        else msg->rs_lun = 2;
        if (PP_ERR == pp_bmc_router_send_msg(msg)) {
            pp_bmc_log_error("[APP]:send_msg:pp_bmc_router_send_msg");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    } else { /* btype == IPMI_BRIDGING_TRACK */
        // is is really okay to copy "shallow" here ?
        imsg_t* cmsg = pp_bmc_imsg_copy(imsg, IMSG_COPY_SHALLOW);
        if (PP_ERR == pp_bmc_mtrack_send_msg(msg, send_msg_mtrack_cb,
                                             send_msg_mtrack_to_cb, cmsg)) {
            pp_bmc_log_error("[APP]:send_msg:pp_bmc_mtrack_send_msg");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    }

    /* send response to Send Message unconditionally (spec v2: p60, p62) */
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int send_msg_mtrack_cb(imsg_t* imsg, void* ctx) {
    /**
     * ATTENTION:
     * This function has not been adapted to the new session
     * management yet. Sessions are not handled and messages
     * might not be returned correctly. Still untested.
     */
    
    imsg_t* smsg = (imsg_t*) ctx;

    assert(IPMI_IS_RESPONSE(imsg->netfn));
        
    /* reformat msg so that it looks like coming from bmc */
    imsg->rs_addr = smsg->rs_addr;
    imsg->rs_lun  = smsg->rs_lun;
    imsg->rq_seq  = smsg->rq_seq;
    
    /* resend to same channel it came from */
    imsg->chan       = smsg->chan;
    imsg->priv_level = smsg->priv_level;
    imsg->session    = pp_bmc_session_dup(smsg->session);

    pp_bmc_imsg_delete(smsg);
    return pp_bmc_router_send_msg(imsg);
}

static int send_msg_mtrack_to_cb(void* ctx) {
    imsg_t* imsg = (imsg_t*) ctx;
    pp_bmc_log_error("[APP]:send_msg: cmd response timeout");
    pp_bmc_imsg_dump(PP_BMC_LOG_DEBUG, "[APP]:orig msg was:", imsg);
    pp_bmc_imsg_delete(imsg);
    return PP_SUC;
}
/********************************************************************/


#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
/*
 * Get System Interface Capabilities
 */
struct get_system_interface_capabilities_rq_s {
    unsigned char system_interface_type;
} __attribute__ ((packed));
typedef struct get_system_interface_capabilities_rq_s get_system_interface_capabilities_rq_t;

struct get_cap_ssif_s {
    unsigned char transaction_support;
    unsigned char input_message_size;
    unsigned char output_message_size;
} __attribute__ ((packed));
typedef struct get_cap_ssif_s get_cap_ssif_t;

struct get_cap_kcs_s {
    unsigned char system_interface_version;
    unsigned char input_message_size;
} __attribute__ ((packed));
typedef struct get_cap_kcs_s get_cap_kcs_t;

struct get_system_interface_capabilities_rs_s {
    unsigned char reserved;
    union {
    	get_cap_ssif_t ssif;
	get_cap_kcs_t kcs;
    } __attribute__ ((packed));
} __attribute__ ((packed));
typedef struct get_system_interface_capabilities_rs_s get_system_interface_capabilities_rs_t;

#define INTERFACE_TYPE_SSIF 0x00
#define INTERFACE_TYPE_KCS  0x01
#define INTERFACE_TYPE_SMIC 0x02

static int app_cmd_get_sys_interface_caps(imsg_t *imsg)
{
    get_system_interface_capabilities_rq_t *req = (void *)imsg->data;
    get_system_interface_capabilities_rs_t *rsp;

    switch (req->system_interface_type) {
    case INTERFACE_TYPE_KCS:
        rsp = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                               sizeof(unsigned char) + sizeof(get_cap_kcs_t)); // incl "reserved"
        rsp->kcs.system_interface_version = 0;
        rsp->kcs.input_message_size = (unsigned char)min(255, PP_BMC_SMI_MAX_MSG_LEN);
        break;
    case INTERFACE_TYPE_SMIC:
        rsp = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(unsigned char));
        break;
    case INTERFACE_TYPE_SSIF:
        rsp = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                               sizeof(unsigned char) + sizeof(get_cap_ssif_t)); // incl "reserved"
        rsp->ssif.transaction_support = 0;
        rsp->ssif.input_message_size = 32;
        rsp->ssif.output_message_size = 32;
        break;
    default:
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    rsp->reserved = 0;
    return pp_bmc_router_send_msg(imsg);
}


/*
 * Get BT Interface Capabilities
 */
struct get_bt_interface_caps_rs_s {
    unsigned char supp_rq_queue_size;
    unsigned char in_buf_size; // for requests
    unsigned char out_buf_size; // for responses
    unsigned char bmc_rq_to_rs_time; // in seconds, max. 30
    unsigned char recommended_retries;
} __attribute__ ((packed));
typedef struct get_bt_interface_caps_rs_s get_bt_interface_caps_rs_t;

static int app_cmd_get_bt_interface_caps(imsg_t *imsg)
{
    get_bt_interface_caps_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_bt_interface_caps_rs_t));
    rs->supp_rq_queue_size = 100;
    rs->in_buf_size = 64;
    rs->out_buf_size = 64;
    rs->bmc_rq_to_rs_time = 1; // we are fast :)
    rs->recommended_retries = 1; // ...and reliable
    return pp_bmc_router_send_msg(imsg);
}
#endif /* PP_FEAT_IPMI_SERVER_SMI_CHAN */

#ifdef PRODUCT_SMIDC
static int app_cmd_get_system_guid(imsg_t *imsg)
{
    char *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 16);
    // use device guid for system guid. this is not entirely correct
    // but system guid can be queried without a session
    pp_bmc_get_guid(rs);
    return pp_bmc_router_send_msg(imsg);
}
#endif

/*
 * Master Write-Read
 */
struct master_read_write_rq_s {
    BITFIELD3(unsigned char, private_bus : 1, bus_id : 3, chan : 4);
    BITFIELD2(unsigned char, : 1, slave_addr : 7);
    unsigned char read_count;
    unsigned char write_data[0];
} __attribute__ ((packed));
typedef struct master_read_write_rq_s master_read_write_rq_t;

#define IPMI_ERR_MWR_LOST_ARBIT 0x81
#define IPMI_ERR_MWR_BUS_ERR    0x82
#define IPMI_ERR_MWR_NAK        0x83
#define IPMI_ERR_MWR_TRUNC      0x84

/* attention, this doesn't include IPMB, so far */
static int app_cmd_master_write_read(imsg_t *imsg)
{
    master_read_write_rq_t *rq = (void*)imsg->data;
    unsigned char *rs;
    int write_count = imsg->data_size - sizeof(master_read_write_rq_t);
    int slave_addr = rq->slave_addr;
    int read_count = rq->read_count;
    int read_len = 0;
    int err = 0, ret = IPMI_ERR_SUCCESS;
    int i2c;

    if (!rq->private_bus)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);

#ifdef PRODUCT_SMIDC
    // only i2c-1 is accessible to public, mapped to IPMI bus-id 2
    if (rq->bus_id != 2) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    rq->bus_id = 1;
#endif

    // open device
    i2c = pp_i2c_open_bus(rq->bus_id, &err);
    if (err != PP_I2C_NO_ERROR) {
        ret = IPMI_ERR_INVALID_DATA_FIELD;
        goto exit;
    }

    // lock device over write _and_ read
    // (e.g. against sensor scanner thread)
    pp_i2c_lock(i2c);

    // write data, if required
    if (write_count > 0) {
	pp_bmc_log_debug("[APP] MWR: write to %s, addr=%#x, cnt=%d",
			 pp_i2c_dev_by_hndl(i2c), slave_addr, write_count);
        pp_i2c_tx_burst_core(i2c, slave_addr, rq->write_data, write_count,
			     &err);
        if (err != PP_I2C_NO_ERROR) {
            ret = IPMI_ERR_MWR_NAK;
        }
    }

    // prepare respone msg
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, read_count);

    // read data, if required
    if (ret == IPMI_ERR_SUCCESS && read_count > 0) {
        read_len = pp_i2c_rx_burst_core(i2c, slave_addr, rs, read_count, &err);
        if (err != PP_I2C_NO_ERROR) {
            read_len = 0;
            ret = IPMI_ERR_MWR_BUS_ERR;
        } else if (read_len < read_count) {
            ret = IPMI_ERR_MWR_TRUNC;
        }
	pp_bmc_log_debug("[APP] MWR: read from %s, addr=%#x, cnt=%d, rcvd=%d",
			 pp_i2c_dev_by_hndl(i2c), slave_addr, read_count,
			 read_len);
    }

    // unlock and close device
    pp_i2c_unlock(i2c);
    pp_i2c_close(i2c);

exit:
    // correct completion code and response data len
    imsg->data[0] = ret;
    imsg->data_size = 1 + read_len;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Cipher Suites
 */

struct get_channel_cipher_suites_rq_s {
    unsigned char channel;
    unsigned char payload_type;
    unsigned char index;
} __attribute__ ((packed));
typedef struct get_channel_cipher_suites_rq_s get_channel_cipher_suites_rq_t;

struct get_channel_cipher_suites_rs_s {
    unsigned char channel;
    unsigned char data_block[16];
} __attribute__ ((packed));
typedef struct get_channel_cipher_suites_rs_s get_channel_cipher_suites_rs_t;

static int app_cmd_get_channel_cipher_suites(imsg_t *imsg)
{
    get_channel_cipher_suites_rq_t *rq = (void*)imsg->data;
    get_channel_cipher_suites_rs_t *rs;
    unsigned char idx = rq->index;
    unsigned char channel = rq->channel;

    if ( (rq->payload_type != IPMI_PAYLOAD_IPMI) &&
	 (rq->payload_type != IPMI_PAYLOAD_SOL) )
    {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }

    if (channel != IPMI_CHAN_LAN) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (!(idx & 0x80)) {
	int n = 0;
	// just list algorithms
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_channel_cipher_suites_rs_t));
	rs->channel = channel;
	
	rs->data_block[n++] = IPMI20_AUTH_NONE;
	rs->data_block[n++] = IPMI20_AUTH_HMAC_SHA1;
	rs->data_block[n++] = IPMI20_AUTH_HMAC_MD5;
	
	rs->data_block[n++] = 0x40 | IPMI20_INTEGRITY_NONE;
	rs->data_block[n++] = 0x40 | IPMI20_INTEGRITY_HMAC_SHA1_96;
	rs->data_block[n++] = 0x40 | IPMI20_INTEGRITY_HMAC_MD5_128;
	rs->data_block[n++] = 0x40 | IPMI20_INTEGRITY_MD5_128;
	
	rs->data_block[n++] = 0x80 | IPMI20_CONF_NONE;
	rs->data_block[n++] = 0x80 | IPMI20_CONF_AES_CBC_128;

	imsg->data_size -= 16 - n; // reduce msg len by amount of unsused cipher slots
    }
    else
    {
        // list cipher suites
        int req_data_index;
        int avail_data_size;
        int data_length;
        
	idx &= 0x7f;
        req_data_index = idx * 16;
        avail_data_size = PP_BMC_LAN_CIPHER_SUITE_COUNT * sizeof(pp_cipher_suite_record_t);
        data_length = avail_data_size - req_data_index;
        
        if (data_length < 0) data_length = 0;
        if (data_length > 16) data_length = 16;
        
	rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1 + data_length); 
	rs->channel = channel;
	memcpy(rs->data_block, ((unsigned char*)pp_bmc_lan_enabled_cipher_suites)+req_data_index, data_length);
    }
    
    return pp_bmc_router_send_msg(imsg);
}

#endif // !PP_FEAT_BMC_OEMCMDS_ONLY

/*
 * Set User Access
 */

struct set_user_access_rq_s {
    BITFIELD5(unsigned char,
        chan_id : 4,
        messaging_en : 1,
        link_auth_en : 1,
        callback_only : 1,
        bits_valid : 1);
    BITFIELD2(unsigned char, uid : 6, : 2);
    unsigned char priv_limit;
    unsigned char sess_limit;
} __attribute__ ((packed));
typedef struct set_user_access_rq_s set_user_access_rq_t;

static int app_cmd_set_user_access(imsg_t *imsg)
{
    set_user_access_rq_t *rq = (void*)imsg->data;
    unsigned char priv_limit = rq->priv_limit & 0x0f;
    int channel = rq->chan_id;
    int result = IPMI_ERR_UNSPECIFIED;

    pp_bmc_log_info("[APP] Set User Access (uid=%d, channel=%d, priv=%d)", rq->uid, rq->chan_id, priv_limit);

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }

    if (priv_limit == IPMI_PRIV_NOACCESS) {
	priv_limit = IPMI_PRIV_UNSPEC;
    } else if (priv_limit > IPMI_PRIV_OEM) {
	result = IPMI_ERR_PARAM_OUT_OF_RANGE;
	goto bail;
    }

    if (rq->bits_valid) {
	if (PP_FAILED( pp_bmc_user_set_priv_level_and_flags(rq->uid, channel,
			priv_limit, rq->callback_only, rq->link_auth_en, rq->messaging_en) )) {
	    pp_bmc_log_warn("[APP] Failed to set access %d for user %d on channel %d", priv_limit, rq->uid, channel);
	    goto bail;
	}
    } else {
        if (PP_ERR == pp_bmc_user_set_priv_level(rq->uid, channel, priv_limit) ) {
            pp_bmc_log_warn("[APP] Failed to set access %d for user %d on channel %d", priv_limit, rq->uid, channel);
            goto bail;
        }
    }

    result = IPMI_ERR_SUCCESS;

bail:
    return pp_bmc_router_resp_err(imsg, result);
}

/*
 * Get User Access
 */

struct get_user_access_rq_s {
    BITFIELD2(unsigned char, channel : 4, : 4);
    BITFIELD2(unsigned char, uid : 6, : 2);
} __attribute__ ((packed));
typedef struct get_user_access_rq_s get_user_access_rq_t;

struct get_user_access_rs_s {
    BITFIELD2(unsigned char, max_uid : 6, : 2);
    BITFIELD2(unsigned char, no_of_enabled_uids : 6, enable_status : 2);
    BITFIELD2(unsigned char, no_of_fixname_uids : 6, : 2);
    BITFIELD5(unsigned char,
        priv_limit : 4,
        messaging_en : 1,
        link_auth_en : 1,
        callback_only : 1,
        res0 : 1);
} __attribute__ ((packed));
typedef struct get_user_access_rs_s get_user_access_rs_t;

static int app_cmd_get_user_access(imsg_t *imsg)
{
    get_user_access_rq_t *rq = (void*)imsg->data;
    int channel = rq->channel;
    int uid = rq->uid;
    int priv, enabled, callback_only, link_auth, msg_enabled;

    if (channel == IPMI_CHAN_PRESENT) {
	channel = imsg->chan;
    }

    priv = pp_bmc_user_get_priv_level(uid, channel);
    enabled = pp_bmc_user_get_enable(uid);
    callback_only = pp_bmc_user_callback_only(uid, channel);
    link_auth = pp_bmc_user_link_auth(uid, channel);
    msg_enabled = pp_bmc_user_msg_enabled(uid, channel);
    get_user_access_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_user_access_rs_t));

    // fill props of user management
    rs->max_uid = pp_bmc_user_get_max_uid(); // UID 1..63
    rs->no_of_enabled_uids = pp_bmc_user_count_enabled(channel);
    rs->no_of_fixname_uids = 1; // UID 1 has fixed name (anonymous)

    // fill props of this user
    rs->enable_status = enabled? 1 : 2; // see IPMI 2.0 errata 3
    rs->priv_limit = (priv != PP_ERR) ? (priv & 0x0f) : IPMI_PRIV_UNSPEC;
    // TODO: combine with chan priv settings
    if (rs->priv_limit == IPMI_PRIV_UNSPEC) rs->priv_limit = IPMI_PRIV_NOACCESS;
    
    rs->callback_only = callback_only;
    rs->link_auth_en = link_auth;
    rs->messaging_en = msg_enabled;

    pp_bmc_log_info("[APP] Get User Access (uid=%d, channel=%d) => priv=%d", uid, channel, rs->priv_limit);

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set User Name
 */

struct set_user_name_rq_s {
    unsigned char uid;
    char name[16];
} __attribute__ ((packed));
typedef struct set_user_name_rq_s set_user_name_rq_t;

static int app_cmd_set_user_name(imsg_t *imsg)
{
    set_user_name_rq_t *rq = (void*)imsg->data;

    // create a zero-terminated string
    char name[17];
    strncpy(name, rq->name, 16);
    name[16] = '\0';

    pp_bmc_log_info("[APP] Set User Name (uid=%d, name=%s)", rq->uid & 0x3f, name);

    if (PP_ERR == pp_bmc_user_set_name(rq->uid & 0x3f, name)) {
	if (errno == PP_UM_ERR_USER_ALREADY_EXISTS) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
	} else {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get User Name
 */

static int app_cmd_get_user_name(imsg_t *imsg)
{
    char name[17] = "";
    unsigned char *rs;

    /* on error: user name stays empty */
    pp_bmc_user_get_name(imsg->data[0] & 0x3f, name, sizeof(name));

    pp_bmc_log_info("[APP] Get User Name (uid=%d) => name=%s", imsg->data[0] & 0x3f, name);


    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 16);
    memset(rs, 0, 16);
    if (strncmp(name, "_ipmi_", 6) != 0) {
	strncpy(rs, name, 16);
    }

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set User Password
 */

#define IPMI_DISABLE_USER   0
#define IPMI_ENABLE_USER    1
#define IPMI_SET_PWD        2
#define IPMI_TEST_PWD       3

struct set_user_password_rq_s {
    BITFIELD2(unsigned char, uid : 6, _rsv1 : 2);
    BITFIELD2(unsigned char, op : 2, _rsv2 : 6);
    char password[0];
} __attribute__ ((packed));
typedef struct set_user_password_rq_s set_user_password_rq_t;

static int app_cmd_set_user_password(imsg_t *imsg)
{
    set_user_password_rq_t *rq = (void*)imsg->data;
    char passwd[21], usrpasswd[64];

    if (rq->op == IPMI_SET_PWD || rq->op == IPMI_TEST_PWD) {
	int sz = imsg->data_size - sizeof(set_user_password_rq_t);
	if (sz != 16 && sz != 20) {
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
	}
	strncpy(passwd, rq->password, sz);
	passwd[sz] = '\0';
	pp_bmc_log_info("[APP] Set User Password (op= %d, uid=%d, passwd=%s)", rq->op, rq->uid, passwd);
    } else {
	pp_bmc_log_info("[APP] Set User Password (op= %d, uid=%d)", rq->op, rq->uid);
    }

    switch (rq->op) {
        case IPMI_DISABLE_USER:
            if (PP_ERR == pp_bmc_user_set_enable(rq->uid, 0)) {
                pp_bmc_log_warn("[APP] Failed to disable user %d", rq->uid);
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
            }
            break;
        case IPMI_ENABLE_USER:
            if (PP_ERR == pp_bmc_user_set_enable(rq->uid, 1)) {
                pp_bmc_log_warn("[APP] Failed to enable user %d", rq->uid);
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
            }
            break;
        case IPMI_SET_PWD:
            if (PP_ERR == pp_bmc_user_set_password(rq->uid, passwd)) {
                pp_bmc_log_warn("[APP] Failed to set password '%s' for user %d", passwd, rq->uid);
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
            }
            break;
        case IPMI_TEST_PWD:
            if (PP_ERR == pp_bmc_user_get_password(rq->uid, usrpasswd, 64)) {
                pp_bmc_log_warn("[APP] Failed to check password of user %d", rq->uid);
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
            }
            if (strcmp(passwd, usrpasswd) != 0)
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_BAD_PASSWD);
            break;
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/********************************************************************
 * App device command table
 */

static const dev_cmd_entry_t app_cmd_tab[] = {
    {
        .cmd_hndlr = app_cmd_get_dev_id,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_DEV_ID,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_cold_reset,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_COLD_RESET,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = app_cmd_warm_reset,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_WARM_RESET,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_ADMIN
    },
#ifndef PP_FEAT_BMC_OEMCMDS_ONLY
    {
        .cmd_hndlr = app_cmd_get_self_test_res,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_SELF_TEST_RES,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_set_acpi_power_state,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SET_ACPI_POWER_STATE,
        .min_data_size = 2, .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = app_cmd_get_acpi_power_state,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_ACPI_POWER_STATE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_get_device_guid,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_DEVICE_GUID,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        // can be issued from SI only
        .cmd_hndlr = app_cmd_set_bmc_global_enables,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SET_BMC_GLOBAL_ENABLES,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_UNSPEC
    },
    {
        .cmd_hndlr = app_cmd_get_bmc_global_enables,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_BMC_GLOBAL_ENABLES,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    {
        .cmd_hndlr = app_cmd_clear_msg_flags,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_CLEAR_MSG_FLAGS,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_get_msg_flags,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_MSG_FLAGS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_get_msg,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_MSG,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
#endif
    {
        .cmd_hndlr = app_cmd_send_msg,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SEND_MSG,
        .min_data_size = 1 + sizeof(ipmi_msg_hdr_t) + 1, .min_priv_level = IPMI_PRIV_USER
    },
#if defined (PP_FEAT_IPMI_SERVER_SMI_CHAN)
    {
	.cmd_hndlr = app_cmd_get_sys_interface_caps,
	.netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_SYSTEM_INTERFACE_CAPABILITIES,
	.min_data_size = 1, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = app_cmd_get_bt_interface_caps,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_BT_INTERFACE_CAPS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
#endif
#ifdef PRODUCT_SMIDC
    {
        .cmd_hndlr = app_cmd_get_system_guid,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_SYSTEM_GUID,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_CALLBACK
    },
#endif
    {
        .cmd_hndlr = app_cmd_master_write_read,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_MASTER_WRITE_READ,
        .min_data_size = sizeof(master_read_write_rq_t), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
	.cmd_hndlr = app_cmd_get_channel_cipher_suites,
	.netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_CHANNEL_CIPHER_SUITES,
	.min_data_size = sizeof(get_channel_cipher_suites_rq_t), .min_priv_level = IPMI_PRIV_UNSPEC
    },
#endif // !PP_FEAT_BMC_OEMCMDS_ONLY
    {
        .cmd_hndlr = app_cmd_set_user_access,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SET_USER_ACCESS,
        .min_data_size = sizeof(set_user_access_rq_t) - 1,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = app_cmd_get_user_access,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_USER_ACCESS,
        .min_data_size = 2, .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = app_cmd_set_user_name,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SET_USER_NAME,
        .min_data_size = sizeof(set_user_name_rq_t), .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = app_cmd_get_user_name,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_GET_USER_NAME,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = app_cmd_set_user_password,
        .netfn = IPMI_NETFN_APP, .cmd = IPMI_CMD_SET_USER_PASSWORD,
        .min_data_size = sizeof(set_user_password_rq_t), .min_priv_level = IPMI_PRIV_ADMIN
    },

    { .cmd_hndlr = NULL }
};

/********************************************************************
 * App device c'tor/d'tor
 */

int pp_bmc_dev_app_init()
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(app_cmd_tab)) return PP_ERR;
    
    /* init channel section */
    if (PP_ERR == pp_bmc_dev_app_channel_init()) return PP_ERR;
    
    pp_bmc_log_info("[APP] device started");
    return PP_SUC;
}

void pp_bmc_dev_app_cleanup()
{
    /* clean up channel section */
    pp_bmc_dev_app_channel_cleanup();
    
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(app_cmd_tab);

    pp_bmc_log_info("[APP] device shut down");
}
