/**
 * bmc_dev_app_channel.c
 *
 * Channel Commands for BMC App Device Header.
 * Implements the channel configuration ipmi commands and stores
 * volatile and non volatile channel configuration data. New
 * volatile configurations are automatically pushed to the channels
 * through the bmc_router.
 *
 * (c) 2005 Peppercon AG, Thomas Weber <tweb@peppercon.de>
 */

#include "pp/base.h"
#include "pp/cfg.h"

#include "pp/bmc/debug.h"

#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"

#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/session_manager.h"

#include "bmc_dev_app_channel.h"

#define IPMI_PRIV_NOACCESS  0x0F    /* ipmi constant only, use IPMI_PRIV_UNSPEC internally */

/* 
 * Write a channel configuration to the config system.
 * The configuration is written unconditionally. Callers
 * must make sure to write only configuration data for 
 * existing ipmi channels.
 */
static int write_config_nv(unsigned char channel,
                           pp_bmc_router_chan_adapter_config_t* config)
{
    const char* s;
    
    s = "";
    switch (config->channel_priv_lvl) {
        case IPMI_PRIV_CALLBACK:
            s = "CALLBACK";
            break;
        case IPMI_PRIV_USER:
            s = "USER";
            break;
        case IPMI_PRIV_OPERATOR:
            s = "OPERATOR";
            break;
        case IPMI_PRIV_ADMIN:
            s = "ADMINISTRATOR";
            break;
        case IPMI_PRIV_OEM:
            s = "OEM";
            break;
        default:
            pp_bmc_log_warn("[CHA] invalid channel_priv_lvl for channel %d", channel);
    }
    pp_cfg_set(s, "bmc.channel_access[%u].privilege_limit", channel);
    
    s = "disable";
    switch (config->access_mode) {
        case IPMI_CHAN_ACCESS_DISABLE:
            s = "disable";
            break;
        case IPMI_CHAN_ACCESS_PREBOOT:
            s = "preboot";
            break;
        case IPMI_CHAN_ACCESS_ALWAYS:
            s = "always";
            break;
        case IPMI_CHAN_ACCESS_SHARED:
            s = "shared";
            break;
        default:
            pp_bmc_log_warn("[CHA] invalid access_mode for channel %d", channel);
    }
    pp_cfg_set(s, "bmc.channel_access[%u].access_mode", channel);

    pp_cfg_set_enabled(config->user_level_authentication_disable, "bmc.channel_access[%u].user_level_authentication_disable", channel);
    pp_cfg_set_enabled(config->per_message_authentication_disable, "bmc.channel_access[%u].per_message_authentication_disable", channel);
    pp_cfg_set_enabled(config->alerting_disable, "bmc.channel_access[%u].alert_disable", channel);

    pp_cfg_save(DO_FLUSH);

    return PP_SUC;
}

/* 
 * Read a channel configuration from the config system.
 * The configuration is read unconditionally. The validity for
 * ipmi messages depends on whether the channel exists or not.
 */
static int read_config_nv(unsigned char channel,
                           pp_bmc_router_chan_adapter_config_t* config)
{
    int i;
    char* s;
    
    memset(config, 0, sizeof(pp_bmc_router_chan_adapter_config_t));

    pp_cfg_get(&s, "bmc.channel_access[%u].privilege_limit", channel);
    if (strcmp(s, "CALLBACK") == 0) {
        config->channel_priv_lvl = IPMI_PRIV_CALLBACK;
    } else if (strcmp(s, "USER") == 0) {
        config->channel_priv_lvl = IPMI_PRIV_USER;
    } else if (strcmp(s, "OPERATOR") == 0) {
        config->channel_priv_lvl = IPMI_PRIV_OPERATOR;
    } else if (strcmp(s, "ADMINISTRATOR") == 0) {
        config->channel_priv_lvl = IPMI_PRIV_ADMIN;
    } else if (strcmp(s, "OEM") == 0) {
        config->channel_priv_lvl = IPMI_PRIV_OEM;
    } else {
        /* not a valid value in this context */
        pp_bmc_log_warn("[CHA] missing configuration value for channel %d", channel);
    }
    free(s);

    pp_cfg_get(&s, "bmc.channel_access[%u].access_mode", channel);
    if (strcmp(s, "disable") == 0) {
        config->access_mode = IPMI_CHAN_ACCESS_DISABLE;
    } else if (strcmp(s, "preboot") == 0) {
        config->access_mode = IPMI_CHAN_ACCESS_PREBOOT;
    } else if (strcmp(s, "always") == 0) {
        config->access_mode = IPMI_CHAN_ACCESS_ALWAYS;
    } else if (strcmp(s, "shared") == 0) {
        config->access_mode = IPMI_CHAN_ACCESS_SHARED;
    } else {
        /* not channel data configured - very bad */
        pp_bmc_log_warn("[CHA] missing configuration value for channel %d", channel);
    }
    free(s);

    pp_cfg_is_enabled_nodflt(&i, "bmc.channel_access[%u].user_level_authentication_disable", channel);
    if (i == 0) {
        config->user_level_authentication_disable = 0;
    } else {
        config->user_level_authentication_disable = 1;
    }
    pp_cfg_is_enabled_nodflt(&i, "bmc.channel_access[%u].per_message_authentication_disable", channel);
    if (i == 0) {
        config->per_message_authentication_disable = 0;
    } else {
        config->per_message_authentication_disable = 1;
    }
    pp_cfg_is_enabled_nodflt(&i, "bmc.channel_access[%u].alert_disable", channel);
    if (i == 0) {
        config->alerting_disable = 0;
    } else {
        config->alerting_disable = 1;
    }
    
    return PP_SUC;
}

int pp_bmc_dev_channel_get_config(unsigned char channel,
                                  pp_bmc_router_chan_adapter_config_t* config)
{
    // this is the same as read_config_nv() internally
    return read_config_nv(channel, config);
}

/*
 * Get channel access command
 */
static int ipmi_cmd_app_get_channel_access(imsg_t* imsg) {
    unsigned char channel;
    unsigned char get_volatile;
    pp_bmc_router_chan_adapter_config_t config;
    pp_bmc_router_chan_adapter_config_t* config_p;
    unsigned char* data_p;
    
    channel = (imsg->data[0] & 0x0f);
    if (channel == IPMI_CHAN_PRESENT) {
        channel = imsg->chan;
    }
    
    get_volatile = 0;
    if ((imsg->data[1] & 0x80) == 0x80) {
        /* volatile */
        get_volatile = 1;
    } else if ((imsg->data[1] & 0x40) == 0x40) {
        /* nonvolatile */
        get_volatile = 0;
    } else {
        /* illegal bit configuration */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    config_p = pp_bmc_router_channel_get_config(channel);
    
    if (config_p == NULL) {
        /* no config, single session channel */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_APP_CHANNEL_SINGLE_SESSION);
    }
    
    if (get_volatile == 0) {
        if (read_config_nv(channel, &config) == PP_ERR) {
            // strange unknown config system error
            pp_bmc_log_warn("[CHA] could not read channel configuration from config system");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
        config_p = &config;
    }

    data_p = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 2);
    memcpy(data_p, config_p, 2);
    // convert IPMI_PRIV_UNSPEC to IPMI_PRIV_NOACCESS
    if ((data_p[1] & 0x0f) == IPMI_PRIV_UNSPEC) data_p[1] |= IPMI_PRIV_NOACCESS;
    
    return pp_bmc_router_send_msg(imsg);
}
 
/*
 * Set channel access command
 */

struct set_channel_access_rq_s {
    BITFIELD2(unsigned char, channel : 4, : 4);
    BITFIELD5(unsigned char,
            access_mode : 3,
            user_level_auth_disable : 1,
            per_message_auth_disable : 1,
            pef_alerting_disable : 1,
            set_channel_access : 2);
    BITFIELD3(unsigned char,
            priv_limit : 4,
            : 2,
            set_channel_priv : 2);
} __attribute__ ((packed));
typedef struct set_channel_access_rq_s set_channel_access_rq_t;

static int ipmi_cmd_app_set_channel_access(imsg_t* imsg) {
    set_channel_access_rq_t *rq = (void *)imsg->data;
    unsigned char channel;
    unsigned char write_volatile = 0;
    unsigned char write_nonvolatile = 0;
    pp_bmc_router_chan_adapter_config_t  config_nv;
    pp_bmc_router_chan_adapter_config_t* config_v;

    pp_bmc_router_chan_adapter_config_t* cur_cfg1;
    pp_bmc_router_chan_adapter_config_t* cur_cfg2;
    
    channel = rq->channel;
    if (channel == IPMI_CHAN_PRESENT) {
        channel = imsg->chan;
    }
    
    config_v = pp_bmc_router_channel_get_config(channel);
    if (config_v == NULL) {
        /* single session channel, cmd not applicable */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_APP_CHANNEL_SINGLE_SESSION);
    }
    
    if (read_config_nv(channel, &config_nv) == PP_ERR) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    /* both volatile and nv params read, now just determine whether *
     * they have to be written back or not                          */
    cur_cfg1 = NULL;
    if (rq->set_channel_access == 0x02) {
        /* volatile */
        cur_cfg1 = config_v;
        write_volatile = 1;
    } else if (rq->set_channel_access == 0x01) {
        /* nonvolatile */
        cur_cfg1 = &config_nv;
        write_nonvolatile = 1;
    } else if (rq->set_channel_access == 0x03) {
        /* reserved */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    cur_cfg2 = NULL;
    if (rq->set_channel_priv == 0x02) {
        /* volatile */
        cur_cfg2 = config_v;
        write_volatile = 1;
    } else if (rq->set_channel_priv == 0x01) {
        /* nonvolatile */
        cur_cfg2 = &config_nv;
        write_nonvolatile = 1;
    } else if (rq->set_channel_priv == 0x03) {
        /* reserved */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    /* now parse the params for v/nv destinations */
    if (cur_cfg1 != NULL) {
        cur_cfg1->alerting_disable = rq->pef_alerting_disable;
        cur_cfg1->per_message_authentication_disable = rq->per_message_auth_disable;
        cur_cfg1->user_level_authentication_disable = rq->user_level_auth_disable;
        if (rq->access_mode > 3) {
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
        } else {
            /* check if the desired access mode is allowed for the selected channel */
            /* not nice to do this hardcoded, maybe we should do this in the channels later on */
            switch (channel) {
            case IPMI_CHAN_LAN:
                if (rq->access_mode != IPMI_CHAN_ACCESS_ALWAYS) {
                    return pp_bmc_router_resp_err(imsg, IPMI_ERR_APP_CHANNEL_ACCESS_MODE_NOT_SUPPORTED);
                }
                break;
            case IPMI_CHAN_SERIAL:
                if ((rq->access_mode != IPMI_CHAN_ACCESS_DISABLE) && (rq->access_mode != IPMI_CHAN_ACCESS_SHARED)) {
                    return pp_bmc_router_resp_err(imsg, IPMI_ERR_APP_CHANNEL_ACCESS_MODE_NOT_SUPPORTED);
                }
            }
            /* access mode is correct */
            cur_cfg1->access_mode = rq->access_mode;
        }
    }
    
    if (cur_cfg2 != NULL) {
        if (rq->priv_limit == IPMI_PRIV_NOACCESS) {
            cur_cfg2->channel_priv_lvl = IPMI_PRIV_UNSPEC;
        } else if (rq->priv_limit >= IPMI_PRIV_CALLBACK && rq->priv_limit <= IPMI_PRIV_OEM) {
            cur_cfg2->channel_priv_lvl = rq->priv_limit;
        } else {
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
        }
    }
    
    /* okay, now actually write the values */
    if (write_volatile == 1) {
        pp_bmc_router_channel_config_updated(channel);
    }
    
    if (write_nonvolatile == 1) {
        if (write_config_nv(channel, &config_nv) == PP_ERR) {
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    }
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get Channel Info Command
 */

struct get_channel_info_rs_s {
    BITFIELD2(unsigned char,    channel : 4,    reserved_1 : 4);
    BITFIELD2(unsigned char,    medium : 7,     reserved_2 : 1);
    BITFIELD2(unsigned char,    proto : 5,      reserved_3 : 3);
    BITFIELD2(unsigned char,    sess_cnt : 6,   sess_supp : 2);
    unsigned char               vendor_id_1;
    unsigned char               vendor_id_2;
    unsigned char               vendor_id_3;
    unsigned char               sms_i;     /* or OEM_1 */
    unsigned char               msg_buf_i; /* or OEM_2 */
} __attribute__ ((packed));
typedef struct get_channel_info_rs_s get_channel_info_rs_t;

static int app_cmd_get_channel_info(imsg_t *imsg) {
    get_channel_info_rs_t *rs;
    unsigned char channel;
    const pp_bmc_router_chan_adapter_info_t *chan_info;
    
    channel = (imsg->data[0] & 0x0f) != IPMI_CHAN_PRESENT ?
              (imsg->data[0] & 0x0f) : imsg->chan;

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_channel_info_rs_t));
    
    rs->channel = channel & 0x0f; // mask out reserved bits
    
    if((chan_info = pp_bmc_router_get_chan_adapter_info(channel)) != NULL) {
        rs->medium = chan_info->medium & 0x7f; // mask out reserved bits
        rs->proto = chan_info->proto & 0x0f; // mask out reserved bits
        rs->sess_supp = chan_info->sess_supp & 0x3; // mask out reserved bits
    }
    
    rs->sess_cnt = pp_bmc_session_manager_get_session_cnt(channel) & 0x1f; 
                       // mask out reserved bits
    
    /* IPMI Enterprise vendor ID */
    rs->vendor_id_1 = 0xf2;
    rs->vendor_id_2 = 0x1b;
    /*  vendor_id_3 = 0x00; no need to set (init) */
    
    if(channel == IPMI_CHAN_SI) {
        /* only set auxiliary info for System Interface, OEM is not supported */
        rs->sms_i = 0xff;
        rs->msg_buf_i = 0xff;
    }
    
    return pp_bmc_router_send_msg(imsg);
}

/********************************************************************
 * App device channel command table
 */

static const dev_cmd_entry_t app_channel_cmd_tab[] = {
    {
        .cmd_hndlr = app_cmd_get_channel_info,
        .netfn = IPMI_NETFN_APP, 
        .cmd = IPMI_CMD_GET_CHANNEL_INFO,
        .min_data_size = 1, 
        .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = ipmi_cmd_app_set_channel_access,
        .netfn = IPMI_NETFN_APP, 
        .cmd = IPMI_CMD_SET_CHANNEL_ACCESS,
        .min_data_size = 3,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = ipmi_cmd_app_get_channel_access,
        .netfn = IPMI_NETFN_APP, 
        .cmd = IPMI_CMD_GET_CHANNEL_ACCESS,
        .min_data_size = 2,
        .min_priv_level = IPMI_PRIV_USER
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * App device c'tor/d'tor
 */

int pp_bmc_dev_app_channel_init()
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(app_channel_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[CHA] device started");
    return PP_SUC;
}

void pp_bmc_dev_app_channel_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(app_channel_cmd_tab);

    pp_bmc_log_info("[CHA] device shut down");
}
