/**
 * bmc_dev_app_session.c
 *
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 * 
 * Session related ipmi commands.
 */

#include <malloc.h>
#include <fcntl.h>
#include <sys/types.h>

#include <pp/base.h>
#include <pp/dlist.h>
#include <pp/xdefs.h>
#include <pp/cfg.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/lan_serv.h>
#include <pp/bmc/ipmi_lan.h>
#include <pp/bmc/session_manager.h>
#include <pp/bmc/user_manager.h>

#include <openssl/md5.h>
#include <openssl/md2.h>

#include "bmc_dev_app_session.h"

#define noSHOW_PASSWORD

#define ARP_CACHE	"/proc/net/arp"

static int get_mac_from_arp(struct in_addr ip, unsigned char *mac);

/*
 * Get channel capabilities (discovery)
 */
static int get_channel_capabilities(imsg_t *imsg)
{
    unsigned char get_extended;
    unsigned char channel;
    unsigned char req_priv_level;
    unsigned short priv_byte;
    const char* priv_lvl_string;

    /* (values are int because config system returns boolean as int) */
    int kg_enable;
    int user_level_auth;
    int per_message_auth;

    int user_login;
    int anonymous_login;
    int role_login;
    
    unsigned char* data;
    char *kg_data = NULL;
    size_t kg_size;
    
    get_extended = ((imsg->data[0] & 0x80) == 0x80) ? 1:0;
    channel = imsg->data[0] & 0x0F;
    req_priv_level = imsg->data[1] & 0x0F;
    
    if ((req_priv_level == 0) || (req_priv_level > IPMI_PRIV_OEM)) {
        /* unsupported privlevel */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    if (channel == IPMI_CHAN_PRESENT) {
        channel = imsg->chan;
    }
    
    /* get correct privilege level bitfield */
    priv_lvl_string = pp_bmc_get_privlevel_string(req_priv_level);
    if (pp_cfg_get_short_nodflt(&priv_byte, "bmc.channel_access[%u].authentication_type_enable[%s]",
                                channel, priv_lvl_string) == PP_ERR) 
    {
        /* no authentication type stored. Default authentication types are stored for *
         * all known multi-session channels so this must be a request to single       *
         * session channel.                                                           *
         * Maybe this return code is invalid ...                                      */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    /* ignore errors when reading, values are mapped to false anyway */
    pp_cfg_is_enabled(&per_message_auth, "bmc.channel_access[%u].per_message_authentication_disable", channel);
    pp_cfg_is_enabled(&user_level_auth, "bmc.channel_access[%u].user_level_authentication_disable", channel);
    
    pp_bmc_user_get_login_types(&anonymous_login, &role_login, &user_login);
    kg_enable = PP_SUCCED( pp_cfg_get_binary(&kg_data, &kg_size, "bmc.channel_access[%u].keygen_key", channel) );
    free(kg_data);
    
    /* create response */
    data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 8);
    
    data[0] = channel;
    
    data[1] = priv_byte & 0x37;
    if (get_extended) data[1] |= 0x80;

    data[2] = 0;
    if (per_message_auth) data[2] |= 0x10;
    if (user_level_auth)  data[2] |= 0x08;
    if (user_login)       data[2] |= 0x04; 
    if (role_login)       data[2] |= 0x02;
    if (anonymous_login)  data[2] |= 0x01;
    if ((get_extended) & (channel == IPMI_CHAN_LAN)) {
        if (kg_enable) data[2] |= 0x20;
    }
    
    data[3] = 0;
    if (get_extended) {
        data[3] = 0x01;  /* support 1.5 connections */
        if (channel == IPMI_CHAN_LAN) {
            data[3] |= 0x02;   /* support 2.0 connections */
        }
    }

    /* datat[4] - data[7]    (OEM fields): initialized to 0  */
        
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Get session challenge command (session activation msg 1 of 2)
 */
static int get_session_challenge(imsg_t *imsg)
{
    unsigned char authtype;
    imsg_session_t* session;
    unsigned int * uip;
    char* buf;

    /* not nice to have this hardcoded ... */
    if ((imsg->chan != IPMI_CHAN_SERIAL) && (imsg->chan != IPMI_CHAN_LAN) && (imsg->chan != IPMI_CHAN_SCSI)) {
        /* this command makes only sense on multi-session channels, return error */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    authtype = imsg->data[0] & 0xf;
    
    if ((authtype != IPMI_AUTH_NONE) && (authtype != IPMI_AUTH_STRAIGHT) &&
        (authtype != IPMI_AUTH_MD2) && (authtype != IPMI_AUTH_MD5))
    {
        /* invalid authtype */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    session = pp_bmc_create_session15(imsg->data+1, imsg->chan);
    if (session == NULL) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SESSION_INVALID_USERNAME);  // invalid username
    }
    session->authtype = authtype;
    if (session->challenge == NULL) {
        /* null challenge may only happen if someone messes up v1.5 and v2.0 sessions */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    session->session_secret = imsg->session_secret; // for IPMI-over-SCSI channel
    
    /* construct response */
    buf = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 20);
    uip = (unsigned int*)buf;
    *uip = cpu_to_le32(session->session_id);
    memcpy(buf+4, session->challenge, 16);
    
    pp_bmc_session_delete(session);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Activate session command (session activation msg 2 of 2)
 */
static int activate_session(imsg_t *imsg)
{
    unsigned char authtype_req;
    unsigned char max_priv_req;
    unsigned char* challenge;
    unsigned int out_seq;
    unsigned int *uip;
    imsg_session_t* session;
    char*  buf;
    const char* priv_level_string;
    unsigned short authtype_enable;
    unsigned short authtype_mask;
    char* channel_priv_string;
    char channel_priv_limit;
    
    /* for single session activation */
    unsigned char bmc_authcode[16];
    MD5_CTX md5_ctx;
    MD2_CTX md2_ctx;
    unsigned int ui;
    
    /* not nice to have this hardcoded ... */
    if ((imsg->chan != IPMI_CHAN_SERIAL) && (imsg->chan != IPMI_CHAN_LAN) && (imsg->chan != IPMI_CHAN_SCSI)) {
        /* this command makes only sense on multi-session channels, return error */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    session = imsg->session;
    if (session == NULL) {
        // this message has no session that can be activated
        pp_bmc_log_debug("[SESS] ignored activate command without session");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SESSION_INVALID_SESSION_ID);
    }

    /*  parse the message */
    authtype_req = imsg->data[0] & 0xF;
    max_priv_req = imsg->data[1] & 0xF;
    if ((max_priv_req == IPMI_PRIV_UNSPEC) || (max_priv_req > IPMI_PRIV_OEM)) {
        /* invalid privilege level */
        pp_bmc_log_debug("[SESS] invalid privilege level");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    challenge = imsg->data+2;
    uip =  (unsigned int*)(imsg->data+18);
    out_seq = le32_to_cpu(*uip);
    if (out_seq == 0) {
        /* sequence must not be 0 */
        pp_bmc_log_debug("[SESS] sequence must not be 0");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    /* check if the requested authtype equals the previously requested authtype */
    if (authtype_req != session->authtype) {
        /* the currently requested authtype MUST match the previously requested *
         * authtype because the user could authenticate with disabled authtypes */
        pp_bmc_log_debug("[SESS] authtype does not match");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    /* check if the requested authtype is allowed for the requested privlevel */
    priv_level_string = pp_bmc_get_privlevel_string(max_priv_req);
    pp_cfg_get_ushort(&authtype_enable,
                      "bmc.channel_access[%u].authentication_type_enable[%s]",
                      imsg->chan, priv_level_string);
    authtype_mask = 0;
    switch (authtype_req) {
        case IPMI_AUTH_NONE:
            authtype_mask = 0x01;
            break;
        case IPMI_AUTH_MD2:
            authtype_mask = 0x02;
            break;
        case IPMI_AUTH_MD5:
            authtype_mask = 0x04;
            break;
        case IPMI_AUTH_STRAIGHT:
            authtype_mask = 0x10;
            break;
    }
    if ((authtype_enable & authtype_mask) != authtype_mask) {
        /* authentication type is not allowed for requested privilege level */
        pp_bmc_log_debug("[SESS] authtype not allowed");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    /* check if the requested privilege level does not exceed user or channel privilege limit */
    channel_priv_limit = IPMI_PRIV_UNSPEC;
    pp_cfg_get(&channel_priv_string, "bmc.channel_access[%u].privilege_limit", imsg->chan);
    channel_priv_limit = pp_bmc_get_privlevel_value(channel_priv_string);
    free(channel_priv_string);
    
    if ((max_priv_req > pp_bmc_user_get_priv_level(session->uid, imsg->chan)) ||
        (max_priv_req > channel_priv_limit))
    {
        pp_bmc_log_debug("[SESS] session privilege level");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SESSION_PRIVILEGE_EXCEEDED);
    }

    /* imsg->data[2-17] is the challenge/authcode */
    if (imsg->chan == IPMI_CHAN_LAN) {
        /* For multisession channels (lan):                       *
         * the challenge must match data from GetSessionChallenge *
         * Authentication is done implicitly in channel interface */
        if (memcmp(challenge, session->challenge, 16) != 0) {
            /* this is not the challenge that we sent, user tries to fool us */
            pp_bmc_log_notice("[SESS] ignored multisession activation with invalid authcode");
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    } else {
        /* For singlesession channels (serial):                   *
         * clear text password or MD2/MD5 authcode                *
         * Authentication must be done manually in this command.  *
         * session->authtype is already known to be correct       */
        switch (session->authtype) {
            case IPMI_AUTH_NONE:
                /* if auth=none, the authcode is not valid/unused                 * 
                 * copy existing data to required data to fake a correct authcode */
                memcpy(bmc_authcode, challenge, 16);
                break;
            case IPMI_AUTH_STRAIGHT:
#if defined(SHOW_PASSWORD)
                {
                    char pw[17];
                    memcpy(pw, challenge, 16);
                    pw[16] = '\0';
                    printf("password: %s\n", pw);
                }
#endif
                /* challenge must match password */
                memcpy(bmc_authcode, session->password, 16);
                break;
            case IPMI_AUTH_MD2:
                /* challenge must match H(password + sessId + challenge string + password) */
                MD2_Init(&md2_ctx);
                MD2_Update(&md2_ctx, session->password, 16);
                ui = cpu_to_le32(session->session_id);
                MD2_Update(&md2_ctx, (unsigned char*)(&ui), 4);     // session_id
                MD2_Update(&md2_ctx, session->challenge, 16);
                MD2_Update(&md2_ctx, session->password, 16);
                MD2_Final(bmc_authcode, &md2_ctx);
                break;
            case IPMI_AUTH_MD5:
                /* challenge must match H(password + sessId + challenge string + password) */
                MD5_Init(&md5_ctx);
                MD5_Update(&md5_ctx, session->password, 16);
                ui = cpu_to_le32(session->session_id);
                MD5_Update(&md5_ctx, &ui, 4);     // session_id
                MD5_Update(&md5_ctx, session->challenge, 16);
                MD5_Update(&md5_ctx, session->password, 16);
                MD5_Final(bmc_authcode, &md5_ctx);
                break;
            default:
                /* should not happen as authtype is checked, but who knows */
                assert(0);
        }
        
        /* compare the calculated authcode and the challenge */
        if (memcmp(bmc_authcode, challenge, 16) != 0) {
            /* the message could not be authenticated */
            pp_bmc_log_notice("[SESS] ignored singlesession activation with invalid authcode");
            return pp_bmc_router_resp_err(imsg,
                imsg->chan == IPMI_CHAN_SCSI ? IPMI_ERR_SESSION_AUTH_FAILED : IPMI_ERR_UNSPECIFIED);
        }
        
    }

    /* everything is fine, activate the session */
    
    /* requested privlevel is correct, assign to session */
    session->max_priv_level = max_priv_req;
    
    /* We could (should ?) use random inbound sequence number here but   *
     * IPMIView requires it to be 1.                                     */
    session->inbound_sequence = 1;
    
    session->outbound_sequence = out_seq;
    session->max_priv_level = max_priv_req;
    if (session->max_priv_level == IPMI_PRIV_CALLBACK) {
        session->cur_priv_level = IPMI_PRIV_CALLBACK;
    } else {
        /* regular sessions start as user, privilege level must be raised */
        session->cur_priv_level = IPMI_PRIV_USER;
    }

    pp_bmc_log_debug("[LAN] activated ipmi15 session %d", session->session_id);

    // construct response
    buf = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 10);
    buf[0] = session->authtype;
    uip  =  (unsigned int*)(buf+1);
    *uip = cpu_to_le32(session->session_id);         // keep the existing session id
    uip  =  (unsigned int*)(buf+5);
    *uip = cpu_to_le32(session->inbound_sequence+1); // we expect the new seq to be higher than the stored one
    
    buf[9] = session->max_priv_level;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set session privilege command
 */
static int set_session_privilege(imsg_t *imsg)
{
    unsigned char req_priv;
    char* buf;
    
    if (imsg->session == NULL) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    req_priv = imsg->data[0] & 0xF;
    
    switch (req_priv) {
    case 0:
        req_priv = imsg->session->cur_priv_level;
        break;
    case 2:
        req_priv = IPMI_PRIV_USER;
        break;
    case 3:
        req_priv = IPMI_PRIV_OPERATOR;
        break;
    case 4:
        req_priv = IPMI_PRIV_ADMIN;
        break;
    case 5:
        req_priv = IPMI_PRIV_OEM;
        break;
    default:
        // unknown priv level
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
    
    if (req_priv > imsg->session->max_priv_level) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SESSION_REQ_PRIVILEGE_EXCEEDED);
    }
    
    imsg->session->cur_priv_level = req_priv;

    buf = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    buf[0] = imsg->session->cur_priv_level;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Close session command. Works for session "owner" only. Admins
 * and System Interface can close any session.
 */
static int close_session(imsg_t *imsg)
{
    unsigned int *uip;
    unsigned int session;
    
    uip = (unsigned int*)(imsg->data);
    session = le32_to_cpu(*uip);
    
    if (imsg->session == NULL) {
        /* only SI can close arbitrary sessions */
        if (imsg->chan != IPMI_CHAN_SI) {
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
        }
    } else {
        if (session != imsg->session->session_id) {
            /* close is not for this users session, only allow if admin */
            if (imsg->session->cur_priv_level != IPMI_PRIV_ADMIN) {
                pp_bmc_log_debug("[SESS] only for admin");
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_INSUFFICIENT_PRIVILEGE_LEVEL);
            }
        }
    }
    
    /* the user is allowed to close the specified session */
    if (pp_bmc_close_session(session) == PP_ERR) {
        pp_bmc_log_debug("[SESS] close session failed");
        /* this usually fails because the session does not exist */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SESSION_CLOSE_NOT_EXISTING);
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get session Info command
 */

struct session_info_common_rs_s {
    unsigned char               handle;
    BITFIELD2(unsigned char,    max_sess : 5,   res_2 : 3);
    BITFIELD2(unsigned char,    act_sess : 5,   res_3 : 3);
    BITFIELD2(unsigned char,    uid : 5,        res_4 : 3);
    BITFIELD2(unsigned char,    priv_level : 4, res_5 : 4);
    BITFIELD2(unsigned char,    channel : 4,    proto_aux : 4);
} __attribute__ ((packed));
    
struct session_info_lan_rs_s {
    unsigned int        ip;
    unsigned char       mac[6];
    unsigned short int  port_le16;
} __attribute__ ((packed));

struct session_info_serial_rs_s {
    unsigned char               activity;
    BITFIELD2(unsigned char,    dest_sel : 4,   res_2 : 4);
    unsigned int                ppp_ip_le32;
} __attribute__ ((packed));

struct session_info_rs_s {
    struct session_info_common_rs_s     common;
    union {
        struct session_info_lan_rs_s    lan;
        struct session_info_serial_rs_s serial;
    } __attribute__ ((packed));
} __attribute__ ((packed));
typedef struct session_info_rs_s session_info_rs_t;

static int get_session_info(imsg_t *imsg) { /* 22.20 */
    /* local data buffer */
    session_info_rs_t rs;
    int data_len;

    /* IPMI message data */
    unsigned char s_idx = imsg->data[0];
    unsigned int session_id;
    unsigned char session_handle;
    imsg_session_t* session;
    
    char *imsg_data;
    
    if( ((s_idx == 0xfe) && (imsg->data_size != 2)) ||
        ((s_idx == 0xff) && (imsg->data_size != 5)) )
    {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    }
    
    /* get desired session... */
    session = NULL;
    if(s_idx == 0x00) {
        /* active session for command */
        if (imsg->session == NULL) {
            /* sessionless messages have no session that could be queried */
            return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
        }
        /* map to session_id query case with our session_id */
        session_id = imsg->session->session_id;
        session = pp_bmc_get_session(session_id);
    } else if(s_idx == 0xff) {
        session_id = le32_to_cpu(*(unsigned int*)(&(imsg->data[1])));
        session = pp_bmc_get_session(session_id);
    } else if(s_idx == 0xfe) {
        session_handle = imsg->data[1];
        session = pp_bmc_get_session_by_handle(session_handle);
    } else {
        // get Nth active session
        session = pp_bmc_get_nth_session(s_idx);
    }
    
    /* generate response in local buffer */
    memset(&rs, 0, sizeof(struct session_info_common_rs_s)); // init

    rs.common.max_sess = 0x3f; /* we allow maximum number of sessions */
    rs.common.act_sess = pp_bmc_session_manager_get_all_session_cnt();
    
    if (session == NULL) {
        /* no active session of that id / handle */
        rs.common.handle = 0; /* no active session */
        data_len = 3;
    } else {
        rs.common.handle = session->session_handle;
        
        rs.common.uid = session->uid;
        rs.common.priv_level = session->cur_priv_level;
        rs.common.channel = session->chan;
        
        if( (session->chan == IPMI_CHAN_LAN) &&
            (session->addr->ipmi_version == IPMI_LAN_VERSION_20) )
        {
            rs.common.proto_aux = 0x1; /* IPMI v20 */
        } else {
            rs.common.proto_aux = 0;  /* IPMI v15 */
        }
        
        data_len = sizeof(struct session_info_common_rs_s);

        /* now add channel specific data */
        if(session->chan == IPMI_CHAN_LAN) {
            struct sockaddr_in *s_addr;
            struct sockaddr_in sa;
            
            data_len += sizeof(struct session_info_lan_rs_s);
            
            s_addr = pp_bmc_lanserv_get_sockaddr(session);
            if (s_addr == NULL) {
                memset(&sa, 0, sizeof(struct sockaddr_in));
                s_addr = &sa;
            }
            
            /* remote console IP address */
            rs.lan.ip = s_addr->sin_addr.s_addr;

	    if (PP_ERR == get_mac_from_arp(s_addr->sin_addr, rs.lan.mac)) {
		memset(rs.lan.mac, 0, 6); /* MAC address of this session */
	    }

            /* port number */
            rs.lan.port_le16 = cpu_to_le16(s_addr->sin_port);
        } else if(session->chan == IPMI_CHAN_SERIAL) {
            data_len += sizeof(struct session_info_serial_rs_s);
            
            memset(&rs, 0, sizeof(struct session_info_serial_rs_s)); // init
            /* These three values are fixed: */
            /* activity type = 0 */
            /* destination selector = 0*/
            /* PPP IP address = 0.0.0.0 */ 
        }
        
        pp_bmc_session_delete(session);
    }
    
    /* create response and fill with local buffer */
    imsg_data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, data_len);
    memcpy(imsg_data, &rs, data_len);
        
    return pp_bmc_router_send_msg(imsg);
}

/**
 * parse /proc/net/arp to gather the mac address of the given ip address.
 * if the format of /proc/net/arp will change in the future, this function
 * will probably not work anymore. furthermore this will only work if the
 * client is located in the same subnet.
 */
static int get_mac_from_arp(struct in_addr ip, unsigned char *mac) {
    int ret = PP_ERR;
    FILE *file;
    char *addr_str, *saveptr;
    char buf[256];

    const char delimiters[] = " \t";
    char *token, *cp;

    addr_str = inet_ntoa(ip);
    
    if ((file = fopen(ARP_CACHE, "r")) != NULL) {
	while (fgets(buf, sizeof(buf), file)) {
	    if (!strncmp(buf, addr_str, strlen(addr_str))) {
		cp = strdupa(buf);
		token = strtok_r(cp, delimiters, &saveptr);
		while ( (token = strtok_r(NULL, delimiters, &saveptr)) != NULL ) {
		    if ( sscanf(token, "%hhx:%hhx:%hhx:%hhx:%hhx:%hhx",
				&mac[0], &mac[1], &mac[2],
				&mac[3], &mac[4], &mac[5]) == 6 ) {
			ret = PP_SUC;
			break;
		    }
		}
	    }
	}
	fclose(file);
    }
    return ret;
}

/********************************************************************
 * Init/Cleanup
 */

static const dev_cmd_entry_t cmd_tab[] = {
    {
        .cmd_hndlr = get_channel_capabilities, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_GET_CHANNEL_CAPABILITIES, 
        .min_data_size = 2,
        .min_priv_level = IPMI_PRIV_UNSPEC
    },
    {
        .cmd_hndlr = get_session_challenge, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_GET_SESSION_CHALLENGE, 
        .min_data_size = 17,
        .min_priv_level = IPMI_PRIV_UNSPEC
    },
    {
        .cmd_hndlr = activate_session, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_ACTIVATE_SESSION, 
        .min_data_size = 22,
        .min_priv_level = IPMI_PRIV_UNSPEC
    },
    {
        .cmd_hndlr = set_session_privilege, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_SET_SESSION_PRIVILEGE, 
        .min_data_size = 1,
        .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = close_session, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_CLOSE_SESSION, 
        .min_data_size = 4,
        .min_priv_level = IPMI_PRIV_CALLBACK
    },
    {
        .cmd_hndlr = get_session_info, 
        .netfn = IPMI_NETFN_APP,
        .cmd = IPMI_CMD_GET_SESSION_INFO, 
        .min_data_size = 1,
        .min_priv_level = IPMI_PRIV_USER
    },
    { .cmd_hndlr = NULL }
};


int pp_bmc_dev_session_init()
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(cmd_tab)) return PP_ERR;
    
    pp_bmc_log_info("[SESS] device started");
    return PP_SUC;
}

void pp_bmc_dev_session_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(cmd_tab);

    pp_bmc_log_info("[SESS] device shut down");
}




