/**
 * \file bmc_dev_chas.c
 *
 * Description: BMC App Device
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/selector.h>
#include <pp_gpio.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_power.h>
#include <pp/bmc/host_hardware.h>
#include <pp/bmc/bmc_chassis.h>
#include "bmc_dev_chassis.h"

/********************************************************************
 * POH Counter
 */
#define POH_FILE	"/flashdisk/bmc/poh_counter"
#define POH_MINS_PER_CNT  1 /* [min/cnt] */
static int poh_timer_id = 0;

/* TODO: should be non-violatile */
static unsigned int poh_counter = 0; /* accumulate host's overall operation time */
struct boot_options_s {
    unsigned char param_valid;
    unsigned char set_in_progress;
    unsigned char service_partition_selector;
    unsigned char bmc_boot_flag_valid_bit_clearing;
    unsigned char boot_info_ack;
    unsigned char boot_flags[4];
    unsigned char boot_initiator_info[9];
    unsigned char boot_initiator_mailbox[80];
} __attribute__ ((packed));
typedef struct boot_options_s boot_options_t;

static boot_options_t boot_options;
static int invalidate_timer_id = 0;

    /* the source of the last power event (see Table 28-14)       */
static unsigned char power_control_source = 0;
static unsigned char power_control_channel = 0;

static const char *bmc_chassis_request_bios_scan_service_partition_str = "bmc.chassis.request_bios_scan_service_partition";
static const char *bmc_chassis_service_partition_discovered_str = "bmc.chassis.service_partition_discovered";

static int poh_counter_load(void);
static int poh_counter_store(void);

static int poh_timer(const int item_id UNUSED, void* ctx UNUSED)
{
    if (HOST_POWER_STATE_POH_IS_ON(pp_bmc_host_power_get_state())) {
    poh_counter++;
	if (PP_ERR == poh_counter_store()) {
	    pp_bmc_log_warn("[CHASSIS] poh counter store failed");
	}
    }
    return PP_SUC;
}

/********************************************************************
 * Chassis command handlers
 */

/* Chassis Capabilities: field Capability Flags */
#define IPMI_CHASSIS_CAPS_PWR_INTERLOCK         0x08
#define IPMI_CHASSIS_CAPS_DIAG_INTR             0x04
#define IPMI_CHASSIS_CAPS_FRONTPANEL_LOCKOUT    0x02
#define IPMI_CHASSIS_CAPS_INTRUSION_SENSOR      0x01

struct chassis_caps_s {
    unsigned char cap_flags;
    unsigned char fru_dev_addr;
    unsigned char sdr_dev_addr;
    unsigned char sel_dev_addr;
    unsigned char sm_dev_addr;
    unsigned char bridge_dev_addr;
} __attribute__ ((packed));
typedef struct chassis_caps_s chassis_caps_t;
  

/*
 * Get Chassis Capabilities
 */

static int chassis_cmd_get_caps(imsg_t* imsg)
{
    chassis_caps_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                                   sizeof(chassis_caps_t));
    static const unsigned char bmc_address = 0x20;
    
    rs->cap_flags       = 0; /* we don't provide any caps ;-) */
    rs->fru_dev_addr    = bmc_address;
    rs->sdr_dev_addr    = bmc_address;
    rs->sel_dev_addr    = bmc_address;
    rs->sm_dev_addr     = bmc_address;
    rs->bridge_dev_addr = 0x20; // mandatory if chassis device implemented on BMC, see spec

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Set Chassis Capabilities
 */

/*
static int chassis_cmd_set_caps(imsg_t* imsg)
{
    // chassis_caps_t *rq = (chassis_caps_t*)imsg->data;

    // do nothing, all values read-only by now

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}
*/

/*
 * Get Chassis Status
 */

struct get_chassis_state_rs_s {
    unsigned char power_state;
    unsigned char last_power_event;
    unsigned char misc_state;
    unsigned char front_panel_caps;
} __attribute__ ((packed));
typedef struct get_chassis_state_rs_s get_chassis_state_rs_t;

static int chassis_cmd_get_state(imsg_t* imsg)
{
    get_chassis_state_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_chassis_state_rs_t));

    rs->power_state = ((pp_bmc_host_power_policy_get() << 5) & 0x60);
    if (pp_bmc_host_power_control_result() == PP_ERR)
        rs->power_state |= 0x10;
    rs->power_state = rs->power_state | (0x01 << pp_bmc_host_power_supply_status());

    /* set power status (on/off) */
    if (pp_bmc_host_power_state_is_on()) rs->power_state |= 0x01;
    else rs->power_state &= ~0x01;

    rs->last_power_event = pp_bmc_host_power_get_last_event();
    
    rs->misc_state = 0;
    if (pp_bmc_hardware_fan_health() != PP_SUC)
        rs->misc_state = rs->misc_state & 0x08;
    if (pp_bmc_hardware_drive_health() != PP_SUC)
        rs->misc_state = rs->misc_state & 0x04;
    if (pp_bmc_hardware_frontpanel_lockout_get() == PP_SUC)
        rs->misc_state = rs->misc_state & 0x02;
    if (pp_bmc_hardware_chassis_closed() != PP_SUC)
        rs->misc_state = rs->misc_state & 0x01;

    rs->front_panel_caps = 0;
    /* check returns whether the button can be disabled or not */
    if (pp_bmc_hardware_button_enable_check(HOST_BUTTON_STANDBY) == HOST_BUTTON_ENABLE)
        rs->front_panel_caps |= 0x80;
    if (pp_bmc_hardware_button_enable_check(HOST_BUTTON_DIAG_INT) == HOST_BUTTON_ENABLE)
        rs->front_panel_caps |= 0x40;
    if (pp_bmc_hardware_button_enable_check(HOST_BUTTON_RESET) == HOST_BUTTON_ENABLE)
        rs->front_panel_caps |= 0x20;
    if (pp_bmc_hardware_button_enable_check(HOST_BUTTON_POWER) == HOST_BUTTON_ENABLE)
        rs->front_panel_caps |= 0x10;
    /* enable_get returns whether the button is disabled or not */
    if (pp_bmc_hardware_button_enable_get(HOST_BUTTON_STANDBY) == HOST_BUTTON_DISABLE)
        rs->front_panel_caps |= 0x08;
    if (pp_bmc_hardware_button_enable_get(HOST_BUTTON_DIAG_INT) == HOST_BUTTON_DISABLE)
        rs->front_panel_caps |= 0x04;
    if (pp_bmc_hardware_button_enable_get(HOST_BUTTON_RESET) == HOST_BUTTON_DISABLE)
        rs->front_panel_caps |= 0x02;
    if (pp_bmc_hardware_button_enable_get(HOST_BUTTON_POWER) == HOST_BUTTON_DISABLE)
        rs->front_panel_caps |= 0x01;
    
#if defined(LARA_KIMMSI)
    /* MSI iConsole does not support the last byte. We will silently discard the last byte */
    imsg->data_size--;
#endif

    return pp_bmc_router_send_msg(imsg);
}

void pp_bmc_dev_chassis_set_restart_cause(int action, unsigned char source, 
					  unsigned char channel) {
    int clear_valid_bit = 1;

    // remember restart cause
    power_control_source = source;
    power_control_channel = channel;

    // Invalidate boot flags unless configured to leave valid by
    // bmc_boot_flag_valid_bit_clearing
    switch (source) {
    case HOST_POWER_CONTROL_SOURCE_PEF_RESET: // Fallthrough
    case HOST_POWER_CONTROL_SOURCE_PEF_CYCLE:
	if (boot_options.bmc_boot_flag_valid_bit_clearing & 0x10) {
	    clear_valid_bit = 0;
	}
	break;
    case HOST_POWER_CONTROL_SOURCE_WATCHDOG:
	if (boot_options.bmc_boot_flag_valid_bit_clearing & 0x04) {
	    clear_valid_bit = 0;
	}
	break;
    case HOST_POWER_CONTROL_SOURCE_RESET_BUTTON: // Fallthrough
    case HOST_POWER_CONTROL_SOURCE_SOFT_RESET:
	if (boot_options.bmc_boot_flag_valid_bit_clearing & 0x02) {
	    clear_valid_bit = 0;
	}
	break;
    case HOST_POWER_CONTROL_SOURCE_POWER_BUTTON_ON: // Fallthrough
    case HOST_POWER_CONTROL_SOURCE_RTC:
	if (boot_options.bmc_boot_flag_valid_bit_clearing & 0x01) {
	    clear_valid_bit = 0;
	}
	break;
    case HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL:
	if ((action == HOST_POWER_CONTROL_CYCLE || 
	     action == HOST_POWER_CONTROL_RESET) &&
	    invalidate_timer_id) 
	    {
		// Stop the invalidation timer
		invalidate_timer_id = pp_select_remove_to(invalidate_timer_id);
		assert(invalidate_timer_id == 0);
		clear_valid_bit = 0;
	    }
	break;
    default:
	clear_valid_bit = 1;
    }

    if (action == HOST_POWER_CONTROL_OFF) {
	clear_valid_bit = 1;
    }
    if (clear_valid_bit) {
	boot_options.boot_flags[0] &= ~0x80;
    }
}

/**
 * boot_flags_invalidate_timeout() is called 60 seconds after an IPMI
 * client sets the boot-flags-valid bit.
 */
static int boot_flags_invalidate_timeout(const int item_id UNUSED, void* ctx UNUSED) {
    if (!(boot_options.bmc_boot_flag_valid_bit_clearing & 0x08)) {
	boot_options.boot_flags[0] &= (~0x80); // clear boot flags valid bit
    }
    return PP_SUC;
}

/*
 * Chassis Control
 */

#define IPMI_CHASSIS_CTRL_PWR_DOWN      0x00
#define IPMI_CHASSIS_CTRL_PWR_UP        0x01
#define IPMI_CHASSIS_CTRL_PWR_CYCLE     0x02
#define IPMI_CHASSIS_CTRL_HARD_RESET    0x03
#define IPMI_CHASSIS_CTRL_DIAG_INTR     0x04
#define IPMI_CHASSIS_CTRL_SOFT_SHUTDOWN 0x05

static int chassis_cmd_control(imsg_t* imsg)
{
    int rv;
    unsigned char ctrl = imsg->data[0];
    unsigned char cc = IPMI_ERR_SUCCESS;

    if (invalidate_timer_id) { // refresh timeout
	pp_select_remove_to(invalidate_timer_id);
	invalidate_timer_id = pp_select_add_to(60000, 0, 
			        boot_flags_invalidate_timeout, NULL);
    }

    switch (ctrl) {
        case IPMI_CHASSIS_CTRL_PWR_DOWN:
            rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_OFF, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);
            if (rv == PP_NOT_APPLICABLE)
                cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
            if (rv == PP_ERR)
                cc = IPMI_ERR_UNSPECIFIED;
            break;

        case IPMI_CHASSIS_CTRL_PWR_UP:
            rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_ON, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);
            if (rv == PP_NOT_APPLICABLE)
                cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
            if (rv == PP_ERR)
                cc = IPMI_ERR_UNSPECIFIED;
            break;
            
        case IPMI_CHASSIS_CTRL_PWR_CYCLE:
            rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_CYCLE, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);
            if (rv == PP_NOT_APPLICABLE)
                cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
            if (rv == PP_ERR)
                cc = IPMI_ERR_UNSPECIFIED;
            break;
            
        case IPMI_CHASSIS_CTRL_HARD_RESET:
            rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_RESET, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);
            if (rv == PP_NOT_APPLICABLE)
                cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
            if (rv == PP_ERR)
                cc = IPMI_ERR_UNSPECIFIED;
            break;
            
        case IPMI_CHASSIS_CTRL_SOFT_SHUTDOWN:
            rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_SOFT_OFF, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);
            if (rv == PP_NOT_APPLICABLE)
                cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
            if (rv == PP_ERR)
                cc = IPMI_ERR_UNSPECIFIED;
            break;

        default:
            cc = IPMI_ERR_INVALID_DATA_FIELD;
    }

    return pp_bmc_router_resp_err(imsg, cc);
}


/*
 * Chassis Reset
 */

static int chassis_cmd_reset(imsg_t* imsg)
{
    unsigned char cc = IPMI_ERR_SUCCESS;
    int rv;

    rv = pp_bmc_host_power_control(HOST_POWER_CONTROL_RESET, HOST_POWER_CONTROL_SOURCE_CHASSIS_CONTROL, imsg->chan);

    if (rv == PP_NOT_APPLICABLE)
        cc = IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE;
    if (rv == PP_ERR)
        cc = IPMI_ERR_UNSPECIFIED;

    return pp_bmc_router_resp_err(imsg, cc);
}

/*
 * Chassis Identify
 */

static int chassis_cmd_identify(imsg_t* imsg)
{
    int duration;

    if (pp_bmc_hardware_chassis_identify_device_exists() == PP_ERR)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_CMD);
    
    if (imsg->data_size > 0)
        duration = imsg->data[0];
    else
        duration = 15;

    if (imsg->data_size > 1)
        if (imsg->data[1] & 0x01)
            /* turn on idenfinitly*/
            duration = -1;

    pp_bmc_hardware_chassis_identify_set(duration);

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Chassis Set Power Restore Policy
 */

static int chassis_cmd_set_power_restore_policy(imsg_t* imsg)
{
    unsigned char policy = imsg->data[0];
    unsigned char *supported = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);

    pp_bmc_log_info("[CHASSIS] Set Power Restore Policy (pol=%d)", policy & 0x07);

    switch (policy & 0x07) {
        case 0x00: pp_bmc_host_power_policy_set(HOST_POWER_POLICY_OFF); break;
        case 0x01: pp_bmc_host_power_policy_set(HOST_POWER_POLICY_RESTORE); break;
        case 0x02: pp_bmc_host_power_policy_set(HOST_POWER_POLICY_ON); break;
        case 0x03: break; // just query supported policies 
        default:   pp_bmc_host_power_policy_set(HOST_POWER_POLICY_UNKNOWN);
    }

    supported = 0;
    return pp_bmc_router_send_msg(imsg);
}

/*
 * Chassis Set Power Cycle Interval
 */

static int chassis_cmd_set_power_cycle_interval(imsg_t* imsg)
{
    pp_bmc_host_power_cycle_interval_set(imsg->data[0]);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Chassis Get POH Counter
 */

struct get_poh_counter_rs_s {
    unsigned char mins_per_count;
    unsigned int counter_le32;
} __attribute__ ((packed));
typedef struct get_poh_counter_rs_s get_poh_counter_rs_t;

static int chassis_cmd_get_poh_counter(imsg_t* imsg UNUSED)
{
    get_poh_counter_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_poh_counter_rs_t));
    rs->mins_per_count = POH_MINS_PER_CNT;
    rs->counter_le32 = cpu_to_le32(poh_counter);

    return pp_bmc_router_send_msg(imsg);
}

struct get_system_restart_cause_rs_s {
    BITFIELD2(unsigned char, cause : 4, reserved : 4);
    unsigned char channel;
} __attribute__ ((packed));
typedef struct get_system_restart_cause_rs_s get_system_restart_cause_rs_t;

static int chassis_cmd_get_system_restart_cause(imsg_t* imsg){
    get_system_restart_cause_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                                        sizeof(get_system_restart_cause_rs_t));
    
    rs->cause = power_control_source & 0x0f;
    rs->channel = power_control_channel;
    return pp_bmc_router_send_msg(imsg);
}

#define CHECK_PDATA_SIZE(s) \
    if (psize < s) { ierr = IPMI_ERR_REQ_DATA_LEN_INVALID; break; }

#define SET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[SOL-CFG] can't write param #%d to cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int chassis_cmd_set_system_boot_options(imsg_t* imsg)
{
    unsigned char param_valid = (imsg->data[0] & 0x80) >> 7;
    int param = imsg->data[0] & 0x7f;
    unsigned char* pdata = &imsg->data[1];
    int psize = imsg->data_size - 1;

    unsigned char ierr = IPMI_ERR_SUCCESS;
    int dirty = 0;
    pp_cfg_tx_t* cfg_trans;

    boot_options.param_valid &= ~(1 << param);
    boot_options.param_valid |= (param_valid << param);

    cfg_trans = pp_cfg_tx_begin(1);
    if (!cfg_trans) {
        pp_bmc_log_error("[CHASSIS] can't create cfg transaction");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    // The client wants to set the parameter and not just the
    // parameter-valid-bit
    switch (param) {
	case 0: // Set in progress
            CHECK_PDATA_SIZE(1);
            switch (pdata[0] & 0x3) {
                case 0: /* set complete */
                    boot_options.set_in_progress = 0;
                    break;
                case 1: /* set in progress */
                    if (boot_options.set_in_progress == 1) {
                        ierr = IPMI_ERR_CHA_SET_IN_PROGRESS;
                    }
                    boot_options.set_in_progress = 1;
                    break;
                default:
                    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            }
            break;
	case 1: // Service partition selector
	    CHECK_PDATA_SIZE(1);
	    boot_options.service_partition_selector = pdata[0];
	    break;
	case 2: // Service partition scan
	    // The only non-volatile option stored in the config system.
	    CHECK_PDATA_SIZE(1);
	    SET_CFG(pp_cfg_set_enabled(pdata[0] & 0x02,
	       bmc_chassis_request_bios_scan_service_partition_str));
	    SET_CFG(pp_cfg_set_enabled(pdata[0] & 0x01,
			       bmc_chassis_service_partition_discovered_str));
            dirty = 1;
	    break;
	case 3: // BMC boot flag valid bit clearing
	    CHECK_PDATA_SIZE(1);
	    boot_options.bmc_boot_flag_valid_bit_clearing = pdata[0] & 0x1F;
	    break;
	case 4: // Boot info acknowledge
	    CHECK_PDATA_SIZE(2);
	    boot_options.boot_info_ack = (boot_options.boot_info_ack & ~pdata[0])
		                          | (pdata[1] & pdata[0]);
	    break;
	case 5: // Boot flags
	    CHECK_PDATA_SIZE(5);
	    memcpy(boot_options.boot_flags, pdata, 4);
	    if ( pdata[0] & 0x80) { // valid bit was set
		invalidate_timer_id = pp_select_add_to(60000, 0, 
	                      boot_flags_invalidate_timeout, NULL);
	    }
	    break;
	case 6: // Boot initiator info
	    CHECK_PDATA_SIZE(9);
	    memcpy(boot_options.boot_initiator_info, pdata, 9);
	    break;
	case 7: // Boot initiator mailbox
	    CHECK_PDATA_SIZE(1);
	    if (pdata[0] >= 5) {
		ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
	    } else if (psize > 17) {
		ierr = IPMI_ERR_REQ_DATA_LEN_LIMIT_EXCEEDED;
	    } else{
		memcpy(&boot_options.boot_initiator_mailbox[16 * pdata[0]],
		       &pdata[1], psize - 1);
	    }
	    break;
	default:
	    ierr = IPMI_ERR_CHA_PARAM_NOT_SUPPORTED;
    }

    if (dirty && ierr == IPMI_ERR_SUCCESS) {
        if (PP_FAILED(pp_cfg_tx_commit(cfg_trans))) {
            pp_bmc_log_error("[CHASSIS] can't commit cfg transaction");
            ierr = IPMI_ERR_UNSPECIFIED;
        }
        pp_cfg_save(DO_FLUSH);
    } else {
        pp_cfg_tx_rollback(cfg_trans);
    }

    return pp_bmc_router_resp_err(imsg, ierr);
}

#define GET_CFG(c) \
    if (PP_FAILED(c)) { \
        pp_bmc_log_error("[CASSIS] can't read param #%d from cfg", param); \
        ierr = IPMI_ERR_UNSPECIFIED; \
        break; \
    }

static int chassis_cmd_get_system_boot_options(imsg_t* imsg)
{
    int param = imsg->data[0] & 0x7f;
    int param_valid = (boot_options.param_valid & (1 << param)) >> param;
    int set_selector = imsg->data[1];
    int block_selector = imsg->data[2];
    unsigned char * resp = NULL;
    int ierr = 0;
    int val;
    unsigned char pdata[20];
    int psize = 0;

    if (block_selector != 0) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    if (imsg->data_size != 3) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    }

    switch (param) {
    case 0: // Set in progress
	psize = 1;
	pdata[0] = boot_options.set_in_progress;
	break;
    case 1: // Service partition selector
	psize = 1;
	pdata[0] = boot_options.service_partition_selector;
	break;
    case 2:  // Service partition scan
	psize = 1;
	GET_CFG(pp_cfg_is_enabled(&val,
			bmc_chassis_request_bios_scan_service_partition_str));
	pdata[0] = val;

	GET_CFG(pp_cfg_is_enabled(&val,
		        bmc_chassis_service_partition_discovered_str));
	pdata[0] |= val << 1;
	break;
    case 3: // BMC boot flag valid bit clearing
	psize = 1;
	pdata[0] = boot_options.bmc_boot_flag_valid_bit_clearing;
	break;
    case 4: // Boot info acknowledge
	psize = 2;
	pdata[0] = 0;
	pdata[1] = boot_options.boot_info_ack;
	break;
    case 5: // Boot flags
	psize = 5;
	memcpy(pdata, boot_options.boot_flags, 4);
	pdata[4] = 0; // reserved
	break;
    case 6: // Boot initiator info
	psize = 9;
	memcpy(pdata, boot_options.boot_initiator_info, 9);
	break;
    case 7: // Boot initiator mailbox
	if (set_selector <= 4) {
	    psize = 17;
	    pdata[0] = set_selector;
	    memcpy(pdata + 1,
		   &boot_options.boot_initiator_mailbox[16 * set_selector], 16);
	}
	else {
	    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
	}
	break;
    default:
	ierr = IPMI_ERR_CHA_PARAM_NOT_SUPPORTED;
    }

    if (ierr != IPMI_ERR_SUCCESS) {
        return pp_bmc_router_resp_err(imsg, ierr);
    }

    resp = pp_bmc_imsg_resp(imsg, ierr, 2 + psize);
    resp[0] = 0x01; // parameter version
    resp[1] = (param_valid << 7) | param;
    memcpy(resp + 2, pdata, psize);
    return pp_bmc_router_send_msg(imsg);
}

static int poh_counter_load(void) {
    int fd, n, ret = PP_ERR;
    
    if ((fd = open(POH_FILE, O_RDONLY)) >= 0) {
	if ((n = read(fd, &poh_counter, sizeof(unsigned int))) != sizeof(unsigned int)) {
	    if (n == -1) {
		pp_log_err("%s(): read(%s) failed", ___F, POH_FILE);
	    } else {
		pp_log("%s(): read(%s) failed\n", ___F, POH_FILE);
	    }
	    poh_counter = 0;
	} else {
	    ret = PP_SUC;
	}
	close(fd);
    }
    return ret;
}

static int poh_counter_store(void) {
    int fd, n;
    char filepath_tmp[128];

    snprintf(filepath_tmp, sizeof(filepath_tmp), "%s.tmp", POH_FILE);
    
    /* open disk file */
    if ((fd = open(filepath_tmp, O_RDWR|O_TRUNC|O_CREAT, S_IRUSR|S_IWUSR)) == -1) {
	pp_log_err("%s(): - open(%s) failed", ___F, filepath_tmp);
	return PP_ERR;
    }

    /* write poh counter to tmp file */
    if ((n = write(fd, &poh_counter, sizeof(unsigned int))) != sizeof(unsigned int)) {
	if (n == -1) {
	    pp_log_err("%s(): write(%s) failed", ___F, filepath_tmp);
	} else {
	    pp_log("%s(): write(%s) failed\n", ___F, filepath_tmp);
	}
	return PP_ERR;
    }

    /* close disk file */
    close(fd);

    /* rename */
    if (rename(filepath_tmp, POH_FILE) == -1) {
	pp_log("%s(): rename(%s, %s) failed\n", ___F, filepath_tmp, POH_FILE);
	return PP_ERR;
    }

    /* make sure everything is stored physically on disk */
    sync(); /* use sync; fsync is not enough - the directory must be fsynced, too! */

    return PP_SUC;
}

/********************************************************************
 * Chassis device command table
 */

static const dev_cmd_entry_t chassis_cmd_tab[] = {
    {
        .cmd_hndlr = chassis_cmd_get_caps,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_GET_CHASSIS_CAPS,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER,
    },
    {
        .cmd_hndlr = chassis_cmd_get_state,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_GET_CHASSIS_STATE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER,
    },
    {
        .cmd_hndlr = chassis_cmd_control,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_CHASSIS_CONTROL,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    {
        .cmd_hndlr = chassis_cmd_reset,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_CHASSIS_RESET,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    {
        .cmd_hndlr = chassis_cmd_identify,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_CHASSIS_IDENTIFY,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    /*
    {
        .cmd_hndlr = chassis_cmd_set_caps,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_SET_CHASSIS_CAPS,
        .min_data_size = 5, .min_priv_level = IPMI_PRIV_ADMIN,
    },
    */
    {
        .cmd_hndlr = chassis_cmd_set_power_restore_policy,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_SET_POWER_RESTORE_POLICY,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    {
        .cmd_hndlr = chassis_cmd_set_power_cycle_interval,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_SET_POWER_CYCLE_INTERVAL,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_ADMIN,
    },
    {
        .cmd_hndlr = chassis_cmd_get_poh_counter,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_GET_POH_COUNTER,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER,
    },
    {
        .cmd_hndlr = chassis_cmd_get_system_restart_cause,
        .netfn = IPMI_NETFN_CHASSIS, 
        .cmd = IPMI_CMD_GET_SYSTEM_RESTART_CAUSE,
        .min_data_size = 0, 
        .min_priv_level = IPMI_PRIV_USER,
    },
    {
        .cmd_hndlr = chassis_cmd_set_system_boot_options,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_SET_SYSTEM_BOOT_OPTIONS,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    {
        .cmd_hndlr = chassis_cmd_get_system_boot_options,
        .netfn = IPMI_NETFN_CHASSIS, .cmd = IPMI_CMD_GET_SYSTEM_BOOT_OPTIONS,
        .min_data_size = 3, .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Chassis device c'tor/d'tor
 */

int pp_bmc_dev_chassis_init()
{
    /* start POH timer */
    if (PP_ERR == poh_counter_load()) {
	pp_bmc_log_warn("[CHASSIS] poh counter load failed");
    }
    poh_timer_id = pp_select_add_to(POH_MINS_PER_CNT * 60 * 1000, 1, poh_timer, NULL);
    bzero(&boot_options, sizeof(boot_options_t));

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(chassis_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[CHASSIS] device started");
    return PP_SUC;
}

void pp_bmc_dev_chassis_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(chassis_cmd_tab);

    pp_select_remove_to(poh_timer_id);

    pp_bmc_log_info("[CHASSIS] device shut down");
}
