/**
 * \file bmc_dev_event.c
 *
 * Description: BMC Event Device
 *
 * (c) 2005 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include "pp/base.h"
#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/ipmi_sess.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_str.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/debug.h"
#include "bmc_dev_event.h"
#include "event_receiver.h"

/********************************************************************
 * Event command handlers
 */

/*
 * Platform Event Message
 */

struct platform_event_msg_rq_s {
    unsigned char ev_m_rev; /* shall be 0x04 (0x03 for IPMI v1.0) */
    unsigned char sensor_type;
    unsigned char sensor_no;
    BITFIELD2(unsigned char, event_type : 7, event_dir : 1);
    unsigned char event_data[3];
} __attribute__ ((packed));
typedef struct platform_event_msg_rq_s platform_event_msg_rq_t;

static int event_cmd_platform_event_msg(imsg_t* imsg)
{
    platform_event_msg_rq_t *rq = (void*)imsg->data;

    if (imsg->chan == IPMI_CHAN_SI) {
        rq = (void*)(imsg->data + 1);
        pp_bmc_receive_event(
            1, imsg->data[0], 0, imsg->chan,
            rq->sensor_type, rq->sensor_no,
            rq->event_dir, rq->event_type,
            rq->event_data[0], rq->event_data[1], rq->event_data[2]);
    } else {
        rq = (void*)imsg->data;
        pp_bmc_receive_event(
            0, imsg->rq_addr, imsg->rq_lun, imsg->chan,
            rq->sensor_type, rq->sensor_no,
            rq->event_dir, rq->event_type,
            rq->event_data[0], rq->event_data[1], rq->event_data[2]);
    }

    pp_bmc_log_info("[EVENT] generated: chan=%d sensor=%d(%s) event='%s %s'",
        imsg->chan,
        rq->sensor_no,
        ipmi_get_sensor_type_string(rq->sensor_type),
        ipmi_get_reading_name(rq->event_type, rq->sensor_type, rq->event_data[1]),
        ipmi_get_event_dir_string(rq->event_dir));

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/********************************************************************
 * Chassis device command table
 */

static const dev_cmd_entry_t event_cmd_tab[] = {
    {
        .cmd_hndlr = event_cmd_platform_event_msg,
        .netfn = IPMI_NETFN_SENSOR_EVENT, .cmd = IPMI_CMD_PLATFORM_EVENT_MSG,
        .min_data_size = sizeof(platform_event_msg_rq_t), .min_priv_level = IPMI_PRIV_OPERATOR,
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Event device c'tor/d'tor
 */

int pp_bmc_dev_event_init()
{
    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(event_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[EVENT] device started");
    return PP_SUC;
}

void pp_bmc_dev_event_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(event_cmd_tab);

    pp_bmc_log_info("[EVENT] device shut down");
}
