/**
 * \file bmc_dev_fru.h
 *
 * Description: BMC Field Replaceable Unit Inventory Device
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <pp/base.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_fru.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/bmc_fru.h>
#include "bmc_dev_fru.h"

#ifdef PRODUCT_SMIDC
  static const char FRU_FILE_MAGIC[] = "SMC-BMC-FRU"; // changed by Edward Pan
  #define FRU_FILE_VER   1
  #define MAX_FRU_SIZE 0x800
  #define IPMI_ERROR_FRU_PROTECTED_AREA	0x80
#else
  static const char FRU_FILE_MAGIC[] = "PPBMC-FRU"; // changed by Edward Pan
  #define FRU_FILE_VER   1
  #define MAX_FRU_SIZE 0x1fff
#endif  

#ifdef PRODUCT_SMIDC
// added by Edward Pan
char fru0_access_lock_flag = 1;  // added by Edward
#endif

struct fru_file_header_s {
    char magic[16];
    int ver;
    int offs; /* offset to data: array of ipmi_fru_header_t */
} __attribute__ ((packed));
typedef struct fru_file_header_s fru_file_header_t;

/* static data */
static struct {
    char *file_name;
} fru = { NULL };

/*
 * FRU device helpers
 */

static int fru_storage_create(void)
{
    assert(fru.file_name);
    int fd = open(fru.file_name, O_WRONLY | O_CREAT, S_RW_RW_RW_);
    if (fd == -1) return PP_ERR; /* file not createable */

    /* write empty file with header */
    fru_file_header_t filehdr;
    memset(filehdr.magic, 0, sizeof(filehdr.magic));
    strcpy(filehdr.magic, FRU_FILE_MAGIC);
    filehdr.ver = FRU_FILE_VER;
    filehdr.offs = 0;
    write(fd, &filehdr, sizeof(filehdr));

    { /* the FRU header */
        ipmi_fru_header_t fru_hdr;
        
#ifdef PRODUCT_SMIDC
        fru_hdr.reserved = 0x0;
        fru_hdr.version = IPMI_FRU_HEADER_VER;
        fru_hdr.offset.internal = 0x00;
        fru_hdr.offset.chassis = 0x00;
        fru_hdr.offset.board = 0x01;    // use different board offset
        fru_hdr.offset.product = 0x08;  // use different product offset
        fru_hdr.offset.multi = 0x00;
        fru_hdr.pad = 0x00;
        fru_hdr.checksum = 0xF6; /* zero checksum */
#else
        fru_hdr.reserved = 0x0;
        fru_hdr.version = IPMI_FRU_HEADER_VER;
        fru_hdr.offset.internal = 0x00;
        fru_hdr.offset.chassis = 0x00;
        fru_hdr.offset.board = 0x00;
        fru_hdr.offset.product = 0x00;
        fru_hdr.offset.multi = 0x00;
        fru_hdr.pad = 0x00;
        fru_hdr.checksum = 0xFE; /* zero checksum */
#endif
        write(fd, &fru_hdr, sizeof(ipmi_fru_header_t));
    }
#if 0 /* LARA_KIMMSI */
    { /* the MSI 9136 FRU data */
        static const unsigned char fru9136[104] =
                            { 0x01, 0x01, 0x02, 0x04, 0x08, 0x00, 0x00, 0xF0,
                              0x01, 0x00, 0x01, 0x04, 0x00, 0x00, 0x03, 0xF7,
                              0x01, 0x02, 0x17, 0xC4, 0x39, 0x31, 0x33, 0x36,
                              0xC0, 0xC0, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x0D,
                              0x01, 0x04, 0x00, 0x00, 0x00, 0x00, 0xC3, 0x4D,
                              0x53, 0x49, 0xC4, 0x39, 0x31, 0x33, 0x36, 0xC0,
                              0xC7, 0x4D, 0x53, 0x2D, 0x39, 0x31, 0x33, 0x36,
                              0xC0, 0xC0, 0xC1, 0x00, 0x00, 0x00, 0x00, 0x50,
                              0x01, 0x05, 0x00, 0xC3, 0x53, 0x50, 0x44, 0xC7,
                              0x4D, 0x53, 0x49, 0x39, 0x31, 0x33, 0x36, 0xC7,
                              0x4D, 0x53, 0x2D, 0x39, 0x31, 0x33, 0x36, 0xC3,
                              0x31, 0x30, 0x30, 0xC0, 0xC0, 0xC0, 0xC0, 0xC1,
                              0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4F };

        write(fd, &fru9136, sizeof(fru9136));
    }
#endif /* LARA_KIMMSI  */
#ifdef PRODUCT_SMIDC
 // changed by Ed
        static const unsigned char supermicro_fru[120] =
                            { 0x01, 0x07, 0x00, 0x00, 0x00, 0x00, 0xCB, 0x53,
                              0x75, 0x70, 0x65, 0x72, 0x20, 0x4D, 0x69, 0x63,
                              0x72, 0x6F, 0xC7, 0x49, 0x50, 0x4D, 0x49, 0x32,
                              0x2E, 0x30, 0xCA, 0x20, 0x20, 0x20, 0x20, 0x20,
                              0x20, 0x20, 0x20, 0x20, 0x20, 0xCC, 0x41, 0x4F,
                              0x43, 0x2D, 0x49, 0x50, 0x4D, 0x49, 0x32, 0x30,
                              0x2D, 0x45, 0xC0, 0xC1, 0x00, 0x00, 0x00, 0x24,
                              0x01, 0x07, 0x00, 0xCB, 0x53, 0x75, 0x70, 0x65,
                              0x72, 0x20, 0x4D, 0x69, 0x63, 0x72, 0x6F, 0xC7,
                              0x49, 0x50, 0x4D, 0x49, 0x32, 0x2E, 0x30, 0xCC,
                              0x41, 0x4F, 0x43, 0x2D, 0x49, 0x50, 0x4D, 0x49,
                              0x32, 0x30, 0x2D, 0x45, 0xC3, 0x31, 0x2E, 0x30,
                              0xCA, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
                              0x20, 0x20, 0x20, 0xC0, 0xC0, 0xC1, 0x00, 0x12,
                              0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

        write(fd, &supermicro_fru, sizeof(supermicro_fru));
  // end of changed
#endif

    close(fd);

    return PP_SUC;
}

static int fru_storage_check(void)
{
    assert(fru.file_name);
    int fd = open(fru.file_name, O_RDONLY);
    if (fd == -1) return PP_ERR; /* not existent or not accessible */

    /* read header */
    fru_file_header_t filehdr;
    ssize_t s = read(fd, &filehdr, sizeof(filehdr));
    close(fd);

    if (s != sizeof(filehdr)
    || 0 != strcmp(filehdr.magic, FRU_FILE_MAGIC)
    || filehdr.ver != FRU_FILE_VER)
    {
        /* bad version or corrupted */
        /* (conversion/repairing goes here in future versions) */
        return PP_ERR;
    }
    return PP_SUC;
}

/********************************************************************
 * FRU command handlers
 */

/*
 * Get FRU Inventory Area Info
 */

struct get_fru_inv_area_info_rs_s {
    unsigned short size_le16;
    BITFIELD2(unsigned char, words : 1, reserved : 7);
} __attribute__ ((packed));
typedef struct get_fru_inv_area_info_rs_s get_fru_inv_area_info_rs_t;

static int fru_cmd_get_fru_inv_area_info(imsg_t *imsg)
{
    if (imsg->data[0] != 0)
    {
        pp_bmc_log_error("[FRU] bad device id %d", imsg->data[0]);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    get_fru_inv_area_info_rs_t *rs =
        pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_fru_inv_area_info_rs_t));

    rs->size_le16 = cpu_to_le16(MAX_FRU_SIZE);
    rs->words = 0;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Read FRU Data
 */

struct read_fru_data_rq_s {
    unsigned char dev_id;
    unsigned short offs_le16;
    unsigned char cnt;
} __attribute__ ((packed));
typedef struct read_fru_data_rq_s read_fru_data_rq_t;

struct read_fru_data_rs_s {
    unsigned char cnt;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct read_fru_data_rs_s read_fru_data_rs_t;

static int fru_cmd_read_fru_data(imsg_t *imsg)
{
    read_fru_data_rq_t *rq = (void*)imsg->data;
    int offs = le16_to_cpu(rq->offs_le16);
    int cnt = rq->cnt;
    
    /* check FRU device id: only 0 supported */
    if (rq->dev_id != 0)
    {
        pp_bmc_log_error("[FRU] bad device id %d", rq->dev_id);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    read_fru_data_rs_t *rs = (void*)pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
        sizeof(read_fru_data_rs_t) + cnt);

    rs->cnt = cnt;
    // init all vals with 0, in case client wants to read more than currently available
    bzero(rs->data, cnt); 
    if (pp_bmc_fru_read_data(rs->data, offs, cnt) == PP_ERR) {
        pp_bmc_log_error("[FRU] error reading FRU (devid=%d offs=%d cnt=%d)",
                        rq->dev_id, offs, cnt);
        pp_bmc_imsg_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    return pp_bmc_router_send_msg(imsg);
}

/**
 * read a piece of fru data of given length into given buffer
 * and return the number of actually read bytes.
 * return -1 (PP_ERR) if error.
 */
int pp_bmc_fru_read_data(void* buf,  off_t offset, size_t cnt) {
    int n, fd;
    fru_file_header_t filehdr;
    
    assert(fru.file_name);
    
    /* open file */
    if (0 > (fd = open(fru.file_name, O_RDONLY))) {
        pp_bmc_log_perror("[FRU] can't open storage file %s:", fru.file_name);
	return PP_ERR;
    }
    
    /* read header */
    if (read(fd, &filehdr, sizeof(filehdr)) != sizeof(filehdr)) {
        close(fd);
        pp_bmc_log_error("[FRU] corrupted storage file %s:", fru.file_name);
        return PP_ERR;
    } else {
        pp_bmc_log_debug("[FRU] %d bytes read from %s (filehdr)",
                        sizeof(filehdr), fru.file_name);
    }
    
    if(lseek(fd, filehdr.offs + offset, SEEK_CUR) == (off_t)-1) {
        pp_bmc_log_error("[FRU] seeking %d + %d bytes failed for file %s:",
                         (int)filehdr.offs, offset, fru.file_name);
    } else {
        pp_bmc_log_debug("[FRU] seeked %d + %d bytes in %s", 
                        (int)filehdr.offs, offset, fru.file_name);
    }
    
    /* read file */
    n = read(fd, buf, cnt);
    if (n < 0) {
        pp_bmc_log_error("[FRU] read error (%u bytes requested from %s)", cnt, fru.file_name);
        close(fd);
        return PP_ERR;
    }

    pp_bmc_log_debug("[FRU]: %d bytes read", n);
    close(fd);
    return n;
}

/*
 * Write FRU Data
 */

struct write_fru_data_rq_s {
    unsigned char dev_id;
    unsigned short offs_le16;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct write_fru_data_rq_s write_fru_data_rq_t;

static int fru_cmd_write_fru_data(imsg_t *imsg)
{
    assert(fru.file_name);

#ifdef PRODUCT_SMIDC
    if(fru0_access_lock_flag == 1) {
        pp_bmc_log_error("[FRU] fru writing area is protected");  
        return pp_bmc_router_resp_err(imsg, IPMI_ERROR_FRU_PROTECTED_AREA);
    }
#endif
    
    /* check FRU device id: only 0 supported */
    if (imsg->data[0] != 0)
    {
        pp_bmc_log_error("[FRU] bad device id %d", imsg->data[0]);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    write_fru_data_rq_t *rq = (void*)imsg->data;
    int offs = le16_to_cpu(rq->offs_le16);
    int cnt = imsg->data_size - sizeof(write_fru_data_rq_t);

    /* open file */
    int fd = open(fru.file_name, O_RDWR);
    if (fd == -1) {
        pp_bmc_log_perror("[FRU] can't open storage file %s", fru.file_name);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }

    /* read header */
    fru_file_header_t filehdr;
    if (read(fd, &filehdr, sizeof(filehdr)) != sizeof(filehdr)) {
        close(fd);
        pp_bmc_log_error("[FRU] corrupted storage file %s", fru.file_name);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
    }
            
    lseek(fd, filehdr.offs + offs, SEEK_CUR);

    /* write file */
    unsigned char size = write(fd, rq->data, cnt);
					 
    if (size != cnt) imsg->data[0] = IPMI_ERR_UNSPECIFIED;
    close(fd);
    
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &size, sizeof(size));
}

/********************************************************************
 * FRU device command table
 */

static const dev_cmd_entry_t fru_cmd_tab[] = {
    {
        .cmd_hndlr = fru_cmd_get_fru_inv_area_info,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_FRU_INV_AREA_INFO,
        .min_data_size = 1, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = fru_cmd_read_fru_data,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_READ_FRU_DATA,
        .min_data_size = sizeof(read_fru_data_rq_t), .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = fru_cmd_write_fru_data,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_WRITE_FRU_DATA,
        .min_data_size = sizeof(write_fru_data_rq_t), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * FRU device c'tor/d'tor
 */

int pp_bmc_dev_fru_init()
{
    /* build file name of persistance storage */
    fru.file_name = malloc(strlen(PP_BMC_FLASH_ROOT) + sizeof("/fru"));
    strcat(strcpy(fru.file_name, PP_BMC_FLASH_ROOT), "/fru");

    /* check existance and version of persistan storage, and try to create it */
    if (PP_ERR == fru_storage_check()) {
        if (errno != ENOENT) {
            pp_bmc_log_error("[FRU] corrupted storage file %s: %s", fru.file_name, pp_error_string(errno));
            return PP_ERR;
        }
        if (PP_ERR == fru_storage_create()) {
            pp_bmc_log_error("[FRU] can't create storage file %s: %s", fru.file_name, pp_error_string(errno));
            return PP_ERR;
        }
    }

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(fru_cmd_tab)) return PP_ERR;
   
    pp_bmc_log_info("[FRU] device started");
    return PP_SUC;
}

void pp_bmc_dev_fru_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(fru_cmd_tab);

    free(fru.file_name); fru.file_name = NULL;
    
    pp_bmc_log_info("[FRU] device shut down");
}
