/**
 * bmc_dev_oem_pp_rpc.c
 *
 * Description: BMC Peppercon OEM commands for RPC
 *
 * (c) 2005 Peppercon AG, Thomas Breitfeld <thomas@peppercon.de>
 */

#include <values.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/selector.h>
#include <pp/hal_rpc.h>

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/host_sensors.h>
#include <pp/bmc/rpc_olled_ctrl.h>
#include <pp/um.h>

#include "bmc_dev_oem_pp_rpc.h"

static const char* bmc_rpc_outlet_state_str = "bmc.rpc.outlet[%u].state";
static const char* bmc_rpc_pwr_on_delay_str = "bmc.rpc.pwr_on_delay";

/*
 * outlet queue implementation
 * =============================
 * philosophy is as follows:
 * - queue has first and second half, a high prio and a low prio half
 * - each single outlet switch on will be queued into high prio half 
 * - each group outlet switch on will be queued into low prio
 * - a single outlet switch on will have precedence over a group switch on
 *   the group switch on will be removed, delay time of previous cmd will
 *   be adjusted so the group requirement will be meat
 * - if two group switch on commands deal with same outlets only the
 *   switch commands of the first group in queue will remain there
 * - outlet switch off will be removed instantly from the queue
 *   no delay time it going to be adjusted, as outlet will not switch
 */

/* -- outlet queue declarations -- */
#define OLQ_TIMER_ACTIVE() (olqueue_timeout_id != 0)

typedef struct olq_entry_s {
    struct list_head lhook;
    int outlet;
    unsigned int t_delay_ms;
} olq_entry_t;

static inline olq_entry_t* olq_entry_create(int ol, int predelay_ms);
static inline void olq_entry_destroy(olq_entry_t* ole);
static inline int olq_set_timer(int delay_ms);
static inline void olq_cancel_timer(void);
static unsigned int olq_twaited_ms(struct timeval* now);
static inline void olq_process_next(void);
static inline void olq_enqueue_hp(olq_entry_t* e);
static inline void olq_enqueue_lp(olq_entry_t* e);
static inline olq_entry_t* olq_dequeue(void);
static inline olq_entry_t* olq_first(void);
static inline olq_entry_t* olq_lp_first(void);
static inline olq_entry_t* olq_remove(olq_entry_t* ole);
static void olq_cleanup(void);
static olq_entry_t* olq_search_outlet(int outlet);
static olq_entry_t* olq_search_hp_outlet(int outlet);
static olq_entry_t* olq_search_lp_outlet(int outlet);
static olq_entry_t* olq_search_outlet_base(int outlet,
					   struct list_head* start,
					   struct list_head* end);
static olq_entry_t* olq_remove_outlet(int outlet);
static int olq_timeout_hndl(const int item_id, void* ctx);
static void outlet_switch(int outlet, int state);

static void rpc_outlet_off(int outlet);
static void rpc_outlet_on(int outlet);
static void rpc_outlet_switch(int outlet, int state);
static void rpc_outlet_on_lowprio(int outlet, int predelay_ms);
static inline unsigned char rpc_oldelay_ms_2_ipmi(int oldelay_ms);
static inline int rpc_ipmi_2_oldelay_ms(unsigned char delay);

/* -- private outlet queue data -- */
static struct list_head  olqueue = LIST_HEAD_INIT(olqueue);
static struct list_head* olq_hp_end = &olqueue;
static struct timeval olq_tlast = { 0, 0};
static int olqueue_timeout_id = 0;

/* -- outlet queue implementation -- */

static inline olq_entry_t* olq_entry_create(int outlet, int t_delay_ms) {
    olq_entry_t* ole = malloc(sizeof(olq_entry_t));
    ole->outlet = outlet;
    ole->t_delay_ms = t_delay_ms;
    return ole;
}

static inline void olq_entry_destroy(olq_entry_t* ole) {
    free(ole);
}

static inline int olq_set_timer(int delay_ms) {
    return pp_select_add_to(delay_ms, 0, olq_timeout_hndl, NULL);
}

static inline void olq_cancel_timer(void) {
    assert(olqueue_timeout_id != 0);
    pp_select_remove_to(olqueue_timeout_id);
    olqueue_timeout_id = 0;
}

static unsigned int olq_twaited_ms(struct timeval* tnow) {
    int64_t tdiff;
    unsigned int t_waited_ms;
    timeval_now(tnow);
    if ((tdiff = timeval_diff_msec(tnow, &olq_tlast)) < 0) {
	olq_tlast = *tnow; /* system time was probably adjusted */
	t_waited_ms = 0;
    } else if (tdiff > (long long)MAXINT) {
	t_waited_ms = MAXINT;
    } else {
	t_waited_ms = (unsigned int)tdiff;
    }
    return t_waited_ms;
}

static inline void olq_process_next() {
    assert(!OLQ_TIMER_ACTIVE());
    olq_timeout_hndl(0, NULL);
}

static inline void olq_enqueue_hp(olq_entry_t* e) {
    list_add(&e->lhook, olq_hp_end);
    olq_hp_end = &e->lhook;
}

static inline void olq_enqueue_lp(olq_entry_t* e) {
    list_add_tail(&e->lhook, &olqueue);
}

static inline olq_entry_t* olq_dequeue() {
    if (list_empty(&olqueue)) return NULL;
    return olq_remove(list_entry(olqueue.next, olq_entry_t, lhook));
}

static inline olq_entry_t* olq_first() {
    if (list_empty(&olqueue)) return NULL;
    return list_entry(olqueue.next, olq_entry_t, lhook);
}

static inline olq_entry_t* olq_lp_first() {
    if (olq_hp_end->next == &olqueue) return NULL;
    return list_entry(olq_hp_end->next, olq_entry_t, lhook);
}

static inline olq_entry_t* olq_remove(olq_entry_t* ole) {
    assert(ole != NULL && &ole->lhook != &olqueue);
    if (olq_hp_end == &ole->lhook) { /* adjust hp_end endtry */
	olq_hp_end = ole->lhook.prev;
    }
    list_del(&ole->lhook);
    return ole;
}   

static void olq_cleanup() {
    olq_entry_t* ole;
    if (olqueue_timeout_id != 0) {
	olq_cancel_timer();
    }
    while ((ole = olq_dequeue()) != NULL) {
	olq_entry_destroy(ole);
    }
}

static inline olq_entry_t* olq_search_outlet(int outlet) {
    return olq_search_outlet_base(outlet, olqueue.next, &olqueue);
}

static inline olq_entry_t* olq_search_hp_outlet(int outlet) {
    return olq_search_outlet_base(outlet, olqueue.next, olq_hp_end->next);
}

static inline olq_entry_t* olq_search_lp_outlet(int outlet) {
    return olq_search_outlet_base(outlet, olq_hp_end->next, &olqueue);
}

static olq_entry_t* olq_search_outlet_base(int outlet,
					   struct list_head* start,
					   struct list_head* end) {
    struct list_head* ehook;
    olq_entry_t* ole, *ret = NULL;
    for (ehook = start; ehook != end; ehook = ehook->next) {
	ole = list_entry(ehook, olq_entry_t, lhook);
	if (ole->outlet == outlet) {
	    ret = ole;
	    break;
	}
    }
    return ret;
}

static olq_entry_t* olq_remove_outlet(int outlet) {
    olq_entry_t* ole;
    if (NULL != (ole = olq_search_outlet(outlet))) {
	olq_remove(ole);
	assert(olq_search_outlet(outlet) == NULL); /* no duplicates check */
    }
    return ole;
}

static int olq_timeout_hndl(const int item_id UNUSED, void* ctx UNUSED) {
    struct timeval tnow;
    olq_entry_t* ole;
    int t_towait_ms;

    if (NULL != (ole = olq_first())) {
	if ((t_towait_ms = ole->t_delay_ms - olq_twaited_ms(&tnow)) <= 10) {
	    outlet_switch(ole->outlet, 0x1);
	    olq_entry_destroy(olq_dequeue());
	    olq_tlast = tnow;
	    if (NULL != (ole = olq_first())) {
		assert(ole->t_delay_ms != 0);
		olqueue_timeout_id = olq_set_timer(ole->t_delay_ms);
	    } else {
		olqueue_timeout_id = 0;
	    }
	} else {
	    olqueue_timeout_id = olq_set_timer(t_towait_ms);
	}
    } else {
	olqueue_timeout_id = 0;
    }
    return PP_SUC;
}

static void outlet_switch(int outlet, int state) {
    static const char* fn = ___F;
    int actor_no = PP_ACTOR_R0_STATE + outlet;
    pp_tp_gpio_act_t* outlet_act;
    
    pp_bmc_log_debug("%s: %d->%s", fn, outlet, state ? "on" : "off");
    if ((outlet_act = pp_bmc_host_sensor_get_gpio_actor(actor_no)) != NULL) {
	pp_tp_gpio_act_set_gpio(outlet_act, state);
    } else {
	pp_bmc_log_error("%s: no relay-act for outlet %d(%#x)",
			 fn, outlet, actor_no);
    }
}

/* principle:
 * remove existing queued outlet from high and low queue
 * and re-add in high queue. this has the effect that last switch command
 * will be overwritten by this latest switch command, what is 'do what I mean'
 */
static void rpc_outlet_on(int outlet) {
    struct timeval tnow;
    int t_global_delay_ms, t_towait_ms;
    olq_entry_t* ole;
    pp_cfg_get_int(&t_global_delay_ms, bmc_rpc_pwr_on_delay_str);
    olq_entry_destroy(olq_remove_outlet(outlet));
    olq_enqueue_hp(ole = olq_entry_create(outlet, t_global_delay_ms));
    if (!OLQ_TIMER_ACTIVE()) {
	olq_process_next();
    } else if (ole == olq_first() && (ole = olq_lp_first()) != NULL) {
	/* long lasting low prio timer is probably running, interrupt it */
	t_towait_ms = ole->t_delay_ms - olq_twaited_ms(&tnow)
	    - t_global_delay_ms;
	if (t_towait_ms < t_global_delay_ms) {
	    t_towait_ms = t_global_delay_ms;
	}
	ole->t_delay_ms = t_towait_ms;
	olq_cancel_timer();
	olq_process_next();
    }
}

/* principle:
 * if outlet is already queued in high queue, don't add again.
 * if outlet is queued in low queue remove and add again.
 * means: single switches have preference over group switches
 * latest group switch has preference over previous switch command.
 */
static void rpc_outlet_on_lowprio(int outlet, int predelay_ms) {
    if (olq_search_hp_outlet(outlet) != NULL) return;
    olq_entry_destroy(olq_remove_outlet(outlet));
    if (predelay_ms <= 0) {
	pp_cfg_get_int(&predelay_ms, bmc_rpc_pwr_on_delay_str);
    }
    olq_enqueue_lp(olq_entry_create(outlet, predelay_ms));
    if (!OLQ_TIMER_ACTIVE()) {
	olq_process_next();
    }
}

static void rpc_outlet_off(int outlet) {
    olq_entry_destroy(olq_remove_outlet(outlet));
    outlet_switch(outlet, 0);
}

static void rpc_outlet_switch(int outlet, int state) {
    if (state) rpc_outlet_on(outlet);
    else rpc_outlet_off(outlet);
}

static inline unsigned char rpc_oldelay_ms_2_ipmi(int oldelay_ms) {
    return (unsigned char)(oldelay_ms / 100);
}

static inline int rpc_ipmi_2_oldelay_ms(unsigned char delay) {
    return delay * 100;
}

/*
 * RPC OEM command implementation
 * ==============================
 */
/*
 * set/get power on delay
 * ------------------------
 */
typedef struct oem_pp_rpc_set_power_on_delay_rq_s {
    unsigned char delay;
} __attribute__ ((packed)) oem_pp_rpc_set_power_on_delay_rq_t;

static int oem_pp_rpc_set_power_on_delay(imsg_t* imsg) {
    oem_pp_rpc_set_power_on_delay_rq_t* rq = (void*)imsg->data;
    if (rq->delay == 0) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    pp_cfg_set_int(rpc_ipmi_2_oldelay_ms(rq->delay), bmc_rpc_pwr_on_delay_str);
    pp_cfg_save(DO_FLUSH);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

typedef struct oem_pp_rpc_get_power_on_delay_rq_s {
    unsigned char delay;
} __attribute__ ((packed)) oem_pp_rpc_get_power_on_delay_rq_t;

static int oem_pp_rpc_get_power_on_delay(imsg_t* imsg) {
    int delay;
    oem_pp_rpc_get_power_on_delay_rq_t* rq;
    rq = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
			  sizeof(oem_pp_rpc_get_power_on_delay_rq_t));	
    pp_cfg_get_int(&delay, bmc_rpc_pwr_on_delay_str);
    rq->delay = rpc_oldelay_ms_2_ipmi(delay);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * set/get receptacle state
 * ------------------------
 */
typedef struct oem_pp_rpc_set_receptacle_state_rq_s {
    unsigned char receptacle;
    unsigned char status;
} __attribute__ ((packed)) oem_pp_rpc_set_receptacle_state_rq_t;

static int oem_pp_rpc_set_receptacle_state(imsg_t* imsg) {
    oem_pp_rpc_set_receptacle_state_rq_t* rq = (void*)imsg->data;
    int new_state = rq->status & 0x01;

    if (rq->receptacle >= pp_hal_rpc_get_no_outlets()) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    rpc_outlet_switch(rq->receptacle, new_state);
    
    pp_cfg_set_enabled(new_state, bmc_rpc_outlet_state_str, rq->receptacle);
    pp_cfg_save(DO_FLUSH);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

typedef struct oem_pp_rpc_get_receptacle_state_rq_s {
    unsigned char outlet;
} __attribute__ ((packed)) oem_pp_rpc_get_receptacle_state_rq_t;

typedef struct oem_pp_rpc_get_receptacle_state_rs_s {
    BITFIELD8(unsigned char, 
              status : 1, 
              off_reason : 1, 
              reserved2 : 1,
              reserved3 : 1,
              green : 1,
              red : 1, 
              blinking : 1,
              reserved7 : 1);
} __attribute__ ((packed)) oem_pp_rpc_get_receptacle_state_rs_t;

static int oem_pp_rpc_get_receptacle_state(imsg_t* imsg) {
    oem_pp_rpc_get_receptacle_state_rq_t* rq = (void*)imsg->data;
    oem_pp_rpc_get_receptacle_state_rs_t* rs;
    int no_outlets = pp_hal_rpc_get_no_outlets();
    pp_tp_ctrl_t *ctrl;
    unsigned char led_state = 0x00;
    
    /* check controller number, has to be at most number of current outlets */
    if (rq->outlet >= no_outlets) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    
    /* get controller */
    if ((ctrl = pp_bmc_host_get_ctrl(rq->outlet + PP_BMC_CTRL_R0_LED))
	== NULL) {
	pp_bmc_log_error("%s: no controller for outlet %d", ___F, rq->outlet);
    } else if (!PP_TP_OBJ_IS_TYPE(PP_TP_OLLED_CTRL, ctrl)) {
	pp_bmc_log_error("%s: wrong controller for outlet %d",
			 ___F, rq->outlet);
    } else {
        led_state = pp_bmc_rpc_olled_ctrl_get_led_state(ctrl);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
			  sizeof(oem_pp_rpc_get_receptacle_state_rs_t));
    
    memset(rs, 0, sizeof(oem_pp_rpc_get_receptacle_state_rs_t));
    rs->status =        0; // FIXME: get from GPIO sensor
    rs->off_reason =    0; // TODO: off_reason 
    rs->green =          led_state       & 0x01; 
    rs->red =           (led_state >> 1) & 0x01; 
    rs->blinking =      (led_state >> 2) & 0x01; 

    return pp_bmc_router_send_msg(imsg);
}

/*
 * set group state
 * ------------------------
 */
typedef struct oem_pp_rpc_set_group_state_rq_s {
    unsigned char group;
    unsigned char status;
} __attribute__ ((packed)) oem_pp_rpc_set_group_state_rq_t;

static int oem_pp_rpc_set_group_state(imsg_t* imsg UNUSED) {
    oem_pp_rpc_set_group_state_rq_t* rq = (void*)imsg->data;
    int memb, memb_cnt, grp_delay, i;
     
    if (rq->group > 23 || rq->status > 1) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    pp_cfg_get_int(&memb_cnt, "bmc.rpc.outlet_groups[%u].members._s_",
		   rq->group);   
    pp_cfg_get_int(&grp_delay, "bmc.rpc.outlet_groups[%u].pwr_on_delay",
		   rq->group);

    for(i = 0; i < memb_cnt; i++) {
    	pp_cfg_get_int(&memb, "bmc.rpc.outlet_groups[%u].members[%u]",
		       rq->group, i);
	if (rq->status) {
	    rpc_outlet_on_lowprio(memb, i == 0 ? 0 : grp_delay);
	} else {
	    rpc_outlet_off(memb);
	}
    	pp_cfg_set_enabled(rq->status, bmc_rpc_outlet_state_str, memb);
    }
    pp_cfg_save(DO_FLUSH);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}


/*
 * set/get group membership command
 * --------------------------------
 */

typedef struct oem_pp_rpc_set_group_membership_rq_s {
    BITFIELD2(unsigned char,
	      group : 5,
	      rsvd1 : 3);
    BITFIELD2(unsigned char,
	      enabled : 1,
	      rsvd2 : 7);
    unsigned char memb[3];
} __attribute__ ((packed)) oem_pp_rpc_set_group_membership_rq_t;

static int oem_pp_rpc_set_group_membership(imsg_t* imsg) {
    oem_pp_rpc_set_group_membership_rq_t* rq = (void*)imsg->data;
    unsigned char group = rq->group;
    int i, j, cnt = 0;

    if (group > 23) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    if (rq->enabled) {
	pp_cfg_set_int(0, "bmc.rpc.outlet_groups[%u].members._s_", group);
	for (j = 0; j < 3; ++j) {
	    for (i = 0; i < 8; i++) {
		if((rq->memb[j] >> i) & 0x01) { 
		    pp_cfg_set_int((8 * j) + i,
				   "bmc.rpc.outlet_groups[%u].members[%u]",
				   group, cnt++);
		}
	    }
	}
	pp_cfg_set_int(cnt, "bmc.rpc.outlet_groups[%u].members._s_", group);
    } else { /* if disabled we delete from config system */
	pp_cfg_remove("bmc.rpc.outlet_groups[%u]", group);
    }
    pp_cfg_save(DO_FLUSH);
    
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

typedef struct oem_pp_rpc_get_group_membership_rq_s {
    unsigned char group;
} __attribute__ ((packed)) oem_pp_rpc_get_group_membership_rq_t;

typedef struct oem_pp_rpc_get_group_membership_rs_s {
    BITFIELD2(unsigned char,
	      enabled : 1,
	      rsvd2 : 7);
    unsigned char outlet7_0;
    unsigned char outlet15_8;
    unsigned char outlet23_16;
} __attribute__ ((packed)) oem_pp_rpc_get_group_membership_rs_t;

static int oem_pp_rpc_get_group_membership(imsg_t* imsg) {
    oem_pp_rpc_get_group_membership_rq_t* rq = (void*)imsg->data;
    oem_pp_rpc_get_group_membership_rs_t *rs;    
    unsigned char group = rq->group;
    int i, outlet, cnt;
    unsigned int memb = 0; 
     
    if (group > 23) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
			  sizeof(oem_pp_rpc_get_group_membership_rs_t));

    if (PP_ERR != pp_cfg_get_int_nodflt(&cnt,
			"bmc.rpc.outlet_groups[%u].members._s_", group)) {
	for (i = 0; i < cnt; ++i){ 
	    pp_cfg_get_int(&outlet, "bmc.rpc.outlet_groups[%u].members[%u]",
			   group,i);
	    memb |= 1 << outlet;
	}
	rs->outlet7_0 = (unsigned char)(memb & 0xFF);
	rs->outlet15_8 = (unsigned char)((memb >> 8) & 0xFF);
	rs->outlet23_16 = (unsigned char)((memb >> 16) & 0xFF);
	rs->enabled = 1;
    } 
    return pp_bmc_router_send_msg(imsg);
}

/*
 * set/get group power on delay command
 * ------------------------------------
 */
typedef struct oem_pp_rpc_set_group_power_on_delay_rq_s {
    unsigned char group;
    unsigned char delay;
} __attribute__ ((packed)) oem_pp_rpc_set_group_power_on_delay_rq_t;

static int oem_pp_rpc_set_group_power_on_delay(imsg_t* imsg) {
    oem_pp_rpc_set_group_power_on_delay_rq_t *rq = (void*)imsg->data;

    if (rq->group > 23) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    pp_cfg_set_int(rpc_ipmi_2_oldelay_ms(rq->delay),
		   "bmc.rpc.outlet_groups[%u].pwr_on_delay", rq->group);
    pp_cfg_save(DO_FLUSH);
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

typedef struct oem_pp_rpc_get_group_power_on_delay_rq_s {
    unsigned char  group;
} __attribute__ ((packed)) oem_pp_rpc_get_group_power_on_delay_rq_t;

typedef struct oem_pp_rpc_get_group_power_on_delay_rs_s {
    unsigned char  delay;
} __attribute__ ((packed)) oem_pp_rpc_get_group_power_on_delay_rs_t;

static int oem_pp_rpc_get_group_power_on_delay(imsg_t* imsg) {
    oem_pp_rpc_get_group_power_on_delay_rq_t *rq= (void*)imsg->data;    
    oem_pp_rpc_get_group_power_on_delay_rs_t *rs;
    int delay;
    
    if (rq->group > 23) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
    }
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
			  sizeof(oem_pp_rpc_get_group_power_on_delay_rs_t));
    pp_cfg_get_int(&delay,"bmc.rpc.outlet_groups[%u].pwr_on_delay",
		   rq->group);
    rs->delay = rpc_oldelay_ms_2_ipmi(delay);
    return pp_bmc_router_send_msg(imsg);
}

/*
 * set/get receptacle acl
 * ------------------------
 */
 
typedef struct oem_pp_rpc_set_receptacle_acl_rq_s {
    unsigned char outlet;
    unsigned char acl_no;	
    BITFIELD3(unsigned char, uid : 6, privge : 1, perm : 1);
} __attribute__ ((packed)) oem_pp_rpc_set_receptacle_acl_rq_t;

static int oem_pp_rpc_set_receptacle_acl(imsg_t* imsg) {
    char* user;
    char perm[11];
    int i, n;
	
    oem_pp_rpc_set_receptacle_acl_rq_t* rq = (void*)imsg->data;
    rq->outlet=imsg->data[0];
    sprintf(perm, "outlet_%d", rq->outlet);   
    n = imsg->data[1];
    for(i=0;i<n;i++) {
	rq->uid=imsg->data[i+2]&0x3f;
	rq->privge=((imsg->data[i+2]&0x40)>>6);
	rq->perm=((imsg->data[i+2]&0x80)>>7);	
	pp_cfg_get(&user, "user[%u].login", rq->uid);
	if(PP_ERR == pp_um_ipmi_user_set_permission(user, perm,
	    rq->perm ? pp_acl_raasip_yes_str : pp_acl_raasip_no_str)) {
	    pp_bmc_log_perror("Error setting acl entry.");
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
	pp_bmc_log_info("Set ACL user: %s, perm: %s, value %d",
			user, perm, rq->perm);
	pp_cfg_save(DO_FLUSH);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

typedef struct oem_pp_rpc_get_receptacle_acl_rq_s { 
    unsigned char num_acl;
    unsigned char acl[256];	
} __attribute__ ((packed)) oem_pp_rpc_get_receptacle_acl_rq_t;

static int oem_pp_rpc_get_receptacle_acl(imsg_t* imsg) {
    char* user;
    char perm[11];
    int i, is_allowed, pp_uid, outlet;

    oem_pp_rpc_get_receptacle_acl_rq_t* rq = (void*)imsg->data;
    outlet=imsg->data[0];
    sprintf(perm, "outlet_%d", outlet);   
    rq = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 20);	
    rq->num_acl = 0;
    vector_t *usrs = pp_um_get_all_users();
    for (i = 0; i < (int)vector_size(usrs); i++) {
	user = vector_get(usrs, i);
	pp_uid = pp_um_user_get_uid(user);
	is_allowed = pp_um_user_has_permission(user, perm,
						   pp_acl_raasip_yes_str);
	rq->acl[rq->num_acl ++] = ((is_allowed<<7)&0x80) + pp_uid;
    }
    vector_delete(usrs);
    imsg->data_size = rq->num_acl+2;
    //pp_bmc_log_info("num of ACL: %d", rq->num_acl);
    return pp_bmc_router_send_msg(imsg);
}


/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_pp_rpc_cmd_tab[] = {
    {
	.cmd_hndlr = oem_pp_rpc_set_power_on_delay,
	.netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_POWER_ON_DELAY,
	.min_data_size = sizeof(oem_pp_rpc_set_power_on_delay_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_get_power_on_delay,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_GET_POWER_ON_DELAY,
        .min_data_size = 0,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_set_receptacle_state,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_RECEPTACLE_STATE,
        .min_data_size = sizeof(oem_pp_rpc_set_receptacle_state_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_get_receptacle_state,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_GET_RECEPTACLE_STATE,
        .min_data_size = sizeof(oem_pp_rpc_get_receptacle_state_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    }, 
    {
        .cmd_hndlr = oem_pp_rpc_set_group_state,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_GROUP_STATE,
        .min_data_size = sizeof(oem_pp_rpc_set_group_state_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_set_group_membership,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_GROUP_MEMBERSHIP,
        .min_data_size = sizeof(oem_pp_rpc_set_group_membership_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_get_group_membership,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_GET_GROUP_MEMBERSHIP,
        .min_data_size = sizeof(oem_pp_rpc_get_group_membership_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_set_group_power_on_delay,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_GROUP_POWER_ON_DELAY,
        .min_data_size = sizeof(oem_pp_rpc_set_group_power_on_delay_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_get_group_power_on_delay,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_GET_GROUP_POWER_ON_DELAY,
        .min_data_size = sizeof(oem_pp_rpc_get_group_power_on_delay_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_set_receptacle_acl,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_SET_RECEPTACLE_ACL,
        .min_data_size = sizeof(oem_pp_rpc_set_receptacle_acl_rq_t),
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_rpc_get_receptacle_acl,
        .netfn = IPMI_NETFN_OEM_PP,
	.cmd = IPMI_CMD_PP_RPC_GET_RECEPTACLE_ACL,
        .min_data_size = 0,
	.min_priv_level = IPMI_PRIV_ADMIN
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */
static int rpc_recover_pwr_state_timeout_id;

static int rpc_recover_pwr_state_hndl (const int item_id UNUSED,
				       void* ctx UNUSED) {
    int i, outlet_is_on;
    int no_outlets = pp_hal_rpc_get_no_outlets();
    for (i = 0; i < no_outlets; ++i) {
	pp_cfg_is_enabled(&outlet_is_on, bmc_rpc_outlet_state_str, i);
	rpc_outlet_switch(i, outlet_is_on);
    }
    return PP_SUC;
}

int pp_bmc_dev_oem_pp_rpc_init()
{
    /* register all entries of cmd tab */
    if (PP_FAILED(pp_bmc_core_reg_cmd_tab(oem_pp_rpc_cmd_tab))) {
	return PP_ERR;
    }
    pp_bmc_log_info("[OEM-PP-RPC] device started");
    return PP_SUC;
}


void pp_bmc_dev_oem_pp_rpc_recover_pwr_state(void) {
    /* do it only after scanner is running and states are probably known */
    rpc_recover_pwr_state_timeout_id = pp_select_add_to(3000, 0,
				  rpc_recover_pwr_state_hndl, NULL);
}

void pp_bmc_dev_oem_pp_rpc_cleanup()
{
    /* remove pending switch commands */
    olq_cleanup();

    /* cancel recover state timeout */
    if (rpc_recover_pwr_state_timeout_id != 0) {
	pp_select_remove_to(rpc_recover_pwr_state_timeout_id);
	rpc_recover_pwr_state_timeout_id = 0;
    }
    
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_pp_rpc_cmd_tab);

    pp_bmc_log_info("[OEM-PP-RPC] device shut down");
}
