/**
 * bmc_dev_oem_pp_self-test.c
 *
 * Description: BMC Peppercon OEM commands for self-test
 *
 * (c) 2006 Raritan, <ralf.guenther@raritan.com>
 */

#include <pp/base.h>
#include <pp/vsc.h>
#include <pp/vsc_types_external.h>
#include <pp/grab.h>
#include <liberic_pthread.h>
#include <liberic_net.h>
#include <pthread.h>
#include <zlib.h>
#include <assert.h>
#include <pp/ipmi.h>

#ifdef PP_FEAT_USB
# include <pp/usb.h>
#endif

#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_msg.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/debug.h>

#include <fcntl.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <net/if_packet.h>
#include <netpacket/packet.h>
#include <linux/if.h>
#include <linux/sockios.h>
#include <linux/if_ether.h>

#ifdef PRODUCT_ASMIDC
# include <fml.h>
# include <pp_i2c.h>
#endif

// hack, to be able to use following kernel includes
#define _LINUX_IF_H // prevent linux/if.h from being used
typedef unsigned long long u64;
typedef __uint32_t u32;
typedef __uint16_t u16;
typedef __uint8_t u8;
#include <linux/mii.h>
#include <linux/ethtool.h>

#include "bmc_dev_oem_pp_selftest.h"

/********************************************************************
 * Types and Constants
 */

#define IPMI_ERR_SLFTST_NOT_SUPPORTED   0x80
#define IPMI_ERR_SLFTST_FAILED          0x81
#define IPMI_ERR_SLFTST_BUSY            0x82
#define IPMI_ERR_SLFTST_NO_RESULTS      0x83

#define IPMI_OEM_PP_SLFTST_VIDEO            1
#define IPMI_OEM_PP_SLFTST_VIDEO_STATUS         1
#define IPMI_OEM_PP_SLFTST_VIDEO_CRC            2
#define IPMI_OEM_PP_SLFTST_DDC              2
#define IPMI_OEM_PP_SLFTST_DDC_INFO             1
#define IPMI_OEM_PP_SLFTST_USB              3
#define IPMI_OEM_PP_SLFTST_USB_OPERATION        1
#define IPMI_OEM_PP_SLFTST_NIC              4
#define IPMI_OEM_PP_SLFTST_NIC_STATUS           1
#define IPMI_OEM_PP_SLFTST_NIC_LOOPBACK         2
#define IPMI_OEM_PP_SLFTST_NIC_PING             3
#define IPMI_OEM_PP_SLFTST_NIC_BCASTPING        4
#define IPMI_OEM_PP_SLFTST_SERIAL           5
#define IPMI_OEM_PP_SLFTST_SERIAL_ECHO          1
#define IPMI_OEM_PP_SLFTST_IPMB             6
#define IPMI_OEM_PP_SLFTST_IPMB_OPERATION       1
#define IPMI_OEM_PP_SLFTST_IPMB_EVALBOARD       2
#define IPMI_OEM_PP_SLFTST_FML              7
#define IPMI_OEM_PP_SLFTST_FML_OPERATION        1
#define IPMI_OEM_PP_SLFTST_FML_EVALBOARD        2

struct oem_pp_selftest_s {
    unsigned char comp; // see IPMI_OEM_PP_SLFTST_XXXX
    unsigned char test; // see IPMI_OEM_PP_SLFTST_XXXX_YYYY
    unsigned char data[0]; // additional params
} __attribute__ ((packed));
typedef struct oem_pp_selftest_s oem_pp_selftest_t;

struct oem_pp_selftest_video_status_rs_s {
    unsigned short xres_le16;
    unsigned short yres_le16;
    unsigned short hfreq_le16; // khz
    unsigned short vfreq_le16; // hz
} __attribute__ ((packed));
typedef struct oem_pp_selftest_video_status_rs_s oem_pp_selftest_video_status_rs_t;

struct oem_pp_selftest_nic_status_rs_s {
    BITFIELD3(unsigned char, speed : 4, duplex : 3, link : 1);
} __attribute__ ((packed));
typedef struct oem_pp_selftest_nic_status_rs_s oem_pp_selftest_nic_status_rs_t;

/********************************************************************
 * Static Vars
 */

static pthread_mutex_t slftst_lock;
static pthread_t slftst_thread;

// will contain params at call time and results once self-test has been finished
static struct {
    unsigned char cc;
    unsigned char comp;
    unsigned char test;
    unsigned char *data;
    unsigned char datalen;
} g_slftst = { .comp = 0, .data = NULL, };

/********************************************************************
 * Helpers
 */

static unsigned char* slftst_set_datalen(int len)
{
    if (g_slftst.data) free(g_slftst.data);
    g_slftst.datalen = len;
    if (len) {
        g_slftst.data = malloc(len);
        memset(g_slftst.data, 0, len);
    } else g_slftst.data = NULL;

    return g_slftst.data;
}

static int slftst_start(const oem_pp_selftest_t *rq, int datalen)
{
    if (pthread_mutex_trylock(&slftst_lock) == 0) {
        // release any result from previous tests and init results
        g_slftst.comp = rq->comp;
        g_slftst.test = rq->test;
        memcpy(slftst_set_datalen(datalen), rq->data, datalen);
        return PP_SUC;
    }
    return PP_ERR;
}

static void slftst_complete(unsigned char cc, unsigned char* data, int datalen)
{
    g_slftst.cc = cc;
    free(g_slftst.data);
    g_slftst.data = data;
    g_slftst.datalen = datalen;
    pthread_mutex_unlock(&slftst_lock);
}

static int slftst_is_running(void)
{
    if (pthread_mutex_trylock(&slftst_lock) != 0) return 1;
    pthread_mutex_unlock(&slftst_lock);
    return 0;
}


static unsigned long calc_image_crc(unsigned long crc,
                                    const unsigned short* buf, int w, int h)
{
    const unsigned short* p;
    int x, y;
    for (y = 0; y < h; y++) {
        p = buf + w * y / 16 + 16 * (y % 16);
        for (x = 0; x < w; x += 16, p += 16 * 16) {
            crc = crc32(crc, (const unsigned char*)p, 
                        16 * sizeof(unsigned short));
        }
    }
    return crc;
}

static void* slftst_video_crc_func(void* arg UNUSED)
{
    unsigned char ret = IPMI_ERR_SLFTST_FAILED;
    pp_grab_client_t *c = NULL;
    unsigned long *crc_le32 = (unsigned long*)malloc(sizeof(unsigned long));

    // get resolution
    fb_format_info_t fb_format;
    if (pp_vsc_get_fb_format(0, &fb_format) != 0) {
        pp_bmc_log_error("[OEM-PP-SLFTST] unable to get video format");
        goto fail;
    }
    pp_bmc_log_debug("[OEM-PP-SLFTST] resolution: w=%d h=%d", 
                    fb_format.g_w, fb_format.g_h);

    // ensure color depth of 16bit!
    fb_color_info_t fb_cinfo;
    if (pp_grab_get_color_info(0, &fb_cinfo) != 0
        || fb_cinfo.bpp != 16) {
        pp_bmc_log_error("[OEM-PP-SLFTST] bad color format");
        goto fail;
    }

    // calc image crc
    c = pp_grab_new_client(0, GRAB_VARIABLE_MEM_DESC);
    if (c == NULL) {
        pp_bmc_log_error("[OEM-PP-SLFTST] unable to register as grab client");
        goto fail;
    }

    // grab one frame and calc crc
    unsigned long crc = crc32(0, Z_NULL, 0);
    BoxRec fullbox = { .x1 = 0, 
                       .y1 = 0, 
                       .x2 = fb_format.g_w - 1, 
                       .y2 = fb_format.g_h - 1, };
    RegionRec req, ack, rem;
    REGION_INIT(&req, NullBox, 0);
    REGION_INIT(&ack, NullBox, 0);
    REGION_INIT(&rem, &fullbox, 0);

    do {
        // grab remaining region
        REGION_COPY(&req, &rem);
        REGION_EMPTY(&ack);
        REGION_EMPTY(&rem);
        if (pp_grab_request_region(c, GRAB_REQ_FLAGS_IGNORE_DIFFMAP,
                                  &req, &ack, &rem) != 0) {
            pp_bmc_log_error("[OEM-PP-SLFTST] error while grabbing");
            goto fail;
        }

        // validate actually grabbed region
        assert(REGION_NUM_RECTS(&ack) == 1);
        BoxPtr ackbox = REGION_RECTS(&ack);
        assert(ackbox->x1 == 0);
        assert(ackbox->x2 + 1 == (int)fb_format.g_w);
        assert((ackbox->y1 % 16) == 0);
        assert(ackbox->y2 + 1 <= (int)fb_format.g_h);

        // continue crc calculation
        crc = calc_image_crc(crc, (unsigned short*)c->buf, 
                             fb_format.g_w, ackbox->y2 - ackbox->y1 + 1);

        // release memory
        pp_grab_release_buffer(c);
    } while (REGION_NOTEMPTY(&rem));

    REGION_UNINIT(&req);
    REGION_UNINIT(&ack);
    REGION_UNINIT(&rem);

    *crc_le32 = cpu_to_le32(crc);
    pp_bmc_log_debug("[OEM-PP-SLFTST] crc: 0x%08x", crc);

    ret = IPMI_ERR_SUCCESS;

fail:
    if (c) pp_grab_remove_client(c);
    slftst_complete(ret, (unsigned char*)crc_le32, sizeof(*crc_le32));
    return NULL;
}

static void* slftst_nic_ping_func(void* arg UNUSED)
{
    unsigned char *ipaddr = g_slftst.data + 1;
    char cmd[256];
    snprintf(cmd, sizeof(cmd), "ping %d.%d.%d.%d", 
             ipaddr[0], ipaddr[1], ipaddr[2], ipaddr[3]);

    int ret = pp_system(cmd);
    pp_bmc_log_debug("[OEM-PP-SLFTST] %s %s", cmd, 
                     ret == 0 ? "succeeded" : "failed");
    slftst_complete(ret == 0 ? IPMI_ERR_SUCCESS : IPMI_ERR_SLFTST_FAILED, NULL, 0);
    return NULL;
}

static int slftst_start_thread(void* (*func)(void*), unsigned char *data)
{
    // start as 'detached' (no join needed)
    return eric_pthread_create(&slftst_thread, 1, 65536,
        func, data) == 0 ? IPMI_ERR_SUCCESS : IPMI_ERR_UNSPECIFIED;
}

/********************************************************************
 * Subtest Routines
 */

static int test_video(void)
{
    switch (g_slftst.test) {
    case IPMI_OEM_PP_SLFTST_VIDEO_STATUS: {
            unsigned char ret = IPMI_ERR_SLFTST_FAILED;
            oem_pp_selftest_video_status_rs_t *rs = 
                (oem_pp_selftest_video_status_rs_t*)malloc
                    (sizeof(oem_pp_selftest_video_status_rs_t));

            if (pp_vsc_has_signal(0) != VIDEO_SIGNAL_ON) {
                pp_bmc_log_warn("[OEM-PP-SLFTST] no video detected");
                goto fail;
            }
            pp_bmc_log_debug("[OEM-PP-SLFTST] video signal detected");

            // get resolution
            fb_format_info_t fb_format;
            if (pp_vsc_get_fb_format(0, &fb_format) != 0) {
                pp_bmc_log_error("[OEM-PP-SLFTST] unable to get video format");
                goto fail;
            }
            pp_bmc_log_debug("[OEM-PP-SLFTST] resolution: w=%d h=%d", 
                            fb_format.g_w, fb_format.g_h);
            rs->xres_le16 = cpu_to_le16(fb_format.g_w);
            rs->yres_le16 = cpu_to_le16(fb_format.g_h);

            // get freqs
            int hf, vf;
            if (pp_vsc_get_freqs(0, &hf, &vf) != 0) {
                pp_bmc_log_error("[OEM-PP-SLFTST] unable to get video freqs");
                goto fail;
            }
            pp_bmc_log_debug("[OEM-PP-SLFTST] freqs: h=%dkHz v=%dHz", 
                             hf / 1000, vf);
            rs->hfreq_le16 = cpu_to_le16(hf / 1000);
            rs->vfreq_le16 = cpu_to_le16(vf);

            ret = IPMI_ERR_SUCCESS;
fail:
            slftst_complete(ret, (unsigned char*)rs, sizeof(*rs));
            return IPMI_ERR_SUCCESS;
        }
    case IPMI_OEM_PP_SLFTST_VIDEO_CRC:
        return slftst_start_thread(slftst_video_crc_func, NULL);
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

static int test_ddc(void)
{
    switch (g_slftst.test) {
#ifdef PP_FEAT_DDC
    case IPMI_OEM_PP_SLFTST_DDC_INFO: {
            unsigned char *data = (unsigned char*)malloc(128);
            pp_vsc_get_ddc_edid(data);
            slftst_complete(IPMI_ERR_SUCCESS, data, 128);
            return IPMI_ERR_SUCCESS;
        }
#endif
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

static int test_ipmb(void)
{
    switch (g_slftst.test) {
#ifdef PRODUCT_ASMIDC
    case IPMI_OEM_PP_SLFTST_IPMB_OPERATION: {
            pp_ipmi_parameter_t p = { .is_cmdline = 0, };
            pp_ipmi_return_t r;
            int e;

            if (PP_FAILED(pp_ipmi_send_command(PP_IPMI_CMD_BMC, 
                                PP_IPMI_BMC_SUBCMD_INFO, &p, &r, &e, NULL))) {
                pp_bmc_log_warn("[OEM-PP-SLFTST] no response from BMC on IPMB");
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
            } else {
                pp_bmc_log_debug("[OEM-PP-SLFTST] got response from BMC on IPMB");
                slftst_complete(IPMI_ERR_SUCCESS, NULL, 0);
            }
            return IPMI_ERR_SUCCESS;
        }
    case IPMI_OEM_PP_SLFTST_IPMB_EVALBOARD: {
            int ret;
            
            /* open the test module file descriptor */
            int test_fd = open("/dev/i2c-ipmb_test", O_RDWR);
            if (test_fd == -1) {
                pp_bmc_log_error("[OEM-PP-SLFTST] Could not open i2c-ipmb_test device!\n");
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                return IPMI_ERR_SUCCESS;
            }
            
            /* now perform the self test */
            ret = ioctl(test_fd, IOCTL_I2C_IPMB_EVALBOARD_TEST, 0);
            close(test_fd);
            
            if (ret == 0) {
                pp_bmc_log_info("[OEM-PP-SLFTST] IPMB self test successful.\n");
                slftst_complete(IPMI_ERR_SUCCESS, NULL, 0);
            } else {
                switch (errno) {
                    case EIO:
                        pp_bmc_log_error("[OEM-PP-SLFTST] IPMB eval board self test failed - I/O error.\n");
                        break;
                    case ETIMEDOUT:
                        pp_bmc_log_error("[OEM-PP-SLFTST] IPMB eval board self test failed - Slave did not respond.\n");
                        break;
                    case ENOMSG:
                        pp_bmc_log_error("[OEM-PP-SLFTST] IPMB eval board self test failed - Message did not match expected format.\n");
                        break;
                    default:
                        pp_bmc_log_error("[OEM-PP-SLFTST] IPMB eval board self test failed - unknown reason.\n");
                }
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
            }
            return IPMI_ERR_SUCCESS;
        }
#endif
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

static int test_fml(void)
{
    switch (g_slftst.test) {
#ifdef PP_FEAT_ESB2_TPT
    case IPMI_OEM_PP_SLFTST_FML_OPERATION: {
            int s = eric_net_tpt_get_status();
            if (s < 0) {
                pp_bmc_log_warn("[OEM-PP-SLFTST] TPT not working (status: %d)",
                                s);
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
            } else {
                pp_bmc_log_debug("[OEM-PP-SLFTST] TPT working (status: %d)", s);
                slftst_complete(IPMI_ERR_SUCCESS, NULL, 0);
            }
            return IPMI_ERR_SUCCESS;
        }
#endif
#ifdef PRODUCT_ASMIDC
    case IPMI_OEM_PP_SLFTST_FML_EVALBOARD: {
            int ret;
            
#ifdef PP_FEAT_ESB2_TPT
            eric_net_esb2_tpt_cleanup();
#endif
            
            /* unload the esb2_tpt module, if not yet done */
            ret = pp_system("lsmod|cut -f 1 -d \" \"|grep -c esb2_tpt");
            if (ret == -1) {
                pp_bmc_log_error("[OEM-PP-SLFTST] Could not determine esb2_tpt module availability!");
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                return IPMI_ERR_SUCCESS;
            }
            if (WEXITSTATUS(ret) == 0) {
                pp_bmc_log_info("[OEM-PP-SLFTST] Removing esb2_tpt module.");
                if (!PP_SUCCED(pp_system("rmmod esb2_tpt"))) {
                    pp_bmc_log_error("[OEM-PP-SLFTST] Could not unload esb2_tpt module!");
                    slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                    return IPMI_ERR_SUCCESS;
                }
            } else {
                pp_bmc_log_info("[OEM-PP-SLFTST] esb2_tpt module not loaded.");
            }

            /* load the fml_test module, if not yet done */
            ret = pp_system("lsmod|cut -f 1 -d \" \"|grep -c fml_test");
            if (ret == -1) {
                pp_bmc_log_error("[OEM-PP-SLFTST] Could not determine fml_test module availability!");
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                return IPMI_ERR_SUCCESS;
            }
            if (WEXITSTATUS(ret) != 0) {
               pp_bmc_log_info("[OEM-PP-SLFTST] Loading fml_test module.");
               if (!PP_SUCCED(pp_system("insmod fml_test"))) {
                    pp_bmc_log_error("[OEM-PP-SLFTST] Could not load fml_test module!");
                    slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                    return IPMI_ERR_SUCCESS;
                }
            } else {
               pp_bmc_log_info("[OEM-PP-SLFTST] fml_test module already loaded.");
            }
            
            /* open the test module file descriptor */
            int test_fd = open("/dev/fml_test", O_RDWR);
            if (test_fd == -1) {
                pp_bmc_log_error("[OEM-PP-SLFTST] Could not open fml_test device!");
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
                return IPMI_ERR_SUCCESS;
            }
            
            /* now perform the self test */
            ret = ioctl(test_fd, FML_IOCTL_PERFORM_TEST, 0);
            close(test_fd);
            
            if (ret == 0) {
                pp_bmc_log_info("[OEM-PP-SLFTST] FML self test successful.");
                slftst_complete(IPMI_ERR_SUCCESS, NULL, 0);
            } else {
                switch (errno) {
                    case EIO:
                        pp_bmc_log_error("[OEM-PP-SLFTST] FML eval board self test failed - I/O error.");
                        break;
                    case ETIMEDOUT:
                        pp_bmc_log_error("[OEM-PP-SLFTST] FML eval board self test failed - Slave did not respond.");
                        break;
                    case ENOMSG:
                        pp_bmc_log_error("[OEM-PP-SLFTST] FML eval board self test failed - Message did not match expected format.");
                        break;
                    default:
                        pp_bmc_log_error("[OEM-PP-SLFTST] FML eval board self test failed - unknown reason.");
                }
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
            }
            
            return IPMI_ERR_SUCCESS;
        }
#endif
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

static int test_usb(void)
{
    switch (g_slftst.test) {
#ifdef PP_FEAT_USB
    case IPMI_OEM_PP_SLFTST_USB_OPERATION: {
            // TODO: parse the USB channel number!
            usb_device_state_t state = pp_usb_get_device_state();
            if (state == PP_USB_DEVSTAT_CONFIGURED) {
                pp_bmc_log_debug("[OEM-PP-SLFTST] USB working, we are enumerated");
                slftst_complete(IPMI_ERR_SUCCESS, NULL, 0);
            } else {
                pp_bmc_log_warn("[OEM-PP-SLFTST] USB not working, state: %d", state);
                slftst_complete(IPMI_ERR_SLFTST_FAILED, NULL, 0);
            }
            return IPMI_ERR_SUCCESS;
        }
#endif
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

static int test_nic(void)
{
    // check params (NIC ID = 0, 1, or 2)
    if (g_slftst.datalen < 1 || g_slftst.data[0] > 2)
        return IPMI_ERR_INVALID_DATA_FIELD;
    int nic_id = g_slftst.data[0] ? g_slftst.data[0] - 1 : 0;

    switch (g_slftst.test) {
    case IPMI_OEM_PP_SLFTST_NIC_STATUS: {
            unsigned char ret = IPMI_ERR_SLFTST_FAILED;

            struct ifreq ifr;
            struct ethtool_value eval;
            struct ethtool_cmd ecmd;

            // enter NIC name
            snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "eth%d", nic_id);

            // open network driver
            int sock = socket(PF_PACKET, SOCK_RAW, 0);
            if (sock == -1) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to open network driver");
                goto fail;
            }

            // get settings
            ecmd.cmd = ETHTOOL_GSET;
            ifr.ifr_data = (void*)&ecmd;
            if (ioctl(sock, SIOCETHTOOL, &ifr) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to get link settings");
                goto fail;
            }

            // get link state
            eval.cmd = ETHTOOL_GLINK;
            ifr.ifr_data = (void*)&eval;
            if (ioctl(sock, SIOCETHTOOL, &ifr) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to get link status");
                goto fail;
            }

            // build up response data
            oem_pp_selftest_nic_status_rs_t *rs = 
                (oem_pp_selftest_nic_status_rs_t*)malloc
                    (sizeof(oem_pp_selftest_nic_status_rs_t));
            rs->link = eval.data ? 1 : 0;
            rs->duplex = ecmd.duplex == DUPLEX_HALF ? 1 
                       : ecmd.duplex == DUPLEX_FULL ? 2 
                       : 0;
            rs->speed = ecmd.speed == SPEED_10 ? 1 
                      : ecmd.speed == SPEED_100 ? 2 
                      : ecmd.speed == SPEED_1000 ? 3 
                      : ecmd.speed == SPEED_10000 ? 4
                      : 0;
            ret = IPMI_ERR_SUCCESS;
            pp_bmc_log_debug("[OEM-PP-SLFTST] NIC %s: speed=%d duplex=%d link=%d",
                             ifr.ifr_name, rs->speed, rs->duplex, rs->link);
fail:
            // clean up
            if (sock != -1) close(sock);
            slftst_complete(ret, (unsigned char*)rs, sizeof(*rs));
            return IPMI_ERR_SUCCESS;
        }
/*
    case IPMI_OEM_PP_SLFTST_NIC_LOOPBACK: {
            unsigned char ret = IPMI_ERR_SLFTST_FAILED;

            struct ifreq ifr;
            struct mii_ioctl_data *mid = if_mii(&ifr);
            unsigned long bmcr = 0;

            // enter NIC name
            snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "eth%d", nic_id);

            // open network driver
            int sock = socket(PF_PACKET, SOCK_RAW, htons(0));
            if (sock == -1) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to open network driver");
                goto fail2;
            }

            // set send and receive timeout to one sec
            struct timeval tv = { .tv_sec = 1, .tv_usec = 0, };
            if (setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO,
                           (void*)&tv, sizeof(tv)) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to set send timeout");
                goto fail2;
            }

            // set receive timeout
            if (setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO,
                           (void*)&tv, sizeof(tv)) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to set receive timeout");
                goto fail2;
            }

            // get PHY addr
            if (ioctl(sock, SIOCGMIIPHY, &ifr) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to get PHY addr");
                goto fail2;
            }
printf("### PHY addr: %d\n", mid->phy_id);

            // backup PHY register
            mid->reg_num = MII_BMCR;
            if (ioctl(sock, SIOCGMIIREG, &ifr) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to backup PHY reg");
                goto fail2;
            }
            bmcr = mid->val_out;
            if (bmcr == 0) {
                pp_bmc_log_error("[OEM-PP-SLFTST] invalid PHY reg");
                goto fail2;
            }
printf("### MII_BMCR: %08lx\n", bmcr);

            // enable test loopback
            mid->reg_num = MII_BMCR;
            mid->val_in = (bmcr | BMCR_LOOPBACK | BMCR_FULLDPLX) & ~BMCR_ANENABLE;
            if (ioctl(sock, SIOCSMIIREG, &ifr) < 0) {
                pp_bmc_log_perror("[OEM-PP-SLFTST] unable to enable PHY test loopback");
                goto fail2;
            }
printf("### loopback enabled\n");


printf("### forking\n");
            int status;
            char send_pkt[] = "Test message!";
            char recv_pkt[sizeof(send_pkt)];
            int send_ret = -1, recv_ret = -1;
            int send_errno = 0, recv_errno = 0;
            pid_t pid = fork();
            switch (pid) {
            case -1: // error
                pp_bmc_log_perror("[OEM-PP-SLFTST] internal fork error");
                goto fail2;
            case 0: { // child
printf("### child\n");
sleep(3);
                    // make sure parent is already in recv
                    usleep(1);

                    // send test packet
                    struct sockaddr_ll sa;
                    memset(&sa, 0, sizeof(sa));
                    sa.sll_family = AF_PACKET;
                    sa.sll_protocol = htons(0);
                    sa.sll_ifindex = if_nametoindex(ifr.ifr_name);
                    sa.sll_halen = ETH_ALEN;
                    memset(sa.sll_addr, 0xff, ETH_ALEN); // bcast
                    send_ret = sendto(sock, send_pkt, sizeof(send_pkt), 0,
                                    (struct sockaddr*)&sa, sizeof(sa));
                    send_errno = errno;
printf("### test msg sent (%d)\n", send_ret);

                    exit(1); // exit child
                }
            default: { // parent
printf("### parent\n");
sleep(3);
                    // receive test packet
                    struct sockaddr_ll sa;
                    socklen_t sa_len = sizeof(sa);
                    recv_ret = recvfrom(sock, recv_pkt, sizeof(recv_pkt), 0,
                                        (struct sockaddr*)&sa, &sa_len);
                    recv_errno = errno;
printf("### test msg received (%d)\n", recv_ret);

                    waitpid(pid, &status, 0); // join with child
                }
            }

            // check transmission and validate test packet
            if (send_ret != sizeof(send_pkt)) {
                errno = send_errno;
                pp_bmc_log_perror("[OEM-PP-SLFTST] error sending test packet");
                goto fail2;
            }
            if (recv_ret != sizeof(recv_pkt)) {
                errno = recv_errno;
                pp_bmc_log_perror("[OEM-PP-SLFTST] error receiving test packet");
                goto fail2;
            }
            if (memcmp(send_pkt, recv_pkt, sizeof(send_pkt)) == 0) ret = IPMI_ERR_SUCCESS;
printf("### test :-%s\n", memcmp(send_pkt, recv_pkt, sizeof(send_pkt)) ? "(" : ")");

fail2:
            // restore PHY regs
            if (bmcr != 0) {
                mid->reg_num = MII_BMCR;
                mid->val_in = bmcr;
                if (ioctl(sock, SIOCSMIIREG, &ifr) < 0) {
                    pp_bmc_log_perror("[OEM-PP-SLFTST] restore PHY reg");
                    goto fail2;
                }
printf("### PHY reg restored\n");
            }

            // clean up
            if (sock != -1) close(sock);
            slftst_complete(ret, NULL, 0);
            return IPMI_ERR_SUCCESS;
        }
*/
    case IPMI_OEM_PP_SLFTST_NIC_PING:
        if (g_slftst.datalen < 4) return IPMI_ERR_INVALID_DATA_FIELD;
        return slftst_start_thread(slftst_nic_ping_func, NULL);
    }

    return IPMI_ERR_SLFTST_NOT_SUPPORTED;
}

/********************************************************************
 * Command handlers
 */

/*
 * Perform Self-Test
 */

static int oem_pp_cmd_perform_selftest(imsg_t* imsg)
{
    oem_pp_selftest_t *rq = (oem_pp_selftest_t*)imsg->data;
    int sz = imsg->data_size - sizeof(oem_pp_selftest_t);
    int ret = PP_ERR;

    // check if another test is still running
    if (PP_FAILED(slftst_start(rq, sz))) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SLFTST_BUSY);
    }

    pp_bmc_log_info("[OEM-PP-SLFTST] Perform Self-Test command (comp=%d, test=%d)",
                    g_slftst.comp, g_slftst.test);

    // branch to the subtest-routines
    switch (rq->comp) {
        case IPMI_OEM_PP_SLFTST_VIDEO:  ret = test_video(); break;
        case IPMI_OEM_PP_SLFTST_DDC:    ret = test_ddc(); break;
        case IPMI_OEM_PP_SLFTST_IPMB:   ret = test_ipmb(); break;
        case IPMI_OEM_PP_SLFTST_FML:    ret = test_fml(); break;
        case IPMI_OEM_PP_SLFTST_USB:    ret = test_usb(); break;
        case IPMI_OEM_PP_SLFTST_NIC:    ret = test_nic(); break;
    }

    if (ret != IPMI_ERR_SUCCESS) {
        g_slftst.comp = 0; // no results avail
        g_slftst.test = 0;
        slftst_complete(0, NULL, 0);
        pp_bmc_log_warn("[OEM-PP-SLFTST] error %d in self-test", ret);
        return pp_bmc_router_resp_err(imsg, ret);
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Get Self-Test Result
 */
static int oem_pp_cmd_get_selftest_result(imsg_t* imsg)
{
    // check if test is still running
    if (slftst_is_running()) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SLFTST_BUSY);
    }

    // check if any results are available
    if (g_slftst.comp == 0) {
        pp_bmc_log_warn("[OEM-PP-SLFTST] client tries to get nonexisting results");
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SLFTST_NO_RESULTS);
    }

    pp_bmc_log_info("[OEM-PP-SLFTST] Get Self-Test Result command (comp=%d, test=%d)",
                    g_slftst.comp, g_slftst.test);

    // return copy of results
    oem_pp_selftest_t *rs = pp_bmc_imsg_resp(imsg, g_slftst.cc, 
        sizeof(oem_pp_selftest_t) + g_slftst.datalen);
    rs->comp = g_slftst.comp;
    rs->test = g_slftst.test;
    memcpy(rs->data, g_slftst.data, g_slftst.datalen);
    return pp_bmc_router_send_msg(imsg);
}

/********************************************************************
 * Device command table
 */

static const dev_cmd_entry_t oem_pp_selftest_cmd_tab[] = {
    {
        .cmd_hndlr = oem_pp_cmd_perform_selftest,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_PERFORM_SELFTEST,
        .min_data_size = sizeof(oem_pp_selftest_t),
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = oem_pp_cmd_get_selftest_result,
        .netfn = IPMI_NETFN_OEM_PP,
        .cmd = IPMI_CMD_PP_GET_SELFTEST_RESULT,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * Device c'tor/d'tor
 */

int pp_bmc_dev_oem_pp_selftest_init()
{
    /* register all entries of cmd tab */
    if (PP_FAILED(pp_bmc_core_reg_cmd_tab(oem_pp_selftest_cmd_tab))) {
        return PP_ERR;
    }
    pp_bmc_log_info("[OEM-PP-SLFTST] device started");
    return PP_SUC;
}

void pp_bmc_dev_oem_pp_selftest_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(oem_pp_selftest_cmd_tab);
    pp_bmc_log_info("[OEM-PP-SLFTST] device shut down");
}
