/**
 * bmc_dev_pef.c
 *
 * BMC Platform Event Filtering (PEF), IPMI command frontend
 * This module only gets and sets parameters and forwards some
 * notifications on parameter changes. There is not one piece
 * of internal logic in this file.
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 * 
 * Note: ICTS's idea of how to use the last-event-processed-by-system-software
 * marker differs from our's. ICTS expects the timer to start if the marker
 * is different than the last bmc marker, we expect that a newer event exist
 * as a second precondition.
 */

#include <string.h>
#include "pp/base.h"
#include "pp/selector.h"

#include "pp/bmc/debug.h"

#include "pp/bmc/bmc_core.h"
#include "pp/bmc/bmc_router.h"

#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/ipmi_cmd.h"
#include "pp/bmc/ipmi_err.h"
#include "pp/bmc/ipmi_msg.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_sess.h"

#include "bmc_dev_sel_nv.h"
#include "bmc_dev_pef.h"
#include "bmc_dev_pef_nv.h"
#include "bmc_dev_pef_internals.h"
#include "bmc_dev_pef_alert.h"
#include "event_receiver.h"


typedef struct {
    // configuration parameters
    unsigned char setInProgress;
} pef_globals_t;

pef_globals_t pef_globals;
pef_globals_t *pefg;


/**
 * PEF commands
 */

static int pef_cmd_get_capabilities(imsg_t* imsg) {
    unsigned char* data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 3);

    data[0] = 0x51;  // still valid for IPMI v2.0
    data[1] = 0;
    data[1] |= 1;    // Alert
    data[1] |= 2;    // Power down
    data[1] |= 4;    // Reset
    data[1] |= 8;    // power cycle
                     // OEM action and diagnostic interrupt are not supported

    data[2] = EFT_SIZE;

    return pp_bmc_router_send_msg(imsg);
}

static int pef_cmd_arm_timer(imsg_t* imsg) {
    unsigned char t = imsg->data[0];
    if (t != 0xFF) {
        // notify logic that timer interval has changed if this is a good value
        bmc_pef_set_timer(t);
    }   
    // else query only
    
    unsigned char* data = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
    data[0] = bmc_pef_get_timer();

    return pp_bmc_router_send_msg(imsg);
}

#define CHECK_PDATA_SIZE(s) \
    if (psize < s) { return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID); }

#define PEF_GET_CFG(c) \
    if (PP_FAILED(pef_get_cfg(c))) { ierr = IPMI_ERR_UNSPECIFIED; break; }

#define PEF_SET_CFG(c) \
    if (PP_FAILED(pef_set_cfg(c))) { ierr = IPMI_ERR_UNSPECIFIED; break; }

static int pef_cmd_set_configuration(imsg_t* imsg) {
    unsigned char param = imsg->data[0] & 0x7F;  // mask out 8th bit
    unsigned char setSel;
    unsigned char blockSel;
    unsigned char* pdata = &imsg->data[1];
    int psize = imsg->data_size - 1;

    unsigned char ierr = IPMI_ERR_SUCCESS;

    pef_configuration_t config;
    event_filter_entry_t filter;
    alert_policy_entry_t alertPolicy;
    alert_string_entry_t alertString;

    switch (param) {
        case 0:
            CHECK_PDATA_SIZE(1);
            switch (pdata[0] & 0x3) {
                case 0: /* set complete */
                    pefg->setInProgress = 0;
                    break;
                case 1: /* set in progress */
                    if (pefg->setInProgress == 1) {
                        ierr = IPMI_ERR_PEF_SET_IN_PROGRESS;
                    }
                    pefg->setInProgress = 1;
                    break;
                default:
                    ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            }
            break;
        case 1:
            CHECK_PDATA_SIZE(1);
            PEF_GET_CFG(&config);

            // notify only if the state changes (if the old and new state are different)
            if (config.enable_pef != (pdata[0] & 0x01)) {
                config.enable_pef = (pdata[0] & 0x01);
                if (config.enable_pef == 1)
                    bmc_pef_enable();
                else {
                    bmc_pef_disable();
                    pp_bmc_log_warn("[PEF] PEF disabled by client!");
                }
            }
            config.enable_action_events = ((pdata[0] & 0x02) >> 1);
            config.disable_startup_delay = 0;       // explicitly disabling delays is not supported
            config.disable_alert_startup_delay = 0; // explicitly disabling delays is not supported
            PEF_SET_CFG(&config);
            break;
        case 2:
            CHECK_PDATA_SIZE(1);
            PEF_GET_CFG(&config);

	    if (pdata[0] & 0x30) { // icts conformance
		ierr = IPMI_ERR_INVALID_DATA_FIELD;
                break;
            }
            config.enable_diagnostic_interrupt = 0;
            config.enable_oem = 0;
            config.enable_power_cycle = (pdata[0] & 0x08) >> 3;
            config.enable_reset = (pdata[0] & 0x04) >> 2;
            config.enable_poweroff = (pdata[0] & 0x02) >> 1;
            config.enable_alert = (pdata[0] & 0x01);
            PEF_SET_CFG(&config);
            break;
        case 3:
            CHECK_PDATA_SIZE(1);
            PEF_GET_CFG(&config);
            config.startup_delay = pdata[0];
            PEF_SET_CFG(&config);
            break;
        case 4:
            CHECK_PDATA_SIZE(1);
            PEF_GET_CFG(&config);
            config.alert_startup_delay = pdata[0];
            PEF_SET_CFG(&config);
            break;
        case 5:
            ierr = IPMI_ERR_PEF_PARAM_READ_ONLY;
            break;
        case 6:
            CHECK_PDATA_SIZE(21);
            setSel = pdata[0];
            if ((setSel == 0) || (setSel > EFT_SIZE)) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }

            pef_get_event_filter(setSel-1, &filter);
            if ((filter.configuration_type == 0x3) || (filter.configuration_type == 0x1)) {
                ierr = IPMI_ERR_INVALID_DATA_FIELD;  // reserved types
                break;
            }
            if (filter.configuration_type == 0x00)
                memcpy(&filter, &(pdata[1]), 20);                             // user defined
            if (filter.configuration_type == 0x2) {
                filter.enable = ((pdata[1] & 0x80) == 0x80);                  // preconfigured filter, only enable
            }
            pef_set_event_filter(setSel-1, &filter);
            break;
        case 7:
            CHECK_PDATA_SIZE(2);
            setSel = pdata[0];
            if ((setSel == 0) || (setSel > EFT_SIZE)) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            pef_get_event_filter(setSel-1, &filter);
            pef_get_event_filter(setSel-1, &filter);
            if ((filter.configuration_type == 0x3) || (filter.configuration_type == 0x1)) {
                ierr = IPMI_ERR_INVALID_DATA_FIELD;  // reserved types
                break;
            }
            if (filter.configuration_type == 0x00)
                memcpy(&filter, &(pdata[1]), 1);                              // user defined
            if (filter.configuration_type == 0x2) {
                filter.enable = ((pdata[1] & 0x80) == 0x80);                  // preconfigured filter, only enable
            }
            pef_set_event_filter(setSel-1, &filter);
            break;
        case 8:
            ierr = IPMI_ERR_PEF_PARAM_READ_ONLY;
            break;
        case 9:
            CHECK_PDATA_SIZE(4);
            setSel = pdata[0];
            if ((setSel==0) || (setSel > APT_SIZE)) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            //pef_get_alert_policy(setSel-1, &alertPolicy);
            memcpy(&alertPolicy, &(pdata[1]), 3);
            pef_set_alert_policy(setSel-1, &alertPolicy);
            break;
        case 10:
            CHECK_PDATA_SIZE(17);
            PEF_GET_CFG(&config);
            config.enable_system_guid = pdata[0] & 0x01;
            memcpy(&(config.system_guid), &(pdata[1]), 16);
            PEF_SET_CFG(&config);
            break;
        case 11:
            ierr = IPMI_ERR_PEF_PARAM_READ_ONLY;
            break;
        case 12:
            CHECK_PDATA_SIZE(3);
            setSel = pdata[0] & 0x7F;
            if ((setSel >= AST_SIZE) || (pdata[1] == 0) || (pdata[2] == 0)) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
                break;
            }
            pef_get_alert_string(setSel, &alertString);
            alertString.event_filter_number = pdata[1] & 0x7F;
            alertString.alert_string_set = pdata[2] & 0x7F;
            pef_set_alert_string(setSel, &alertString);
            break;
        case 13:
            CHECK_PDATA_SIZE(3);
            setSel = pdata[0] & 0x7F;
            blockSel = pdata[1];
            if ((setSel >= AST_SIZE) || (blockSel == 0)) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            } else if (blockSel >= ALERT_STRING_BLOCK_CNT) {
                ierr = IPMI_ERR_PARAM_OUT_OF_RANGE;
            } else if (psize > 17) {
                ierr = IPMI_ERR_REQ_DATA_LEN_LIMIT_EXCEEDED;
            } else {
                pef_get_alert_string(setSel, &alertString);
                memcpy((&(alertString.alertString[0]) + (blockSel-1)*16), &(pdata[2]), psize - 2);
                pef_set_alert_string(setSel, &alertString);
            }
            break;
        default:
            ierr = IPMI_ERR_PEF_PARAM_NOT_SUPPORTED;
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int pef_cmd_get_configuration(imsg_t* imsg) {
    unsigned char pdata[256];
    int psize;
    char *resp;

    pef_configuration_t config;
    // event_filter_entry_t filter;
    // alert_policy_entry_t alertPolicy;
    alert_string_entry_t alertString;
    unsigned int alert_strlen;
    
    unsigned char param = imsg->data[0] & 0x7F;  // mask out 8th bit
    unsigned char setSel = imsg->data[1];
    unsigned int  blockSel = imsg->data[2];

    if (imsg->data_size != 3)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);

    if (imsg->data[0] & 0x80) {
        psize=0;
    } else
    switch (param) {
        case 0:
            psize = 1;
            pdata[0] = pefg->setInProgress;
            break;
        case 1:
            pef_get_cfg(&config);
            psize = 1;
            pdata[0] = 0;
            pdata[0] |= (config.enable_pef);
            pdata[0] |= (config.enable_action_events << 1);
            pdata[0] |= (config.disable_startup_delay << 2);
            pdata[0] |= (config.disable_alert_startup_delay << 3);
            break;
        case 2:
            pef_get_cfg(&config);
            psize = 1;
            pdata[0] = 0;
            pdata[0] |= (config.enable_diagnostic_interrupt << 5);
            pdata[0] |= (config.enable_oem << 4);
            pdata[0] |= (config.enable_power_cycle << 3);
            pdata[0] |= (config.enable_reset << 2);
            pdata[0] |= (config.enable_poweroff << 1);
            pdata[0] |= (config.enable_alert);
            break;
        case 3:
            psize = 1;
            pef_get_cfg(&config);
            pdata[0] = config.startup_delay;
            break;
        case 4:
            psize = 1;
            pef_get_cfg(&config);
            pdata[0] = config.alert_startup_delay;
            break;
        case 5:
            psize = 1;
            pdata[0] = EFT_SIZE;
            break;
        case 6:
            psize = 21;
            if ((setSel == 0) || (setSel > EFT_SIZE))
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
            pdata[0] = setSel;
            pef_get_event_filter(setSel-1, (event_filter_entry_t*)&(pdata[1]));
            break;
        case 7:
            psize = 2;
            if ((setSel == 0) || (setSel > EFT_SIZE))
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
            pdata[0] = setSel;
            pef_get_event_filter(setSel-1, (event_filter_entry_t*)&(pdata[1]));
            break;
        case 8:
            psize = 1;
            pdata[0] = APT_SIZE;
            break;
        case 9:
            psize = 4;
            if ((setSel == 0) || (setSel > APT_SIZE))
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
            pdata[0] = setSel;
            pef_get_alert_policy(setSel-1, (alert_policy_entry_t*)(&(pdata[1])));
            break;
        case 10:
            psize = 17;
            pef_get_cfg(&config);
            memcpy (&(pdata[0]),&(config.system_guid[0]) - 1, psize);
            // (-1 because we also have to copy the enable_system_guid_bit in the proceeding byte)
            break;
        case 11:
            psize = 1;
            pdata[0] = AST_SIZE-1;  // minus the volatile string 0
            break;
        case 12:
            psize = 3;
            setSel = setSel & 0x7F;
            if (setSel >= AST_SIZE)
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
            pdata[0] = setSel;
            pef_get_alert_string(setSel, &alertString);
            memcpy (&(pdata[1]),&(alertString), psize-1);
            break;
        case 13:
            psize = 18;
            setSel = setSel & 0x7F;
            if ((setSel >= AST_SIZE) || (blockSel == 0) )
                return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
            pdata[0] = setSel;
            pdata[1] = blockSel;
            pef_get_alert_string(setSel, &alertString);
	    alert_strlen = strnlen(alertString.alertString, 
				   sizeof(alertString.alertString));
	    assert(alert_strlen < sizeof(alertString.alertString));
	    if ( (blockSel -1)*16 > alert_strlen)
		return pp_bmc_router_resp_err(imsg, IPMI_ERR_PARAM_OUT_OF_RANGE);
	    if ( blockSel*16 > alert_strlen) {
		strcpy(&(pdata[2]), &(alertString.alertString[0]) + (blockSel-1)*16);
	    } else {
		memcpy (&(pdata[2]),&(alertString.alertString[0]) + (blockSel-1)*16, 16);	    }
            break;
        default:
            return pp_bmc_router_resp_err(imsg, 0x80);  // parameter not supported
    }


    resp = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, psize + 1);
    resp[0] = 0x11;
    memcpy(resp + 1, pdata, psize);

    return pp_bmc_router_send_msg(imsg);
}

struct set_last_proc_event_rq_s {
    BITFIELD2(unsigned char, _res1 : 7, by_bmc : 1);
    unsigned short id_le16;
} __attribute__ ((packed));
typedef struct set_last_proc_event_rq_s set_last_proc_event_rq_t;

static int pef_cmd_set_last_processed_event(imsg_t* imsg)
{
    set_last_proc_event_rq_t *rq = (void*)imsg->data;
    unsigned short id = le16_to_cpu(rq->id_le16);

    if (rq->by_bmc) {
        pp_bmc_log_notice("[PEF] setting last processed event ID for BMC to %d", id);
        bmc_pef_set_lpe_bmc(id);
    } else {
        pp_bmc_log_notice("[PEF] setting last processed event ID for SMS to %d", id);
        bmc_pef_set_lpe_sms(id);
    }

    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

struct get_last_proc_event_rs_s {
    unsigned int timestamp_le32;
    unsigned short last_sel_id_le16;
    unsigned short last_sms_id_le16;
    unsigned short last_bmc_id_le16;
} __attribute__ ((packed));
typedef struct get_last_proc_event_rs_s get_last_proc_event_rs_t;

static int pef_cmd_get_last_processed_event(imsg_t* imsg) {
    pef_configuration_t config;
    get_last_proc_event_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
                                                    sizeof(get_last_proc_event_rs_t));

    rs->timestamp_le32 = cpu_to_le32(pp_bmc_get_most_recent_addition_timestamp());
    rs->last_sel_id_le16 = cpu_to_le16(pp_bmc_get_last_event_id());
    pef_get_cfg(&config);
    rs->last_sms_id_le16 = cpu_to_le16(config.lpe_sms);
    rs->last_bmc_id_le16 = cpu_to_le16(config.lpe_bmc);

    return pp_bmc_router_send_msg(imsg);
}

static int pef_cmd_alert_immediate(imsg_t* imsg) {
    unsigned char channel;
    unsigned char destination;
    unsigned char alert_string;
    unsigned char operation;
    sel_entry_t entry;
    unsigned char* cp;
    unsigned char c;
    
    memset(&entry, 0, sizeof(sel_entry_t));
    entry.type = 0x02;
    operation = (imsg->data[1] & 0xc0) >> 6;
    channel = imsg->data[0] & 0x0f;
    
    switch (operation) {
    case 0x00:
        /* initiate an immediate alert, check if data is extended */
        destination = imsg->data[1] & 0x0f;
        alert_string = imsg->data[2] & 0x7f;  // ignore bit 7

        /* read (optional) extended data for event */
        if (imsg->data_size == 11) {
            /* the 'generator id' byte is ambiguous (should be two bytes) */
            /* we copy the byte to generator id byte 1                    */
            memcpy(((unsigned char*)&entry)+7, imsg->data+3, 1);
            /* the rest of the event can be mapped directly */
            memcpy(((unsigned char*)&entry)+9, imsg->data+4, 7);
        }
        
        c = bmc_pef_add_immediate_alert(channel, destination, alert_string, &entry);
        pp_bmc_imsg_resp(imsg, c, 0);
        
        break;
        
    case 0x01:
        /* get immediate alert status */
        cp = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 1);
        *cp =bmc_pef_get_immediate_alert_status(channel);
        break;

    case 0x02:
        /* clear immediate alert status */
        bmc_pef_clear_immediate_alert_status(channel);
        pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, 0);
        break;
    default:
        /* unknown cmd (reserved) */
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }
    
    return pp_bmc_router_send_msg(imsg);
}

static int pef_cmd_acknowledge(imsg_t* imsg) {
    unsigned short* usp_le16;
    
    if (imsg->data_size != 12)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);
    
    usp_le16 = (unsigned short*)(imsg->data);
    if (PP_SUC == bmc_pef_acknowledge_alert(le16_to_cpu(*usp_le16)))
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
    else
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
}


 /********************************************************************
 * PEF device c'tor/d'tor
 */

static const dev_cmd_entry_t pef_cmd_tab[] = {
    {
        .cmd_hndlr = pef_cmd_get_capabilities, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_GET_PEF_CAPABILITES, 
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = pef_cmd_arm_timer, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_ARM_PEF_POSTPONE_TIMER,
        .min_data_size = 1,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = pef_cmd_set_configuration, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_SET_PEF_CONFIGURATION,
        .min_data_size = 2,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = pef_cmd_get_configuration, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_GET_PEF_CONFIGURATION,
        .min_data_size = 3,
        .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = pef_cmd_set_last_processed_event, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_SET_LASTPROCESSED_EVENT,
        .min_data_size = 3,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = pef_cmd_get_last_processed_event, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_GET_LASTPROCESSED_EVENT,
        .min_data_size = 0,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = pef_cmd_alert_immediate, 
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_ALERT_IMMEDIATE,
        .min_data_size = 3,
        .min_priv_level = IPMI_PRIV_ADMIN
    },
    {
        .cmd_hndlr = pef_cmd_acknowledge,
        .netfn = IPMI_NETFN_SENSOR_EVENT,
        .cmd = IPMI_CMD_PET_ACKNOWLEDGE,
        .min_data_size = 12,
        .min_priv_level = IPMI_PRIV_UNSPEC
    },
    { .cmd_hndlr = NULL }
};

static int pp_bmc_nv_init(void) {
    if (pef_complete_check() == PP_SUC) {
        return PP_SUC;
    }

    // initialize pef
    pef_configuration_t config;
    event_filter_entry_t filter;
    alert_policy_entry_t alertPolicy;
    alert_string_entry_t alertString;

    memset(&(config), 0, sizeof(pef_configuration_t));
    config.enable_pef = 1;
    config.enable_alert = 1;
    config.enable_poweroff = 1;
    config.enable_reset = 1;
    config.enable_power_cycle = 1;

    config.startup_delay = 60;
    config.alert_startup_delay = 60;
    
    config.lpe_bmc = 0xFFFF;
    config.lpe_sms = 0xFFFF;
    config.next_unprocessed_event = 0xFFFF;

    memset(&filter, 0, sizeof(event_filter_entry_t));
    memset(&alertPolicy, 0, sizeof(alert_policy_entry_t));
    memset(&alertString, 0, sizeof(alert_string_entry_t));

    if (pef_initialize(&config, &filter, &alertPolicy, &alertString) == PP_ERR)
        return PP_ERR;

    // Manufacturer preconfigured events could be defined here ...
#ifdef LARA_KIMMSI
    /* Filters as defined in 'MS-9246_9247_FW_Spec_101.pdf', section 19.2 */
    int i = 0;

    const unsigned char temp_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x01, 0xff,
          0x01, 0x04, 0x02, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&temp_1);
    i++;

    const unsigned char temp_2[20] =
        { 0x40, 0x01, 0x02, 0x08,
          0x20, 0xff, 0x01, 0xff,
          0x01, 0x81, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&temp_2);
    i++;

    const unsigned char temp_3[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x01, 0xff,
          0x81, 0xff, 0xff, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&temp_3);
    i++;


    const unsigned char voltage_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x02, 0xff,
          0x01, 0x04, 0x02, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&voltage_1);
    i++;

    const unsigned char voltage_2[20] =
        { 0x40, 0x01, 0x02, 0x08,
          0x20, 0xff, 0x02, 0xff,
          0x01, 0x81, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&voltage_2);
    i++;

    const unsigned char voltage_3[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x02, 0xff,
          0x81, 0xff, 0xff, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&voltage_3);
    i++;


    const unsigned char thres_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x04, 0xff,
          0x01, 0x04, 0x02, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&thres_1);
    i++;

    const unsigned char thres_2[20] =
        { 0x40, 0x01, 0x02, 0x08,
          0x20, 0xff, 0x04, 0xff,
          0x01, 0x81, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&thres_2);
    i++;

    const unsigned char thres_3[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x04, 0xff,
          0x81, 0xff, 0xff, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&thres_3);
    i++;

    const unsigned char thres_4[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x04, 0xff,
          0x03, 0x00, 0x02, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&thres_4);
    i++;

    const unsigned char thres_5[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x04, 0xff,
          0x83, 0x00, 0x01, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&thres_5);
    i++;


#if 0
    /* chassis intrusion events. not activated because we (currently)
     * do not have a chassis intrusion sensor */
    const unsigned char security_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x05, 0xff,
          0x6f, 0x0f, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&security_1);
    i++;

    const unsigned char security_2[20] =
        { 0x40, 0x01, 0x02, 0x08,
          0x20, 0xff, 0x05, 0xff,
          0x6f, 0x70, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&security_2);
    i++;

    const unsigned char security_3[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x05, 0xff,
          0xef, 0x7f, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&security_3);
    i++;
#endif


    /* watchdog events */
    const unsigned char asr_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x23, 0xff,
          0x6f, 0x04, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&asr_1);
    i++;

    const unsigned char asr_2[20] =
        { 0x40, 0x01, 0x02, 0x08,
          0x20, 0xff, 0x23, 0xff,
          0x6f, 0x0b, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&asr_2);
    i++;

    const unsigned char asr_3[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x23, 0xff,
          0xef, 0xff, 0xff, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&asr_3);
    i++;


#if 0
    /* module/board, removed because we have no sensor */
    const unsigned char board_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x15, 0xff,
          0x03, 0x02, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&board_1);
    i++;

    const unsigned char board_2[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x15, 0xff,
          0x83, 0x01, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&board_2);
    i++;
#endif

#if 0
    /* power device removed / inserted, removed because we have no sensor */
    const unsigned char power_1[20] =
        { 0x40, 0x01, 0x01, 0x02,
          0x20, 0xff, 0x08, 0xff,
          0x08, 0x03, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&power_1);
    i++;
#endif

#if 0
    /* memory fault, removed because we have no sensor */
    const unsigned char memory_1[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x0c, 0xff,
          0x6f, 0x01, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&memory_1);
    i++;

    const unsigned char memory_2[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x0f, 0xff,
          0x6f, 0x01, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&memory_2);
    i++;

    const unsigned char memory_3[20] =
        { 0x40, 0x01, 0x03, 0x10,
          0x20, 0xff, 0x13, 0xff,
          0x6f, 0x01, 0x00, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&memory_3);
    i++;
#endif

    /* overtemperature poweroff */
    const unsigned char poweroff_1[20] =
        { 0x40, 0x02, 0x04, 0x20,
          0x20, 0xff, 0x01, 0x20,
          0x01, 0x00, 0x08, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&poweroff_1);
    i++;

    const unsigned char poweroff_2[20] =
        { 0x40, 0x02, 0x04, 0x20,
          0x20, 0xff, 0x01, 0x21,
          0x01, 0x00, 0x08, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&poweroff_2);
    i++;

    const unsigned char alert1[3] = { 0x18, 0x11, 0x00 };
    pef_set_alert_policy(0, (const alert_policy_entry_t*)alert1);
    const unsigned char alert2[3] = { 0x28, 0x11, 0x00 };
    pef_set_alert_policy(1, (const alert_policy_entry_t*)alert2);
    const unsigned char alert3[3] = { 0x38, 0x11, 0x00 };
    pef_set_alert_policy(2, (const alert_policy_entry_t*)alert3);

    /* predefined filter from peppercon, not included in msi-spec */
    /* powerstate-sensor */
    const unsigned char acpi_powerstate[20] =
        { 0x40, 0x01, 0x01, 0x01,
          0x20, 0xff, 0x22, 0xff,
          0xff, 0xff, 0x3f, 0x00,
          0xff, 0x00, 0x00, 0xff,
          0x00, 0x00, 0xff, 0x00 };
    pef_set_event_filter(i, (event_filter_entry_t*)&acpi_powerstate);
    i++;

#endif
    
    return PP_SUC;
}

int pp_bmc_dev_pef_init()
{
    pef_configuration_t config;

    assert(sizeof(event_filter_entry_t) == 20);
    assert(sizeof(alert_policy_entry_t) == 3);

    pefg = &pef_globals;
    pefg->setInProgress = 0;    // set complete
    
    if (pp_bmc_nv_init() == PP_ERR)
        return PP_ERR;

    // clear the "volatile" alert String 0
    alert_string_entry_t as;
    pef_get_alert_string(0, &as);
    memset(&as, 0, sizeof(alert_string_entry_t));
    pef_set_alert_string(0, &as);

    if (bmc_pef_init() != PP_SUC)   // init internals
        return PP_ERR;


    if (PP_FAILED(pef_get_cfg(&config))) {
        pp_bmc_log_error("[PEF] failed to read non-volatile configuration data");
    } else {
        if (config.enable_pef != 1)
            pp_bmc_log_warn("[PEF] PEF has been disabled by client!");
    }

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(pef_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[PEF] device started");

    return PP_SUC;
}

void pp_bmc_dev_pef_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(pef_cmd_tab);

    bmc_pef_cleanup();
    
    pp_bmc_log_info("[PEF] device shut down");
}
