/**
 * bmc_dev_pef_alert.c
 *
 * Encapsulates the alert handling of the PEF device. This module
 * alerts all necessary destinations of a certain alert policy,
 * deferres alerts if necessary and notifies the calling module
 * when an alert is completely processed.
 *
 * Concrete channel alert implementations can register through
 * a generic interface.
 *
 * TODO:
 * - check alert policy chaining (choose next destination if unsuccessful, ...)
 * 
 * Notes:
 * - alert destinations in progress cannot be stopped (results are ignored)
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <sys/time.h>
#include <time.h>
#include <sys/select.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>

#include "bmc_dev_sel_nv.h"
#include "bmc_dev_pef_alert.h"
#include "bmc_dev_pef_nv.h"

#include "pp/base.h"
#include "pp/selector.h"
#include "pp/bmc/utils.h"
#include "pp/bmc/debug.h"
#include "pp/bmc/bmc_config.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/ipmi_err.h"


#define ALERT_IMMEDIATE_NO_STATUS     0x00
#define ALERT_IMMEDIATE_COMPLETE      0x01
#define ALERT_IMMEDIATE_RETRY_FAILED  0x02
#define ALERT_IMMEDIATE_NO_ACK        0x03
#define ALERT_IMMEDIATE_PENDING       0xff

/**
 * All alerts are stored in a queue and have this structure.
 * Then the current_policy index iterates over all entries in
 * the policy table to find valid alert destinations.
 * 
 * During the processing of a valid alert, alert_in_progress is set to 1
 * to "lock" further handling while the result of the pending destination
 * is not clear. Theoretically, several alerts could be handled in parallel.
 * This is not done in the moment.
 */
typedef struct {
    struct list_head link;
    int filter_number;
    sel_entry_t event;
    alert_complete_hndl_t alert_complete_handler;
    pp_bmc_alert_acknowledge_t acknowledge; // a handle to acknowledge a channel alert
    void* alert_hndl;                // "context" for the acknowledge at the channel
} alert_queue_entry_t;

/* PEF channel registrations */
typedef struct {
    struct list_head link;
    unsigned char channel;
    pp_bmc_alert_initiate_t start_alert;
    pp_bmc_alert_acknowledge_t acknowledge;
    
    /* immediate alert on this channel */
    unsigned char immediate_status;  // channel busy if ALERT_IMMEDIATE_PENDING
    sel_entry_t   immediate_event;
    void*         immediate_alert_hndl;
} alert_channel_t;

/* PEF alert globals */
typedef struct {
    char enabled;

    int current_policy_index;
    char last_successful;
    char chan_type;   // -1=any -2=none  x=any channel other than x
    char dest_type;   // -1=any -2=none  y=any dest other than y
    char destination_in_progress;

    struct list_head alertq_root;
    
    struct list_head channels;
} pef_alert_globals_t;

static pef_alert_globals_t pef_alert_globals;
static pef_alert_globals_t* pefag;


/* some internal function blueprints */
static unsigned char immediate_alerts_active(void);
static void next_alert(void);
static void send_alert(alert_queue_entry_t* aqe,
                       int filter_index,
                       int policy_index,
                       int channel,
                       int destination);


/**
 * Interface to other pef modules
 */

int bmc_pef_alert_init() {
    pefag = &pef_alert_globals;

    INIT_LIST_HEAD(&(pefag->alertq_root));
    pefag->enabled = 1;
    bmc_pef_hold_alert();
    
    INIT_LIST_HEAD(&(pefag->channels));
    
    return PP_SUC;
}

void bmc_pef_alert_cleanup() {
    bmc_pef_clear_alert();
}

void bmc_pef_enable_alert() {
    if (pefag->enabled == 0) {
        pefag->enabled = 1;

        // immediately process alerts
        if (immediate_alerts_active() == 0) {
            next_alert();
        }
    }
}

void bmc_pef_hold_alert() {
    if (pefag->enabled == 1) {
        pefag->enabled = 0;
        pefag->destination_in_progress = -1;
        pefag->current_policy_index = -1;
    }
}

void bmc_pef_clear_alert() {
    struct list_head* act;
    while (list_empty(&(pefag->alertq_root)) == 0) {
        act = pefag->alertq_root.next;
        list_del(act);
        free(list_entry(act, alert_queue_entry_t, link));
    }
    pefag->destination_in_progress = -1;
    pefag->current_policy_index = -1;
}

void bmc_pef_add_alert(sel_entry_t* evt, int filterNo, alert_complete_hndl_t alert_complete_handler) {
    pef_configuration_t config;
    char is_new_entry;
    
    is_new_entry=0;
    sel_entry_t* event = evt;

    pef_get_cfg(&config);
    if (config.enable_alert == 0)
        return;   // add alert only if alerting is enabled
    
    is_new_entry = list_empty(&pefag->alertq_root);
    
    alert_queue_entry_t* aqe;
    aqe = malloc(sizeof(alert_queue_entry_t));
    
    aqe->filter_number = filterNo;
    aqe->event = *event;                     // copy event contents
    aqe->alert_complete_handler = alert_complete_handler;
    aqe->alert_hndl = NULL;
    aqe->acknowledge = NULL;
    INIT_LIST_HEAD(&(aqe->link));
    list_add_tail(&(aqe->link), &(pefag->alertq_root));
    
    /* Call next_alert() if alerting is enabled and no other alerts were active.
     * Active alerts can be pef alerts (aqe not empty) or immediate alerts 
     * (channel immediate_status != ALERT_IMMEDIATE_PENDING).
     * (otherwise calls to next_alert() are ignored until the pending alert
     * destination is processed - and then next_alert() is called anyway)
     */
    if ((is_new_entry == 1) && (pefag->enabled == 1) && (immediate_alerts_active() == 0)) {
        next_alert();
    }
}

unsigned char bmc_pef_add_immediate_alert(unsigned char channel, 
                                unsigned char destination,
                                unsigned char alert_string UNUSED,
                                sel_entry_t* event)
{
    unsigned char guid[16];
    struct list_head* act;
    alert_channel_t* ach;
    pef_configuration_t config;
    void* rv;
    
    if (pefag->alertq_root.next != &(pefag->alertq_root)) {
        /* we have pending pef alerts, reject any immediate alerts */
        return IPMI_ERR_PEF_ALERT_IN_PROGRESS;
    }
    
    act = pefag->channels.next;
    while(act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        
        if (ach->channel == channel) {
            if (ach->immediate_status == ALERT_IMMEDIATE_PENDING) {
                return IPMI_ERR_PEF_ALERT_IN_PROGRESS;
            }

            /* TODO: get alert string here if we ever need it */
    
            // system guid, either from pet config or global guid
            pef_get_cfg(&config);
            if (config.enable_system_guid == 1) {
                memcpy(guid, config.system_guid, 16);
            }
            else {
               pp_bmc_get_guid(guid);
            }
            
            ach->immediate_event = *event;
            ach->immediate_event.id_le16 = cpu_to_le16(sel_nv_get_next_entry_id());
            
            pp_bmc_log_debug("[PEF] sending immediate alert to channel %d, destination %d", channel, destination);
            
            ach->immediate_status = ALERT_IMMEDIATE_PENDING;
            rv = ach->start_alert(destination, NULL, &(ach->immediate_event), 0, guid, 1, ach);
            if (rv != NULL) {
                ach->immediate_alert_hndl = rv;
            }
            /* else: Alert immediately finished (maybe because of error)  *
             * immediate_status is set accordingly                        */
            return IPMI_ERR_SUCCESS;
        }
        act = act->next;
    }
    
    return IPMI_ERR_DEST_UNAVAILABLE;     // may be the wrong completion code
}

unsigned char bmc_pef_get_immediate_alert_status(unsigned char channel) {
    struct list_head* act;
    alert_channel_t* ach;

    act = pefag->channels.next;
    while (act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        if (ach->channel == channel) {
            return ach->immediate_status;
        }
        act = act->next;
    }
    
    return ALERT_IMMEDIATE_NO_STATUS;
}

void bmc_pef_clear_immediate_alert_status(unsigned char channel) {
    struct list_head* act;
    alert_channel_t* ach;

    act = pefag->channels.next;
    while (act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        if (ach->channel == channel) {
            ach->immediate_status = ALERT_IMMEDIATE_NO_STATUS;
            return;
        }
        act = act->next;
    }
    // if we come here the channel is unknown, but thats not a problem
}


/**
 * Internals
 */

/**
 * check if we have active immediate alerts.
 * @return 1 if yes, 0 otherweise
 */
static unsigned char immediate_alerts_active() {
    struct list_head* act;
    alert_channel_t* ach;

    act = pefag->channels.next;
    while (act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        if (ach->immediate_status == ALERT_IMMEDIATE_PENDING) {
            return 1;
        }
        act = act->next;
    }
    
    return 0;
}

/**
 * process pending alerts ...
 */
static void next_alert() {
    alert_queue_entry_t* aqe;
    alert_policy_entry_t policy;
    event_filter_entry_t filter;
    struct list_head* next;
    int i;
    int apply;

restart:
    // dont do anything if alerting is disabled (hold)
    if (pefag->enabled == 0)
        return;

    if (list_empty(&(pefag->alertq_root)) == 0) {
        // process one destination for the current alert policy
        // get current policy
        next = pefag->alertq_root.next;
        aqe = list_entry(next, alert_queue_entry_t, link);
        sel_entry_t* event = &(aqe->event);

        if (aqe->filter_number != -1) {
            pef_get_event_filter(aqe->filter_number, &filter);
        }

        if (pefag->current_policy_index == -1) {
            // This is a new alert (without any processed destinations)
            //pp_bmc_log_debug("[PEF] processing alert for event=%.4x filter=%d", event->id, aqe->filter_number);
            pefag->last_successful = 0;
            pefag->chan_type = -1;
            pefag->dest_type = -1;
            pefag->destination_in_progress = 0;

            if (aqe->filter_number == -1) {
                // dummy alert, no need to process, set as if already processed
                pefag->current_policy_index = APT_SIZE - 1;
            }
        }
        
        if (pefag->destination_in_progress == 1) {
            // an alert destination is currently being processed
            pp_bmc_log_notice("[PEF] next_alert called without necessity - alert destination in progress");
            return;
        }
        
        // proceed to next unchecked policy
        pefag->current_policy_index++;
        
        // find next destination in the policy set
        //pp_bmc_log_debug("[PEF] find next destination (chan=%d, dest=%d)", aqe->chan_type, aqe->dest_type);
        for (i=pefag->current_policy_index; i<APT_SIZE; i++) {
            pef_get_alert_policy(i, &policy);
            //pp_bmc_log_debug("[PEF] checking destination #%d = %x (policyNo=%u, enable=%u)",i, ((char*)(&policy))[0] ,policy.number,policy.enable);
            if ( (policy.number == filter.alert_policy_number) && (policy.enable == 1) ) {
                //pp_bmc_log_debug("[PEF] checking destination II #%d",i);
                // the (char) casting is needed because the ppc compiler complains for some unknown reason
                if (  (pefag->chan_type != (char)(-2)) && (pefag->dest_type != (char)(-2)) &&
                     ((pefag->chan_type == (char)(-1)) || (pefag->chan_type != policy.channel)) &&
                     ((pefag->dest_type == (char)(-1)) || (pefag->dest_type != policy.destination)) ) {
                    // send alert to this entry
                    // reset chan_type and dest_type because we found the next entry
                    pefag->chan_type = -1;
                    pefag->dest_type = -1;
                    pefag->last_successful = 0; // is this correct ?
                    break;
                }
            }
        }
        pefag->current_policy_index = i;
        
        // either this alert is processed or we found another valid destination
        if (pefag->current_policy_index == APT_SIZE) {
            // all policies checked and all destinations notified
            // notify the internals module to update the next_unprocessed_event
            aqe->alert_complete_handler(le16_to_cpu(event->id_le16));
            
            // delete this element
            list_del(next);
            free(aqe);
            //pp_bmc_log_debug("[PEF] policy completed");
            
            pefag->current_policy_index = -1;
            
            // call next_alert() again
            if (list_empty(&(pefag->alertq_root)) == 0) {
                goto restart;
            }
        }
        else {
            // this policy belongs to our set and is enabled
            // check if the corresponding trap should be sent
            //pp_bmc_log_debug("[PEF] checking policy #%d in policy set #%u", aqe->current_policy_index, policy.number);
            
            apply = -1;
            if (policy.policy == 0) {
                // always apply
                apply = 1;
            }
            if ((policy.policy == 1) || (policy.policy == 2) || (policy.policy == 3) || (policy.policy == 4)) {
                // apply only if previous destination was unsuccessful
                apply = 0;
                if (pefag->last_successful == 0)
                    apply = 1;
                if (policy.policy == 2) {
                    // break
                    pefag->chan_type = -2;
                    pefag->dest_type = -2;
                }
                if (policy.policy == 3) {
                    // goto next channel after processing
                    pefag->chan_type = policy.channel;
                }
                if (policy.policy == 4) {
                    // goto next destination after processing
                    pefag->dest_type = policy.destination;
                }
            }

            if (apply == -1) {
                pp_bmc_log_warn("[PEF] found invalid policy(%u) in policy entry #%d", policy.policy, pefag->current_policy_index);
                goto restart;
            } else if (apply == 1) {
                pp_bmc_log_debug("[PEF] evt %.4x/filt %d/pol %d -> alert ch%u/dst%u",  \
                                 le16_to_cpu(event->id_le16), aqe->filter_number, pefag->current_policy_index, policy.channel, policy.destination);
                pefag->destination_in_progress = 1;
                // alert has to call pp_bmc_pef_report_alert_status() after success or timeout
                send_alert(aqe, aqe->filter_number, pefag->current_policy_index, policy.channel, policy.destination);  // send_alert() will restart this function when it has finished
            } else {
                pp_bmc_log_debug("[PEF] evt%.4x/filt %d/pol %d -> drop ch%u/dst%u",  \
                                 le16_to_cpu(event->id_le16), aqe->filter_number, pefag->current_policy_index, policy.channel, policy.destination);
                goto restart;
            }
        }
        
    } // if (alerts exist)
}


/**
 * Called by the channels at the end of an alert to indicate whether
 * the alert was successful or not.
 * success == 1: destination successfully alerted
 * success == 0: alert destination failed
 */
void pp_bmc_pef_report_alert_status(void* ctx, unsigned char success) {
    alert_channel_t* ach;

    if (ctx == NULL) {
         /* regular alert ended */
 
        if (pefag->enabled == 1) {
            pefag->destination_in_progress = 0;
            pefag->last_successful = success;
        
            next_alert();
        }
    } else {
        /* this must be an immediate alert, context is pointer to source alert channel structure */
        ach = (alert_channel_t*)(ctx);
        
        if (success == 1) {
            ach->immediate_status = ALERT_IMMEDIATE_COMPLETE;
        } else {
            ach->immediate_status = ALERT_IMMEDIATE_NO_ACK;
        }
        
        // call next_alert() if no more immediate alerts pending and aqe exists
        if (pefag->alertq_root.next == &(pefag->alertq_root) && 
            (immediate_alerts_active() == 0))
        {
            next_alert();
        }
    }
}

/**
 * Send an alert to the specified channel, no further checks,
 * calls pp_bmc_pef_report_alert_status after alert processing has ended
 */
static void send_alert(alert_queue_entry_t* aqe,
                       int filter_index UNUSED,
                       int policy_index UNUSED,
                       int channel,
                       int destination)
{
    pef_configuration_t config;
    event_filter_entry_t filter;
    void* rv;

    unsigned char guid[16];
    //unsigned char alert_string[16*ALERT_STRING_BLOCK_CNT];

    struct list_head* act;
    alert_channel_t* ach;

    pef_get_cfg(&config);
    pef_get_event_filter(aqe->filter_number, &filter);

    // prepare alert parameters

    // system guid, either from pet config or global guid
    if (config.enable_system_guid == 1) {
        memcpy(guid, config.system_guid, 16);
    }
    else {
        pp_bmc_get_guid(guid);
    }
    
    /* TODO: get alert string here if we ever need it */
    
    act = pefag->channels.next;
    while(act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        
        if (ach->channel == channel) {
            pp_bmc_log_debug("[PEF] sending alert to channel %d, destination %d", channel, destination);
            aqe->acknowledge = ach->acknowledge;
            rv = ach->start_alert(destination, NULL, &(aqe->event), filter.event_severity, guid, 0, NULL); 
            if (rv != NULL) {
                aqe->alert_hndl = rv;
            }
            /* else:
             * Alert immediately finished, aqe is deleted ! */
                
            return;
        }
        act = act->next;
    }
    
    pp_bmc_log_warn("[PEF] alerts to channel %d not supported", channel);
    pp_bmc_pef_report_alert_status(NULL, 0);
}

int bmc_pef_acknowledge_alert(unsigned short alert_seq) {
    struct list_head* act;
    alert_queue_entry_t* aqe;
    alert_channel_t* ach;
    int has_active;
    int accepted = PP_ERR;

    has_active = 0;
    
    /* check aqe  */
    act = pefag->alertq_root.next;
    if (act != &(pefag->alertq_root)) {
        /* alert queue not empty, check first element */
        has_active = 1;
        aqe = list_entry(act, alert_queue_entry_t, link);
        if (le16_to_cpu(aqe->event.id_le16) == alert_seq) {
            // this alert is acknowledged
            if (aqe->acknowledge == NULL) {
                // very strange, should not happen, perhaps we are wrong, better dont do anything
                pp_bmc_log_warn("[PET] unable to acknowledge alert with aqe->acknowledge == NULL");
            }
            else {
		accepted = PP_SUC;
                aqe->acknowledge(aqe->alert_hndl);
            }
            return accepted;
        }
    }
    
    act = pefag->channels.next;
    while (act != &(pefag->channels)) {
        /* check immediate alert on this channel */
        ach = list_entry(act, alert_channel_t, link);
        if (ach->immediate_status == ALERT_IMMEDIATE_PENDING) {
            has_active = 1;
            if (le16_to_cpu(ach->immediate_event.id_le16) == alert_seq) {
		accepted = PP_SUC;
                ach->acknowledge(ach->immediate_alert_hndl);
            }
            return accepted;
        }
	act = act->next;
    }
    
    if (has_active == 1) {
        pp_bmc_log_warn("[PET] received acknowledge with unknown alert sequence");
    }
    return accepted;
}


/**
 * Registers a new channel alert module at the pef.
 * @return a handle used for unregistration
 */
int pp_bmc_pef_alert_register_channel(unsigned char channel,
                                   pp_bmc_alert_initiate_t start_alert,
                                   pp_bmc_alert_acknowledge_t acknowledge_alert)
{
    alert_channel_t* ach;
    ach = malloc(sizeof(alert_channel_t));
    
    ach->channel = channel;
    ach->start_alert = start_alert;
    ach->acknowledge = acknowledge_alert;
    ach->immediate_status = ALERT_IMMEDIATE_NO_STATUS;
    
    list_add_tail(&(ach->link), &(pefag->channels));
    
    return (int)(ach);
}

/**
 * Unregister a channel at the pef using the registration handle.
 */
int pp_bmc_pef_unregister_channel(int hndl) {
    struct list_head* act;
    alert_channel_t* ach;
    
    act = pefag->channels.next;
    while(act != &(pefag->channels)) {
        ach = list_entry(act, alert_channel_t, link);
        if ((int)ach == hndl) {
            list_del(act);
            free(ach);
            return PP_SUC;
        }
        act = act->next;
    }
    
    return PP_ERR;
}    
