/**
 * bmc_dev_pef_alert.h
 *
 * Encapsulates the alert handling of the PEF device. This module
 * alerts all necessary destinations of a certain alert policy,
 * deferres alerts if necessary and notifies the calling module
 * when an alert is completely processed.
 * 
 * Concrete channel alert implementations can register through
 * a generic interface.
 *
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __BMC_DEV_PEF_ALERT_H__
#define __BMC_DEV_PEF_ALERT_H__

#include "event_receiver.h"
#include "pp/bmc/bmc_channel_alert.h"

/**
 * Initialize alerting (alert queue).
 * Alerting starts disabled.
 */
int bmc_pef_alert_init(void);

/**
 * Cleanup the pef alert module. Stop pending alerts and
 * cleanup alertqueue.
 */
void bmc_pef_alert_cleanup(void);

/**
 * (Re)Start processing of pending alerts (used to indicate that
 * the alert startup delay has passed)
 */
void bmc_pef_enable_alert(void);

/**
 * Disable alert processing. Alerts are still accepted, but
 * are held until alerting is reenabled. Usually called in
 * conjunction with clear_alert(void)
 */
void bmc_pef_hold_alert(void);

/**
 * Clear all pending and all queued alerts. Usually called
 * in conjunction with hold_alert(void)
 */
void bmc_pef_clear_alert(void);

/**
 * Called by bmc_dev_pef_alert if an alert is completely processed.
 */
typedef void (*alert_complete_hndl_t)(unsigned short eventID);

/**
 * Add a new alert to the alert queue. The new alert
 * will be processed as soon as possible (when all older
 * alerts have been processed). The pef internals module
 * will be notified when the alert has been completely
 * processed.
 * @param eventID the id of the event that caused the alert
 * @param filterNo the index of the filter that caused the alert
 *        If filterNo equals -1 then the alert is empty. It has to be queued
 *        anyway to ensure that the next_unprocessed_event id is updated
 *        correct.
 * @param alert_complete_handler the function that is called when this alert
 *        is completely processed.
 */
void bmc_pef_add_alert(sel_entry_t* event, int filterNo, alert_complete_hndl_t alert_complete_handler);

/**
 * Try to initiate an immediate alert to the specified channel/destination.
 * The alert will be started only if no other immediate alert is active
 * on this channel and no pef alert is active overall.
 * @param channel the channel to alert
 * @param destination the destination to alert
 * @param alert_string the alert string number
 * @param event the event data to be packed in the alert, may contain empty values
 * @param alert_complete_handler this function will be called once the alert is
 *        finished.
 * @return the ipmi error code for this functionality, IPMI_ERR_SUCCESS if the
 *        alert was started.
 */
unsigned char bmc_pef_add_immediate_alert(unsigned char channel, 
                                unsigned char destination,
                                unsigned char alert_string,
                                sel_entry_t* event);

/**
 * Get the status of an immediate alert on the specified channel.
 */
unsigned char bmc_pef_get_immediate_alert_status(unsigned char channel);

/**
 * Clear the status of an immediate alert on the specified channel.
 */
void bmc_pef_clear_immediate_alert_status(unsigned char channel);

/**
 * Acknowledge an ipmi alert. Called by the ipmi pet acknowledge command to
 * forward acknowledgements to the pef alert module.
 * @param alert_seq the sequence number of the alert, can be taken directly
 *        from the command data
 */
int bmc_pef_acknowledge_alert(unsigned short alert_seq);


#endif
