/**
 * bmc_dev_pef_internals.c
 *
 * BMC Platform Event Filtering (PEF), Internal logic, filtering and
 * alert mechanisms.
 * 
 * TODO
 * - check lpe_bmc         (see below)
 * 
 * Notes:
 * - alerts can be processed before their corresponding actions during startup delay
 * - does not support ICMB groupcontrol
 * 
 * Last processed event mechanisms
 * As the usage of the 'last_processed_bmc' and 'last_processed_sms' event ids
 * as proposed in the spec is quite confusing and maybe unreliable, we chose to
 * keep two internal event ids for this purpose.
 * - the non volatile 'next_unprocessed_event' id is updated whenever an event
 *   is completly processed (including all alerts). 
 * - the volatile 'next_unqueued_event' id is updated whenever an event is
 *   filtered and the appropriate actions and alers are enqueued.
 * - the other ids (from the spec) are kept for compatibility reasons only
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */
#include <stdlib.h>

#include <pp/selector.h>

#include <pp/bmc/host_power.h>
#include <pp/bmc/bmc_event.h>
#include <pp/bmc/ipmi_bits.h>
#include <pp/bmc/ipmi_chan.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/debug.h>

#include "bmc_dev_pef_internals.h"
#include "bmc_dev_pef_alert.h"
#include "bmc_dev_pef_nv.h"
#include "bmc_dev_sdrr.h"
#include "event_receiver.h"



typedef struct {
    int timer_handle;
    unsigned char timer_interval;
    
    int startup_delay_handle;
    int action;
    int alert_delay_timer_handle;
    
    char enable;
    unsigned short next_unqueued_event;
    
} pef_internal_globals_t;

pef_internal_globals_t pef_internal_globals;
pef_internal_globals_t* pefig;


/** possible actions for addAction() */
#define ACTION_RESET         1  
#define ACTION_POWER_CYCLE   2
#define ACTION_POWEROFF      3

/** some internal function blueprints, not ordered */
static void check_timer(void);
static int pef_postpone_timeout(const int item_id, void* ctx);
static int applyFilter(event_filter_entry_t* filter, sel_entry_t* event);
static void addAction(int actionType);
static int startAction(const int item_id, void* ctx);
static int alert_delay_timeout(const int item_id, void* ctx);
static void alert_processed_handler(unsigned short eventID);
static void power_status_changed(int action);
static void pef_sel_add(unsigned short id);
static void pef_sel_delete(unsigned short id, unsigned short next_id);
static void pef_sel_cleared(void);
static int renew_startup_delays(void);
static void action(void);


static sel_listener_hndl_t selHandles = {
    .add   = pef_sel_add,
    .del   = pef_sel_delete,
    .clear = pef_sel_cleared,
};

static ipmi_sdr_eventonly_sensor_t pef_sdr = {
    header: {
        /* id: reserved to be used by SDRR */
        version: 0x51,
        type:    0x03,
        length:  sizeof(ipmi_sdr_eventonly_sensor_t) - sizeof(ipmi_sdr_header_t)
    },
    key: {
        owner_id:   0x20,
        lun:        0,
        channel:    IPMI_CHAN_PRIMARY_IPMB,
        sensor_num: IPMI_SENSOR_NUMBER_STATIC_PEF
    },
    entity: {
        id:         IPMI_ENTITY_ID_MGMT_CONTROLLER_FIRMWARE,
        instance:   0x1,
        is_logical: 0
    },
    
    sensor_type: IPMI_SENSOR_TYPE_SYSTEM_EVENT, 
    event_type: IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC,
    
    share: {
        count:      0,
        mod_type:   0,  // not sure if this is correct
        sens_dir:   0,
        mod_offset: 0,
        entity_inst:0
    }, 
    
    oem: 0,
    
    id: {
        length: 3, 
        code:   IPMI_STR_ASCII8,
        string: "pef"
    }
};



int bmc_pef_init() {
    pef_configuration_t config;

    bmc_pef_alert_init();

    pefig = &pef_internal_globals;
    pefig->timer_handle = -1;
    pefig->timer_interval = 0;
    pefig->action = -1;
    pefig->enable = 0;

    pp_bmc_host_power_control_subscribe(power_status_changed);
    
    pef_get_cfg(&config);
    pefig->next_unqueued_event = config.next_unprocessed_event;

    if (config.enable_pef == 1)  {
        // enable alert (postpone timer running)
        bmc_pef_enable();

        // we simply assume that the host is starting up right now and that we need a startup delay.
        if (renew_startup_delays() == PP_ERR)
            return PP_ERR;
    }

    // allocate pef_sdr because it gets free()d by clear-sdr.
    pp_bmc_sdrr_add_sdr_transiently(&pef_sdr.header);
    
    pp_event_set_listener(selHandles);

    return PP_SUC;
}

void bmc_pef_cleanup() {
    
    pp_bmc_host_power_control_unsubscribe(power_status_changed);
    selHandles.add = NULL;
    selHandles.del = NULL;
    selHandles.clear = NULL;
    pp_event_set_listener(selHandles);
    
    bmc_pef_alert_cleanup();
}

/**
 * Enable/disable pef and related functions (timers)
 */

void bmc_pef_enable() {
    if (pefig->enable == 1)
        return;  // already enabled
    
    pefig->enable = 1;
    // start postpone timer if necessary
    check_timer();
    // make sure alerts are enabled (they might have been stopped due to alert delays
    bmc_pef_enable_alert();
}

void bmc_pef_disable() {
    pef_configuration_t config;
    
    if (pefig->enable == 0)
        return;  // already disabled
        
    pefig->enable = 0;
    
    // stop postpone timers
    if (pefig->timer_handle != -1)
        pp_select_remove_to(pefig->timer_handle);
    pefig->timer_handle = -1;
    
    // stop alert delays
    if (pefig->startup_delay_handle != -1) 
        pp_select_remove_to(pefig->startup_delay_handle);
    pefig->startup_delay_handle = -1;
    if (pefig->alert_delay_timer_handle != -1) 
        pp_select_remove_to(pefig->alert_delay_timer_handle);
    pefig->alert_delay_timer_handle = -1;

    // clear all pending alerts
    bmc_pef_clear_alert();

    pef_get_cfg(&config);
    pefig->next_unqueued_event = 0xffff;
    config.next_unprocessed_event = 0xffff;
    pef_set_cfg(&config);
}

static void power_status_changed(int act) {
    if ((act == HOST_POWER_CONTROL_ON) ||
        (act == HOST_POWER_CONTROL_CYCLE) ||
        (act == HOST_POWER_CONTROL_RESET) ||
        (act == HOST_POWER_CONTROL_SOFT_RESET))
    {
        // renew startup delay to allow bios+os to run before we can power them down
        renew_startup_delays();
    }
}

/**
 * Init startup delay: disable alerts, clear alerts, start startup_delay timers
 */
static int renew_startup_delays(void) {
    pef_configuration_t config;
    pef_get_cfg(&config);
    
    // clear all pending alerts and hold new alerts until startup delay has passed
    bmc_pef_hold_alert();
    bmc_pef_clear_alert();
    
    pefig->startup_delay_handle = pp_select_add_to(config.startup_delay*1000, 0, startAction, NULL);
    if (pefig->startup_delay_handle < 0) {
        pp_bmc_log_error("[PEF] could not create startup delay timeout");
        return PP_ERR;
    }
    pefig->alert_delay_timer_handle = pp_select_add_to(config.alert_startup_delay*1000+1, 0, alert_delay_timeout, NULL);
    if (pefig->alert_delay_timer_handle < 0) {
        pp_bmc_log_error("[PEF] could not create alert startup delay timeout");
        return PP_ERR;
    }
    return PP_SUC;
}

/**
 * The alert delay has passed. Enable alerting and stop the timer.
 */
static int alert_delay_timeout(const int item_id UNUSED, void* ctx UNUSED) {
    pefig->alert_delay_timer_handle = -1;
    bmc_pef_enable_alert();
    return PP_SUC;
}

/**
 * Called to indicate that the startup delay has passed and actions can now be taken
 */
static int startAction(const int item_id UNUSED, void* ctx UNUSED) {
    //pp_bmc_log_debug("[PEF] startup delay timeout has passed, actions now enabled");
    pefig->startup_delay_handle = -1;  // not running any more
    if (pefig->action != -1)
        action();
    return PP_SUC;
}


void bmc_pef_set_lpe_bmc(unsigned short id) {
    pef_configuration_t config;
    unsigned short event;

    pef_get_cfg(&config);
    config.lpe_bmc = id;

    if (id == 0xffff)
        event = id;
    else
        event = pp_bmc_get_next_event_id(id);
    
    if (event == 0x0000) {
        // specified event does not exist, do nothing
    } else {
        // specified event exists, update next_unprocessed_event
        // TODO: we assume here that the given event id is newer than all existing ids
        pefig->next_unqueued_event = event;
        config.next_unprocessed_event = event;
        //pp_bmc_log_debug("[PEF] next unprocessed event set to %.4x", event);
    }
    
    pef_set_cfg(&config);

    check_timer();
}

void bmc_pef_set_lpe_sms(unsigned short id) {
    pef_configuration_t config;
    unsigned short event;

    pef_get_cfg(&config);
    config.lpe_sms = id;

    if (id == 0xffff)
        event = id;
    else
        event = pp_bmc_get_next_event_id(id);
    if (event == 0x0000) {
        // specified event does not exist, do nothing
    } else {
        // specified event exists, update next_unprocessed_event
        // TODO: we assume here that the given event id is newer than all existing ids
        pefig->next_unqueued_event = event;
        config.next_unprocessed_event = event;
        //pp_bmc_log_debug("[PEF] next unprocessed event set to %.4x", event);
    }
    
    pef_set_cfg(&config);

    check_timer();
}

static void pef_sel_add(unsigned short id) {
    pef_configuration_t config;
    pef_get_cfg(&config);
    if (pefig->enable == 1) {
        if (pefig->next_unqueued_event == 0xFFFF) {
            pefig->next_unqueued_event = id;
            config.next_unprocessed_event = id;
            pp_bmc_log_debug("[PEF] next unqueued event set to %.4x", id);
            pef_set_cfg(&config);
        }
        check_timer();
    } else {
        // pef disabled
        pefig->next_unqueued_event = 0xFFFF;    // should be 0xffff anyway
        config.next_unprocessed_event = 0xFFFF; // should be 0xffff anyway
        config.lpe_bmc = id;
        pef_set_cfg(&config);
    }
}

static void pef_sel_delete(unsigned short id, unsigned short next_id) {
    pef_configuration_t config;

    // sel deletes are only of interest if pef is enabled
    if (pefig->enable != 1)
        return;
    
    pef_get_cfg(&config);

    // if the first unprocessed event is deleted then update id
    if (config.next_unprocessed_event == id) {
        config.next_unprocessed_event = next_id;
        //pp_bmc_log_debug("[PEF] next unprocessed event set to %.4x", next_id);
        pef_set_cfg(&config);
    }
    if (pefig->next_unqueued_event == id) {
        pefig->next_unqueued_event = next_id;
        if (next_id == 0xFFFF)
            check_timer();
    }
}

static void pef_sel_cleared(void) {
    pef_configuration_t config;
    
    // sel clears are only of interest if pef is enabled
    if (pefig->enable != 1)
        return;

    pef_get_cfg(&config);
    config.next_unprocessed_event = 0xFFFF; // no unprocessed events
    pefig->next_unqueued_event = 0xFFFF;
    //pp_bmc_log_debug("[PEF] next unprocessed event set to %.4x", 0xffff);
    pef_set_cfg(&config);
    
    check_timer();
}

void bmc_pef_set_timer(unsigned char interval) {
    pefig->timer_interval = interval;
    check_timer();
}

unsigned char bmc_pef_get_timer() {
    return pefig->timer_interval;
}

/**
 * check if unprocessed events exist
 */
static int exist_unprocessed_events(void) {
    return (pefig->next_unqueued_event != 0xffff);
}

/**
 * Check the timer
 */
static void check_timer(void) {
    if (exist_unprocessed_events()) {
        //pp_bmc_log_debug("[PEF] has events to filter");
        // we have events to filter

        // I don't think that there are any other cases in which we have to
        // restart the timer (except perhaps reconfigurations) ?
        
        if (pefig->timer_handle == -1) {
            pefig->timer_handle = pp_select_add_to(pefig->timer_interval*1000, 0, pef_postpone_timeout, NULL);
            //pp_bmc_log_debug("[PEF] starting timer (%d)", pefig->timer_interval);
            if (pefig->timer_handle < 0) {
                // we cannot really recover here  -  just print error for now
                //pp_bmc_log_critical("[PEF] cannot add pef postpone timer");
            }
        }
    } else {
        // pp_bmc_log_debug("[PEF] no events to filter");
        // no (more) events to filter, timer is obsolete
        if (pefig->timer_handle != -1) {
            // stop timer
            //pp_bmc_log_debug("[PEF] stopping timer");
            if (pp_select_remove_to(pefig->timer_handle) == PP_ERR) {
                // we cannot really recover here  -  just print error for now
                //pp_bmc_log_critical("[PEF] cannot remove pef postpone timer");
            }
            pefig->timer_handle = -1;
        }
    }
}

/**
 * A postpone timeout has occured and now we have to filter unprocessed events
 */
static int pef_postpone_timeout(const int item_id UNUSED, void* ctx UNUSED) {
    event_filter_entry_t filter;
    sel_entry_t* event;
    int i;
    int match;
    
    int cur_action;
    int policyNo;
    int filterIndex;
    
    pef_configuration_t config;
    if (pef_get_cfg(&config) != PP_SUC) memset(&config, 0, sizeof(pef_configuration_t));

    //pp_bmc_log_debug("[PEF] postpone timeout occured");
    
    pefig->timer_handle = -1;  // the timer is not running any more
    while(exist_unprocessed_events()) {
        // get event, check if valid
        event = pp_bmc_get_event(pefig->next_unqueued_event);
        if (event == NULL) {
            // somehow the event disappeared. this should not happen, but who knows
            pp_bmc_log_error("[PEF] next_unqueued_event not in SEL. Setting to 0xffff");
            pefig->next_unqueued_event = 0xffff;
            return PP_SUC;
        }
        
        // handle this event as queued/processed from now on
        pefig->next_unqueued_event = pp_bmc_get_next_event_id(pefig->next_unqueued_event);
        if (pefig->next_unqueued_event == 0x0000)
            pefig->next_unqueued_event = 0xffff;

        cur_action = -1;
        policyNo = 1000;
        filterIndex = -1;
        
        //pp_bmc_log_debug("[PEF] checking event %.4x", le16_to_cpu(event->id_le16)); 
        for (i=0; i<EFT_SIZE; i++) {
            pef_get_event_filter(i, &filter);
            match = applyFilter(&filter, event);
            if (match == 1) {
                pp_bmc_log_debug("[PEF] Event %.4x matches filter %d", le16_to_cpu(event->id_le16), i);
                
                if ((filter.action.reset == 1) &&
		    (cur_action < ACTION_RESET)) {
                    cur_action = ACTION_RESET;
		}
                if ((filter.action.power_cycle == 1) &&
		    (cur_action < ACTION_POWER_CYCLE)) {
                    cur_action = ACTION_POWER_CYCLE;
		}
                if (filter.action.poweroff == 1) {
                    cur_action = ACTION_POWEROFF;
		}
                if (filter.action.alert == 1) {
                    if (filter.alert_policy_number < policyNo) {
                        policyNo = filter.alert_policy_number;
                        filterIndex = i;
                    }
                }
            } // if (match == 1)
        } // for all filters

        // create action event
        if ( ((cur_action != -1) || (policyNo != 1000)) &&                   // if that event matched a filter
             (config.enable_action_events == 1) &&                       // and event messages for actions enabled
             (event->sys.sensor_no != IPMI_SENSOR_NUMBER_STATIC_PEF))    // make sure that the user cannot generate endless loops
        {
            unsigned char data2 = 0;
            switch(cur_action) {
                case ACTION_POWEROFF: data2 = 0x02; break;
                case ACTION_RESET:    data2 = 0x04; break;
                case ACTION_POWER_CYCLE: data2 = 0x08; break;
            }
            if (policyNo != 1000) data2 |= 0x01;

            pp_bmc_receive_event(0, 0x20, 
                                 0, IPMI_CHAN_PRIMARY_IPMB, 
                                 IPMI_SENSOR_TYPE_SYSTEM_EVENT, IPMI_SENSOR_NUMBER_STATIC_PEF,
                                 0, IPMI_EVENT_READING_TYPE_SENSOR_SPECIFIC,
                                 0xb4,   // offset4, data2 sensor specific, data3 unspecified
                                 data2, 0);
        }
        
        if (cur_action != -1) {
            addAction(cur_action);
        }
        if (policyNo != 1000) {
            bmc_pef_add_alert(event, filterIndex, alert_processed_handler);
        } else {
            // add empty alert to ensure proper next_unprocessed_event update
            bmc_pef_add_alert(event, -1, alert_processed_handler);
        }
        
        free(event);
    } // while has_unprocessed_events
    
    return PP_SUC;
}

/**
 * Apply the specified filter on the specified event.
 * @returns 1 if the event matches the filter
 * @returns 0 if the event does not match the filter
 */
static int applyEventDataMask(unsigned char value, unsigned char *mask) {
    unsigned char match;
    unsigned char nonExactMask = ~(mask[1]);

    //pp_bmc_log_debug("[PEF] applyEventDataMask: value=%.2x, mask=%.2x, exact=%.2x, data=%.2x", value, mask[0], mask[1], mask[2]);
    
    // mask value
    value = value & (mask[0]);
    
    // value XOR data
    // bitwise not
    // => matching bits are 1, non-matching bits are 0
    match = ~(value ^ (mask[2]));
   
    // chose the bits that should match exact and check that they match exact
    if ((match & mask[1]) != mask[1])
        return 0;

    // if we have bits that should match nonexact, at lease one bit must be set
    if ( (nonExactMask > 0) & ((match & nonExactMask) == 0))
        return 0;

    // no problems encountered. value matches
    return 1;
}

/**
 * Apply the specified filter on the specified event.
 * @returns 1 if the event matches the filter
 * @returns 0 if the event does not match the filter
 */
static int applyFilter(event_filter_entry_t* filter, sel_entry_t* event) {
    int match = 0;
    unsigned char* eventB = (unsigned char*)event;
    unsigned char d1LN;
    unsigned short cmpByte;

    if (filter->enable == 1) {
        // Apply this filter
        match = 1;
        if ((filter->generatorID1 != 0xFF) &&
            (filter->generatorID1 != (event->sys.addr_swid & 0x7F)) )
            {
                match = 0;
            }
        if ((filter->generatorID2 != 0xFF) &&
            (filter->generatorID2 != eventB[8]) )
            {
                match = 0;
            }
        if ((filter->sensor_type != 0xFF) &&
            (filter->sensor_type != event->sys.sensor_type) )
            {
                match = 0;
            }
        if ((filter->sensor_number != 0xFF) &&
            (filter->sensor_number != event->sys.sensor_no) )
            {
                match = 0;
            }
        if ((filter->event_trigger != 0xFF) &&
            (filter->event_trigger != eventB[12]) )
            {
                match = 0;
            }
        //pp_bmc_log_debug("[PEF] match before nibble compare = %d", match);

        // now compare the remaining three event data bytes
        // compare data 1 lower nibble
        d1LN = event->sys.event_data[0] & 0x0F;  // get the least significant nibble
        cmpByte = le16_to_cpu(filter->offset_mask);
        //pp_bmc_log_debug("data1=%.2x  cmyByte=%.4x  d1LN=%.2x", event->sys.event_data[0], cmpByte, d1LN);
        
        // the value of d1LN is the number of the bit in cmpByte that must be set to 1
        // range d1LN: 0..15
        cmpByte = cmpByte >> d1LN;
        if ((cmpByte & 0x01) == 0) {
            match=0;
        }

        //pp_bmc_log_debug("[PEF] match before free compare masks = %d", match);

        // now compare the 3 data bytes with the masks
        if (applyEventDataMask(event->sys.event_data[0],
                               (unsigned char*)&(filter->data_filter[0])) == 0)
            match=0;
        if (applyEventDataMask(event->sys.event_data[1],
                               (unsigned char*)&(filter->data_filter[1])) == 0)
            match=0;
        if (applyEventDataMask(event->sys.event_data[2],
                               (unsigned char*)&(filter->data_filter[2])) == 0)
            match=0;
    } // if filter is enabled
    
    return match;
}


/** Process all stored actions */
static void action(void) {
    pp_bmc_log_debug("[PEF] action %d", pefig->action);

    if (pefig->action == ACTION_POWEROFF) {
        pp_bmc_host_power_control(HOST_POWER_CONTROL_OFF, HOST_POWER_CONTROL_SOURCE_PEF_RESET, 0);
    } else if (pefig->action == ACTION_POWER_CYCLE) {
        pp_bmc_host_power_control(HOST_POWER_CONTROL_CYCLE, HOST_POWER_CONTROL_SOURCE_PEF_CYCLE, 0);
    } else if (pefig->action == ACTION_RESET) {
        pp_bmc_host_power_control(HOST_POWER_CONTROL_RESET, HOST_POWER_CONTROL_SOURCE_PEF_RESET, 0);
    }
    
    pefig->action = -1;
}

/**
 * Add a new action (reset/powercycle/poweroff) to the pef internal action
 * queue. The queue will handle this action as soon as possible. (After
 * startup delay and if the action is allowed)
 */
static void addAction(int actionType) {
    pef_configuration_t config;
    pef_get_cfg(&config);
    
    //pp_bmc_log_debug("[PEF] addAction()");

    // ignore selected action if it is not enabled
    if ((actionType == ACTION_RESET) && (config.enable_reset == 0))
        return;
    if ((actionType == ACTION_POWEROFF) && (config.enable_poweroff == 0))
        return;
    if ((actionType == ACTION_POWER_CYCLE) && (config.enable_power_cycle == 0))
        return;

    if (pefig->action < actionType)
        pefig->action = actionType;
    // else no need to do low priority action if we have a pending high priority action

    if (pefig->startup_delay_handle == -1)
        action();
    // else deferred action
}

/**
 * Called by the alert module once the alert associated with the
 * specified eventID is completely processed
 */
static void alert_processed_handler(unsigned short eventID) {
    pef_configuration_t config;
    pef_get_cfg(&config);
    config.lpe_bmc = eventID;
    config.next_unprocessed_event = pp_bmc_get_next_event_id(eventID);
    if (config.next_unprocessed_event == 0x0000)
        config.next_unprocessed_event = 0xffff;
    pef_set_cfg(&config);
    
    pp_bmc_release_event(eventID);
}

