/**
 * \file bmc_dev_sdrr.c
 *
 * Description: BMC Sensor Data Record Repository Device
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <pp/base.h>
#include <pp/vector.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/ipmi_sdr.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/bmc_config.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/utils.h>
#include <pp/atomic_write.h>
#include "bmc_dev_sdrr.h"
#include "bmc_dev_sel.h"  // "get/set time" 

static const char SDR_FILE_MAGIC[] = "PPBMC-SDR";
#define SDR_FILE_VER   2

struct sdrr_file_header_s {
    char magic[16];
    int ver;
    int offs; /* offset to data: array of ipmi_sdr_header_t */
} __attribute__ ((packed));
typedef struct sdrr_file_header_s sdrr_file_header_t;

/* static data */
static struct {
    vector_t *repo; /* vector of ipmi_sdr_header_t* */

    reserv_t *reserv;
    
    char *file_name;
    int update_mode;
    unsigned char is_persistent;

    time_t last_add_time;
    time_t last_del_time;
} sdrr = { NULL, NULL, NULL, 0, 0, 0, 0 };

/*
 * SDR file methods
 */

static int sdrr_storage_load(void)
{
    assert(sdrr.file_name);
    assert(sdrr.repo);
    
    if (pp_fa_at_regular_file_absent(sdrr.file_name)) {
	return PP_SUC;
    }
    sdrr.is_persistent = 1;

    int fd = pp_fa_at_open(sdrr.file_name, PP_FA_AT_READ);
    if (fd == -1) {
	return PP_ERR;
    }
    /* read file header */
    sdrr_file_header_t filehdr;
    if (pp_fa_at_read(fd, &filehdr, sizeof(filehdr)) != sizeof(filehdr)) {
        pp_fa_at_close(fd);
        return PP_ERR;
    }
    lseek(fd, filehdr.offs, SEEK_CUR);

    /* loop, reading records */
    int i;
    for (i = 0; ; i++) {
        /* read header */
        ipmi_sdr_header_t hdr;
        int c = pp_fa_at_read(fd, &hdr, sizeof(hdr));
        if (c == 0) break; /* EOF: gracefully exit the loop */
        else if (c != sizeof(hdr)) {
            pp_fa_at_close(fd);
            return PP_ERR; /* corrupted header */
        }
        assert(i == le16_to_cpu(hdr.id_le16));
        void *sdr = malloc(sizeof(hdr) + hdr.length);
        memcpy(sdr, &hdr, sizeof(hdr));

        /* read body */
        if (pp_fa_at_read(fd, sdr + sizeof(hdr), hdr.length) != hdr.length) {
            pp_fa_at_close(fd);
            return PP_ERR; /* corrupted body */
        }
        
        vector_add(sdrr.repo, sdr);
    }

    pp_fa_at_close(fd);
    return PP_SUC;
}

static int sdrr_storage_write(void)
{
    assert(sdrr.file_name);
    assert(sdrr.repo);

    int fd = pp_fa_at_open(sdrr.file_name, PP_FA_AT_WRITE);
    if (fd == -1) {
	return PP_ERR;
    }
    sdrr.is_persistent = 1;
    
    /* write header */
    sdrr_file_header_t filehdr;
    memset(filehdr.magic, 0, sizeof(filehdr.magic));
    strcpy(filehdr.magic, SDR_FILE_MAGIC);
    filehdr.ver = SDR_FILE_VER;
    filehdr.offs = 0;
    pp_fa_at_write(fd, &filehdr, sizeof(filehdr));

    /* loop, writing records */
    unsigned int i;
    for (i = 0; i < vector_size(sdrr.repo); i++) {
	ipmi_sdr_header_t* sdr = vector_get(sdrr.repo, i);
	size_t sz = (sizeof(ipmi_sdr_header_t) + sdr->length);
	if (pp_fa_at_write(fd, sdr, sz) != sz) {
	    pp_bmc_log_perror("[SDR] sdrr_storage_write()");
	    pp_fa_at_close(fd);

	    return PP_ERR;
	}
    }

    if (pp_fa_at_close(fd) == PP_ERR) {
	return PP_ERR;
    }
    return PP_SUC;
}

static int sdrr_storage_check(void)
{
    sdrr_file_header_t filehdr;
    ssize_t s;
    int fd;
    assert(sdrr.file_name);

    if (pp_fa_at_regular_file_absent(sdrr.file_name)) {
	return PP_SUC;
    }

    if (0 > (fd = pp_fa_at_open(sdrr.file_name, PP_FA_AT_READ))) {
	return PP_ERR;
    }

    /* read header */
    s = pp_fa_at_read(fd, &filehdr, sizeof(filehdr));
    pp_fa_at_close(fd);

    if (s != sizeof(filehdr)
	|| 0 != strcmp(filehdr.magic, SDR_FILE_MAGIC)
	|| filehdr.ver != SDR_FILE_VER) {
        /* bad version or corrupted */
        /* (conversion/repairing goes here in future versions) */
	pp_bmc_log_perror("[SDRR] invalid file, deleting!");
	if (0 > unlink(sdrr.file_name)) {
	    pp_bmc_log_perror("[SDRR] sdrr_storage_check(): unlink(%s)", 
			      sdrr.file_name);
	    return PP_ERR;
	}
	return PP_SUC;
    }

    return PP_SUC;
}

/****************************
 * SDRR operations, callable locally, intended for devices that
 * need to add transient SDRs, such as Watchdog, etc.
 */
int pp_bmc_sdrr_is_persistent() {
    return sdrr.is_persistent;
}

ipmi_sdr_header_t*
pp_bmc_sdrr_lookup_by_key_sensor(ipmi_sdr_key_sensor_t* key) {
    size_t i, s = vector_size(sdrr.repo);
    ipmi_sdr_header_t* sdr;
    ipmi_sdr_key_sensor_t* sdrkey;
    
    if (key == NULL) return NULL;
    
    for (i = 0; i < s; ++i) {
	sdr = vector_get(sdrr.repo, i);
	if (NULL == (sdrkey = ipmi_sdr_get_key_sensor_ptr(sdr))) {
	    continue;
	}
	if (memcmp(sdrkey, key, sizeof(ipmi_sdr_key_sensor_t)) == 0) {
	    return sdr;
	}
    }
    return NULL;
}

static unsigned short sdrr_add_sdr(ipmi_sdr_header_t* sdr) {
    ipmi_sdr_header_t* newsdr;
    assert(sdr != NULL);
    newsdr = ipmi_sdr_copy(sdr);
    newsdr->id_le16 = cpu_to_le16(vector_size(sdrr.repo)); /* new record id */
    vector_add(sdrr.repo, newsdr);
    return newsdr->id_le16;
}

unsigned short pp_bmc_sdrr_add_sdr_transiently(ipmi_sdr_header_t* sdr) {
    if (pp_bmc_sdrr_is_persistent()) return PP_ERR;
    return sdrr_add_sdr(sdr);
}

/********************************************************************
 * SDRR command handlers
 */

/*
 * Get SDR Repository Info
 */

#define IPMI_SDR_SPACE_FULL     0x0000
#define IPMI_SDR_SPACE_HUGE     0xfffe
#define IPMI_SDR_SPACE_UNSPEC   0xffff

#define IPMI_SDR_SUPP_MODAL         0x40
#define IPMI_SDR_SUPP_NON_MODAL     0x20
#define IPMI_SDR_SUPP_DELETE        0x08
#define IPMI_SDR_SUPP_PARTIAL_ADD   0x04
#define IPMI_SDR_SUPP_RESERVE       0x02
#define IPMI_SDR_SUPP_GET_ALLOC     0x01
 
struct get_sdrr_info_rs_s {
    unsigned char ver;
    unsigned short cnt_le16;
    unsigned short free_le16; /* see IPMI_SDR_SPACE_xxx */
    unsigned int last_add_time_le32;
    unsigned int last_del_time_le32;
    BITFIELD2(unsigned char, supported : 7, overflow : 1);
} __attribute__ ((packed));
typedef struct get_sdrr_info_rs_s get_sdrr_info_rs_t;

static int sdrr_cmd_get_sdrr_info(imsg_t *imsg)
{
    assert(sdrr.repo);
    get_sdrr_info_rs_t *rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(get_sdrr_info_rs_t));

    rs->ver = 0x51; /* IPMI v1.5 */
    rs->cnt_le16 = cpu_to_le16(vector_size(sdrr.repo));
    rs->free_le16 = cpu_to_le16(IPMI_SDR_SPACE_HUGE); /* we provide unlimited space :) */
    rs->last_add_time_le32 = cpu_to_le32(sdrr.last_add_time);
    rs->last_del_time_le32 = cpu_to_le32(sdrr.last_del_time);
    rs->supported = IPMI_SDR_SUPP_MODAL | IPMI_SDR_SUPP_RESERVE;
    rs->overflow = 0;

    return pp_bmc_router_send_msg(imsg);
}

/*
 * Reserve SDR Repository
 */

static int sdrr_cmd_reserve_sdrr(imsg_t *imsg)
{
    reserv_cancel(sdrr.reserv);
    unsigned short id = cpu_to_le16(reserv_make(sdrr.reserv, imsg));
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &id, sizeof(id));
}

/*
 * Get SDR
 */

struct get_sdr_rq_s {
    unsigned short res_id_le16; /* 0x0000 = unspec */
    unsigned short rec_id_le16;
    unsigned char offs;
    unsigned char size; /* 0xff = all */
} __attribute__ ((packed));
typedef struct get_sdr_rq_s get_sdr_rq_t;

struct get_sdr_rs_s {
    unsigned short rec_id_le16;
    unsigned char data[0];
} __attribute__ ((packed));
typedef struct get_sdr_rs_s get_sdr_rs_t;

static int sdrr_cmd_get_sdr(imsg_t *imsg)
{
    assert(sdrr.repo);
    get_sdr_rq_t *rq = (void*)imsg->data;
    unsigned short rec_id = le16_to_cpu(rq->rec_id_le16);
    unsigned char  rq_offs = rq->offs;
    get_sdr_rs_t* rs;

    /* reservation is required for partial reads with non-zero offset only*/
    int needs_reserv = (rq_offs == 0) ?
	PP_BMC_RESERV_OPTIONAL : PP_BMC_RESERV_MANDATORY;
    
    /* check reservation */
    if (sdrr.update_mode) {
	assert(le16_to_cpu(rq->res_id_le16) == 0);
	if (rq->res_id_le16 != 0) {
	    pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }
    else if (PP_ERR == reserv_check(sdrr.reserv, le16_to_cpu(rq->res_id_le16),
				    needs_reserv, imsg))
    {
	pp_bmc_log_error("[SDR] reservation invalid (partial Get SDR (offs=%d size=%d resid=%d)",
			 rq->offs, rq->size, le16_to_cpu(rq->res_id_le16));
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }

    if (rec_id == 0xFFFF) {
        /* get last record */
        rec_id = vector_size(sdrr.repo) - 1;
    }
    
    if (rec_id >= vector_size(sdrr.repo)) {
        pp_bmc_log_error("[SDR] invalid rec_id %d, repo size is %d", 
                         rec_id, vector_size(sdrr.repo));
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }

    ipmi_sdr_header_t* sdr = vector_get(sdrr.repo, rec_id);
    size_t rs_size = sizeof(ipmi_sdr_header_t) + sdr->length - rq_offs;
    /* rs_size is the size that could be sent */
    if ((rq->size != 0xff) && (rq->size < rs_size)) {  /* cut size if needed */
        rs_size = rq->size;
    }
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS,
				 sizeof(get_sdr_rs_t) + rs_size);
    /* calculate next record id */
    rec_id++;
    rs->rec_id_le16 = (rec_id >= vector_size(sdrr.repo)) ?
	0xffff : cpu_to_le16(rec_id);
    memcpy(rs->data, ((unsigned char*)sdr) + rq_offs, rs_size);

    return pp_bmc_router_send_msg(imsg);
}
 
/*
 * Add SDR
 */

static int sdrr_cmd_add_sdr(imsg_t *imsg)
{
    assert(sdrr.repo);
    ipmi_sdr_header_t *sdr = (void*)imsg->data;
    unsigned short rid;

    /* body length validation */
    if (sizeof(ipmi_sdr_header_t) + sdr->length > imsg->data_size)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_INVALID);

    rid = sdrr_add_sdr(sdr);
    if (PP_ERR == sdrr_storage_write()) {
	// The added SDRR doesn't fit into flash. Also remove it from RAM.
	vector_remove(sdrr.repo, vector_size(sdrr.repo) - 1);
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_OUT_OF_SPACE);
    }

    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &rid, sizeof(rid));
}
 
/*
 * Clear SDR
 */

struct clear_sdrr_rq_s {
    unsigned short res_id_le16;
    unsigned int magic_le24 : 24; /* see IPMI_ERASE_MAGIC */
    unsigned char cmd; /* see IPMI_ERASE_xxx */
} __attribute__ ((packed));
typedef struct clear_sdrr_rq_s clear_sdrr_rq_t;

static int sdrr_cmd_clear_sdrr(imsg_t *imsg)
{
    assert(sdrr.repo);
    clear_sdrr_rq_t *rq = (void*)imsg->data;

    /* check reservation */
    if (sdrr.update_mode) {
	assert(le16_to_cpu(rq->res_id_le16) == 0);
	if (rq->res_id_le16 != 0) {
	    pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }
    else if (PP_ERR == reserv_check(sdrr.reserv, le16_to_cpu(rq->res_id_le16),
				    PP_BMC_RESERV_MANDATORY, imsg)) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_RESERVATION);
    }
    /* security double check for heavy operation */
    if (rq->magic_le24 != cpu_to_le24(IPMI_ERASE_MAGIC)) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_INVALID_DATA_FIELD);
    }

    /* really want to erase...? */
    if (rq->cmd == IPMI_ERASE_INITIATE) {
        int i;
 
	for (i = vector_size(sdrr.repo) - 1; i >= 0; --i) {
	    vector_remove(sdrr.repo, i);
	}
	sdrr_storage_write(); // write the file header
        sdrr.last_del_time = time(NULL);
        reserv_cancel(sdrr.reserv);
    }

    /* erease complete (always work synchron) */
    unsigned char state = 0x1;
    return pp_bmc_router_resp_msg(imsg, IPMI_ERR_SUCCESS, &state, sizeof(state));
}

/*
 * Enter SDR Update Mode
 */

static int sdrr_cmd_enter_sdrr_update_mode(imsg_t *imsg)
{
    if (sdrr.update_mode)
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_SDR_REPO_IN_UPDATE_MODE);
    
    sdrr.update_mode = 1;
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/*
 * Exit SDR Update Mode
 */

static int sdrr_cmd_exit_sdrr_update_mode(imsg_t *imsg)
{
    if (!sdrr.update_mode)
        return pp_bmc_router_resp_err(imsg,
				      IPMI_ERR_NOT_SUPPORTED_IN_PRESENT_STATE);
    
    sdrr.update_mode = 0;
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

/********************************************************************
 * SDR device command table
 */

static const dev_cmd_entry_t sdrr_cmd_tab[] = {
    {
        .cmd_hndlr = sdrr_cmd_get_sdrr_info,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SDR_REPO_INFO,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sdrr_cmd_reserve_sdrr,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_RESERVE_SDR_REPO,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sdrr_cmd_get_sdr,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SDR,
        .min_data_size = sizeof(get_sdr_rq_t), .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = sdrr_cmd_add_sdr,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_ADD_SDR,
        .min_data_size = sizeof(ipmi_sdr_header_t),
	.min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = sdrr_cmd_clear_sdrr,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_CLEAR_SDR_REPO,
        .min_data_size = sizeof(clear_sdrr_rq_t), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = pp_sel_cmd_get_sel_sdrr_time,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_GET_SDR_REPO_TIME,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_USER
    },
    {
        .cmd_hndlr = pp_sel_cmd_set_sel_sdrr_time,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_SET_SDR_REPO_TIME,
        .min_data_size = sizeof(int), .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = sdrr_cmd_enter_sdrr_update_mode,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_ENTER_SDR_REPO_UPDATE_MODE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_OPERATOR
    },
    {
        .cmd_hndlr = sdrr_cmd_exit_sdrr_update_mode,
        .netfn = IPMI_NETFN_STORAGE, .cmd = IPMI_CMD_EXIT_SDR_REPO_UPDATE_MODE,
        .min_data_size = 0, .min_priv_level = IPMI_PRIV_OPERATOR
    },
    { .cmd_hndlr = NULL }
};

/********************************************************************
 * SDR device c'tor/d'tor
 */

int pp_bmc_dev_sdrr_init()
{
    /* build file name of persistance storage */
    sdrr.file_name = malloc(strlen(PP_BMC_FLASH_ROOT) + sizeof("/sdr"));
    strcat(strcpy(sdrr.file_name, PP_BMC_FLASH_ROOT), "/sdr");

    /* check existance and version of persistan storage, and try to create it*/
    if (PP_ERR == sdrr_storage_check()) {
        if (errno != ENOENT) {
            pp_bmc_log_error("[SDR] corrupted storage file %s",
			     sdrr.file_name);
            return PP_ERR;
        }
    }


    sdrr.repo = vector_new(NULL, 0, free);

    if (PP_ERR == sdrr_storage_load()) {
        pp_bmc_log_perror("[SDR] can't read storage file %s, though it exists",
			  sdrr.file_name);
        return PP_ERR;
    }

    sdrr.reserv = reserv_new();

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(sdrr_cmd_tab)) return PP_ERR;

    pp_bmc_log_info("[SDRR] device started");
    return PP_SUC;
}

void pp_bmc_dev_sdrr_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(sdrr_cmd_tab);

    reserv_delete(sdrr.reserv); sdrr.reserv = NULL;
    vector_delete(sdrr.repo); sdrr.repo = NULL;
    free(sdrr.file_name); sdrr.file_name = NULL;
    
    pp_bmc_log_info("[SDRR] device shut down");
}
