/**
 * bmc_dev_sel_nv.h
 *
 * Provides a non volatile storage for SEL entries. The entries are ordered
 * as a linked list and can be retrieved by their id. The id does not have to
 * be sequential. Internally the entries are stored in an array and the
 * next id is calculated by retrieving the id of the next element of the
 * array.
 * 
 * (c) 2004 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __BMC_DEV_SEL_NV_H__
#define __BMC_DEV_SEL_NV_H__

#include <pp/bmc/utils.h>
#include <pp/bmc/bmc_sel_record.h>



#define SEL_MAX_SIZE 512

#define IPMI_SEL_FIRST_REC  0x0000
#define IPMI_SEL_LAST_REC   0xFFFF


/**
 * Initialize the SEL file module.
 * @returns PP_SUC after correct initialization
 * @returns PP_ERR if the initialization failed
 */
int sel_nv_init(void);

/**
 * Cleanup the SEL file module.
 */
void sel_nv_cleanup(void);

/**
 * Get the current size of the SEL.
 */
int sel_nv_get_size(void);

/**
 * Get the id of the first element in the SEL.
 * @returns the id of the first element in the SEL
 * @returns 0xffff if the SEL is empty
 */
unsigned short sel_nv_get_first_event_id(void);

/**
 * Get the id of the last (newest) sel record.
 */
unsigned short sel_nv_get_last_event_id(void);

/**
 * Get an unused SEL id for new events
 */
unsigned short sel_nv_get_next_entry_id(void);

/**
 * Retrieve the event with the specified id from the SEL.
 * The event will be copied to the specified address. The
 * caller must provide the necessary memory.
 * 
 * If next_id is not NULL, the id of the next element will also
 * be returned (0xffff if the current event is the last event)
 * 
 * @returns PP_SUC if the event could be retrieved 
 * @returns PP_ERR if the event with the specified id does not exist
 */
int sel_nv_get_event(unsigned short event_id, sel_entry_t* entry, unsigned short* next_id);

/**
 * Add an event to the SEL. It will be inserted at the end of the
 * linked list. The id of the entry must already be set.
 * 
 * @returns the id of the newly created SEL entry
 * @returns 0xFFFF if the event could not be added
 */
unsigned short sel_nv_add_event(sel_entry_t* entry);

/**
 * Delete the event with the specified id from the SEL.
 * @returns PP_SUC if successful
 * @returns PP_ERR if the event with the specified id does not exist.
 */
int sel_nv_delete_event(unsigned short id);

/**
 * Clear the SEL and delete all entries.
 * @returns PP_SUC if successful
 * @returns PP_ERR if the SEL could not be cleared
 */
int sel_nv_clear_sel(void); 


/**
 * Check the consistency of the SEL. If this succeeds, all future operations
 * on the file should also succeed.
 * @returns PP_SUC if the SEL exists and has the correct number of records.
 * @returns PP_ERR if no valid SEL could be read.
 */
int sel_nv_complete_check(void);

/**
 * Creates a fresh and empty new SEL file and enable SEL logging.
 * @returns PP_SUC after correct initialization
 * @returns PP_ERR if the initialization failed
 */
int sel_nv_create(void);


/**
 * Notification handler for new sel entries
 */
typedef void (*sel_add_hndl_t)(unsigned short id);

/**
 * Notification handler for deleted sel entries
 */
typedef void (*sel_delete_hndl_t)(unsigned short id, unsigned short next_id);

/**
 * Notification handler for sel clears
 */
typedef void (*sel_clear_hndl_t)(void);

struct sel_listener_hndl_s {
    sel_add_hndl_t    add;
    sel_delete_hndl_t del;
    sel_clear_hndl_t  clear;
};
typedef struct sel_listener_hndl_s sel_listener_hndl_t;

/**
 * Set an event receiver listener. This listener is notified of
 * changes in the sel. The device provides two listeners, one
 * for the event receiver and one for the sel device.
 */
void sel_set_listener_evt_r(sel_listener_hndl_t handlers);
void sel_set_listener_sel_dev(sel_listener_hndl_t handlers);



#endif
