/**
 * bmc_dev_sensor.c
 *
 * BMC sensor device 
 * 
 * (c) 2004 Peppercon AG, 12/14/2004, tbr@peppecon.de
 */
#include <malloc.h>
#include <pp/base.h>
#include <pp/cfg.h>
#include <pp/rb_tree.h>
#include <pp/bmc/debug.h>
#include <pp/bmc/ipmi_err.h>
#include <pp/bmc/ipmi_cmd.h>
#include <pp/bmc/ipmi_sess.h>
#include <pp/bmc/bmc_imsg.h>
#include <pp/bmc/bmc_router.h>
#include <pp/bmc/bmc_core.h>
#include <pp/bmc/utils.h>
#include <pp/bmc/topo_classes.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "bmc_dev_sdrr.h"
#include "bmc_dev_sensor.h"


/* the sensor table,
 * i.e. the list of all sensors managed by this device */
static pp_rb_tree_t* sensor_map = NULL;


/* some private prototypes */
static int sensor_config_addsensor(unsigned char sensor_nr,
                                   pp_tp_ipmi_sens_t* sensor);


#if defined (PP_FEAT_IPMI_THRESHOLD_PERSIST)
//static const char *bmc_sensors_name = "bmc.sensors[%u].name";
static const char *bmc_sensors_thresh = "bmc.sensors[%u].thresh.%s";
#endif /* PP_FEAT_IPMI_THRESHOLD_PERSIST */


/**************************************
 * static helpers
 */
static void sensor_delete(pp_mallocator_t* a UNUSED, void* sensor) {
    pp_tp_ipmi_sens_release((pp_tp_ipmi_sens_t*)sensor);
}

/*************************************
 * misc stuff
 */
static int sd_sensor_add_sensor(unsigned char sensor_nr,
				 pp_tp_ipmi_sens_t* sensor) {
    if (0 != pp_rb_tree_insert(sensor_map, (void*)(long)sensor_nr,
			       pp_tp_ipmi_sens_duplicate(sensor),
			       sensor_delete, 0)) {
	errno = EINVAL;
	return PP_ERR;
    }
    return PP_SUC;
}

pp_tp_ipmi_sens_t* get_sensor_by_index(unsigned char idx)
{
    return pp_rb_tree_search(sensor_map, (void*)(long)idx);
}

/********************************************************************
 * sensor device command handlers
 */

static int sd_cmd_get_sensor_reading_factors(imsg_t *imsg)
{
    ipmi_get_sensor_reading_factors_rs_t *rs;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(*rs));

    if (PP_ERR == pp_tp_ipmi_sens_get_reading_factors(sensor, imsg->data[1], rs)) {
	pp_bmc_imsg_err(imsg, IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_send_msg(imsg);
}

static int sd_cmd_set_sensor_hysteresis(imsg_t *imsg)
{
    ipmi_set_sensor_hysteresis_rq_t *rq = (void*)imsg->data;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    if (PP_ERR == pp_tp_ipmi_sens_set_hysteresis(sensor, rq)) {
	return pp_bmc_router_resp_err(imsg,IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int sd_cmd_get_sensor_hysteresis(imsg_t *imsg)
{
    ipmi_get_sensor_hysteresis_rs_t *rs;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(*rs));
    
    if (PP_ERR == pp_tp_ipmi_sens_get_hysteresis(sensor, rs)) {
	pp_bmc_imsg_err(imsg, IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_send_msg(imsg);
}

static int sd_cmd_set_sensor_thresholds(imsg_t *imsg)
{
    ipmi_set_sensor_thresholds_rq_t *rq = (void*)imsg->data;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    if (PP_ERR ==  pp_tp_ipmi_sens_set_thresholds(sensor, rq)) {
	return pp_bmc_router_resp_err(imsg,IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
#if defined (PP_FEAT_IPMI_THRESHOLD_PERSIST)
    {
        ipmi_get_sensor_thresholds_rs_t _rs, *rs = &_rs;

        if(PP_ERR != pp_tp_ipmi_sens_get_thresholds(sensor, rs)) {
            int do_save = 0;
            
            if(rs->mask & IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT) {
                pp_cfg_set_int(rs->lower_non_crit, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "low_noncrit");
                ++do_save;
            }
            if(rs->mask & IPMI_SENSOR_THRESHOLD_LOWER_CRIT) {
                pp_cfg_set_int(rs->lower_crit, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "low_crit");
                ++do_save;
            }
            if(rs->mask & IPMI_SENSOR_THRESHOLD_LOWER_NON_RECOV) {
                pp_cfg_set_int(rs->lower_non_recov, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "low_nonrecover");
                ++do_save;
            }
            if(rs->mask & IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT) {
                pp_cfg_set_int(rs->upper_non_crit, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "up_noncrit");
                ++do_save;
            }
            if(rs->mask & IPMI_SENSOR_THRESHOLD_UPPER_CRIT) {
                pp_cfg_set_int(rs->upper_crit, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "up_crit");
                ++do_save;
            }
            if(rs->mask & IPMI_SENSOR_THRESHOLD_UPPER_NON_RECOV) {
                pp_cfg_set_int(rs->upper_non_recov, bmc_sensors_thresh, 
                               (int)rq->sensor_no, "up_nonrecover");
                ++do_save;
            }
            if(do_save) {
//                pp_cfg_set(sensor->base.id, bmc_sensors_name,
//                           (int)rq->sensor_no);
                pp_cfg_save(DO_FLUSH);
            }
        }
    }
#endif
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int sd_cmd_get_sensor_thresholds(imsg_t *imsg)
{
    ipmi_get_sensor_thresholds_rs_t *rs;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }

    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(*rs));
    
    if (PP_ERR == pp_tp_ipmi_sens_get_thresholds(sensor, rs)) {
	pp_bmc_imsg_err(imsg, IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_send_msg(imsg);
}

static int sd_cmd_set_sensor_event_enable(imsg_t *imsg)
{
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    /* the request can be 2 to 6 bytes long. Any missing byte is to be
       treated as zeroes.
    */
    ipmi_set_sensor_event_enable_rq_t full_rq;

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    assert(imsg->data_size >=2);
    
    if (imsg->data_size > sizeof(full_rq)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_LIMIT_EXCEEDED);
    }
    bzero(&full_rq, sizeof(full_rq));
    memcpy(&full_rq, imsg->data, imsg->data_size);
    
    if (PP_ERR == pp_tp_ipmi_sens_set_event_enable(sensor, &full_rq)) {
	return pp_bmc_router_resp_err(imsg,IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int sd_cmd_get_sensor_event_enable(imsg_t *imsg)
{
    ipmi_get_sensor_event_enable_rs_t *rs;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(*rs));
    bzero(rs, sizeof(ipmi_get_sensor_event_enable_rs_t));
    
    if (PP_ERR == pp_tp_ipmi_sens_get_event_enable(sensor, rs)) {
	pp_bmc_imsg_err(imsg, IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_send_msg(imsg);
}

static int sd_cmd_rearm_sensor_events(imsg_t *imsg)
{
    ipmi_rearm_sensor_events_rq_t full_rq;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);
    
    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }

    if (imsg->data_size > sizeof(full_rq)) {
	return pp_bmc_router_resp_err(imsg, IPMI_ERR_REQ_DATA_LEN_LIMIT_EXCEEDED);
    }
    bzero(&full_rq, sizeof(full_rq));
    memcpy(&full_rq, imsg->data, imsg->data_size);

    if (PP_ERR == pp_tp_ipmi_sens_rearm_events(sensor, &full_rq)) {
	return pp_bmc_router_resp_err(imsg,IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_resp_err(imsg, IPMI_ERR_SUCCESS);
}

static int sd_cmd_get_sensor_event_status(imsg_t *imsg)
{
    ipmi_sensor_get_event_status_rs_t *rs;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);

    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    
    rs = pp_bmc_imsg_resp(imsg, IPMI_ERR_SUCCESS, sizeof(*rs));
    
    if (PP_ERR == pp_tp_ipmi_sens_get_event_status(sensor, rs)) {
	pp_bmc_imsg_err(imsg, IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
    }
    return pp_bmc_router_send_msg(imsg);
}

static int sd_cmd_get_sensor_reading(imsg_t *imsg)
{
    unsigned char resp_code;
    ipmi_sensor_get_reading_rs_t resp_data;
    pp_tp_ipmi_sens_t* sensor = get_sensor_by_index(imsg->data[0]);
    
    if (!sensor) {
        return pp_bmc_router_resp_err(imsg, IPMI_ERR_ITEM_NOT_PRESENT);
    }
    if (PP_ERR == pp_tp_ipmi_sens_get_reading(sensor, &resp_code, &resp_data)){
	if (errno == ENOSYS) {
	    return pp_bmc_router_resp_err(imsg,
					  IPMI_ERR_CMD_ILLEGAL_FOR_THIS_ITEM);
	} else {
	    pp_bmc_log_perror("[SD]: pp_tp_ipmi_sens_get_reading from %s "
			      "failed", pp_tp_ipmi_sens_to_string(sensor));
	    return pp_bmc_router_resp_err(imsg, IPMI_ERR_UNSPECIFIED);
	}
    }
    return pp_bmc_router_resp_msg(imsg, resp_code, &resp_data,
				  sizeof(resp_data));
}

/********************************************************************
 * SDR device command table
 */

static const dev_cmd_entry_t sd_cmd_tab[] = {
    /* not implemented 
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_DEVICE_SDR_INFO,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_DEVICE_SDR,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_RESERVE_DEVICE_SDR_REPO,
    },
    */
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_READING_FACTORS,
      cmd_hndlr:      sd_cmd_get_sensor_reading_factors,
      min_data_size:  1,
      min_priv_level: IPMI_PRIV_USER,     
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_SET_SENSOR_HYSTERESIS,
      cmd_hndlr:      sd_cmd_set_sensor_hysteresis,
      min_data_size:  sizeof(ipmi_set_sensor_hysteresis_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_HYSTERESIS,
      cmd_hndlr:      sd_cmd_get_sensor_hysteresis,
      min_data_size:  2,
      min_priv_level: IPMI_PRIV_USER,
    },

    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_SET_SENSOR_TRESHOLDS,
      cmd_hndlr:      sd_cmd_set_sensor_thresholds,
      min_data_size:  sizeof(ipmi_set_sensor_thresholds_rq_t),
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_TRESHOLDS,
      cmd_hndlr:      sd_cmd_get_sensor_thresholds,
      min_data_size:  1,
      min_priv_level: IPMI_PRIV_USER,
    },

    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_SET_SENSOR_EVENT_ENABLE,
      cmd_hndlr:      sd_cmd_set_sensor_event_enable,
      min_data_size:  2,
      min_priv_level: IPMI_PRIV_USER,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_EVENT_ENABLE,
      cmd_hndlr:      sd_cmd_get_sensor_event_enable,
      min_data_size:  1,
      min_priv_level: IPMI_PRIV_USER,
    },

    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_REARM_SENSOR_EVENTS,
      cmd_hndlr:      sd_cmd_rearm_sensor_events,
      min_data_size:  2,
      min_priv_level: IPMI_PRIV_USER,
    },

    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_EVENT_STATUS,
      cmd_hndlr:      sd_cmd_get_sensor_event_status,
      min_data_size:  1,
      min_priv_level: IPMI_PRIV_USER,
    },

    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_READING,
      cmd_hndlr:      sd_cmd_get_sensor_reading,
      min_data_size:  1,
      min_priv_level: IPMI_PRIV_USER,
    },

    /* not implemented
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_SET_SENSOR_TYPE,
    },
    { netfn:          IPMI_NETFN_SENSOR_EVENT,
      cmd:            IPMI_CMD_GET_SENSOR_TYPE,
    },
    */
    
    { cmd_hndlr:      NULL,
    }
};

/********************************************************************
 * sensor devices c'tor/d'tor
 */

/**
 * Callback for the sensor library to report new object. If the object is
 * a pp_tp_ipmi_sens_t, it will be passed to the sensor_config_addsensor
 * function to add it to the bmc sensor devices.
 */
void pp_bmc_dev_sensor_add_object(pp_tp_obj_t* obj)
{
    if (PP_TP_OBJ_IS_TYPE(PP_TP_IPMI_SENS, obj)) {
        sensor_config_addsensor(((pp_tp_ipmi_sens_t*)obj)->number,
				(pp_tp_ipmi_sens_t*)obj);
    }
}

/*
 * Used to report new sensor objects to the bmc.
 * 1. SDRR is in persistent mode:
 *    - lookup SDR by key
 *    - if found: set that SDR into sensor and add to device
 *    - else ignore sensor
 * 2. SDRR is in transient mode:
 *    - add SDR transiently to SDRR
 *    - add sensor to device
 */
static int sensor_config_addsensor(unsigned char sensor_nr,
                                   pp_tp_ipmi_sens_t* sensor)
{
    ipmi_sdr_header_t* defsdr, *newsdr;
    ipmi_sdr_key_sensor_t* sdrkey;
    
    defsdr = pp_tp_ipmi_sens_get_sdr(sensor);
    assert(defsdr != NULL);

    /* check for existing SDR in SDRR */
    if (pp_bmc_sdrr_is_persistent()) {
	sdrkey = ipmi_sdr_get_key_sensor_ptr(defsdr);
	if (NULL != (newsdr = pp_bmc_sdrr_lookup_by_key_sensor(sdrkey))) {
	    if (PP_ERR == pp_tp_ipmi_sens_set_sdr(sensor, newsdr)) {
		pp_bmc_log_warn("[SD] failed to set SDR for sensor (%s), "
				"skipped!", pp_tp_ipmi_sens_to_string(sensor));
		return PP_ERR;
	    }
	} else {
	    pp_bmc_log_notice("[SD] no SDR for sensor (%s), skipped!",
			      pp_tp_ipmi_sens_to_string(sensor));
	    return PP_SUC;
	}
    }

    /* add to sensor device */
    if (PP_ERR == sd_sensor_add_sensor(sensor_nr, sensor)) {
	pp_bmc_log_error("[SD] sensor (%s) has duplicate id (0x%x), skipped!",
			 pp_tp_ipmi_sens_to_string(sensor), sensor_nr);
	return PP_ERR;
    }

    /* add default SDR if we have a transient SDRR */
    if (!pp_bmc_sdrr_is_persistent()) {
	pp_bmc_sdrr_add_sdr_transiently(defsdr);
    }

    pp_bmc_log_info("[SD] sensor %#.2x (%s) added",
		    sensor_nr, pp_tp_ipmi_sens_to_string(sensor));
    return PP_SUC;
}

#if defined (PP_FEAT_IPMI_THRESHOLD_PERSIST)
typedef struct {
    BITFIELD7(unsigned char,        low_noncrit:1,
                                    low_crit:1,
                                    low_nonrecover:1,
                                    up_noncrit:1,
                                    up_crit:1,
                                    up_nonrecover:1,
                                    reserved:2);
} sensor_thresh_mask_t;

#define SET_THRESH_AND_MASK(__cfg_thresh__, __rq_thresh__) \
    if(PP_ERR != pp_cfg_get_ushort_nodflt(&u, bmc_sensors_thresh, i, \
                                          #__cfg_thresh__)) { \
        rq->__rq_thresh__ = (unsigned char)u; \
        mask->__cfg_thresh__ = 1; \
    }

void pp_bmc_dev_sensor_init_thresholds() {
    /* notes:
     1.store persistent thresholds by sensor number
         - remember name + 6 thresholds = threshold record
       while init thresholds check
         - if sensor number exist
	 - if name fits to sensor number
	 if false, then
	 - delete threshold record
       _NOTE_! for the moment, we trust in sensor number no matter what name
     2.implement OEM command sensor_unset_threshold
       will delete threshold record and
       set thresholds from SDR
    */
//    char* name;
    unsigned char i;
    pp_tp_ipmi_sens_t* sensor;
    u_short u;
    ipmi_set_sensor_thresholds_rq_t *rq =
	malloc(sizeof(ipmi_set_sensor_thresholds_rq_t));
    sensor_thresh_mask_t *mask = (sensor_thresh_mask_t*)&rq->mask;
    int do_save = 0;
        
    for(i=0;i<255;i++) {
	if((sensor = (pp_tp_ipmi_sens_t*) get_sensor_by_index(i)) != NULL) {
	    if (PP_TP_OBJ_IS_TYPE(PP_TP_THRESH_IPMI_SENS, sensor)) {
	       memset(rq, 0, sizeof(ipmi_set_sensor_thresholds_rq_t));			
//		if((pp_cfg_get_nodflt(&name, bmc_sensors_name, i) == PP_SUC) &&
//		(strcmp(sensor->base.id, name) == 0)) {
		    rq->mask = 0x00;

                    SET_THRESH_AND_MASK(low_noncrit, lower_non_crit);
                    SET_THRESH_AND_MASK(low_crit, lower_crit);
                    SET_THRESH_AND_MASK(low_nonrecover, lower_non_recov);
                    SET_THRESH_AND_MASK(up_noncrit, upper_non_crit);
                    SET_THRESH_AND_MASK(up_crit, upper_crit);
                    SET_THRESH_AND_MASK(up_nonrecover, upper_non_recov);
                    
		    if (PP_ERR ==  pp_tp_ipmi_sens_set_thresholds(sensor, rq)) {
			pp_bmc_log_info("IPMI Sensor id %d --> %s set threshold fail",
					i, sensor->base.id);
		    }
//		} else {
//		    pp_cfg_remove("bmc.sensors[%u]", i);
//                    ++do_save;
//		}
	    }
	}
    }
    
    if(do_save) {
        /* only save if we modified configuration */
        pp_cfg_save(DO_FLUSH);
    }
    
    free(rq);
}   
#endif

int pp_bmc_dev_sensor_init()
{
    assert(sensor_map == NULL);
    sensor_map = pp_rb_tree_int_new();

    /* register all entries of cmd tab */
    if (PP_ERR == pp_bmc_core_reg_cmd_tab(sd_cmd_tab))
	return PP_ERR;

    pp_bmc_log_info("[SD] started");
    return PP_SUC;
}

void pp_bmc_dev_sensor_cleanup()
{
    /* unregister all entries of cmd tab */
    pp_bmc_core_unreg_cmd_tab(sd_cmd_tab);

    pp_rb_tree_destroy(sensor_map);
    sensor_map = NULL;
    pp_bmc_log_info("[SD] shut down");
}
