/**
 * bmc_router.h
 *
 * Description: BMC Message Router
 * 
 * Note: The local loopi channels (IPMI_CHAN_LOOP_*) are hardwired hidden
 * in this implementation. Messages can be sent to the channel, but
 * the channel does not have a configuration nor an information
 * structure.
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#include <malloc.h>
#include "pp/base.h"
#include "pp/bmc/ipmi_chan.h"
#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/bmc_router.h"
#include "pp/bmc/debug.h"

#include "bmc_dev_app_channel.h"
#include "receive_message_queue.h"
 
/*
 *  Static Data
 */

typedef struct {
    int (*msg_hndlr)(imsg_t*);
    void (*reset_hndlr)(void);
    void (*config_hndlr)(void);
    const pp_bmc_router_chan_adapter_info_t *chan_info;  // this struct will never change, so a reference is just fine
    pp_bmc_router_chan_adapter_config_t volatile_config; // current (volatile) config for channel
} chan_adapter_entry_t;

static chan_adapter_entry_t *chan_adapter_tab = 0;

static void(*nc_interrupt_actor)(unsigned char) = NULL;

/*
 *  Interface
 */

int pp_bmc_router_init()
{
    int sz = sizeof(chan_adapter_entry_t) * (IPMI_CHAN_MAX + 1);
    chan_adapter_tab = malloc(sz);
    memset(chan_adapter_tab, 0, sz);
    
    pp_bmc_log_info("[BMCrouter] router started");
    return PP_SUC;
}

void pp_bmc_router_cleanup()
{
    free(chan_adapter_tab);
    chan_adapter_tab = NULL;
    nc_interrupt_actor = NULL; // clean up actor
    
    pp_bmc_log_info("[BMCrouter] router shut down");
}

/******************* Message routing functionality *************************/

int pp_bmc_router_send_msg(imsg_t* imsg)
{
    assert(chan_adapter_tab); /* must be init */
    if (imsg->chan > IPMI_CHAN_MAX) return PP_ERR;
    if (!chan_adapter_tab[imsg->chan].msg_hndlr) return PP_ERR;

    pp_bmc_imsg_dump(PP_BMC_LOG_DEBUG, "[BMCrouter]", imsg);
    
    return chan_adapter_tab[imsg->chan].msg_hndlr(imsg);
}

int pp_bmc_router_resp_err(imsg_t* imsg, unsigned char compl_code)
{
    return pp_bmc_router_resp_msg(imsg, compl_code, NULL, 0);
}

int pp_bmc_router_resp_msg(imsg_t* imsg,
                           unsigned char compl_code,
                           const void* resp_data,
                           unsigned int resp_data_size)
{
    void *p = pp_bmc_imsg_resp(imsg, compl_code, resp_data_size);
    memcpy(p, resp_data, resp_data_size);
    return pp_bmc_router_send_msg(imsg);
}

/******************* Non communication interrupt *****************************/

int pp_bmc_router_register_nc_interrupt(void(*nci_actor)(unsigned char value)) {
    if (nci_actor == NULL) {
        nc_interrupt_actor = NULL;
    } else {
        if (nc_interrupt_actor == NULL) {
            nc_interrupt_actor = nci_actor;
        } else {
            // interrupt actor already registered
            return PP_ERR;
        }
    }
    return PP_SUC;
}

void pp_bmc_router_set_nc_interrupt(unsigned char value) {
    if (nc_interrupt_actor == NULL) {
        pp_bmc_log_notice("[BMCrouter] No non communication interrupt actor registered. Set to %x ignored", value);
    } else {
        pp_bmc_log_debug("[BMCrouter] non communcation interrupt set to %x", value);
        nc_interrupt_actor(value);
    }
}



/******************* Special/internal channel functions **********************/

pp_bmc_router_chan_adapter_config_t*
pp_bmc_router_reg_chan(unsigned char chan_id,
                       int (*msg_hndlr)(imsg_t*),
                       void (*reset_hndlr)(void),
                       void (*config_update_hndlr)(void),
                       const pp_bmc_router_chan_adapter_info_t* channel_info)
{
    assert(chan_adapter_tab); /* must be init */
    if (chan_id > IPMI_CHAN_MAX) return NULL;
    
    /* chan_id already registered? */
    if (chan_adapter_tab[chan_id].msg_hndlr) return NULL;

#if !defined (PP_FEAT_BMC_OEMCMDS_ONLY)
    if (config_update_hndlr != NULL) {
        if (pp_bmc_dev_channel_get_config(chan_id, &(chan_adapter_tab[chan_id].volatile_config)) == PP_ERR) {
            /* no configuration available (should no happen ?) */
            return NULL;
        }
    }
#endif
    
    chan_adapter_tab[chan_id].msg_hndlr = msg_hndlr;
    chan_adapter_tab[chan_id].reset_hndlr = reset_hndlr;
    chan_adapter_tab[chan_id].config_hndlr = config_update_hndlr;
    chan_adapter_tab[chan_id].chan_info = channel_info;
    
    return &(chan_adapter_tab[chan_id].volatile_config);
}

int pp_bmc_router_unreg_chan(unsigned char chan_id)
{
    assert(chan_adapter_tab); /* must be init */
    if (chan_id > IPMI_CHAN_MAX) return PP_ERR;
    
    /* chan_id really registered? */
    if (!chan_adapter_tab[chan_id].msg_hndlr) return PP_ERR;
    
    chan_adapter_tab[chan_id].msg_hndlr = NULL;

    return PP_SUC;
}

const pp_bmc_router_chan_adapter_info_t* 
pp_bmc_router_get_chan_adapter_info(unsigned char chan_id) {
    assert(chan_adapter_tab); /* must be init */
    if (chan_id > IPMI_CHAN_MAX) return NULL;

    if ((chan_id == IPMI_CHAN_LOOP_ERIC) || (chan_id == IPMI_CHAN_LOOP_SNMPD)) {
        /* loop interface is invisible */
        return NULL;
    }
        
    /* chan_id really registered? */
    if (!chan_adapter_tab[chan_id].msg_hndlr) return NULL;
    
    return chan_adapter_tab[chan_id].chan_info;
}

void pp_bmc_router_channel_reset() {
    int i;
    assert(chan_adapter_tab);
    
    for (i=0; i<IPMI_CHAN_MAX; i++) {
        if (chan_adapter_tab[i].reset_hndlr)
            chan_adapter_tab[i].reset_hndlr();
    }
}

pp_bmc_router_chan_adapter_config_t*
pp_bmc_router_channel_get_config(unsigned char chan_id) {
    assert (chan_adapter_tab); /* must be init */
    if (chan_id > IPMI_CHAN_MAX) return NULL;
    
    if ((chan_id == IPMI_CHAN_LOOP_ERIC) || (chan_id == IPMI_CHAN_LOOP_SNMPD)) {
        /* loop interface is invisible */
        return NULL;
    }
    
    if ((chan_adapter_tab[chan_id].msg_hndlr != NULL) && 
        (chan_adapter_tab[chan_id].config_hndlr != NULL))
    {
        return &(chan_adapter_tab[chan_id].volatile_config);
    }
    else
    {
        return NULL;
    }
}

void pp_bmc_router_channel_config_updated(unsigned char chan_id) {
    assert (chan_adapter_tab); /* must be init */
    assert (chan_adapter_tab[chan_id].msg_hndlr != NULL);
    
    if (chan_adapter_tab[chan_id].config_hndlr != NULL) {
        chan_adapter_tab[chan_id].config_hndlr();
    }
}     
