/**
 * event_receiver.h
 *
 * The event receiver is the central point for adding new events. It can
 * be configured to disable events from the PCI management bus and ICMB.
 * Valid events are forwarded to the SEL. If the SEL is full, they are
 * stored in an internal, volatile queue to guarantee correct PEF
 * operation.
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __EVENT_RECEIVER_H__
#define __EVENT_RECEIVER_H__

#include <string.h>
#include "pp/base.h"

#include <pp/bmc/bmc_event.h>
#include "pp/bmc/utils.h"
#include "bmc_dev_sel_nv.h"

/**
 * Initialize the event receiver
 */
int pp_bmc_event_receiver_init(void);

/**
 * Clean up the event receiver
 */
void pp_bmc_event_receiver_cleanup(void);


/**
 * Returns a reference to the specified SEL event.
 * Attention: memory for the event is allocated and must
 * be freed by the caller. Returns NULL if the specified
 * event does not exist.
 */
sel_entry_t* pp_bmc_get_event(unsigned short eventID);

/**
 * Gets the event with the specified id and returns the
 * id of the next event. Returns 0xFFFF if the specified
 * event is the last event and 0x0000 if the specified
 * event does not exist.
 */
unsigned short pp_bmc_get_next_event_id(unsigned short eventID);

/**
 * The PEF calls this function to notify the event receiver that the
 * specified id is not needed any more. This takes effect only if the
 * SEL is full and events are stored in an internal queue.
 */
int pp_bmc_release_event(unsigned short id);

/**
 * Get the timestamp of the most recent addition to the SEL.
 */
time_t pp_bmc_get_most_recent_addition_timestamp(void);

/**
 * Set the timestamp of the most recent addition to the SEL.
 */
void pp_bmc_set_most_recent_addition_timestamp(time_t la_time);

/**
 * Get the id of the last event
 */
unsigned short pp_bmc_get_last_event_id(void);

/**
 * Sets one listener that is notified of event changes. This is usually
 * the pef. Multiple calls of this function overwrite the handlers.
 */
void pp_event_set_listener(sel_listener_hndl_t handlers);


#endif
