/**
 * bmc_channel_alert.h
 * 
 * The interface for channel specific alert mechanisms. Channels can
 * register alert code through this interface. Header file only, implemented
 * by bmc_dev_pef_alert.c
 * 
 * (c) 2005 Peppercon AG, Georg Hoesch <geo@peppercon.de>
 */

#ifndef __BMC_CHANNEL_ALERT_H__
#define __BMC_CHANNEL_ALERT_H__

#include <pp/bmc/bmc_sel_record.h>

/**
 * Channel alert initiation handle. Call code within channel to initiate
 * an alert to the specified destination and alert_string. Note that all
 * references will be invalid after the function returns and thus must be
 * copied by the channel if needed.
 * @param destination channel alert destination
 * @param alert_string the alert string, must not be freed, may be NULL
 * @param event a pointer to the event causing the alert, must not be freed, may be NULL
 * @param event_severity event severity as specified in the matching filter
 * @param guid a pointer to the guid to use in the alert, must not be freed
 * @param force set to 1 to force the alert even if alerting on that channel is deactivated
 * @param context is the context of the alert, must be returned when reporting
 *        the final alert status
 * @return a handle to the alert if successful
 * @return NULL if the alert could not be initiated (another alert in progress, ...)
 */
typedef void* (*pp_bmc_alert_initiate_t)(unsigned char destination,
                                         unsigned char* alert_string,
                                         sel_entry_t* event,
                                         unsigned char event_severity,
                                         unsigned char* guid,
                                         unsigned char force,
                                         void* context);

/**
 * Acknowledge the specified alert that was initiated at the channel.
 * Upon reception, the channel may stop further alerting and report a
 * successfull alert to the bmc_dev_pef_alert module.
 */
typedef void (*pp_bmc_alert_acknowledge_t)(void* alert_hndl);

/**
 * Called by the channels at the end of an alert to indicate whether
 * the alert was successful or not.
 */
void pp_bmc_pef_report_alert_status(void*, unsigned char success);

/**
 * Registers a new channel alert module at the pef.
 * @return a handle used for unregistration
 */
int pp_bmc_pef_alert_register_channel(unsigned char channel,
                                   pp_bmc_alert_initiate_t start_alert,
                                   pp_bmc_alert_acknowledge_t acknowledge_alert);

/**
 * Unregister a channel at the pef using the registration handle.
 */
int pp_bmc_pef_unregister_channel(int hndl);


#endif
