/**
 * \file bmc_core.h
 *
 * Description: BMC Core
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __BMC_CORE_H__
#define __BMC_CORE_H__

#include "pp/bmc/bmc_imsg.h"

/**
 * Initializes the BMC Core.
 *
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_core_init(void);

/**
 * Finishes the BMC Core.
 */
void pp_bmc_core_cleanup(void);

/**
 * (Re)Initializes the BMC sensor topology.
 *
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_sensor_topo_parse_new(void);

/**
 * Message handler of BMC Core (used by Channel Adapters).
 *
 * @param imsg The IPMI message structure
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_core_handle_msg(imsg_t* imsg);

/**
 * Registers a command handler of a BMC Device at BMC Core.
 *
 * @param netfn Desired network function
 * @param cmd Desired command code 
 * @param min_data_size Minimal size of body (-1 = don't care)
 * @param min_priv_level Minimal required privilege level of session (0 = don't
care)
 * @param cmd_hndlr Command handler of device
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_core_reg_cmd(unsigned char netfn, unsigned char cmd,
                        int min_data_size, int min_priv_level,
                        int (*cmd_hndlr)(imsg_t*));

/**
 * Unregisters a command handler on a BMC Device at BMC Core.
 *
 * @param netfn Desired network function
 * @param cmd Desired command code 
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_core_unreg_cmd(unsigned char netfn, unsigned char cmd);

/**
 * Command table typically used by a BMC Core.
 */
typedef struct {
    int (*cmd_hndlr)(imsg_t*);
    unsigned char netfn;
    unsigned char cmd;
    unsigned int min_data_size;
    unsigned char min_priv_level;
} dev_cmd_entry_t;

/**
 * Registers an entire command handler table of a BMC Device at BMC Core.
 *
 * @param tab Pointer to table (last entry: cmd_hndlr == NULL)
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_core_reg_cmd_tab(const dev_cmd_entry_t* tab);

/**
 * Unregisters an entire command handler table of a BMC Device at BMC Core.
 *
 * @param tab Pointer to table (last entry: cmd_hndlr == NULL)
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
*/
int pp_bmc_core_unreg_cmd_tab(const dev_cmd_entry_t* tab);

#endif /* !__BMC_CORE_H__ */
