/**
 * bmc_router.h
 *
 * Description: BMC Message Router
 *
 * (c) 2004 Peppercon AG, Ralf Guenther <rgue@peppercon.de>
 */

#ifndef __BMC_ROUTER_H__
#define __BMC_ROUTER_H__

#include "pp/bmc/bmc_imsg.h"
#include "pp/bmc/ipmi_chan.h"

/**
 * The channel info structure, provided by channel. A constant structure that 
 * describes the channels general properties.
 */
typedef struct {
    unsigned char       medium;    /* Medium Type (see ipmi spec table 6-3)   */
    unsigned char       proto;     /* Protocol Type (see ipmi spec table 6-2) */
    unsigned char       sess_supp; /* Session support:
                                    *     0x00 channel is session-less
                                    *     0x01 channel is single-session
                                    *     0x10 channel is multi-session
                                    *     0x11 channel is session-based */
    /* constants for this struct are defined in 'ipmi_chan.h' */
} pp_bmc_router_chan_adapter_info_t;

/*
 * The channel configuration structure. The bmc_router passes this
 * structure to the bmc whenever the channel needs to be configured.
 * This is the case after channel startup or when the channel
 * configuration options have been changed by an ipmi command.
 * The channel must not change this structure itself.
 */
struct pp_bmc_router_chan_adapter_config_s {
    BITFIELD5(unsigned char,
              access_mode: 3,
              user_level_authentication_disable: 1,
              per_message_authentication_disable: 1,
              alerting_disable: 1,
              dummy: 2);
    BITFIELD2(unsigned char,
              channel_priv_lvl: 4,
              dummy2: 4);
} __attribute__ ((packed));
typedef struct pp_bmc_router_chan_adapter_config_s pp_bmc_router_chan_adapter_config_t;



/**
 * Initializes the BMC Message Router.
 *
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 */
int pp_bmc_router_init(void);

/**
 * Finishes the BMC Message Router.
 */
void pp_bmc_router_cleanup(void);



/**
 * Registers a Channel Adapter at the BMC Core.
 *
 * @param chan_id Desired Channel ID
 * @param msg_hndlr Message handler of adapter
 * @param reset_hndlr The channel warm reset handler, may be NULL if
 *        the channel does not support warm resets
 * @param config_hndlr The channel config handler, called if the channel
 *        configuration data has changed. May be NULL if the channel
 *        does not support configuration parameters (for example System Intf)
 * 
 * @return a reference to the channels configuration data if the channel
 *         could be registered. This will be valid while the channel is
 *         registered. It may be changed from other locations but the
 *         channel will be notified about those changes to allow immediate
 *         reactions.
 *         This reference will also be returned for channels which do not
 *         support configuration options (config_hndlr == NULL). In this case
 *         it can be savely ignored.
 * @return NULL if the channel could not be registered
 * @see pp_bmc_router_unreg_chan()
 */
pp_bmc_router_chan_adapter_config_t*
pp_bmc_router_reg_chan(unsigned char chan_id,
            /* send a msg to the channel */
            int (*msg_hndlr)(imsg_t*),
            /* reset the channel */
            void (*reset_hndlr)(void),
            /* notify the channel that the config structure has changed */
            void (*config_update_hndlr)(void),
            /* a reference to the channel info,
	     * must remain valid until deregistration */
            const pp_bmc_router_chan_adapter_info_t* channel_info);

/**
 * Unregisters a Channel Adapter at the BMC Core.
 *
 * @param chan_id Desired Channel ID
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 * @see pp_bmc_router_reg_chan()
 */
int pp_bmc_router_unreg_chan(unsigned char chan_id);

/**
 * Perform a warm reset on all registered channels.
 * Each channel should reinitialize its communication state
 * without loosing its major interface configuration.
 */
void pp_bmc_router_channel_reset(void);

/**
 * @return The configuration structure for the specified channel.
 *         Make sure to call pp_bmc_router_channel_config_updated()
 *         after you changed this structure.
 * @return NULL if the channel is not registered
 */
pp_bmc_router_chan_adapter_config_t*
pp_bmc_router_channel_get_config(unsigned char channel);

/**
 * Notify the channel that the configuration structure has changed.
 */
void pp_bmc_router_channel_config_updated(unsigned char channel);

/**
 * Gets channel adapter info for requested channel
 * 
 * @param chan_id Desired Channel ID
 * @return reference to channel adapter info, must not be changed or freed
 * @return NULL if the channel is not registered
 */
const pp_bmc_router_chan_adapter_info_t* 
pp_bmc_router_get_chan_adapter_info(unsigned char chan_id);



/**
 * Register the interrupt actor for non communication interrupts.
 * This handler will be called by the bmc to set/unset the SMS-ATN
 * bit through the System Interface to request attention from the
 * System management software.
 * @param set_nc_interrupt The interrupt actor from the System interface
 *     As only one system interface can exist, only one interrupt actor
 *     can be registered. Register a NULL interrupt to remove the stored
 *     interrupt actor.
 * @return PP_SUC if registration was successfull
 * @return PP_ERR if another interrupt actor is already registered.
 */
int pp_bmc_router_register_nc_interrupt(void(*nc_interrupt_actor)(unsigned char value));

/**
 * Set the non communication interrupt status of the host. This interrupt
 * will be forwarded through the System Interface to the SMS_ATN bit.
 */
void pp_bmc_router_set_nc_interrupt(unsigned char value);



/**
 * Sends an IPMI message to a Channel Adapter (used by BMC Devices).
 * For regular ipmi responses your have to call pp_bmc_imsg_resp()
 * to convert the ipmi request message to a response message before
 * calling this function.
 *
 * @param chan_id Destination channel ID
 * @param imsg IPMI Message to route (chan field determines destination)
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error (bad channel)
 */
int pp_bmc_router_send_msg(imsg_t* imsg);

/**
 * Send error response without data (used by BMC Devices).
 *
 * @param imsg Request message that should be responded
 * @param compl_code Completion code (see IPMI_ERR_xxx)
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 * @see ipmi_err.h
 */
int pp_bmc_router_resp_err(imsg_t* imsg, unsigned char compl_code);

/**
 * Responses an IPMI command (used by BMC Devices).
 *
 * @param imsg Request message that should be responded
 * @param compl_code Completion code (see IPMI_ERR_xxx)
 * @param resp_data Pointer to the response body (completion code not included)
 * @param resp_data_size Size of response body (without completion code)
 * @return PP_SUC (0) if success
 *         PP_ERR (-1) if error
 * @see ipmi_err.h
 */
int pp_bmc_router_resp_msg(imsg_t* imsg,
                           unsigned char compl_code,
                           const void* resp_data,
                           unsigned int resp_data_size);

#endif /* !__BMC_ROUTER_H__ */
